<?php

if ( ! defined( 'ABSPATH' ) ) {
	exit; // Exit if accessed directly
}


if ( ! class_exists( 'Tco_Woo_Validations' ) ) {

    /**
	 * TcoWooCheckout WooCommerce Validation class
     *
	 * This is a fallback validations once Woocommerce fails to validate
     *
	 * @package TcoWooCheckout
     * @since 2.2.0
	 */

    class Tco_Woo_Validations {

        /**
         * The single instance of the class
         *
         * @since 2.2.0
         */
        protected static $_instance = null;


        /**
         * Get the instance
         *
         * @since 2.2.0
         */
        public static function instance() {
            if ( is_null( self::$_instance ) ) {
                self::$_instance = new self();
            }
            return self::$_instance;
        }


        public function __construct() {

            //Predefined validations

            add_action( 'tco_woo_checkout_validation_field_postcode', array(__CLASS__, 'validate_postcode' ), 10, 4 );

            add_action( 'tco_woo_checkout_validation_field_phone', array(__CLASS__, 'validate_phone' ), 10, 4 );

            add_action( 'tco_woo_checkout_validation_field_email', array(__CLASS__, 'validate_email' ), 10, 4 );

            add_action( 'tco_woo_checkout_validation_field_state', array(__CLASS__, 'validate_state' ), 10, 4 );

        }

        /**
         * @param Array $name . Field's name
         * @param Array $text . Notice
         * @since 2.2.0
         */

        static function show_error_message ( $name, $text ) {

            wc_add_notice( '<span class="checkout_error" id="tco_woo_checkout_error_for_'.$name.'">'.$text.'</span>', 'error' );

        }

        /**
         * @param Array $name . Field's name
         * @param Array $value . Field's value
         * @param Array $field . Field's definition
         * @param Array $post_data . Submitted post data
         * @since 2.2.0
         */
        static function validate_phone ( $name, $value, $field, $post_data ){

            $value = wc_format_phone_number( $value );

            if ( !WC_Validation::is_phone( $value ) ) {

                self::show_error_message( $name, sprintf( __( '%s is not a valid phone number.' , 'tco_woo_checkout' ), '<strong>' . esc_html( $field['label'] ) . '</strong>' ) );

            }

        }

        /**
         * @param Array $name . Field's name
         * @param Array $value . Field's value
         * @param Array $field . Field's definition
         * @param Array $post_data . Submitted post data
         * @since 2.2.0
         */
        static function validate_postcode ( $name, $value, $field, $post_data ){

            $section = explode('_', $name );
            
            $country = isset( $post_data[ $section[0] . '_country' ] ) ? $post_data[ $section[0] . '_country' ] : WC()->customer->{"get_{$section[0]}_country"}();

            $value = wc_format_postcode( $value, $country );

            if ( !WC_Validation::is_postcode( $value, $country ) ) {

                self::show_error_message( $name, sprintf( __( '%s is not a valid postcode / ZIP.' , 'tco_woo_checkout' ), '<strong>' . esc_html( $field['label'] ) . '</strong>' ) );

            }

        }

        /**
         * @param Array $name . Field's name
         * @param Array $value . Field's value
         * @param Array $field . Field's definition
         * @param Array $post_data . Submitted post data
         * @since 2.2.0
         */
        static function validate_email ( $name, $value, $field, $post_data ){

            $value = sanitize_email( $value );

            if ( !is_email( $value ) ) {

                self::show_error_message( $name, sprintf( __( '%s is not a valid email address.' , 'tco_woo_checkout' ), '<strong>' . esc_html( $field['label'] ) . '</strong>' ) );
                
            }      

        }

        /**
         * @param Array $name . Field's name
         * @param Array $value . Field's value
         * @param Array $field . Field's definition
         * @param Array $post_data . Submitted post data
         * @since 2.2.0
         */
        static function validate_state ( $name, $value, $field, $post_data ){

            //Do nothing, this is handled flawlessly by Woocommerce itself

        }
        
    }

    Tco_Woo_Validations::instance();

}
?>
