<?php

/**
 * QR Code element
 */

namespace Cornerstone\QRCodes\Element;

use function Cornerstone\QRCodes\Enqueue\enqueue;

// Values extended from div element
$divDefinition = cs_get_element('layout-div');

$values = $divDefinition->def['values'];

$values['qrcode_text'] = cs_value('https://theme.co', 'markup');
$values['qrcode_size'] = cs_value('auto', 'markup');
$values['qrcode_colorDark'] = cs_value('#000', 'markup:color');
$values['qrcode_colorLight'] = cs_value('#FFF', 'markup:color');
$values['qrcode_correctLevel'] = cs_value(1, 'markup:int');

// Change flexbox for better looking default
$values['layout_div_flexbox'][0] = true;
$values['layout_div_flex_justify'][0] = 'center';
$values['layout_div_flex_align'][0] = 'center';


/**
 * Controls
 * Extends from Div element adding in a new tab called QR
 */
function controls() {
  $divDefinition = cs_get_element('layout-div');
  $controls = $divDefinition->def['builder']();

  // No children for qrcodes
  unset($controls['control_nav']['layout_div:children']);

  // Add QR Code tab
  $controls['control_nav'] = array_merge([
    'qrcode' => __('QR Code', 'cornerstone'),
    'qrcode:general' => __('General', 'cornerstone'),
  ], $controls['control_nav']);

  // Add controls in QR Code tab
  $controls['controls'][] = [
    'type' => 'group',
    'group' => 'qrcode:general',
    'controls' => [

      // URL
      [
        'key' => 'qrcode_text',
        'label' => __('URL', 'cornerstone'),
        'type' => 'text',
        'options' => [
          'placeholder' => 'https://theme.co',
        ],
      ],

      // Color Dark
      [
        'key' => 'qrcode_colorDark',
        'label' => __('Color Dark', 'cornerstone'),
        'type' => 'color',
        'description' => __('The color of the QR Code bars', 'cornerstone'),
      ],

      // Color Light
      [
        'key' => 'qrcode_colorLight',
        'label' => __('Color Light', 'cornerstone'),
        'type' => 'color',
        'description' => __('The color of the QR Code background', 'cornerstone'),
      ],

      // Correct Level
      [
        'key' => 'qrcode_correctLevel',
        'label' => __('Correct Level', 'cornerstone'),
        'description' => __('A higher correct level will result in more QR Code bars from being created. Allowing you to add icons or images inside the QR Code and still being readable by a QR Code reader.', 'cornerstone'),
        'type' => 'select',
        'options' => [
          'choices' => [
            [
              'label' => 'L',
              'value' => 1,
            ],
            [
              'label' => 'M',
              'value' => 0,
            ],
            [
              'label' => 'Q',
              'value' => 3,
            ],
            [
              'label' => 'H',
              'value' => 2,
            ],
          ],
        ],
      ],

      // Size
      [
        'key'        => 'qrcode_size',
        'type'       => 'unit-slider',
        'label'      => __('Size', 'cornerstone'),
        'description' => __('Setting the size to auto will result in the QR Code taking up as most space as possible while keeping a box aspect ratio. You can futher control how the size will interact in the Primary tab which will control the QR Codes container', 'cornerstone'),
        'options'    => [
          'available_units' => [ 'px' ],
          'valid_keywords'  => [ 'auto' ],
          'fallback_value'  => '250px',
          'ranges'          => [
            'px'  => [ 'min' => 50, 'max' => 500, 'step' => 10  ],
          ],
        ],
      ],

    ],
  ];

  return $controls;
}

/**
 * Render
 * Extends from Div element adding in custom attributes so it has our special attributes
 */
function render($data) {
  $divDefinition = cs_get_element('layout-div');

  // Enqueue qrcodes script
  enqueue();

  $qrcodeData = cs_split_to_object($data, 'qrcode');

  // Set width and height for library
  $qrcodeData['width'] = $qrcodeData['height'] = str_replace('px', '', $qrcodeData['size']);
  unset($qrcodeData['size']);

  $customAtts = $data['custom_atts'];

  if (empty($customAtts)) {
    $customAtts = [];
  } else {
    $customAtts = cs_maybe_json_decode($customAtts);
  }


  $customAtts['data-cs-qrcode'] = json_encode($qrcodeData);

  $data['custom_atts'] = $customAtts;

  return $divDefinition->getRenderFn()($data);
}

// Register Element

cs_register_element( 'qrcode', [
  'title' => __( 'QR Code', 'cornerstone' ),
  'icon' => '<svg xmlns="http://www.w3.org/2000/svg" viewBox="0 0 448 512"><!--!Font Awesome Pro 6.7.2 by @fontawesome - https://fontawesome.com License - https://fontawesome.com/license (Commercial License) Copyright 2025 Fonticons, Inc.--><path d="M144 80l0 96-96 0 0-96 96 0zM48 32C21.5 32 0 53.5 0 80l0 96c0 26.5 21.5 48 48 48l96 0c26.5 0 48-21.5 48-48l0-96c0-26.5-21.5-48-48-48L48 32zm96 304l0 96-96 0 0-96 96 0zM48 288c-26.5 0-48 21.5-48 48l0 96c0 26.5 21.5 48 48 48l96 0c26.5 0 48-21.5 48-48l0-96c0-26.5-21.5-48-48-48l-96 0zM304 80l96 0 0 96-96 0 0-96zm-48 0l0 96c0 26.5 21.5 48 48 48l96 0c26.5 0 48-21.5 48-48l0-96c0-26.5-21.5-48-48-48l-96 0c-26.5 0-48 21.5-48 48zM72 120l0 16c0 8.8 7.2 16 16 16l16 0c8.8 0 16-7.2 16-16l0-16c0-8.8-7.2-16-16-16l-16 0c-8.8 0-16 7.2-16 16zM88 360c-8.8 0-16 7.2-16 16l0 16c0 8.8 7.2 16 16 16l16 0c8.8 0 16-7.2 16-16l0-16c0-8.8-7.2-16-16-16l-16 0zM328 120l0 16c0 8.8 7.2 16 16 16l16 0c8.8 0 16-7.2 16-16l0-16c0-8.8-7.2-16-16-16l-16 0c-8.8 0-16 7.2-16 16zM256 304l0 160c0 8.8 7.2 16 16 16l32 0c8.8 0 16-7.2 16-16l0-64c0-8.8 7.2-16 16-16s16 7.2 16 16s7.2 16 16 16l64 0c8.8 0 16-7.2 16-16l0-96c0-8.8-7.2-16-16-16s-16 7.2-16 16s-7.2 16-16 16l-32 0c-8.8 0-16-7.2-16-16s-7.2-16-16-16l-64 0c-8.8 0-16 7.2-16 16zM368 448a16 16 0 1 0 0 32 16 16 0 1 0 0-32zm64 0a16 16 0 1 0 0 32 16 16 0 1 0 0-32z"/></svg>',
  'values' => $values,
  'includes' => [ 'bg', 'effects' ],
  'builder' => __NAMESPACE__ . '\controls',
  'render' => __NAMESPACE__ . '\render',
  'group' => 'content',
  'tss' => 'x_element_tss_layout_div',
  'options' => []
]);
