<?php

if ( ! defined( 'ABSPATH' ) ) exit;

if ( ! class_exists( 'WPGMP_Integration_Zapier' ) ) {

class WPGMP_Integration_Zapier {

	public function __construct() {
		add_filter( 'wpgmp_integrations_list', [ $this, 'register_extension' ] );
		add_filter( 'wpgmp_integration_nav_zapier', [ $this, 'register_nav_tabs' ] );
		add_action( 'wpgmp_render_integration_zapier_settings', [ $this, 'render_settings_tab' ] );
		add_action( 'wpgmp_render_integration_zapier_help', [ $this, 'render_help_tab' ] );
		add_action( 'wp_footer', [ $this, 'inject_tracking_script' ] );
		// existing hooks
		add_action( 'wp_ajax_wpgmp_zapier_proxy', [ $this, 'send_to_zapier' ] );
		add_action( 'wp_ajax_nopriv_wpgmp_zapier_proxy', [ $this, 'send_to_zapier' ] );
	}

	public function send_to_zapier() {
		$settings = maybe_unserialize( get_option( 'wpgmp_integrations_data' ) );
		$settings = $settings['zapier'] ?? [];
	
		if ( empty( $settings['zapier_webhook_url'] ) ) {
			wp_send_json_error('No Zapier URL configured');
		}
	
		$payload = [
			'map_id'       => sanitize_text_field($_POST['map_id'] ?? ''),
			'provider'     => sanitize_text_field($_POST['provider'] ?? ''),
			'marker_id'    => sanitize_text_field($_POST['marker_id'] ?? ''),
			'marker_title' => sanitize_text_field($_POST['marker_title'] ?? ''),
			'clicked_at'   => sanitize_text_field($_POST['clicked_at'] ?? ''),
		];
	
		wp_remote_post( esc_url_raw( $settings['zapier_webhook_url'] ), [
			'method'  => 'POST',
			'headers' => [ 'Content-Type' => 'application/json' ],
			'body'    => wp_json_encode( $payload ),
		]);
	
		wp_send_json_success('Sent to Zapier');
	}
	
	public function register_extension( $integrations ) {
		$integrations['zapier'] = [
			'title' => __( 'Zapier', 'wpgmp-google-map' ),
			'slug'  => 'zapier',
		];
		return $integrations;
	}

	public function register_nav_tabs( $tabs ) {
		return [
			'settings' => __( 'Settings', 'wpgmp-google-map' ),
			'help'     => __( 'Help', 'wpgmp-google-map' ),
		];
	}

	public function render_settings_tab() {
		echo '<div class="fc-box">';
		echo '<h3>' . esc_html__( 'Zapier Integration Settings', 'wpgmp-google-map' ) . '</h3>';
		echo '<p>' . esc_html__( 'Enter your Zapier webhook URL to receive marker click data.', 'wpgmp-google-map' ) . '</p>';

		$form_fields = [
			[
				'type'  => 'text',
				'name'  => 'zapier_webhook_url',
				'label' => __( 'Zapier Webhook URL', 'wpgmp-google-map' ),
				'desc'  => __( 'Paste your Zapier "Catch Hook" URL here.', 'wpgmp-google-map' ),
			],
			[
				'type'  => 'checkbox',
				'name'  => 'zapier_enable_marker_click',
				'label' => __( 'Enable Marker Click Tracking', 'wpgmp-google-map' ),
				'desc'  => __( 'Send marker_click event data to Zapier.', 'wpgmp-google-map' ),
			],
		];

		$form = new WPGMP_Integration_Form( 'zapier', $form_fields );
		$form->render_form();

		echo '</div>';
	}

	public function render_help_tab() {
		echo '<div class="fc-box">';
		echo '<h3>' . esc_html__( 'Zapier Help & Instructions', 'wpgmp-google-map' ) . '</h3>';
		echo '<p>' . esc_html__( '1. In Zapier, create a new Zap using "Webhooks by Zapier" → "Catch Hook"', 'wpgmp-google-map' ) . '</p>';
		echo '<p>' . esc_html__( '2. Copy the webhook URL provided by Zapier and paste it into the Webhook URL field here.', 'wpgmp-google-map' ) . '</p>';
		echo '<p>' . esc_html__( '3. Enable the marker click option and click a marker to test.', 'wpgmp-google-map' ) . '</p>';
		echo '<p><strong>Fields sent:</strong> map_id, marker_id, marker_title, provider, clicked_at</p>';
		echo '</div>';
	}

	public function inject_tracking_script() {
		$settings = maybe_unserialize( get_option( 'wpgmp_integrations_data' ) );
		$settings = $settings['zapier'] ?? [];

		$enabled = !empty( $settings['zapier_enable_marker_click'] );
		$webhook = esc_url_raw( $settings['zapier_webhook_url'] ?? '' );

		if ( ! $enabled || empty( $webhook ) ) return;

		?>
		<script>
			(function($) {
				$(document).on("wpgmp_marker_clicked", function(e, mapObj, place) {
					if (!mapObj || !place) return;

					const data = {
						action: 'wpgmp_zapier_proxy',
						map_id: mapObj.map_data?.map_property?.map_id || '',
						provider: mapObj.map_data?.provider || '',
						marker_id: place.id || '',
						marker_title: place.title || '',
						clicked_at: new Date().toISOString()
					};

					$.post("<?= admin_url('admin-ajax.php'); ?>", data);
				});
			})(jQuery);
		</script>
		<?php
	}
}

}
new WPGMP_Integration_Zapier();

