<?php
/**
 * Custom Functions
 */

/**
 * Check a plugin active
 *
 * @param $plugin_dir
 * @param $plugin_file
 *
 * @return bool
 */
function thim_plugin_active( $plugin_dir, $plugin_file = null ) {
	$plugin_file            = $plugin_file ? $plugin_file : ( $plugin_dir . '.php' );
	$plugin                 = $plugin_dir . '/' . $plugin_file;
	$active_plugins_network = get_site_option( 'active_sitewide_plugins' );

	if ( isset( $active_plugins_network[ $plugin ] ) ) {
		return true;
	}

	$active_plugins = get_option( 'active_plugins' );

	if ( in_array( $plugin, $active_plugins ) ) {
		return true;
	}

	return false;
}

/**
 * Get header layouts
 *
 * @return string CLASS for header layouts
 */
function thim_header_layout_class() {
    echo ' ' . get_theme_mod( 'header_style', 'layout-1' );

	if ( get_theme_mod( 'header_position', 'default' ) === 'default' ) {
		echo ' header-default';
	} else {
		echo ' header-overlay';
	}

	if ( get_theme_mod( 'show_sticky_menu', true ) ) {
		echo ' sticky-header';
	}

	if ( get_theme_mod( 'sticky_menu_style', 'same' ) === 'custom' ) {
		echo ' custom-sticky';
	} else {
		echo '';
	}

	if ( isset( $thim_options['header_retina_logo'] ) && get_theme_mod( 'header_retina_logo' ) ) {
		echo ' has-retina-logo';
	}
}

/**
 * Get Header Logo
 *
 * @return string
 */
if ( ! function_exists( 'thim_header_logo' ) ) {
	function thim_header_logo() {
		$thim_options         = get_theme_mods();
		$thim_logo_src        = THIM_URI . "assets/images/logo.png";
		$thim_mobile_logo_src = THIM_URI . "assets/images/logo.png";
		$thim_retina_logo_src = THIM_URI . "assets/images/logo.png";

		if ( isset( $thim_options['header_logo'] ) && $thim_options['header_logo'] <> '' ) {
			$thim_logo_src = get_theme_mod( 'header_logo' );
			if ( is_numeric( $thim_logo_src ) ) {
				$logo_attachment = wp_get_attachment_image_src( $thim_logo_src, 'full' );
				$thim_logo_src   = $logo_attachment[0];
			}
		}

		if ( isset( $thim_options['mobile_logo'] ) && $thim_options['mobile_logo'] <> '' ) {
			$thim_mobile_logo_src = get_theme_mod( 'mobile_logo' );
			if ( is_numeric( $thim_mobile_logo_src ) ) {
				$logo_attachment      = wp_get_attachment_image_src( $thim_mobile_logo_src, 'full' );
				$thim_mobile_logo_src = $logo_attachment[0];
			}
		}

		echo '<a class="no-sticky-logo" href="' . esc_url( home_url( '/' ) ) . '" title="' . esc_attr( get_bloginfo( 'name', 'display' ) ) . ' - ' . esc_attr( get_bloginfo( 'description' ) ) . '" rel="home">';
		echo '<img class="logo" src="' . esc_url( $thim_logo_src ) . '" alt="' . esc_attr( get_bloginfo( 'name', 'display' ) ) . '" />';

		if ( get_theme_mod( 'header_retina_logo', false ) ) {
			$thim_retina_logo_src = get_theme_mod( 'header_retina_logo' );
			if ( is_numeric( $thim_retina_logo_src ) ) {
				$logo_attachment      = wp_get_attachment_image_src( $thim_retina_logo_src, 'full' );
				$thim_retina_logo_src = $logo_attachment[0];
			}
		}
        echo '<img class="retina-logo" src="' . esc_url( $thim_retina_logo_src ) . '" alt="' . esc_attr( get_bloginfo( 'name', 'display' ) ) . '" />';
		echo '</a>';
	}
}
add_action( 'thim_header_logo', 'thim_header_logo' );

/**
 * Get Header Sticky logo
 *
 * @return string
 */
if ( ! function_exists( 'thim_header_sticky_logo' ) ) {
	function thim_header_sticky_logo() {
		if ( get_theme_mod( 'header_sticky_logo' ) != '' ) {
			$thim_logo_stick_logo     = get_theme_mod( 'header_sticky_logo' );
			$thim_logo_stick_logo_src = $thim_logo_stick_logo; // For the default value
			if ( is_numeric( $thim_logo_stick_logo ) ) {
				$logo_attachment = wp_get_attachment_image_src( $thim_logo_stick_logo, 'full' );
				if ( $logo_attachment ) {
					$thim_logo_stick_logo_src = $logo_attachment[0];
				} else {
					$thim_logo_stick_logo_src = THIM_URI . 'assets/images/sticky-logo.png';
				}
			}
			$thim_logo_size = @getimagesize( $thim_logo_stick_logo_src );
			$logo_size      = $thim_logo_size[3];
			$site_title     = esc_attr( get_bloginfo( 'name', 'display' ) );
			echo '<a href="' . esc_url( home_url( '/' ) ) . '" title="' . esc_attr( get_bloginfo( 'name', 'display' ) ) . ' - ' . esc_attr( get_bloginfo( 'description' ) ) . '" rel="home" class="sticky-logo">
					<img src="' . esc_url($thim_logo_stick_logo_src) . '" alt="' . esc_attr($site_title) . '" ' . $logo_size . ' /></a>';
		} else {
			echo '<a href="' . esc_url( home_url( '/' ) ) . '" title="' . esc_attr( get_bloginfo( 'name', 'display' ) ) . ' - ' . esc_attr( get_bloginfo( 'description' ) ) . '" rel="home" class="sticky-logo">' . esc_attr( get_bloginfo( 'name' ) ) . '</a>';
		}
	}
}
add_action( 'thim_header_sticky_logo', 'thim_header_sticky_logo' );

/**
 * Get Page Title Content For Single
 *
 * @return string HTML for Page title bar
 */
function thim_get_single_page_title_content() {
	$post_id = get_the_ID();

	if ( get_post_type( $post_id ) == 'post' ) {
		$categories = get_the_category();
	} elseif ( get_post_type( $post_id ) == 'attachment' ) {
		echo '<h2 class="title">' . esc_html__( 'Attachment', 'vividly' ) . '</h2>';

		return;
	} else {// Custom post type
		$categories = get_the_terms( $post_id, 'taxonomy' );
	}
	if ( ! empty( $categories ) ) {
		echo '<h2 class="title">' . esc_html( $categories[0]->name ) . '</h2>';
	}
}

/**
 * Get Page Title Content For Date Format
 *
 * @return string HTML for Page title bar
 */
function thim_get_page_title_date() {
	if ( is_year() ) {
		echo '<h2 class="title">' . esc_html__( 'Year', 'vividly' ) . '</h2>';
	} elseif ( is_month() ) {
		echo '<h2 class="title">' . esc_html__( 'Month', 'vividly' ) . '</h2>';
	} elseif ( is_day() ) {
		echo '<h2 class="title">' . esc_html__( 'Day', 'vividly' ) . '</h2>';
	}

	$date  = '';
	$day   = intval( get_query_var( 'day' ) );
	$month = intval( get_query_var( 'monthnum' ) );
	$year  = intval( get_query_var( 'year' ) );
	$m     = get_query_var( 'm' );

	if ( ! empty( $m ) ) {
		$year  = intval( substr( $m, 0, 4 ) );
		$month = intval( substr( $m, 4, 2 ) );
		$day   = substr( $m, 6, 2 );

		if ( strlen( $day ) > 1 ) {
			$day = intval( $day );
		} else {
			$day = 0;
		}
	}

	if ( $day > 0 ) {
		$date .= $day . ' ';
	}
	if ( $month > 0 ) {
		global $wp_locale;
		$date .= $wp_locale->get_month( $month ) . ' ';
	}
	$date .= $year;
	echo '<div class="description">' . esc_attr( $date ) . '</div>';
}

/**
 * Get Page Title Content
 *
 * @return string HTML for Page title bar
 */
if ( ! function_exists( 'thim_page_title_content' ) ) {
	function thim_page_title_content() {
		if ( is_front_page() ) {// Front page
			echo '<h2 class="title">' . get_bloginfo( 'name' ) . '</h2>';
			echo '<div class="description">' . get_bloginfo( 'description' ) . '</div>';
		} elseif ( is_home() ) {// Post page
			echo '<h2 class="title">' . esc_html__( 'Blog', 'vividly' ) . '</h2>';
			echo '<div class="description">' . get_bloginfo( 'description' ) . '</div>';
		} elseif ( is_page() ) {// Page
			echo '<h2 class="title">' . get_the_title() . '</h2>';
		} elseif ( is_single() ) {// Single
			thim_get_single_page_title_content();
		} elseif ( is_author() ) {// Author
			echo '<h2 class="title">' . esc_html__( 'Author', 'vividly' ) . '</h2>';
			echo '<div class="description">' . get_the_author() . '</div>';
		} elseif ( is_search() ) {// Search
			echo '<h2 class="title">' . esc_html__( 'Search', 'vividly' ) . '</h2>';
			echo '<div class="description">' . get_search_query() . '</div>';
		} elseif ( is_tag() ) {// Tag
			echo '<h2 class="title">' . esc_html__( 'Tag', 'vividly' ) . '</h2>';
			echo '<div class="description">' . single_tag_title( '', false ) . '</div>';
		} elseif ( is_category() ) {// Archive
			echo '<h2 class="title">' . esc_html__( 'Category', 'vividly' ) . '</h2>';
			echo '<div class="description">' . single_cat_title( '', false ) . '</div>';
		} elseif ( is_404() ) {
			echo '<h2 class="title">' . esc_html__( 'Page Not Found!', 'vividly' ) . '</h2>';
		} elseif ( is_date() ) {
			thim_get_page_title_date();
		}
	}
}
add_action( 'thim_page_title_content', 'thim_page_title_content' );

/**
 * Get breadcrumb for page
 *
 * @return string
 */
function thim_get_breadcrumb_items_other() {
	global $author;
	$userdata   = get_userdata( $author );
	$categories = get_the_category();
	if ( is_front_page() ) { // Do not display on the homepage
		return;
	}
	if ( is_home() ) {
		echo '<li itemprop="itemListElement" itemscope itemtype="http://schema.org/ListItem"><span itemprop="name" title="' . esc_attr( get_the_title() ) . '">' . esc_html__( 'Blog', 'vividly' ) . '</span></li>';
	} else if ( is_category() ) { // Category page
		echo '<li itemprop="itemListElement" itemscope itemtype="http://schema.org/ListItem"><span itemprop="name">' . esc_html( $categories[0]->cat_name ) . '</span></li>';
	} else if ( is_tag() ) {
		echo '<li itemprop="itemListElement" itemscope itemtype="http://schema.org/ListItem"><span itemprop="name" title="' . esc_attr( single_term_title( '', false ) ) . '">' . esc_html( single_term_title( '', false ) ) . '</span></li>';
	} else if ( is_year() ) {
		echo '<li itemprop="itemListElement" itemscope itemtype="http://schema.org/ListItem"><span itemprop="name" title="' . esc_attr( get_the_time( 'Y' ) ) . '">' . esc_html( get_the_time( 'Y' ) ) . ' ' . esc_html__( 'Archives', 'vividly' ) . '</span></li>';
	} else if ( is_author() ) { // Auhor archive
		echo '<li itemprop="itemListElement" itemscope itemtype="http://schema.org/ListItem"><span itemprop="name" title="' . esc_attr( $userdata->display_name ) . '">' . esc_attr__( 'Author', 'vividly' ) . ' ' . esc_html( $userdata->display_name ) . '</span></li>';
	} else if ( get_query_var( 'paged' ) ) {
		echo '<li itemprop="itemListElement" itemscope itemtype="http://schema.org/ListItem"><span itemprop="name" title="' . esc_attr__( 'Page', 'vividly' ) . ' ' . get_query_var( 'paged' ) . '">' . esc_html__( 'Page', 'vividly' ) . ' ' . esc_html( get_query_var( 'paged' ) ) . '</span></li>';
	} else if ( is_search() ) {
		echo '<li itemprop="itemListElement" itemscope itemtype="http://schema.org/ListItem"><span itemprop="name" title="' . esc_attr__( 'Search results for:', 'vividly' ) . ' ' . esc_attr( get_search_query() ) . '">' . esc_html__( 'Search results for:', 'vividly' ) . ' ' . esc_html( get_search_query() ) . '</span></li>';
	} elseif ( is_404() ) {
		echo '<li itemprop="itemListElement" itemscope itemtype="http://schema.org/ListItem"><span itemprop="name" title="' . esc_attr__( '404 Page', 'vividly' ) . '">' . esc_html__( '404 Page', 'vividly' ) . '</span></li>';
	}
}

/**
 * Get content breadcrumbs
 *
 * @return string
 */
if ( ! function_exists( 'thim_breadcrumbs' ) ) {
	function thim_breadcrumbs() {
		global $post;
		if ( is_front_page() ) { // Do not display on the homepage
			return;
		}
		$categories   = get_the_category();
		$thim_options = get_theme_mods();
		$icon         = '/';
		if ( isset( $thim_options['breadcrumb_icon'] ) ) {
			$icon = html_entity_decode( get_theme_mod( 'breadcrumb_icon' ) );
		}
		// Build the breadcrums
		echo '<ul itemprop="breadcrumb" itemscope itemtype="http://schema.org/BreadcrumbList" id="breadcrumbs" class="breadcrumbs">';
		echo '<li itemprop="itemListElement" itemscope itemtype="http://schema.org/ListItem"><a itemprop="item" href="' . esc_url( home_url() ) . '" title="' . esc_attr__( 'Home', 'vividly' ) . '"><span itemprop="name">' . esc_html__( 'Home', 'vividly' ) . '</span></a><span class="breadcrum-icon">' . ent2ncr( $icon ) . '</span></li>';
		if ( is_single() ) { // Single post (Only display the first category)
			if ( isset( $categories[0] ) ) {
				echo '<li itemprop="itemListElement" itemscope itemtype="http://schema.org/ListItem"><a itemprop="item" href="' . esc_url( get_category_link( $categories[0]->term_id ) ) . '" title="' . esc_attr( $categories[0]->cat_name ) . '"><span itemprop="name">' . esc_html( $categories[0]->cat_name ) . '</span></a>';
				if(get_the_title()) {
				    echo '<span class="breadcrum-icon">' . ent2ncr( $icon ) . '</span>';
                }
                echo '</li>';
			}
			echo '<li itemprop="itemListElement" itemscope itemtype="http://schema.org/ListItem"><span itemprop="name" title="' . esc_attr( get_the_title() ) . '">' . esc_html( get_the_title() ) . '</span></li>';
		} else if ( is_page() ) {
			// Standard page
			if ( $post->post_parent ) {
				$anc = get_post_ancestors( $post->ID );
				$anc = array_reverse( $anc );
				// Parent page loop
				foreach ( $anc as $ancestor ) {
					echo '<li itemprop="itemListElement" itemscope itemtype="http://schema.org/ListItem"><a itemprop="item" href="' . esc_url( get_permalink( $ancestor ) ) . '" title="' . esc_attr( get_the_title( $ancestor ) ) . '"><span itemprop="name">' . esc_html( get_the_title( $ancestor ) ) . '</span></a><span class="breadcrum-icon">' . ent2ncr( $icon ) . '</span></li>';
				}
			}
			// Current page
			echo '<li itemprop="itemListElement" itemscope itemtype="http://schema.org/ListItem"><span itemprop="name" title="' . esc_attr( get_the_title() ) . '"> ' . esc_html( get_the_title() ) . '</span></li>';
		} elseif ( is_day() ) {// Day archive
			// Year link
			echo '<li itemprop="itemListElement" itemscope itemtype="http://schema.org/ListItem"><a itemprop="item" href="' . esc_url( get_year_link( get_the_time( 'Y' ) ) ) . '" title="' . esc_attr( get_the_time( 'Y' ) ) . '"><span itemprop="name">' . esc_html( get_the_time( 'Y' ) ) . ' ' . esc_html__( 'Archives', 'vividly' ) . '</span></a><span class="breadcrum-icon">' . ent2ncr( $icon ) . '</span></li>';
			// Month link
			echo '<li itemprop="itemListElement" itemscope itemtype="http://schema.org/ListItem"><a itemprop="item" href="' . esc_url( get_month_link( get_the_time( 'Y' ), get_the_time( 'm' ) ) ) . '" title="' . esc_attr( get_the_time( 'M' ) ) . '"><span itemprop="name">' . esc_html( get_the_time( 'M' ) ) . ' ' . esc_html__( 'Archives', 'vividly' ) . '</span></a><span class="breadcrum-icon">' . ent2ncr( $icon ) . '</span></li>';
			// Day display
			echo '<li itemprop="itemListElement" itemscope itemtype="http://schema.org/ListItem"><span itemprop="name" title="' . esc_attr( get_the_time( 'jS' ) ) . '"> ' . esc_html( get_the_time( 'jS' ) ) . ' ' . esc_html( get_the_time( 'M' ) ) . ' ' . esc_html__( 'Archives', 'vividly' ) . '</span></li>';

		} else if ( is_month() ) {
			// Year link
			echo '<li itemprop="itemListElement" itemscope itemtype="http://schema.org/ListItem"><a itemprop="item" href="' . esc_url( get_year_link( get_the_time( 'Y' ) ) ) . '" title="' . esc_attr( get_the_time( 'Y' ) ) . '"><span itemprop="name">' . esc_html( get_the_time( 'Y' ) ) . ' ' . esc_html__( 'Archives', 'vividly' ) . '</span></a><span class="breadcrum-icon">' . ent2ncr( $icon ) . '</span></li>';
			echo '<li itemprop="itemListElement" itemscope itemtype="http://schema.org/ListItem"><span itemprop="name" title="' . esc_attr( get_the_time( 'M' ) ) . '">' . esc_html( get_the_time( 'M' ) ) . ' ' . esc_html__( 'Archives', 'vividly' ) . '</span></li>';
		}
		thim_get_breadcrumb_items_other();
		echo '</ul>';
	}
}

/**
 * Get list sidebars
 */
if ( ! function_exists( 'thim_get_list_sidebar' ) ) {
	function thim_get_list_sidebar() {
		global $wp_registered_sidebars;

		$sidebar_array = array();
		$dp_sidebars   = $wp_registered_sidebars;

		$sidebar_array[''] = esc_attr__( '-- Select Sidebar --', 'vividly' );

		foreach ( $dp_sidebars as $sidebar ) {
			$sidebar_array[ $sidebar['name'] ] = $sidebar['name'];
		}

		return $sidebar_array;
	}
}

/**
 * Turn on and get the back to top
 *
 * @return string HTML for the back to top
 */
if ( ! class_exists( 'thim_back_to_top' ) ) {
	function thim_back_to_top() {
		if ( get_theme_mod( 'feature_backtotop', true ) ) {
            get_template_part( 'templates/footer/back-to-top' );
		}
	}
}
add_action( 'thim_space_body', 'thim_back_to_top', 10 );


/**
 * Switch footer layout
 *
 * @return string HTML footer layout
 */
if ( ! function_exists( 'thim_footer_layout' ) ) {
	function thim_footer_layout() {
		$template_name = 'templates/footer/' . get_theme_mod( 'footer_style', 'layout-1' );
		get_template_part( $template_name );
	}
}

/**
 * Footer Widgets
 *
 * @return bool
 * @return string
 */
if ( ! function_exists( 'thim_footer_widgets' ) ) {
	function thim_footer_widgets() {
	    ?>
        <div class="thim-<?php echo get_theme_mod( 'footer_columns', 4 );?>-col">
            <?php for ( $i = 1; $i <= get_theme_mod( 'footer_columns', 4 ); $i ++ ): ?>
                <?php dynamic_sidebar( 'footer-sidebar-' . $i ); ?>
            <?php endfor; ?>
        </div>
		<?php
	}
}


/**
 * Footer Copyright bar
 *
 * @return bool
 * @return string
 */
if ( ! function_exists( 'thim_copyright_bar' ) ) {
	function thim_copyright_bar() {
		if ( get_theme_mod( 'copyright_bar', true ) ) : ?>
			<div class="copyright-text">
				<?php
				$copyright_text = get_theme_mod( 'copyright_text', '' );
				echo ent2ncr( $copyright_text );
				?>
			</div>
		<?php endif;
	}
}

/**
 * Footer menu
 *
 * @return bool
 * @return array
 */
if ( ! function_exists( 'thim_copyright_menu' ) ) {
	function thim_copyright_menu() {
		if ( get_theme_mod( 'copyright_menu', true ) ) :
			if ( has_nav_menu( 'copyright_menu' ) ) {
				wp_nav_menu( array(
					'theme_location' => 'copyright_menu',
					'container'      => false,
					'items_wrap'     => '<ul id="copyright-menu" class="list-inline">%3$s</ul>',
				) );
			}
		endif;
	}
}

/**
 * Theme Feature: Open Graph insert doctype
 *
 * @param $output
 */
if ( ! function_exists( 'thim_doctype_opengraph' ) ) {
	function thim_doctype_opengraph( $output ) {
		if ( get_theme_mod( 'feature_open_graph_meta', true ) ) {
			return $output . ' prefix="og: http://ogp.me/ns# fb: http://ogp.me/ns/fb#"';
		}
	}

	add_filter( 'language_attributes', 'thim_doctype_opengraph' );
}

/**
 * Theme Feature: Preload
 *
 * @return bool
 * @return string HTML for preload
 */
if ( ! function_exists( 'thim_preloading' ) ) {
	function thim_preloading() {
		$preloading = get_theme_mod( 'theme_feature_preloading', 'off' );
		if ( $preloading != 'off' ) {

			echo '<div id="thim-preloading">';

			switch ( $preloading ) {
				case 'custom-image':
					$preloading_image = get_theme_mod( 'theme_feature_preloading_custom_image', false );
					if ( $preloading_image ) {
						if ( locate_template( 'templates/features/preloading/' . $preloading . '.php' ) ) {
							include locate_template( 'templates/features/preloading/' . $preloading . '.php' );
						}
					}
					break;
				default:
					if ( locate_template( 'templates/features/preloading/' . $preloading . '.php' ) ) {
						include locate_template( 'templates/features/preloading/' . $preloading . '.php' );
					}
					break;
			}

			echo '</div>';

		}
	}

	add_action( 'thim_before_body', 'thim_preloading', 10 );
}

/**
 * Theme Feature: Open Graph meta tag
 *
 * @param string
 */
if ( ! function_exists( 'thim_add_opengraph' ) ) {
	function thim_add_opengraph() {
		global $post;

		if ( get_theme_mod( 'feature_open_graph_meta', true ) ) {
			if ( is_single() ) {
				if ( has_post_thumbnail( $post->ID ) ) {
					$img_src = wp_get_attachment_image_src( get_post_thumbnail_id( $post->ID ), 'medium' );
					$img_src = esc_attr( $img_src[0] );
				} else {
					$img_src = THIM_URI . 'assets/images/opengraph.png';
				}
				if ( $excerpt = $post->post_excerpt ) {
					$excerpt = strip_tags( $post->post_excerpt );
					$excerpt = str_replace( "", "'", $excerpt );
				} else {
					$excerpt = get_bloginfo( 'description' );
				}
				?>

				<meta property="og:title" content="<?php echo the_title(); ?>" />
				<meta property="og:description" content="<?php echo esc_attr( $excerpt ); ?>" />
				<meta property="og:type" content="article" />
				<meta property="og:url" content="<?php echo the_permalink(); ?>" />
				<meta property="og:site_name" content="<?php echo get_bloginfo(); ?>" />
				<meta property="og:image" content="<?php echo esc_attr( $img_src ); ?>" />

				<?php
			} else {
				return;
			}
		}
	}

	add_action( 'wp_head', 'thim_add_opengraph', 10 );
}


/**
 * Theme Feature: Google theme color
 */
if ( ! function_exists( 'thim_google_theme_color' ) ) {
	function thim_google_theme_color() {
		if ( get_theme_mod( 'feature_google_theme', false ) ) { ?>
			<meta name="theme-color" content="<?php echo esc_attr( get_theme_mod( 'feature_google_theme_color', '#333333' ) ) ?>">
			<?php
		}
	}

	add_action( 'wp_head', 'thim_google_theme_color', 10 );
}

/**
 * Responsive: enable or disable responsive
 *
 * @return string
 * @return bool
 */
if ( ! function_exists( 'thim_enable_responsive' ) ) {
	function thim_enable_responsive() {
		if ( get_theme_mod( 'enable_responsive', true ) ) {
			echo '<meta name="viewport" content="width=device-width, initial-scale=1">';
		}
	}

	add_action( 'wp_head', 'thim_enable_responsive', 1 );
}


/**
 *
 * Display Topbar
 *
 * @return void
 *
 */
if ( ! function_exists( 'thim_topbar' ) ) {
	function thim_topbar() {
		$display = get_theme_mod( 'header_topbar_display', false );
		$style   = get_theme_mod( 'header_position', 'default' );
		if ( $display ) {
			echo '<div id="thim-header-topbar" class="style-' . $style . '">';
			get_template_part( 'templates/header/topbar' );
			echo '</div>';
		}
	}

	add_action( 'thim_topbar', 'thim_topbar', 10 );
}


/**
 * Override ajax-loader contact form
 *
 * $return mixed
 */

function thim_wpcf7_ajax_loader() {
	return THIM_URI . 'assets/images/icons/ajax-loader.gif';
}

add_filter( 'wpcf7_ajax_loader', 'thim_wpcf7_ajax_loader' );




/**
 * Get feature image
 *
 * @param int  $width
 * @param int  $height
 * @param bool $link
 *
 * @return string
 */
function thim_feature_image( $width = 1024, $height = 768, $link = true ) {
	global $post;
	if ( has_post_thumbnail() ) {
		if ( $link != true && $link != false ) {
			the_post_thumbnail( $post->ID, $link );
		} else {
            $src   = wp_get_attachment_image_src( get_post_thumbnail_id( $post->ID ), 'full' );
			if ( $src ) {
				$width_data    = $src[1];
				$height_data   = $src[2];
				if ( $link ) {
					if ( ( $width_data < $width ) || ( $height_data < $height ) ) {
						echo '<div class="thumbnail"><a href="' . esc_url( get_permalink() ) . '" title = "' . get_the_title() . '">';
						echo '<img src="' . esc_url($src[0]) . '" alt= "' . get_the_title() . '" title = "' . get_the_title() . '" />';
						echo '</a></div>';
					} else {
						$image_crop = thim_aq_resize( $src[0], $width, $height, true );
						echo '<div class="thumbnail"><a href="' . esc_url( get_permalink() ) . '" title = "' . get_the_title() . '">';
						echo '<img src="' . esc_url($image_crop) . '" alt= "' . get_the_title() . '" title = "' . get_the_title() . '" />';
						echo '</a></div>';
					}
				} else {
					if ( ( $width_data < $width ) || ( $height_data < $height ) ) {
						return '<img src="' . esc_url($src[0]) . '" alt= "' . get_the_title() . '" title = "' . get_the_title() . '" />';
					} else {
						$image_crop = thim_aq_resize( $src[0], $width, $height, true );

						return '<img src="' . esc_url($image_crop) . '" alt= "' . get_the_title() . '" title = "' . get_the_title() . '" />';
					}
				}
			}
		}
	}
}

/**
 * aq_resize function fake.
 * Aq_Resize
 */
if ( ! function_exists( 'thim_aq_resize' ) ) {
    function thim_aq_resize( $url, $width = null, $height = null, $crop = null, $single = true, $upscale = false ) {
        /* WPML Fix */
        if ( defined( 'ICL_SITEPRESS_VERSION' ) ) {
            global $sitepress;
            $url = $sitepress->convert_url( $url, $sitepress->get_default_language() );
        }
        /* WPML Fix */
        $aq_resize = Aq_Resize::getInstance();

        return $aq_resize->process( $url, $width, $height, $crop, $single, $upscale );
    }
}


/**
 * Get template.
 *
 * Search for the template and include the file.
 *
 * @since 1.0.0
 *
 * @see   wcpt_locate_template()
 *
 * @param string $template_name Template to load.
 * @param array  $args          Args passed for the template file.
 * @param string $string        $template_path    Path to templates.
 * @param string $default_path  Default path to template files.
 */
function thim_get_template( $template_name, $args = array(), $tempate_path = '', $default_path = '' ) {
	if ( is_array( $args ) && isset( $args ) ) :
		extract( $args );
	endif;

	$template_name = $template_name . '.php';
	$posts         = isset( $args['posts'] ) ? $args['posts'] : array();
	$params        = isset( $args['params'] ) ? $args['params'] : array();

	$template_file = thim_locate_template( $template_name, $tempate_path, $default_path );

	if ( ! file_exists( $template_file ) ) :
		_doing_it_wrong( __FUNCTION__, sprintf( '<code>%s</code> does not exist.', $template_file ), '1.0.0' );

		return;
	endif;

	include $template_file;
}

if ( ! function_exists( 'thim_entry_top_archive' ) ) {
    function thim_entry_top_archive() {
        $size = 0;
        $html = '';
        switch ( get_post_format() ) {
            case 'image':
                $image = thim_get_image( array(
                    'size'     => $size,
                    'format'   => 'src',
                    'meta_key' => 'thim_image',
                    'echo'     => false,
                ) );
                if ( ! $image ) {
                    break;
                }

                $html = sprintf(
                    '<a class="post-image" href="%1$s" title="%2$s"><img src="%3$s" alt="%2$s"></a>',
                    esc_url( get_permalink() ),
                    esc_attr( the_title_attribute( 'echo=0' ) ),
                    $image
                );
                break;
            case 'audio':
                $audio = thim_meta( 'thim_audio' );
                if ( ! $audio ) {
                    break;
                }
                // If URL: show oEmbed HTML or jPlayer
                if ( filter_var( $audio, FILTER_VALIDATE_URL ) ) {
                    wp_enqueue_style( 'thim-pixel-industry' );
                    wp_enqueue_script( 'thim-jplayer' );
                    // Try oEmbed first
                    if ( $oembed = @wp_oembed_get( $audio ) ) {
                        $html .= $oembed;
                    } // Use jPlayer
                    else {
                        $id   = uniqid();
                        $html .= "<div data-player='$id' class='jp-jplayer' data-audio='$audio'></div>";
                        $html .= thim_jplayer( $id );
                    }
                } // If embed code: just display
                else {
                    $html .= $audio;
                }
                break;
            case 'video':

                $video = thim_meta( 'thim_video' );
                if ( ! $video ) {
                    break;
                }
                // If URL: show oEmbed HTML
                if ( filter_var( $video, FILTER_VALIDATE_URL ) ) {
                    if ( $oembed = @wp_oembed_get( $video ) ) {
                        $html .= $oembed;
                    }
                } // If embed code: just display
                else {
                    $html .= $video;
                }
                break;
            default:
                $thumb = get_the_post_thumbnail( get_the_ID(), 'full' );
                if ( empty( $thumb ) ) {
                    return;
                }
                $html .= '<a class="post-image" href="' . esc_url( get_permalink() ) . '">';
                $html .= $thumb;
                $html .= '</a>';
        }
        if ( $html ) {
            echo "<div class='post-formats-wrapper'>$html</div>";
        }
    }
}
/**
 * Locate template.
 *
 * Locate the called template.
 * Search Order:
 * 1. /themes/theme/woocommerce-plugin-templates/$template_name
 * 2. /themes/theme/$template_name
 * 3. /plugins/woocommerce-plugin-templates/templates/$template_name.
 *
 * @since 1.0.0
 *
 * @param    string $template_name Template to load.
 * @param    string $string        $template_path    Path to templates.
 * @param    string $default_path  Default path to template files.
 *
 * @return    string                            Path to the template file.
 */
function thim_locate_template( $template_name, $template_path = '', $default_path = '' ) {
	// Set variable to search in woocommerce-plugin-templates folder of theme.
	if ( ! $template_path ) :
		$template_path = 'templates/';
	endif;

	// Set default plugin templates path.
	if ( ! $default_path ) :
		$default_path = THIM_MAGWP_PATH . $template_path; // Path to the template folder
	endif;

	// Search template file in theme folder.
	$template = locate_template( array(
		$template_path . $template_name,
		$template_name
	) );

	// Get plugins template file.
	if ( ! $template ) :
		$template = $default_path . $template_name;
	endif;

	return apply_filters( 'thim_locate_template', $template, $template_name, $template_path, $default_path );
}


/**
 * Get post meta
 *
 * @param $key
 * @param $args
 * @param $post_id
 *
 * @return string
 * @return bool
 */
if ( ! function_exists( 'thim_meta' ) ) {
    function thim_meta( $key, $args = array(), $post_id = null ) {
        $post_id = empty( $post_id ) ? get_the_ID() : $post_id;

        $args = wp_parse_args( $args, array(
            'type' => 'text',
        ) );

        // Image
        if ( in_array( $args['type'], array( 'image' ) ) ) {
            if ( isset( $args['single'] ) && $args['single'] == "false" ) {
                // Gallery
                $temp          = array();
                $data          = array();
                $attachment_id = get_post_meta( $post_id, $key, false );
                if ( ! $attachment_id ) {
                    return $data;
                }

                if ( empty( $attachment_id ) ) {
                    return $data;
                }
                foreach ( $attachment_id as $k => $v ) {
                    $image_attributes = wp_get_attachment_image_src( $v, $args['size'] );
                    $temp['url']      = $image_attributes[0];
                    $data[]           = $temp;
                }

                return $data;
            } else {
                // Single Image
                $attachment_id    = get_post_meta( $post_id, $key, true );
                $image_attributes = wp_get_attachment_image_src( $attachment_id, $args['size'] );

                return $image_attributes;
            }
        }

        return get_post_meta( $post_id, $key, $args );
    }
}


/**
 * Get image features
 *
 * @param $args
 *
 * @return array|void
 */
if ( ! function_exists( 'thim_get_image' ) ) {
    function thim_get_image( $args = array() ) {
        $default = apply_filters( 'thim_get_image_default_args', array(
            'post_id'  => get_the_ID(),
            'size'     => 'thumbnail',
            'format'   => 'html', // html or src
            'attr'     => '',
            'meta_key' => '',
            'scan'     => true,
            'default'  => '',
            'echo'     => true,
        ) );

        $args = wp_parse_args( $args, $default );

        if ( ! $args['post_id'] ) {
            $args['post_id'] = get_the_ID();
        }

        // Get image from cache
        $key         = md5( serialize( $args ) );
        $image_cache = wp_cache_get( $args['post_id'], 'thim_get_image' );

        if ( ! is_array( $image_cache ) ) {
            $image_cache = array();
        }

        if ( empty( $image_cache[ $key ] ) ) {
            // Get post thumbnail
            if ( has_post_thumbnail( $args['post_id'] ) ) {
                $id   = get_post_thumbnail_id();
                $html = wp_get_attachment_image( $id, $args['size'], false, $args['attr'] );
                list( $src ) = wp_get_attachment_image_src( $id, $args['size'], false, $args['attr'] );
            }

            // Get the first image in the custom field
            if ( ! isset( $html, $src ) && $args['meta_key'] ) {
                $id = get_post_meta( $args['post_id'], $args['meta_key'], true );

                // Check if this post has attached images
                if ( $id ) {
                    $html = wp_get_attachment_image( $id, $args['size'], false, $args['attr'] );
                    list( $src ) = wp_get_attachment_image_src( $id, $args['size'], false, $args['attr'] );
                }
            }

            // Get the first attached image
            if ( ! isset( $html, $src ) ) {
                $image_ids = array_keys( get_children( array(
                    'post_parent'    => $args['post_id'],
                    'post_type'      => 'attachment',
                    'post_mime_type' => 'image',
                    'orderby'        => 'menu_order',
                    'order'          => 'ASC',
                ) ) );

                // Check if this post has attached images
                if ( ! empty( $image_ids ) ) {
                    $id   = $image_ids[0];
                    $html = wp_get_attachment_image( $id, $args['size'], false, $args['attr'] );
                    list( $src ) = wp_get_attachment_image_src( $id, $args['size'], false, $args['attr'] );
                }
            }

            // Get the first image in the post content
            if ( ! isset( $html, $src ) && ( $args['scan'] ) ) {
                preg_match( '|<img.*?src=[\'"](.*?)[\'"].*?>|i', get_post_field( 'post_content', $args['post_id'] ), $matches );

                if ( ! empty( $matches ) ) {
                    $html = $matches[0];
                    $src  = $matches[1];
                }
            }

            // Use default when nothing found
            if ( ! isset( $html, $src ) && ! empty( $args['default'] ) ) {
                if ( is_array( $args['default'] ) ) {
                    $html = $args['html'];
                    $src  = $args['src'];
                } else {
                    $html = $src = $args['default'];
                }
            }

            // Still no images found?
            if ( ! isset( $html, $src ) ) {
                return false;
            }

            $output = 'html' === strtolower( $args['format'] ) ? $html : $src;

            $image_cache[ $key ] = $output;
            wp_cache_set( $args['post_id'], $image_cache, 'thim_get_image' );
        } // If image already cached
        else {
            $output = $image_cache[ $key ];
        }

        $output = apply_filters( 'thim_get_image', $output, $args );

        if ( ! $args['echo'] ) {
            return $output;
        }

        echo ent2ncr( $output );
    }
}

/**
 * Show entry format images, video, gallery, audio, etc.
 *
 * @return void
 */
if ( ! function_exists( 'thim_post_formats' ) ):
    function thim_post_formats( $size ) {
        $html = '';
        switch ( get_post_format() ) {
            case 'image':
                $image = thim_get_image( array(
                    'size'     => $size,
                    'format'   => 'src',
                    'meta_key' => 'thim_image',
                    'echo'     => false,
                ) );
                if ( ! $image ) {
                    break;
                }

                $html = sprintf( '<a class="post-image" href="%1$s" title="%2$s"><img src="%3$s" alt="%2$s"></a>', esc_url( get_permalink() ), esc_attr( the_title_attribute( 'echo=0' ) ), $image );
                break;
            case 'gallery':
                $images = thim_meta( 'thim_gallery', "type=image&single=false&size=$size" );
                $thumbs = thim_meta( 'thim_gallery', "type=image&single=false&size=thumbnail" );
                if ( empty( $images ) ) {
                    if( has_post_thumbnail() ) $html .= get_the_post_thumbnail(get_the_ID(), 'full');
                    break;
                }
                $html .= '<div class="slide-images js-call-slick-col" data-numofshow="1" data-numofscroll="1" data-loopslide="1" data-autoscroll="0" data-speedauto="6000" data-responsive="[1, 1], [1, 1], [1, 1], [1, 1], [1, 1]">';
                $html .= '<div class="wrap-arrow-slick">';
                $html .= '<div class="arow-slick prev-slick">';
                $html .= '<i class="ion ion-ios-arrow-left"></i>';
                $html  .='</div>';

                $html .= '<div class="arow-slick next-slick">';
                $html .= '<i class="ion ion-ios-arrow-right"></i>';
                $html .= '</div>';
                $html .= '</div>';
                $html .= '<div class="slide-slick">';
                foreach ( $images as $key => $image ) {

                    if ( ! empty( $image['url'] ) ) {
                        $html .= sprintf(
                            '<div class="item-slick" data-thumb="%s"><div class="item-img"><img src="%s" alt="' . esc_attr__( 'Gallery', 'vividly' ) . '"></div></div>',
                            esc_url( $thumbs[ $key ]['url'] ),
                            esc_url( get_permalink() ),
                            esc_url( $image['url'] )
                        );
                    }
                }
                $html .= '</div>';
                $html .= '<div class="wrap-dot-slick"></div>';
                $html .= '</div>';
                break;
            case 'audio':
                $audio = thim_meta( 'thim_audio' );
                if ( ! $audio ) {
                    break;
                }
                // If URL: show oEmbed HTML or jPlayer
                if ( filter_var( $audio, FILTER_VALIDATE_URL ) ) {
                    //jsplayer
                    wp_enqueue_style( 'thim-pixel-industry', THIM_CORE_ADMIN_URI . '/assets/js/jplayer/skin/pixel-industry/pixel-industry.css' );
                    wp_enqueue_script( 'thim-jplayer', THIM_CORE_ADMIN_URI . '/assets/js/jplayer/jquery.jplayer.min.js', array( 'jquery' ), '', true );

                    // Try oEmbed first
                    if ( $oembed = @wp_oembed_get( $audio ) ) {
                        $html .= $oembed;
                    } // Use jPlayer
                    else {
                        $id   = uniqid();
                        $html .= "<div data-player='$id' class='jp-jplayer' data-audio='$audio'></div>";
                        $html .= thim_jplayer( $id );
                    }
                } // If embed code: just display
                else {
                    $html .= $audio;
                }
                break;
            case 'video':

                $video = thim_meta( 'thim_video' );
                if ( ! $video ) {
                    break;
                }
                // If URL: show oEmbed HTML
                if ( filter_var( $video, FILTER_VALIDATE_URL ) ) {
                    if ( $oembed = @wp_oembed_get( $video ) ) {
                        $html .= $oembed;
                    }
                } // If embed code: just display
                else {
                    $html .= $video;
                }
                break;
            default:
                $thumb = get_the_post_thumbnail( get_the_ID(), $size );
                if ( empty( $thumb ) ) {
                    return;
                }
                $html .= '<a class="post-image" href="' . esc_url( get_permalink() ) . '">';
                $html .= $thumb;
                $html .= '</a>';
        }
        if ( $html ) {
            echo "<div class='post-formats-wrapper'>$html</div>";
        }
    }
endif;
add_action( 'thim_entry_top', 'thim_post_formats', 10 );

//Edit Widget Categories
add_filter( 'wp_list_categories', 'thim_add_span_cat_count' );
function thim_add_span_cat_count( $links ) {
    $links = str_replace( '<span class="count">(', '<span class="count">', $links );
    $links = str_replace( ')</span>', '</span>', $links );


    $links = str_replace( '(', '<span class="count">', $links );
    $links = str_replace( ')', '</span>', $links );

    return $links;
}

// remove input website in comment form
add_filter('comment_form_default_fields', 'thim_remove_website');
function thim_remove_website($fields)
{
    if(isset($fields['url']))
        unset($fields['url']);
    return $fields;
}

// add custom class to tag
function thim_add_class_the_tags($html){
    $postid = get_the_ID();
    $html = str_replace('<a','<a class="tag-item"',$html);
    return $html;
}
add_filter('the_tags','thim_add_class_the_tags');

function thim_modify_comment_form_fields( $fields ) {
    $commenter = wp_get_current_commenter();
    $req       = get_option( 'require_name_email' );
    $aria_req  = ( $req ? " aria-required='true'" : '' );

    $fields['author'] = '<div class="comment-meta"><div class="row"><div class="col-md-6"><p class="comment-form-author">' .
        '<input placeholder="' . esc_attr__( 'Your Name *', 'vividly' ) . '" id="author" name="author" type="text" value="' . esc_attr( $commenter['comment_author'] ) .
        '" size="30"' . $aria_req . ' /></p></div>';
    $fields['email']  = '<div class="col-md-6"><p class="comment-form-email">' .
        '<input placeholder="' . esc_attr__( 'Email *', 'vividly' ) . '" id="email" name="email" type="text" value="' . esc_attr( $commenter['comment_author_email'] ) .
        '" size="30"' . $aria_req . ' /></p></div></div></div>';
    $fields['url']    = '';


    return $fields;
}

add_filter( 'comment_form_default_fields', 'thim_modify_comment_form_fields' );

/*
 *    navigate blog next
 * */
function thim_navigate_next_blog() {
        global $next_post;
        $next_post = get_next_post();
        if (!empty( $next_post )):
            ?>
            <div class="navi-item next-blog">
                <div class="navi-text">
                    <div class="name-navi">
                        <?php echo esc_html__('Next','vividly');?>
                    </div>

                    <div class="title-navi">
                        <a href="<?php echo esc_url(get_permalink($next_post->ID)); ?>">
                            <?php echo esc_html($next_post->post_title); ?>
                        </a>
                    </div>

                    <div class="info-navi">
                        <?php echo esc_html(get_the_date('F j, Y',$next_post->ID)); ?>
                    </div>
                </div>

                <a href="<?php echo esc_url(get_permalink($next_post->ID)); ?>" class="navi-arrow">
                    <i class="ion ion-ios-arrow-thin-right"></i>
                </a>
            </div>
        <?php
        endif;
}
add_action('thim_navigate_next_blog','thim_navigate_next_blog');
/*
 *    navigate blog preview
 * */
function thim_navigate_preview_blog() {
    global $prev_post;
    $prev_post = get_previous_post();

        if (!empty( $prev_post )):
            ?>
            <div class="navi-item prev-blog">
                <a href="<?php echo esc_url(get_permalink($prev_post->ID)) ?>" class="navi-arrow">
                    <i class="ion ion-ios-arrow-thin-left"></i>
                </a>

                <div class="navi-text">
                    <div class="name-navi">
                        <?php echo esc_html__('Previous','vividly');?>
                    </div>

                    <div class="title-navi">
                        <a href="<?php echo esc_url(get_permalink($prev_post->ID)) ?>">
                            <?php echo esc_html($prev_post->post_title); ?>
                        </a>
                    </div>

                    <div class="info-navi">
                        <?php echo esc_html(get_the_date('F j, Y',$prev_post->ID)); ?>
                    </div>
                </div>
            </div>
        <?php
        endif;
}
add_action('thim_navigate_preview_blog','thim_navigate_preview_blog');

if( !function_exists('thim_get_list_authors') ) {
    function thim_get_list_authors() {
        $args = array(
            'role'         => 'author',
        );
        $users = get_users( $args );
        return $users;
    }
}

/**
 * Check import demo data page-builder
 */
add_action( 'wp_ajax_thim_update_theme_mods', 'thim_import_demo_page_builder' );
if ( ! function_exists( 'thim_import_demo_page_builder' ) ) {
    function thim_import_demo_page_builder() {
        $thim_key   = sanitize_text_field( $_POST["thim_key"] );
        $thim_value = sanitize_text_field( $_POST["thim_value"] );
        if ( ! is_multisite() ) {
            $active_plugins = get_option( 'active_plugins' );

            if ( $thim_value == 'visual_composer' ) {
                if ( ( $key = array_search( 'siteorigin-panels/siteorigin-panels.php', $active_plugins ) ) !== false ) {
                    unset( $active_plugins[ $key ] );
                }
                if ( ( $key = array_search( 'elementor/elementor.php', $active_plugins ) ) !== false ) {
                    unset( $active_plugins[ $key ] );
                }
                if ( ( $key = array_search( 'anywhere-elementor/anywhere-elementor.php', $active_plugins ) ) !== false ) {
                    unset( $active_plugins[ $key ] );
                }
                if ( ! in_array( 'js_composer/js_composer.php', $active_plugins ) ) {
                    $active_plugins[] = 'js_composer/js_composer.php';
                }

            } else if ( $thim_value == 'site_origin' ) {
                if ( ( $key = array_search( 'js_composer/js_composer.php', $active_plugins ) ) !== false ) {
                    unset( $active_plugins[ $key ] );
                }
                if ( ( $key = array_search( 'elementor/elementor.php', $active_plugins ) ) !== false ) {
                    unset( $active_plugins[ $key ] );
                }
                if ( ( $key = array_search( 'anywhere-elementor/anywhere-elementor.php', $active_plugins ) ) !== false ) {
                    unset( $active_plugins[ $key ] );
                }
                if ( ! in_array( 'siteorigin-panels/siteorigin-panels.php', $active_plugins ) ) {
                    $active_plugins[] = 'siteorigin-panels/siteorigin-panels.php';
                }

            } else if ( $thim_value == 'elementor' ) {
                if ( ( $key = array_search( 'js_composer/js_composer.php', $active_plugins ) ) !== false ) {
                    unset( $active_plugins[ $key ] );
                }
                if ( ( $key = array_search( 'siteorigin-panels/siteorigin-panels.php', $active_plugins ) ) !== false ) {
                    unset( $active_plugins[ $key ] );
                }
                if ( ! in_array( 'elementor/elementor.php', $active_plugins ) ) {
                    $active_plugins[] = 'elementor/elementor.php';
                }
                if ( ! in_array( 'anywhere-elementor/anywhere-elementor.php', $active_plugins ) ) {
                    $active_plugins[] = 'anywhere-elementor/anywhere-elementor.php';
                }

            }
            update_option( 'active_plugins', $active_plugins );
        }

        if ( empty( $thim_key ) || empty( $thim_value ) ) {
            $output = 'update fail';
        } else {
            set_theme_mod( $thim_key, $thim_value );
            $output = 'update success';
        }

        echo ent2ncr( $output );
        die();
    }
}

/**
 * Filter redirect plugin tp chameleon
 */
if ( ! function_exists( 'thim_tp_chameleon_redirect' ) ) {
    function thim_tp_chameleon_redirect( $option ) {
        if ( ( ! is_admin() && ! is_home() && ! is_front_page() ) || is_customize_preview() ) {
            return false;
        } else {
            return $option;
        }
    }
}
add_filter( 'tp_chameleon_redirect_iframe', 'thim_tp_chameleon_redirect' );

/**
 * Filter image all-demo tp-chameleon
 */
if ( ! function_exists( 'thim_override_demo_image_tp_chameleon' ) ) {
    function thim_override_demo_image_tp_chameleon() {
        return THIM_URI . 'assets/images/VideoBlog_chameleon.png';
    }
}
add_filter( 'tp_chameleon_get_image_sprite_demos', 'thim_override_demo_image_tp_chameleon' );

/**
 * Filter demos path
 */
function thim_filter_site_demos( $demo_datas ) {
    $demo_data_file_path = get_template_directory() . '/inc/data/demos.php';
    if ( is_file( $demo_data_file_path ) ) {
        require $demo_data_file_path;
    }

    return $demo_datas;
}

add_filter( 'tp_chameleon_get_site_demos', 'thim_filter_site_demos' );

/**
 * List child themes.
 *
 * @return array
 */
function thim_video_blog_list_child_themes() {
    return array(
        'vividly-child'              => array(
            'name'       => 'Vividly Child',
            'slug'       => 'vividly-child',
            'screenshot' => 'https://thimpresswp.github.io/demo-data/video-blog/child-themes/vividly-child.png',
            'source'     => 'https://thimpresswp.github.io/demo-data/video-blog/child-themes/vividly-child.zip',
            'version'    => '1.0.0'
        ),
    );
}
add_filter( 'thim_core_list_child_themes', 'thim_video_blog_list_child_themes' );

//Add info for Dashboard Admin
if ( ! function_exists( 'thim_links_guide_user' ) ) {
    function thim_links_guide_user() {
        return array(
            'docs'      => 'http://docspress.thimpress.com/vividly/',
            'support'   => 'https://thimpress.com/forums/forum/vividly/',
            'knowledge' => 'https://thimpress.com/knowledge-base/',
        );
    }
}
add_filter( 'thim_theme_links_guide_user', 'thim_links_guide_user' );

/**
 * Link purchase theme.
 */
if ( ! function_exists( 'thim_link_purchase' ) ) {
    function thim_link_purchase() {
        return 'https://themeforest.net/item/vividly-video-blog-personal-vlog-wordpress-theme/23168294?ref=thimpress&utm_source=thimcoredashboard&utm_medium=dashboard';
    }
}
add_filter( 'thim_envato_link_purchase', 'thim_link_purchase' );

/**
 * Envato id.
 */
if ( ! function_exists( 'thim_envato_item_id' ) ) {
    function thim_envato_item_id() {
        return '23168294';
    }
}
add_filter( 'thim_envato_item_id', 'thim_envato_item_id' );
