<?php

function boxcar_get_listings( $params = array() ) {
	$params = wp_parse_args( $params, array(
		'limit' => -1,
		'post_status' => 'publish',
		'get_listings_by' => 'recent',
		'orderby' => '',
		'order' => '',
		'post__in' => array(),
		'fields' => null, // ids
		'author' => null,
		'type' => array(), 'category' => array(), 'condition' => array(), 'make' => array(), 'model' => array()
	));
	extract($params);

	$query_args = array(
		'post_type'         => 'listing',
		'posts_per_page'    => $limit,
		'post_status'       => $post_status,
		'orderby'       => $orderby,
		'order'       => $order,
	);

	$meta_query = array();
	switch ($get_listings_by) {
		case 'recent':
			$query_args['orderby'] = 'date';
			$query_args['order'] = 'DESC';
			break;
		case 'featured':
			$meta_query[] = array(
				'key' => WP_CARDEALER_LISTING_PREFIX.'featured',
	           	'value' => 'on',
	           	'compare' => '=',
			);
			break;
		case 'popular':
			$query_args['meta_key'] = WP_CARDEALER_LISTING_PREFIX.'views';
			$query_args['order'] = 'DESC';
			$query_args['orderby'] = 'meta_value_num';
			break;
	}

	if ( !empty($post__in) ) {
    	$query_args['post__in'] = $post__in;
    }

    if ( !empty($fields) ) {
    	$query_args['fields'] = $fields;
    }

    if ( !empty($author) ) {
    	$query_args['author'] = $author;
    }

    $tax_query = array();

    $tax_keys = apply_filters('boxcar-listing-get-tax-keys', array('type', 'category', 'condition', 'make', 'model'));
    foreach ($tax_keys as $tax_key) {
    	if ( !empty($params[$tax_key]) ) {
    		$tax_query[] = array(
	            'taxonomy'      => 'listing_'.$tax_key,
	            'field'         => 'slug',
	            'terms'         => $params[$tax_key],
	            'operator'      => 'IN'
	        );
    	}
    }
    
    if ( !empty($tax_query) ) {
    	$query_args['tax_query'] = $tax_query;
    }
    
    if ( !empty($meta_query) ) {
    	$query_args['meta_query'] = $meta_query;
    }
    
	return new WP_Query( $query_args );
}

function boxcar_listing_template_folder_name($folder) {
	$folder = 'template-listings';
	return $folder;
}
add_filter( 'wp-cardealer-theme-folder-name', 'boxcar_listing_template_folder_name', 10 );

if ( !function_exists('boxcar_get_listings_layout_configs') ) {
	function boxcar_get_listings_layout_configs() {
		$layout_sidebar = boxcar_get_listings_layout_sidebar();

		$sidebar = 'sidebar-default';
		switch ( $layout_sidebar ) {
		 	case 'left-main':
		 		$configs['left'] = array( 'sidebar' => $sidebar, 'class' => 'col-md-4 col-lg-3 col-sm-12 col-xs-12'  );
		 		$configs['main'] = array( 'class' => 'col-md-8 col-lg-9 col-sm-12 col-xs-12' );
		 		break;
		 	case 'main-right':
		 	default:
		 		$configs['right'] = array( 'sidebar' => $sidebar,  'class' => 'col-md-4 col-lg-3 col-sm-12 col-xs-12' ); 
		 		$configs['main'] = array( 'class' => 'col-md-8 col-lg-9 col-sm-12 col-xs-12' );
		 		break;
	 		case 'main':
	 			$configs['main'] = array( 'class' => 'col-md-12 col-sm-12 col-xs-12' );
	 			break;
		}
		return $configs; 
	}
}

function boxcar_get_listings_layout_sidebar() {
	global $post;
	if ( is_page() && is_object($post) ) {
		$layout_type = get_post_meta( $post->ID, 'apus_page_layout', true );
	}
	if ( empty($layout_type) ) {
		$layout_type = boxcar_get_config('listings_layout_sidebar', 'main-right');
	}
	return apply_filters( 'boxcar_get_listings_layout_sidebar', $layout_type );
}

function boxcar_get_listing_layout_type() {
	global $post;
	
	$layout_type = get_post_meta($post->ID, WP_CARDEALER_LISTING_PREFIX.'layout_type', true);
	
	if ( empty($layout_type) ) {
		$layout_type = boxcar_get_config( 'listing_elementor_template' );
	}
	return apply_filters( 'boxcar_get_listing_layout_type', $layout_type );
}

function boxcar_listing_scripts() {
	
	wp_enqueue_style( 'leaflet' );
	wp_enqueue_script( 'jquery-highlight' );
    wp_enqueue_script( 'leaflet' );
    wp_enqueue_script( 'control-geocoder' );
    wp_enqueue_script( 'esri-leaflet' );
    wp_enqueue_script( 'esri-leaflet-geocoder' );
    wp_enqueue_script( 'leaflet-markercluster' );
    wp_enqueue_script( 'leaflet-HtmlIcon' );
    
    if ( wp_cardealer_get_option('map_service') == 'google-map' ) {
    	wp_enqueue_script( 'leaflet-GoogleMutant' );
    }
    
	wp_register_script( 'boxcar-listing', get_template_directory_uri() . '/js/listing.js', array( 'jquery', 'wp-cardealer-main', 'perfect-scrollbar', 'imagesloaded' ), '20150330', true );

	$current_currency = WP_CarDealer_Price::get_current_currency();
	$multi_currencies = WP_CarDealer_Price::get_currencies_settings();

	if ( !empty($multi_currencies) && !empty($multi_currencies[$current_currency]) ) {
		$currency_args = $multi_currencies[$current_currency];
	}

	if ( !empty($currency_args) ) {
		$currency_symbol = !empty($currency_args['custom_symbol']) ? $currency_args['custom_symbol'] : '';
		if ( empty($currency_symbol) ) {
			$currency = !empty($currency_args['currency']) ? $currency_args['currency'] : 'USD';
			$currency_symbol = WP_CarDealer_Price::currency_symbol($currency);
		}
	}

	$currency_symbol = ! empty( $currency_symbol ) ? $currency_symbol : '$';
	$dec_point = ! empty( wp_cardealer_get_option('money_dec_point') ) ? wp_cardealer_get_option('money_dec_point') : '.';
	$thousands_separator = ! empty( wp_cardealer_get_option('money_thousands_separator') ) ? wp_cardealer_get_option('money_thousands_separator') : '';

	wp_localize_script( 'boxcar-listing', 'boxcar_listing_opts', array(
		'ajaxurl' => admin_url( 'admin-ajax.php' ),

		'dec_point' => $dec_point,
		'thousands_separator' => $thousands_separator,
		'currency' => esc_attr($currency_symbol),
		'currency_position' => esc_attr(wp_cardealer_get_option('currency_position')),
		'compare_added_title' => esc_html__('Compared', 'boxcar'),
		'compare_title' => esc_html__('Compare', 'boxcar'),
		'compare_added_tooltip_title' => esc_html__('Remove Compare', 'boxcar'),
		'compare_add_tooltip_title' => esc_html__('Add Compare', 'boxcar'),
		'favorite_added_tooltip_title' => esc_html__('Remove Favorite', 'boxcar'),
		'favorite_add_tooltip_title' => esc_html__('Add Favorite', 'boxcar'),
		
		'template' => apply_filters( 'boxcar_autocompleate_search_template', '<a href="{{url}}" class="media autocompleate-media">
			<div class="media-left media-middle">
				<img src="{{image}}" class="media-object" height="50" width="50">
			</div>
			<div class="media-body media-middle">
				<h4>{{title}}</h4>
				{{{price}}}
				</div></a>' ),
        'empty_msg' => apply_filters( 'boxcar_autocompleate_search_empty_msg', esc_html__( 'Unable to find any listing that match the currenty query', 'boxcar' ) ),
        'show' => __( 'Show', 'boxcar' ),
		'hide' => __( 'Hide', 'boxcar' ),
	));
	wp_enqueue_script( 'boxcar-listing' );

	$here_map_api_key = '';
	$here_style = '';
	$mapbox_token = '';
	$mapbox_style = '';
	$custom_style = '';
	$googlemap_type = wp_cardealer_get_option('googlemap_type', 'roadmap');
	if ( empty($googlemap_type) ) {
		$googlemap_type = 'roadmap';
	}
	$map_service = wp_cardealer_get_option('map_service', '');
	if ( $map_service == 'mapbox' ) {
		$mapbox_token = wp_cardealer_get_option('mapbox_token', '');
		$mapbox_style = wp_cardealer_get_option('mapbox_style', 'streets-v11');
		if ( empty($mapbox_style) || !in_array($mapbox_style, array( 'streets-v11', 'light-v10', 'dark-v10', 'outdoors-v11', 'satellite-v9' )) ) {
			$mapbox_style = 'streets-v11';
		}
	} elseif ( $map_service == 'here' ) {
		$here_map_api_key = wp_cardealer_get_option('here_map_api_key', '');
		$here_style = wp_cardealer_get_option('here_map_style', 'normal.day');
	} else {
		$custom_style = wp_cardealer_get_option('google_map_style', '');
	}

	wp_register_script( 'boxcar-listing-map', get_template_directory_uri() . '/js/listing-map.js', array( 'jquery' ), '20150330', true );
	wp_localize_script( 'boxcar-listing-map', 'boxcar_listing_map_opts', array(
		'map_service' => $map_service,
		'mapbox_token' => $mapbox_token,
		'mapbox_style' => $mapbox_style,
		'here_map_api_key' => $here_map_api_key,
		'here_style' => $here_style,
		'custom_style' => $custom_style,
		'googlemap_type' => $googlemap_type,
		'default_latitude' => wp_cardealer_get_option('default_maps_location_latitude', '43.6568'),
		'default_longitude' => wp_cardealer_get_option('default_maps_location_longitude', '-79.4512'),
		'default_pin' => wp_cardealer_get_option('default_maps_pin', ''),
		
	));
	wp_enqueue_script( 'boxcar-listing-map' );
}
add_action( 'wp_enqueue_scripts', 'boxcar_listing_scripts', 10 );

function boxcar_is_listings_page() {
	if( is_post_type_archive('listing') || is_tax('listing_category') || is_tax('listing_color') || is_tax('listing_condition') || is_tax('listing_cylinder') || is_tax('listing_door') || is_tax('listing_drive_type') || is_tax('listing_feature') || is_tax('listing_fuel_type') || is_tax('listing_location') || is_tax('listing_make') || is_tax('listing_model') || is_tax('listing_offer_type') || is_tax('listing_safety_type') || is_tax('listing_transmission') || is_tax('listing_type') ) {
		return true;
	}
	return false;
}

function boxcar_is_listing_single_page() {
	if ( is_singular('listing') || apply_filters('boxcar_is_listings_page', false) ) {
		return true;
	}
	return false;
}


function boxcar_listing_metaboxes($fields) {
	// listing

	$elementor_options = ['' => esc_html__('Global Settings', 'boxcar')];
    if ( did_action( 'elementor/loaded' ) ) {
        $ele_obj = \Elementor\Plugin::$instance;
        $templates = $ele_obj->templates_manager->get_source( 'local' )->get_items();
        
        if ( !empty( $templates ) ) {
            foreach ( $templates as $template ) {
                $elementor_options[ $template['template_id'] ] = $template['title'] . ' (' . $template['type'] . ')';
            }
        }
    }

	$prefix = WP_CARDEALER_LISTING_PREFIX;
	if ( !empty($fields) ) {
		$fields[ $prefix . 'tab-layout-version' ] = array(
			'id' => $prefix . 'tab-layout-version',
			'icon' => 'dashicons-admin-appearance',
			'title' => esc_html__( 'Layout Type', 'boxcar' ),
			'fields' => array(
				array(
					'name'              => esc_html__( 'Layout Type', 'boxcar' ),
					'id'                => $prefix . 'layout_type',
					'type'              => 'select',
					'options'			=> $elementor_options,
				)
			)
		);
	}
	
	return $fields;
}
add_filter( 'wp-cardealer-admin-custom-fields', 'boxcar_listing_metaboxes' );


function boxcar_compare_footer_html() {
	if ( !boxcar_get_config('listing_enable_compare', true) ) {
		return;
	}
	$compare_ids = WP_CarDealer_Compare::get_compare_items(); ?>
	<div id="compare-sidebar" class="<?php echo esc_attr(count($compare_ids) > 0 ? 'active' : ''); ?>">
		<h3 class="title"><?php echo esc_html__('Compare Listings', 'boxcar'); ?></h3>
		<div class="compare-sidebar-inner">
			<div class="compare-list">
				<?php
					if ( count($compare_ids) > 0 ) {
						$page_id = wp_cardealer_get_option('compare_listings_page_id');
	            		$submit_url = $page_id ? get_permalink($page_id) : home_url( '/' );
						
						foreach ($compare_ids as $listing_id) {
							$post_object = get_post( $listing_id );
	                        if ( $post_object ) {
	                            setup_postdata( $GLOBALS['post'] =& $post_object );
	                            echo WP_CarDealer_Template_Loader::get_template_part( 'listings-styles/inner-list-compare-small' );
	                        }
						}
					}
				?>
			</div>
			<?php if ( count($compare_ids) > 0 ) { ?>
				<div class="compare-actions">
					<div class="row row-20 clearfix">
						<div class="col-6">
						<a href="<?php echo esc_url($submit_url); ?>" class="btn btn-theme w-100 btn-sm"><?php echo esc_html__('Compare', 'boxcar'); ?><svg xmlns="http://www.w3.org/2000/svg" class="next" width="14" height="14" viewBox="0 0 14 14" fill="none">
            <path d="M13.6109 0H5.05533C4.84037 0 4.66643 0.173943 4.66643 0.388901C4.66643 0.603859 4.84037 0.777802 5.05533 0.777802H12.6721L0.113697 13.3362C-0.0382246 13.4881 -0.0382246 13.7342 0.113697 13.8861C0.18964 13.962 0.289171 14 0.388666 14C0.488161 14 0.587656 13.962 0.663635 13.8861L13.222 1.3277V8.94447C13.222 9.15943 13.3959 9.33337 13.6109 9.33337C13.8259 9.33337 13.9998 9.15943 13.9998 8.94447V0.388901C13.9998 0.173943 13.8258 0 13.6109 0Z" fill="currentColor"></path>
            </svg></a>
						</div>
						<div class="col-6">
						<a href="javascript:void(0);" class="btn-remove-compare-all btn btn-danger w-100 btn-sm" data-nonce="<?php echo esc_attr(wp_create_nonce( 'wp-cardealer-remove-listing-compare-nonce' )); ?>"><?php echo esc_html__('Clear', 'boxcar'); ?><svg xmlns="http://www.w3.org/2000/svg" class="next" width="14" height="14" viewBox="0 0 14 14" fill="none">
            <path d="M13.6109 0H5.05533C4.84037 0 4.66643 0.173943 4.66643 0.388901C4.66643 0.603859 4.84037 0.777802 5.05533 0.777802H12.6721L0.113697 13.3362C-0.0382246 13.4881 -0.0382246 13.7342 0.113697 13.8861C0.18964 13.962 0.289171 14 0.388666 14C0.488161 14 0.587656 13.962 0.663635 13.8861L13.222 1.3277V8.94447C13.222 9.15943 13.3959 9.33337 13.6109 9.33337C13.8259 9.33337 13.9998 9.15943 13.9998 8.94447V0.388901C13.9998 0.173943 13.8258 0 13.6109 0Z" fill="currentColor"></path>
            </svg></a>
						</div>
					</div>
				</div>
			<?php } ?>
		</div>
		<div class="compare-sidebar-btn">
			<?php esc_html_e( 'Compare', 'boxcar' ); ?> (<span class="count"><?php echo count($compare_ids); ?></span>)
		</div>
	</div><!-- .widget-area -->
<?php
}
add_action( 'wp_footer', 'boxcar_compare_footer_html', 10 );

function boxcar_add_remove_listing_compare_return($return) {
	$compare_ids = WP_CarDealer_Compare::get_compare_items();
	$output = '';
	if ( !empty($compare_ids) && count($compare_ids) > 0 ) {
		ob_start();
		$page_id = wp_cardealer_get_option('compare_listings_page_id');
		$submit_url = $page_id ? get_permalink($page_id) : home_url( '/' );
		?>
		<div class="compare-list">
			<?php
			foreach ($compare_ids as $listing_id) {
				$post_object = get_post( $listing_id );
                if ( $post_object ) {
                    setup_postdata( $GLOBALS['post'] =& $post_object );
                    echo WP_CarDealer_Template_Loader::get_template_part( 'listings-styles/inner-list-compare-small' );
                }
			}
			?>
		</div>
		<div class="compare-actions">
			<div class="row row-20 clearfix">
				<div class="col-6">
				<a href="<?php echo esc_url($submit_url); ?>" class="btn btn-theme w-100 btn-sm"><?php echo esc_html__('Compare', 'boxcar'); ?><svg xmlns="http://www.w3.org/2000/svg" class="next" width="14" height="14" viewBox="0 0 14 14" fill="none">
            <path d="M13.6109 0H5.05533C4.84037 0 4.66643 0.173943 4.66643 0.388901C4.66643 0.603859 4.84037 0.777802 5.05533 0.777802H12.6721L0.113697 13.3362C-0.0382246 13.4881 -0.0382246 13.7342 0.113697 13.8861C0.18964 13.962 0.289171 14 0.388666 14C0.488161 14 0.587656 13.962 0.663635 13.8861L13.222 1.3277V8.94447C13.222 9.15943 13.3959 9.33337 13.6109 9.33337C13.8259 9.33337 13.9998 9.15943 13.9998 8.94447V0.388901C13.9998 0.173943 13.8258 0 13.6109 0Z" fill="currentColor"></path>
            </svg></a>
				</div>
				<div class="col-6">
				<a href="javascript:void(0);" class="btn-remove-compare-all btn btn-danger w-100 btn-sm" data-nonce="<?php echo esc_attr(wp_create_nonce( 'wp-cardealer-remove-listing-compare-nonce' )); ?>"><?php echo esc_html__('Clear', 'boxcar'); ?><svg xmlns="http://www.w3.org/2000/svg" class="next" width="14" height="14" viewBox="0 0 14 14" fill="none">
            <path d="M13.6109 0H5.05533C4.84037 0 4.66643 0.173943 4.66643 0.388901C4.66643 0.603859 4.84037 0.777802 5.05533 0.777802H12.6721L0.113697 13.3362C-0.0382246 13.4881 -0.0382246 13.7342 0.113697 13.8861C0.18964 13.962 0.289171 14 0.388666 14C0.488161 14 0.587656 13.962 0.663635 13.8861L13.222 1.3277V8.94447C13.222 9.15943 13.3959 9.33337 13.6109 9.33337C13.8259 9.33337 13.9998 9.15943 13.9998 8.94447V0.388901C13.9998 0.173943 13.8258 0 13.6109 0Z" fill="currentColor"></path>
            </svg></a>
				</div>
			</div>
		</div>
		<?php
		$output = ob_get_clean();
	}
	$return['html_output'] = $output;
	$return['count'] = !empty($compare_ids) ? count($compare_ids) : 0;
	

	return $return;
}
add_filter( 'wp-cardealer-process-add-listing-compare-return', 'boxcar_add_remove_listing_compare_return', 10, 1 );
add_filter( 'wp-cardealer-process-remove-listing-compare-return', 'boxcar_add_remove_listing_compare_return', 10, 1 );



// autocomplete search listings
add_action( 'wpcd_ajax_boxcar_autocomplete_search_listings', 'boxcar_autocomplete_search_listings' );
function boxcar_autocomplete_search_listings() {

    // Query for suggestions
    $suggestions = array();
    $args = array(
		'post_type' => 'listing',
		'posts_per_page' => 10,
		'fields' => 'ids'
	);
    $filter_params = isset($_REQUEST['data']) ? $_REQUEST['data'] : null;

	$listings = WP_CarDealer_Query::get_posts( $args, $filter_params );

	if ( !empty($listings->posts) ) {
		foreach ($listings->posts as $post_id) {
			$suggestion['title'] = get_the_title($post_id);
			$suggestion['url'] = get_permalink($post_id);

			if ( has_post_thumbnail( $post_id ) ) {
	            $image = wp_get_attachment_image_src( get_post_thumbnail_id( $post_id ), 'thumbnail' );
	            $suggestion['image'] = $image[0];
	        } else {
	            $suggestion['image'] = boxcar_placeholder_img_src();
	        }
	        
	        $suggestion['price'] = boxcar_listing_display_price($post_id, 'icon', false);


        	$suggestions[] = $suggestion;
		}
		wp_reset_postdata();
	}
    echo json_encode( $suggestions );
 
    exit;
}


function boxcar_user_display_phone($phone, $display_type = 'no-title', $echo = true, $always_show_phone = false, $show_url = true) {
    ob_start();
    if ( $phone ) {
        $show_full = boxcar_get_config('listing_show_full_phone', false);
        $hide_phone = $show_full ? false : true;
        $hide_phone = apply_filters('boxcar_phone_hide_number', $hide_phone );
        if ( $always_show_phone ) {
        	$hide_phone = false;
        }
        $add_class = '';
        if ( $hide_phone ) {
            $add_class = 'phone-hide';
        }
        if ( $display_type == 'title' ) {
            ?>
            <div class="phone-wrapper agent-phone with-title <?php echo esc_attr($add_class); ?>">
                <span><?php esc_html_e('Phone: ', 'boxcar'); ?></span>
            <?php
        } elseif ($display_type == 'icon') {
            ?>
            <div class="phone-wrapper agent-phone with-icon <?php echo esc_attr($add_class); ?>">
                <i class="ti-headphone-alt"></i>
        <?php
        } else {
            ?>
            <div class="phone-wrapper agent-phone <?php echo esc_attr($add_class); ?>">
            <?php
        }

        ?>
        	<?php if ( $show_url ) { ?>
            <a class="phone" href="tel:<?php echo trim($phone); ?>">
            <?php } ?>
            	<?php echo trim($phone); ?>
    		<?php if ( $show_url ) { ?>
            	</a>
        	<?php } ?>

            <?php if ( $hide_phone ) {
                $dispnum = substr($phone, 0, (strlen($phone)-3) ) . str_repeat("*", 3);
            ?>
                <span class="phone-show" onclick="this.parentNode.classList.add('show');"><?php echo trim($dispnum); ?> <span><?php esc_html_e('show', 'boxcar'); ?></span></span>
            <?php } ?>
        </div>
        <?php
    }
    $output = ob_get_clean();
    if ( $echo ) {
        echo trim($output);
    } else {
        return $output;
    }
}


add_action( 'wp_ajax_nopriv_boxcar_ajax_print_listing', 'boxcar_ajax_print_listing' );
add_action( 'wp_ajax_boxcar_ajax_print_listing', 'boxcar_ajax_print_listing' );

add_action( 'wpcd_ajax_boxcar_ajax_print_listing', 'boxcar_ajax_print_listing' );

function boxcar_ajax_print_listing () {
	if ( !isset( $_POST['nonce'] ) || ! wp_verify_nonce( $_POST['nonce'], 'boxcar-printer-listing-nonce' )  ) {
		exit();
	}
	if( !isset($_POST['listing_id'])|| !is_numeric($_POST['listing_id']) ){
        exit();
    }

    $listing_id = intval($_POST['listing_id']);
    $the_post = get_post( $listing_id );

    if( $the_post->post_type != 'listing' || $the_post->post_status != 'publish' ) {
        exit();
    }
    setup_postdata( $GLOBALS['post'] =& $the_post );
    global $post;

    $dir = '';
    $body_class = '';
    if ( is_rtl() ) {
    	$dir = 'dir="rtl"';
    	$body_class = 'rtl';
    }

    print  '<html '.$dir.'><head><link href="'.get_stylesheet_uri().'" rel="stylesheet" type="text/css" />';
    if( is_rtl() ) {
    	print '<link href="'.get_template_directory_uri().'/css/bootstrap-rtl.css" rel="stylesheet" type="text/css" />';
    } else {
	    print  '<html><head><link href="'.get_template_directory_uri().'/css/bootstrap.css" rel="stylesheet" type="text/css" />';
	}
    print  '<html><head><link href="'.get_template_directory_uri().'/css/all-awesome.css" rel="stylesheet" type="text/css" />';
    print  '<html><head><link href="'.get_template_directory_uri().'/css/flaticon.css" rel="stylesheet" type="text/css" />';
    print  '<html><head><link href="'.get_template_directory_uri().'/css/themify-icons.css" rel="stylesheet" type="text/css" />';
    print  '<html><head><link href="'.get_template_directory_uri().'/css/template.css" rel="stylesheet" type="text/css" />';


    print '</head>';
    print '<script>window.onload = function() { window.print(); }</script>';
    print '<body class="'.$body_class.'">';

    $print_logo = boxcar_get_config('print-logo');
    if( !empty($print_logo) ) {
    	$print_logo = $print_logo;
    } else {
    	$print_logo = get_template_directory_uri().'/images/logo.svg';
    }
    $title = get_the_title( $listing_id );

    $image_id = get_post_thumbnail_id( $listing_id );
    $full_img = wp_get_attachment_image_src($image_id, 'boxcar-slider');
    $full_img = $full_img [0];

    ?>

    <section id="section-body">
        <!--start detail content-->
        <section class="section-detail-content">
            <div class="detail-bar print-detail">
                
                <?php if ( boxcar_get_config('show_print_header', true) ) { ?>
	            	<div class="print-header-top">
	                    <div class="inner">
                            <img src="<?php echo esc_url($print_logo); ?>" alt="<?php esc_attr_e('Logo', 'boxcar'); ?>">
	                    </div>
	                </div>
	            <?php } ?>

                <div class="print-header-middle">
                    <div class="print-header-middle-left">
                        <h1><?php echo esc_attr($title); ?></h1>
                        <?php boxcar_listing_display_full_location_without_url($post,'no-icon-title',true); ?>
                    </div>
                    <div class="print-header-middle-right">
                        <?php boxcar_listing_display_price($post); ?>
                    </div>
                </div>

                <?php if( !empty($full_img) ) { ?>
	                <div class="print-banner">
	                    <div class="print-main-image">
                            <img src="<?php echo esc_url( $full_img ); ?>" alt="<?php echo esc_attr($title); ?>">
                            <?php if ( boxcar_get_config('show_print_qrcode', true) ) { ?>
	                            <img class="qr-image" src="https://chart.googleapis.com/chart?chs=105x104&cht=qr&chl=<?php echo esc_url( get_permalink($listing_id) ); ?>&choe=UTF-8" title="<?php echo esc_attr($title); ?>" />
	                        <?php } ?>
	                    </div>
	                </div>
                <?php } ?>
                <?php
                
                if ( boxcar_get_config('show_print_author', true) ) {
					$user_id = $post->post_author;
					$author_email = get_the_author_meta('user_email');
					$a_title = get_the_author_meta('display_name');
					$a_phone = get_user_meta($user_id, '_user_phone', true);
					$a_phone = boxcar_user_display_phone($a_phone, 'no-title', false, true, false);
					$a_website = get_user_meta($user_id, '_user_url', true);

            	?>
                    <div class="print-block">
                    	<h3><?php esc_html_e( 'Contact Author', 'boxcar' ); ?></h3>
                        <div class="agent-media">
                            <div class="media-image-left">
                                <?php echo boxcar_get_avatar($post->post_author, 180); ?>
                            </div>
                            <div class="media-body-right">
                                
                                <h4 class="title"><?php echo trim($a_title); ?></h4>
								<div class="phone"><?php echo trim($a_phone); ?></div>
								<div class="email"><?php echo trim($author_email); ?></div>
								<div class="website"><?php echo trim($a_website); ?></div>

                            </div>
                        </div>
                    </div>
                <?php } ?>

                <div id="listing-single-details">
					<?php
					if ( boxcar_get_config('show_print_description', true) ) {
						?>
						<div class="description inner">
						    <h3 class="title"><?php esc_html_e('Overview', 'boxcar'); ?></h3>
						    <div class="description-inner">
						        <?php the_content(); ?>
						        <?php do_action('wp-cardealer-single-listing-description', $post); ?>
						    </div>
						</div>
						<?php
					}
					?>

					<?php
					if ( boxcar_get_config('show_print_detail', true) ) {
						echo WP_CarDealer_Template_Loader::get_template_part( 'single-listing/detail', array('print_car' => true) );
					}
					?>

				</div>

				<?php
				if ( boxcar_get_config('show_print_features', true) ) {
					echo WP_CarDealer_Template_Loader::get_template_part( 'single-listing/features' );
				}
				?>

				<?php

				$obj_listing_meta = WP_CarDealer_Listing_Meta::get_instance($post->ID);
				$gallery = $obj_listing_meta->get_post_meta( 'gallery' );
				if ( boxcar_get_config('show_print_gallery', true) && $gallery ) {
				?>
					<div class="print-gallery">
						<div class="detail-title-inner">
                            <h4 class="title-inner"><?php esc_html_e('Listing images', 'boxcar'); ?></h4>
                        </div>
                        <div class="row">
							<?php foreach ( $gallery as $id => $src ) { ?>
				                <div class="print-gallery-image col-xs-12 col-sm-6">
				                    <?php echo wp_get_attachment_image( $id, 'boxcar-slider' ); ?>
				                </div>
			                <?php } ?>
		                </div>
		          	</div>
	          	<?php } ?>
				
            </div>
        </section>
    </section>


    <?php
    
    wp_reset_postdata();

    print '</body></html>';
    wp_die();
}


function boxcar_load_select2() {
	wp_enqueue_script('wpcd-select2');
	wp_enqueue_style('wpcd-select2');
}


add_filter( 'wp-cardealer-type-available-fields', 'boxcar_listing_type_available_fields' );
function boxcar_listing_type_available_fields($fields) {
	$prefix = WP_CARDEALER_LISTING_PREFIX;
	$fields[] = array(
        'name'              => __( 'Old Price', 'boxcar' ),
        'id'                => $prefix . 'old_price',
        'type'              => 'text',
        'show_compare'      => false
    );
	$fields[] = array(
        'name'              => __( 'Tagline', 'boxcar' ),
        'id'                => $prefix . 'tagline',
        'type'              => 'text',
        'show_compare'      => false
    );
	$fields[] = array(
        'name'              => __( 'Height', 'boxcar' ),
        'id'                => $prefix . 'height',
        'type'              => 'text',
        'show_compare'      => true
    );
    $fields[] = array(
        'name'              => __( 'Height (including roof rails)', 'boxcar' ),
        'id'                => $prefix . 'height_including_roof_rails',
        'type'              => 'text',
        'show_compare'      => true
    );
    $fields[] = array(
        'name'              => __( 'Length', 'boxcar' ),
        'id'                => $prefix . 'length',
        'type'              => 'text',
        'show_compare'      => true
    );
    $fields[] = array(
        'name'              => __( 'Wheelbase', 'boxcar' ),
        'id'                => $prefix . 'wheelbase',
        'type'              => 'text',
        'show_compare'      => true
    );
    $fields[] = array(
        'name'              => __( 'Width', 'boxcar' ),
        'id'                => $prefix . 'width',
        'type'              => 'text',
        'show_compare'      => true
    );
    $fields[] = array(
        'name'              => __( 'Width (including mirrors)', 'boxcar' ),
        'id'                => $prefix . 'width_including_mirrors',
        'type'              => 'text',
        'show_compare'      => true
    );
    $fields[] = array(
        'name'              => __( 'Fuel Tank Capacity (Litres)', 'boxcar' ),
        'id'                => $prefix . 'fuel_tank_capacity',
        'type'              => 'text',
        'show_compare'      => true
    );
    $fields[] = array(
        'name'              => __( 'Gross Vehicle Weight (kg)', 'boxcar' ),
        'id'                => $prefix . 'gross_vehicle_weight',
        'type'              => 'text',
        'show_compare'      => true
    );
    $fields[] = array(
        'name'              => __( 'Luggage Capacity (Seats Down - Litres)', 'boxcar' ),
        'id'                => $prefix . 'luggage_capacity_seats_down',
        'type'              => 'text',
        'show_compare'      => true
    );
    $fields[] = array(
        'name'              => __( 'Luggage Capacity (Seats Up - Litres)', 'boxcar' ),
        'id'                => $prefix . 'luggage_capacity_seats_up',
        'type'              => 'text',
        'show_compare'      => true
    );
    $fields[] = array(
        'name'              => __( 'Max. Loading Weight (kg)', 'boxcar' ),
        'id'                => $prefix . 'max_loading_weight',
        'type'              => 'text',
        'show_compare'      => true
    );
    $fields[] = array(
        'name'              => __( 'Max. Roof Load (kg)', 'boxcar' ),
        'id'                => $prefix . 'max_roof_load',
        'type'              => 'text',
        'show_compare'      => true
    );
    $fields[] = array(
        'name'              => __( 'Max. Towing Weight - Braked (kg)', 'boxcar' ),
        'id'                => $prefix . 'max_towing_weight_braked',
        'type'              => 'text',
        'show_compare'      => true
    );
    $fields[] = array(
        'name'              => __( 'Max. Towing Weight - Unbraked (kg)', 'boxcar' ),
        'id'                => $prefix . 'max_towing_weight_unbraked',
        'type'              => 'text',
        'show_compare'      => true
    );
    $fields[] = array(
        'name'              => __( 'Minimum Kerbweight (kg)', 'boxcar' ),
        'id'                => $prefix . 'minimum_kerbweight',
        'type'              => 'text',
        'show_compare'      => true
    );
    $fields[] = array(
        'name'              => __( 'No. of Seats', 'boxcar' ),
        'id'                => $prefix . 'number_of_seats',
        'type'              => 'text',
        'show_compare'      => true
    );
    $fields[] = array(
        'name'              => __( 'Turning Circle - Kerb to Kerb (m)', 'boxcar' ),
        'id'                => $prefix . 'turning_circle_kerb',
        'type'              => 'text',
        'show_compare'      => true
    );
    return $fields;
}

add_filter( 'wp-cardealer-user-type-available-fields', 'boxcar_user_type_available_fields', 10 );
function boxcar_user_type_available_fields($fields) {
	$prefix = WP_CARDEALER_USER_PREFIX;
	$fields[] = array(
        'name'              => __( 'Hours', 'boxcar' ),
        'id'                => $prefix . 'hours',
        'type'              => 'wpcd_hours',
    );
    return $fields;
}

add_filter( 'wp_cardealer_list_simple_type', 'boxcar_listing_list_simple_type', 10 );
function boxcar_listing_list_simple_type($fields) {
	$prefix = WP_CARDEALER_LISTING_PREFIX;
	$fields = array_merge($fields, array($prefix . 'old_price', $prefix . 'tagline', $prefix . 'height', $prefix . 'height_including_roof_rails', $prefix . 'length', $prefix . 'wheelbase', $prefix . 'width', $prefix . 'width_including_mirrors', $prefix . 'fuel_tank_capacity', $prefix . 'gross_vehicle_weight', $prefix . 'luggage_capacity_seats_down', $prefix . 'luggage_capacity_seats_up', $prefix . 'max_loading_weight', $prefix . 'max_roof_load', $prefix . 'max_towing_weight_braked', $prefix . 'max_towing_weight_unbraked', $prefix . 'minimum_kerbweight', $prefix . 'number_of_seats', $prefix . 'turning_circle_kerb', WP_CARDEALER_USER_PREFIX . 'hours'));
	return $fields;
}

function boxcar_get_days_of_week() {
	$days = array(0, 1, 2, 3, 4, 5, 6);

	$start_day = get_option( 'start_of_week' );

	$first = array_splice( $days, $start_day, count( $days ) - $start_day );

	$second = array_splice( $days, 0, $start_day );

	$days = array_merge( $first, $second );

	return $days;
}

function boxcar_get_day_hours($hours) {
	global $wp_locale;
	if (empty($hours) || !is_array($hours)) {
		return;
	}
    $numericdays = boxcar_get_days_of_week();
    $days = array();

    foreach ( $numericdays as $key => $i ) {
        $day = $wp_locale->get_weekday( $i );
        if ( isset($hours[ $i ][ 'type' ]) && $hours[ $i ][ 'type' ] == 'enter_hours' ) {
        	if ( !empty($hours[ $i ][ 'from' ]) && !empty($hours[ $i ][ 'to' ]) ) {
        		$t_day = array();
        		foreach ($hours[ $i ][ 'from' ] as $key => $value) {
        			$start = $value;
        			$end = !empty($hours[ $i ][ 'to' ][$key]) ? $hours[ $i ][ 'to' ][$key] : false;
        			if ( $start && $end ) {
        				$start = strtotime($start);
        				$end = strtotime($end);
	        			$t_day[] = array( date(get_option('time_format'), $start), date(get_option('time_format'), $end) );
	        		}
        		}
        		$days[ $day ] = $t_day;
        	}

	    } elseif ( isset($hours[ $i ][ 'type' ]) && $hours[ $i ][ 'type' ] == 'open_all_day' ) {
	    	$days[ $day ] = 'open';
	    } elseif ( isset($hours[ $i ][ 'type' ]) && $hours[ $i ][ 'type' ] == 'closed_all_day' ) {
	    	$days[ $day ] = 'closed';
	    }
    }
    return $days;
}


function boxcar_get_current_time($timezone) {
	global $wp_locale;
	$timezones = timezone_identifiers_list();

	if ( $timezone && in_array( $timezone, $timezones ) ) {
		$timezone_date = new DateTime( 'now', new DateTimeZone( $timezone ) );

		$time = $timezone_date->format(get_option('time_format'));
		$date = $timezone_date->format('Y-m-d');
	} else {
		$timezone  = get_option('gmt_offset');
		
		$time = gmdate(get_option('time_format'), time() + 3600*($timezone+date("I"))); 
		$date = gmdate('Y-m-d', time() + 3600*($timezone+date("I")));
	}

	$time = strtotime($time);
	$date = strtotime($date);

	$day_of_week = date('N',$date);
	if ( $day_of_week == 7 ) {
		$day_of_week = 0;
	}
	$day = $wp_locale->get_weekday( $day_of_week );
	
	$day = ucfirst($day);
	$return = array( 'day' => $day, 'time' => $time );
	return $return;
}

function boxcar_get_current_time_status($hours) {
	$timezone = !empty($hours['timezone']) ? $hours['timezone'] : '';

	$current = boxcar_get_current_time($timezone);
	$current_day = strtolower($current['day']);
	$current_time = $current['time'];
	
	if ( !empty($hours['day']) ) {
		$days = boxcar_get_day_hours($hours['day']);
		
		if ( !empty($days[$current_day]) || !empty($days[$current['day']]) ) {

			$times = !empty($days[$current_day]) ? $days[$current_day] : $days[$current['day']];

			if ( is_array($times) ) {
				foreach ($times as $time) {
					$opentime = strtotime($time[0]);
					$closedtime = strtotime($time[1]);
					if ( $opentime <= $closedtime ) {
						if ( $current_time >= $opentime && $current_time <= $closedtime ) {
							return true;
						}
					} else {
						$is_open = true;
						if( $current_time < $opentime ){
				            if( $current_time > $closedtime ){
				                $is_open = false;
				            }
				        }
				        return $is_open;
					}
				}
			} elseif ( $times == 'open' ) {
				return true;
			} elseif ( $times == 'closed' ) {
				return false;
			}
		} else {
			return true;
		}
	}
	return false;
}

function boxcar_display_time_status($user_id) {
	if ( boxcar_get_config('listing_show_hour_status', true) ) {
		$hours = get_user_meta( $user_id, '_user_hours', true );
		$status = boxcar_get_current_time_status( $hours );

		if ( $status ) { ?>
			<div class="listing-time opening">
				<?php esc_html_e( 'Open', 'boxcar' ); ?>
			</div>
		<?php } else { ?>
			<div class="listing-time closed">
				<?php esc_html_e( 'Closed', 'boxcar' ); ?>
			</div>
		<?php }
	}
}


add_filter( 'template_include', 'boxcar_listing_set_template', 100 );
function boxcar_listing_set_template($template) {
    if ( is_embed() ) {
        return $template;
    }
    if ( is_singular( 'listing' ) ) {
    	$template_id = boxcar_get_listing_layout_type();
        if ( $template_id ) {
            $template =WP_CarDealer_Template_Loader::locate('template-listings/single-listing-elementor');
        }
    } elseif ( boxcar_is_listings_page() ) {
        if ( boxcar_get_config( 'listing_archive_elementor_template' ) ) {
            $template = WP_CarDealer_Template_Loader::locate('template-listings/archive-listing-elementor');
        }
    }
    return $template;
}

add_action( 'boxcar_listing_detail_content', 'boxcar_listing_detail_builder_content', 5 );
function boxcar_listing_detail_builder_content() {
    $template_id = boxcar_get_listing_layout_type();
    if ( $template_id ) {
        $post = get_post($template_id);
        echo apply_filters( 'boxcar_generate_post_builder', '', $post, $template_id);
    }
}

add_action( 'boxcar_listing_archive_content', 'boxcar_listing_archive_builder_content', 5 );
function boxcar_listing_archive_builder_content() {
    $template_id = boxcar_get_config('listing_archive_elementor_template');
    if ( $template_id ) {
        $post = get_post($template_id);
        echo apply_filters( 'boxcar_generate_post_builder', '', $post, $template_id);
    }
}

add_action( 'boxcar_listing_author_content', 'boxcar_listing_author_builder_content', 5 );
function boxcar_listing_author_builder_content() {
    $template_id = boxcar_get_config('user_profile_elementor_template');
    if ( $template_id ) {
        $post = get_post($template_id);
        echo apply_filters( 'boxcar_generate_post_builder', '', $post, $template_id);
    }
}



// demo function
function boxcar_check_demo_account() {
	if ( defined('BOXCAR_DEMO_MODE') && BOXCAR_DEMO_MODE ) {
		$user_id = get_current_user_id();
		$user_obj = get_user_by('ID', $user_id);
		if ( strtolower($user_obj->data->user_login) == 'demo') {
			$return = array( 'status' => false, 'msg' => esc_html__('Demo users are not allowed to modify information.', 'boxcar') );
		   	echo wp_json_encode($return);
		   	exit;
		}
	}
}

add_action('wp-cardealer-process-forgot-password', 'boxcar_check_demo_account', 10);
add_action('wp-cardealer-process-change-password', 'boxcar_check_demo_account', 10);
add_action('wp-cardealer-before-delete-profile', 'boxcar_check_demo_account', 10);
add_action('wp-cardealer-before-remove-listing-alert', 'boxcar_check_demo_account', 10 );
add_action('wp-cardealer-before-change-profile-normal', 'boxcar_check_demo_account', 10 );
add_action('wp-cardealer-process-add-agent', 'boxcar_check_demo_account', 10 );
add_action('wp-cardealer-process-remove-agent', 'boxcar_check_demo_account', 10 );
add_action('wp-cardealer-process-remove-before-save', 'boxcar_check_demo_account', 10);

function boxcar_check_demo_account2($error) {
	if ( defined('BOXCAR_DEMO_MODE') && BOXCAR_DEMO_MODE ) {
		$user_id = get_current_user_id();
		$user_obj = get_user_by('ID', $user_id);
		if ( strtolower($user_obj->data->user_login) == 'demo') {
			$error[] = esc_html__('Demo users are not allowed to modify information.', 'boxcar');
		}
	}
	return $error;
}
add_filter('wp-cardealer-submission-validate', 'boxcar_check_demo_account2', 10, 2);
add_filter('wp-cardealer-edit-validate', 'boxcar_check_demo_account2', 10, 2);

function boxcar_check_demo_account3($post_id, $prefix) {
	if ( defined('BOXCAR_DEMO_MODE') && BOXCAR_DEMO_MODE ) {
		$user_id = get_current_user_id();
		$user_obj = get_user_by('ID', $user_id);
		if ( strtolower($user_obj->data->user_login) == 'demo') {
			$_SESSION['messages'][] = array( 'danger', esc_html__('Demo users are not allowed to modify information.', 'boxcar') );
			$redirect_url = get_permalink( wp_cardealer_get_option('edit_profile_page_id') );
			WP_CarDealer_Mixes::redirect( $redirect_url );
			exit();
		}
	}
}
add_action('wp-cardealer-process-profile-before-change', 'boxcar_check_demo_account3', 10, 2);

function boxcar_check_demo_account4() {
	if ( defined('BOXCAR_DEMO_MODE') && BOXCAR_DEMO_MODE ) {
		$user_id = get_current_user_id();
		$user_obj = get_user_by('ID', $user_id);
		if ( strtolower($user_obj->data->user_login) == 'demo') {
			$return['msg'] = esc_html__('Demo users are not allowed to modify information.', 'boxcar');
			$return['status'] = false;
			echo json_encode($return); exit;
		}
	}
}
add_action('wp-private-message-before-reply-message', 'boxcar_check_demo_account4');
add_action('wp-private-message-before-add-message', 'boxcar_check_demo_account4');
add_action('wp-private-message-before-delete-message', 'boxcar_check_demo_account4');

function boxcar_title_connect() {
	if ( WP_CarDealer_Mixes::check_social_login_enable() ) {
		echo '<div class="line-header"><span>'.esc_html__('or', 'boxcar').'</span></div>';
	}
}
add_action('login_form', 'boxcar_title_connect', 2);