<?php

/**
 * Elementor Single Widget
 * @package techco-core
 * @since 1.0.0
 */

namespace Elementor;

if (!defined('ABSPATH')) exit; // Exit if accessed directly.

class Techco_Page_Title extends Widget_Base
{

    /**
     * Get widget name.
     *
     * Retrieve Elementor widget name.
     *
     * @return string Widget name.
     * @since 1.0.0
     * @access public
     *
     */
    public function get_name()
    {
        return 'int-page-tile';
    }

    /**
     * Get widget title.
     *
     * Retrieve Elementor widget title.
     *
     * @return string Widget title.
     * @since 1.0.0
     * @access public
     *
     */
    public function get_title()
    {
        return esc_html__('Page Title', 'techco-core');
    }

    /**
     * Get widget icon.
     *
     * Retrieve Elementor widget icon.
     *
     * @return string Widget icon.
     * @since 1.0.0
     * @access public
     *
     */
    public function get_icon()
    {
        return 'techco-custom-icon';
    }

    /**
     * Get widget categories.
     *
     * Retrieve the list of categories the Elementor widget belongs to.
     *
     * @return array Widget categories.
     * @since 1.0.0
     * @access public
     *
     */
    public function get_categories()
    {
        return ['techco_widgets'];
    }


    protected function register_controls()
    {

        $this->start_controls_section(
            'content_option',
            [
                'label' => esc_html__('Content', 'techco-core'),
                'tab' => Controls_Manager::TAB_CONTENT,
            ]
        );

        $this->add_control(
            'sub_title',
            [
                'type' => Controls_Manager::TEXT,
                'label' => esc_html__('Sub Title', 'techco-core'),
                'label_block' => true,
            ]
        );
        $this->add_control(
            'sub_title_mark',
            [
                'type' => Controls_Manager::TEXT,
                'label' => esc_html__('SubTitle Badge Text', 'techco-core'),
            ]
        );
        $this->add_control(
            'badge_align',
            [
                'label' => __('Badge Alignment', 'techco-core'),
                'type' => \Elementor\Controls_Manager::SELECT,
                'options' => [
                    'badge_left' => __('Badge Left', 'techco-core'),
                    'badge_right' => __('Badge Right', 'techco-core'),
                ],
                'default' => 'badge_left',
            ]
        );
        $this->add_control(
            'title_hr',
            [
                'type' => \Elementor\Controls_Manager::DIVIDER,
            ]
        );
        $this->add_control(
            'title',
            [
                'type' => Controls_Manager::TEXTAREA,
                'label' => esc_html__('Title', 'techco-core'),
                'label_block' => true,
            ]
        );

        $this->add_responsive_control(
            'text_align',
            [
                'label' => esc_html__('Text Align', 'techco-core'),
                'type' => Controls_Manager::CHOOSE,
                'label_block' => false,

                'options' => [
                    'start' => [
                        'title' => __('Left', 'techco-core'),
                        'icon' => 'eicon-text-align-left',
                    ],

                    'center' => [
                        'title' => __('Center', 'techco-core'),
                        'icon' => 'eicon-text-align-center',
                    ],

                    'end' => [
                        'title' => __('Right', 'techco-core'),
                        'icon' => 'eicon-text-align-right',
                    ],
                ],

                'devices' => ['desktop', 'tablet', 'mobile'],

                'selectors' => [
                    '{{WRAPPER}} .page_banner' => 'text-align: {{VALUE}};',
                ],
            ]
        );
        $this->end_controls_section();

        $this->start_controls_section(
            'content_style',
            [
                'label' => esc_html__('Style', 'techco-core'),
                'tab' => Controls_Manager::TAB_STYLE,
            ]
        );
        $this->add_control(
            'sub_title-c',
            [
                'label' => esc_html__('Sub Title Color', 'techco-core'),
                'type' => Controls_Manager::COLOR,
                'selectors' => [
                    '{{WRAPPER}} .heading_focus_text' => 'color: {{VALUE}} !important',
                ],
            ]
        );
        $this->add_control(
            'sub_title_badge-c',
            [
                'label' => esc_html__('Mark Text Color', 'techco-core'),
                'type' => Controls_Manager::COLOR,
                'selectors' => [
                    '{{WRAPPER}} .heading_focus_text .badge' => 'color: {{VALUE}}',
                ],
            ]
        );
        $this->add_control(
            'sub_title_bg-c',
            [
                'label' => esc_html__('Mark Text BG Color', 'techco-core'),
                'type' => Controls_Manager::COLOR,
                'selectors' => [
                    '{{WRAPPER}} .heading_focus_text .badge' => 'background-color: {{VALUE}} !important',
                ],
            ]
        );
        $this->add_group_control(
            Group_Control_Typography::get_type(),
            [
                'name' => 'sub_title_typography',
                'selector' => '{{WRAPPER}} .heading_focus_text, {{WRAPPER}} .heading_focus_text .badge',
            ]
        );
        $this->add_control(
            'title_style_hr',
            [
                'type' => \Elementor\Controls_Manager::DIVIDER,
            ]
        );
        $this->add_control(
            'title-c',
            [
                'label' => esc_html__('Title Color', 'techco-core'),
                'type' => Controls_Manager::COLOR,
                'selectors' => [
                    '{{WRAPPER}} .page_title' => 'color: {{VALUE}} !important',
                ],
            ]
        );
        $this->add_group_control(
            Group_Control_Typography::get_type(),
            [
                'name' => 'title_typography',
                'selector' => '{{WRAPPER}} .page_title',
            ]
        );
        $this->add_control(
            'title_padding',
            [
                'label' => esc_html__('Title Padding', 'techco-core'),
                'type' => \Elementor\Controls_Manager::DIMENSIONS,
                'size_units' => ['px', '%', 'em', 'rem', 'custom'],
                'selectors' => [
                    '{{WRAPPER}} .page_title' => 'padding: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
                ],
            ]
        );

        $this->end_controls_section();

    }

    protected function render()
    {
        $settings = $this->get_settings_for_display();
        ?>

        <div class="page_banner">
            <?php if (!empty($settings['sub_title'])): ?>
                <div class="heading_focus_text text-white">
                    <?php if (!empty($settings['sub_title_mark']) && $settings['badge_align'] === 'badge_left'): ?>
                        <span class="badge bg-secondary"><?php echo esc_html($settings['sub_title_mark']); ?></span>
                    <?php endif; ?>
                    <?php echo esc_html($settings['sub_title']); ?>
                    <?php if (!empty($settings['sub_title_mark']) && $settings['badge_align'] === 'badge_right'): ?>
                        <span class="badge bg-secondary"><?php echo esc_html($settings['sub_title_mark']); ?></span>
                    <?php endif; ?>
                </div>
            <?php endif; ?>
            <?php if (!empty($settings['title'])): ?>
                <h1 class="page_title"><?php echo wp_kses($settings['title'], true); ?></h1>
            <?php endif; ?>
        </div>
        <?php
    }


}


Plugin::instance()->widgets_manager->register(new Techco_Page_Title());