<?php

/**
 * Elementor Single Widget
 * @package techco-core
 * @since 1.0.0
 */

namespace Elementor;

if (!defined('ABSPATH')) exit; // Exit if accessed directly.

class Techco_Similar_Project extends Widget_Base
{

    /**
     * Get widget name.
     *
     * Retrieve Elementor widget name.
     *
     * @return string Widget name.
     * @since 1.0.0
     * @access public
     *
     */
    public function get_name()
    {
        return 'int-similar-project';
    }

    /**
     * Get widget title.
     *
     * Retrieve Elementor widget title.
     *
     * @return string Widget title.
     * @since 1.0.0
     * @access public
     *
     */
    public function get_title()
    {
        return esc_html__('Similar Project', 'techco-core');
    }

    /**
     * Get widget icon.
     *
     * Retrieve Elementor widget icon.
     *
     * @return string Widget icon.
     * @since 1.0.0
     * @access public
     *
     */
    public function get_icon()
    {
        return 'techco-custom-icon';
    }

    /**
     * Get widget categories.
     *
     * Retrieve the list of categories the Elementor widget belongs to.
     *
     * @return array Widget categories.
     * @since 1.0.0
     * @access public
     *
     */
    public function get_categories()
    {
        return ['techco_widgets'];
    }


    protected function register_controls()
    {


        $this->start_controls_section(
            'project_option',
            [
                'label' => esc_html__('Project', 'techco-core'),
                'tab' => Controls_Manager::TAB_CONTENT,
            ]
        );
        $this->add_control(
            'projectorder',
            [
                'label' => esc_html__('Project Order', 'techco-core'),
                'type' => Controls_Manager::SELECT,
                'default' => 'ASC',
                'options' => [
                    'ASC' => esc_html__('Ascending', 'techco-core'),
                    'DESC' => esc_html__('Descending', 'techco-core'),
                ],
            ]
        );
        $this->add_control(
            'project_per_page',
            [
                'label' => __('Project Per Page', 'techco-core'),
                'type' => Controls_Manager::NUMBER,
                'min' => 1,
                'default' => 2,
            ]
        );
        $this->add_control(
            'project_categories',
            [
                'type' => Controls_Manager::SELECT2,
                'label' => esc_html__('Select Project Categories', 'techco-core'),
                'options' => $this->get_project_categories(),
                'label_block' => true,
                'multiple' => true,
            ]
        );
        $this->end_controls_section();

        $this->start_controls_section(
            'project_style',
            [
                'label' => esc_html__('Style', 'techco-core'),
                'tab' => Controls_Manager::TAB_STYLE,
            ]
        );
        $this->add_control(
            'bg-c',
            [
                'label' => esc_html__('Background Color', 'techco-core'),
                'type' => Controls_Manager::COLOR,
                'selectors' => [
                    '{{WRAPPER}} .portfolio_block' => 'background-color: {{VALUE}}',
                ],
            ]
        );
        $this->add_control(
            'port_padding',
            [
                'label' => esc_html__('Padding', 'techco-core'),
                'type' => \Elementor\Controls_Manager::DIMENSIONS,
                'size_units' => ['px', '%', 'em', 'rem', 'custom'],
                'selectors' => [
                    '{{WRAPPER}} .portfolio_block' => 'padding: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
                ],
            ]
        );
        $this->add_control(
            'port_cat_hr',
            [
                'type' => \Elementor\Controls_Manager::DIVIDER,
            ]
        );
        $this->add_control(
            'cat_color',
            [
                'label' => esc_html__('Category Color', 'techco-core'),
                'type' => Controls_Manager::COLOR,
                'selectors' => [
                    '{{WRAPPER}} .category_list a' => 'color: {{VALUE}}',
                ],
            ]
        );
        $this->add_control(
            'cat_hover_color',
            [
                'label' => esc_html__('Category Hover Color', 'techco-core'),
                'type' => Controls_Manager::COLOR,
                'selectors' => [
                    '{{WRAPPER}} .category_list a:hover' => 'color: {{VALUE}}',
                ],
            ]
        );
        $this->add_group_control(
            Group_Control_Typography::get_type(),
            [
                'name' => 'cat_typography',
                'selector' => '{{WRAPPER}} .category_list a',
            ]
        );
        $this->add_control(
            'port_title_hr',
            [
                'type' => \Elementor\Controls_Manager::DIVIDER,
            ]
        );
        $this->add_control(
            'title_color',
            [
                'label' => esc_html__('Title Color', 'techco-core'),
                'type' => Controls_Manager::COLOR,
                'selectors' => [
                    '{{WRAPPER}} .portfolio_block .portfolio_title a' => 'color: {{VALUE}}',
                ],
            ]
        );
        $this->add_control(
            'title_hover_color',
            [
                'label' => esc_html__('Title Hover Color', 'techco-core'),
                'type' => Controls_Manager::COLOR,
                'selectors' => [
                    '{{WRAPPER}} .portfolio_block .portfolio_title a:hover' => 'color: {{VALUE}}',
                ],
            ]
        );
        $this->add_group_control(
            Group_Control_Typography::get_type(),
            [
                'name' => 'title_typography',
                'selector' => '{{WRAPPER}} .portfolio_block .portfolio_title',
            ]
        );
        $this->end_controls_section();

    }


    private function get_project_categories() {
        $terms = get_terms([
            'taxonomy' => 'project_cat',
            'hide_empty' => true,
        ]);

        $cat_list = [];
        if (!empty($terms) && !is_wp_error($terms)) {
            foreach ($terms as $term) {
                $cat_list[$term->term_id] = $term->name;
            }
        }
        return $cat_list;
    }

    protected function render() {
        $settings = $this->get_settings_for_display();

        $args = [
            'post_type' => 'techco_project',
            'posts_per_page' => $settings['project_per_page'] ?? 1,
            'ignore_sticky_posts' => true,
            'post_status' => 'publish',
            'order' => $settings['projectorder'],
            'tax_query' => [],
        ];

        if (!empty($settings['project_categories'])) {
            $args['tax_query'][] = [
                'taxonomy' => 'project_cat',
                'field' => 'term_id',
                'terms' => $settings['project_categories'],
            ];
        }

        $query = new \WP_Query($args);

        if ($query->have_posts()) :
            echo '<div class="row justify-content-center">';
            while ($query->have_posts()) : $query->the_post();
                $this->render_project_item();
            endwhile;
            echo '</div>';
            wp_reset_postdata();
        endif;
    }

    private function render_project_item() {
        ?>
        <div class="col-lg-6">
            <div class="portfolio_block portfolio_layout_2">
                <div class="portfolio_image">
                    <a class="portfolio_image_wrap bg-light" href="<?php the_permalink(); ?>">
                        <?php the_post_thumbnail('techco-project'); ?>
                    </a>
                </div>
                <div class="portfolio_content">
                    <h3 class="portfolio_title">
                        <a href="<?php the_permalink(); ?>">
                            <?php the_title(); ?>
                        </a>
                    </h3>
                    <ul class="category_list unordered_list">
                        <?php
                        $projectInerCates = get_the_terms(get_the_ID(), 'project_cat');
                        if ($projectInerCates && !is_wp_error($projectInerCates)) :
                            foreach ($projectInerCates as $cate) :
                                ?>
                                <li>
                                    <a href="<?php echo esc_url(get_term_link($cate)); ?>">
                                        <i class="fa-solid fa-tags"></i><?php echo esc_html($cate->name); ?>
                                    </a>
                                </li>
                            <?php
                            endforeach;
                        endif;
                        ?>
                    </ul>
                </div>
            </div>
        </div>
        <?php
    }


}


Plugin::instance()->widgets_manager->register(new Techco_Similar_Project());