<?php

/**
 * Elementor Single Widget
 * @package techco-core
 * @since 1.0.0
 */

namespace Elementor;

if (!defined('ABSPATH')) exit; // Exit if accessed directly.

class Techco_Service extends Widget_Base
{

    /**
     * Get widget name.
     *
     * Retrieve Elementor widget name.
     *
     * @return string Widget name.
     * @since 1.0.0
     * @access public
     *
     */
    public function get_name()
    {
        return 'int-service';
    }

    /**
     * Get widget title.
     *
     * Retrieve Elementor widget title.
     *
     * @return string Widget title.
     * @since 1.0.0
     * @access public
     *
     */
    public function get_title()
    {
        return esc_html__('Service', 'techco-core');
    }

    /**
     * Get widget icon.
     *
     * Retrieve Elementor widget icon.
     *
     * @return string Widget icon.
     * @since 1.0.0
     * @access public
     *
     */
    public function get_icon()
    {
        return 'techco-custom-icon';
    }

    /**
     * Get widget categories.
     *
     * Retrieve the list of categories the Elementor widget belongs to.
     *
     * @return array Widget categories.
     * @since 1.0.0
     * @access public
     *
     */
    public function get_categories()
    {
        return ['techco_widgets'];
    }


    protected function register_controls()
    {

        $this->start_controls_section(
            'service_option',
            [
                'label' => esc_html__('Service', 'techco-core'),
                'tab' => Controls_Manager::TAB_CONTENT,
            ]
        );
        $this->add_control(
            'template',
            [
                'type' => Controls_Manager::SELECT2,
                'options' => $this->template_select(),
                'multiple' => false,
                'label' => esc_html__('Service', 'techco-core'),
                'label_block' => true,
            ]
        );
        $this->add_control(
            'note_label',
            [
                'type' => Controls_Manager::RAW_HTML,
                'raw' => sprintf(
                    '<div class="elementor-panel-alert elementor-panel-alert-info">
                <p>%s</p>
            </div>',
                    sprintf(
                        __('Please note! Customize your selected Service using Elementor. You can find it under <a href="%s" target="_blank">Services</a>.', 'techco-core'),
                        admin_url('edit.php?post_type=techco_service')
                    )
                ),
                'content_classes' => 'elementor-notice-container',
            ]
        );
        $this->end_controls_section();

        $this->start_controls_section(
            'service_style_option',
            [
                'label' => esc_html__('Style', 'techco-core'),
                'tab' => Controls_Manager::TAB_STYLE,
            ]
        );
        $this->add_control(
            'service_padding',
            [
                'label' => esc_html__('Padding', 'techco-core'),
                'type' => \Elementor\Controls_Manager::DIMENSIONS,
                'size_units' => ['px', '%', 'em', 'rem', 'custom'],
                'selectors' => [
                    '{{WRAPPER}} .service_block .service_content' => 'padding: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
                ],
            ]
        );
        $this->add_control(
            'title_hr',
            [
                'type' => \Elementor\Controls_Manager::DIVIDER,
            ]
        );
        $this->add_control(
            'service_title-c',
            [
                'label' => esc_html__('Title Color', 'techco-core'),
                'type' => Controls_Manager::COLOR,
                'selectors' => [
                    '{{WRAPPER}} .service_block .service_title a' => 'color: {{VALUE}}',
                ],
            ]
        );
        $this->add_control(
            'h_service_title-c',
            [
                'label' => esc_html__('Title Hover Color', 'techco-core'),
                'type' => Controls_Manager::COLOR,
                'selectors' => [
                    '{{WRAPPER}} .service_block .service_title a:hover' => 'color: {{VALUE}}',
                ],
            ]
        );
        $this->add_group_control(
            Group_Control_Typography::get_type(),
            [
                'name' => 'service_title_typography',
                'selector' => '{{WRAPPER}} .service_block .service_title',
            ]
        );
        $this->add_control(
            'service_link_hr',
            [
                'type' => \Elementor\Controls_Manager::DIVIDER,
            ]
        );
        $this->add_control(
            'link_style',
            [
                'label' => esc_html__('Link Style', 'techco-core'),
                'type' => \Elementor\Controls_Manager::HEADING,
                'separator' => 'before',
            ]
        );
        $this->start_controls_tabs(
            'style_tabs'
        );

        $this->start_controls_tab(
            'style_normal_tab',
            [
                'label' => esc_html__('Normal', 'techco-core'),
            ]
        );

        $this->add_control(
            'service_link-c',
            [
                'label' => esc_html__('Link Color', 'techco-core'),
                'type' => Controls_Manager::COLOR,
                'selectors' => [
                    '{{WRAPPER}} .service_block .icon_block' => 'color: {{VALUE}}',
                ],
            ]
        );
        $this->add_control(
            'link_border_c',
            [
                'label' => esc_html__('Border Color', 'techco-core'),
                'type' => Controls_Manager::COLOR,
                'selectors' => [
                    '{{WRAPPER}} .service_block .icon_block' => 'border-color: {{VALUE}}',
                ],
            ]
        );
        $this->add_control(
            'link_bg-c',
            [
                'label' => esc_html__('Background Color', 'techco-core'),
                'type' => Controls_Manager::COLOR,
                'selectors' => [
                    '{{WRAPPER}} .service_block .icon_block' => 'background-color: {{VALUE}}',
                ],
            ]
        );

        $this->end_controls_tab();

        $this->start_controls_tab(
            'style_hover_tab',
            [
                'label' => esc_html__('Hover', 'techco-core'),
            ]
        );

        $this->add_control(
            'h_service_link-c',
            [
                'label' => esc_html__('Link Color', 'techco-core'),
                'type' => Controls_Manager::COLOR,
                'selectors' => [
                    '{{WRAPPER}} .service_block .icon_block:hover' => 'color: {{VALUE}}',
                ],
            ]
        );
        $this->add_control(
            'h_link_border_c',
            [
                'label' => esc_html__('Border Color', 'techco-core'),
                'type' => Controls_Manager::COLOR,
                'selectors' => [
                    '{{WRAPPER}} .service_block .icon_block:hover' => 'border-color: {{VALUE}}',
                ],
            ]
        );
        $this->add_control(
            'h_link_bg-c',
            [
                'label' => esc_html__('Background Color', 'techco-core'),
                'type' => Controls_Manager::COLOR,
                'selectors' => [
                    '{{WRAPPER}} .service_block .icon_block:hover' => 'background-color: {{VALUE}}',
                ],
            ]
        );
        $this->end_controls_tab();

        $this->end_controls_tabs();

        $this->add_control(
            'tags_style',
            [
                'label' => esc_html__('Tag Style', 'techco-core'),
                'type' => \Elementor\Controls_Manager::HEADING,
                'separator' => 'before',
            ]
        );
        $this->start_controls_tabs(
            'tag_style_tabs'
        );

        $this->start_controls_tab(
            'tags_style_normal_tab',
            [
                'label' => esc_html__('Normal', 'techco-core'),
            ]
        );

        $this->add_control(
            'service_tag-c',
            [
                'label' => esc_html__('Tag Color', 'techco-core'),
                'type' => Controls_Manager::COLOR,
                'selectors' => [
                    '{{WRAPPER}} .service_block .category_btns_group a' => 'color: {{VALUE}}',
                ],
            ]
        );
        $this->add_control(
            'tag_border_c',
            [
                'label' => esc_html__('Border Color', 'techco-core'),
                'type' => Controls_Manager::COLOR,
                'selectors' => [
                    '{{WRAPPER}} .service_block .category_btns_group a' => 'border-color: {{VALUE}}',
                ],
            ]
        );
        $this->add_control(
            'tag_bg-c',
            [
                'label' => esc_html__('Background Color', 'techco-core'),
                'type' => Controls_Manager::COLOR,
                'selectors' => [
                    '{{WRAPPER}} .service_block .category_btns_group a' => 'background-color: {{VALUE}}',
                ],
            ]
        );

        $this->end_controls_tab();

        $this->start_controls_tab(
            'tag_style_hover_tab',
            [
                'label' => esc_html__('Hover', 'techco-core'),
            ]
        );

        $this->add_control(
            'h_service_tag-c',
            [
                'label' => esc_html__('Tag Color', 'techco-core'),
                'type' => Controls_Manager::COLOR,
                'selectors' => [
                    '{{WRAPPER}} .service_block .category_btns_group a:hover' => 'color: {{VALUE}}',
                ],
            ]
        );
        $this->add_control(
            'h_tag_border_c',
            [
                'label' => esc_html__('Border Color', 'techco-core'),
                'type' => Controls_Manager::COLOR,
                'selectors' => [
                    '{{WRAPPER}} .service_block .category_btns_group a:hover' => 'border-color: {{VALUE}}',
                ],
            ]
        );
        $this->add_control(
            'h_tag_bg-c',
            [
                'label' => esc_html__('Background Color', 'techco-core'),
                'type' => Controls_Manager::COLOR,
                'selectors' => [
                    '{{WRAPPER}} .service_block .category_btns_group a:hover' => 'background-color: {{VALUE}}',
                ],
            ]
        );

        $this->end_controls_tab();

        $this->end_controls_tabs();
        $this->end_controls_section();

    }

    protected function render()
    {
        $settings = $this->get_settings_for_display();

            echo '<div class="xb-service-block">';
            if (isset($settings['template']) && !empty($settings['template'])) {
                $service_post = get_post($settings['template']);

                if ($service_post instanceof \WP_Post) {
                    $service_title = esc_html($service_post->post_title);
                    $service_permalink = esc_url(get_permalink($service_post));
                    $service_image_url = get_the_post_thumbnail_url($service_post, 'full');
                    $categories = wp_get_post_terms($service_post->ID, 'service_cat', array('fields' => 'all'));
                    ?>

                    <div class="service_block">
                        <?php if ($service_image_url): ?>
                            <div class="service_image">
                                <img src="<?php echo esc_url($service_image_url); ?>" alt="<?php echo esc_attr($service_title); ?>">
                            </div>
                        <?php endif; ?>
                        <div class="service_content">
                            <h3 class="service_title">
                                <a href="<?php echo $service_permalink; ?>">
                                    <?php echo $service_title; ?>
                                </a>
                            </h3>
                            <div class="links_wrapper">
                                <ul class="category_btns_group unordered_list">
                                    <?php
                                    if (!empty($categories) && !is_wp_error($categories)) {
                                        $count = 0;
                                        foreach ($categories as $category) {
                                            if ($count < 2) {
                                                echo '<li><a href="' . esc_url(get_term_link($category)) . '">' . esc_html($category->name) . '</a></li>';
                                                $count++;
                                            } else {
                                                break;
                                            }
                                        }
                                    }
                                    ?>
                                </ul>
                                <a class="icon_block" href="<?php echo $service_permalink; ?>">
                                    <i class="fa-regular fa-arrow-up-right"></i>
                                </a>
                            </div>
                        </div>
                    </div>
                    <?php
                }
            }
            echo '</div>';

    }

    protected function template_select()
    {
        $type = 'techco_service';
        global $post;
        $args = array('numberposts' => -1, 'post_type' => $type,);
        $posts = get_posts($args);
        $categories = array(
            '' => __('Select', 'techco-core'),
        );
        foreach ($posts as $pn_cat) {
            $categories[$pn_cat->ID] = get_the_title($pn_cat->ID);
        }
        return $categories;
    }

    protected function render_template($id)
    {
        return Plugin::instance()->frontend->get_builder_content_for_display($id);
    }

}


Plugin::instance()->widgets_manager->register(new Techco_Service());