<?php

/**
 * Elementor Single Widget
 * @package techco-core
 * @since 1.0.0
 */

namespace Elementor;

if (!defined('ABSPATH')) exit; // Exit if accessed directly.

class Techco_service_V4 extends Widget_Base
{

    /**
     * Get widget name.
     *
     * Retrieve Elementor widget name.
     *
     * @return string Widget name.
     * @since 1.0.0
     * @access public
     *
     */
    public function get_name()
    {
        return 'int-service-v4';
    }

    /**
     * Get widget title.
     *
     * Retrieve Elementor widget title.
     *
     * @return string Widget title.
     * @since 1.0.0
     * @access public
     *
     */
    public function get_title()
    {
        return esc_html__('Service V4', 'techco-core');
    }

    /**
     * Get widget icon.
     *
     * Retrieve Elementor widget icon.
     *
     * @return string Widget icon.
     * @since 1.0.0
     * @access public
     *
     */
    public function get_icon()
    {
        return 'techco-custom-icon';
    }

    /**
     * Get widget categories.
     *
     * Retrieve the list of categories the Elementor widget belongs to.
     *
     * @return array Widget categories.
     * @since 1.0.0
     * @access public
     *
     */
    public function get_categories()
    {
        return ['techco_widgets'];
    }


    protected function register_controls()
    {


        $this->start_controls_section(
            'service_option',
            [
                'label' => esc_html__('Service List', 'techco-core'),
                'tab' => Controls_Manager::TAB_CONTENT,
            ]
        );

        $repeater = new \Elementor\Repeater();

        $repeater->add_control(
            'template',
            [
                'type' => Controls_Manager::SELECT2,
                'options' => $this->template_select(),
                'multiple' => false,
                'label' => esc_html__('Service', 'techco-core'),
                'label_block' => true,
            ]
        );
        $repeater->add_control(
            'icon',
            [
                'label' => esc_html__('Icon', 'techco-core'),
                'type' => \Elementor\Controls_Manager::ICONS,
            ]
        );
        $repeater->add_control(
            'content',
            [
                'type' => Controls_Manager::TEXTAREA,
                'label' => esc_html__('Content', 'techco-core'),
                'label_block' => true,
            ]
        );

        $this->add_control(
            'lists',
            [
                'type' => Controls_Manager::REPEATER,
                'fields' => $repeater->get_controls(),
                'prevent_empty' => false,
            ]
        );
        $this->add_control(
            'btn_label',
            [
                'type' => Controls_Manager::TEXT,
                'label' => esc_html__('Button Label', 'techco-core'),
                'label_block' => true,
                'default' => esc_html__( 'Find More', 'techco-core' ),
            ]
        );
        $this->add_control(
            'note_label',
            [
                'type' => Controls_Manager::RAW_HTML,
                'raw' => sprintf(
                    '<div class="elementor-panel-alert elementor-panel-alert-info">
                <p>%s</p>
            </div>',
                    sprintf(
                        __('Please note! Customize your selected Service using Elementor. You can find it under <a href="%s" target="_blank">Services</a>.', 'techco-core'),
                        admin_url('edit.php?post_type=techco_service')
                    )
                ),
                'content_classes' => 'elementor-notice-container',
            ]
        );
        $this->end_controls_section();

        $this->start_controls_section(
            'service_style_option',
            [
                'label' => esc_html__('Service Style', 'techco-core'),
                'tab' => Controls_Manager::TAB_STYLE,
            ]
        );
        $this->add_control(
            'bg-c',
            [
                'label' => esc_html__('Background Color', 'techco-core'),
                'type' => Controls_Manager::COLOR,
                'selectors' => [
                    '{{WRAPPER}} .iconbox_block' => 'background-color: {{VALUE}}',
                ],
            ]
        );
        $this->add_control(
            'service_padding',
            [
                'label' => esc_html__('Padding', 'techco-core'),
                'type' => \Elementor\Controls_Manager::DIMENSIONS,
                'size_units' => ['px', '%', 'em', 'rem', 'custom'],
                'selectors' => [
                    '{{WRAPPER}} .iconbox_block' => 'padding: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
                ],
            ]
        );
        $this->add_control(
            'icon_hr',
            [
                'type' => \Elementor\Controls_Manager::DIVIDER,
            ]
        );
        $this->add_control(
            'icon-bg-c',
            [
                'label' => esc_html__('Icon BG Color', 'techco-core'),
                'type' => Controls_Manager::COLOR,
                'selectors' => [
                    '{{WRAPPER}} .iconbox_block .iconbox_icon' => 'background-color: {{VALUE}}',
                ],
            ]
        );
        $this->add_control(
            'icon-border-c',
            [
                'label' => esc_html__('Icon Border Color', 'techco-core'),
                'type' => Controls_Manager::COLOR,
                'selectors' => [
                    '{{WRAPPER}} .iconbox_block .iconbox_icon' => 'border-color: {{VALUE}}',
                ],
            ]
        );
        $this->add_control(
            'title-c',
            [
                'label' => esc_html__('Title Color', 'techco-core'),
                'type' => Controls_Manager::COLOR,
                'selectors' => [
                    '{{WRAPPER}} .iconbox_block .iconbox_title a' => 'color: {{VALUE}}',
                ],
            ]
        );
        $this->add_control(
            'title_hover-c',
            [
                'label' => esc_html__('Title Hover Color', 'techco-core'),
                'type' => Controls_Manager::COLOR,
                'selectors' => [
                    '{{WRAPPER}} .row.business_consulting_services .iconbox_block .iconbox_title a:hover' => 'color: {{VALUE}}',
                ],
            ]
        );
        $this->add_group_control(
            Group_Control_Typography::get_type(),
            [
                'name' => 'title_typography',
                'selector' => '{{WRAPPER}} .iconbox_block .iconbox_title',
            ]
        );
        $this->add_control(
            'content_hr',
            [
                'type' => \Elementor\Controls_Manager::DIVIDER,
            ]
        );
        $this->add_control(
            'content_color',
            [
                'label' => esc_html__('Content Color', 'techco-core'),
                'type' => Controls_Manager::COLOR,
                'selectors' => [
                    '{{WRAPPER}} .iconbox_block p' => 'color: {{VALUE}}',
                ],
            ]
        );
        $this->add_group_control(
            Group_Control_Typography::get_type(),
            [
                'name' => 'content_typography',
                'selector' => '{{WRAPPER}} .iconbox_block p',
            ]
        );
        $this->end_controls_section();

    }

    protected function render()
    {
        $settings = $this->get_settings_for_display();

        if (!empty($settings['lists'])) {
            echo '<div class="container">';
            echo '<div class="sc_service row g-0 mt-none-30">';
            foreach ($settings['lists'] as $a) {
                if (isset($a['template']) && !empty($a['template'])) {
                    $service_post = get_post($a['template']);

                    if ($service_post instanceof \WP_Post) {
                        $service_title = esc_html($service_post->post_title);
                        $service_permalink = esc_url(get_permalink($service_post));
                        ?>
                        <div class="col-lg-3 col-md-6 col-sm-6 mt-30">
                            <div class="sc_service_item">
                                <div class="iconbox_icon">
                                    <?php \Elementor\Icons_Manager::render_icon($a['icon'], ['aria-hidden' => 'true']); ?>
                                </div>
                                <div class="sc_service_content">
                                    <h3 class="sc_service_title">
                                        <a href="<?php echo $service_permalink; ?>">
                                            <?php echo $service_title; ?>
                                        </a>
                                    </h3>
                                    <?php if (!empty($a['content'])): ?>
                                        <p class="mb-0">
                                            <?php echo wp_kses($a['content'], true); ?>
                                        </p>
                                    <?php endif; ?>
                                    <?php if (!empty($settings['btn_label'])): ?>
                                        <a class="sc_service_link" href="<?php echo $service_permalink; ?>"><?php echo esc_html($settings['btn_label']); ?><svg width="16" height="16" viewBox="0 0 16 16" fill="none" xmlns="http://www.w3.org/2000/svg">
                                                <path d="M15.7071 8.70711C16.0976 8.31658 16.0976 7.68342 15.7071 7.29289L9.34315 0.928933C8.95262 0.538409 8.31946 0.538409 7.92893 0.928933C7.53841 1.31946 7.53841 1.95262 7.92893 2.34315L13.5858 8L7.92893 13.6569C7.53841 14.0474 7.53841 14.6805 7.92893 15.0711C8.31946 15.4616 8.95262 15.4616 9.34315 15.0711L15.7071 8.70711ZM-8.74228e-08 9L15 9L15 7L8.74228e-08 7L-8.74228e-08 9Z" fill="#49515B" />
                                            </svg></a>
                                    <?php endif; ?>
                                </div>
                            </div>
                        </div>

                        <?php
                    }
                }
            }
            echo '</div>';
            echo '</div>';
        }
    }

    protected function template_select()
    {
        $type = 'techco_service';
        global $post;
        $args = array('numberposts' => -1, 'post_type' => $type,);
        $posts = get_posts($args);
        $categories = array(
            '' => __('Select', 'techco-core'),
        );
        foreach ($posts as $pn_cat) {
            $categories[$pn_cat->ID] = get_the_title($pn_cat->ID);
        }
        return $categories;
    }

    protected function render_template($id)
    {
        return Plugin::instance()->frontend->get_builder_content_for_display($id);
    }

}


Plugin::instance()->widgets_manager->register(new Techco_service_V4());