<?php

/**
 * Elementor Single Widget
 * @package techco-core
 * @since 1.0.0
 */

namespace Elementor;

if (!defined('ABSPATH')) exit; // Exit if accessed directly.

class Techco_Service_V2 extends Widget_Base
{

    /**
     * Get widget name.
     *
     * Retrieve Elementor widget name.
     *
     * @return string Widget name.
     * @since 1.0.0
     * @access public
     *
     */
    public function get_name()
    {
        return 'int-service-v2';
    }

    /**
     * Get widget title.
     *
     * Retrieve Elementor widget title.
     *
     * @return string Widget title.
     * @since 1.0.0
     * @access public
     *
     */
    public function get_title()
    {
        return esc_html__('Service V2', 'techco-core');
    }

    /**
     * Get widget icon.
     *
     * Retrieve Elementor widget icon.
     *
     * @return string Widget icon.
     * @since 1.0.0
     * @access public
     *
     */
    public function get_icon()
    {
        return 'techco-custom-icon';
    }

    /**
     * Get widget categories.
     *
     * Retrieve the list of categories the Elementor widget belongs to.
     *
     * @return array Widget categories.
     * @since 1.0.0
     * @access public
     *
     */
    public function get_categories()
    {
        return ['techco_widgets'];
    }


    protected function register_controls()
    {

        $this->start_controls_section(
            'service_option',
            [
                'label' => esc_html__('Service', 'techco-core'),
                'tab' => Controls_Manager::TAB_CONTENT,
            ]
        );
        $this->add_control(
            'template',
            [
                'type' => Controls_Manager::SELECT2,
                'options' => $this->template_select(),
                'multiple' => false,
                'label' => esc_html__('Service', 'techco-core'),
                'label_block' => true,
            ]
        );
        $this->add_control(
            'icon',
            [
                'label' => esc_html__('Icon', 'techco-core'),
                'type' => \Elementor\Controls_Manager::ICONS,
            ]
        );
        $repeater = new \Elementor\Repeater();

        $repeater->add_control(
            'list',
            [
                'type' => Controls_Manager::TEXT,
                'label' => esc_html__('List Text', 'techco-core'),
                'label_block' => true,
            ]
        );
        $this->add_control(
            'lists',
            [
                'label' => esc_html__('Add List', 'techco-core'),
                'type' => Controls_Manager::REPEATER,
                'fields' => $repeater->get_controls(),
                'title_field' => '{{{ list }}}',
            ]
        );
        $this->add_control(
            'note_label',
            [
                'type' => Controls_Manager::RAW_HTML,
                'raw' => sprintf(
                    '<div class="elementor-panel-alert elementor-panel-alert-info">
                <p>%s</p>
            </div>',
                    sprintf(
                        __('Please note! Customize your selected Service using Elementor. You can find it under <a href="%s" target="_blank">Services</a>.', 'techco-core'),
                        admin_url('edit.php?post_type=techco_service')
                    )
                ),
                'content_classes' => 'elementor-notice-container',
            ]
        );
        $this->end_controls_section();

        $this->start_controls_section(
            'service_style_option',
            [
                'label' => esc_html__('Style', 'techco-core'),
                'tab' => Controls_Manager::TAB_STYLE,
            ]
        );
        $this->add_control(
            'service_bg-c',
            [
                'label' => esc_html__('Background Color', 'techco-core'),
                'type' => Controls_Manager::COLOR,
                'selectors' => [
                    '{{WRAPPER}} .service_block_2' => 'background-color: {{VALUE}}',
                ],
            ]
        );
        $this->add_control(
            'service_padding',
            [
                'label' => esc_html__('Padding', 'techco-core'),
                'type' => \Elementor\Controls_Manager::DIMENSIONS,
                'size_units' => ['px', '%', 'em', 'rem', 'custom'],
                'selectors' => [
                    '{{WRAPPER}} .service_block_2' => 'padding: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
                ],
            ]
        );
        $this->add_control(
            'icon_hr',
            [
                'type' => \Elementor\Controls_Manager::DIVIDER,
            ]
        );
        $this->add_control(
            'icon_color',
            [
                'label' => esc_html__('Icon Color', 'techco-core'),
                'type' => Controls_Manager::COLOR,
                'selectors' => [
                    '{{WRAPPER}} .service_block_2 .service_icon svg path' => 'fill: {{VALUE}}',
                ],
            ]
        );
        $this->add_control(
            'service_title_hr',
            [
                'type' => \Elementor\Controls_Manager::DIVIDER,
            ]
        );
        $this->add_control(
            'service_title-c',
            [
                'label' => esc_html__('Title Color', 'techco-core'),
                'type' => Controls_Manager::COLOR,
                'selectors' => [
                    '{{WRAPPER}} .service_block_2 .service_title a' => 'color: {{VALUE}}',
                ],
            ]
        );
        $this->add_control(
            'h_service_title-c',
            [
                'label' => esc_html__('Title Hover Color', 'techco-core'),
                'type' => Controls_Manager::COLOR,
                'selectors' => [
                    '{{WRAPPER}} .service_block_2 .service_title a:hover' => 'color: {{VALUE}}',
                    '{{WRAPPER}} .service_block_2 .service_title a:hover::after' => 'color: {{VALUE}}',
                ],
            ]
        );
        $this->add_group_control(
            Group_Control_Typography::get_type(),
            [
                'name' => 'service_title_typography',
                'selector' => '{{WRAPPER}} .service_block_2 .service_title',
            ]
        );
        $this->add_control(
            'service_list_hr',
            [
                'type' => \Elementor\Controls_Manager::DIVIDER,
            ]
        );
        $this->add_control(
            'service_list-c',
            [
                'label' => esc_html__('List Color', 'techco-core'),
                'type' => Controls_Manager::COLOR,
                'selectors' => [
                    '{{WRAPPER}} .service_block_2 .icon_list .icon_list_text' => 'color: {{VALUE}}',
                ],
            ]
        );
        $this->add_group_control(
            Group_Control_Background::get_type(),
            [
                'name' => 'icon_bg_color',
                'types' => ['gradient'],
                'exclude' => ['image'],
                'selector' => '{{WRAPPER}} .service_block_2 .icon_list > li',
                'fields_options' => [
                    'background' => [
                        'label' => esc_html__('Icon BG Color', 'xpress-core'),
                    ]
                ]
            ]
        );
        $this->add_control(
            'icon_dot-c',
            [
                'label' => esc_html__('Icon Dot Color', 'techco-core'),
                'type' => Controls_Manager::COLOR,
                'selectors' => [
                    '{{WRAPPER}} .service_block_2 .icon_list .icon_list_icon' => 'color: {{VALUE}}',
                ],
            ]
        );
        $this->add_group_control(
            Group_Control_Typography::get_type(),
            [
                'name' => 'service_list_typography',
                'selector' => '{{WRAPPER}} .service_block_2 .icon_list > li',
            ]
        );
        $this->end_controls_section();

    }

    protected function render()
    {
        $settings = $this->get_settings_for_display();

        echo '<div class="xb-service-block-v2">';
        if (isset($settings['template']) && !empty($settings['template'])) {
            $service_post = get_post($settings['template']);

            if ($service_post instanceof \WP_Post) {
                $service_title = esc_html($service_post->post_title);
                $service_permalink = esc_url(get_permalink($service_post));
                ?>

                <div class="service_block_2">
                    <div class="service_icon">
                        <?php \Elementor\Icons_Manager::render_icon($settings['icon'], ['aria-hidden' => 'true']); ?>
                    </div>
                    <h3 class="service_title">
                        <a href="<?php echo $service_permalink; ?>">
                            <?php echo $service_title; ?>
                        </a>
                    </h3>
                    <ul class="icon_list unordered_list_block">
                        <?php
                        foreach ($settings['lists'] as $list):
                            ?>
                            <li>
                              <span class="icon_list_icon">
                                <i class="fa-regular fa-circle-dot"></i>
                              </span>
                                <span class="icon_list_text">
                                    <?php echo esc_html($list['list']); ?>
                                  </span>
                            </li>
                        <?php endforeach; ?>
                    </ul>
                </div>
                <?php
            }
        }
        echo '</div>';

    }

    protected function template_select()
    {
        $type = 'techco_service';
        global $post;
        $args = array('numberposts' => -1, 'post_type' => $type,);
        $posts = get_posts($args);
        $categories = array(
            '' => __('Select', 'techco-core'),
        );
        foreach ($posts as $pn_cat) {
            $categories[$pn_cat->ID] = get_the_title($pn_cat->ID);
        }
        return $categories;
    }

    protected function render_template($id)
    {
        return Plugin::instance()->frontend->get_builder_content_for_display($id);
    }

}


Plugin::instance()->widgets_manager->register(new Techco_Service_V2());