<?php

/**
 * Elementor Single Widget
 * @package techco-core
 * @since 1.0.0
 */

namespace Elementor;

if (!defined('ABSPATH')) exit; // Exit if accessed directly.

class Techco_Section_Title extends Widget_Base
{

    /**
     * Get widget name.
     *
     * Retrieve Elementor widget name.
     *
     * @return string Widget name.
     * @since 1.0.0
     * @access public
     *
     */
    public function get_name()
    {
        return 'int-sec-title';
    }

    /**
     * Get widget title.
     *
     * Retrieve Elementor widget title.
     *
     * @return string Widget title.
     * @since 1.0.0
     * @access public
     *
     */
    public function get_title()
    {
        return esc_html__('Section Title', 'techco-core');
    }

    /**
     * Get widget icon.
     *
     * Retrieve Elementor widget icon.
     *
     * @return string Widget icon.
     * @since 1.0.0
     * @access public
     *
     */
    public function get_icon()
    {
        return 'techco-custom-icon';
    }

    /**
     * Get widget categories.
     *
     * Retrieve the list of categories the Elementor widget belongs to.
     *
     * @return array Widget categories.
     * @since 1.0.0
     * @access public
     *
     */
    public function get_categories()
    {
        return ['techco_widgets'];
    }


    protected function register_controls()
    {

        $this->start_controls_section(
            'section_title_option',
            [
                'label' => esc_html__('Section Title', 'techco-core'),
                'tab' => Controls_Manager::TAB_CONTENT,
            ]
        );

        $this->add_control(
            'design_style',
            [
                'label' => __('Design Style', 'techco-core'),
                'type' => Controls_Manager::SELECT,
                'options' => [
                    'style_1' => __('Style 1', 'techco-core'),
                    'style_2' => __('Style 2', 'techco-core'),
                    'style_3' => __('Style 3', 'techco-core'),
                    'style_4' => __('Style 4', 'techco-core'),
                    'style_5' => __('Style 5', 'techco-core'),
                    'style_6' => __('Style 6', 'techco-core'),
                ],
                'default' => 'style_1',
            ]
        );
        $this->add_control(
            'sub_title_img',
            [
                'label' => esc_html__('Sub Title Image', 'techco-core'),
                'type' => \Elementor\Controls_Manager::MEDIA,
                'condition' => [
                    'design_style' => ['style_2'],
                ],
            ]
        );
        $this->add_control(
            'sub_title', [
                'label' => esc_html__('Sub Title', 'techco-core'),
                'type' => Controls_Manager::TEXT,
                'label_block' => true,
                'condition' => [
                    'design_style' => ['style_1', 'style_2', 'style_4', 'style_5', 'style_6'],
                ],
            ]
        );

        $this->add_control(
            'sub_title_img_left',
            [
                'label' => esc_html__('Sub Title Image Left', 'techco-core'),
                'type' => \Elementor\Controls_Manager::MEDIA,
                'condition' => [
                    'design_style' => ['style_4', 'style_5', 'style_6'],
                ],
            ]
        );

        $this->add_control(
            'sub_title_img_right',
            [
                'label' => esc_html__('Sub Title Image Right', 'techco-core'),
                'type' => \Elementor\Controls_Manager::MEDIA,
                'condition' => [
                    'design_style' => ['style_4', 'style_5', 'style_6'],
                ],
            ]
        );
        $this->add_control(
            'sub_title_badge', [
                'label' => esc_html__('Badge', 'techco-core'),
                'type' => Controls_Manager::TEXT,
                'condition' => [
                    'design_style' => ['style_1'],
                ],
            ]
        );
        $this->add_control(
            'badge_align',
            [
                'label' => __('Badge Alignment', 'techco-core'),
                'type' => \Elementor\Controls_Manager::SELECT,
                'options' => [
                    'badge_left' => __('Badge Left', 'techco-core'),
                    'badge_right' => __('Badge Right', 'techco-core'),
                ],
                'default' => 'badge_left',
                'condition' => [
                    'design_style' => ['style_1'],
                ],
            ]
        );
        $this->add_control(
            'title_hr',
            [
                'type' => \Elementor\Controls_Manager::DIVIDER,
            ]
        );
        $this->add_control(
            'title', [
                'label' => esc_html__('Title', 'techco-core'),
                'type' => Controls_Manager::TEXTAREA,
                'label_block' => true,
            ]
        );
        $this->add_control(
            'content_hr',
            [
                'type' => \Elementor\Controls_Manager::DIVIDER,
            ]
        );
        $this->add_control(
            'content', [
                'label' => esc_html__('Content', 'techco-core'),
                'type' => Controls_Manager::TEXTAREA,
                'label_block' => true,
            ]
        );
        $this->add_responsive_control(
            'text_align',
            [
                'label' => esc_html__('Text Align', 'techco-core'),
                'type' => Controls_Manager::CHOOSE,
                'label_block' => false,

                'options' => [
                    'start' => [
                        'title' => __('Left', 'techco-core'),
                        'icon' => 'eicon-text-align-left',
                    ],

                    'center' => [
                        'title' => __('Center', 'techco-core'),
                        'icon' => 'eicon-text-align-center',
                    ],

                    'end' => [
                        'title' => __('Right', 'techco-core'),
                        'icon' => 'eicon-text-align-right',
                    ],
                ],

                'devices' => ['desktop', 'tablet', 'mobile'],

                'selectors' => [
                    '{{WRAPPER}} .heading_block' => 'text-align: {{VALUE}};',
                ],
            ]
        );

        $this->end_controls_section();

        $this->start_controls_section(
            'color_option',
            [
                'label' => esc_html__('Style', 'techco-core'),
                'tab' => Controls_Manager::TAB_STYLE,
            ]
        );
        $this->add_control(
            'sub_title-c',
            [
                'label' => esc_html__('Sub Title Color', 'techco-core'),
                'type' => Controls_Manager::COLOR,
                'selectors' => [
                    '{{WRAPPER}} .heading_focus_text' => 'color: {{VALUE}}',
                    '{{WRAPPER}} .cs_sec_title .subtitle' => 'color: {{VALUE}}',
                    '{{WRAPPER}} .ml_sec_title .subtitle' => 'color: {{VALUE}}',
                ],
            ]
        );
        $this->add_control(
            'badge-c',
            [
                'label' => esc_html__('Badge Color', 'techco-core'),
                'type' => Controls_Manager::COLOR,
                'selectors' => [
                    '{{WRAPPER}} .heading_focus_text .badge' => 'color: {{VALUE}} !important',
                ],
                'condition' => [
                    'design_style' => ['style_1'],
                ],
            ]
        );
        $this->add_control(
            'badge_bg-c',
            [
                'label' => esc_html__('Badge BG Color', 'techco-core'),
                'type' => Controls_Manager::COLOR,
                'selectors' => [
                    '{{WRAPPER}} .heading_focus_text .badge' => 'background-color: {{VALUE}} !important',
                ],
                'condition' => [
                    'design_style' => ['style_1'],
                ],
            ]
        );
        $this->add_group_control(
            Group_Control_Typography::get_type(),
            [
                'name' => 'subtitle_typography',
                'selector' => '{{WRAPPER}} .heading_focus_text, {{WRAPPER}} .heading_focus_text .badge, {{WRAPPER}} .cs_sec_title .subtitle',
            ]
        );
        $this->add_control(
            'title-c',
            [
                'label' => esc_html__('Title Color', 'techco-core'),
                'type' => Controls_Manager::COLOR,
                'selectors' => [
                    '{{WRAPPER}} .heading_block .heading_text' => 'color: {{VALUE}}',
                ],
            ]
        );
        $this->add_control(
            'title-mark-c',
            [
                'label' => esc_html__('Title Mark Color', 'techco-core'),
                'type' => Controls_Manager::COLOR,
                'selectors' => [
                    '{{WRAPPER}} .heading_text mark' => 'color: {{VALUE}}',
                ],
            ]
        );
        $this->add_group_control(
            Group_Control_Typography::get_type(),
            [
                'name' => 'title_typography',
                'selector' => '{{WRAPPER}} .heading_block .heading_text',
            ]
        );
        $this->add_control(
            'title_padding',
            [
                'label' => esc_html__('Title Padding', 'techco-core'),
                'type' => \Elementor\Controls_Manager::DIMENSIONS,
                'size_units' => ['px', '%', 'em', 'rem', 'custom'],
                'selectors' => [
                    '{{WRAPPER}} .heading_block .heading_text' => 'padding: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
                ],
            ]
        );
        $this->add_control(
            'title_margin',
            [
                'label' => esc_html__('Title Margin', 'techco-core'),
                'type' => \Elementor\Controls_Manager::DIMENSIONS,
                'size_units' => ['px', '%', 'em', 'rem', 'custom'],
                'selectors' => [
                    '{{WRAPPER}} .heading_block .heading_text' => 'margin: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
                ],
            ]
        );
        $this->add_control(
            'content_style_hr',
            [
                'type' => \Elementor\Controls_Manager::DIVIDER,
            ]
        );
        $this->add_control(
            'content-c',
            [
                'label' => esc_html__('Content Color', 'techco-core'),
                'type' => Controls_Manager::COLOR,
                'selectors' => [
                    '{{WRAPPER}} .heading_block .heading_description' => 'color: {{VALUE}}',
                ],
            ]
        );
        $this->add_group_control(
            Group_Control_Typography::get_type(),
            [
                'name' => 'content_typography',
                'selector' => '{{WRAPPER}} .heading_block .heading_description',
            ]
        );
        $this->end_controls_section();


    }


    protected function render()
    {
        $settings = $this->get_settings_for_display();
        ?>

        <?php if ($settings['design_style'] === 'style_2'): ?>
        <div class="heading_block">
            <div class="heading_focus_text has_underline d-inline-flex" <?php if (!empty($settings['sub_title_img']['url'])): ?>
                 style="background-image: url('<?php echo esc_url($settings['sub_title_img']['url']); ?>') <?php endif; ?>">
                <?php echo esc_html($settings['sub_title']); ?>
            </div>
            <?php if (!empty($settings['title'])): ?>
                <h2 class="heading_text">
                    <?php echo wp_kses($settings['title'], true); ?>
                </h2>
            <?php endif; ?>
            <?php if (!empty($settings['content'])): ?>
                <p class="heading_description mb-0">
                    <?php echo wp_kses($settings['content'], true); ?>
                </p>
            <?php endif; ?>
        </div>
    <?php elseif ($settings['design_style'] === 'style_3'): ?>
        <div class="heading_block">
            <?php if (!empty($settings['title'])): ?>
                <h2 class="heading_text">
                    <?php echo wp_kses($settings['title'], true); ?>
                </h2>
            <?php endif; ?>
            <?php if (!empty($settings['content'])): ?>
                <p class="heading_description">
                    <?php echo wp_kses($settings['content'], true); ?>
                </p>
            <?php endif; ?>
        </div>
    <?php elseif ($settings['design_style'] === 'style_4'): ?>
        <div class="heading_block cs_sec_title">
            <span class="subtitle"><?php echo wp_get_attachment_image($settings['sub_title_img_left']['id'], 'thumbnail');
                echo esc_html($settings['sub_title']);
                echo wp_get_attachment_image($settings['sub_title_img_right']['id'], 'thumbnail'); ?></span>
            <?php if (!empty($settings['title'])): ?>
                <h2 class="heading_text">
                    <?php echo wp_kses($settings['title'], true); ?>
                </h2>
            <?php endif; ?>
            <?php if (!empty($settings['content'])): ?>
                <p class="heading_description mb-0">
                    <?php echo wp_kses($settings['content'], true); ?>
                </p>
            <?php endif; ?>
        </div>
    <?php elseif ($settings['design_style'] === 'style_5'): ?>
        <div class="heading_block ml_sec_title">
            <span class="subtitle"><?php echo wp_get_attachment_image($settings['sub_title_img_left']['id'], 'thumbnail');
                echo esc_html($settings['sub_title']);
                echo wp_get_attachment_image($settings['sub_title_img_right']['id'], 'thumbnail'); ?></span>
            <?php if (!empty($settings['title'])): ?>
                <h2 class="heading_text">
                    <?php echo wp_kses($settings['title'], true); ?>
                </h2>
            <?php endif; ?>
            <?php if (!empty($settings['content'])): ?>
                <p class="heading_description mb-0">
                    <?php echo wp_kses($settings['content'], true); ?>
                </p>
            <?php endif; ?>
        </div>
    <?php elseif ($settings['design_style'] === 'style_6'): ?>
        <div class="heading_block ml_sec_title ml_sec_title_two">
            <span class="subtitle"><?php echo wp_get_attachment_image($settings['sub_title_img_left']['id'], 'thumbnail');
                echo esc_html($settings['sub_title']);
                echo wp_get_attachment_image($settings['sub_title_img_right']['id'], 'thumbnail'); ?></span>
            <?php if (!empty($settings['title'])): ?>
                <h2 class="heading_text">
                    <?php echo wp_kses($settings['title'], true); ?>
                </h2>
            <?php endif; ?>
            <?php if (!empty($settings['content'])): ?>
                <p class="heading_description mb-0">
                    <?php echo wp_kses($settings['content'], true); ?>
                </p>
            <?php endif; ?>
        </div>
    <?php else: ?>
        <div class="heading_block">
            <div class="heading_focus_text">
                <?php if ($settings['badge_align'] === 'badge_left'): ?>
                    <span class="badge bg-secondary text-white"><?php echo esc_html($settings['sub_title_badge']); ?></span>
                <?php endif;
                echo esc_html($settings['sub_title']);
                if ($settings['badge_align'] === 'badge_right'): ?>
                    <span class="badge bg-secondary text-white"><?php echo esc_html($settings['sub_title_badge']); ?></span>
                <?php endif; ?>
            </div>
            <?php if (!empty($settings['title'])): ?>
                <h2 class="heading_text">
                    <?php echo wp_kses($settings['title'], true); ?>
                </h2>
            <?php endif; ?>
            <?php if (!empty($settings['content'])): ?>
                <p class="heading_description mb-0">
                    <?php echo wp_kses($settings['content'], true); ?>
                </p>
            <?php endif; ?>
        </div>
    <?php
    endif;
    }


}


Plugin::instance()->widgets_manager->register(new Techco_Section_Title());