<?php

/**
 * Elementor Single Widget
 * @package techco-core
 * @since 1.0.0
 */

namespace Elementor;

if (!defined('ABSPATH')) exit; // Exit if accessed directly.

class Techco_Project extends Widget_Base
{

    /**
     * Get widget name.
     *
     * Retrieve Elementor widget name.
     *
     * @return string Widget name.
     * @since 1.0.0
     * @access public
     *
     */
    public function get_name()
    {
        return 'int-project';
    }

    /**
     * Get widget title.
     *
     * Retrieve Elementor widget title.
     *
     * @return string Widget title.
     * @since 1.0.0
     * @access public
     *
     */
    public function get_title()
    {
        return esc_html__('Project', 'techco-core');
    }

    /**
     * Get widget icon.
     *
     * Retrieve Elementor widget icon.
     *
     * @return string Widget icon.
     * @since 1.0.0
     * @access public
     *
     */
    public function get_icon()
    {
        return 'techco-custom-icon';
    }

    /**
     * Get widget categories.
     *
     * Retrieve the list of categories the Elementor widget belongs to.
     *
     * @return array Widget categories.
     * @since 1.0.0
     * @access public
     *
     */
    public function get_categories()
    {
        return ['techco_widgets'];
    }


    protected function register_controls()
    {

        $this->start_controls_section(
            'project_option',
            [
                'label' => esc_html__('Project', 'techco-core'),
                'tab' => Controls_Manager::TAB_CONTENT,
            ]
        );

        $repeater = new \Elementor\Repeater();

        $repeater->add_control(
            'template',
            [
                'type' => Controls_Manager::SELECT2,
                'options' => $this->template_select(),
                'multiple' => false,
                'label' => esc_html__('Project', 'techco-core'),
                'label_block' => true,
            ]
        );

        $repeater->add_control(
            'image',
            [
                'label' => esc_html__('Image', 'techco-core'),
                'type' => \Elementor\Controls_Manager::MEDIA,
            ]
        );

        $this->add_control(
            'lists',
            [
                'type' => Controls_Manager::REPEATER,
                'fields' => $repeater->get_controls(),
                'prevent_empty' => false,
            ]
        );
        $this->add_control(
            'btn_label',
            [
                'type' => Controls_Manager::TEXT,
                'label' => esc_html__('Button Label', 'techco-core'),
                'default' => esc_html__('Explore', 'techco-core'),
                'label_block' => true,
            ]
        );
        $this->add_control(
            'note_label',
            [
                'type' => Controls_Manager::RAW_HTML,
                'raw' => sprintf(
                    '<div class="elementor-panel-alert elementor-panel-alert-info">
                <p>%s</p>
            </div>',
                    sprintf(
                        __('Please note! Customize your selected Project using Elementor. You can find it under <a href="%s" target="_blank">Projects</a>.', 'techco-core'),
                        admin_url('edit.php?post_type=techco_project')
                    )
                ),
                'content_classes' => 'elementor-notice-container',
            ]
        );
        $this->end_controls_section();


        $this->start_controls_section(
            'project_style_option',
            [
                'label' => esc_html__('Project Style', 'techco-core'),
                'tab' => Controls_Manager::TAB_STYLE,
            ]
        );
        $this->add_control(
            'bg-c',
            [
                'label' => esc_html__('Background Color', 'techco-core'),
                'type' => Controls_Manager::COLOR,
                'selectors' => [
                    '{{WRAPPER}} .portfolio_block' => 'background-color: {{VALUE}}',
                ],
            ]
        );
        $this->add_control(
            'project_padding',
            [
                'label' => esc_html__('Padding', 'techco-core'),
                'type' => \Elementor\Controls_Manager::DIMENSIONS,
                'size_units' => ['px', '%', 'em', 'rem', 'custom'],
                'selectors' => [
                    '{{WRAPPER}} .portfolio_block .portfolio_content' => 'padding: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
                ],
            ]
        );

        $this->add_control(
            'title_hr',
            [
                'type' => \Elementor\Controls_Manager::DIVIDER,
            ]
        );
        $this->add_control(
            'title_color',
            [
                'label' => esc_html__('Title Color', 'techco-core'),
                'type' => Controls_Manager::COLOR,
                'selectors' => [
                    '{{WRAPPER}} .portfolio_block .portfolio_title a' => 'color: {{VALUE}}',
                ],
            ]
        );
        $this->add_control(
            'title_hover_color',
            [
                'label' => esc_html__('Title Hover Color', 'techco-core'),
                'type' => Controls_Manager::COLOR,
                'selectors' => [
                    '{{WRAPPER}} .portfolio_block .portfolio_title a:hover' => 'color: {{VALUE}}',
                ],
            ]
        );
        $this->add_group_control(
            Group_Control_Typography::get_type(),
            [
                'name' => 'title_typography',
                'selector' => '{{WRAPPER}} .portfolio_block .portfolio_title',
            ]
        );

        $this->add_control(
            'cat_hr',
            [
                'type' => \Elementor\Controls_Manager::DIVIDER,
            ]
        );
        $this->add_control(
            'cat_c',
            [
                'label' => esc_html__('Category Color', 'techco-core'),
                'type' => Controls_Manager::COLOR,
                'selectors' => [
                    '{{WRAPPER}} .category_list a' => 'color: {{VALUE}}',
                ],
            ]
        );
        $this->add_control(
            'cat_hover_c',
            [
                'label' => esc_html__('Category Hover Color', 'techco-core'),
                'type' => Controls_Manager::COLOR,
                'selectors' => [
                    '{{WRAPPER}} .category_list a:hover' => 'color: {{VALUE}}',
                ],
            ]
        );
        $this->add_group_control(
            Group_Control_Typography::get_type(),
            [
                'name' => 'cat_typography',
                'selector' => '{{WRAPPER}} .category_list a',
            ]
        );
        $this->add_control(
            'button_style',
            [
                'label' => esc_html__('Button Style', 'techco-core'),
                'type' => \Elementor\Controls_Manager::HEADING,
                'separator' => 'before',
            ]
        );
        $this->add_control(
            'btn_padding',
            [
                'label' => esc_html__('Padding', 'techco-core'),
                'type' => \Elementor\Controls_Manager::DIMENSIONS,
                'size_units' => ['px', '%', 'em', 'rem', 'custom'],
                'selectors' => [
                    '{{WRAPPER}} .btn' => 'padding: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
                ],
            ]
        );
        $this->add_control(
            'btn_radius',
            [
                'label' => esc_html__('Border Radius', 'techco-core'),
                'type' => \Elementor\Controls_Manager::DIMENSIONS,
                'size_units' => ['px', '%', 'em', 'rem', 'custom'],
                'selectors' => [
                    '{{WRAPPER}} .btn' => 'border-radius: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
                ],
            ]
        );
        $this->add_group_control(
            Group_Control_Typography::get_type(),
            [
                'name' => 'btn_typography',
                'selector' => '{{WRAPPER}} .btn',
            ]
        );

        $this->start_controls_tabs(
            'style_tabs'
        );

        $this->start_controls_tab(
            'style_normal_tab',
            [
                'label' => esc_html__('Normal', 'techco-core'),
            ]
        );

        $this->add_control(
            'btn_color',
            [
                'label' => esc_html__('Button Color', 'techco-core'),
                'type' => Controls_Manager::COLOR,
                'selectors' => [
                    '{{WRAPPER}} .btn' => 'color: {{VALUE}}',
                ],
            ]
        );
        $this->add_control(
            'btn_border_color',
            [
                'label' => esc_html__('Button Border Color', 'techco-core'),
                'type' => Controls_Manager::COLOR,
                'selectors' => [
                    '{{WRAPPER}} .btn' => 'border-color: {{VALUE}}',
                ],
            ]
        );
        $this->add_control(
            'btn_bg_color',
            [
                'label' => esc_html__('Background Color', 'techco-core'),
                'type' => Controls_Manager::COLOR,
                'selectors' => [
                    '{{WRAPPER}} .btn' => 'background-color: {{VALUE}}',
                ],
            ]
        );

        $this->end_controls_tab();

        $this->start_controls_tab(
            'style_hover_tab',
            [
                'label' => esc_html__('Hover', 'techco-core'),
            ]
        );

        $this->add_control(
            'h_btn_color',
            [
                'label' => esc_html__('Button Color', 'techco-core'),
                'type' => Controls_Manager::COLOR,
                'selectors' => [
                    '{{WRAPPER}} .btn' => 'color: {{VALUE}}',
                ],
            ]
        );
        $this->add_control(
            'h_border_c',
            [
                'label' => esc_html__('Border Color', 'techco-core'),
                'type' => Controls_Manager::COLOR,
                'selectors' => [
                    '{{WRAPPER}} .btn' => 'border-color: {{VALUE}}',
                ],
            ]
        );
        $this->add_control(
            'hbtn_bg_color',
            [
                'label' => esc_html__('Background Color', 'techco-core'),
                'type' => Controls_Manager::COLOR,
                'selectors' => [
                    '{{WRAPPER}} .btn:hover' => 'background-color: {{VALUE}}',
                ],
            ]
        );
        $this->end_controls_tab();
        $this->end_controls_tabs();

        $this->add_control(
            'bullet_style',
            [
                'label' => esc_html__('Bullet Style', 'techco-core'),
                'type' => \Elementor\Controls_Manager::HEADING,
                'separator' => 'before',
            ]
        );
        $this->add_control(
            'bullet-c',
            [
                'label' => esc_html__('Bullet Color', 'techco-core'),
                'type' => Controls_Manager::COLOR,
                'selectors' => [
                    '{{WRAPPER}} .swiper-pagination-bullets .swiper-pagination-bullet' => 'background-color: {{VALUE}}',
                ],
            ]
        );
        $this->add_control(
            'bullet-active-c',
            [
                'label' => esc_html__('Bullet Active Color', 'techco-core'),
                'type' => Controls_Manager::COLOR,
                'selectors' => [
                    '{{WRAPPER}} .swiper-pagination-bullets .swiper-pagination-bullet.swiper-pagination-bullet-active' => 'background-color: {{VALUE}}',
                ],
            ]
        );
        $this->end_controls_section();

    }

    protected function render()
    {
        $settings = $this->get_settings_for_display();

        if (!empty($settings['lists'])) {
            echo '<div class="portfolio_carousel">';
            echo '<div class="swiper-wrapper">';
            foreach ($settings['lists'] as $a) {
                if (isset($a['template']) && !empty($a['template'])) {
                    $project_post = get_post($a['template']);

                    if ($project_post instanceof \WP_Post) {
                        $project_title = esc_html($project_post->post_title);
                        $project_permalink = esc_url(get_permalink($project_post));
                        $project_image_url = get_the_post_thumbnail_url($project_post, 'full');
                        $project_cats = get_the_terms($project_post->ID, 'project_cat');
                        ?>

                        <div class="swiper-slide">
                            <div class="portfolio_block">
                                <?php if ($project_image_url): ?>
                                    <div class="portfolio_image">
                                        <a class="portfolio_image_wrap bg-light"
                                           href="<?php echo $project_permalink; ?>">
                                            <img src="<?php echo esc_url($project_image_url); ?>"
                                                 alt="<?php echo esc_attr($project_title); ?>">
                                        </a>
                                    </div>
                                <?php endif; ?>
                                <div class="portfolio_content">
                                    <h3 class="portfolio_title">
                                        <a href="<?php echo $project_permalink; ?>">
                                            <?php echo $project_title; ?>
                                        </a>
                                    </h3>
                                    <?php if ($project_cats && !is_wp_error($project_cats)): ?>
                                        <ul class="category_list unordered_list">
                                            <?php foreach ($project_cats as $cat): ?>
                                                <li>
                                                    <a href="<?php echo esc_url(get_term_link($cat)); ?>">
                                                        <?php echo esc_html($cat->name); ?>
                                                    </a>
                                                </li>
                                            <?php endforeach; ?>
                                        </ul>
                                    <?php endif; ?>
                                    <?php if (!empty($settings['btn_label'])): ?>
                                        <a class="btn btn-outline-light" href="<?php echo $project_permalink; ?>">
                                            <span class="btn_label" data-text="<?php echo esc_attr($settings['btn_label']); ?>"><?php echo esc_html($settings['btn_label']); ?></span>
                                            <span class="btn_icon">
                                            <i class="fa-solid fa-arrow-up-right"></i>
                                          </span>
                                        </a>
                                    <?php endif; ?>
                                </div>
                            </div>
                        </div>
                        <?php
                    }
                }
            }
            echo '</div>';
            echo '<div class="portfolio-swiper-pagination"></div>';
            echo '</div>';
        }
    }

    protected function template_select()
    {
        $type = 'techco_project';
        global $post;
        $args = array('numberposts' => -1, 'post_type' => $type,);
        $posts = get_posts($args);
        $categories = array(
            '' => __('Select', 'techco-core'),
        );
        foreach ($posts as $pn_cat) {
            $categories[$pn_cat->ID] = get_the_title($pn_cat->ID);
        }
        return $categories;
    }

    protected function render_template($id)
    {
        return Plugin::instance()->frontend->get_builder_content_for_display($id);
    }

}


Plugin::instance()->widgets_manager->register(new Techco_Project());