<?php

/**
 * Elementor Single Widget
 * @package techco-core
 * @since 1.0.0
 */

namespace Elementor;

if (!defined('ABSPATH')) exit; // Exit if accessed directly.

class Techco_Project_V3 extends Widget_Base
{

    /**
     * Get widget name.
     *
     * Retrieve Elementor widget name.
     *
     * @return string Widget name.
     * @since 1.0.0
     * @access public
     *
     */
    public function get_name()
    {
        return 'int-project-v3';
    }

    /**
     * Get widget title.
     *
     * Retrieve Elementor widget title.
     *
     * @return string Widget title.
     * @since 1.0.0
     * @access public
     *
     */
    public function get_title()
    {
        return esc_html__('Project V3', 'techco-core');
    }

    /**
     * Get widget icon.
     *
     * Retrieve Elementor widget icon.
     *
     * @return string Widget icon.
     * @since 1.0.0
     * @access public
     *
     */
    public function get_icon()
    {
        return 'techco-custom-icon';
    }

    /**
     * Get widget categories.
     *
     * Retrieve the list of categories the Elementor widget belongs to.
     *
     * @return array Widget categories.
     * @since 1.0.0
     * @access public
     *
     */
    public function get_categories()
    {
        return ['techco_widgets'];
    }


    protected function register_controls()
    {


        $this->start_controls_section(
            'project_option',
            [
                'label' => esc_html__('Project', 'techco-core'),
                'tab' => Controls_Manager::TAB_CONTENT,
            ]
        );
        $this->add_control(
            'projectorder',
            [
                'label' => esc_html__('Project Order', 'techco-core'),
                'type' => Controls_Manager::SELECT,
                'default' => 'ASC',
                'options' => [
                    'ASC' => esc_html__('Ascending', 'techco-core'),
                    'DESC' => esc_html__('Descending', 'techco-core'),
                ],
            ]
        );
        $this->add_control(
            'project_per_page',
            [
                'label' => __('Project Per Page', 'techco-core'),
                'type' => Controls_Manager::NUMBER,
                'min' => 1,
                'default' => 9,
            ]
        );
        $this->add_control(
            'project_categories',
            [
                'type' => Controls_Manager::SELECT2,
                'label' => esc_html__('Select Project Categories', 'techco-core'),
                'options' => techco_project_category(),
                'label_block' => true,
                'multiple' => true,
            ]
        );
        $this->end_controls_section();

        $this->start_controls_section(
            'project_cat_style',
            [
                'label' => esc_html__('Project Category', 'techco-core'),
                'tab' => Controls_Manager::TAB_STYLE,
            ]
        );

        $this->add_control(
            'nav_padding',
            [
                'label' => esc_html__('Nav Padding', 'techco-core'),
                'type' => \Elementor\Controls_Manager::DIMENSIONS,
                'size_units' => ['px', '%', 'em', 'rem', 'custom'],
                'selectors' => [
                    '{{WRAPPER}} .filter_elements_nav' => 'padding: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
                ],
            ]
        );
        $this->add_group_control(
            Group_Control_Typography::get_type(),
            [
                'name' => 'nav_typography',
                'selector' => '{{WRAPPER}} .filter_elements_nav > button',
            ]
        );

        $this->start_controls_tabs(
            'style_tabs'
        );

        $this->start_controls_tab(
            'style_normal_tab',
            [
                'label' => esc_html__('Normal', 'techco-core'),
            ]
        );

        $this->add_control(
            'nav-c',
            [
                'label' => esc_html__('Nav Color', 'techco-core'),
                'type' => Controls_Manager::COLOR,
                'selectors' => [
                    '{{WRAPPER}} .filter_elements_nav > button' => 'color: {{VALUE}}',
                ],
            ]
        );
        $this->add_control(
            'nav-bg-c',
            [
                'label' => esc_html__('Nav BG Color', 'techco-core'),
                'type' => Controls_Manager::COLOR,
                'selectors' => [
                    '{{WRAPPER}} .filter_elements_nav > button' => 'background-color: {{VALUE}}',
                ],
            ]
        );

        $this->end_controls_tab();

        $this->start_controls_tab(
            'style_hover_tab',
            [
                'label' => esc_html__('Hover', 'techco-core'),
            ]
        );

        $this->add_control(
            'h-nav-c',
            [
                'label' => esc_html__('Nav Color', 'techco-core'),
                'type' => Controls_Manager::COLOR,
                'selectors' => [
                    '{{WRAPPER}} .filter_elements_nav > button.active' => 'color: {{VALUE}}',
                    '{{WRAPPER}} .filter_elements_nav > button:hover' => 'color: {{VALUE}}',
                ],
            ]
        );
        $this->add_control(
            'h-nav-bg-c',
            [
                'label' => esc_html__('Nav BG Color', 'techco-core'),
                'type' => Controls_Manager::COLOR,
                'selectors' => [
                    '{{WRAPPER}} .filter_elements_nav > button.active' => 'background-color: {{VALUE}}',
                    '{{WRAPPER}} .filter_elements_nav > button:hover' => 'background-color: {{VALUE}}',
                ],
            ]
        );

        $this->end_controls_tab();

        $this->end_controls_tabs();

        $this->end_controls_section();

        $this->start_controls_section(
            'project_list_style',
            [
                'label' => esc_html__('Project List', 'techco-core'),
                'tab' => Controls_Manager::TAB_STYLE,
            ]
        );
        $this->add_control(
            'bg-c',
            [
                'label' => esc_html__('Background Color', 'techco-core'),
                'type' => Controls_Manager::COLOR,
                'selectors' => [
                    '{{WRAPPER}} .portfolio_block' => 'background-color: {{VALUE}}',
                ],
            ]
        );
        $this->add_control(
            'port_padding',
            [
                'label' => esc_html__('Padding', 'techco-core'),
                'type' => \Elementor\Controls_Manager::DIMENSIONS,
                'size_units' => ['px', '%', 'em', 'rem', 'custom'],
                'selectors' => [
                    '{{WRAPPER}} .portfolio_block' => 'padding: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
                ],
            ]
        );
        $this->add_control(
            'port_cat_hr',
            [
                'type' => \Elementor\Controls_Manager::DIVIDER,
            ]
        );
        $this->add_control(
            'cat_color',
            [
                'label' => esc_html__('Category Color', 'techco-core'),
                'type' => Controls_Manager::COLOR,
                'selectors' => [
                    '{{WRAPPER}} .category_list a' => 'color: {{VALUE}}',
                ],
            ]
        );
        $this->add_control(
            'cat_hover_color',
            [
                'label' => esc_html__('Category Hover Color', 'techco-core'),
                'type' => Controls_Manager::COLOR,
                'selectors' => [
                    '{{WRAPPER}} .category_list a:hover' => 'color: {{VALUE}}',
                ],
            ]
        );
        $this->add_group_control(
            Group_Control_Typography::get_type(),
            [
                'name' => 'cat_typography',
                'selector' => '{{WRAPPER}} .category_list a',
            ]
        );
        $this->add_control(
            'port_title_hr',
            [
                'type' => \Elementor\Controls_Manager::DIVIDER,
            ]
        );
        $this->add_control(
            'title_color',
            [
                'label' => esc_html__('Title Color', 'techco-core'),
                'type' => Controls_Manager::COLOR,
                'selectors' => [
                    '{{WRAPPER}} .portfolio_block .portfolio_title a' => 'color: {{VALUE}}',
                ],
            ]
        );
        $this->add_control(
            'title_hover_color',
            [
                'label' => esc_html__('Title Hover Color', 'techco-core'),
                'type' => Controls_Manager::COLOR,
                'selectors' => [
                    '{{WRAPPER}} .portfolio_block .portfolio_title a:hover' => 'color: {{VALUE}}',
                ],
            ]
        );
        $this->add_group_control(
            Group_Control_Typography::get_type(),
            [
                'name' => 'title_typography',
                'selector' => '{{WRAPPER}} .portfolio_block .portfolio_title',
            ]
        );
        $this->end_controls_section();

    }


    protected function render()
    {
        $settings = $this->get_settings_for_display();
        $selected_categories = !empty($settings['project_categories']) ? $settings['project_categories'] : [];

        if (!empty($selected_categories)) {
            $projectCategory = get_terms([
                'taxonomy' => 'project_cat',
                'include' => $selected_categories,
            ]);
        } else {
            $projectCategory = get_terms('project_cat');
        }

        ?>
        <div class="text-center">
            <div class="filter_elements_nav project-menu mb-70 text-center">
                <button class="active" data-filter="*"><?php esc_html_e('SEE ALL', 'techco-core') ?></button>
                <?php
                if (!empty($projectCategory) && !is_wp_error($projectCategory)):
                    foreach ($projectCategory as $cate):
                        ?>
                        <button data-filter=".<?php echo esc_attr($cate->slug) ?>"
                                class=""><?php echo esc_html($cate->name) ?></button>
                    <?php endforeach; endif; ?>
            </div>
        </div>
        <div class="row grid mt-none-30">
            <?php
            $args = array(
                'post_type' => 'techco_project',
                'posts_per_page' => !empty($settings['project_per_page']) ? $settings['project_per_page'] : 1,
                'ignore_sticky_posts' => true,
                'post_status' => 'publish',
                'suppress_filters' => false,
                'order' => $settings['projectorder'],
            );

            if (!empty($selected_categories)) {
                $args['tax_query'] = array(
                    array(
                        'taxonomy' => 'project_cat',
                        'field' => 'term_id',
                        'terms' => $selected_categories,
                    ),
                );
            }

            $query = new \WP_Query($args);
            if ($query->have_posts()):

                while ($query->have_posts()) {
                    $query->the_post();
                    $idd = get_the_ID();

                    $projectInerCates = get_the_terms($idd, 'project_cat');
                    if ($projectInerCates && !is_wp_error($projectInerCates)) {
                        $project_cat_list = array();
                        foreach ($projectInerCates as $cate) {
                            $project_cat_list[] = $cate->slug;
                        }
                        $project_cate_asign_list = join(' ', $project_cat_list);
                    } else {
                        $project_cate_asign_list = '';
                    }

                    ?>
                    <div class="col-lg-6 grid-item mt-30 <?php echo esc_attr($project_cate_asign_list) ?>">
                        <div class="portfolio_block portfolio_layout_2">
                            <div class="portfolio_image">
                                <a href="<?php the_permalink(); ?>"><?php the_post_thumbnail('techco-project-2'); ?></a>
                            </div>
                            <div class="portfolio_content">
                                <h3 class="portfolio_title">
                                    <a href="<?php the_permalink(); ?>"><?php the_title(); ?></a>
                                </h3>
                                <ul class="category_list unordered_list">
                                    <?php
                                    $projectInerCates = get_the_terms(get_the_ID(), 'project_cat');
                                    if ($projectInerCates && !is_wp_error($projectInerCates)) :
                                        foreach ($projectInerCates as $cate) :
                                            ?>
                                            <li>
                                                <a href="<?php echo esc_url(get_term_link($cate)); ?>">
                                                    <i class="fa-solid fa-tags"></i><?php echo esc_html($cate->name); ?>
                                                </a>
                                            </li>
                                        <?php
                                        endforeach;
                                    endif;
                                    ?>
                                </ul>
                            </div>
                        </div>
                    </div>
                    <?php
                }
                wp_reset_query();
            endif;
            ?>
        </div>
        <?php
    }


}


Plugin::instance()->widgets_manager->register(new Techco_Project_V3());