<?php

/**
 * Elementor Single Widget
 * @package techco-core
 * @since 1.0.0
 */

namespace Elementor;

if (!defined('ABSPATH')) exit; // Exit if accessed directly.

class Techco_Feature_V4 extends Widget_Base
{

    /**
     * Get widget name.
     *
     * Retrieve Elementor widget name.
     *
     * @return string Widget name.
     * @since 1.0.0
     * @access public
     *
     */
    public function get_name()
    {
        return 'int-feature-v4';
    }

    /**
     * Get widget title.
     *
     * Retrieve Elementor widget title.
     *
     * @return string Widget title.
     * @since 1.0.0
     * @access public
     *
     */
    public function get_title()
    {
        return esc_html__('Feature V4', 'techco-core');
    }

    /**
     * Get widget icon.
     *
     * Retrieve Elementor widget icon.
     *
     * @return string Widget icon.
     * @since 1.0.0
     * @access public
     *
     */
    public function get_icon()
    {
        return 'techco-custom-icon';
    }

    /**
     * Get widget categories.
     *
     * Retrieve the list of categories the Elementor widget belongs to.
     *
     * @return array Widget categories.
     * @since 1.0.0
     * @access public
     *
     */
    public function get_categories()
    {
        return ['techco_widgets'];
    }


    protected function register_controls()
    {

        $this->start_controls_section(
            'section_title_option',
            [
                'label' => esc_html__('Sticky Content', 'techco-core'),
                'tab' => Controls_Manager::TAB_CONTENT,
            ]
        );
        $this->add_control(
            'sub_title', [
                'label' => esc_html__('Sub Title', 'techco-core'),
                'type' => Controls_Manager::TEXT,
                'label_block' => true,
            ]
        );
        $this->add_control(
            'sub_title_img',
            [
                'label' => esc_html__('Sub Title Image', 'techco-core'),
                'type' => \Elementor\Controls_Manager::MEDIA,
            ]
        );
        $this->add_control(
            'title_hr',
            [
                'type' => \Elementor\Controls_Manager::DIVIDER,
            ]
        );
        $this->add_control(
            'title', [
                'label' => esc_html__('Title', 'techco-core'),
                'type' => Controls_Manager::TEXTAREA,
                'label_block' => true,
            ]
        );
        $this->add_control(
            'btn_hr',
            [
                'label' => esc_html__('Button', 'textdomain'),
                'type' => \Elementor\Controls_Manager::HEADING,
                'separator' => 'before',
            ]
        );
        $this->add_control(
            'btn_label',
            [
                'type' => Controls_Manager::TEXT,
                'label' => esc_html__('Button Label', 'textdomain'),
                'label_block' => true,
            ]
        );
        $this->add_control(
            'btn_url',
            [
                'label' => esc_html__('Button Link', 'textdomain'),
                'type' => \Elementor\Controls_Manager::URL,
                'options' => ['url'],
                'default' => [
                    'url' => '#',
                ],
                'label_block' => true,
            ]
        );
        $this->end_controls_section();

        $this->start_controls_section(
            'feature_list_option',
            [
                'label' => esc_html__('Feature List', 'techco-core'),
                'tab' => Controls_Manager::TAB_CONTENT,
            ]
        );
        $repeater = new \Elementor\Repeater();

        $repeater->add_control(
            'icon',
            [
                'label' => esc_html__('Icon', 'textarea'),
                'type' => \Elementor\Controls_Manager::ICONS,
            ]
        );
        $repeater->add_control(
            'title',
            [
                'type' => Controls_Manager::TEXTAREA,
                'label' => esc_html__('Title', 'purefit-core'),
                'label_block' => true,
            ]
        );
        $repeater->add_control(
            'content',
            [
                'type' => Controls_Manager::TEXTAREA,
                'label' => esc_html__('Content', 'purefit-core'),
                'label_block' => true,
            ]
        );
        $repeater->add_control(
            'image',
            [
                'label' => esc_html__('Image', 'textdomain'),
                'type' => \Elementor\Controls_Manager::MEDIA,
            ]
        );
        $repeater->add_control(
            'blur',
            [
                'label' => esc_html__('Blur Image', 'textdomain'),
                'type' => \Elementor\Controls_Manager::MEDIA,
            ]
        );
        $this->add_control(
            'lists',
            [
                'type' => Controls_Manager::REPEATER,
                'fields' => $repeater->get_controls(),
                'title_field' => '{{{ title }}}',
            ]
        );
        $this->end_controls_section();

        $this->start_controls_section(
            'feature_style',
            [
                'label' => esc_html__('Style', 'techco-core'),
                'tab' => Controls_Manager::TAB_STYLE,
            ]
        );
        $this->add_control(
            'title-c',
            [
                'label' => esc_html__('Title Color', 'techco-core'),
                'type' => Controls_Manager::COLOR,
                'selectors' => [
                    '{{WRAPPER}} .sc-feature h2' => 'color: {{VALUE}}',
                ],
            ]
        );
        $this->add_group_control(
            Group_Control_Typography::get_type(),
            [
                'name' => 'title_typography',
                'selector' => '{{WRAPPER}} .sc-feature h2',
            ]
        );
        $this->add_control(
            'content_hr',
            [
                'type' => \Elementor\Controls_Manager::DIVIDER,
            ]
        );
        $this->add_control(
            'content-c',
            [
                'label' => esc_html__('Content Color', 'techco-core'),
                'type' => Controls_Manager::COLOR,
                'selectors' => [
                    '{{WRAPPER}} .sc-feature p' => 'color: {{VALUE}}',
                ],
            ]
        );
        $this->add_group_control(
            Group_Control_Typography::get_type(),
            [
                'name' => 'content_typography',
                'selector' => '{{WRAPPER}} .sc-feature p',
            ]
        );
        $this->end_controls_section();

        $this->start_controls_section(
            'shape_option',
            [
                'label' => esc_html__('Shape Option', 'techco-core'),
                'tab' => Controls_Manager::TAB_CONTENT,
            ]
        );
        $this->add_control(
            'shape',
            [
                'label' => esc_html__('Shape Image', 'textdomain'),
                'type' => \Elementor\Controls_Manager::MEDIA,
            ]
        );
        $this->end_controls_section();

    }

    protected function render()
    {
        $settings = $this->get_settings_for_display();
        ?>
        <div class="ml_feature">
            <div class="row">
                <div class="col-lg-5">
                    <div class="ml_feature_title">
                        <div class="heading_block ml_sec_title mb-55">
                            <span class="subtitle"><?php echo esc_html($settings['sub_title']);
                                echo wp_get_attachment_image($settings['sub_title_img']['id'], 'thumbnail'); ?></span>
                            <?php if (!empty($settings['title'])): ?>
                                <h2 class="title">
                                    <?php echo wp_kses($settings['title'], true); ?>
                                </h2>
                            <?php endif; ?>
                        </div>
                        <?php if (!empty($settings['btn_label'])): ?>
                            <div class="ml_feature_btn">
                                <div class="button-item grow">
                                    <a href="<?php echo esc_html($settings['btn_url']['url']); ?>"
                                       class="button primary grow w-inline-block">
                                        <div class="semibold-s button-text"><?php echo esc_html($settings['btn_label']); ?>
                                            <span><svg width="10"
                                                       height="10"
                                                       viewBox="0 0 10 10"
                                                       fill="none"
                                                       xmlns="http://www.w3.org/2000/svg">
  <path d="M9.75 1C9.75 0.585787 9.41421 0.25 9 0.25H2.25C1.83579 0.25 1.5 0.585787 1.5 1C1.5 1.41421 1.83579 1.75 2.25 1.75H8.25V7.75C8.25 8.16421 8.58579 8.5 9 8.5C9.41421 8.5 9.75 8.16421 9.75 7.75V1ZM1.53033 9.53033L9.53033 1.53033L8.46967 0.46967L0.46967 8.46967L1.53033 9.53033Z"
        fill="white"/>
</svg></span></div>
                                        <div class="button-linear-primary"></div>
                                    </a>
                                    <div class="glow-border button-border"></div>
                                </div>
                            </div>
                        <?php endif; ?>
                    </div>
                </div>
                <div class="col-lg-7">
                    <div class="ml_feature_list">
                        <?php
                        foreach ($settings['lists'] as $list):
                            ?>
                            <div class="ml-icon-card-content">
                                <div class="card-shape-item">
                                    <div class="card-shape-wrapper">
                                        <?php \Elementor\Icons_Manager::render_icon($list['icon'], ['aria-hidden' => 'true']); ?>
                                    </div>
                                    <div class="glow-border card-shape-border"></div>
                                </div>
                                <div class="ml-icon-card-text-content">
                                    <?php if (!empty($list['title'])): ?>
                                        <h4>
                                            <?php echo wp_kses($list['title'], true); ?>
                                        </h4>
                                    <?php endif; ?>
                                    <?php if (!empty($list['content'])): ?>
                                        <p><?php echo wp_kses($list['content'], true); ?></p>
                                    <?php endif; ?>
                                </div>
                                <?php if (!empty($list['image']['id'])): ?>
                                    <div class="ml_card_img">
                                        <?php echo wp_get_attachment_image($list['image']['id'], 'full'); ?>
                                    </div>
                                <?php endif; ?>
                                <?php if (!empty($list['blur']['id'])): ?>
                                    <div class="ml_card_blur">
                                        <?php echo wp_get_attachment_image($list['blur']['id'], 'full'); ?>
                                    </div>
                                <?php endif; ?>
                            </div>
                        <?php endforeach; ?>
                    </div>
                </div>
            </div>
            <?php if(!empty($settings['shape']['id'])): ?>
            <div class="ml_feature_shape">
                <?php echo wp_get_attachment_image($settings['shape']['id'], 'full'); ?>
            </div>
            <?php endif; ?>
        </div>

        <?php
    }


}


Plugin::instance()->widgets_manager->register(new Techco_Feature_V4());