<?php

/**
 * Elementor Single Widget
 * @package techco-core
 * @since 1.0.0
 */

namespace Elementor;

if (!defined('ABSPATH')) exit; // Exit if accessed directly.

class Techco_Brand_V2 extends Widget_Base
{

    /**
     * Get widget name.
     *
     * Retrieve Elementor widget name.
     *
     * @return string Widget name.
     * @since 1.0.0
     * @access public
     *
     */
    public function get_name()
    {
        return 'int-brand-v2';
    }

    /**
     * Get widget title.
     *
     * Retrieve Elementor widget title.
     *
     * @return string Widget title.
     * @since 1.0.0
     * @access public
     *
     */
    public function get_title()
    {
        return esc_html__('Brand V2', 'techco-core');
    }

    /**
     * Get widget icon.
     *
     * Retrieve Elementor widget icon.
     *
     * @return string Widget icon.
     * @since 1.0.0
     * @access public
     *
     */
    public function get_icon()
    {
        return 'techco-custom-icon';
    }

    /**
     * Get widget categories.
     *
     * Retrieve the list of categories the Elementor widget belongs to.
     *
     * @return array Widget categories.
     * @since 1.0.0
     * @access public
     *
     */
    public function get_categories()
    {
        return ['techco_widgets'];
    }


    protected function register_controls()
    {

        $this->start_controls_section(
            'content_option',
            [
                'label' => esc_html__('Content', 'techco-core'),
                'tab' => Controls_Manager::TAB_CONTENT,
            ]
        );
        $this->add_control(
            'title',
            [
                'type' => Controls_Manager::TEXT,
                'label' => esc_html__('Title', 'techco-core'),
                'label_block' => true,
            ]
        );
        $repeater = new \Elementor\Repeater();

        $repeater->add_control(
            'logo',
            [
                'label' => esc_html__('Brand Logo', 'techco-core'),
                'type' => \Elementor\Controls_Manager::MEDIA,
            ]
        );
        $this->add_control(
            'brands',
            [
                'label' => esc_html__('Add Brand', 'techco-core'),
                'type' => Controls_Manager::REPEATER,
                'fields' => $repeater->get_controls(),
            ]
        );

        $this->end_controls_section();

        $this->start_controls_section(
            'brand_style',
            [
                'label' => esc_html__('Style', 'techco-core'),
                'tab' => Controls_Manager::TAB_STYLE,
            ]
        );
        $this->add_control(
            'bg-c',
            [
                'label' => esc_html__('Background Color', 'techco-core'),
                'type' => Controls_Manager::COLOR,
                'selectors' => [
                    '{{WRAPPER}} .feature_partners_section' => 'background-color: {{VALUE}}',
                    '{{WRAPPER}} .feature_partners_section::before' => 'background-color: {{VALUE}}',
                    '{{WRAPPER}} .feature_partners_section .title_text' => 'background-color: {{VALUE}}',
                ],
            ]
        );
        $this->add_control(
            'wrap_padding',
            [
                'label' => esc_html__('Padding', 'techco-core'),
                'type' => \Elementor\Controls_Manager::DIMENSIONS,
                'size_units' => ['px', '%', 'em', 'rem', 'custom'],
                'selectors' => [
                    '{{WRAPPER}} .feature_partners_section' => 'padding: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
                ],
            ]
        );
        $this->add_control(
            'title_hr',
            [
                'type' => \Elementor\Controls_Manager::DIVIDER,
            ]
        );
        $this->add_control(
            'title-c',
            [
                'label' => esc_html__('Title Color', 'techco-core'),
                'type' => Controls_Manager::COLOR,
                'selectors' => [
                    '{{WRAPPER}} .feature_partners_section .title_text' => 'color: {{VALUE}} !important',
                ],
            ]
        );
        $this->add_group_control(
            Group_Control_Typography::get_type(),
            [
                'name' => 'title_typography',
                'selector' => '{{WRAPPER}} .feature_partners_section .title_text',
            ]
        );
        $this->add_control(
            'brand_item_style',
            [
                'label' => esc_html__('Brand Logo Style', 'techco-core'),
                'type' => \Elementor\Controls_Manager::HEADING,
                'separator' => 'before',
            ]
        );
        $this->add_control(
            'brand_bg-c',
            [
                'label' => esc_html__('Background Color', 'techco-core'),
                'type' => Controls_Manager::COLOR,
                'selectors' => [
                    '{{WRAPPER}} .client_logo_item' => 'background-color: {{VALUE}}',
                ],
            ]
        );
        $this->add_control(
            'height',
            [
                'label' => esc_html__('Height', 'techco-core'),
                'type' => \Elementor\Controls_Manager::SLIDER,
                'size_units' => ['px', '%', 'em', 'rem', 'custom'],
                'range' => [
                    'px' => [
                        'min' => 0,
                        'max' => 1000,
                        'step' => 5,
                    ],
                    '%' => [
                        'min' => 0,
                        'max' => 100,
                    ],
                ],
                'selectors' => [
                    '{{WRAPPER}} .client_logo_item' => 'height: {{SIZE}}{{UNIT}};',
                ],
            ]
        );
        $this->end_controls_section();

    }

    protected function render()
    {
        $settings = $this->get_settings_for_display();
        ?>
        <div class="feature_partners_section">
            <div class="container position-relative">
                <?php if (!empty($settings['title'])): ?>
                    <div class="title_text text-white">
                        <?php echo esc_html($settings['title']); ?>
                    </div>
                <?php endif; ?>
                <div class="client_logo_carousel_inner marquee-left">
                    <?php
                    foreach ($settings['brands'] as $brand):
                        ?>
                        <div class="client_logo_item">
                            <?php echo wp_get_attachment_image($brand['logo']['id'], 'full'); ?>
                        </div>
                    <?php endforeach; ?>
                </div>
            </div>
        </div>

        <?php
    }


}


Plugin::instance()->widgets_manager->register(new Techco_Brand_V2());