<?php

/**
 * Elementor Single Widget
 * @package techco-core
 * @since 1.0.0
 */

namespace Elementor;

if (!defined('ABSPATH')) exit; // Exit if accessed directly.

class Techco_Blog_V4 extends Widget_Base
{

    /**
     * Get widget name.
     *
     * Retrieve Elementor widget name.
     *
     * @return string Widget name.
     * @since 1.0.0
     * @access public
     *
     */
    public function get_name()
    {
        return 'int-blog-v4';
    }

    /**
     * Get widget title.
     *
     * Retrieve Elementor widget title.
     *
     * @return string Widget title.
     * @since 1.0.0
     * @access public
     *
     */
    public function get_title()
    {
        return esc_html__('Blog V4', 'techco-core');
    }

    /**
     * Get widget icon.
     *
     * Retrieve Elementor widget icon.
     *
     * @return string Widget icon.
     * @since 1.0.0
     * @access public
     *
     */
    public function get_icon()
    {
        return 'techco-custom-icon';
    }

    /**
     * Get widget categories.
     *
     * Retrieve the list of categories the Elementor widget belongs to.
     *
     * @return array Widget categories.
     * @since 1.0.0
     * @access public
     *
     */
    public function get_categories()
    {
        return ['techco_widgets'];
    }

    protected function register_controls()
    {

        $this->start_controls_section(
            'post_option',
            [
                'label' => esc_html__('Post Option', 'techco-core'),
                'tab' => Controls_Manager::TAB_CONTENT,
            ]
        );
        $this->add_control(
            'post_order',
            [
                'label' => esc_html__('Post Order', 'techco-core'),
                'type' => Controls_Manager::SELECT,
                'default' => 'ASC',
                'options' => [
                    'ASC' => esc_html__('Ascending', 'techco-core'),
                    'DESC' => esc_html__('Descending', 'techco-core'),
                ],
            ]
        );

        $this->add_control(
            'post_per_page',
            [
                'label' => __('Posts Per Page', 'techco-core'),
                'type' => Controls_Manager::NUMBER,
                'min' => 1,
                'default' => 3,
            ]
        );
        $this->add_control(
            'post_categories',
            [
                'type' => Controls_Manager::SELECT2,
                'label' => esc_html__('Select Categories', 'techco-core'),
                'options' => techco_blog_category(),
                'label_block' => true,
                'multiple' => true,
            ]
        );
        $this->add_control(
            'title_length',
            [
                'label' => __('Title Length', 'techco-core'),
                'type' => Controls_Manager::NUMBER,
                'step' => 1,
                'default' => 12,
            ]
        );
        $this->add_control(
            'button_label', [
                'label' => esc_html__('Readmore Button', 'techco-core'),
                'default' => esc_html__('Read More', 'techco-core'),
                'type' => Controls_Manager::TEXT,
            ]
        );
        $this->end_controls_section();

        $this->start_controls_section(
            'post_style',
            [
                'label' => esc_html__('Post Style', 'techco-core'),
                'tab' => Controls_Manager::TAB_STYLE,
            ]
        );
        $this->add_control(
            'bg-c',
            [
                'label' => esc_html__('Background Color', 'techco-core'),
                'type' => Controls_Manager::COLOR,
                'selectors' => [
                    '{{WRAPPER}} .blog_post_block .blog_post_content' => 'background-color: {{VALUE}}',
                ],
            ]
        );
        $this->add_control(
            'padding',
            [
                'label' => esc_html__('Padding', 'techco-core'),
                'type' => \Elementor\Controls_Manager::DIMENSIONS,
                'size_units' => ['px', '%', 'em', 'rem', 'custom'],
                'selectors' => [
                    '{{WRAPPER}} .blog_post_block .blog_post_content' => 'padding: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
                ],
            ]
        );
        $this->add_control(
            'cat_hr',
            [
                'label' => esc_html__('Category Style', 'techco-core'),
                'type' => \Elementor\Controls_Manager::HEADING,
                'separator' => 'before',
            ]
        );
        $this->add_group_control(
            Group_Control_Typography::get_type(),
            [
                'name' => 'cat_typography',
                'selector' => '{{WRAPPER}} .blog_post_block .blog-post-cat',
            ]
        );
        $this->add_control(
            'cat_padding',
            [
                'label' => esc_html__('Padding', 'techco-core'),
                'type' => \Elementor\Controls_Manager::DIMENSIONS,
                'size_units' => ['px', '%', 'em', 'rem', 'custom'],
                'selectors' => [
                    '{{WRAPPER}} .blog_post_block .blog-post-cat' => 'padding: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
                ],
            ]
        );
        $this->start_controls_tabs(
            'style_tabs'
        );

        $this->start_controls_tab(
            'style_normal_tab',
            [
                'label' => esc_html__('Normal', 'techco-core'),
            ]
        );

        $this->add_control(
            'cat-c',
            [
                'label' => esc_html__('Category Color', 'techco-core'),
                'type' => Controls_Manager::COLOR,
                'selectors' => [
                    '{{WRAPPER}} .blog_post_block .blog-post-cat' => 'color: {{VALUE}}',
                ],
            ]
        );
        $this->add_control(
            'cat_bg-c',
            [
                'label' => esc_html__('Background Color', 'techco-core'),
                'type' => Controls_Manager::COLOR,
                'selectors' => [
                    '{{WRAPPER}} .blog_post_block .blog-post-cat' => 'background-color: {{VALUE}}',
                ],
            ]
        );
        $this->add_control(
            'cat_border_c',
            [
                'label' => esc_html__('Border Color', 'techco-core'),
                'type' => Controls_Manager::COLOR,
                'selectors' => [
                    '{{WRAPPER}} .blog_post_block .blog-post-cat' => 'border-color: {{VALUE}}',
                ],
            ]
        );

        $this->end_controls_tab();

        $this->start_controls_tab(
            'style_hover_tab',
            [
                'label' => esc_html__('Hover', 'techco-core'),
            ]
        );

        $this->add_control(
            'h_cat-c',
            [
                'label' => esc_html__('Category Color', 'techco-core'),
                'type' => Controls_Manager::COLOR,
                'selectors' => [
                    '{{WRAPPER}} .blog_post_block .blog-post-cat:hover' => 'color: {{VALUE}}',
                ],
            ]
        );
        $this->add_control(
            'h_cat_bg-c',
            [
                'label' => esc_html__('Background Color', 'techco-core'),
                'type' => Controls_Manager::COLOR,
                'selectors' => [
                    '{{WRAPPER}} .blog_post_block .blog-post-cat:hover' => 'background-color: {{VALUE}}',
                ],
            ]
        );
        $this->add_control(
            'h_cat_border_c',
            [
                'label' => esc_html__('Border Color', 'techco-core'),
                'type' => Controls_Manager::COLOR,
                'selectors' => [
                    '{{WRAPPER}} .blog_post_block .blog-post-cat:hover' => 'border-color: {{VALUE}}',
                ],
            ]
        );

        $this->end_controls_tab();

        $this->end_controls_tabs();

        $this->add_control(
            'meta_hr',
            [
                'type' => \Elementor\Controls_Manager::DIVIDER,
            ]
        );
        $this->add_control(
            'meta-c',
            [
                'label' => esc_html__('Meta Color', 'techco-core'),
                'type' => Controls_Manager::COLOR,
                'selectors' => [
                    '{{WRAPPER}} .post_meta > li a' => 'color: {{VALUE}}',
                ],
            ]
        );

        $this->add_group_control(
            Group_Control_Typography::get_type(),
            [
                'name' => 'meta_typography',
                'selector' => '{{WRAPPER}} .post_meta > li a',
            ]
        );

        $this->add_control(
            'p_title_hr',
            [
                'type' => \Elementor\Controls_Manager::DIVIDER,
            ]
        );
        $this->add_control(
            'p-title-c',
            [
                'label' => esc_html__('Title Color', 'techco-core'),
                'type' => Controls_Manager::COLOR,
                'selectors' => [
                    '{{WRAPPER}} .blog_post_block .blog_post_title a' => 'color: {{VALUE}}',
                ],
            ]
        );
        $this->add_control(
            'p-hover-title-c',
            [
                'label' => esc_html__('Title Hover Color', 'techco-core'),
                'type' => Controls_Manager::COLOR,
                'selectors' => [
                    '{{WRAPPER}} .blog_post_block .blog_post_title a:hover' => 'color: {{VALUE}}',
                ],
            ]
        );
        $this->add_group_control(
            Group_Control_Typography::get_type(),
            [
                'name' => 'p_title_typography',
                'selector' => '{{WRAPPER}} .blog_post_block .blog_post_title a',
            ]
        );
        $this->add_control(
            'readmore_hr',
            [
                'type' => \Elementor\Controls_Manager::DIVIDER,
            ]
        );
        $this->add_control(
            'p-readmore-c',
            [
                'label' => esc_html__('Readmore Color', 'techco-core'),
                'type' => Controls_Manager::COLOR,
                'selectors' => [
                    '{{WRAPPER}} .btn_unfill' => 'color: {{VALUE}}',
                    '{{WRAPPER}} .btn_unfill .btn_icon' => 'background-color: {{VALUE}}',
                ],
            ]
        );
        $this->add_control(
            'p-readmore-hover-c',
            [
                'label' => esc_html__('Readmore Hover Color', 'techco-core'),
                'type' => Controls_Manager::COLOR,
                'selectors' => [
                    '{{WRAPPER}} .btn_unfill:hover' => 'color: {{VALUE}}',
                    '{{WRAPPER}} .btn_unfill:hover .btn_icon' => 'background-color: {{VALUE}}',
                ],
            ]
        );
        $this->add_group_control(
            Group_Control_Typography::get_type(),
            [
                'name' => 'p_readmore_typography',
                'selector' => '{{WRAPPER}} .btn_unfill',
            ]
        );
        $this->end_controls_section();

    }

    protected function render()
    {
        $settings = $this->get_settings_for_display();
        $calendar_icon = TECHCO_DIR_URL . 'assets/img/icon_calendar.svg';
        $args = array(
            'post_type' => 'post',
            'posts_per_page' => !empty($settings['post_per_page']) ? $settings['post_per_page'] : 1,
            'post_status' => 'publish',
            'ignore_sticky_posts' => 1,
            'order' => $settings['post_order'],
        );
        if (!empty($settings['post_categories'])) {
            $args['category_name'] = implode(',', $settings['post_categories']);
        }

        $query = new \WP_Query($args);
        ?>

        <div class="xb-blog-inner">
            <div class="row justify-content-center mt-none-30">
                <?php
                if ($query->have_posts()) {
                    $counter = 0;
                    while ($query->have_posts()) {
                        $query->the_post();
                        $delay = ($counter * 150) . 'ms';
                        $counter++;
                        ?>
                        <div class="col-lg-4 col-md-6 mt-30">
                            <div class="xb_blog_item wow fadeInUp" data-wow-delay="<?php echo esc_attr($delay); ?>" data-wow-duration=".6s">
                                <?php if (has_post_thumbnail()) : ?>
                                    <div class="blog_post_image">
                                        <a class="image_wrap" href="<?php the_permalink(); ?>"><?php the_post_thumbnail('techco-blog'); ?></a>
                                    </div>
                                <?php endif; ?>
                                <div class="blog_post_content">
                                    <div class="post_meta_wrap">
                                        <ul class="post_meta unordered_list">
                                            <li>
                                                <?php
                                                $categories = get_the_category();
                                                if (!empty($categories)) {
                                                    echo '<a class="blog-post-cat" href="' . esc_url(get_category_link($categories[0]->term_id)) . '">' . esc_html($categories[0]->name) . '</a>';
                                                }
                                                ?>
                                            <li class="post_date">
                                                <a href="<?php echo esc_url(get_day_link(get_the_date('Y'), get_the_date('m'), get_the_date('d'))); ?>">
                                                    <?php echo get_the_date('F j, Y'); ?>
                                                </a>
                                            </li>
                                        </ul>
                                    </div>
                                    <h3 class="blog_post_title">
                                        <a href="<?php the_permalink(); ?>">
                                            <?php echo wp_trim_words(get_the_title(), $settings['title_length'], ''); ?>
                                        </a>
                                    </h3>
                                    <a class="blog_link" href="<?php the_permalink(); ?>">
                                        <span class="btn_label"><?php echo esc_html($settings['button_label']); ?></span>
                                        <span class="btn_icon">
                                            <svg width="20" height="16" viewBox="0 0 20 16" fill="none" xmlns="http://www.w3.org/2000/svg">
                                                <path d="M19.7071 8.70711C20.0976 8.31658 20.0976 7.68342 19.7071 7.29289L13.3431 0.928932C12.9526 0.538408 12.3195 0.538408 11.9289 0.928932C11.5384 1.31946 11.5384 1.95262 11.9289 2.34315L17.5858 8L11.9289 13.6569C11.5384 14.0474 11.5384 14.6805 11.9289 15.0711C12.3195 15.4616 12.9526 15.4616 13.3431 15.0711L19.7071 8.70711ZM0 9H19V7H0V9Z" fill="#615959"></path>
                                              </svg>
                                          </span>
                                    </a>
                                </div>
                            </div>
                        </div>
                    <?php }
                    wp_reset_query();
                } ?>
            </div>
        </div>
        <?php
    }


}


Plugin::instance()->widgets_manager->register(new Techco_Blog_V4());