<?php

/**
 * Elementor Single Widget
 * @package techco-core
 * @since 1.0.0
 */

namespace Elementor;

if (!defined('ABSPATH')) exit; // Exit if accessed directly.

class Techco_Blog_V3 extends Widget_Base
{

    /**
     * Get widget name.
     *
     * Retrieve Elementor widget name.
     *
     * @return string Widget name.
     * @since 1.0.0
     * @access public
     *
     */
    public function get_name()
    {
        return 'int-blog-v3';
    }

    /**
     * Get widget title.
     *
     * Retrieve Elementor widget title.
     *
     * @return string Widget title.
     * @since 1.0.0
     * @access public
     *
     */
    public function get_title()
    {
        return esc_html__('Blog V3', 'techco-core');
    }

    /**
     * Get widget icon.
     *
     * Retrieve Elementor widget icon.
     *
     * @return string Widget icon.
     * @since 1.0.0
     * @access public
     *
     */
    public function get_icon()
    {
        return 'techco-custom-icon';
    }

    /**
     * Get widget categories.
     *
     * Retrieve the list of categories the Elementor widget belongs to.
     *
     * @return array Widget categories.
     * @since 1.0.0
     * @access public
     *
     */
    public function get_categories()
    {
        return ['techco_widgets'];
    }

    protected function register_controls()
    {

        $this->start_controls_section(
            'post_option',
            [
                'label' => esc_html__('Post Option', 'techco-core'),
                'tab' => Controls_Manager::TAB_CONTENT,
            ]
        );
        $this->add_control(
            'post_order',
            [
                'label' => esc_html__('Post Order', 'techco-core'),
                'type' => Controls_Manager::SELECT,
                'default' => 'ASC',
                'options' => [
                    'ASC' => esc_html__('Ascending', 'techco-core'),
                    'DESC' => esc_html__('Descending', 'techco-core'),
                ],
            ]
        );

        $this->add_control(
            'post_per_page',
            [
                'label' => __('Posts Per Page', 'techco-core'),
                'type' => Controls_Manager::NUMBER,
                'min' => 1,
                'default' => 3,
            ]
        );
        $this->add_control(
            'post_categories',
            [
                'type' => Controls_Manager::SELECT2,
                'label' => esc_html__('Select Categories', 'techco-core'),
                'options' => techco_blog_category(),
                'label_block' => true,
                'multiple' => true,
            ]
        );
        $this->add_control(
            'title_length',
            [
                'label' => __('Title Length', 'techco-core'),
                'type' => Controls_Manager::NUMBER,
                'step' => 1,
                'default' => 8,
            ]
        );
        $this->end_controls_section();

        $this->start_controls_section(
            'post_style',
            [
                'label' => esc_html__('Post Style', 'techco-core'),
                'tab' => Controls_Manager::TAB_STYLE,
            ]
        );
        $this->add_control(
            'bg-c',
            [
                'label' => esc_html__('Background Color', 'techco-core'),
                'type' => Controls_Manager::COLOR,
                'selectors' => [
                    '{{WRAPPER}} .blog_post_block_2 .blog_post_content' => 'background-color: {{VALUE}}',
                ],
            ]
        );
        $this->add_control(
            'padding',
            [
                'label' => esc_html__('Padding', 'techco-core'),
                'type' => \Elementor\Controls_Manager::DIMENSIONS,
                'size_units' => ['px', '%', 'em', 'rem', 'custom'],
                'selectors' => [
                    '{{WRAPPER}} .blog_post_block_2 .blog_post_content' => 'padding: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
                ],
            ]
        );

        $this->add_control(
            'cat_hr',
            [
                'type' => \Elementor\Controls_Manager::DIVIDER,
            ]
        );
        $this->add_control(
            'cat-c',
            [
                'label' => esc_html__('Category Color', 'techco-core'),
                'type' => Controls_Manager::COLOR,
                'selectors' => [
                    '{{WRAPPER}} .category_list a' => 'color: {{VALUE}}',
                ],
            ]
        );
        $this->add_control(
            'cat_hover-c',
            [
                'label' => esc_html__('Category Hover Color', 'techco-core'),
                'type' => Controls_Manager::COLOR,
                'selectors' => [
                    '{{WRAPPER}} .category_list a:hover' => 'color: {{VALUE}}',
                ],
            ]
        );

        $this->add_group_control(
            Group_Control_Typography::get_type(),
            [
                'name' => 'cat_typography',
                'selector' => '{{WRAPPER}} .category_list a',
            ]
        );

        $this->add_control(
            'p_title_hr',
            [
                'type' => \Elementor\Controls_Manager::DIVIDER,
            ]
        );
        $this->add_control(
            'p-title-c',
            [
                'label' => esc_html__('Title Color', 'techco-core'),
                'type' => Controls_Manager::COLOR,
                'selectors' => [
                    '{{WRAPPER}} .blog_post_block_2 .post_title a' => 'color: {{VALUE}}',
                ],
            ]
        );
        $this->add_control(
            'p-hover-title-c',
            [
                'label' => esc_html__('Title Hover Color', 'techco-core'),
                'type' => Controls_Manager::COLOR,
                'selectors' => [
                    '{{WRAPPER}} .blog_post_block_2:hover .post_title > a' => 'color: {{VALUE}}',
                ],
            ]
        );
        $this->add_group_control(
            Group_Control_Typography::get_type(),
            [
                'name' => 'p_title_typography',
                'selector' => '{{WRAPPER}} .blog_post_block_2 .post_title',
            ]
        );
        $this->add_control(
            'meta_hr',
            [
                'type' => \Elementor\Controls_Manager::DIVIDER,
            ]
        );
        $this->add_control(
            'p-meta-c',
            [
                'label' => esc_html__('Meta Color', 'techco-core'),
                'type' => Controls_Manager::COLOR,
                'selectors' => [
                    '{{WRAPPER}} .post_meta > li a' => 'color: {{VALUE}}',
                ],
            ]
        );
        $this->add_group_control(
            Group_Control_Typography::get_type(),
            [
                'name' => 'p_meta_typography',
                'selector' => '{{WRAPPER}} .post_meta > li a',
            ]
        );
        $this->end_controls_section();

    }

    protected function render()
    {
        $settings = $this->get_settings_for_display();
        $calendar_icon = TECHCO_DIR_URL . 'assets/img/icon_calendar.svg';
        $args = array(
            'post_type' => 'post',
            'posts_per_page' => !empty($settings['post_per_page']) ? $settings['post_per_page'] : 1,
            'post_status' => 'publish',
            'ignore_sticky_posts' => 1,
            'order' => $settings['post_order'],
        );
        if (!empty($settings['post_categories'])) {
            $args['category_name'] = implode(',', $settings['post_categories']);
        }

        $query = new \WP_Query($args);
        ?>

        <div class="xb-blog-inner">
            <div class="row mt-none-30">
                <?php
                if ($query->have_posts()) {
                    while ($query->have_posts()) {
                        $query->the_post();

                        ?>
                        <div class="col-lg-6 mt-30">
                            <div class="blog_post_block_2">
                                <?php if (has_post_thumbnail()) : ?>
                                    <div class="blog_post_image">
                                        <a href="<?php the_permalink(); ?>">
                                            <?php the_post_thumbnail('techco-blog-2'); ?>
                                        </a>
                                    </div>
                                <?php endif; ?>
                                <div class="blog_post_content">
                                    <?php
                                    $categories = get_the_category();
                                    if (!empty($categories) && is_array($categories)) : ?>
                                        <ul class="category_list unordered_list">
                                            <li>
                                                <a href="<?php echo esc_url(get_category_link($categories[0]->term_id)); ?>">
                                                    <?php echo esc_html($categories[0]->name); ?>
                                                </a>
                                            </li>
                                        </ul>
                                    <?php endif; ?>
                                    <h3 class="post_title">
                                        <a href="<?php the_permalink(); ?>">
                                            <?php echo wp_trim_words(get_the_title(), $settings['title_length'], ''); ?>
                                        </a>
                                    </h3>
                                    <ul class="post_meta unordered_list">
                                        <li>
                                            <a href="<?php echo esc_url(get_author_posts_url(get_the_author_meta('ID'))); ?>">
                                                <?php _e('By'); ?> <b><?php the_author(); ?></b>
                                            </a>
                                        </li>
                                        <li>
                                            <a href="<?php echo esc_url(get_day_link(get_the_date('Y'), get_the_date('m'), get_the_date('d'))); ?>">
                                                <?php echo get_the_date('F j, Y'); ?>
                                            </a>
                                        </li>
                                    </ul>
                                </div>
                            </div>
                        </div>
                    <?php }
                    wp_reset_query();
                } ?>
            </div>
        </div>
        <?php
    }


}


Plugin::instance()->widgets_manager->register(new Techco_Blog_V3());