<?php
/**
 * @package techco
 */

if (!function_exists('wp_body_open')) :
    /**
     * Shim for sites older than 5.2.
     *
     * @link https://core.trac.wordpress.org/ticket/12563
     */
    function wp_body_open()
    {
        do_action('wp_body_open');
    }
endif;

/**
 * Adds custom classes to the array of body classes.
 */
function techco_body_classes($classes)
{
    // Adds a class of hfeed to non-singular pages.
    if (!is_singular()) {
        $classes[] = 'hfeed';
    }

    //Check Elementor Page Builder Used or not
    $elementor_used = get_post_meta(get_the_ID(), '_elementor_edit_mode', true);

    if (is_archive() || is_search()) {
        $classes[] = !!$elementor_used ? 'page-builder-not-used' : 'page-builder-not-used';
    } else {
        $classes[] = !!$elementor_used ? 'page-builder-used' : 'page-builder-not-used';
    }

    return $classes;
}

add_filter('body_class', 'techco_body_classes');

/**
 * Add a pingback url auto-discovery header for single posts, pages, or attachments.
 */
function techco_pingback_header()
{
    if (is_singular() && pings_open()) {
        printf('<link rel="pingback" href="%s">', esc_url(get_bloginfo('pingback_url')));
    }
}

add_action('wp_head', 'techco_pingback_header');


/**
 * Words limit
 */
function techco_words_limit($text, $limit)
{
    $words = explode(' ', $text, ($limit + 1));

    if (count($words) > $limit) {
        array_pop($words);
    }

    return implode(' ', $words);
}


/**
 * Get excluded sidebar list
 */
if (!function_exists('techco_sidebars')) {
    function techco_sidebars()
    {
        $default = esc_html__('Default', 'techco');
        $options = array($default);
        // set ids of the registered sidebars for exclude
        $exclude = array('footer-widget');

        global $wp_registered_sidebars;

        if (!empty($wp_registered_sidebars)) {
            foreach ($wp_registered_sidebars as $sidebar) {
                if (!in_array($sidebar['id'], $exclude)) {
                    $options[$sidebar['id']] = $sidebar['name'];
                }
            }
        }

        return $options;
    }
}


/**
 * Iframe embed
 */

function techco_iframe_embed($tags, $context)
{
    if ('post' === $context) {
        $tags['iframe'] = array(
            'src' => true,
            'height' => true,
            'width' => true,
            'frameborder' => true,
            'allowfullscreen' => true,
        );
    }
    return $tags;
}

add_filter('wp_kses_allowed_html', 'techco_iframe_embed', 10, 2);


/**
 * Check if a post is a custom post type.
 *
 * @param mixed $post Post object or ID
 *
 * @return boolean
 */
function techco_custom_post_types($post = null)
{
    $custom_post_list = get_post_types(array('_builtin' => false));

    // there are no custom post types
    if (empty ($custom_post_list)) {
        return false;
    }

    $custom_types = array_keys($custom_post_list);
    $current_post_type = get_post_type($post);

    // could not detect current type
    if (!$current_post_type) {
        return false;
    }

    return in_array($current_post_type, $custom_types);
}


/**
 * Add span tag in archive list count number
 */
function techco_add_span_archive_count($links)
{
    $links = str_replace('</a>&nbsp;(', '</a> <span class="post-count-number">(', $links);
    $links = str_replace(')', ')</span>', $links);
    return $links;
}

add_filter('get_archives_link', 'techco_add_span_archive_count');


/**
 * Add span tag in category list count number
 */

function techco_add_span_category_count($links)
{
    $links = str_replace('</a> (', '</a> <span class="post-count-number">(', $links);
    $links = str_replace(')', ')</span>', $links);
    return $links;
}

add_filter('wp_list_categories', 'techco_add_span_category_count');

/**
 * Preloader
 *
 * @return  [type]  [return description]
 */
function techco_preloader()
{
    $preloader_enable = techco_option('preloader_enable', true);
    if ($preloader_enable == true):
        ?>
        <div id="preloader" class="preloader">
            <div class="loader-circle">
                <div class="loader-line-mask">
                    <div class="loader-line"></div>
                </div>
                <div class="loader-logo">
                    
                </div>
            </div>
        </div>
    <?php
    endif;
}

/**
 * Scroll Up
 *
 * @return  [type]  [return description]
 */
function techco_scroll_up_btn()
{
    $scroll_up_btn = techco_option('scroll_up_btn', true);
    if ($scroll_up_btn == true):
        ?>
        <div class="xb-backtotop">
            <a href="#" class="scroll">
                <i class="far fa-arrow-up"></i>
            </a>
        </div>
    <?php
    endif;
}

/**
 * Image id from url
 */
if (!function_exists('techco_image_id_by_url')) {
    function techco_image_id_by_url($image_url)
    {
        global $wpdb;
        $attachment = $wpdb->get_col($wpdb->prepare("SELECT ID FROM $wpdb->posts WHERE guid='%s';", $image_url));
        return $attachment[0];
    }
}

/**
 * Prints HTML with meta information for the current post-date/time.
 */
if (!function_exists('techco_posted_on')) :

    function techco_posted_on()
    {
        $time_string = '<time class="entry-date published updated" datetime="%1$s">%2$s</time>';
        if (get_the_time('U') !== get_the_modified_time('U')) {
            $time_string = '<time class="entry-date published" datetime="%1$s">%2$s</time><time class="updated" datetime="%3$s">%4$s</time>';
        }

        $time_string = sprintf($time_string,
            esc_attr(get_the_date(DATE_W3C)),
            esc_html(get_the_date()),
            esc_attr(get_the_modified_date(DATE_W3C)),
            esc_html(get_the_modified_date())
        );

        $posted_on = sprintf(
        /* translators: %s: post date. */
            esc_html_x(' %s', 'post date', 'techco'),
            '<a href="' . esc_url(get_permalink()) . '" rel="bookmark">' . $time_string . '</a>'
        );

        echo '<span class="posted-on"><i class="far fa-clock"></i>' . $posted_on . '</span>'; // WPCS: XSS OK.

    }
endif;


/**
 * Prints HTML with meta information for the current author.
 */
if (!function_exists('techco_posted_by')) :

    function techco_posted_by()
    {
        $byline = sprintf(
        /* translators: %s: post author. */
            esc_html_x(' %s', 'post author', 'techco'),
            '<span class="author vcard">' . esc_html(get_the_author()) . '</span>'
        );

        echo '<span class="byline"><i class="far fa-user"></i> ' . $byline . '</span>'; // WPCS: XSS OK.

    }
endif;

/**
 * Prints HTML with meta information for the tags.
 */
if (!function_exists('techco_post_tags')) :

    function techco_post_tags()
    {
        // Hide category and tag text for pages.
        if ('post' === get_post_type()) {

            /* translators: used between list items, there is a space after the comma */
            $tags_list = get_the_tag_list('', esc_html_x('', 'list item separator', 'techco'));
            if ($tags_list) {
                /* translators: 1: list of tags. */
                printf('<span class="tags-links"><span class="tag-title">' . esc_html__('Tags:', 'techco') . '</span>' . esc_html__(' %1$s', 'techco') . '</span>', $tags_list); // WPCS: XSS OK.


            }

        }
    }
endif;

/**
 * Prints HTML with meta information for the categories.
 */

if (!function_exists('techco_theme_post_categories')) :

    function techco_theme_post_categories()
    {
        // Hide category and tag text for pages.
        if ('post' === get_post_type()) {

            /* translators: used between list items, there is a space after the comma */
            $categories_list = get_the_category_list(esc_html__(', ', 'techco'));
            if ($categories_list) {
                /* translators: 1: list of categories. */
                printf('<span class="cat-links"><i class="far fa-folder"></i>' . esc_html__('%1$s', 'techco') . '</span>', $categories_list); // WPCS: XSS OK.
            }

        }
    }
endif;

/**
 * Prints post's first category
 */

if (!function_exists('techco_post_first_category')) :

    function techco_post_first_category()
    {

        $post_category_list = get_the_terms(get_the_ID(), 'category');

        $post_first_category = $post_category_list[0];
        if (!empty($post_first_category->slug)) {
            echo '<span class="cat-links"><i class="far fa-folder"></i><a href="' . get_term_link($post_first_category->slug, 'category') . '">' . $post_first_category->name . '</a></span>';
        }

    }
endif;

/**
 * Prints HTML with meta information for the comments.
 */
if (!function_exists('techco_comment_count')) :

    function techco_comment_count()
    {
        if (!post_password_required() && (comments_open() || get_comments_number()) && get_comments_number() != 0) {
            echo '<span class="comments-link"><i class="far fa-comments"></i>';
            comments_popup_link('', '' . esc_html__('1', 'techco') . ' <span class="comment-count-text">' . esc_html__('Comment', 'techco') . '</span>', '% <span class="comment-count-text">' . esc_html__('Comments', 'techco') . '</span>');
            echo '</span>';
        }
    }
endif;


/**
 * Site Logo
 */
function techco_default_logo()
{
    $site_logo_img = techco_option('header_default_logo');
    ?>
    <?php
    if (has_custom_logo()) {
        the_custom_logo();
    } else {
        if (!empty($site_logo_img['url'])) { ?>
            <a href="<?php echo esc_url(home_url('/')); ?>">
                <img src="<?php echo esc_url($site_logo_img['url']); ?>"
                     alt="<?php echo esc_attr(get_post_meta($site_logo_img['id'], '_wp_attachment_image_alt', true)); ?>">
            </a>

            <?php

        } else { ?>
            <h1 class="site-title"><a href="<?php echo esc_url(home_url('/')); ?>"
                                      rel="home"><?php bloginfo('name'); ?></a></h1>

            <?php

        }
    }
    ?>
<?php }


function techco_main_menu()
{
    wp_nav_menu(array(
        'theme_location' => 'main-menu',
        'menu_id' => 'main-nav',
        'link_before' => '<span>', 'link_after' => '</span>',
        'container' => false,
        'fallback_cb' => 'Techco_Bootstrap_Navwalker::fallback',
        'walker'         => class_exists( 'EFramework_Mega_Menu_Walker' ) ? new EFramework_Mega_Menu_Walker : '',
    ));
}

function techco_mobiel_menu()
{
    wp_nav_menu([
        'theme_location' => 'main-menu',
        'menu_id' => 'xb-menu-primary',
        'menu_class' => 'xb-menu-primary clearfix',
        'link_before' => '<span>', 'link_after' => '</span>',
        'container' => false,
        'fallback_cb' => 'Techco_Bootstrap_Navwalker::fallback',
        'walker'         => class_exists( 'EFramework_Mega_Menu_Walker' ) ? new EFramework_Mega_Menu_Walker : '',
    ]);

}

/**
 * Mobile Search
 *
 * @return  [type]  [return description]
 */
function techco_moble_search()
{
    ?>
    <div class="xb-header-mobile-search xb-hide-xl">
        <form role="search" action="<?php echo esc_url(home_url('/')); ?>">
            <input class="search-field" type="text" name="s"
                   placeholder="<?php esc_attr_e('Search Keywords', 'techco'); ?>"
                   value="<?php the_search_query(); ?>">
            <button class="search-submit" type="submit"><i class="far fa-search"></i></button>
        </form>
    </div>
    <?php
}


/**
 * Sidebar Info
 *
 * @return  [type]  [return description]
 */
function techco_sidebar_info()
{
    $is_enable_offcanvas_nav = techco_option('is_enable_offcanvas_nav');
    $about_title = techco_option('about_title');
    $contact_info_title = techco_option('contact_info_title');
    $contact_info = techco_option('contact_info');

    $off_call_title = techco_option('off_call_title');
    $off_call = techco_option('off_call');
    $off_email = techco_option('off_email');
    $offc_social_icons = techco_option('offc_social_icons');
    ?>
    <!-- sidebar-info start -->
    <?php if ($is_enable_offcanvas_nav == true): ?>
    <div class="offcanvas-sidebar black-sidebar">
        <div class="sidebar-menu-close">
            <a class="xb-close" href="javascript:void(0);"></a>
        </div>
        <div class="sidebar-top mb-65">
            <div class="sidebar-logo mb-40">
                <?php techco_default_logo(); ?>
            </div>
            <div class="sidebar-content">
                <?php echo wp_kses($about_title, true); ?>
            </div>
        </div>

        <div class="sidebar-contact-info mb-65">
            <h4 class="sidebar-heading"><?php echo wp_kses($contact_info_title, true); ?></h4>
            <ul class="sidebar-info-list list-unstyled">
                <?php foreach ($contact_info as $info): ?>
                    <li><span><svg width="17" height="17" viewBox="0 0 17 17" fill="none" xmlns="http://www.w3.org/2000/svg">
<path d="M9.69477 5.28863L9.74259 5.50013L9.92613 5.38427L14.5104 2.48953L11.6156 7.07378L11.4998 7.25769L11.7114 7.3055L17 8.49969L11.7114 9.69449L11.4998 9.74231L11.6156 9.9256L14.5104 14.5105L9.92613 11.6151L9.74259 11.4993L9.69477 11.7114L8.50012 17L7.30548 11.7114L7.25766 11.4993L7.07412 11.6151L2.48964 14.5105L5.38463 9.9256L5.50049 9.74231L5.28881 9.69449L0 8.49969L5.28881 7.3055L5.50049 7.25769L5.38463 7.07378L2.48964 2.48953L7.07412 5.38427L7.25766 5.50013L7.30548 5.28863L8.50012 0L9.69477 5.28863Z" fill="#FF4D24"/>
</svg>
</span><?php echo esc_html($info['info']);?></li>
                <?php endforeach; ?>
            </ul>
        </div>
        <div class="xb-content-wrap d-flex">
            <div class="xb-title col-auto"><?php echo esc_html($off_call_title);?></div>
            <div class="xb-inf-content-wrap col">
                <div class="xb-item-wrap row">
                    <div class="xb-item col-auto ">
                        <span class="item-content"><a href="tel:<?php echo esc_url($off_call);?>" class="tel"><?php echo esc_html($off_call);?></a></span>
                    </div>
                    <div class="xb-item col-auto "> <span class="item-content"><a href="mailto:<?php echo esc_url($off_email);?>"><?php echo esc_html($off_email);?></a></span></div>
                </div>
            </div>
        </div>
        <?php if(!empty($offc_social_icons)):?>
            <div class="sidebar-socials-wrap mt-30">
                <?php foreach($offc_social_icons as $link):?>
                    <a class="social-item" href="<?php echo esc_url($link['link']);?>" target="_blank"><?php echo esc_html($link['social_title']);?></a>
                <?php endforeach;?>
            </div>
        <?php endif;?>

    </div>
<?php endif; ?>
    <!-- sidebar-info end -->
    <?php
}


/**
 * Offcanvash Menu
 *
 * @return  [type]  [return description]
 */
function techco_offcanvasmenu(){
    ?>
    <div class="xb-header-wrap">
        <div class="xb-header-menu">
            <div class="xb-header-menu-scroll">
                <div class="xb-menu-close xb-hide-xl xb-close"></div>
                <div class="xb-logo-mobile xb-hide-xl">
                    <?php techco_default_logo(); ?>
                </div>
                <nav class="xb-header-nav">
                    <?php
                    techco_moble_search();
                    techco_mobiel_menu();
                    ?>
                </nav>
            </div>
        </div>
        <div class="xb-header-menu-backdrop"></div>
    </div>
    <?php
}

/**
 * Header Search
 *
 * @return  [type]  [return description]
 */
function techco_heder_search()
{ ?>
    <div class="header-search-form-wrapper">
        <div class="xb-search-close xb-close"></div>
        <div class="header-search-container container">
            <form role="search" class="search-form" action="<?php echo esc_url(home_url('/')); ?>">
                <input type="search" class="search-field"
                       placeholder="<?php esc_attr_e('Search Keywords', 'techco'); ?>"
                       value="<?php echo get_search_query(); ?>" name="s">
                <button class="search-submit" type="submit"><i class="far fa-search"></i></button>
            </form>
        </div>
    </div>
    <?php
}

/**
 * Product Per Page Count
 *
 * @param [type] $per_page
 * @return void
 */

function techco_product_count($cols)
{
    $product_count = techco_option('product_count');
    $cols = !empty($product_count) ? $product_count : '12';
    return $cols;
}

add_filter('loop_shop_per_page', 'techco_product_count', 20);

function prinox_megamenu_enable() {
    return true;
}
add_filter( 'th_enable_megamenu', 'prinox_megamenu_enable' );