<?php
/**
 * Define the Youtube Importer functionality
 *
 *
 * @link       https://themeforest.net/user/phpface
 * @since      2.0
 *
 * @package    Streamtube_Core
 * @subpackage Streamtube_Core/includes
 */

/**
 *
 * @since      1.0.0
 * @package    Streamtube_Core
 * @subpackage Streamtube_Core/includes
 * @author     phpface <nttoanbrvt@gmail.com>
 */

if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

class StreamTube_Core_Youtube_Importer_Admin {

	/**
	 *
	 * Define advertising admin menu slug
	 *
	 * @since 1.3
	 * 
	 */
	const ADMIN_MENU_SLUG = 'edit.php?post_type=video';

	/**
	 *
	 * Unregistered Menu
	 * 
	 */
	public function unregistered() {
		add_submenu_page(
			self::ADMIN_MENU_SLUG,
			esc_html__( 'YouTube Importers', 'streamtube-core' ),
			esc_html__( 'YouTube Importers', 'streamtube-core' ),
			'administrator',
			'youtube-importers',
			array( 'Streamtube_Core_License', 'unregistered_template' ),
			50
		);
	}

	/**
	 *
	 * @see add_meta_box()
	 *
	 * @since 2.0
	 * 
	 */
	public function add_meta_boxes( $post_type, $post = null ) {

		if ( ! $post ) {
			return;
		}

		$post_type = StreamTube_Core_Youtube_Importer_Post_Type::POST_TYPE;

		$settings = $this->get_settings( $post->ID );

		streamtube_add_meta_box(
			$post_type . '_settings',
			esc_html__( 'Search Settings', 'streamtube-core' ),
			array( $this, 'search_settings_box' ),
			array( 'youtube_importer', 'dashboard_youtube_importer' ),
			'advanced',
			'core'
		);

		streamtube_add_meta_box(
			$post_type . '_cpt_settings',
			esc_html__( 'Import To', 'streamtube-core' ),
			array( $this, 'video_settings_box' ),
			array( 'youtube_importer', 'dashboard_youtube_importer' ),
			'advanced',
			'core'
		);

		streamtube_add_meta_box(
			$post_type . '_search',
			esc_html__( 'Search', 'streamtube-core' ),
			array( $this, 'search_box' ),
			array( 'youtube_importer', 'dashboard_youtube_importer' ),
			'advanced',
			'core'
		);

		streamtube_add_meta_box(
			$post_type . '_schedule',
			esc_html__( 'Schedule', 'streamtube-core' ),
			array( $this, 'schedule' ),
			array( 'youtube_importer', 'dashboard_youtube_importer' ),
			'side',
			'core'
		);

		streamtube_add_meta_box(
			$post_type . '_videos',
			esc_html__( 'Latest Imported Items', 'streamtube-core' ),
			array( $this, 'imported_box' ),
			array( 'youtube_importer', 'dashboard_youtube_importer' ),
			'side',
			'core'
		);

		$taxonomies = get_object_taxonomies( $settings['post_type'], 'object' );

		foreach ( $taxonomies as $tax => $object ) {

			if ( ! $object->public || ( ! is_admin() && ! $object->show_ui_frontend ) || ! current_user_can( $object->cap->assign_terms ) ) {
				continue;
			}

			streamtube_add_meta_box(
				$post_type . '_' . $tax,
				sprintf(
					esc_html__( 'Import to %s', 'streamtube-core' ),
					$object->labels->name
				),
				array( $this, 'taxonomies_box' ),
				array( 'youtube_importer', 'dashboard_youtube_importer' ),
				'side',
				'core',
				compact( 'tax' )
			);
		}
	}

	/**
	 *
	 * The settings metabox
	 * 
	 * @param  object $post
	 *
	 * @since 2.0
	 *
	 */
	public function search_settings_box( $post ) {
		load_template( plugin_dir_path( __FILE__ ) . 'admin/search-settings.php' );
	}

	/**
	 *
	 * The box settings metabox
	 * 
	 * @param  object $post
	 *
	 * @since 2.0
	 *
	 */
	public function video_settings_box( $post ) {
		load_template( plugin_dir_path( __FILE__ ) . 'admin/video-settings.php' );
	}

	/**
	 *
	 * The search metabox
	 * 
	 * @param  object $post
	 *
	 * @since 2.0
	 *
	 */
	public function search_box( $post ) {
		load_template( plugin_dir_path( __FILE__ ) . 'admin/search.php' );
	}

	public function schedule( $post ) {
		load_template( plugin_dir_path( __FILE__ ) . 'admin/schedule.php' );
	}

	/**
	 *
	 * The imported metabox
	 * 
	 * @param  object $post
	 *
	 * @since 2.0
	 *
	 */
	public function imported_box( $post ) {
		load_template( plugin_dir_path( __FILE__ ) . 'admin/imported-videos.php' );
	}

	/**
	 *
	 * The taxonomies metabox
	 * 
	 * @param  object $post
	 *
	 * @since 2.0
	 * 
	 */
	public function taxonomies_box( $post, $args ) {
		load_template( plugin_dir_path( __FILE__ ) . 'admin/taxonomy.php', false, $args['args'] );
	}

	/**
	 *
	 * Get default settings;
	 * 
	 * @return array
	 *
	 * @since 2.0
	 * 
	 */
	private function get_default() {
		return array(
			'apikey'                => current_user_can( 'manage_options' ) ? get_option( 'youtube_api_key' ) : '',
			'oauth'                 => false,
			'enable'                => '',
			'cron_tag_key'          => uniqid(),
			'q'                     => '',
			'forMine'               => '',
			'searchIn'              => 'channel',
			'channelId'             => '',
			'maxResults'            => 10,
			'publishedAfter'        => '',
			'publishedBefore'       => '',
			'regionCode'            => '',
			'relevanceLanguage'     => '',
			'type'                  => 'video',
			'videoType'             => 'any',
			'safeSearch'            => 'moderate',
			'eventType'             => '',
			'videoDefinition'       => 'any',
			'videoDimension'        => 'any',
			'videoDuration'         => 'any',
			'videoEmbeddable'       => true,
			'videoLicense'          => 'any',
			'order'                 => 'date',
			'order_playlist_date'   => '',
			'order_asc'             => '',
			'post_status'           => 'pending',
			'post_type'             => 'video',
			'post_meta_field'       => '',
			'post_author'           => '',
			'post_tags'             => '',
			'update_number'         => 5,
			'update_frequency'      => 5,
			'update_frequency_unit' => 'minutes',
			'cron_interval'         => ''
		);
	}

	/**
	 *
	 * Save settings metabox
	 * 
	 * @param  int $post_id
	 *
	 * @since 2.0
	 * 
	 */
	public function save_settings( $importer_id ) {

		$http_data = wp_parse_args( $_REQUEST, array(
			'save_importer' => ''
		) );

		if ( ! wp_verify_nonce( $http_data['save_importer'], 'save_importer_nonce' ) ) {
			return;
		}

		if ( defined( 'DOING_AUTOSAVE' ) && DOING_AUTOSAVE ) {
			return;
		}

		if ( ! isset( $_POST['yt_importer'] ) ) {
			return;
		}

		$settings = wp_parse_args( $_POST['yt_importer'], $this->get_default() );

		foreach ( $settings as $key => $value ) {
			update_post_meta( $importer_id, $key, wp_unslash( trim( $value ) ) );
		}
	}

	/**
	 *
	 * Get settings
	 * 
	 * @param  integer $importer_id
	 * 
	 * @since 2.0
	 * 
	 */
	public function get_settings( $importer_id = 0 ) {

		if ( ! get_post( $importer_id ) ) {
			return false;
		}

		$settings = array();

		$defaults = $this->get_default();

		foreach ( $defaults as $key => $value ) {
			$_meta = get_post_meta( $importer_id, $key, true );

			if ( ! $_meta ) {
				$_meta = $value;
			}

			$settings[ $key ] = $_meta;
		}

		/**
		 *
		 * Filter settings;
		 *
		 * @param array $settings
		 * @param int $post_id ($importer_id)
		 * 
		 */
		return apply_filters( 'streamtube/core/youtube_importer/get_settings', $settings, $importer_id );
	}

	/**
	 * Add custom fields to the Video table
	 *
	 * @param array $columns
	 */
	public function post_table( $columns ) {
		unset( $columns['date'] );
		unset( $columns['title'] );

		$new_columns = array(
			'run'       => esc_html__( 'Action', 'streamtube-core' ),
			'title'     => esc_html__( 'Importer', 'streamtube-core' ),
			'schedule'  => esc_html__( 'Schedule', 'streamtube-core' ),
			'last_run'  => esc_html__( 'Last Run', 'streamtube-core' ),
			'next_run'  => esc_html__( 'Next Run', 'streamtube-core' ),
			'next_page' => esc_html__( 'Next Page Token', 'streamtube-core' ),
			'status'    => esc_html__( 'Status', 'streamtube-core' ),
			'date'      => esc_html__( 'Date', 'streamtube-core' )
		);

		return array_merge( $columns, $new_columns );
	}

	/**
	 *
	 * Custom Columns callback
	 * 
	 * @param  string $column
	 * @param  int $post_id
	 * 
	 */
	public function post_table_columns( $column, $importer_id ) {

		switch ($column) {

			case 'run':
				printf(
					'<button data-text-run="%s" data-text-running="%s" type="button" class="button button-primary btn btn-primary button-yt-bulk-import btn-sm" data-importer-id="%s" data-key="%s" data-nonce="%s">',
					esc_html__( 'Run Now', 'streamtube-core' ),
					esc_html__( 'Running', 'streamtube-core' ),
					esc_attr( $importer_id ),
					esc_attr( get_post_meta( $importer_id, 'cron_tag_key', true ) ),
					esc_attr( wp_create_nonce( 'run_schedule_' . $importer_id ) )
				);

				esc_html_e( 'Run Now', 'streamtube-core' ); ?>
				</button>
				<?php
				break;

			case 'schedule':
				if ( wp_next_scheduled( 'streamtube_youtube_imports', array( $importer_id ) ) ) {
					printf(
						'<span class="badge bg-success">%s</span>',
						esc_html__( 'Scheduled', 'streamtube-core' )
					);
				} else {
					printf(
						'<span class="badge bg-warning">%s</span>',
						esc_html__( 'Not scheduled', 'streamtube-core' )
					);
				}
				break;

			case 'last_run':

				$last_check = get_post_meta( $importer_id, 'last_check', true );

				if ( $last_check ) {
					printf(
						esc_html__( '%s ago', 'streamtube-core' ),
						human_time_diff( get_post_meta( $importer_id, 'last_check', true ), current_time( 'timestamp' ) )
					);
				}
				break;

			case 'next_run':

				$event = wp_get_scheduled_event( 'streamtube_youtube_imports', array( $importer_id ) );

				if ( $event ) {
					echo human_time_diff( $event->timestamp, current_time( 'timestamp' ) );
				} else {
					esc_html_e( 'No event', 'streamtube-core' );
				}

				break;

			case 'next_page':

				$maybe_next_page = get_post_meta( $importer_id, 'next_page_token', true );

				if ( $maybe_next_page ) {
					echo $maybe_next_page;
				}
				break;

			case 'status':

				$count = 0;

				$imported_videos = streamtube_core()->get()->yt_importer->get_imported_videos( $importer_id, 10000000000 );

				if ( $imported_videos ) {
					$count = count( $imported_videos );
				}

				printf(
					esc_html__( '%s items were imported', 'streamtube-core' ),
					number_format_i18n( $count )
				);
				break;
		}
	}

	/**
	 *
	 * Filter $query
	 * Query posts based on `importer_id` meta value
	 * 
	 */
	public function pre_get_posts( $query ) {
		if ( is_admin() ) {
			if ( isset( $_GET['post_type'] ) && $_GET['post_type'] == 'video' ) {
				if ( isset( $_GET['importer_id'] ) && absint( $_GET['importer_id'] ) > 0 ) {

					$query->set( 'meta_query', array(
						array(
							'key'   => 'yt_importer_id',
							'value' => absint( $_GET['importer_id'] )
						)
					) );
				}
			}
		}
	}

	/**
	 *
	 * Filter `streamtube/core/posts_list_table/query_args`
	 *
	 * Dashboard posts list table
	 *
	 * @param array $q;
	 * 
	 */
	public function filter_posts_list_table_query_args( $q, $post_type, $wp_query ) {
		$importer_id = isset( $_REQUEST['importer_id'] ) ? absint( $_REQUEST['importer_id'] ) : 0;

		if ( $importer_id ) {
			$q = wp_parse_args( $q, array(
				'meta_query' => array()
			) );

			$q['meta_query'][] = array(
				'key'   => 'yt_importer_id',
				'value' => $importer_id
			);
		}
		return $q;
	}

	/**
	 *
	 * Enqueue frontend scripts
	 * 
	 */
	public function enqueue_scripts() {
		wp_enqueue_style(
			'youtube-importer',
			plugin_dir_url( __FILE__ ) . 'assets/style.css',
			array(),
			filemtime( plugin_dir_path( __FILE__ ) . 'assets/style.css' )
		);

		wp_enqueue_script(
			'youtube-importer',
			plugin_dir_url( __FILE__ ) . 'assets/scripts.js',
			array( 'jquery' ),
			filemtime( plugin_dir_path( __FILE__ ) . 'assets/scripts.js' ),
			true
		);

		wp_localize_script( 'youtube-importer', 'youtube_importer', array(
			'is_admin' => is_admin()
		) );
	}
}