<?php
if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

/**
 *
 * Get all public post types
 * 
 * @return array
 */
function streamtube_core_get_statistics_post_types( $filtered = false ) {

	$post_types = streamtube_core_get_public_post_types();

	if ( function_exists( 'WC' ) && post_type_exists( 'product' ) ) {
		$post_types = array_merge( $post_types, array(
			'product' => 'product'
		) );
	}

	if ( $filtered ) {
		foreach ( $post_types as $post_type => $name ) {
			if ( ! current_user_can( get_post_type_object( $post_type )->cap->edit_posts ) ) {
				unset( $post_types[ $post_type ] );
			}
		}
	}

	return apply_filters( 'streamtube_core_get_statistics_post_types', $post_types, $filtered );
}

/**
 *
 * Get resource types based of current user post type capability
 * 
 * @return array
 */
function streamtube_core_get_statistics_resource_types( $filtered = false ) {

	$resource_types = array();

	$post_types = streamtube_core_get_statistics_post_types( $filtered );

	if ( ! $post_types ) {
		return array();
	}

	foreach ( $post_types as $key => $value ) {
		if ( ! in_array( $key, array( 'post', 'page', 'attachment' ) ) ) {
			$resource_types[] = 'post_type_' . $key;
		} else {
			$resource_types[] = $key;
		}
	}

	return apply_filters( 'streamtube_core_get_statistics_resource_types', $resource_types, $filtered );
}

/**
 * Generates a URL for a specific statistics page within a user's dashboard.
 *
 * @param string $page    (Optional) The specific statistics page to link to. Default is empty string.
 * @param int    $user_id (Optional) The ID of the user whose statistics page is being linked to. Defaults to the current user's ID.
 *
 * @return string The URL to the specified statistics page.
 */
function streamtube_core_statistics_page_url( $page = '', $user_id = 0 ) {

	if ( ! $user_id ) {
		$user_id = get_current_user_id();
	}

	$url = untrailingslashit( get_author_posts_url( $user_id ) ) . '/dashboard/statistics/';

	if ( $page ) {
		$url .= $page;
	}

	return $url;
}

/**
 * Check if a WP Statistics page is designed for public users or admin only.
 *
 * WP Statistics pages aren't designed for multi-author environments.
 * This function verifies if the page is accessible to public users.
 * 
 * @param string $page The page slug to check.
 * @return bool True if the page is public, false otherwise.
 */
function streamtube_core_is_statistics_public_pages( $page = '' ) {
	$public_pages = array(
		'pages',
		'content-analytics',
		'visitors',
		'geographic',
		'referrals',
		'devices'
	);

	/**
	 *
	 * Filter public pages
	 * 
	 */
	$public_pages = apply_filters( 'streamtube/core/wpstatistics/public_pages', $public_pages );

	return apply_filters(
		'streamtube/core/wpstatistics/is_public_page',
		in_array( $page, $public_pages ),
		$page,
		$public_pages
	);
}

/**
 * Filters the admin URL to redirect it to a user-specific dashboard URL.
 *
 * This function checks if the URL contains 'admin.php' and modifies it to point
 * to the user's dashboard URL instead. It also extracts the relevant page name
 * from the query parameters and appends it to the dashboard URL.
 *
 * @param string $good_protocol_url The URL with the correct protocol (http/https).
 * @param string $original_url The original URL before protocol correction.
 * @param string $_context The context in which the URL is being filtered.
 * @return string The filtered URL.
 */
function streamtube_core_filter_statistics_admin_url( $good_protocol_url = '', $original_url = '', $_context = '' ) {

	if ( str_contains( $original_url, WP_STATISTICS_SITE_URL ) ) {
		return add_query_arg( array(
			'ref' => StreamTube_Core_WP_Statistics::AFFID
		), $original_url );
	}

	// Return the URL as is if it doesn't contain 'admin.php'
	if ( ! str_contains( $original_url, 'admin.php' ) ) {
		return $good_protocol_url;
	}

	// Generate the base user dashboard URL
	$user_url = untrailingslashit( streamtube_core_statistics_page_url() );

	// Parse the original URL
	$parsed_url = parse_url( $original_url );

	// Return the URL as is if there's no query string
	if ( ! isset( $parsed_url['query'] ) ) {
		return $good_protocol_url;
	}

	// Parse the query string into an array
	$params = array();
	wp_parse_str( $parsed_url['query'], $params );

	// Extract the 'page' parameter and modify the user URL if a match is found
	$page = $params['page'] ?? '';
	if ( preg_match( '/wps_(.*?)_page/', $page, $matches ) ) {
		if ( $matches[1] !== 'overview' ) {
			$user_url .= '/' . $matches[1];
		}

		if ( in_array( $matches[1], array( 'settings', 'optimization', 'privacy-audit', 'plugins' ) ) ) {
			if ( current_user_can( WP_STATISTICS\Option::get( 'manage_capability', 'manage_options' ) ) ) {
				return $good_protocol_url;
			} else {
				return '#';
			}
		}

		if ( ! streamtube_core_is_statistics_public_pages( $matches[1] ) && ! StreamTube_Core_WP_Statistics_Permission::view_all_statistics() ) {
			return '#';
		}
	}

	// Remove the 'page' parameter from the query parameters
	unset( $params['page'] );

	if ( isset( $params['from'] ) && $params['from'] === '1970-01-01' && isset( $params['post_id'] ) ) {
		$params['from'] = get_the_date( 'Y-m-d', $params['post_id'] );
	}

	// Rebuild the URL with the remaining query parameters
	return add_query_arg( $params, $user_url );
}

/**
 * Sanitizes the URL used for tab navigation.
 *
 * @param string $safe_text The already sanitized text.
 * @param string $text      The original text before sanitization.
 * @return string The filtered and sanitized URL.
 * 
 */
function streamtube_core_filter_statistics_attribute_escape( $safe_text, $text ) {
	return streamtube_core_filter_statistics_admin_url( $safe_text, $text );
}

/**
 *
 * Metrics options
 * 
 * @return array
 */
function streamtube_core_get_statistics_metrics() {
	return array(
		'views'    => esc_html__( 'Views (Hits)', 'streamtube-core' ),
		'visitors' => esc_html__( 'Visitors (Unique Views)', 'streamtube-core' )
	);
}

/**
 *
 * Get metric
 * 
 * @return array
 */
function streamtube_core_get_statistics_metric() {
	$metric = get_option( 'wpstatistics_metric', 'views' );

	if ( ! $metric || empty( $metric ) ) {
		$metric = 'views';
	}

	return apply_filters(
		'streamtube/core/wpstatistics/metric',
		$metric
	);
}