<?php
if (!defined('ABSPATH')) {
    exit; // Exit if accessed directly
}

use WP_Statistics\Utils\Request;
use WP_STATISTICS\Admin_Template;
use WP_STATISTICS\Admin_Assets;
use WP_STATISTICS\Menus;
use WP_Statistics\Components\View;
use WP_Statistics\Service\Admin\Referrals\Views\TabsView;

class StreamTube_Core_WP_Statistics_Referrals_TabsView extends TabsView{

    protected $defaultTab = 'referrers';

    protected $tabs = [
        'referrers',
        'search-engines'
    ];    

    protected $visitorsModel;

    protected $args;

    public function __construct(){
        $this->args = $this->getFilters();

        $this->visitorsModel = new StreamTube_Core_WP_Statistics_VisitorsModel();

        if( $this->canViewAll() ){
            $this->defaultTab = 'referred-visitors';

            $this->tabs[] = 'referred-visitors';
        }
    }    

    public function canViewAll(){
        return StreamTube_Core_WP_Statistics_Permission::view_all_statistics();
    }

    private function getFilters(){
        $args = array(
            'referrer'          => Request::get('referrer'),
            'source_channel'    => Request::get('source_channel'),
            'order'             => Request::get('order', 'DESC'),
            'per_page'          => Admin_Template::$item_per_page,
            'page'              => Admin_Template::getCurrentPaged()
        );

        if( ! $this->canViewAll() ){
            $args = array_merge( $args, array(
                'author_id'     => get_current_user_id(),
                'post_type'     => array_keys( streamtube_core_get_statistics_post_types() ),
                'resource_type' => streamtube_core_get_statistics_resource_types()
            ) );
        }        

        return $args;
    }    

    public function getReferredVisitorsData(){

        return [
            'visitors' => $this->visitorsModel->getReferredVisitors( $this->args ),
            'total'    => $this->visitorsModel->countReferredVisitors( $this->args )
        ];
    }    

    public function getReferrersData(){

        return [
            'referrers' => $this->visitorsModel->getReferrers( array_merge( $this->args, [
                'decorate' => true,
                'group_by' => ['visitor.referred', 'visitor.source_channel']
            ])),
            'total'     => $this->visitorsModel->countReferrers( $this->args )
        ];
    }    

    private function getSearchEngineChart(){
        $args = [
            'source_channel' => Request::get('source_channel', ['search', 'paid_search'])
        ];        

        $SearchEnginesChartsData = new StreamTube_Core_WP_Statistics_SearchEngineChartDataProvider(
            array_merge( $this->args, $args )
        );

        return array(
            'search_engine_chart_data'  =>  $SearchEnginesChartsData->getData()
        );        
    }

    public function getSearchEnginesData(){

        wp_localize_script(
            Admin_Assets::$prefix, 'Wp_Statistics_Referrals_Object', 
            $this->getSearchEngineChart()
        );

        return [
            'referrers' => $this->visitorsModel->getReferrers(array_merge( $this->args, [
                'source_channel'    => Request::get('source_channel', ['search', 'paid_search']),
                'decorate'          => true,
                'group_by'          => ['visitor.referred', 'visitor.source_channel']
            ])),
            'total'     => $this->visitorsModel->countReferrers(array_merge($this->args, ['source_channel' => Request::get('source_channel', ['search', 'paid_search'])]))
        ];
    }    

    private function getSocialMediaChartsData(){
        $args = [
            'source_channel' => Request::get('source_channel', ['social', 'paid_social'])
        ];

        $socialMediaChart = new StreamTube_Core_WP_Statistics_SocialMediaChartDataProvider( array_merge($this->args, $args ) );

        return [
            'social_media_chart_data' => $socialMediaChart->getData()
        ];
    }

    public function getSocialMediaData(){
        wp_localize_script(
            Admin_Assets::$prefix, 'Wp_Statistics_Referrals_Object', 
            $this->getSocialMediaChartsData()
        );

        return [
            'referrers' => $this->visitorsModel->getReferrers(array_merge($this->args, [
                'source_channel'    => Request::get('source_channel', ['social', 'paid_social']),
                'decorate'          => true,
                'group_by'          => ['visitor.referred', 'visitor.source_channel']
            ])),
            'total'     => $this->visitorsModel->countReferrers(array_merge($this->args, ['source_channel' => Request::get('source_channel', ['social', 'paid_social'])]))
        ];
    }    

    private function getSourceCategoryChartsData(){
        $SourceCategoryChartDataProvider = new StreamTube_Core_WP_Statistics_SourceCategoryChartDataProvider($this->args);

        return [
            'source_category_chart_data' => $SourceCategoryChartDataProvider->getData()
        ];        
    }

    public function getSourceCategoriesData(){

        wp_localize_script(
            Admin_Assets::$prefix, 'Wp_Statistics_Referrals_Object', 
            $this->getSourceCategoryChartsData()
        );

        $sourceCategories = $this->visitorsModel->getReferrers(array_merge($this->args, [
            'group_by' => ['visitor.source_channel'],
            'decorate' => true,
            'not_null' => false
        ]));

        $total = 0;
        foreach ($sourceCategories as $sourceCategory) {
            $total += $sourceCategory->getTotalReferrals(true);
        }

        return [
            'categories' => $sourceCategories,
            'total'      => $total
        ];
    }    

    public function render(){
        try {
            $data       = $this->getTabData();
            $template   = $this->getCurrentTab();

            $args = [
                'title'       => esc_html__('Referrals', 'wp-statistics'),
                'pageName'    => Menus::get_page_slug('referrals'),
                'custom_get'  => [
                    'tab'       => $this->getCurrentTab(),
                    'order_by'  => Request::get('order_by'),
                    'order'     => Request::get('order'),
                    'referrer'  => Request::get('referrer')
                ],
                'filters'     => ['source-channels'],
                'DateRang'    => Admin_Template::DateRange(),
                'hasDateRang' => true,
                'data'        => $data,
                'pagination'  => Admin_Template::paginate_links([
                    'total' => $data['total'] ?? 0,
                    'echo'  => false
                ]),
                'tabs'        => [
                    [
                        'link'  => Menus::admin_url('referrals', ['tab' => 'referrers']),
                        'title' => esc_html__('Referrers', 'wp-statistics'),
                        'class' => $this->isTab('referrers') ? 'current' : '',
                    ],
                    [
                        'link'  => Menus::admin_url('referrals', ['tab' => 'search-engines']),
                        'title' => esc_html__('Search Engines', 'wp-statistics'),
                        'class' => $this->isTab('search-engines') ? 'current' : '',
                    ],
                    [
                        'link'  => Menus::admin_url('referrals', ['tab' => 'social-media']),
                        'title' => esc_html__('Social Media', 'wp-statistics'),
                        'class' => $this->isTab('social-media') ? 'current' : '',
                    ],
                    [
                        'link'  => Menus::admin_url('referrals', ['tab' => 'source-categories']),
                        'title' => esc_html__('Source Categories', 'wp-statistics'),
                        'class' => $this->isTab('source-categories') ? 'current' : '',
                    ]
                ]
            ];

            if( $this->canViewAll() ){
                $args['tabs'] = array_merge( array(
                    [
                        'link'  => Menus::admin_url('referrals', ['tab' => 'referred-visitors']),
                        'title' => esc_html__('Referred Visitors', 'wp-statistics'),
                        'class' => $this->isTab('referred-visitors') ? 'current' : '',
                    ]
                ), $args['tabs'] );
            }

            // Add referrer filter if tab is referred visitors
            if ($this->isTab('referred-visitors')) {
                array_unshift($args['filters'], 'referrer');
            }

            // Remove source channels filter if tab is source categories
            if ($this->isTab('source-categories')) {
                $args['filters'] = array_values(array_diff($args['filters'], ['source-channels']));
            }

            // Add search channels filter if tab is search engines
            if ($this->isTab('search-engines')) {
                $args['filters'] = ['search-channels'];
            }

            // Add social channels filter if tab is it's social media tab
            if ($this->isTab('social-media')) {
                $args['filters'] = ['social-channels'];
            }

            Admin_Template::get_template(['layout/header', 'layout/tabbed-page-header'], $args);
            View::load("pages/referrals/$template", $args);
            Admin_Template::get_template(['layout/postbox.hide', 'layout/referrer.filter', 'layout/footer'], $args);
        } catch (Exception $e) {
            Notice::renderNotice($e->getMessage(), $e->getCode(), 'error');
        }
    }    
}