<?php
/**
 * WooCommerce
 *
 * @link       https://themeforest.net/user/phpface
 * @since      1.0.0
 *
 * @package    Streamtube_Core
 * @subpackage Streamtube_Core/includes
 */

if ( ! defined( 'ABSPATH' ) ) {
	exit;
}
class Streamtube_Core_Woocommerce {

	const SLUG = 'shopping';

	/**
	 * Holds the Sell Content object for managing product sales.
	 *
	 * @var object
	 */

	public $sell_content;

	public $sell_capability;

	public $stripe;
	public function __construct() {

		$this->load_dependencies();

		$this->sell_content = new StreamTube_Core_Woocommerce_Sell_Content();

		$this->sell_capability = new StreamTube_Core_Woocommerce_Sell_Capability();

		$this->stripe = new StreamTube_Core_Woocommerce_Stripe();
	}

	/**
	 *
	 * Get store slug
	 * 
	 * @return string
	 */
	public static function get_shopping_slug() {
		return apply_filters( 'streamtube/core/user/profile/shopping_slug', self::SLUG );
	}

	/**
	 *
	 * Load the required dependencies for this plugin.
	 * 
	 * @since 1.1
	 */
	private function load_dependencies() {
		$this->include_file( 'class-streamtube-core-woocommerce-stripe.php' );
		$this->include_file( 'class-streamtube-core-woocommerce-permission.php' );
		$this->include_file( 'class-streamtube-core-woocommerce-sell-content.php' );
		$this->include_file( 'class-streamtube-core-woocommerce-sell-capability.php' );
	}

	/**
	 * Loads required WooCommerce functions and classes.
	 * Used for generating metaboxes
	 */
	private function load_wc_admin_dependencies() {

		if ( ! defined( 'WC_PLUGIN_FILE' ) ) {
			return;
		}

		$wc_plugin_dir = dirname( WC_PLUGIN_FILE );

		if ( ! function_exists( 'wc_get_default_product_type_options' ) ) {
			require_once $wc_plugin_dir . '/includes/admin/wc-admin-functions.php';
		}

		if ( ! function_exists( 'woocommerce_wp_text_input' ) ) {
			require_once $wc_plugin_dir . '/includes/admin/wc-meta-box-functions.php';
		}

		if ( ! class_exists( 'WC_Marketplace_Suggestions' ) ) {
			require_once $wc_plugin_dir . '/includes/admin/marketplace-suggestions/class-wc-marketplace-suggestions.php';
		}
	}

	/**
	 *
	 * Include file in WP environment
	 * 
	 * @param  string $file
	 *
	 * @since 1.0.9
	 * 
	 */
	protected function include_file( $file ) {
		require_once trailingslashit( plugin_dir_path( __FILE__ ) ) . $file;
	}

	/**
	 *
	 * Remove default Woocommerce hooks
	 * 
	 */
	public function remove_default() {

		// Remove default single product title
		remove_action( 'woocommerce_single_product_summary', 'woocommerce_template_single_title', 5 );

		// Remove tab description title
		add_filter( 'woocommerce_product_description_heading', '__return_null' );
		add_filter( 'woocommerce_product_additional_information_heading', '__return_null' );

		remove_action( 'woocommerce_before_shop_loop_item', 'woocommerce_template_loop_product_link_open', 10 );
		remove_action( 'woocommerce_after_shop_loop_item', 'woocommerce_template_loop_product_link_close', 5 );

		remove_action( 'woocommerce_before_shop_loop_item_title', 'woocommerce_template_loop_product_thumbnail', 10 );

		remove_action( 'woocommerce_shop_loop_item_title', 'woocommerce_template_loop_product_title', 10 );

		// Remove default product thumbnails
		//remove_action( 'woocommerce_product_thumbnails', 'woocommerce_show_product_thumbnails', 20 );

		// Remove the default rating and display it under the product title instead
		// display_single_product_rating()
		remove_action( 'woocommerce_single_product_summary', 'woocommerce_template_single_rating', 10 );

		// Remove WC lost password
		remove_filter( 'lostpassword_url', 'wc_lostpassword_url', 10 );

		remove_action( 'template_redirect', 'wc_disable_author_archives_for_customers', 10 );
	}

	/**
	 *
	 * Check if within StreamTube Dashboard
	 * 
	 * @return boolean
	 */
	public function is_shopping( $wp = null ) {

		if ( ! $wp || $wp === null ) {
			global $wp;
		}

		if ( ! is_object( $wp ) || ! is_array( $wp->query_vars ) ) {
			return false;
		}

		if ( array_key_exists( 'dashboard', $wp->query_vars ) ) {
			if ( strpos( $wp->query_vars['dashboard'], self::get_shopping_slug() ) !== false ) {
				return true;
			}
		}

		return false;
	}

	/**
	 *
	 * Preparse query
	 * 
	 */
	public function parse_request( $wp ) {
		if ( $this->is_shopping( $wp ) ) {
			$request = explode( '/', untrailingslashit( $wp->query_vars['dashboard'] ) );
			if ( count( $request ) > 1 ) {
				$wp->query_vars[ $request[1] ] = isset( $request[2] ) ? $request[2] : '';
			}
		}
	}

	/**
	 *
	 * Extra endpoints that are not available under Shopping menu item
	 * 
	 * @return array
	 */
	public static function get_extra_endpoints() {
		return apply_filters( 'streamtube/core/woocommerce/extra_endpoints', array(
			'delete-payment-method',
			'add-payment-method',
			'set-default-payment-method',
			'view-rma-requests',
			'request-warranty',
			'view-order',
			'edit-account'
		) );
	}

	/**
	 *
	 * Filter the my-account shortcode menu items
	 * Remove logout, dashboard and other items if already supported by the theme
	 *
	 * Add extra endpoints to the my-acount shortcode
	 * 
	 */
	public function filter_woocommerce_account_menu_items( $items, $endpoints ) {

		unset( $items['dashboard'] );
		//unset( $items['edit-account'] );
		unset( $items['customer-logout'] );

		$new_items = array(
			'purchased-products' => esc_html__( 'Purchased products', 'streamtube-core' )
		);

		if ( get_option( 'woocommerce_sell_content', 'on' ) ) {
			$new_items['purchased-videos'] = esc_html__( 'Purchased videos', 'streamtube-core' );
		}

		if ( function_exists( 'WPPL' ) ) {
			$new_items['liked-products'] = esc_html__( 'Liked products', 'streamtube-core' );
		}

		return array_merge( $items, $new_items );
	}

	/**
	 *
	 * Fix the current menu item if any extra params found but not belong to current shopping menu
	 *
	 * Dokan add extra endpoints such as `support-tickets`, `request-a-quote` ...
	 *
	 * We catch the params and return the `shopping` as current requested menu
	 * 
	 */
	public function set_current_dashboard_shopping_menu( $current, $request ) {

		// Get all WooCommerce account menu endpoints
		$endpoints = array_merge( array_keys( wc_get_account_menu_items() ), self::get_extra_endpoints() );

		// Build the dashboard query variable once
		$dashboard_query = get_query_var( 'dashboard' );

		foreach ( $endpoints as $endpoint ) {
			// Construct the dynamic pattern for each endpoint
			$pattern = sprintf(
				'/%s\/%s\/([^\/]+)(\/|$)/',
				preg_quote( self::get_shopping_slug(), '/' ),
				preg_quote( $endpoint, '/' )
			);

			// Check if the pattern matches the current dashboard query
			if ( preg_match( $pattern, $dashboard_query, $matches ) ) {
				$current = self::get_shopping_slug();
				break;
			}
		}

		return $current;
	}

	/**
	 * Filter the WC endpoint
	 * 
	 */
	public function filter_woocommerce_get_endpoint_url( $url, $endpoint, $value, $permalink ) {

		$endpoints = array_merge( array_keys( wc_get_account_menu_items() ), self::get_extra_endpoints() );

		/**
		 *
		 * Filter endpoints
		 * Redirect to dashboard if found
		 * 
		 */
		$endpoints = apply_filters( 'streamtube/core/woocommerce/endpoints', $endpoints );

		if ( in_array( $endpoint, $endpoints ) && is_user_logged_in() && $this->is_shopping() ) {

			$author_url = trailingslashit( get_author_posts_url( get_current_user_id() ) );

			if ( $endpoints === 'edit-account' ) {
				$url = $author_url . 'dashboard/account/personal/';
			} else {
				$url = $author_url . 'dashboard/' . self::get_shopping_slug() . '/' . $endpoint;
			}

			$url = trailingslashit( $url );

			if ( $value ) {
				$url = trailingslashit( $url ) . $value;
			}
		}

		return $url;
	}

	/**
	 * Redirects the WooCommerce edit account URL to the user's profile page if the user is shopping and on the edit account page.
	 *
	 * This function checks if the user is currently in a shopping context and if the request URI contains 'edit-account'.
	 * If both conditions are met, it redirects the user to their profile page's personal account settings.
	 *
	 * @return void
	 */
	public function redirect_woocommerce_edit_account_url() {
		if ( $this->is_shopping() && strpos( $_SERVER['REQUEST_URI'], 'edit-account' ) !== false ) {
			wp_redirect( trailingslashit( get_author_posts_url( get_current_user_id() ) ) . 'dashboard/account/personal/' );
			exit;
		}
	}

	/**
	 *
	 * Add `woocommerce` class to bodyclass
	 * 
	 */
	public function filter_body_classes( $classes ) {

		global $wp_query;

		if (
			isset( $wp_query->query_vars['dashboard'] ) &&
			( strpos( $wp_query->query_vars['dashboard'], 'shopping' ) === 0 || empty( $wp_query->query_vars['dashboard'] ) ) ||
			( function_exists( 'is_cart' ) && is_cart() ) ||
			( function_exists( 'is_checkout' ) && is_checkout() )
		) {
			$classes[] = 'woocommerce';
		}

		return $classes;
	}

	/**
	 *
	 * Get product URL
	 * 
	 */
	private function get_product_url() {

		global $product;

		return apply_filters( 'woocommerce_loop_product_link', get_the_permalink(), $product );
	}

	/**
	 *
	 * Get the thumbnail image ratio
	 * 16x9 is default
	 * 
	 * @return string
	 */
	private function get_thumbnail_ratio() {
		return apply_filters(
			'woocommerce_loop_product_thumbnail_ratio',
			get_option( 'woocommerce_thumbnail_ratio', '16x9' )
		);
	}

	/**
	 *
	 * Get the thumbnail image ratio
	 * 16x9 is default
	 * 
	 * @return string
	 */
	private function get_thumbnail_size() {
		return apply_filters(
			'woocommerce_loop_product_thumbnail_size',
			get_option( 'woocommerce_thumbnail_size', 'streamtube-image-medium' )
		);
	}

	/**
	 *
	 * Get the single thumbnail image ratio
	 * 16x9 is default
	 * 
	 * @return string
	 */
	private function get_single_thumbnail_ratio() {
		return apply_filters(
			'woocommerce_single_product_thumbnail_ratio',
			get_option( 'woocommerce_single_thumbnail_ratio', '16x9' )
		);
	}

	/**
	 *
	 * Display the product loop thumbnail
	 * 
	 */
	public function display_template_loop_product_thumbnail() {
		printf(
			'<a href="%s"><div class="post-thumbnail ratio ratio-%s rounded overflow-hidden bg-dark">%s</div></a>',
			esc_url( $this->get_product_url() ),
			$this->get_thumbnail_ratio(),
			woocommerce_get_product_thumbnail()
		);
	}

	/**
	 *
	 * Display the product loop title
	 * 
	 */
	public function display_template_loop_product_title() {

		$class = apply_filters( 'woocommerce_product_loop_title_classes', 'woocommerce-loop-product__title post-title' );

		printf(
			'<h2 class="%s"><a href="%s" class="text-body">%s</a></h2>',
			esc_attr( $class ),
			esc_url( $this->get_product_url() ),
			get_the_title()
		);
	}

	/**
	 *
	 * Get cart content
	 * 
	 * @return array
	 *
	 * @since 1.0.5
	 * 
	 */
	public function get_cart_total() {

		$item_count = WC()->cart->get_cart_contents_count();

		return array(
			'item_count'      => (int) $item_count,
			'item_count_text' => sprintf(
				_n( '%s item', '%s items', $item_count, 'streamtube-core' ),
				number_format_i18n( $item_count )
			),
			'total'           => wc_price( WC()->cart->total )
		);
	}

	/**
	 *
	 * AJAX get cart content
	 * 
	 * @return prints JSON results
	 *
	 * @since 1.0.5
	 * 
	 */
	public function ajax_get_cart_total() {

		check_ajax_referer( '_wpnonce' );

		wp_send_json_success( $this->get_cart_total() );
	}

	/**
	 *
	 * The Cart button
	 * 
	 */
	public function the_cart_button() {

		if (
			! get_option( 'woocommerce_enable_header_cart', 'on' ) ||
			streamtube_core_has_mobile_footer_bar() ) {
			return;
		}

		$count = WC()->cart->get_cart_contents_count();

		?>
		<div class="header-user-woocommerce woocommerce">
			<div class="header-user__cart">
				<div class="dropdown">
					<button class="btn btn-cart shadow-none px-2 position-relative" data-bs-toggle="dropdown"
						data-bs-display="static">
						<span class="btn__icon icon-cart-plus"></span>
						<?php printf(
							'<span class="badge cart-count bg-danger position-absolute top-0 end-0 %s">%s</span>',
							(int) $count == 0 ? 'd-none' : '',
							(int) $count > 0 ? number_format_i18n( $count ) : ''
						) ?>
					</button>

					<div class="dropdown-menu dropdown-menu-end dropdown-menu-mini-cart">
						<div class="widget-title-wrap d-flex m-0 p-0 border-bottom">
							<h2 class="widget-title p-3 m-0 no-after"><?php esc_html_e( 'Shopping cart', 'streamtube-core' ); ?>
							</h2>
						</div>
						<div class="widget_shopping_cart_content">
							<?php woocommerce_mini_cart(); ?>
						</div>
					</div>
				</div>
			</div>
		</div>
		<?php
	}

	/**
	 *
	 * Only completed orders will be considered as paid
	 * 
	 * @param  array  $statuses
	 * 
	 */
	public function filter_order_is_paid_statuses( $statuses = array() ) {
		return array( 'completed' );
	}

	/**
	 *
	 * Always return true if current page is User profile page
	 * 
	 * @return boolean
	 */
	public function filter_is_account_page( $retvar ) {
		if ( get_query_var( 'dashboard' ) && is_author() ) {
			return true;
		}

		return $retvar;
	}

	/**
	 *
	 * Filter is_purchasable
	 *
	 * By default, only published product is purchasable
	 * We support "unlist" status as well
	 * 
	 */
	public function filter_is_purchasable( $retvar, $product ) {

		if (
			$product->exists() &&
			in_array( $product->get_status(), array( 'publish', 'unlist' ) ) &&
			'' !== $product->get_price() &&
			$this->sell_content->is_builitin_product( $product->get_id() )
		) {
			$retvar = true;
		}

		return $retvar;
	}

	/**
	 *
	 * Display cart product count
	 * 
	 */
	public function filter_wp_menu_item_title( $title, $wpmi, $item, $args, $depth ) {

		if ( ! is_object( $item ) ) {
			return $title;
		}

		if ( $item->url == wc_get_cart_url() ) {
			$count = WC()->cart->get_cart_contents_count();
			$title .= sprintf(
				'<span class="menu-badge badge bg-danger cart-count %s">%s</span>',
				(int) $count == 0 ? 'd-none' : '',
				number_format_i18n( $count )
			);
		}

		return $title;
	}

	/**
	 *
	 * Add more attributes to the [products] shortcode
	 * 
	 */
	public function filter_shortcode_atts_products( $out, $pairs, $atts, $shortcode ) {

		$atts = wp_parse_args( $atts, array(
			'author'      => 0,
			'post_status' => '',
			'search'      => ''

		) );

		return array_merge( $out, $atts );
	}

	/**
	 *
	 * Filter the [products] shortcode query
	 * 
	 */
	public function filter_shortcode_products_query( $args, $atts = array(), $type = '' ) {
		$atts = wp_parse_args( $atts, array(
			'author'      => 0,
			'post_status' => '',
			'search'      => ''
		) );

		if ( $atts['author'] ) {
			$args['author'] = (int) $atts['author'];
		}

		if ( $atts['post_status'] ) {
			$args['post_status'] = array_map( 'trim', array_filter( explode( ',', $atts['post_status'] ) ) );
		}

		if ( $atts['search'] ) {
			$args['s'] = wp_unslash( $atts['search'] );
		}

		return $args;
	}

	/**
	 *
	 * Display unlist product as well if "unlist" status found
	 * 
	 */
	public function filter_product_is_visible() {
		global $product;

		if ( is_object( $product ) && in_array( $product->get_status(), array( 'publish', 'unlist' ) ) ) {
			return true;
		}

		return false;
	}

	public function display_single_product_rating() {
		woocommerce_template_single_rating();
	}

	/**
	 *
	 * Add wrapper to single thumbnail image
	 * Make it appear within defined aspect ratio
	 *
	 * @param string $html
	 * @param int $post_thumbnail_id
	 * 
	 */
	public function filter_single_product_image_thumbnail_html( $html, $post_thumbnail_id ) {
		//wp_get_attachment_image( $post_thumbnail_id, $this->get_single_thumbnail_size() )
		return sprintf(
			'<div class="shadow-sm product-thumbnail post-thumbnail ratio ratio-%s rounded overflow-hidden bg-dark">%s</div>',
			esc_attr( $this->get_single_thumbnail_ratio() ),
			$html
		);
	}

	/**
	 *
	 * Set default thumbnail size
	 * 
	 * @param  string $size
	 * 
	 */
	public function filter_thumbnail_image_size( $size ) {
		return $this->get_thumbnail_size();
	}

	/**
	 *
	 * `streamtube/core/dashboard/menu/current`
	 * 
	 */
	public function filter_dashboard_current_menu( $current, $request ) {

		if ( is_array( $request ) ) {
			if ( in_array( 'view-order', $request ) || in_array( 'request-warranty', $request ) ) {
				$current = self::get_shopping_slug();
			}
		}

		return $current;
	}

	/**
	 *
	 * Filter dashboard page heading
	 *
	 */
	public function filter_page_heading( $heading = '' ) {

		if ( ! $this->is_shopping() ) {
			return $heading;
		}

		foreach ( wc_get_account_menu_items() as $endpoint => $label ) {
			if ( strpos( get_query_var( 'dashboard' ), $endpoint ) !== false ) {
				return $label;
			}
		}

		return $heading;
	}

	/**
	 * Filters the JSON search results for products.
	 *
	 * This function removes products from the search results that do not belong
	 * to the currently logged-in user, unless the user has the 'edit_others_posts'
	 * capability (e.g., administrator or editor).
	 *
	 * @param array $products An associative array of product IDs and values.
	 * @return array The filtered array of product IDs and values.
	 */
	public function filter_json_search_found_products( $products = array() ) {

		if ( ! $products ) {
			return $products;
		}

		if ( ! current_user_can( get_post_type_object( 'product' )->cap->edit_others_posts ) ) {
			foreach ( $products as $product_id => $value ) {
				if ( get_post( $product_id )->post_author !== get_current_user_id() ) {
					unset( $products[ $product_id ] );
				}
			}
		}

		return $products;
	}

	/**
	 *
	 * Display notifications within user dashboard
	 * 
	 */
	public function display_dashboard_notices() {
		if ( function_exists( 'wc_print_notices' ) ) {
			wc_print_notices();
		}
	}

	/**
	 *
	 * Display the purchased products if `purchased-products` query found
	 * 
	 */
	public function display_purchased_products() {
		streamtube_core_load_template( 'user/dashboard/shopping/purchased-products.php' );
	}

	/**
	 *
	 * Display the purchased videos if `purchased-products` query found
	 * 
	 * @return [type] [description]
	 */
	public function display_purchased_videos() {
		streamtube_core_load_template( 'user/dashboard/shopping/purchased-videos.php' );
	}

	/**
	 *
	 * Display the liked products if `purchased-products` query found
	 * 
	 */
	public function display_liked_products() {
		streamtube_core_load_template( 'user/dashboard/shopping/liked-products.php' );
	}

	/**
	 * Adds WooCommerce product data meta boxes to a custom dashboard screen.
	 *
	 * @param string $screen The current screen ID.
	 * @param object $box    The meta box object.
	 */
	public function add_meta_boxes( $screen, $box ) {
		if ( $screen === 'dashboard_product' ) {

			streamtube_remove_meta_box( 'product-discussion', $screen, 'normal' );
			streamtube_remove_meta_box( 'product-excerpt', $screen, 'advanced' );

			if ( class_exists( 'WC_Meta_Box_Product_Data' ) ) {
				// Add the WooCommerce product data meta box.
				streamtube_add_meta_box(
					'woocommerce-product-data',
					esc_html__( 'Product Data', 'streamtube-core' ),
					'WC_Meta_Box_Product_Data::output',
					$screen,
					'advanced'
				);
			}

			if ( class_exists( 'WC_Meta_Box_Product_Data' ) ) {
				// Add the WooCommerce product data meta box.
				streamtube_add_meta_box(
					'woocommerce-postexcerpt',
					esc_html__( 'Product short description', 'streamtube-core' ),
					'WC_Meta_Box_Product_Short_Description::output',
					$screen,
					'normal'
				);
			}

			if ( class_exists( 'WC_Meta_Box_Product_Images' ) ) {
				// Add the WooCommerce product data meta box.
				streamtube_add_meta_box(
					'woocommerce-product-images',
					esc_html__( 'Product gallery', 'streamtube-core' ),
					'WC_Meta_Box_Product_Images::output',
					$screen,
					'side'
				);
			}
		}
	}

	/**
	 *
	 * Add `Shopping` dashboard menu
	 * 
	 */
	public function add_dashboard_menu( $menu_items ) {

		$menu_items[ self::get_shopping_slug()] = array(
			'title'    => esc_html__( 'Shopping', 'streamtube-core' ),
			'icon'     => 'icon-cart-plus',
			'callback' => function () {
				streamtube_core_load_template( 'user/dashboard/shopping.php' );
			},
			'parent'   => 'dashboard',
			'cap'      => 'read',
			'url'      => '#',
			'priority' => 5,
			'submenu'  => array()
		);

		foreach ( wc_get_account_menu_items() as $endpoint => $label ) {
			$menu_items['shopping']['submenu'][ $endpoint ] = array(
				'title'    => $label,
				'callback' => function () {
					streamtube_core_load_template( 'user/dashboard/shopping.php' );
				},
				'cap'      => 'read',
				'icon'     => 'icon-minus',
				'priority' => 20
			);
		}

		return $menu_items;
	}

	/**
	 * Add `woocommerce_sell_content` post type support
	 * 
	 */
	public function add_post_type_supports() {
		add_post_type_support( 'video', 'woocommerce_sell_content' );
	}

	/**
	 *
	 * Filter Product register post type args
	 * Enable `show_ui_frontend`
	 * 
	 */
	public function filter_register_post_type_args( $args, $post_type ) {
		if ( $post_type === 'product' && ! function_exists( 'dokan' ) ) {
			$args['show_ui_frontend'] = true;
		}
		return $args;
	}

	/**
	 *
	 * Permission screen
	 * 
	 */
	public function register_permission_module() {
		streamtube_core_register_module(
			'woocommerce',
			esc_html__( 'WooCommerce', 'streamtube-core' ),
			array(
				'woocommerce_sell_content'
			)
		);
	}

	/**
	 * Enqueues WooCommerce admin scripts and styles for product pages.
	 *
	 * This function is used to load necessary assets for generating metaboxes
	 * on WooCommerce product pages.
	 */
	public function enqueue_admin_scripts() {

		global $wp_query;

		if ( isset( $wp_query->query_vars['dashboard'] ) && preg_match( '/product\/([0-9]+|\?view=add-post)/', $_SERVER['REQUEST_URI'] ) ) {
			global $current_screen, $temp_current_screen;

			// Ensure WooCommerce dependencies are loaded.
			$this->load_wc_admin_dependencies();

			if ( class_exists( 'WC_Admin_Assets' ) ) {

				if ( $current_screen ) {
					// Store the original screen object.
					$temp_current_screen = $current_screen;
				}

				// Temporarily set the screen to 'edit-product' to load WooCommerce assets.
				set_current_screen( 'edit-product' );

				// Instantiate and enqueue WooCommerce admin assets.
				$wcAdminAssets = new WC_Admin_Assets();
				$wcAdminAssets->admin_scripts();
				$wcAdminAssets->admin_styles();

				// Restore the original screen object.
				unset( $GLOBALS['current_screen'] );
			}

			if ( $temp_current_screen ) {
				// Restore the original screen object after processing.
				$GLOBALS['current_screen'] = $temp_current_screen;
			}

		}
	}
}

