<?php
/**
 * Define the Storage Role functionality
 *
 *
 * @link       https://themeforest.net/user/phpface
 * @since      1.0.0
 *
 * @package    Streamtube_Core
 * @subpackage Streamtube_Core/includes
 */

/**
 *
 * @since      1.0.8
 * @package    Streamtube_Core
 * @subpackage Streamtube_Core/includes
 * @author     phpface <nttoanbrvt@gmail.com>
 */

if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

class StreamTube_Core_Storage {

	public function is_valid_quota_capacity( $capacity = '' ) {
		if ( is_string( $capacity ) && strpos( $capacity, 'storage_capacity_' ) === 0 && preg_match( '/\d+/', $capacity, $matches ) ) {
			return absint( $matches[0] ) > 0 ? absint( $matches[0] ) : false;
		}

		return false;
	}

	/**
	 *
	 * Add `storage_capacity_` to administrator role.
	 * 
	 * @param WP_Error|true
	 */
	public function add_quota_capacity( $size = 0 ) {

		$size = absint( $size );

		if ( $size < 1 ) {
			return new WP_Error(
				'invalid_capacity',
				esc_html__( 'Invalid capacity', 'streamtube-core' )
			);
		}

		$capacity = 'storage_capacity_' . $size;

		$role = get_role( 'administrator' );

		if ( $role->has_cap( $capacity ) ) {
			return new WP_Error(
				'capacity_exists',
				esc_html__( 'Capacity already exists', 'streamtube-core' )
			);
		}

		$role->add_cap( $capacity );

		return $capacity;
	}

	/**
	 *
	 * Retrieve all added capacities that were added to administrator role
	 * 
	 * @return array
	 */
	public function get_quota_capacities() {

		$capacities = array();

		foreach ( StreamTube_Core_Role::get_all_capabilities( 'administrator' ) as $capability ) {

			$size = $this->is_valid_quota_capacity( $capability );

			if ( $size ) {
				$bytes                     = $size * 1024 * 1024;
				$capacities[ $capability ] = array(
					'bytes'    => $bytes,
					'formated' => size_format( $bytes )
				);
			}
		}

		uasort( $capacities, function ($item1, $item2) {
			return $item1['bytes'] <=> $item2['bytes'] ?? 0;
		} );

		return $capacities;
	}
	/**
	 *
	 * Get Max upload file size for given role
	 * 
	 * @param  string $role
	 *
	 * @return int bytes
	 * 
	 */
	public function get_role_upload_size( $role = '' ) {

		$size = (int) get_option( 'max_upload_size_' . sanitize_key( $role ), streamtube_core_get_default_max_upload_size() );

		if ( ! $size ) {
			$size = 0;
		}

		/**
		 *
		 * Filter max upload file size
		 *
		 * @param int $size
		 * @param string $role
		 * 
		 */
		return apply_filters( 'streamtube/core/storage/get_role_size', $size, $role );
	}

	/**
	 * Get the maximum storage quota for a specific role.
	 *
	 * This function retrieves the maximum storage quota for the given role.
	 * If no specific quota is set, it defaults to 10GB (1024 * 10 * 1024 * 1024 bytes).
	 *
	 * @param string $role The role for which to retrieve the storage quota.
	 * @return int The maximum storage quota in bytes.
	 * 
	 */
	public function get_role_storage_quota( $role = '' ) {
		$size = (int) get_option( 'max_storage_quota_' . sanitize_key( $role ), 0 );

		if ( ! $size ) {
			$size = 0;
		}

		$size = min( $size, PHP_INT_MAX );

		/**
		 *
		 * Filter max upload file size
		 *
		 * @param int $size
		 * @param string $role
		 * 
		 */
		return apply_filters( 'streamtube/core/storage/get_role_quota', $size, $role );
	}

	/**
	 * Get the maximum storage quota of given roles.
	 *
	 * This function calculates the maximum storage quota from an array of roles.
	 * It considers individual quotas set for each role and returns the highest value.
	 *
	 * @param array $roles An array of role names.
	 * @return int|null The maximum storage quota, or null if no roles are provided.
	 */
	public function get_roles_max_storage_quota( $roles = array() ) {

		if ( empty( $roles ) ) {
			return -1;
		}

		$sizes = array();

		foreach ( $roles as $role ) {
			$role_object = get_role( $role );

			if ( $role_object ) {
				$sizes[] = (int) $this->get_role_storage_quota( $role );
			}
		}

		$min = min( $sizes );

		// If the smallest quota is less than or equal to zero, return the minimum
		if ( $min <= 0 ) {
			return $min;
		}

		// Return the maximum quota from the sizes array
		return max( $sizes );
	}

	/**
	 * Update the maximum storage quota for roles
	 *
	 * This function updates the maximum storage quota for specified roles based on
	 * the data received in the request.
	 *
	 * @return bool True if the operation was successful, false otherwise.
	 * 
	 */
	public function update_roles_upload_sizes() {
		// Retrieve and sanitize roles data from the request
		$roles = isset( $_REQUEST['roles'] ) ? wp_unslash( $_REQUEST['roles'] ) : array();

		// Ensure roles is an array
		if ( ! is_array( $roles ) || empty( $roles ) ) {
			return false;
		}

		// Get all editable roles
		$editable_roles = streamtube_core_get_roles();

		// Loop through provided roles and update their upload size
		foreach ( $roles as $role => $size ) {
			if ( array_key_exists( $role, $editable_roles ) ) {

				$bytes = (int) $size != $size ? wp_convert_hr_to_bytes( $size ) : (int) $size * 1024 * 1024;

				$bytes = min( $bytes, PHP_INT_MAX );

				update_option( 'max_upload_size_' . sanitize_key( $role ), $bytes );
			}
		}

		return true;
	}

	/**
	 * Update the maximum storage quota for roles
	 *
	 * This function updates the maximum storage quota for specified roles based on
	 * the data received in the request.
	 *
	 * @return bool True if the operation was successful, false otherwise.
	 */
	public function update_roles_quota() {
		// Retrieve and sanitize roles data from the request
		$roles = isset( $_REQUEST['roles'] ) ? wp_unslash( $_REQUEST['roles'] ) : array();

		// Ensure roles is an array
		if ( ! is_array( $roles ) || empty( $roles ) ) {
			return false;
		}

		// Get all editable roles
		$all_roles = streamtube_core_get_roles();

		// Loop through provided roles and update their upload size
		foreach ( $roles as $role => $size ) {
			if ( array_key_exists( $role, $all_roles ) ) {
				$bytes = (int) $size != $size ? wp_convert_hr_to_bytes( $size ) : (int) $size * 1024 * 1024;

				$bytes = min( $bytes, PHP_INT_MAX );

				update_option( 'max_storage_quota_' . sanitize_key( $role ), $bytes );
			}
		}

		return true;
	}

	/**
	 *
	 * Get max upload file size for given user
	 * 
	 * @param  integer $user_id
	 * @return return false|0|int
	 */
	public function get_user_max_upload_size( $user_id = 0 ) {

		if ( ! $user_id ) {
			$user_id = get_current_user_id();
		}

		$user_roles = get_userdata( $user_id )->roles;

		if ( ! $user_roles ) {
			return false;
		}

		$max_sizes = array();

		$roles = array_values( $user_roles );

		for ( $i = 0; $i < count( $roles ); $i++ ) {

			$role = get_role( $roles[ $i ] );

			$max_sizes[] = $this->get_role_upload_size( $roles[ $i ] );
		}

		if ( ! $max_sizes || min( $max_sizes ) < 0 ) {
			return -1;
		}

		if ( min( $max_sizes ) === 0 ) {
			return 0;
		}

		// Otherwise, we get the max size.
		return max( $max_sizes );
	}

	/**
	 * Retrieves the list of object types that are considered for granted storage sizes.
	 *
	 * This includes default object types like 'initial' and 'purchase'. 
	 * Developers can modify the list by using the `streamtube/core/storage/get_user_granted_sizes/object_types` filter.
	 *
	 * @return array The list of granted object types.
	 */
	public function get_custom_object_types() {

		$types = array(
			'initial'         => esc_html__( 'Initial', 'streamtube-core' ),
			'manual'          => esc_html__( 'Manual', 'streamtube-core' ),
			'custom_capacity' => esc_html__( 'Custom Capacity', 'streamtube-core' )
		);

		if ( is_plugin_active( 'woocommerce/woocommerce.php' ) ) {
			$types['shop_order_placehold'] = esc_html__( 'Purchased', 'streamtube-core' );
		}

		// Apply a filter to allow customization of the granted object types
		return apply_filters(
			'streamtube/core/storage/custom_quota_types',
			$types
		);
	}

	public function get_user_storage_details( $user_id ) {
		$storage = $this->get_user_storage( $user_id );

		$args = array(
			'user_id' => $user_id,
			'quota'   => array(
				'bytes'    => $storage->quota,
				'readable' => size_format( $storage->quota )
			),
			'usage'   => array(
				'bytes'    => $storage->usage,
				'readable' => size_format( $storage->usage )
			),
			'free'    => array(
				'bytes'    => $storage->free,
				'readable' => size_format( $storage->free )
			)
		);

		return apply_filters(
			'streamtube/core/storage/get_user_storage_details',
			$args,
			$storage,
			$user_id
		);
	}

	public function set_user_storage_quota( $user_id = 0, $size = 0 ) {

		if ( ! $user_id ) {
			$user_id = get_current_user_id();
		}

		$size = (int) $size;

		if ( is_numeric( $size ) ) {

			$size = min( $size, PHP_INT_MAX );

			$_db_args = array_merge( compact( 'user_id' ), array(
				'object_type' => 'initial'
			) );

			if ( StreamTube_Core_Storage_DB::exists( $_db_args ) ) {
				StreamTube_Core_Storage_DB::update( array_merge(
					$_db_args,
					compact( 'size' )
				) );
			} else {
				StreamTube_Core_Storage_DB::add( array_merge(
					$_db_args,
					compact( 'size' )
				) );
			}

			/**
			 * Fires after adjusting max quota
			 *
			 * @param int $user_id
			 * @param int $size
			 * 
			 */
			do_action( 'streamtube/core/storage/updated_user_quota', $user_id, $size );

			return $this->get_user_storage_details( $user_id );
		}

		return false;
	}

	/**
	 * Resets the storage usage for a user.
	 *
	 * @param int $user_id The ID of the user whose storage usage is to be reset. Defaults to the current user if not provided.
	 * @return bool|int The number of rows affected on success, false on failure.
	 * 
	 */
	public function reset_user_storage_usage( $user_id = 0 ) {

		if ( ! $user_id ) {
			$user_id = get_current_user_id();
		}

		return StreamTube_Core_Storage_DB::delete( compact( 'user_id' ) );
	}

	/**
	 * Calculates the total initial granted storage sizes for a user.
	 *
	 * @param int $user_id Optional. The ID of the user whose initial granted storage size is being calculated. 
	 *                     Defaults to the current logged-in user if not provided.
	 * @return int The total initial granted storage size in bytes.
	 */
	public function get_user_storage_initial_quota( $user_id = 0 ) {

		// Use the current logged-in user if no user ID is provided
		if ( ! $user_id ) {
			$user_id = get_current_user_id();
		}

		$object_type = 'initial';

		// Retrieve and return the total sizes using the storage database class
		$total = StreamTube_Core_Storage_DB::get_total_sizes(
			compact( 'user_id', 'object_type' )
		);

		if ( is_null( $total ) ) {
			$total = $this->get_roles_max_storage_quota( get_userdata( $user_id )->roles );
		}

		return apply_filters( 'streamtube/core/storage/get_user_initial_quota', $total, $user_id );
	}

	/**
	 * Calculates the total granted storage sizes for a user.
	 * Including initials
	 *
	 * This includes storage sizes of types like 'initial', 'purchase', 'reward', 
	 * or any other types as defined by the `streamtube/core/storage/custom_quota_types` filter.
	 *
	 * @param int $user_id The ID of the user whose granted storage sizes are being calculated.
	 * @return int The total granted storage size in bytes.
	 */
	public function get_user_storage_custom_quota( $user_id = 0 ) {

		if ( ! $user_id ) {
			$user_id = get_current_user_id();
		}

		$initial = $this->get_user_storage_initial_quota( $user_id );

		if ( $initial < 0 ) {
			return $initial;
		}

		$object_type = array_keys( $this->get_custom_object_types() );

		$object_type_not_in = 'initial';

		// Retrieve and return the total sizes using the storage database class
		$total = StreamTube_Core_Storage_DB::get_total_sizes(
			compact( 'user_id', 'object_type', 'object_type_not_in' )
		);

		if ( $initial > 0 ) {
			$total += $initial;
		}

		return apply_filters( 'streamtube/core/storage/get_user_custom_quota', $total, $user_id, $initial );
	}

	/**
	 *
	 * Get total quota including initials and any other granted sizes.
	 * 
	 * @param  integer $user_id
	 * @return return false|0|int
	 */
	public function get_user_storage_quota( $user_id = 0 ) {

		if ( ! $user_id ) {
			$user_id = get_current_user_id();
		}

		$roles = get_userdata( $user_id )->roles;

		if ( $roles ) {
			$roles = array_values( $roles );
		}

		$maybe_user_custom_quota = $this->get_user_storage_custom_quota( $user_id );

		if ( ! is_null( $maybe_user_custom_quota ) ) {

			// Otherwise, we get the max size.
			/**
			 *
			 * Filter the max quota
			 * Let other plugin extend this quota
			 *
			 * @param int $max
			 * @param int $user_id
			 * @param array $roles
			 * 
			 */
			return apply_filters( 'streamtube/core/storage/get_user_quota', (int) $maybe_user_custom_quota, $user_id, $roles );
		}

		return apply_filters(
			'streamtube/core/storage/get_user_quota',
			$this->get_roles_max_storage_quota( $roles ),
			$user_id,
			$roles
		);
	}

	/**
	 * Get the user's current storage usage.
	 *
	 * This function calculates and retrieves the storage usage for the specified user.
	 * If no user ID is provided, it defaults to the current user.
	 *
	 * @param int $user_id The ID of the user. Defaults to the current user if not provided.
	 * @return int The user's current storage usage in bytes.
	 * 
	 */
	public function get_user_storage_usage( $user_id = 0 ) {

		if ( ! $user_id ) {
			$user_id = get_current_user_id();
		}

		$usage = StreamTube_Core_Storage_DB::get_total_sizes( array_merge( compact( 'user_id' ), array(
			'object_type' => array( 'attachment' )
		) ) );

		return $usage ?? 0;
	}

	public function get_user_storage( $user_id = 0 ) {
		if ( ! $user_id ) {
			$user_id = get_current_user_id();
		}

		$quota = $this->get_user_storage_quota( $user_id );
		$usage = $this->get_user_storage_usage( $user_id );
		$free  = max( 0, $quota - $usage );

		return (object) compact( 'quota', 'usage', 'free' );
	}

	/**
	 *
	 * The user storage progress
	 */
	public function get_user_storage_progres( $args = array() ) {

		$args = wp_parse_args( $args, array(
			'user_id'                          => get_current_user_id(),
			'progress_height'                  => '20px',
			'hide_if_unlimited_or_not_allowed' => false,
			'wrap'                             => false
		) );

		extract( $args );

		$storage = $this->get_user_storage( $user_id );

		if ( $storage->quota <= 0 && wp_validate_boolean( $hide_if_unlimited_or_not_allowed ) ) {
			// Unlimited or now allowed
			return;
		}

		$output   = '';
		$progress = 0;

		if ( $storage->quota > 0 ) {
			$progress = ( absint( $storage->usage ) * 100 ) / floatval( $storage->quota );
			$progress = min( 100, max( 0, $progress ) ); // Clamp between 0 and 100
		}

		if ( is_numeric( $progress ) ) {
			$progress = ceil( $progress );
		}

		$progress_class = 'danger';
		$progress_text  = '';

		if ( $storage->quota < 0 ) {
			;
			$progress      = 100;
			$progress_text = esc_html__( 'No quota', 'streamtube-core' );
		} elseif ( $storage->quota === 0 ) {
			$progress_class = 'success';
			$progress       = 100;
			$progress_text  = esc_html__( 'Unlimited', 'streamtube-core' );
		} else {
			$progress_text = $progress . '%';
		}

		$output .= sprintf(
			'<p class="text-muted">%s: %s/%s, %s %s</p>',
			'<span class="small">' . esc_html__( 'Storage quota', 'streamtube-core' ) . '</span>',
			'<span class="small storage-usage">' . size_format( $storage->usage ) . '</span>',
			'<strong class="text-success small storage-quota">' . size_format( $storage->quota ) . '</strong>',
			$storage->free ? '<strong>' . esc_html__( 'Free:', 'streamtube-core' ) . '</strong>' : '',
			$storage->free ? '<strong class="text-success small storage-free">' . size_format( $storage->free ) . '</strong>' : '',
		);

		$output .= sprintf(
			'<div class="%s" role="%s" aria-label="%s" aria-valuenow="%d" aria-valuemin="0" aria-valuemax="100" style="height:%s">',
			'progress bg-success',
			'progressbar',
			esc_attr__( 'Storage progress', 'streamtube-core' ),
			intval( $progress ),
			esc_attr( $progress_height )
		);

		$output .= sprintf(
			'<div class="progress-bar progress-bar-animated bg-%1$s text-white %2$s" style="width: %3$s;">%4$s</div>',
			esc_attr( $progress_class ),
			$progress < 100 ? 'progress-bar-striped' : '',
			$storage->quota === 0 ? '100%' : esc_attr( $progress . '%' ),
			$progress_text
		);

		$output .= '</div>';

		$output = sprintf(
			'<div id="storage-quota-progress-%s" class="storage-quota-progress">%s</div>',
			esc_attr( $user_id ),
			$output
		);

		if ( ! wp_validate_boolean( $wrap ) ) {
			return $output;
		}

		return sprintf(
			'<div class="p-4 border bg-white my-4">%s</div>',
			$output
		);
	}

	/**
	 *
	 * Add user capacity
	 * 
	 * @param integer $user_id
	 * @param string  $capacity
	 */
	public function add_user_quota_capacity( $args = array() ) {

		$args = wp_parse_args( $args, array(
			'user_id'     => get_current_user_id(),
			'size'        => 0,// mb
			'capacity'    => '',
			'description' => ''
		) );

		extract( $args );

		$size = $size ? $size : $this->is_valid_quota_capacity( $capacity );

		if ( ! $user_id || ! $size || ! is_numeric( $size ) || $size < 0 ) {
			return false;
		}

		StreamTube_Core_Storage_DB::add(
			array(
				'user_id'     => $user_id,
				'size'        => $size * 1024 * 1024,
				'object_id'   => $capacity,
				'object_type' => 'custom_capacity',
				'description' => $description
			)
		);

		/**
		 *
		 * Fires after adding capacity
		 *
		 * @param int $user_id
		 * @param int $size in byte
		 * @param string $capacity
		 * 
		 */
		do_action( 'streamtube/core/storage/added_user_quota_capacity', $user_id, $size, $capacity );

		return true;
	}

	public function remove_user_quota_capacity( $args = array() ) {
		$args = wp_parse_args( $args, array(
			'user_id'  => get_current_user_id(),
			'capacity' => ''
		) );

		extract( $args );

		if ( ! $user_id || ! $this->is_valid_quota_capacity( $capacity ) ) {
			return false;
		}

		StreamTube_Core_Storage_DB::remove( array(
			'user_id'   => $user_id,
			'object_id' => $capacity
		) );
	}

	public function add_user_manual_quota( $args = array() ) {

		$args = wp_parse_args( $args, array(
			'user_id'     => get_current_user_id(),
			'object_type' => 'manual',
			'size'        => 0,
			'description' => ''
		) );

		return StreamTube_Core_Storage_DB::add( $args );
	}

	public function delete_quota_capacities( $role, $capability ) {
		return StreamTube_Core_Storage_DB::delete( array(
			'object_id' => $capability
		) );
	}

	/**
	 *
	 * Check if capacity exists
	 * 
	 * @param  integer $user_id
	 * @param  string  $capacity
	 * 
	 */
	public function exists_capacity( $args = array() ) {

		$args = wp_parse_args( $args, array(
			'user_id'  => '',
			'capacity' => ''
		) );

		extract( $args );

		if ( ! $capacity ) {
			return false;
		}

		$args = array(
			'object_id' => $capacity
		);

		if ( $user_id ) {
			$args['user_id'] = $user_id;
		}

		return StreamTube_Core_Storage_DB::exists( $args );
	}
}