<?php
/**
 * The Storage Usage table list class
 * 
 * @link       https://1.envato.market/mgXE4y
 * @since      2.1
 *
 * @package    WordPress
 * @subpackage StreamTube
 * @author     phpface <nttoanbrvt@gmail.com>
 */

if ( ! defined( 'ABSPATH' ) ) {
    exit;
}

if ( ! class_exists( 'WP_List_Table' ) ) {
    require_once ABSPATH . 'wp-admin/includes/class-wp-list-table.php';
}

class StreamTube_Core_Storage_Usage_Table extends WP_List_Table {

    protected $user_id;

    protected $items_per_page    = 50;

    protected $screen_id;

    public function __construct( $args = array() ) {

        $args = wp_parse_args( $args, array(
            'screen'            => 'edit-user-storage',
            'user_id'           => 0,
            'items_per_page'    => 0,
            'singular'          => esc_html__( 'Usage', 'streamtube-core' ),
            'plural'            => esc_html__( 'Usages', 'streamtube-core' ),
            'ajax'              => true
        ) );

        $this->user_id          = $args['user_id'];

        $this->items_per_page   = $args['items_per_page'] ?? $this->get_items_per_page();

        $this->screen_id        = $args['screen'];

        parent::__construct( $args );
    } 

    public function get_columns() {
        $columns = array(
            'cb'            => '<input type="checkbox" />',
            'user_id'       => esc_html__( 'User', 'streamtube-core' ),
            'object_id'     => esc_html__( 'Content', 'streamtube-core' ),
            'object_type'   => esc_html__( 'Type', 'streamtube-core' ),
            'size'          => esc_html__( 'Size', 'streamtube-core' ),
            'description'   => esc_html__( 'Description', 'streamtube-core' ),
            'date'          => esc_html__( 'Date', 'streamtube-core' )
        );

        if( ! current_user_can( 'edit_users' ) ){
            unset( $columns['cb'] );
            unset( $columns['user_id'] );
        }

        return apply_filters( 'manage_storage_usage_columns', $columns );
    }    

    /**
     * Define what data to show on each column of the table
     *
     * @param  Array $item        Data
     * @param  String $column_name - Current column name
     *
     * @return Mixed
     */
    public function column_default( $item, $column_name ){

        global $streamtube;

        switch ( $column_name ) {
            case 'user_id':

                $row_actions = '';

                $maybe_user = get_userdata( $item[ $column_name ]  );

                $output = sprintf(
                    '<a class="text-%s" href="%s">[%s:%s] %s</a>',
                    $maybe_user ? 'body' : 'danger',
                    $maybe_user ? esc_url( get_author_posts_url( $maybe_user->ID )) : '#',
                    esc_html__( 'ID', 'streamtube-core' ),
                    $maybe_user ? $maybe_user->ID : 0,
                    $maybe_user ? $maybe_user->display_name : esc_html__( 'Unknown', 'streamtube-core' )
                );

                if( current_user_can( 'edit_user', $item[ $column_name ] ) ){
                    $row_actions = sprintf(
                        '<span class="trash"><a href="%s">%s</a></span>',
                        esc_url(
                            add_query_arg(
                                array(
                                    'doaction'  =>  'delete_usage',
                                    'id'        =>  $item['id']
                                )
                            ),
                            $_SERVER['REQUEST_URI']
                        ),
                        esc_html__( 'Delete', 'streamtube-core' )
                    );
                }

                if( $row_actions ){
                    $output .= sprintf( '<div class="row-actions">%s</div>', $row_actions );
                }


                return $output;
            break;

            case 'object_id':

                if( $item[ 'object_type' ] && post_type_exists( $item[ 'object_type' ] ) ){

                    if( $item[ 'object_type' ] === 'shop_order_placehold' ){
                        return sprintf(
                            esc_html__( 'Order ID: %s', 'streamtube-core' ),
                            $item[ $column_name ]
                        );
                    }

                    if( ! is_post_type_viewable( $item[ 'object_type' ] ) ){
                        return sprintf(
                            esc_html__( '%s ID: %s', 'streamtube-core' ),
                            get_post_type_object( $item[ 'object_type' ] )->labels->name,
                            $item[ $column_name ]
                        );
                    }

                    $post = get_post( (int)$item[ $column_name ] );

                    if( is_object( $post ) ){
                        return sprintf(
                            '<a target="_blank" href="%s">%s</a>',
                            $post->post_parent ? esc_url( get_permalink( $post->post_parent ) ) : '#',
                            $post->post_title
                        );
                    }
                }

                return $item[ $column_name ];

            break;

            case 'object_type':
                global $streamtube;

                $type   = $item[ $column_name ];
                $types  = $streamtube->get()->storage->get_custom_object_types();
                $badge  = 'secondary';

                if( array_key_exists( $type , $types ) ){
                    $type = $types[ $type ];
                }

                if( $item[ $column_name ] === 'initial' ){
                    $badge = 'primary';
                }

                if( $item[ $column_name ] === 'shop_order_placehold' ){
                    $badge = 'success';
                }

                if( $item[ $column_name ] === 'manual' ){
                    $badge = 'info';
                }

                return sprintf(
                    '<span class="badge text-white bg-%s">%s</span>',
                    esc_attr( $badge ),
                    $type
                );
                
            break;

            case 'size':

                $size = size_format( absint( $item[ $column_name ] ) );

                if( ! in_array( $item[ 'object_type' ], array_keys( $streamtube->get()->storage->get_custom_object_types() ) ) ){
                    return sprintf(
                        '<span class="text-danger">%s</span>',
                        $size
                    );
                }

                return sprintf(
                    '<span class="text-%s">%s%s</span>',
                    (int)$item[ $column_name ] > 0 ? 'success' : 'danger',
                    (int)$item[ $column_name ] > 0 ? '+' : '-',
                    $size
                );
            break;

            case 'date':
                return sprintf(
                    esc_html__( '%s ago', 'streamtube-core' ),
                    human_time_diff( strtotime( $item[ $column_name ] ), current_time( 'U' ) )
                );
            break;            
            
            default:
                return $item[ $column_name ];
            break;
        }
        
    }    

    /**
     * Define the sortable columns
     *
     * @return array
     */
    public function get_sortable_columns() {
        return array(
            'id'        => array( 'id', false ),
            'user_id'   => array( 'user_id', false ),
            'size'      => array( 'size', false )
        );
    }      

    /**
     * Render the bulk edit checkbox
     *
     * @param array $item
     *
     * @return string
     */
    function column_cb( $item ) {
        return sprintf(
            '<input type="checkbox" name="id[]" value="%s" />', $item['id']
        );
    }     

    /**
     * Define which columns are hidden
     *
     * @return array
     */
    public function get_hidden_columns() {
        $hidden_columns = (array)get_user_option( 'manage' . $this->screen->id . 'columnshidden' );

        return is_array( $hidden_columns ) ? $hidden_columns : array();
    }

    protected function get_items_per_page( $option = '', $default_value = 20 ) {
        $per_page = (int)get_user_option( 'edit_'. $this->screen->id .'_per_page' );

        return $per_page !== 0 ? absint( $per_page ) : $default_value;
    }

    protected function get_views() {

        global $streamtube;

        $url    = remove_query_arg( array(
            's', 'paged', 'object_type'
        ), $_SERVER['REQUEST_URI'] );

        $views = array();

        $views['all']  = array(
            'url'     => $url,
            'label'   => esc_html__( 'All', 'streamtube-core' ),
            'current' => ! isset( $_REQUEST['object_type'] ) && ! isset( $_REQUEST['object_type'] )
        );

        if( current_user_can( 'edit_others_posts' ) && ! $this->user_id ){
            $views[ 'mine' ]  = array(
                'url'     =>  esc_url( add_query_arg( array(
                    'object_type'    =>  'mine'
                ),   $url ) ),
                'label'   => esc_html__( 'Mine', 'streamtube-core' ),
                'current' => isset( $_REQUEST['object_type'] ) && 'mine' === $_REQUEST['object_type']
            );             
        }        

        foreach ($streamtube->get()->storage->get_custom_object_types() as $key => $value ) {
            $views[ $key ]  = array(
                'url'     =>  esc_url( add_query_arg( array(
                    'object_type'    =>  $key
                ),   $url ) ),
                'label'   => $value,
                'current' => isset( $_REQUEST['object_type'] ) && $key === $_REQUEST['object_type']
            );            
        }

        return $this->get_views_links( $views );
    }           

    public function get_bulk_actions(){

        $action = array();
        if( current_user_can( 'edit_users' ) ){
            $action['delete_usage'] = esc_html__( 'Delete', 'streamtube-core' );
        }
        
        return apply_filters( 'bulk_actions-' . $this->screen_id, $action ); 
    }      

    private function _get_query_args(){
        $q = array(
            'limit'         =>  $this->get_items_per_page(),
            'output'        =>  ARRAY_A,
            'paged'         =>  isset( $_REQUEST['paged'] )             ? absint( $_REQUEST['paged'] )     : 1,
            's'             =>  isset( $_REQUEST['s'] )                 ? wp_unslash( $_REQUEST['s'] )      : '',
            'object_type'   =>  isset( $_REQUEST['object_type'] )       ? wp_unslash( $_REQUEST['object_type'] )   : ''
        );

        if( ! current_user_can( 'edit_users' ) ){
            $q['user_id'] = get_current_user_id();
        }

        if( $this->user_id ){
            $q['user_id'] = $this->user_id;
        }

        if( $q['object_type'] === 'mine' ){
            $q['user_id'] = get_current_user_id();

            unset( $q['object_type'] );
        }

        return $q;  
    }

    private function _get_total(){
        return StreamTube_Core_Storage_DB::query(
            apply_filters( 'streamtube/core/storage/usage_table/query_args', array_merge( $this->_get_query_args(), array(
                'count' =>  true
            ) ) )
        );        
    }

    /**
     * Get the table data
     *
     * @return array
     */
    private function table_data() {
        return StreamTube_Core_Storage_DB::query(
            apply_filters( 'streamtube/core/storage/usage_table/query_args', $this->_get_query_args() )
        );
    }    

    /**
     * Sort data based on the selected column and order
     *
     * @param array $a
     * @param array $b
     * @return int
     */
    private function sort_data( $a, $b ) {
        $orderby = isset( $_GET['orderby'] )    && is_string( $_GET['orderby'] )    ? sanitize_text_field( $_GET['orderby'] )   : 'id'; 
        $order   = isset( $_GET['order'] )      && is_string( $_GET['order'] )      ? sanitize_text_field( $_GET['order'] )     : 'desc';

        if ( ! is_string( $a[$orderby] ) || ! is_string( $b[$orderby] ) || ! isset( $a[$orderby] ) || ! isset( $b[$orderby] ) ) {
            return 0; // Handle cases where the sort column doesn't exist in the data
        }

        $result = strcmp( $a[$orderby], $b[$orderby] ); 

        return ( $order === 'asc' ) ? $result : -$result;
    }    

    public function process_bulk_action() {

        $action     = $this->current_action();
        $ids        = isset( $_REQUEST['id'] ) ? wp_unslash( $_REQUEST['id'] ) : false;

        if( is_string( $ids ) ){
            $ids = array( $ids );
        }

        if( ! $ids ){
            return;
        }

        if( isset( $_REQUEST['doaction'] ) ){
            $action = wp_unslash( $_REQUEST['doaction'] );
        }

        if( ! current_user_can( 'edit_users' ) || ! $ids ){
            return;
        }

        if( in_array( $action , array( 'delete_usage', 'storage-quota' ) ) ){
            for ( $i =0;  $i < count( $ids );  $i++) {
                StreamTube_Core_Storage_DB::delete( array(
                    'id'    =>  $ids[$i]
                ) );
            }

            wp_redirect( remove_query_arg( array(
                'doaction', 'id'
            ), $_SERVER['REQUEST_URI'] ) );
        }

        do_action( 'handle_bulk_actions-' . $this->screen_id, $action, $ids );
    }

    public function prepare_items() {

        $this->process_bulk_action();

        $columns  = $this->get_columns();
        $hidden   = $this->get_hidden_columns();
        $sortable = $this->get_sortable_columns();

        $data = $this->table_data();

        usort( $data, array( $this, 'sort_data' ) );

        $perPage     = $this->get_items_per_page();
        $currentPage = $this->get_pagenum();
        $totalItems  = $this->_get_total();

        $this->set_pagination_args( array(
            'total_items' => $totalItems,
            'per_page'    => $perPage,
        ) );

        //$data = array_slice( $data, ( ( $currentPage - 1 ) * $perPage ), $perPage );

        $this->_column_headers = array( $columns, $hidden, $sortable );
        $this->items = $data;
    }
}