<?php
/**
 * Define the Storage functionality
 *
 *
 * @link       https://themeforest.net/user/phpface
 * @since      1.0.0
 *
 * @package    Streamtube_Core
 * @subpackage Streamtube_Core/includes
 */

/**
 *
 * @since      1.0.8
 * @package    Streamtube_Core
 * @subpackage Streamtube_Core/includes
 * @author     phpface <nttoanbrvt@gmail.com>
 */

if( ! defined('ABSPATH' ) ){
    exit;
}

require_once( untrailingslashit( dirname( __FILE__ ) ) . '/class-streamtube-core-storage-db.php' );
require_once( untrailingslashit( dirname( __FILE__ ) ) . '/class-streamtube-core-storage-table.php' );
require_once( untrailingslashit( dirname( __FILE__ ) ) . '/class-streamtube-core-storage.php' );
require_once( untrailingslashit( dirname( __FILE__ ) ) . '/functions.php' );

class StreamTube_Core_Storage_Loader extends StreamTube_Core_Storage{

    /**
     *
     * Hooks into `upload_size_limit`
     * Set max size for current user
     * 
     * @param  $size
     * 
     */
    public function filter_upload_size_limit( $size ){

        if( class_exists( 'BigFileUploads' ) ){
            return $size;
        }

        if( ! is_user_logged_in() ){
            return $size;
        }

        $_size = $this->get_user_max_upload_size( get_current_user_id() );

        if( $_size !== false ){
            $size = $_size;
        }

        return $size;
    }

    /**
     * Hooks into `wp_handle_upload_prefilter`.
     * Filters the uploaded file based on the user's role and storage limits.
     * 
     * @param  array $file The file array containing file details (e.g., name, size, etc.).
     * @return array Modified file array with an error if the upload is disallowed.
     */
    public function filter_wp_handle_upload_prefilter( $file = array() ){

        if( ! $file || $file['error'] ){
            return $file;
        }
        
        $user_max_upload_fize   = $this->get_user_max_upload_size();
        $storage                = $this->get_user_storage();

        if( $user_max_upload_fize === false || $storage->quota < 0 ){
            $file['error'] = esc_html__( 'Your maximum upload file size has not been set yet, or you are not allowed to upload files.', 'streamtube-core' );
        }

        if( $user_max_upload_fize > 0 && (int)$file['size'] > $user_max_upload_fize ){
            $file['error'] = sprintf(
                esc_html__( 'The file size (%s) exceeds the maximum allowed file size (%s).', 'streamtube-core' ),
                size_format( $file['size'] ),
                size_format( $user_max_upload_fize )
            );
        }

        if( $storage->quota > 0 && is_numeric( $storage->free ) && (int)$file['size'] > $storage->free ){
            $file['error'] = sprintf(
                esc_html__( 'The file size (%s) exceeds your available free storage quota (%s).', 'streamtube-core' ),
                size_format( $file['size'] ),
                size_format( $storage->free )
            );
        }

        return $file;
    }

    /**
     *
     * Filter our custom upload errors
     * 
     * @param  WP_Error $errors
     * @param  array  $file
     * 
     */
    public function filter_upload_video_error_handler( $errors, $file = array() ){

        $file = $this->filter_wp_handle_upload_prefilter( $file );

        if( isset( $file['error'] ) && ! empty( $file['error'] ) ){
            $errors->add( 'upload_error', $file['error'] );
        }

        return $errors;
    }

    public function add_attachment_usage( $post_id ){

        $post = get_post( $post_id );

        $size = 0;

        // Get the file path and calculate its size
        $file_path = get_attached_file( $post_id );
        if ( $file_path && file_exists( $file_path ) ) {
            $size = filesize( $file_path );
        }

        // Check the metadata for a filesize key and use it if available
        $metadata = get_post_meta( $post_id, '_wp_attachment_metadata', true );
        if ( is_array( $metadata ) && isset( $metadata['filesize'] ) ) {
            $size = (int) $metadata['filesize'];
        }

        // Update the user's usage quota, subtracting the file size
        $user_id = $post->post_author; // Get the author of the post

        if ( $user_id && $size > 0 ) {
            return StreamTube_Core_Storage_DB::add( array_merge( compact( 'user_id', 'size' ), array(
                'object_id'     =>  $post_id,
                'object_type'   =>  $post->post_type  
            ) ) );
        }
   
        return false;
    }    

    public function delete_attachment_usage( $post_id = 0, $post = null ){

        if( is_object( $post ) ){
            $user_id = $post->post_author;    
        }else{
            $user_id = get_post( $post_id )->post_author;
        }

        return StreamTube_Core_Storage_DB::remove( array_merge( compact( 'user_id' ), array(
            'object_id'     =>  $post_id,
            'object_type'   =>  $post->post_type  
        ) ) );        
    }

    /**
    * Adds temporary Bunny video usage record.
    *
    * This record is used during the import process and will be removed
    * after the video is imported into the WordPress Media Library.
    *
    * @param array $response The response data containing video information.
    * @return int|bool The ID of the added record on success, false on failure.
    * 
    */
    public function added_direct_upload_service( $args ){

        extract( $args );

        $user_id = get_current_user_id();

        return StreamTube_Core_Storage_DB::add( array_merge( compact( 'user_id', 'size' ), array(
            'object_id'     =>  $uid,
            'object_type'   =>  isset( $args['service'] ) ? $args['service'] : ''
        ) ) );
    }

    /**
     *
     * Remove bunny token usage after importing
     * 
     */
    public function remove_bunny_resumable_token( $post_id, $attachment_id, $video_data ){

        $user_id = get_post( $post_id )->post_author;

        return StreamTube_Core_Storage_DB::remove( array_merge( compact( 'user_id' ), array(
            'object_id'     =>  $video_data['guid']
        ) ) );
    }

    /**
     *
     * Remove cloudflare token usage after importing
     * 
     */
    public function remove_cloudflare_resumable_token( $post_id, $attachment_id, $stream = array() ){
        $user_id = get_post( $post_id )->post_author;

        return StreamTube_Core_Storage_DB::remove( array_merge( compact( 'user_id' ), array(
            'object_id'     =>  $stream['uid']
        ) ) );          
    }

    /**
     *
     * AJAX Update max upload size for all requested roles
     * 
     */    
    public function ajax_update_roles_sizes(){

        check_ajax_referer( 'max_file_size' );

        if( ! current_user_can( 'manage_options' ) ){
            wp_send_json_error(
                new WP_Error(
                    'no_permission',
                    esc_html__( 'You do not have permission to do this action.', 'streamtube-core' )
                )
            );
        }        

        $this->update_roles_upload_sizes();

        wp_send_json_success( array(
            'message'   =>  esc_html__( 'Save Changes', 'streamtube-core' )
        ) );
    }

    public function ajax_update_roles_quotes(){
        check_ajax_referer( 'storage_quota' );

        if( ! current_user_can( 'manage_options' ) ){
            wp_send_json_error(
                new WP_Error(
                    'no_permission',
                    esc_html__( 'You do not have permission to do this action.', 'streamtube-core' )
                )
            );
        }        

        $this->update_roles_quota();

        wp_send_json_success( array(
            'message'   =>  esc_html__( 'Save Changes', 'streamtube-core' )
        ) );        
    }

    public function ajax_update_user_storage_quota(){

        $http_data = wp_parse_args( $_REQUEST, array(
            '_wpnonce'  =>  '',
            'user_id'   =>  0,
            'size'      =>  0
        ) );

        extract( $http_data );

        check_ajax_referer( 'update_user_storage_quota_' . $user_id );

        if( ! current_user_can( 'edit_user', $user_id ) ){
            wp_send_json_error(
                new WP_Error(
                    'no_permission',
                    esc_html__( 'You do not have permission to update user storage quota.', 'streamtube-core' )
                )
            );
        }

        $bytes = (int)$size != $size ? wp_convert_hr_to_bytes($size) : (int)$size *1024*1024;        

        $response = $this->set_user_storage_quota( $user_id, $bytes );

        if( is_wp_error( $response ) ){
            wp_send_json_error( $response );
        }

        wp_send_json_success( array_merge( $response, array(
            'message'   =>  esc_html__( 'Saved Changes.', 'streamtube-core' )
        )) );        
    }

    public function ajax_reset_users_storage_quota(){
        $http_data = wp_parse_args( $_REQUEST, array(
            'confirm'   =>  '',
            'users'     =>  '',
            '_wpnonce'  =>  ''
        ) );

        extract( $http_data );

        $users = explode(',', $users );

        if( ! $users ){
            wp_send_json_error(
                new WP_Error(
                    'empty_users',
                    esc_html__( 'Users were not found.', 'streamtube-core' )
                ),
                403
            );
        }

        check_ajax_referer( 'reset_users_storage_quota_' . join(',', $users ) );  

        if( ! $confirm || $confirm !== 'RESET' ){
            wp_send_json_error(
                new WP_Error(
                    'invalid_confirm',
                    esc_html__( 'Please enter RESET to confirm action.', 'streamtube-core' )
                ),
                403
            );            
        }      

        for ( $i=0; $i < count( $users ); $i++) { 
            if( current_user_can( 'edit_user', $users[$i] ) ){
                $this->reset_user_storage_usage( $users[$i] );
            }
        }

        wp_send_json_success( array(
            'message'   =>  esc_html__( 'Saved Changes.', 'streamtube-core' )
        ) );
    }

    public function ajax_add_user_manual_quota(){
        $http_data = wp_parse_args( $_REQUEST, array(
            'user_id'       =>  0,
            'size'          =>  0,
            'description'   =>  '',
            '_wpnonce'      =>  ''
        ) );

        extract( $http_data );

        check_ajax_referer( 'add_user_manual_' . $user_id );

        $size = (int)$size;

        if( $size === 0 ){
            wp_send_json_error(
                new WP_Error(
                    'invalid_size',
                    esc_html__( 'Invalid size.', 'streamtube-core' )
                ),
                403
            );
        }

        if( current_user_can( 'edit_user', $user_id ) ){
            $response = $this->add_user_manual_quota( array_merge( compact( 'user_id', 'description' ), array(
                'size'  =>  $size *1024*1024
            ) ) );

            if( ! is_wp_error( $response ) || $response === true ){
                wp_send_json_success( array_merge( $this->get_user_storage_details( $user_id ), array(
                    'message'   =>  esc_html__( 'Quota added.', 'streamtube-core' )
                ) ) );
            }
        }

        wp_send_json_error(
            new WP_Error(
                'no_permission',
                esc_html__( 'You do not have permission to add manual quota for this user.', 'streamtube-core' ),
            ),
            403
        );
    }

    /**
     *
     * Auto add user capacity after assigning capability
     * Hooks into `streamtube/core/woocommerce/xcapabilities/update_capabilities` action.
     *
     * @param WP_User $user
     * @param string $capability assigned capability
     * @param array $old_capabilities
     * 
     */
    public function wc_purchase_capacity_quota( $customer_id, $capabilities, $order_line_item, $order ){

        if( ! $capabilities ){
            return;
        }

        $status = $order->get_status();

        for ( $i=0;  $i < count( $capabilities );  $i++) { 

            $maybe_capacity = $this->is_valid_quota_capacity( $capabilities[$i] );

            if( $maybe_capacity ){

                $size = $maybe_capacity*1024*1024 * $order_line_item->get_quantity();

                $args = array(
                    'user_id'       =>  $customer_id,
                    'object_id'     =>  $order->get_id(),
                    'object_type'   =>  get_post_type( $order->get_id() ),
                    'size'          =>  $size
                );

                if( $status === 'completed' ){
                    StreamTube_Core_Storage_DB::add( array_merge( $args, array(
                        'description'   =>  sprintf(
                            esc_html__( 'Paid for %s, order ID: %s', 'streamtube-core' ),
                            size_format( $size ),
                            $order->get_id()
                        )
                    ) ) );
                }else{
                    StreamTube_Core_Storage_DB::remove( $args);                    
                }
            }
        }
    }   

    /**
     *
     * AJAX add custom capacity to administrator role
     * 
     */
    public function ajax_add_quota_capacity(){
        check_ajax_referer( 'add_quota_capability' );

        $http_data = wp_parse_args( $_REQUEST, array(
            'capacity'  =>  0
        ) );

        $response = $this->add_quota_capacity( $http_data['capacity'] );

        if( is_wp_error( $response ) ){
            wp_send_json_error( $response, 403 );
        }

        wp_send_json_success( array(
            'capacity'  =>  $response,
            'message'   =>  esc_html__( 'Capacity added.', 'streamtube-core' ),
            'output'    =>  streamtube_core_get_storage_capacity_list()
        ) );
    }

    /**
     * Run after deleting user
     */
    public function deleted_user( $id, $reassign, $user ){

        if( is_null( $reassign ) ){
            return StreamTube_Core_Storage_DB::delete( array(
                'user_id'   =>  (int)$id
            ) );
        }
        else{
            return StreamTube_Core_Storage_DB::assign_user( $reassign, $id );
        }
    }

    /**
     * Hook into `after_delete_post` to delete any quotas associated with the given post ID and post type.
     *
     * @param int    $post_id The ID of the post being deleted.
     * @param object $post    The post object associated with the post ID.
     * @return void
     */
    public function deleted_post( $post_id, $post ) {
        return StreamTube_Core_Storage_DB::delete( array(
            'object_id'     =>  $post_id,
            'object_type'   =>  $post->post_type
        ) );
    }

    public function display_current_user_storage_progress(){
        echo $this->get_user_storage_progres( array(
            'hide_if_unlimited_or_not_allowed' =>  false,
            'wrap'                             =>  true
        ) );
    }

    /**
     *
     * [user_storage_table] shortcode
     * 
     * @param  array  $args
     * 
     */
    public function shortcode_user_storage_table( $args = array() ){
        ob_start();

        load_template( trailingslashit( plugin_dir_path( __FILE__ ) ) . 'public/usage-table.php', true, $args );

        return ob_get_clean();
    }

    /**
     *
     * [user_storage_progress] shortcode
     * 
     * @return array
     */
    public function register_shortcode_user_storage_progress(){
        add_shortcode( 'user_storage_progress', array( $this , 'get_user_storage_progres' ) );
    }

    /**
     *
     * [user_storage_table] shortcode
     * 
     * @return array
     */
    public function register_shortcode_user_storage_table(){
        add_shortcode( 'user_storage_table', array( $this , 'shortcode_user_storage_table' ) );
    }    

    /**
     * Add custom columns to the Users List Table.
     *
     * @param array $columns The existing columns in the Users List Table.
     * @return array Modified columns with the 'Storage' column added.
     */
    public function users_list_table_columns( $columns ) {
        $columns['storage'] = esc_html__( 'Storage', 'streamtube-core' );
        return $columns;
    }

    /**
     * Render content for custom columns in the Users List Table.
     *
     * @param string $output      The column content (default empty).
     * @param string $column_name The name of the column being rendered.
     * @param int    $user_id     The ID of the user for this row.
     * @return string Modified column content for the 'Storage' column.
     */
    public function users_list_table_columns_content( $output, $column_name, $user_id ) {
        if ( 'storage' === $column_name ) {
            $output = $this->get_user_storage_progres( 
                array_merge( compact( 'user_id' ) , array(
                     'hide_if_unlimited_or_not_allowed' =>  false
                ) )
            );
        }
        return $output;
    }

    public function users_list_table_actions( $actions ){
        return array_merge( $actions, array(
            'reset_storage_quota'   =>  esc_html__( 'Reset Storage Quota', 'streamtube-core' )
        ) );
    }

    public function get_setting_tabs(){
        $tabs = array(
            'quota' =>  array(
                'title'     =>  esc_html__( 'Quota', 'streamtube-core' ),
                'callback'  =>  function(){
                    streamtube_core_load_template( 'site-settings/storage/quota.php' );
                }
            ),
            'capacity'    =>  array(
                'title'     =>  esc_html__( 'Capacity', 'streamtube-core' ),
                'callback'  =>  function(){
                    streamtube_core_load_template( 'site-settings/storage/capacity.php' );
                }
            )
        );
        return apply_filters( 'streamtube/core/storage/setting_tabs', $tabs );
    }
}