<?php
/**
 * Define the Storage DB functionality
 *
 *
 * @link       https://themeforest.net/user/phpface
 * @since      1.0.0
 *
 * @package    Streamtube_Core
 * @subpackage Streamtube_Core/includes
 */

/**
 *
 * @since      1.0.8
 * @package    Streamtube_Core
 * @subpackage Streamtube_Core/includes
 * @author     phpface <nttoanbrvt@gmail.com>
 */

if( ! defined('ABSPATH' ) ){
    exit;
}

class StreamTube_Core_Storage_DB{

    const STORAGE_USAGE_TABLE = 'storage_usage';

    public static function get_table(){
        global $wpdb;

        return $wpdb->prefix . self::STORAGE_USAGE_TABLE;
    }

    /**
     *
     * create a new `wp_storage_usage` table 
     * 
     */
    public static function install_db(){

        if( ! function_exists( 'maybe_create_table' ) ){
            require_once( ABSPATH . 'wp-admin/includes/upgrade.php' );
        }

        global $wpdb;

        $table_name = self::get_table();

        $charset_collate = $wpdb->get_charset_collate();

        $sql = "CREATE TABLE $table_name (
            id BIGINT(20) UNSIGNED NOT NULL AUTO_INCREMENT,
            user_id BIGINT(20) UNSIGNED NOT NULL,
            object_id varchar(255) DEFAULT NULL,
            object_type varchar(255) NOT NULL,
            size BIGINT(20) SIGNED NOT NULL,
            description TEXT DEFAULT NULL,
            date DATETIME NOT NULL,
            PRIMARY KEY  (id)
        ) $charset_collate;";

        return maybe_create_table( $table_name, $sql );
    }    

    private static function _default_args(){
        return array(
            'user_id'       =>  '',
            'object_id'     =>  '',
            'object_type'   =>  '',
            'size'          =>  '',
            'description'   =>  '',
            'date'          =>  ''
        );
    }

    /**
     *
     * Parse default $args
     * 
     * @param  array  $args
     * @return array $args
     * 
     */
    private static function parse_args( $args = array() ){
        return wp_parse_args( $args, self::_default_args() );        
    }

    /**
     *
     * Add entry
     * 
     * @param array $args
     *
     * @return $entry_id
     */
    public static function add( $args = array() ){
        global $wpdb;

        $args = self::parse_args( array_merge( $args, array(
            'date'  =>  current_time( 'Y-m-d H:i:s' )
        ) ) );

        extract( $args );

        $wpdb->insert(
            self::get_table(),
            compact( 'user_id', 'object_id', 'object_type', 'size', 'description', 'date' ),
            array( '%d', '%s', '%s', '%d', '%s', '%s' )
        );  

        return $wpdb->insert_id; 
    }

    public static function remove( $args = array() ){

        global $wpdb;

        $args = self::parse_args( $args );

        extract( $args );

        $where = array();
        $where_format = array();

        foreach ( $args as $field => $value ) {
            if( ! empty( $value ) ){
                $where[ $field ] = $value;
                $where_format[] = is_int( $value ) ? '%d' : '%s';
            }
        }

        if( $where ){
            return $wpdb->delete(
                self::get_table(),
                $where,
                $where_format
            );
        }
        return false;
    }

    public static function delete( $args = array() ){
        return self::remove( $args );
    }

    public static function update( $args = array() ){

        global $wpdb;

        $args = self::parse_args( array_merge( $args, array(
            'date'  =>  current_time( 'Y-m-d H:i:s' )
        ) ) );

        extract( $args );        

        return $wpdb->update(
            self::get_table(),
            compact( 'object_id', 'object_type', 'size', 'description', 'date' ),
            compact( 'user_id', 'object_id', 'object_type' ),
            array( '%s', '%s', '%d', '%s', '%s' ),
            array( '%d', '%s', '%s' )
        );
    }

    public static function assign_user( $to = 0, $from = 0 ){
        if( ! $to || ! $from ){
            return false;
        }

        global $wpdb;

        return $wpdb->update(
            self::get_table(),
            array(
                'user_id'   =>  $to
            ),
            array(
                'user_id'   =>  $from
            ),
            array( '%d' ),
            array( '%d' )
        );
    }

    public static function get_total_sizes( $args = array() ){

        global $wpdb;

        $args = wp_parse_args( self::parse_args( $args ), array(
            'object_type_not_in'    =>  ''
        ) );

        extract( $args );

        $table = self::get_table();

        $where = array();

        if( $user_id ){
            $where[] = $wpdb->prepare( 'user_id=%d', $user_id );
        }

        if( $object_id ){
            $where[] = $wpdb->prepare( 'object_id=%s', $object_id );
        }

        if ( $object_type ) {
            if ( is_string( $object_type ) ) {
                $where[] = $wpdb->prepare( 'object_type = %s', $object_type );
            } elseif ( is_array( $object_type ) ) {
                $placeholders = implode( ', ', array_fill( 0, count( $object_type ), '%s' ) );
                $where[] = $wpdb->prepare( 'object_type IN (' . $placeholders . ')', ...$object_type );
            }
        }

        if ( $object_type_not_in ) {
            if ( is_string( $object_type_not_in ) ) {
                $where[] = $wpdb->prepare( 'object_type != %s', $object_type_not_in );
            } elseif ( is_array( $object_type_not_in ) ) {
                $placeholders = implode( ', ', array_fill( 0, count( $object_type_not_in ), '%s' ) );
                $where[] = $wpdb->prepare( 'object_type NOT IN (' . $placeholders . ')', ...$object_type_not_in );
            }
        }
                    
        $where = $where ? implode(' AND ', $where ) : 1;

        $query = "
            SELECT SUM(size) FROM $table WHERE $where
        ";

        $total_size = $wpdb->get_var( $query );

        if( is_null( $total_size ) ){
            return $total_size;
        }

        $total_size = (int)$total_size;

        return min( $total_size, PHP_INT_MAX );
    }

    public static function exists( $args = array() ){
        return self::query( $args );
    }

    public static function query( $args = array() ){

        global $wpdb;

        $table = self::get_table();

        $args = wp_parse_args( self::parse_args( $args ), array(
            'limit'     =>  50,
            'paged'     =>  1,
            'orderby'   =>  array( 'id' ) ,
            'order'     =>  'desc',
            's'         =>  '',
            'count'     =>  false,
            'output'    =>  OBJECT
        ) );

        $where = array();

        extract( $args );

        if( $user_id ){
            $where[] = $wpdb->prepare( 'user_id=%d', $user_id );
        }

        if( $object_id ){
            $where[] = $wpdb->prepare( 'object_id=%s', $object_id );
        }

        if( $object_type ){
            $where[] = $wpdb->prepare( 'object_type=%s', $object_type );
        }

        if( $size ){
            $where[] = $wpdb->prepare( 'size=%d', $size );
        } 

        if( $description ){
            $where[] = $wpdb->prepare( 'description like %s', '%'.$description.'%' );
        }

        if( $date ){
            $where[] = $wpdb->prepare( 'date = %s', $date );
        }

        if( is_string( $s ) && ! empty( $s ) ){
            $where[] = $wpdb->prepare(
                '(user_id LIKE %s OR object_id LIKE %s OR object_type LIKE %s OR description LIKE %s)',
                '%' . $wpdb->esc_like( $s ) . '%',
                '%' . $wpdb->esc_like( $s ) . '%',
                '%' . $wpdb->esc_like( $s ) . '%',
                '%' . $wpdb->esc_like( $s ) . '%'
            );
        }

        if( $where ){
            $where = implode(' AND ', $where );
        }else{
            $where = '1';
        }

        if ( (int)$args['limit'] > 0 ) {
            $offset = ( max( 1, $args['paged'] ) - 1 ) * $args['limit'];
            $limit = $wpdb->prepare( 'LIMIT %d OFFSET %d', $args['limit'], $offset );
        }

        if( $orderby ){
            if( is_string( $orderby ) ){
                $orderby = array( $orderby );
            }

            $orderby = 'ORDER BY ' . implode(',', $orderby) . ' ' . $order;
        }

        if( wp_validate_boolean( $args['count'] ) ){
            $query = "
                SELECT COUNT(id) AS total FROM {$table}
                WHERE {$where}
            ";

            return $wpdb->get_var( $query );
        }else{
            $query = "
                SELECT * FROM {$table}
                WHERE {$where} {$orderby} {$limit}
            ";   
            return $wpdb->get_results( $query, $output );
        }

    }
}