<?php
/**
 * Define the Paid Member Subscriptions functionality
 * 
 * @link       https://wordpress.org/plugins/paid-member-subscriptions/
 *
 *
 * @link       https://themeforest.net/user/phpface
 * @since      4.50
 *
 * @package    Streamtube_Core
 * @subpackage Streamtube_Core/includes
 */

/**
 *
 * @since      4.50
 * @package    Streamtube_Core
 * @subpackage Streamtube_Core/includes
 * @author     phpface <nttoanbrvt@gmail.com>
 */

if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

class StreamTube_Core_PMSubscriptions {
	public const PAGE_SLUG = 'pmsubscriptions';

	/**
	 * Removes the 'pms_filter_attachment_url' filter from 'wp_get_attachment_url' hook on single video pages.
	 *
	 * This function is used to disable the Paid Member Subscriptions (PMS) filter that restricts attachment URLs
	 * on single video pages, likely to prevent unintended access restrictions on video attachments.
	 *
	 * @return void
	 * */
	public function remove_attachment_url_filter() {
		if ( is_singular( 'video' ) && function_exists( 'pms_filter_attachment_url' ) ) {
			remove_filter( 'wp_get_attachment_url', 'pms_filter_attachment_url', 10, 2 );
		}
	}

	/**
	 * Filters the player output based on Paid Memberships Pro (PMS) restrictions.
	 *
	 * This function checks if the current user has the required subscription to access the content.
	 * If not, it wraps the restricted content with a message indicating that a paid membership is required.
	 *
	 * @param string $output The original player output.
	 * @param array  $setup  An array of setup parameters for the player, including 'mediaid' and 'poster' URLs.
	 *
	 * @return string The filtered player output, either the original output if the user has access,
	 *                or a wrapped output with a restriction message if the user does not have access.
	 */
	public function filter_player_output( $output, $setup ) {

		if (
			! function_exists( 'pms_filter_content' ) ||
			! function_exists( 'pms_is_post_restricted' ) ||
			! pms_is_post_restricted( $setup['mediaid'] ) ||
			current_user_can( 'edit_others_videos' ) ) {
			return $output;
		}

		if ( $setup['mediaid'] ) {
			$_post = get_post( $setup['mediaid'] );
			if ( $_post && (int) $_post->post_author == get_current_user_id() && is_user_logged_in() ) {
				return $output;
			}
		}

		$restricted_output = pms_filter_content( $output );

		$plans   = get_post_meta( $setup['mediaid'], 'pms-content-restrict-subscription-plan' );
		$wrapper = '';

		if (
			is_main_query() &&
			apply_filters( 'streamtube/core/pmsubscriptions_unfilter_content', true ) ) {
			remove_filter( 'the_content', 'pms_filter_content', 11 );
		}

		$wrapper = '<div class="require-membership require-pms-levels">';
		$wrapper .= '<div class="top-50 start-50 translate-middle position-absolute">';
		$wrapper .= sprintf(
			'<div class="pmpro_card pmpro_content_message"><h2 class="pmpro_card_title">%s</h2></div>',
			esc_html__( 'Paid Membership Required', 'streamtube-core' )
		);
		$wrapper .= balanceTags( $restricted_output, true );

		if ( $plans && is_user_logged_in() ) {
			$plan_name = array();
			foreach ( $plans as $plan ) {
				$_plan = get_post( $plan );
				if ( $_plan ) {
					$plan_name[] = $_plan->post_title;
				}
			}

			$plan_names = count( $plan_name ) > 1 ? join( '<span class="text-white">' . esc_html__( ' or ', 'streamtube-core' ) . '</span>', $plan_name ) : $plan_name[0];

			$wrapper .= '<div class="pms-message">' . sprintf(
				esc_html__( 'You must have an active %s subscription to access this content.', 'streamtube-core' ),
				'<strong class="text-success">' . $plan_names . '</strong>'
			) . '</div>';
		}

		$wrapper .= '</div>';
		$wrapper .= '</div>';

		$wrapper .= sprintf(
			'<div class="player-poster bg-cover" style="background-image:url(%s)"></div>',
			$setup['poster2'] ? $setup['poster2'] : $setup['poster']
		);

		return balanceTags( $wrapper, true );
	}

	/**
	 * Filters the player embed output.
	 *
	 * @param string $embed The original embed output.
	 * @param array  $setup The setup parameters.
	 *
	 * @return string The filtered embed output.
	 */
	public function filter_player_embed_output( $embed, $setup ) {
		return $this->filter_player_output( $embed, $setup );
	}

	/**
	 * Filters the member account tabs displayed in the Profile Builder member account page.
	 *
	 * This function removes the 'profile' tab and disables the logout tab if the 'dashboard' query variable is set.
	 *
	 * @param array $tabs An array of tabs to be displayed in the member account page.
	 *                    The keys are the tab slugs and the values are the tab labels.
	 * @param array $args An array of arguments passed to the member account page. Default is an empty array.
	 *
	 * @return array The filtered array of tabs.
	 */
	public function filter_member_account_tabs( $tabs, $args = array() ) {
		global $wp_query;
		if ( isset( $wp_query->query_vars['dashboard'] ) ) {
			unset( $tabs['profile'] );
			add_filter( 'pms_member_account_logout_tab', '__return_false' );
		}
		return $tabs;
	}

	/**
	 * Wraps the given output with a div having the class "pmsubscriptions".
	 *
	 * This function is used to add a wrapper around the output of a shortcode,
	 * specifically for Paid Member Subscriptions (PMS). The wrapper provides a
	 * container with a specific class that can be used for styling or other
	 * purposes.
	 *
	 * @param string $output The output to be wrapped.
	 *
	 * @return string The wrapped output, enclosed in a div with the class "pmsubscriptions".
	 */
	public function add_shortcode_wrapper( $output ) {
		return sprintf(
			'<div class="pmsubscriptions">%s</div>',
			$output
		);
	}

	/**
	 * Display a membership badge if Paid Member Subscriptions plugin is active.
	 *
	 * @param string $badge   The existing badge HTML (if any).
	 * @param int    $post_id The ID of the post.
	 *
	 * @return string The membership badge HTML.
	 */
	public function display_membership_badge( $badge = '', $post_id = 0 ) {
		if ( function_exists( 'pms_is_post_restricted' ) && pms_is_post_restricted( $post_id ) ) {
			ob_start();
			streamtube_core_load_template( 'misc/premium-badge.php', false, array(
				'service' => 'pms'
			) );
			$badge = ob_get_clean();
		}
		return $badge;
	}

	/**
	 * Add "membership" label to status links
	 */
	public function filter_posts_list_table_status_links( $status_links, $post_type, $url ) {
		if ( post_type_supports( $post_type, 'paid_membership_subscriptions' ) ) {
			$status_links['membership'] = array(
				'url'     => add_query_arg( array( 'post_status' => 'membership' ), $url ),
				'label'   => esc_html__( 'Membership', 'streamtube-core' ),
				'current' => isset( $_REQUEST['post_status'] ) && $_REQUEST['post_status'] == 'membership' ? true : false
			);
		}

		return $status_links;
	}


	/**
	 * Filters the query arguments for the posts list table to include only posts restricted to a membership plan.
	 *
	 * This function modifies the query arguments when the 'post_status' request parameter is set to 'membership'.
	 * It adds a meta query to filter posts that have the 'pms-content-restrict-subscription-plan' meta key.
	 *
	 * @param array  $q         An array of query variables.
	 * @param string $post_type The post type being queried.
	 * @param object $wp_query  The WP_Query object.
	 *
	 * @return array The modified array of query variables.
	 */
	public function filter_posts_list_table_query_args( $q ) {
		if ( isset( $_REQUEST['post_status'] ) && wp_unslash( $_REQUEST['post_status'] ) === 'membership' ) {

			$meta_query = isset( $q['meta_query'] ) && is_array( $q['meta_query'] ) ? $q['meta_query'] : array();

			$meta_query[] = array(
				'key'     => 'pms-content-restrict-subscription-plan',
				'compare' => 'EXISTS'
			);

			$meta_query[] = array(
				'key'     => 'pms-content-restrict-user-status',
				'compare' => 'EXISTS'
			);

			$meta_query['relation'] = 'OR';

			$q['meta_query'] = $meta_query;
		}

		return $q;
	}

	/**
	 *
	 * Add `paid_membership_pro` support for post types set by PMP CPT addon
	 * 
	 */
	public function add_post_type_support_membership() {
		$post_types = apply_filters( 'pmpro_restrictable_post_types', array( 'page', 'post', 'video' ) );

		for ( $i = 0; $i < count( $post_types ); $i++ ) {
			add_post_type_support( $post_types[ $i ], 'paid_membership_subscriptions' );
		}
	}

	/**
	 *
	 * Add dashboard menu
	 *
	 * @since 2.2
	 * ]
	 */
	public function add_dashboard_menu( $items ) {

		$items[ self::PAGE_SLUG ] = array(
			'title'    => esc_html__( 'Subscriptions', 'streamtube-core' ),
			'desc'     => esc_html__( 'Your subscriptions', 'streamtube-core' ),
			'icon'     => 'icon-diamond',
			'callback' => function () {
				load_template( plugin_dir_path( __FILE__ ) . 'dashboard/subscriptions.php', true );
			},
			'parent'   => 'dashboard',
			'cap'      => 'read',
			'priority' => 5
		);

		return $items;
	}

	/**
	 * Adds meta boxes to the 'dashboard_video' post type for display options.
	 *
	 * This method checks if the Paid Member Subscriptions plugin is active and its
	 * `PMS_Meta_Box_Content_Restriction` class exists. If so, it adds a meta box
	 * to the 'dashboard_video' post type, allowing users to configure display options
	 * for videos.
	 *
	 * @return void
	 */
	public function add_meta_boxes() {
		if ( class_exists( 'PMS_Meta_Box_Content_Restriction' ) ) {
			global $post_type_screen;

			if ( post_type_supports( $post_type_screen, 'paid_membership_subscriptions' ) ) {
				streamtube_add_meta_box(
					self::PAGE_SLUG . '-box',
					esc_html__( 'Display Options', 'streamtube-core' ),
					array( $this, 'metabox_callback' ),
					streamtube_get_current_screen(),
					'advanced',
					'high'
				);
			}
		}
	}

	/**
	 * Callback function to display the metabox for content restriction.
	 *
	 * This function checks if the 'PMS_Meta_Box_Content_Restriction' class exists
	 * and if so, instantiates it and calls its 'output' method to render the metabox.
	 *
	 * @param WP_Post $post The current post object.
	 *
	 * @return string|void The output of the metabox, or void if the class doesn't exist.
	 */
	public function metabox_callback( $post ) {

		if ( class_exists( 'PMS_Meta_Box_Content_Restriction' ) ) {
			$metabox = new PMS_Meta_Box_Content_Restriction();
			ob_start();
			$metabox->output( $post );

			echo balanceTags( ob_get_clean(), true );
		}
	}

	/**
	 * Enqueues the necessary scripts and styles for the plugin.
	 *
	 * This method registers and enqueues the stylesheet for the plugin,
	 * ensuring it is properly loaded with dependencies and versioning.
	 *
	 * @return void
	 */
	public function enqueue_scripts() {

		global $wp_query;

		if ( defined( 'PMS_PLUGIN_DIR_URL' ) && isset( $wp_query->query_vars['dashboard'] ) ) {
			wp_enqueue_style(
				'pms-style-back-end',
				PMS_PLUGIN_DIR_URL . 'assets/css/style-back-end.css',
				array(),
				PMS_VERSION
			);

			wp_enqueue_script(
				'pms_post_content_restriction-js',
				PMS_PLUGIN_DIR_URL . 'assets//js/admin/meta-box-post-content-restriction.js',
				array( 'jquery' ),
				PMS_VERSION,
				true
			);
		}

		wp_enqueue_style(
			self::PAGE_SLUG,
			untrailingslashit( plugin_dir_url( __FILE__ ) ) . '/assets/style.css',
			array( 'bootstrap' ),
			filemtime( untrailingslashit( plugin_dir_path( __FILE__ ) ) . '/assets/style.css' )
		);
	}
}