<?php
/**
 * Define the BunnyCDN functionality
 *
 *
 * @link       https://themeforest.net/user/phpface
 * @since      2.1
 *
 * @package    Streamtube_Core
 * @subpackage Streamtube_Core/includes
 */

/**
 *
 * @since      2.1
 * @package    Streamtube_Core
 * @subpackage Streamtube_Core/includes
 * @author     phpface <nttoanbrvt@gmail.com>
 */

if ( ! defined( 'ABSPATH' ) ) {
	exit;
}
class Streamtube_Core_BunnyCDN {

	/**
	 *
	 * Holds the settings
	 * 
	 * @var array
	 *
	 * @since 2.1
	 * 
	 */
	public $settings = array();

	/**
	 *
	 * Holds the Bunny API object
	 * 
	 * @var object
	 *
	 * @since 2.1
	 * 
	 */
	public $bunnyAPI;

	/**
	 *
	 * Holds the admin
	 * 
	 * @var object
	 *
	 * @since 2.1
	 * 
	 */
	public $admin;

	private $Post;

	public $restApi;

	public function __construct() {

		$this->load_dependencies();

		$this->settings = Streamtube_Core_BunnyCDN_Settings::get_settings();

		$this->admin = new Streamtube_Core_BunnyCDN_Admin();

		$this->bunnyAPI = new Streamtube_Core_BunnyCDN_API( array(
			'AccessKey'      => $this->settings['AccessKey'],
			'libraryId'      => $this->settings['libraryId'],
			'cdn_hostname'   => $this->settings['cdn_hostname'],
			'token_auth_key' => $this->settings['token_auth_key'],
			'token_expires'  => $this->settings['token_expires']
		) );

		$this->Post = new Streamtube_Core_Post();

		$this->restApi = new StreamTube_Core_Bunny_Rest_Controller();
	}

	/**
	 *
	 * Include file
	 * 
	 * @param  string $file
	 *
	 * @since 2.1
	 * 
	 */
	private function include_file( $file ) {
		require_once plugin_dir_path( __FILE__ ) . $file;
	}

	/**
	 *
	 * Load dependencies
	 *
	 * @since 2.1
	 * 
	 */
	private function load_dependencies() {
		if ( ! function_exists( 'media_sideload_image' ) ) {
			require_once( ABSPATH . 'wp-admin/includes/media.php' );
			require_once( ABSPATH . 'wp-admin/includes/file.php' );
			require_once( ABSPATH . 'wp-admin/includes/image.php' );
		}
		$this->include_file( 'class-streamtube-core-bunnycdn-perm.php' );
		$this->include_file( 'class-streamtube-core-bunnycdn-settings.php' );
		$this->include_file( 'class-streamtube-core-bunnycdn-admin.php' );
		$this->include_file( 'class-streamtube-core-bunnycdn-api.php' );
		$this->include_file( 'class-streamtube-core-bunnycdn-rest.php' );
	}

	/**
	 *
	 * Settings Tabs
	 * 
	 * @return array
	 *
	 * @since 2.1
	 * 
	 */
	public function get_setting_tabs() {
		$tabs = array(
			'general'       => array(
				'heading' => esc_html__( 'General', 'streamtube-core' ),
				'inform'  => true
			),
			'security'      => array(
				'heading' => esc_html__( 'Security', 'streamtube-core' ),
				'inform'  => true
			),
			'import'        => array(
				'heading' => esc_html__( 'Import', 'streamtube-core' ),
				'inform'  => true
			),
			'bulk-import'   => array(
				'heading' => esc_html__( 'Bulk Import', 'streamtube-core' ),
				'inform'  => false
			),
			'notifications' => array(
				'heading' => esc_html__( 'Notifications', 'streamtube-core' ),
				'inform'  => true
			)
		);

		return apply_filters( 'streamtube/core/bunnycdn/admin/tabs', $tabs );
	}

	/**
	 *
	 * Get sync types
	 * 
	 * @return array
	 *
	 * @since 2.1
	 * 
	 */
	public function get_upload_types() {
		return array(
			'auto'   => esc_html__( 'Auto', 'streamtube-core' ),
			'manual' => esc_html__( 'Manual', 'streamtube-core' )
		);
	}

	/**
	 *
	 * Get upload handlers
	 * 
	 * @return array
	 *
	 * @since 2.1
	 * 
	 */
	public function get_upload_handlers() {
		return array(
			'resumable' => esc_html__( 'Direct and Resumable (recommended)', 'streamtube-core' ),
			'normal'    => esc_html__( 'Native', 'streamtube-core' )
		);
	}

	/**
	 *
	 * Get sync types
	 * 
	 * @return array
	 *
	 * @since 2.1
	 * 
	 */
	public function get_sync_types() {
		return array(
			'fetching'   => esc_html__( 'Fetching', 'streamtube-core' ),
			'php_curl'   => esc_html__( 'PHP cURL', 'streamtube-core' ),
			'shell_curl' => esc_html__( 'Shell cURL', 'streamtube-core' )
		);
	}

	/**
	 *
	 * Get webhook URL
	 * 
	 * @return string
	 *
	 * @since 2.1
	 * 
	 */
	public function get_webhook_url() {
		return add_query_arg( array(
			'webhook' => 'bunnycdn',
			'key'     => $this->settings['webhook_key']
		), home_url( '/' ) );
	}

	public function is_enabled() {
		return $this->settings['enable'] && $this->settings['is_connected'] ? true : false;
	}

	/**
	 *
	 * Check if auto sync enabled
	 * 
	 * @return boolean
	 *
	 * @since 2.1
	 * 
	 */
	public function is_auto_sync() {
		return $this->is_enabled() && ( $this->settings['upload_type'] == 'auto' ) ? true : false;
	}

	/**
	 *
	 * Check if Bulk Sync supported
	 * 
	 * @return boolean
	 *
	 * @since 2.1
	 * 
	 */
	public function is_bulk_sync_supported() {
		return ( $this->settings['sync_type'] == 'php_curl' ) ? false : true;
	}

	/**
	 *
	 * Check if post is synced
	 * 
	 * @param  int  $post_id
	 * @return true|false
	 *
	 * @since 2.1
	 * 
	 */
	public function is_synced( $post_id ) {

		$has_data   = get_post_meta( $post_id, '_bunnycdn', true );
		$is_encoded = (int) get_post_meta( $post_id, '_bunnycdn_status', true );

		return $has_data && $is_encoded == 3 ? true : false;
	}

	/**
	 *
	 * Get WP Post ID (attachment_id) from bunny guid
	 * 
	 * @param  string $videoId
	 * @return false|int
	 *
	 * @since 2.1
	 * 
	 */
	public function get_post_id_from_videoId( $videoId ) {
		global $wpdb;

		$results = $wpdb->get_var(
			$wpdb->prepare(
				"SELECT post_id FROM {$wpdb->postmeta} WHERE meta_key = %s AND meta_value = %s",
				'_bunnycdn_guid',
				$videoId
			)
		);

		if ( $results ) {
			return (int) $results;
		}

		return false;
	}

	/**
	 *
	 * Get bunny videoId
	 * 
	 * @param  int $post_id
	 * @return false|string
	 *
	 * @since 2.1
	 * 
	 */
	public function get_video_guid( $post_id ) {

		$post_id = (int) $post_id;

		if ( get_post_type( $post_id ) === 'video' ) {
			$post_id = (int) get_post_meta( $post_id, 'video_url', true );
		}

		return get_post_meta( $post_id, '_bunnycdn_guid', true );
	}

	public function get_downloadable_url( $attachment_id ) {

		$videoId = $this->get_video_guid( $attachment_id );

		$url = wp_get_attachment_url( $attachment_id );

		if ( $videoId ) {
			$url = add_query_arg( array(
				'download' => 'true',
				'name'     => sanitize_file_name( basename( get_attached_file( $attachment_id ) ) )
			), $this->bunnyAPI->get_direct_file_url( $videoId ) );
		}

		/**
		 *
		 * Filter the URL
		 * 
		 */
		return apply_filters( 'streamtube/core/bunnycdn/downloadable_url', $url, $attachment_id, $videoId );
	}

	private function is_valid_user_collection( $collection = array() ) {
		if ( ! $collection || ! is_array( $collection ) ) {
			return false;
		}

		if ( ! isset( $collection['videoLibraryId'] ) || empty( $collection['videoLibraryId'] ) ) {
			return false;
		}
		return true;
	}

	/**
	 *
	 * Update user collection metadata
	 * 
	 * @param  int $user_id
	 * @param  array $collection
	 * @return update_user_meta()
	 *
	 * @since 2.1
	 * 
	 */
	private function _update_user_collection_metadata( $user_id, $collection = array() ) {

		if ( $this->is_valid_user_collection( $collection ) ) {
			return update_user_meta( $user_id, '_bunnycdn_collection', $collection );
		}
	}

	/**
	 *
	 * Get user collection metadata
	 * 
	 * @param  int $user_id
	 * @return get_user_meta()
	 *
	 * @since 2.1
	 * 
	 */
	private function _get_user_collection_metadata( $user_id ) {
		return (array) get_user_meta( $user_id, '_bunnycdn_collection', true );
	}

	/**
	 *
	 * Delete user collection metadata
	 * 
	 * @param  integer $user_id
	 */
	private function _delete_user_collection_metadata( $user_id = 0 ) {
		return delete_user_meta( $user_id, '_bunnycdn_collection' );
	}

	/**
	 * @param  integer $user_id
	 */
	private function _get_collection_name( $user_id = 0 ) {
		$user_data = get_userdata( $user_id );

		$name = sprintf(
			'(#%s - %s) %s',
			$user_id,
			$user_data->user_login,
			$user_data->display_name
		);

		return apply_filters( 'streamtube/core/bunnycdn/get_collection_name/name', $name, $user_data );
	}

	/**
	 *
	 * Create collection
	 * 
	 * @param  int $user_id
	 * @return WP_Error|Array
	 *
	 * @since 2.1
	 * 
	 */
	public function create_user_collection( $user_id = 0, $name = '' ) {

		if ( ! $user_id && is_user_logged_in() ) {
			$user_id = get_current_user_id();
		}

		$user_id = (int) $user_id;

		if ( ! $name ) {
			$name = $this->_get_collection_name( $user_id );
		}

		$collection = $this->get_user_collection( $user_id );

		if (
			! $this->is_valid_user_collection( $collection ) ||
			( is_array( $collection ) && isset( $collection['guid'] ) && is_wp_error( $this->bunnyAPI->get_collection( $collection['guid'] ) ) )
		) {
			$collection = $this->bunnyAPI->create_collection( $name );

			if ( ! is_wp_error( $collection ) ) {
				$this->_update_user_collection_metadata( $user_id, $collection );
			}
		}

		return $collection;
	}

	/**
	 *
	 * Delete user collection
	 * 
	 * @param  string $collectionId
	 * 
	 */
	public function delete_user_collection( $collectionId = '' ) {
		$response = $this->bunnyAPI->delete_collection( $collectionId );

		if ( ! is_wp_error( $response ) ) {

			global $wpdb;

			$wpdb->query(
				$wpdb->prepare(
					"DELETE FROM {$wpdb->usermeta} WHERE meta_key = %s AND meta_value LIKE %s",
					'_bunnycdn_collection',
					'%' . $wpdb->esc_like( $collectionId ) . '%'
				)
			);

			/**
			 * Fires after deleting collection
			 *
			 * @param string $collectionId
			 */
			do_action( 'streamtube/core/bunnycdn/deleted_user_collection', $collectionId );
		}

		return $response;
	}

	/**
	 *
	 * Call API to update user collection if user updated
	 *
	 * @since 2.1.1
	 */
	public function update_user_collection( $user_id, $old_user_data = array(), $userdata = array() ) {

		if ( ! $this->is_enabled() ) {
			return $user_id;
		}

		$collectionId = $this->get_user_collection_id( $user_id );
		$name         = $this->_get_collection_name( $user_id );

		if ( ! $collectionId || ! $name ) {
			return $user_id;
		}

		$response = $this->bunnyAPI->update_collection( $collectionId, $name );

		if ( ! is_wp_error( $response ) ) {
			$collection = $this->bunnyAPI->get_collection( $collectionId );

			if ( ! is_wp_error( $collection ) ) {
				$this->_update_user_collection_metadata( $user_id, $collection );
			}
		}

		return $response;
	}

	/**
	 * Update user collection immediately after webhook is fired.
	 *
	 * @param int $attachment_id ID of the uploaded video or file.
	 * @param array $webhook_data Data received from the webhook.
	 * @param array $video_data Additional video-related data (optional).
	 * 
	 */
	public function update_user_collection_webhook( $attachment_id, $webhook_data = array(), $video_data = array() ) {

		if ( in_array( $webhook_data['Status'], array( 3, 4 ) ) ) {

			$attachment = get_post( $attachment_id );

			$collection = $this->get_user_collection( $attachment->post_author );

			if ( $collection ) {
				$collection = $this->bunnyAPI->get_collection( $collection['guid'] );

				if ( ! is_wp_error( $collection ) ) {
					return $this->_update_user_collection_metadata( $attachment->post_author, $collection );
				}
			}
		}

	}

	/**
	 *
	 * Get collection
	 * 
	 * @param  int $user_id
	 * @return false|array
	 *
	 * @since 2.1
	 * 
	 */
	public function get_user_collection( $user_id = 0 ) {

		if ( ! $user_id ) {
			$user_id = get_current_user_id();
		}

		$user_id = (int) $user_id;

		return $this->_get_user_collection_metadata( $user_id );
	}

	/**
	 *
	 * Get collection id
	 * 
	 * @param  int $user_id
	 * @return false|string
	 *
	 * @since 2.1
	 * 
	 */
	public function get_user_collection_id( $user_id = 0 ) {

		if ( ! $user_id ) {
			$user_id = get_current_user_id();
		}

		$collection = $this->get_user_collection( $user_id );

		if ( is_array( $collection ) ) {
			return isset( $collection['guid'] ) ? $collection['guid'] : false;
		}

		return false;
	}

	/**
	 *
	 * Get user ID from given collection id
	 * 
	 * @param  string $collectionId
	 * @return int user_id|false
	 */
	public function get_user_id_from_collection( $collectionId = '' ) {

		$collectionId = trim( $collectionId );

		if ( ! streamtube_core_is_valid_uuid( $collectionId ) ) {
			return false;
		}

		global $wpdb;

		$results = $wpdb->get_col(
			$wpdb->prepare(
				"SELECT user_id FROM {$wpdb->usermeta} WHERE meta_key = %s AND meta_value LIKE %s",
				'_bunnycdn_collection',
				'%' . $collectionId . '%'
			)
		);

		return $results ? absint( $results[0] ) : false;
	}

	/**
	 *
	 * Get bunny iframe
	 * 
	 */
	public function get_iframe( $videoId ) {
		$allow = array(
			'accelerometer',
			'autoplay',
			'clipboard-write',
			'encrypted-media',
			'gyroscope',
			'picture-in-picture',
			'fullscreen'
		);

		$iframe = sprintf(
			'<iframe allow="%s" src="%s"></iframe>',
			esc_attr( join( ';', $allow ) ),
			$this->bunnyAPI->get_direct_player( $this->settings['libraryId'], $videoId )
		);

		/**
		 *
		 * Filter the player
		 *
		 * @param string $iframe
		 * @param string $video UID
		 * @param string $library UID
		 * 
		 */
		return apply_filters( 'streamtube/core/bunnycdn/bunny_player', $iframe, $videoId, $this->settings['libraryId'] );
	}

	/**
	 *
	 * Delete attachment files
	 * 
	 * @param  int $post_id attachment_id
	 *
	 * @return wp_delete_file_from_directory();
	 * 
	 * @since 2.1
	 */
	public function delete_attachment_file( $post_id ) {
		$uploadpath = wp_get_upload_dir();
		return wp_delete_file_from_directory( get_attached_file( $post_id ), $uploadpath['basedir'] );
	}

	/**
	 *
	 * Generate attachment metadata from given stream
	 * Stream input is data of api->get_video()
	 * 
	 */
	public function _generate_attachment_metadata( $video_data = array() ) {
		$metadata = array();

		$video_data = wp_parse_args( $video_data, array(
			'size'        => 0,
			'storageSize' => 0
		) );

		if ( $video_data['storageSize'] ) {
			$metadata['filesize'] = $video_data['storageSize'];
		}

		if ( $video_data['size'] ) {
			$metadata['filesize'] = $video_data['size'];
		}

		if ( isset( $video_data['length'] ) ) {
			$metadata['length'] = $video_data['length'];
		}

		if ( isset( $video_data['width'] ) ) {
			$metadata['width'] = $video_data['width'];
		}

		if ( isset( $video_data['height'] ) ) {
			$metadata['height'] = $video_data['height'];
		}

		return $metadata;
	}

	/**
	 *
	 * Create new Video after adding attachment
	 * 
	 * @param int $post_id
	 *
	 * @since 2.1
	 * 
	 */
	public function _add_attachment( $post_id ) {

		$post           = get_post( $post_id );
		$user_id        = $post->post_author;
		$attachment_url = wp_get_attachment_url( $post_id );
		$collection     = '';
		$upload         = false;

		if ( get_post_meta( $post_id, '_bunnycdn', true ) ) {
			return;
		}

		if ( get_post_type( $post->post_parent ) === 'ad_tag' ) {
			return;
		}

		if ( $this->settings['file_organize'] ) {
			$collection = $this->create_user_collection( $user_id );

			if ( is_wp_error( $collection ) ) {

				$this->bunnyAPI->write_log_file(
					get_attached_file( $post_id ),
					$collection->get_error_code() . ' ' . $collection->get_error_message(),
					$collection->get_error_code()
				);
				return $post_id;
			}
		}

		$create = $this->bunnyAPI->create_video( get_the_title( $post_id ), $collection['guid'] );

		if ( ! is_wp_error( $create ) ) {
			update_post_meta( $post_id, '_bunnycdn', $create );
			update_post_meta( $post_id, '_bunnycdn_guid', $create['guid'] );
			update_post_meta( $post_id, '_bunnycdn_status', '-1' );// uploading

			/**
			 *
			 * Fires after Video created
			 *
			 * @param array $create
			 * @param int $post_id (attachment_id)
			 *
			 * @since 2.1
			 * 
			 */
			do_action( 'streamtube/core/bunnycdn/video/created', $create, $post_id );

			$file = get_post_meta( $post_id, '_wp_attached_file', true );

			$this->bunnyAPI->delete_log_file( get_attached_file( $post_id ) );

			if ( wp_http_validate_url( $file ) ) {
				$upload = $this->bunnyAPI->fetch_video( $create['guid'], $file );
			} else {

				$file = get_attached_file( $post_id );

				$this->bunnyAPI->create_empty_log_file( $file );

				switch ($this->settings['sync_type']) {
					case 'shell_curl':
						$upload = $this->bunnyAPI->shell_curl_upload_video(
							$create['guid'],
							$file,
							$this->settings['curl_path'],
							wp_validate_boolean( $this->settings['tsp'] ),
							$this->settings['tsp_path']
						);
						break;

					case 'php_curl':
						$upload = $this->bunnyAPI->php_curl_upload_video( $create['guid'], $file );
						break;

					default:
						$upload = $this->bunnyAPI->fetch_video( $create['guid'], $attachment_url );
						break;
				}
			}
			/**
			 *
			 * Fires after Video uploaded
			 *
			 * @param array $upload
			 * @param array $create
			 * @param int $post_id (attachment_id)
			 *
			 * @since 2.1
			 * 
			 */
			do_action( 'streamtube/core/bunnycdn/video/uploaded', $upload, $create, $post_id );
		}

		return $create;
	}

	/**
	 *
	 * Create new Video after adding attachment
	 * 
	 * @param int $post_id
	 *
	 * @since 2.1
	 * 
	 */
	public function add_attachment( $post_id ) {

		if ( ! $this->is_auto_sync() || get_post_meta( $post_id, 'live_status', true ) ) {
			return $post_id;
		}

		if ( wp_attachment_is( 'video', $post_id ) || wp_attachment_is( 'audio', $post_id ) ) {
			return $this->_add_attachment( $post_id );
		}

		return $post_id;
	}

	/**
	 *
	 * Update Video after updating attachment
	 * 
	 * @param  int $post_id
	 * @return update_video()
	 *
	 * @since 2.1
	 * 
	 */
	public function _attachment_updated( $post_id ) {

		$videoId = $this->get_video_guid( $post_id );

		$post_author = get_post( $post_id )->post_author;

		if ( ! $videoId ) {
			return $post_id;
		}

		$title        = get_the_title( $post_id );
		$collectionId = $this->get_user_collection_id( $post_author );

		$args = compact( 'videoId', 'title' );

		if ( is_string( $collectionId ) ) {
			$args['collectionId'] = $collectionId;
		}

		return $this->bunnyAPI->update_video( $args );
	}

	/**
	 *
	 * Update Video after updating attachment
	 * 
	 * @param  int $post_id
	 * @return update_video()
	 *
	 * @since 2.1
	 * 
	 */
	public function attachment_updated( $post_id ) {
		return $this->_attachment_updated( $post_id );
	}

	/**
	 *
	 * Delete video while deleting attachment
	 * 
	 * @param  int $post_id
	 * @return delete_video()
	 *
	 * @since 2.1
	 * 
	 */
	public function _delete_attachment( $post_id ) {

		$videoId = $this->get_video_guid( $post_id );

		if ( ! $videoId ) {
			return $post_id;
		}

		$this->bunnyAPI->delete_log_file( get_attached_file( $post_id ) );

		return $this->bunnyAPI->delete_video( $videoId );
	}

	/**
	 *
	 * Delete video while deleting attachment
	 * 
	 * @param  int $post_id
	 * @return delete_video()
	 *
	 * @since 2.1
	 * 
	 */
	public function delete_attachment( $post_id ) {
		return $this->_delete_attachment( $post_id );
	}

	/**
	 *
	 * Call API to Get bunny video data
	 * 
	 * @param  integer $attachment_id
	 * @return WP_Error|array
	 * 
	 */
	public function get_video_data( $attachment_id = 0 ) {

		$videoId = $this->get_video_guid( $attachment_id );

		if ( $videoId ) {
			return $this->bunnyAPI->get_video( $videoId );
		}

		return false;
	}

	/**
	 *
	 * Update bunny video data
	 * 
	 * @param  integer $attachment_id
	 * @param array $video_data
	 * @return WP_Error|array
	 * 
	 */
	public function update_video_data( $attachment_id, $video_data = array() ) {

		if ( ! $video_data ) {
			$video_data = $this->get_video_data( $attachment_id );
		}

		if ( is_array( $video_data ) ) {
			update_post_meta( $attachment_id, '_bunnycdn', $video_data );
			update_post_meta( $attachment_id, '_bunnycdn_guid', $video_data['guid'] );
			update_post_meta( $attachment_id, '_bunnycdn_status', $video_data['status'] );

			update_post_meta( $attachment_id, '_wp_attachment_metadata', $this->_generate_attachment_metadata( $video_data ) );
		}

		return $video_data;
	}

	/**
	 *
	 * Get video status
	 * 
	 * @param  int $attachment_id
	 * @return html
	 *
	 * @since 2.1
	 * 
	 */
	public function get_video_status( $attachment_id ) {

		$status   = -1;
		$progress = 0;
		$statuses = $this->bunnyAPI->get_webhook_video_statuses();

		$video_data = $this->update_video_data( $attachment_id );

		if ( is_wp_error( $video_data ) ) {
			$message = $video_data->get_error_message();
		} else {

			$video_data = wp_parse_args( $video_data, array(
				'status'               => -1,
				'encodeProgress'       => 0,
				'availableResolutions' => ''
			) );

			$status   = (int) $video_data['status'];
			$progress = (int) $video_data['encodeProgress'];

			/**
			 *
			 * Fires once webhook updated
			 * Fake Webhook
			 *
			 * @param attachment_id
			 * @param array $data
			 *
			 * @since 2.1
			 * 
			 */
			do_action( 'streamtube/core/bunny/webhook/update', $attachment_id, array(
				'Status'    => $status,
				'VideoGuid' => $video_data['guid']
			), $video_data );

			if ( in_array( $status, array( 0, -1, 1, 2, 5, 6, 7, 8, 9, 10 ) ) ) {
				$message = $statuses[ (string) $status ][1];
			}

			if ( in_array( $status, array( 3, 4 ) ) && ( $progress < 100 || ! $video_data['availableResolutions'] ) ) {
				$message = esc_html__( 'Encoding is almost finished. Please wait a moment.', 'streamtube-core' );
			}

			// playable
			if ( $progress == 100 || $video_data['availableResolutions'] ) {
				$message = '';
			}
		}

		if ( ! empty( $message ) ) {
			return new WP_Error( 'bunny_status_' . $status, $message, array_merge( compact( 'status', 'progress' ), array(
				'handler' => 'bunny'
			) ) );
		}

		return true;
	}

	/**
	 * Retrieves video captions for a given attachment ID.
	 *
	 * This method fetches the captions associated with a video attachment.
	 * If the video data is not provided, it attempts to retrieve it using the
	 * `get_video_data` method. If the video data is invalid or an error occurs,
	 * the method returns false. Otherwise, it checks if captions are available
	 * in the video data and returns them.
	 *
	 * @param int   $attachment_id The ID of the video attachment. Default is 0.
	 * @param array $video_data    Optional. The video data array. If not provided,
	 *                             it will be fetched using the attachment ID.
	 *                             Default is an empty array.
	 * 
	 * @return array|false Returns an array of captions if available, or false if
	 *                     no captions are found or an error occurs.
	 */
	public function get_video_captions( $attachment_id = 0, $video_data = array() ) {
		if ( ! $video_data ) {
			$video_data = get_post_meta( $attachment_id, '_bunnycdn', true );
		}

		if ( ! is_array( $video_data ) || is_wp_error( $video_data ) ) {
			return false;
		}

		if ( isset( $video_data['captions'] ) && $video_data['captions'] ) {
			return $video_data['captions'];
		}

		return false;
	}

	/**
	 *
	 * Sync video
	 * 
	 * @param  int $post_id
	 * @return _add_attachment()
	 *
	 * @since 2.1
	 * 
	 */
	public function sync_video( $post_id ) {

		if ( $this->is_synced( $post_id ) ) {
			return new WP_Error(
				'synced',
				esc_html__( 'This video is already synced', 'streamtube-core' )
			);
		}

		if ( ! Streamtube_Core_BunnyCDN_Perms::manual_sync() ) {
			return new WP_Error(
				'no_permission',
				esc_html__( 'You do not have permission to sync this video.', 'streamtube-core' )
			);
		}

		return $this->_add_attachment( $post_id );
	}

	/**
	 *
	 * Retry sunc video
	 * 
	 * @param  int $post_id attachment_id
	 * @return _add_attachment()
	 *
	 * @since 2.1
	 * 
	 */
	public function retry_sync_video( $post_id ) {

		if ( $this->is_synced( $post_id ) ) {
			return new WP_Error(
				'synced',
				esc_html__( 'This video is already synced', 'streamtube-core' )
			);
		}

		if ( "" != $videoId = $this->get_video_guid( $post_id ) ) {
			$this->bunnyAPI->delete_video( $videoId );
		}

		delete_post_meta( $post_id, '_bunnycdn' );
		delete_post_meta( $post_id, '_bunnycdn_guid' );
		delete_post_meta( $post_id, '_bunnycdn_status' );

		return $this->_add_attachment( $post_id );
	}

	/**
	 *
	 * AJAX get video player status
	 * 
	 * @since 2.1
	 * 
	 */
	public function ajax_get_video_status() {
		check_ajax_referer( '_wpnonce' );

		if ( ! isset( $_GET['attachment_id'] ) ) {
			wp_send_json_error( new WP_Error(
				'attachment_not_found',
				esc_html__( 'Attachment ID was not found', 'streamtube-core' )
			) );
		}

		$output = $this->get_transcoding_progress( $_GET['attachment_id'] );

		if ( strpos( $output, '<iframe' ) !== false ) {
			wp_send_json_success( $output );
		}

		wp_send_json_error( $output );
	}

	/**
	 *
	 * Refresh Bunny data
	 * 
	 */
	public function ajax_refresh_bunny_data() {

		check_ajax_referer( '_wpnonce' );

		if ( ! isset( $_POST['attachment_id'] ) ) {
			wp_send_json_error( new WP_Error(
				'attachment_not_found',
				esc_html__( 'Attachment ID was not found', 'streamtube-core' )
			) );
		}

		if ( ! current_user_can( 'edit_post', $_POST['attachment_id'] ) ) {
			wp_send_json_error( new WP_Error(
				'no_permission',
				esc_html__( 'You do not have permission to process this action.', 'streamtube-core' )
			) );
		}

		$response = $this->update_video_data( $_POST['attachment_id'] );

		if ( is_wp_error( $response ) ) {
			wp_send_json_error( $response );
		}

		wp_send_json_success( $response );
	}

	/**
	 *
	 * AJAX sync
	 * 
	 * @since 2.1
	 */
	public function ajax_sync() {

		if ( ! isset( $_POST ) || ! isset( $_POST['attachment_id'] ) ) {
			wp_send_json_error( new WP_Error(
				'invalid_requested',
				esc_html__( 'Invalid Requested', 'streamtube-core' )
			) );
		}

		check_ajax_referer( 'bunny-sync-' . absint( $_POST['attachment_id'] ), 'nonce' );

		$results = $this->sync_video( $_POST['attachment_id'] );

		if ( is_wp_error( $results ) ) {
			wp_send_json_error( $results );
		}

		wp_send_json_success( array(
			'results' => $results,
			'message' => esc_html__( 'Syncing', 'streamtube-core' )
		) );
	}

	/**
	 *
	 * AJAX retry sync
	 * 
	 * @since 2.1
	 */
	public function ajax_retry_sync() {

		if ( ! isset( $_POST ) || ! isset( $_POST['attachment_id'] ) ) {
			exit;
		}

		$results = $this->retry_sync_video( $_POST['attachment_id'] );

		if ( is_wp_error( $results ) ) {
			wp_send_json_error( $results );
		}

		wp_send_json_success( array(
			'results' => $results,
			'message' => esc_html__( 'Syncing', 'streamtube-core' )
		) );
	}

	/**
	 *
	 * Bulk sync
	 * 
	 * @param  array $post_ids $attachment_ids
	 *
	 * @return array $queued array of queued post ids
	 *
	 * @since 2.1
	 * 
	 */
	public function bulk_media_sync( $post_ids = array() ) {

		$sync_types = $this->get_sync_types();

		if ( ! $post_ids ) {
			return new WP_Error(
				'empty_posts',
				esc_html__( 'Empty Posts', 'streamtube-core' )
			);
		}

		if ( $this->settings['sync_type'] == 'php_curl' ) {
			return new WP_Error(
				'php_curl_not_supported',
				sprintf(
					esc_html__( 'Bulk Sync does not support %s type', 'streamtube-core' ),
					$sync_types[ $this->settings['sync_type'] ]
				)
			);
		}

		$queued = array();

		foreach ( $post_ids as $post_id ) {
			$_queue = $this->_add_attachment( $post_id );

			if ( ! is_wp_error( $_queue ) ) {
				$queued[] = $post_id;
			}
		}

		return $queued;
	}

	/**
	 *
	 * Filter attachment URL
	 * 
	 * @param  string $url
	 * @param  int $post_id
	 * @return string
	 *
	 * @since 2.1
	 * 
	 */
	public function filter_wp_get_attachment_url( $url, $attachment_id ) {

		$videoId = $this->get_video_guid( $attachment_id );

		if ( $this->is_enabled() && $videoId ) {
			if ( wp_attachment_is( 'audio', $attachment_id ) ) {
				$url = $this->bunnyAPI->get_direct_file_url( $videoId );
			} else {
				$url = $this->bunnyAPI->get_video_hls_url( $videoId );
			}
		}

		return $url;
	}

	/**
	 *
	 * Display the transcoding progress in case Bunny player is enabled
	 * 
	 * @param  string|int $attachment_id
	 * @return string
	 * 
	 */
	public function get_transcoding_progress( $attachment_id ) {

		$output = '';

		$response = $this->get_video_status( $attachment_id );

		if ( is_wp_error( $response ) ) {

			$error_data = $response->get_error_data();

			$status   = $error_data['status'];
			$progress = $error_data['progress'];
			$message  = $response->get_error_message();
			$code     = $response->get_error_code();

			$spinner = true;

			// The video has finished processing but failed.
			if ( in_array( (int) $status, array( 5, -1 ) ) ) {
				$spinner = false;
			}

			$args = compact( 'code', 'message', 'spinner', 'attachment_id', 'progress', 'status' );

			ob_start();

			/**
			 *
			 * Filter the output args
			 * 
			 * @since 2.1
			 */
			$args = apply_filters( 'streamtube/core/bunnycdn/video_player_status', $args );

			load_template(
				plugin_dir_path( __FILE__ ) . 'frontend/video-status.php',
				true,
				$args
			);

			$output = ob_get_clean();
		}

		if ( $output ) {
			return $output;
		}

		return $this->get_iframe( $this->get_video_guid( $attachment_id ) );
	}

	/**
	 *
	 * Filter player setup
	 *
	 */
	public function filter_player_setup( $setup, $source ) {

		if ( ! $this->is_enabled() || ! $this->get_video_guid( $source ) ) {
			return $setup;
		}

		$playerLoadSource = array();

		$video_data = wp_parse_args( (array) get_post_meta( $source, '_bunnycdn', true ), array(
			'encodeProgress'       => 0,
			'availableResolutions' => ''
		) );

		if ( ! $video_data['availableResolutions'] ) {
			$playerLoadSource = array(
				'message' => esc_html__( 'Waiting ...', 'streamtube-core' )
			);
		}

		if ( $playerLoadSource ) {
			$setup['plugins']['playerLoadSource'] = $playerLoadSource;
			// Reset sources
			$setup['sources'] = array();
		}

		return $setup;
	}

	/**
	 *
	 * Hooked into "streamtube/core/player/check_video_source" filter
	 *
	 * 
	 * @param  string|WP_Error $source
	 * @param  int $post_id
	 * 
	 */
	public function filter_player_load_source( $source, $post_id, $data = array() ) {

		$attachment_id = get_post_meta( $post_id, 'video_url', true );

		$videoId = $this->get_video_guid( $attachment_id );

		if ( ! $this->is_enabled() || ! $videoId ) {
			return $source;
		}

		$src = '';

		$status = $this->get_video_status( $attachment_id );

		if ( is_wp_error( $status ) ) {
			return $status;
		} else {
			$src = wp_get_attachment_url( $attachment_id );
		}

		if ( $src ) {
			return array(
				'type' => 'application/x-mpegURL',
				'src'  => $src
			);
		}

		return $source;
	}

	/**
	 *
	 * Filter player output
	 *
	 * @since 2.1
	 * 
	 */
	public function filter_player_output( $player, $setup, $source ) {

		if ( ! $this->is_enabled() || ! $this->settings['bunny_player'] ) {
			return $player;
		}

		if ( "" == ( $videoId = $this->get_video_guid( $source ) ) ) {
			return $player;
		}

		$video_data = wp_parse_args( (array) get_post_meta( $source, '_bunnycdn', true ), array(
			'encodeProgress'       => 0,
			'availableResolutions' => ''
		) );

		if ( ! $video_data['availableResolutions'] ) {
			return $this->get_transcoding_progress( $source );
		}

		return $this->get_iframe( $videoId );
	}

	public function filter_download_file_url( $url, $post_id ) {

		if ( ! $this->is_enabled() ) {
			return $url;
		}

		$mediaid = $this->Post->get_source( $post_id );

		if ( wp_attachment_is( 'video', $mediaid ) || wp_attachment_is( 'audio', $mediaid ) ) {
			$url = $this->get_downloadable_url( $mediaid );
		}

		return $url;
	}

	/**
	 * Filters and adds subtitles (captions) to the video player tracks.
	 *
	 * This method retrieves video captions associated with the given source,
	 * normalizes the language codes, and appends them to the tracks array.
	 *
	 * @param array  $tracks  The existing array of subtitle tracks for the video player.
	 * @param string $default The default subtitle track (not used in this method).
	 * @param array  $setup   The player setup configuration (not used in this method).
	 * @param string $source  The video source identifier.
	 *
	 * @return array The updated array of subtitle tracks with added captions.
	 */
	public function filter_player_subtitles( $tracks, $default, $setup, $source ) {
		$captions = $this->get_video_captions( $source );

		if ( is_array( $captions ) && ! empty( $captions ) ) {
			foreach ( $captions as $caption ) {
				// Append to tracks array
				$tracks[] = array(
					'language' => strtolower( str_replace( '-auto', '', $caption['srclang'] ) ),
					'name'     => $caption['label'],
					'source'   => $this->bunnyAPI->get_caption_url( $this->get_video_guid( $source ), $caption['srclang'] )
				);
			}
		}

		return $tracks;
	}

	/**
	 *
	 * Display the notice within thumbnail field on the Edit Post form
	 * 
	 */
	public function thumbnail_notice( $post ) {
		if ( ! has_post_thumbnail( $post ) ) {

			$Post = new Streamtube_Core_Post();

			if ( $this->get_video_guid( $Post->get_source( $post->ID ) ) ) {
				load_template( untrailingslashit( plugin_dir_path( __FILE__ ) ) . '/frontend/process-thumbnail.php' );
			}
		}
	}

	/**
	 *
	 * Update attachment metadata once webhook fires
	 * 
	 */
	public function update_attachment_metadata( $attachment_id, $webhook_data = array(), $video_data = array() ) {
		return $this->update_video_data( $attachment_id, $video_data );
	}

	/**
	 *
	 * Generate thumbnail image
	 * 
	 * @param  int $attachment_id
	 * @param  string $videoId
	 * @return WP_Error|int
	 *
	 * @since 2.1
	 * 
	 */
	public function generate_thumbnail_image( $attachment_id, $videoId = '' ) {

		if ( has_post_thumbnail( $attachment_id ) ) {
			return new WP_Error(
				'thumbnail_exists',
				esc_html__( 'Thumbnail Image is already existed', 'streamtube-core' )
			);
		}

		if ( ! $videoId ) {
			$videoId = $this->get_video_guid( $attachment_id );
		}

		if ( ! $videoId ) {
			return new WP_Error(
				'videoId_not_found',
				esc_html__( 'VideoId was not found', 'streamtube-core' )
			);
		}

		$thumbnail_url = $this->bunnyAPI->get_video_thumbnail_url( $videoId );

		if ( apply_filters( 'streamtube/core/hash_bunny_image_file_name', true ) ) {
			add_filter(
				'sanitize_file_name',
				'streamtube_core_hash_file_name',
				10,
				1
			);
		}

		$thumbnail_id = media_sideload_image( $thumbnail_url, $attachment_id, null, 'id' );

		if ( ! is_wp_error( $thumbnail_id ) ) {

			set_post_thumbnail( $attachment_id, $thumbnail_id );

			wp_update_post( array(
				'ID'          => $thumbnail_id,
				'post_parent' => $attachment_id,
				'post_author' => get_post( $attachment_id )->post_author
			) );

			$attachment = get_post( $attachment_id );

			if ( $attachment->post_parent && ! has_post_thumbnail( $attachment->post_parent ) ) {
				set_post_thumbnail( $attachment->post_parent, $thumbnail_id );
			}
		}

		return $thumbnail_id;
	}

	/**
	 *
	 * Generate webp image
	 * 
	 * @param  int $post_id
	 * @param  string $videoId
	 * @return WP_Error|int
	 *
	 * @since 2.1
	 * 
	 */
	public function generate_webp_image( $attachment_id, $videoId = '' ) {

		if ( ! $videoId ) {
			$videoId = $this->get_video_guid( $attachment_id );
		}

		if ( ! $videoId ) {
			return new WP_Error(
				'videoId_not_found',
				esc_html__( 'VideoId was not found', 'streamtube-core' )
			);
		}

		$attachment = get_post( $attachment_id );
		$maybe_url  = $this->Post->get_thumbnail_image_url_2( $attachment_id );

		if ( $maybe_url && ( wp_http_validate_url( $maybe_url ) || wp_attachment_is_image( $maybe_url ) ) && $attachment->post_parent ) {
			return $this->Post->update_thumbnail_image_url_2( $attachment->post_parent, $maybe_url );
		}

		$webp_url = $this->bunnyAPI->get_video_preview_webp_url( $videoId );

		if ( apply_filters( 'streamtube/core/hash_bunny_image_file_name', true ) ) {
			add_filter(
				'sanitize_file_name',
				'streamtube_core_hash_file_name',
				10,
				1
			);
		}

		$webp_id = media_sideload_image( $webp_url, $attachment_id, null, 'id' );

		if ( ! is_wp_error( $webp_id ) ) {

			$this->Post->update_thumbnail_image_url_2( $attachment_id, $webp_id );

			if ( $attachment->post_parent ) {
				$this->Post->update_thumbnail_image_url_2( $attachment->post_parent, $webp_id );
			}

			wp_update_post( array(
				'ID'          => $webp_id,
				'post_parent' => $attachment_id,
				'post_author' => $attachment->post_author
			) );
		}

		return (int) $webp_id;
	}

	/**
	 *
	 * Generate thumbnail images
	 * 
	 * @param  int $post_id
	 * @param  array $data webhook response
	 *
	 * @since 2.1
	 * 
	 */
	public function update_thumbnail_images( $attachment_id, $webhook_data = array(), $video_data = array() ) {

		$webhook_data = wp_parse_args( $webhook_data, array(
			'status' => 0
		) );

		$video_data = wp_parse_args( $video_data, array(
			'status' => 0
		) );

		$statuses = array( 3, 4 );

		if ( in_array( $webhook_data['Status'], $statuses ) || in_array( $video_data['status'], $statuses ) ) {

			if ( $this->settings['auto_import_thumbnail'] ) {
				$this->generate_thumbnail_image( $attachment_id, $webhook_data['VideoGuid'] );
			}

			if ( $this->settings['animation_image'] ) {
				$this->generate_webp_image( $attachment_id, $webhook_data['VideoGuid'] );
			}
		}
	}

	/**
	 *
	 * Filter webp image URL
	 * 
	 * @param  string $image_url
	 * @param  int $post_id
	 * @return $image_url
	 *
	 * @since 2.1.10
	 * 
	 */
	public function filter_thumbnail_image_2( $image_url, $image_id, $post_id ) {

		if ( ! $this->is_enabled() || ! empty( $image_url ) ) {
			return $image_url;
		}

		if ( ! apply_filters( 'streamtube/core/bunnycdn/load_webp', true ) ) {
			return $image_url;
		}

		$attachment_id = get_post_meta( $post_id, 'video_url', true );

		if ( ! wp_attachment_is( 'video', $attachment_id ) ) {
			return $image_url;
		}

		return $this->bunnyAPI->get_video_preview_webp_url( $this->get_video_guid( $attachment_id ) );
	}

	/**
	 *
	 * Delete orignial file
	 * 
	 * @param  int $post_id
	 * @param  array $data webhook response
	 *
	 * @since 2.1
	 * 
	 */
	public function delete_original_file( $attachment_id, $webhook_data, $video_data = array() ) {

		if ( in_array( $webhook_data['Status'], array( 3, 4 ) ) ) {

			$attachment = get_post( $attachment_id );

			if ( get_post_type( get_post_parent( $attachment ) ) != 'ad_tag' && wp_validate_boolean( $this->settings['delete_original'] ) ) {
				$this->delete_attachment_file( $attachment_id );
			}
		}
	}

	public function auto_categorization( $attachment_id, $webhook_data, $video_data = array() ) {
		if ( $this->settings['auto_import_uploads'] && $this->settings['auto_import_auto_cat'] ) {
			if ( $video_data['category'] && $video_data['category'] != 'unknown' && get_post_meta( $attachment_id, '_bunnycdn_import', true ) ) {

				if ( ! term_exists( $video_data['category'], 'categories', null ) ) {
					$term = wp_insert_term( ucwords( $video_data['category'] ), 'categories' );

					if ( is_array( $term ) ) {
						wp_set_post_terms( get_post( $attachment_id )->post_parent, (int) $term['term_id'], 'categories', true );
					}
				} else {
					$term = get_term_by( 'slug', $video_data['category'], 'categories' );

					if ( is_object( $term ) ) {
						wp_set_post_terms( get_post( $attachment_id )->post_parent, (int) $term->term_id, 'categories', true );
					}
				}
			}
		}
	}

	public function ajax_read_log_content() {
		$attachment_id = isset( $_GET['attachment_id'] ) ? (int) $_GET['attachment_id'] : 0;

		if ( ! $attachment_id || ! Streamtube_Core_BunnyCDN_Perms::view_encode_logs() ) {
			exit;
		}

		$log_content = $this->bunnyAPI->read_log_file( get_attached_file( $attachment_id ) );

		if ( ! $log_content ) {
			esc_html_e( 'No log content available', 'streamtube-core' );

		} else {
			printf(
				'<pre>%s</pre>',
				$log_content
			);
		}
		exit;
	}

	/**
	 *
	 * AJAX view log file content
	 * 
	 * @since 2.1
	 */
	public function ajax_read_task_log_content() {

		$task_id = isset( $_GET['task_id'] ) ? (int) $_GET['task_id'] : -1;

		if ( $task_id == -1 || ! Streamtube_Core_BunnyCDN_Perms::view_encode_logs() ) {
			exit;
		}

		$log_content = $this->bunnyAPI->read_task_log_content( $task_id );

		if ( $log_content ) {
			printf(
				'<pre>%s</pre>',
				$log_content
			);
		} else {
			esc_html_e( 'No log content', 'streamtube-core' );
		}
		exit;
	}

	/**
	 *
	 * Try to update video data after updating post
	 *
	 * Hooked into "edit_post_video"
	 *
	 * @param int $post_id video post ID
	 * 
	 */
	public function refresh_bunny_data( $post_id ) {

		if ( ! $this->is_enabled() ) {
			return;
		}

		$maybe_attachment_id = (int) get_post_meta( $post_id, 'video_url', true );

		if ( $maybe_attachment_id ) {
			return $this->update_video_data( $maybe_attachment_id );
		}
	}

	/**
	 *
	 * Create draft video and sha resumable upload url
	 * 
	 * @return WP_Error|array
	 */
	public function create_resumable_token( $args = array() ) {

		$user_id = get_current_user_id();

		$errors = false;

		$args = wp_parse_args( $args, array(
			'name' => '',
			'size' => 0
		) );

		extract( $args );

		$errors = new Streamtube_Core_Upload( $args );

		if ( ! $this->is_enabled() ) {
			$errors->get_errors()->add(
				'not_enabled',
				esc_html__( 'Bunny is not enabled', 'streamtube-core' )
			);
		}

		// Verify the collection
		$collection = $this->bunnyAPI->get_collection( $this->get_user_collection_id( $user_id ) );

		if ( is_wp_error( $collection ) ) {
			$errors->get_errors()->add(
				$collection->get_error_code(),
				$collection->get_error_message()
			);
		}

		$errors = $errors->get_errors();

		/**
		 *
		 * Filter errors
		 * 
		 */
		$errors = apply_filters(
			'streamtube/core/bunnycdn/create_resumable_token/errors',
			$errors,
			$args
		);

		if ( $errors->get_error_codes() ) {
			return $errors;
		}

		if ( ! $args['name'] ) {
			$args['name'] = 'Draft';
		} else {
			$args['name'] = 'Draft-' . $args['name'];
		}

		$expiration = 60 * 60;

		$create = $this->bunnyAPI->create_video( $args['name'], $collection['guid'] );

		if ( is_wp_error( $create ) ) {
			return $create;
		}

		$video_id = $create['guid'];

		$library_id = $this->bunnyAPI->get_library_id();

		$sha256 = $this->bunnyAPI->generate_sha256_tus( array(
			'videoId'    => $create['guid'],
			'expiration' => $expiration
		) );

		$response = array_merge( compact( 'sha256', 'library_id', 'video_id', 'size' ), array(
			'collection' => $collection['guid']
		) );

		/**
		 *
		 * Fires after creating the token
		 *
		 * @param $args sent via rest
		 * @param $response results of create_direct_upload() call
		 * 
		 */
		do_action( 'created_direct_upload_url', array_merge( $args, array(
			'uid'     => $video_id,
			'service' => 'bunny'
		) ) );

		/**
		 *
		 * Fires after creating resumable upload
		 * 
		 */
		do_action( 'streamtube/core/bunnycdn/created_resumable_token', $response );

		return $response;
	}

	/**
	 *
	 * Disable Big Uploads if active
	 * We don't need it anymore as resumable upload is enabled by default
	 * 
	 */
	public function disable_big_uploads( $jsvars ) {
		if ( $this->is_enabled() && $this->settings['upload_handler'] === 'resumable' && $this->settings['upload_type'] === 'auto' ) {
			$jsvars['chunkUpload'] = 'off';
		}
		return $jsvars;
	}

	/**
	 * Updates the metadata of a BunnyCDN video.
	 *
	 * This function updates the metadata of a video hosted on BunnyCDN
	 * based on the provided post ID, attachment ID, and video data.
	 *
	 * @param int   $post_id        The ID of the post associated with the video. Default is 0.
	 * @param int   $attachment_id  The ID of the attachment associated with the video. Default is 0.
	 * @param array $video_data     An array containing the video metadata to be updated. Default is an empty array.
	 * @return void
	 */
	public function update_bunny_video_metadata( $post_id = 0, $attachment_id = 0, $video_data = array() ) {
		if ( ! $post_id ) {
			return;
		}

		$post_content = '';

		$meta_input = array();

		if ( isset( $video_data['metaTags'] ) ) {
			if ( is_array( $video_data['metaTags'] ) && $video_data['metaTags'] ) {
				$metadata = $video_data['metaTags'];

				for ( $i = 0; $i < count( $metadata ); $i++ ) {

					$key   = isset( $metadata[ $i ]['property'] ) ? $metadata[ $i ]['property'] : '';
					$value = isset( $metadata[ $i ]['value'] ) ? $metadata[ $i ]['value'] : '';

					if ( ! $key || ! $value ) {
						continue;
					}

					if ( $key === 'description' ) {
						$post_content = $value;
						continue;
					}

					$meta_input['meta_key'] = $value;
				}
			}

			if ( $post_content || $meta_input ) {

				$args = array( 'ID' => $post_id );

				if ( $post_content && ! get_post( $post_id )->post_content ) {
					$args['post_content'] = $post_content;
				}

				if ( $meta_input ) {
					$args['meta_input'] = $meta_input;
				}

				return wp_update_post( $args );
			}
		}
	}

	/**
	 *
	 * Call API to delete remote bunny video
	 * 
	 * @param  string $videoId
	 * 
	 */
	public function delete_bunny_video( $videoId = '' ) {

		if ( ! Streamtube_Core_BunnyCDN_Perms::delete_videos() ) {
			return new WP_Error(
				'no_permission',
				esc_html__( 'You do not have permission to delete this video.', 'streamtube-core' )
			);
		}

		$attachment_id = $this->get_post_id_from_videoId( $videoId );

		if ( $attachment_id ) {
			$attachment = get_post( $attachment_id );

			if ( $attachment->post_parent ) {
				wp_delete_post( $attachment->post_parent, true );
			} else {
				wp_delete_attachment( $attachment_id, true );
			}
		}

		return $this->bunnyAPI->delete_video( $videoId );
	}

	/**
	 * Handles the AJAX request to delete a BunnyCDN video.
	 *
	 * This function processes the AJAX request to delete a video from BunnyCDN.
	 * It expects a 'data' parameter in the request, which should contain the video identifier.
	 * The function checks the AJAX nonce for security, attempts to delete the video,
	 * and returns a JSON response indicating success or failure.
	 *
	 * @return void
	 */
	public function ajax_delete_bunny_video() {

		$http_data = wp_parse_args( $_REQUEST, array(
			'data' => ''
		) );

		extract( $http_data );

		check_ajax_referer( 'delete-' . $data );

		$response = $this->delete_bunny_video( $data );

		if ( is_wp_error( $response ) ) {
			wp_send_json_error( $response );
		}

		wp_send_json_success( array_merge( $response, array(
			'message' => esc_html__( 'Video deleted', 'streamtube-core' )
		) ) );
	}

	/**
	 *
	 * Import bunny video
	 * 
	 * @param  string  $videoId
	 * @param  boolean $include_tax
	 *
	 * @return WP_Error|Array
	 */
	public function import_bunny_video( $args = array() ) {

		$args = wp_parse_args( $args, array(
			'guid'          => '',
			'size'          => 0,
			'type'          => 'video/mp4',
			'include_tax'   => false,
			'status'        => '',
			'attachment_id' => 0
		) );

		if ( ! $args['type'] ) {
			$args['type'] = 'video/mp4';
		}

		extract( $args );

		$video_data = $this->bunnyAPI->get_video( $guid );

		if ( is_wp_error( $video_data ) ) {
			return $video_data;
		}

		if ( "" != $attachment = get_post( $attachment_id ) ) {

			update_post_meta(
				$attachment_id,
				'_wp_attachment_metadata',
				$this->_generate_attachment_metadata( array_merge( $video_data, array(
					'size' => $size
				) ) )
			);

			/**
			 * Fires after checking if a post already exists.
			 *
			 * @param int   $post_id       The ID of the parent post.
			 * @param int   $attachment_id The ID of the attachment.
			 * @param array $video_data    An array containing video data.
			 */
			do_action( 'streamtube/core/bunny/existed/post', $attachment->post_parent, $attachment_id, $video_data );

			return new WP_Error(
				'existed',
				sprintf(
					esc_html__( '%s (%s) was already imported.', 'streamtube-core' ),
					$attachment->post_title,
					'<a target="_blank" href="' . esc_url( get_permalink( $attachment->post_parent ) ) . '">' . $guid . '</a>'
				)
			);
		}

		$post_title = preg_replace( '/\.[^.]+$/', '', $video_data['title'] );

		/**
		 *
		 * Filter post title
		 * 
		 */
		$post_title = apply_filters( 'streamtube/core/bunny/import/post_title', $post_title, $video_data );

		$post_author = get_current_user_id() ?? absint( $this->settings['auto_import_author_id'] );

		$maybe_author_id = $this->get_user_id_from_collection( $video_data['collectionId'] );

		if ( $maybe_author_id ) {
			$post_author = $maybe_author_id;
		}

		$attachment_id = wp_insert_post( array(
			'post_title'     => $post_title,
			'post_type'      => 'attachment',
			'post_mime_type' => $args['type'],
			'post_status'    => 'inherit',
			'post_author'    => $post_author,
			'meta_input'     => array(
				'_wp_attachment_metadata' => $this->_generate_attachment_metadata( array_merge( $video_data, array(
					'size' => $size
				) ) ),
				'_wp_attached_file'       => $video_data['guid'],
				'_bunnycdn'               => $video_data,
				'_bunnycdn_guid'          => $video_data['guid'],
				'_bunnycdn_status'        => $video_data['status'],
				'_bunnycdn_import'        => 'on'
			)
		), true );

		if ( is_wp_error( $attachment_id ) ) {
			return $attachment_id;
		}

		$post_status = $this->settings['auto_import_post_status'];

		if ( is_string( $status ) && in_array( $status, array_keys( get_post_stati() ) ) ) {
			$post_status = $status;
		}

		if ( $status === 'publish' && ! user_can( $post_author, get_post_type_object( 'video' )->cap->publish_posts ) ) {
			$post_status = 'pending';
		}

		$post_args = apply_filters( 'streamtube/core/bunny/import/post_args', array(
			'post_title'  => $post_title,
			'post_type'   => 'video',
			'post_status' => $post_status,
			'post_author' => $post_author,
			'meta_input'  => array(
				'video_url' => $attachment_id
			)
		), $attachment_id, $video_data );

		$post_id = wp_insert_post( $post_args, true );

		if ( is_wp_error( $post_id ) ) {
			wp_delete_attachment( $attachment_id, true );

			return $post_id;
		}

		if ( $include_tax ) {
			$taxonomies = get_object_taxonomies( 'video', 'object' );

			foreach ( $taxonomies as $tax => $object ) {
				if ( array_key_exists( 'auto_import_tax_' . $tax, $this->settings ) && $this->settings[ 'auto_import_tax_' . $tax ] ) {
					if ( is_taxonomy_hierarchical( $tax ) ) {
						$terms = array_map( 'absint', $this->settings[ 'auto_import_tax_' . $tax ] );
					} else {
						$terms = array_map( 'strval', $this->settings[ 'auto_import_tax_' . $tax ] );
					}

					if ( $terms ) {
						wp_set_post_terms( $post_id, $terms, $tax, true );
					}
				}
			}
		}

		wp_update_post( array(
			'ID'          => $attachment_id,
			'post_parent' => $post_id
		) );

		$this->update_thumbnail_images( $attachment_id, array(
			'Status'    => $video_data['status'],
			'VideoGuid' => $video_data['guid']
		), $video_data );

		/**
		 *
		 * Fires after post imported
		 *
		 * @param int $post_id
		 * @param int $attachment_id
		 * @param array $video_data
		 * 
		 */
		do_action( 'streamtube/core/bunny/imported/post', $post_id, $attachment_id, $video_data );

		return compact( 'post_id', 'attachment_id', 'video_data' );
	}

	public function ajax_import_bunny_video() {

		$data = wp_parse_args( $_POST, array(
			'guid' => ''
		) );

		extract( $data );

		check_ajax_referer( $guid );

		$attachment_id = $this->get_post_id_from_videoId( $data['guid'] );

		if ( $attachment_id ) {
			wp_send_json_error( new WP_Error(
				'existed',
				esc_html__( '%s was already imported.', 'streamtube-core' )
			) );
		}

		$response = $this->import_bunny_video( compact( 'guid' ) );

		if ( is_wp_error( $response ) ) {
			wp_send_json_error( $response );
		}

		wp_send_json_success( $response );
	}

	/**
	 *
	 * Import all available bunny videos
	 * 
	 */
	public function get_bunny_videos( $paged = 1, $per_page = 10 ) {

		$args = array(
			'page'         => absint( $paged ),
			'itemsPerPage' => absint( $per_page )
		);

		$response = $this->bunnyAPI->get_videos( $args );

		if ( is_wp_error( $response ) ) {
			return $response;
		}

		if ( absint( $response['totalItems'] ) == 0 ) {
			return new WP_Error(
				'empty',
				esc_html__( 'Library is empty', 'streamtube-core' )
			);
		}

		return $response;
	}

	/**
	 *
	 * AJAX import bunny videos
	 * 
	 */
	public function ajax_import_bunny_videos() {

		check_ajax_referer( 'import_all_bunny_videos' );

		$http_data = wp_parse_args( $_REQUEST, array(
			'paged'    => 1,
			'per_page' => 10,
			'guid'     => ''
		) );

		$response = array();
		$post     = array();

		extract( $http_data );

		if ( ! $guid ) {
			$response = $this->get_bunny_videos( $paged, $per_page );
		} else {

			$response = $this->import_bunny_video( array_merge( compact( 'guid' ), array(
				'include_tax'   => false,
				'attachment_id' => $this->get_post_id_from_videoId( $guid )
			) ) );

			if ( ! is_wp_error( $response ) ) {
				$post = array(
					'guid' => $guid,
					'url'  => get_permalink( $response['post_id'] )
				);
			}
		}

		if ( is_wp_error( $response ) ) {
			wp_send_json_error( $response );
		}

		wp_send_json_success( array_merge( (array) $response, array(
			'did_action' => $guid ? 'imported' : 'fetched',
			'post'       => $post
		) ) );
	}

	/**
	 *
	 * Delete given bunny collection
	 * 
	 * @param  string $collectionId
	 * 
	 */
	public function delete_bunny_collection( $collectionId = '' ) {

		if ( ! Streamtube_Core_BunnyCDN_Perms::delete_collections() ) {
			return new WP_Error(
				'no_permission',
				esc_html__( 'You do not have permission to delete this collection.', 'streamtube-core' )
			);
		}

		return $this->delete_user_collection( $collectionId );
	}

	/**
	 *
	 * AJAX delete bunny collecton
	 * 
	 * 
	 */
	public function ajax_delete_bunny_collection() {
		$http_data = wp_parse_args( $_REQUEST, array(
			'data' => ''
		) );

		extract( $http_data );

		check_ajax_referer( 'delete-' . $data );

		$response = $this->delete_bunny_collection( $data );

		if ( is_wp_error( $response ) ) {
			wp_send_json_error( $response );
		}

		wp_send_json_success( $response );
	}

	/**
	 *
	 * Auto publish video after encoding successfully
	 * 
	 * @param  $post
	 * @param  array $data webhook response
	 *
	 * @since 2.1
	 * 
	 */
	public function auto_publish_after_success_encoding( $attachment_id, $webhook_data, $video_data = array() ) {

		/**
		 * Return if Auto Publish is disabled
		 */
		if ( ! $this->settings['auto_publish'] ) {
			return $attachment_id;
		}

		$video_data = wp_parse_args( $video_data, array(
			'encodeProgress'       => '',
			'availableResolutions' => ''
		) );

		if ( in_array( $webhook_data['Status'], array( 3, 4 ) ) && ( (int) $video_data['encodeProgress'] == 100 || $video_data['availableResolutions'] ) ) {

			$attachment = get_post( $attachment_id );

			if ( $attachment->post_parent ) {

				wp_update_post( array(
					'ID'          => $attachment->post_parent,
					'post_status' => 'publish'
				) );

				if ( $this->settings['author_notify_publish'] ) {
					streamtube_core_notify_author_after_video_publish( $attachment->post_parent, array(
						'subject' => trim( $this->settings['author_notify_publish_subject'] ),
						'content' => trim( $this->settings['author_notify_publish_content'] )
					) );
				}

			}

			/**
			 *
			 * Fires after publishing video
			 *
			 * @param  int $post_id video id
			 * @param  array $webhook_data webhook response
			 *
			 * @since 2.1
			 * 
			 */
			do_action( 'streamtube/core/bunnycdn/auto_publish', $attachment->post_parent, $webhook_data, $video_data );
		}
	}

	/**
	 *
	 * Auto send notify to author after encoding failed
	 * 
	 * @param  $attachment_id
	 * @param  array $data webhook response
	 *
	 * @since 2.1
	 * 
	 */
	public function notify_author_after_encoding_failed( $attachment_id, $webhook_data, $video_data = array() ) {
		if ( $this->settings['author_notify_fail'] && $webhook_data['Status'] == 5 ) {
			streamtube_core_notify_author_after_video_encoding_failed( $attachment_id, array(
				'subject' => trim( $this->settings['author_notify_fail_subject'] ),
				'content' => trim( $this->settings['author_notify_fail_content'] )
			) );
		}
	}

	/**
	 *
	 * Filter Allow Formats
	 * 
	 * @param  array  $allow_formats
	 *
	 * @return array
	 * 
	 */
	public function filter_allow_formats( $allow_formats = array() ) {

		if ( ! empty( $this->settings['allow_formats'] ) && $this->is_enabled() ) {
			$_allow_formats = array_map( 'trim', explode( ',', $this->settings['allow_formats'] ) );

			if ( is_array( $_allow_formats ) ) {
				$allow_formats = array_merge( $allow_formats, $_allow_formats );

				$allow_formats = array_values( array_unique( $allow_formats ) );
			}
		}

		return $allow_formats;
	}

	/**
	 *
	 * Filter the better messages meta
	 * 
	 */
	public function filter_better_messages_rest_message_meta( $meta, $message_id, $thread_id, $content ) {

		if ( ! array_key_exists( 'files', $meta ) ) {
			return $meta;
		}

		if ( is_array( $meta['files'] ) ) {
			$files = $meta['files'];

			if ( ! is_array( $files ) ) {
				return $meta;
			}

			for ( $i = 0; $i < count( $files ); $i++ ) {

				if ( wp_attachment_is( 'video', $files[ $i ]['id'] ) || wp_attachment_is( 'audio', $files[ $i ]['id'] ) ) {

					if ( false != $videoId = $this->get_video_guid( $files[ $i ]['id'] ) ) {
						$files[ $i ]['url'] = $this->get_downloadable_url( $files[ $i ]['id'] );
					}
				}

			}

			$meta['files'] = $files;
		}

		return $meta;
	}

	/**
	 *
	 * Process webhook data
	 * 
	 * @since 2.1
	 * 
	 */
	public function _webhook_callback( $data ) {
		$data = wp_parse_args( json_decode( $data, true ), array(
			'VideoLibraryId' => '',
			'VideoGuid'      => '',
			'Status'         => 0
		) );

		$statuses = $this->bunnyAPI->get_webhook_video_statuses();

		$data['Status'] = (int) $data['Status'];

		$video_data = array();

		$attachment_id = $this->get_post_id_from_videoId( $data['VideoGuid'] );

		if ( ! $attachment_id && $this->settings['auto_import_uploads'] ) {
			$import = $this->import_bunny_video( array(
				'guid'        => $data['VideoGuid'],
				'include_tax' => true
			) );

			if ( is_wp_error( $import ) ) {
				return $import;
			}

			$attachment_id = $import['attachment_id'];
			$video_data    = $import['video_data'];
		}

		if ( $attachment_id ) {

			$file = get_attached_file( $attachment_id );

			if ( ! $video_data || is_wp_error( $video_data ) ) {
				// Get and update video data
				$video_data = $this->bunnyAPI->get_video( $data['VideoGuid'] );

				if ( is_wp_error( $video_data ) ) {
					return $video_data;
				} else {
					update_post_meta( $attachment_id, '_bunnycdn', $video_data );
					update_post_meta( $attachment_id, '_bunnycdn_status', $data['Status'] );
				}
			}

			if ( $file ) {
				// Update logs
				if ( $this->settings['tsp'] ) {
					$this->bunnyAPI->write_log_file( $file );
				}

				$this->bunnyAPI->write_log_file(
					$file,
					json_encode( $data ),
					sprintf(
						esc_html__( 'Webhook Request Status %s', 'streamtube-core' ),
						$statuses[ $data['Status'] ][0]
					)
				);
			}

			/**
			 *
			 * Fires once webhook updated
			 *
			 * @param $attachment_id
			 * @param array $data
			 * @param array $video_data
			 *
			 * @since 2.1
			 * 
			 */
			do_action( 'streamtube/core/bunny/webhook/update', $attachment_id, $data, $video_data );
		}
	}

	/**
	 *
	 * Process webhook data
	 * 
	 * @since 2.1
	 * 
	 */
	public function webhook_callback() {
		$request = wp_parse_args( $_GET, array(
			'webhook' => '',
			'key'     => ''
		) );

		if ( $request['webhook'] != 'bunnycdn' || $request['key'] != $this->settings['webhook_key'] ) {
			return;
		}

		if ( ! $this->is_enabled() ) {
			wp_send_json_error( 'Not Enabled' );
		}

		$data = file_get_contents( "php://input" );

		if ( $data ) {
			$this->_webhook_callback( $data );
		}

		wp_send_json_success( 'Webhook' );

	}

	/**
	 *
	 * The Video table
	 *
	 * @since 2.1
	 * 
	 */
	public function admin_post_table( $columns ) {

		if ( ! $this->is_enabled() ) {
			return $columns;
		}

		unset( $columns['date'] );

		$new_columns = array();

		if ( Streamtube_Core_BunnyCDN_Perms::view_stream() && $this->is_enabled() ) {
			$new_columns['bunnycdn_sync'] = esc_html__( 'Bunny', 'streamtube-core' );
		}

		$new_columns['date'] = esc_html__( 'Date', 'streamtube-core' );

		return array_merge( $columns, $new_columns );
	}

	/**
	 *
	 * The Video table
	 *
	 * @since 2.1
	 * 
	 */
	public function admin_post_table_columns( $column, $post_id ) {

		$attachment_id = get_post_meta( $post_id, 'video_url', true );

		switch ($column) {

			case 'bunnycdn_sync':

				if ( wp_attachment_is( 'video', $attachment_id ) || wp_attachment_is( 'audio', $attachment_id ) ) {
					load_template(
						plugin_dir_path( __FILE__ ) . 'admin/sync-control.php',
						false,
						compact( 'attachment_id' )
					);
				}
				break;

		}
	}

	/**
	 *
	 * The media table
	 * 
	 * @since  2.1
	 * 
	 */
	public function admin_media_table( $columns ) {

		if ( ! $this->is_enabled() ) {
			return $columns;
		}

		unset( $columns['date'] );

		$new_columns = array();

		if ( Streamtube_Core_BunnyCDN_Perms::view_stream() && $this->is_enabled() ) {
			$new_columns['bunnycdn_sync'] = esc_html__( 'Bunny Stream', 'streamtube-core' );
		}

		$new_columns['date'] = esc_html__( 'Date', 'streamtube-core' );

		return array_merge( $columns, $new_columns );
	}

	/**
	 *
	 * The media table
	 * 
	 * @since  2.1
	 * 
	 */
	public function admin_media_table_columns( $column, $post_id ) {

		switch ($column) {

			case 'bunnycdn_sync':

				$attachment_id = $post_id;

				if ( wp_attachment_is( 'video', $attachment_id ) || wp_attachment_is( 'audio', $attachment_id ) ) {
					load_template(
						plugin_dir_path( __FILE__ ) . 'admin/sync-control.php',
						false,
						compact( 'attachment_id' )
					);
				}
				break;

		}
	}

	/**
	 *
	 * Add Bulk actions
	 * 
	 * @return array
	 *
	 * @since 2.1
	 * 
	 */
	public function admin_bulk_actions( $bulk_actions ) {

		if ( ! $this->is_enabled() || ! Streamtube_Core_Permission::moderate_posts() ) {
			return $bulk_actions;
		}

		$bulk_actions = array_merge( $bulk_actions, array(
			'bulk_bunnycdn_sync'                => esc_html__( 'Sync with Bunny', 'streamtube-core' ),
			'bulk_bunnycdn_generate_image'      => esc_html__( 'Generate thumbnail image', 'streamtube-core' ),
			'bulk_bunnycdn_generate_webp_image' => esc_html__( 'Generate animated image', 'streamtube-core' )
		) );

		return $bulk_actions;
	}

	/**
	 *
	 * Bulk actions handler
	 * 
	 * @param  string $redirect_url
	 * @param  string $action
	 * @param  int $post_ids
	 *
	 * @since 2.1
	 * 
	 */
	public function admin_handle_bulk_actions( $redirect_url, $action, $post_ids ) {

		if ( ! $this->settings['is_connected'] ) {
			return $redirect_url;
		}

		$queued = array();

		$_post_ids = array();

		foreach ( $post_ids as $post_id ) {
			if ( get_post_type( $post_id ) == 'video' ) {
				$post_id = get_post_meta( $post_id, 'video_url', true );
			}

			$_post_ids[] = $post_id;
		}

		switch ($action) {
			case 'bulk_bunnycdn_sync':

				$is_bulk_sync_supported = $this->is_bulk_sync_supported();

				for ( $i = 0; $i < count( $_post_ids ); $i++ ) {

					if ( $is_bulk_sync_supported ) {

						$result = $this->retry_sync_video( $_post_ids[ $i ] );

						if ( ! is_wp_error( $result ) ) {
							$queued[] = $_post_ids[ $i ];
						}
					}
				}

				if ( count( $queued ) > 0 ) {
					$redirect_url = add_query_arg( array(
						$action => count( $queued )
					), $redirect_url );
				}

				if ( ! $is_bulk_sync_supported ) {
					$redirect_url = add_query_arg( array(
						$action => 'bulk_sync_not_supported',
						'ref'   => 'php_curl'
					), $redirect_url );
				}

				break;

			case 'bulk_bunnycdn_generate_image':
				for ( $i = 0; $i < count( $_post_ids ); $i++ ) {
					$result = $this->generate_thumbnail_image( $_post_ids[ $i ] );

					if ( ! is_wp_error( $result ) ) {
						$queued[] = $_post_ids[ $i ];
					}
				}

				if ( count( $queued ) > 0 ) {
					$redirect_url = add_query_arg( array(
						$action => count( $queued )
					), $redirect_url );
				}

				break;

			case 'bulk_bunnycdn_generate_webp_image':
				for ( $i = 0; $i < count( $_post_ids ); $i++ ) {
					$result = $this->generate_webp_image( $_post_ids[ $i ] );

					if ( ! is_wp_error( $result ) ) {
						$queued[] = $_post_ids[ $i ];
					}
				}

				if ( count( $queued ) > 0 ) {
					$redirect_url = add_query_arg( array(
						$action => count( $queued )
					), $redirect_url );
				}
				break;
		}

		return $redirect_url;
	}

	/**
	 *
	 * Show admin notice 
	 * 
	 * @since 2.1
	 */
	public function admin_handle_bulk_admin_notices() {
		if ( isset( $_REQUEST['bulk_bunnycdn_sync'] ) ) {

			if ( $_REQUEST['bulk_bunnycdn_sync'] == 'bulk_sync_not_supported' ) {
				printf(
					'<div class="notice notice-error"><p>%s</p></div>',
					sprintf(
						esc_html__( 'Bulk Sync is not supported since you have selected %s Sync Type from %s page', 'streamtube-core' ),
						'<strong>' . esc_html__( 'PHP Curl', 'streamtube-core' ) . '</strong>',
						'<strong><a href="' . esc_url( admin_url( 'options-general.php?page=sync-bunnycdn' ) ) . '">' . esc_html__( 'Settings', 'streamtube-core' ) . '</a></strong>',
					)
				);
			} else {
				echo '<div class="notice notice-success"><p>';
				$count = (int) $_REQUEST['bulk_bunnycdn_sync'];
				printf(
					_n(
						'%s has been queued for syncing onto Bunny CDN',
						'%s have been queued for syncing onto Bunny CDN',
						$count,
						'streamtube-core'
					),
					number_format_i18n( $count )
				);
				echo '</p></div>';
			}
		}
	}

	/**
	 * 
	 *
	 * Rest API generate thumbnail image
	 * 
	 * @param  int $thumbnail_id
	 * @param  int $attachment_id
	 * @return int
	 */
	public function rest_generate_thumbnail_image( $thumbnail_id = 0, $attachment_id = 0 ) {

		if ( ! $this->is_enabled() ) {
			return $thumbnail_id;
		}

		if ( ! $thumbnail_id || is_wp_error( $thumbnail_id ) ) {
			$thumbnail_id = $this->generate_thumbnail_image( $attachment_id );
		}

		return $thumbnail_id;
	}

	public function rest_generate_animated_image_from_file( $thumbnail_url, $attachment_id ) {

		if ( ! $this->is_enabled() ) {
			return $thumbnail_url;
		}

		if ( ! $thumbnail_url || is_wp_error( $thumbnail_url ) ) {
			$thumbnail_id = $this->generate_webp_image( $attachment_id );

			if ( is_int( $thumbnail_id ) ) {
				$thumbnail_url = wp_get_attachment_image_url( $thumbnail_id, 'large' );
			}
		}

		return $thumbnail_url;
	}

	/**
	 * Generates subtitles for a video via a REST API endpoint.
	 *
	 * This method checks if the feature is enabled and retrieves the video ID
	 * associated with the provided post ID. If the video ID is valid and a transcription
	 * is not already queued, it sends a request to the BunnyCDN API to transcribe the video
	 * in the specified language. A transient is set to prevent duplicate transcription requests.
	 *
	 * @param mixed           $response The current response, which may be modified.
	 * @param WP_REST_Request $request  The REST API request object containing parameters.
	 * 
	 * @return mixed|WP_Error The updated response or a WP_Error object if an error occurs.
	 *
	 * @throws WP_Error If the transcription is already queued or if the BunnyCDN API request fails.
	 */
	public function rest_generate_subtitles( $response, $request ) {

		if ( ! $this->is_enabled() || ! current_user_can( 'ai_generate_subtitles' ) ) {
			return $response;
		}

		/**
		 * Filters the BunnyCDN AI-generated subtitles process and performs additional checks.
		 *
		 * This code snippet applies a filter to allow further customization or validation
		 * of the BunnyCDN AI-generated subtitles process. If an error is detected during
		 * the process, it returns a WP_Error object with the corresponding error code.
		 *
		 * @param WP_Error $maybe_further_check An instance of WP_Error to hold any potential errors.
		 * @param mixed    $response            The response data from the BunnyCDN API.
		 * @param mixed    $request             The original request data sent to the BunnyCDN API.
		 *
		 * @return WP_Error Returns a WP_Error object if an error occurs and has an error code.
		 *                  Otherwise, the process continues without interruption.
		 */
		$maybe_further_check = new WP_Error();

		$maybe_further_check = apply_filters( 'streamtube/core/bunny/ai_generate_subtitles', $maybe_further_check, $response, $request );

		if ( is_wp_error( $maybe_further_check ) && $maybe_further_check->get_error_codes() ) {
			return $maybe_further_check;
		}

		$attachment_id = (int) get_post_meta( $request['post_id'], 'video_url', true );

		$videoId = $this->get_video_guid( $attachment_id );

		if ( ! $attachment_id || ! $videoId ) {
			return $response;
		}

		$language = $request['language'];

		if ( get_transient( "transcribe_queued_{$attachment_id}_{$language}" ) !== false ) {
			return new WP_Error(
				'transcription_queued',
				esc_html__( 'The transcription is in queue or already completed.', 'streamtube-core' )
			);
		}

		$response = $this->bunnyAPI->transcribe( $videoId, $language, 'true' );

		if ( is_wp_error( $response ) ) {
			return $response;
		}

		set_transient( "transcribe_queued_{$attachment_id}_{$language}", current_time( 'timestamp' ), 60 * 60 * 24 );

		return $response;
	}

	public function register_dashboard_menu( $menu_items ) {

		if ( ! $this->is_enabled() ) {
			return $menu_items;
		}

		$menu_items['bunny'] = array(
			'title'     => esc_html__( 'Bunny', 'streamtube-core' ),
			'alt_title' => esc_html__( 'Videos', 'streamtube-core' ),
			'desc'      => esc_html__( 'All videos', 'streamtube-core' ),
			'icon'      => 'dashicons dashicons-cloud-saved',
			'callback'  => function () {
				load_template( trailingslashit( plugin_dir_path( __FILE__ ) ) . 'admin/bulk-import.php' );
			},
			'cap'       => Streamtube_Core_BunnyCDN_Perms::get_capabilities()->list_videos,
			'parent'    => 'dashboard',
			'submenu'   => array(
				'collections' => array(
					'title'    => esc_html__( 'Collections', 'streamtube-core' ),
					'desc'     => esc_html__( 'All Collections', 'streamtube-core' ),
					'cap'      => Streamtube_Core_BunnyCDN_Perms::get_capabilities()->list_collections,
					'callback' => function () {
						load_template( trailingslashit( plugin_dir_path( __FILE__ ) ) . 'admin/collections.php' );
					},
					'priority' => 10,
					'icon'     => 'icon-folder-open'
				)
			),
			'priority'  => 50
		);

		return $menu_items;
	}

	/**
	 *
	 * Permission screen
	 * 
	 */
	public function register_permission_module() {
		streamtube_core_register_module(
			'bunnycdn',
			esc_html__( 'Bunny Stream', 'streamtube-core' ),
			(array) Streamtube_Core_BunnyCDN_Perms::get_capabilities()
		);
	}

	/**
	 *
	 * Load assets
	 * 
	 */
	public function enqueue_scripts() {

		if ( ! $this->is_enabled() ) {
			return;
		}

		if ( current_user_can( get_post_type_object( 'video' )->cap->create_posts ) ) {
			if ( $this->settings['upload_handler'] === 'resumable' && $this->settings['upload_type'] === 'auto' ) {
				wp_enqueue_script(
					'streamtube-core-bunny',
					plugin_dir_url( __FILE__ ) . 'assets/scripts.js',
					array( 'jquery', 'streamtube-core-scripts' ),
					filemtime( plugin_dir_path( __FILE__ ) . 'assets/scripts.js' ),
					true
				);
			}

			wp_enqueue_script(
				'streamtube-core-admin-bunny',
				plugin_dir_url( __FILE__ ) . 'assets/admin-scripts.js',
				array( 'jquery' ),
				filemtime( plugin_dir_path( __FILE__ ) . 'assets/admin-scripts.js' ),
				true
			);
		}
	}
}