<?php
/**
 * The table list class
 * 
 * @link       https://1.envato.market/mgXE4y
 * @since      2.1
 *
 * @package    WordPress
 * @subpackage StreamTube
 * @author     phpface <nttoanbrvt@gmail.com>
 */

if ( ! defined( 'ABSPATH' ) ) {
    exit;
}

if ( ! class_exists( 'WP_List_Table' ) ) {
    require_once ABSPATH . 'wp-admin/includes/class-wp-list-table.php';
}

class StreamTube_Core_Bunny_Videos_List_Table extends WP_List_Table {

    private $bunnycdn;

    public function __construct() {
        parent::__construct( array(
            'screen'    => 'edit-bunny-stream',
            'singular'  => esc_html__( 'Video', 'streamtube-core' ),
            'plural'    => esc_html__( 'Videos', 'streamtube-core' ),
            'ajax'      => true
        ) );

        $this->bunnycdn = streamtube_core()->get()->bunnycdn;
    }

    public function get_columns() {
        $columns = array(
            'cb'            => '<input type="checkbox" />',
            'title'         => esc_html__( 'Title', 'streamtube-core' ),
            'thumbnail'     => esc_html__( 'Thumbnail', 'streamtube-core' ),
            'guid'          => esc_html__( 'GUID', 'streamtube-core' ),
            'collectionId'  => esc_html__( 'Collection', 'streamtube-core' ),
            'category'      => esc_html__( 'Category', 'streamtube-core' ),
            'storageSize'   => esc_html__( 'Storage Size', 'streamtube-core' ),
            'dateUploaded'  => esc_html__( 'Date', 'streamtube-core' ),
            'action'        => esc_html__( 'Action', 'streamtube-core' )
        );
        return $columns;
    }

    /**
     * Define what data to show on each column of the table
     *
     * @param array $item        Data
     * @param string $column_name Current column name
     *
     * @return string
     */
    public function column_default( $item, $column_name ){
        switch( $column_name ) {
            case 'title':
                $output = sprintf(
                    '<p><a class="row-title" target="_blank" href="%s">%s</a></p>',
                    esc_url( $this->bunnycdn->bunnyAPI->get_direct_player( $item['videoLibraryId'], $item['guid'] ) ),
                    esc_html( $item['title'] )
                );
            break;

            case 'thumbnail':

                $output = sprintf(
                    '<div class="ratio ratio-16x9"><a target="_blank" href="%s"><img class="img-thumbnail" src="%s"></a></div>',
                    esc_url( $this->bunnycdn->bunnyAPI->get_direct_player( $item['videoLibraryId'], $item['guid'] ) ),
                    $this->bunnycdn->bunnyAPI->get_video_thumbnail_url( $item['guid'] )
                );

            break;

            case 'storageSize':
                $output = size_format( $item[ $column_name ] );
            break;

            case 'dateUploaded':
                $output = sprintf(
                    esc_html__( '%s ago', 'streamtube-core' ),
                    human_time_diff( strtotime( $item[ $column_name ] ), current_time( 'U' ) )
                );
            break;

            case 'action':
                if( "" != $attachment_id = $this->bunnycdn->get_post_id_from_videoId( $item['guid'] ) ){
                    $output = sprintf(
                        '<span class="badge bg-success"><a target="_blank" class="text-white" href="%s">%s</a></span>',
                        esc_url( get_permalink( get_post( $attachment_id )->post_parent ) ),
                        esc_html__( 'Imported', 'streamtube-core' )
                    );
                }else{

                    if( Streamtube_Core_BunnyCDN_Perms::import_videos() ){
                        $output = sprintf(
                            '<button class="btn btn-sm button button-small button-primary button-import-bunny-video" data-guid="%s" data-nonce="%s" type="button">%s</button>',
                            esc_attr( $item['guid'] ),
                            wp_create_nonce( $item['guid'] ),
                            esc_html__( 'Import', 'streamtube-core' )
                        );
                    }
                }

                if( Streamtube_Core_BunnyCDN_Perms::delete_videos() && ! is_admin() ){
                    $output .= sprintf(
                        '<button 
                            type="button" 
                            class="%s" 
                            data-nonce="%s" 
                            data-action="%s" 
                            data-params="%s"
                            data-confirm="%s"
                        >%s</button>',
                        esc_attr( 'btn btn-sm bg-danger button button-small text-white delete-bunny-video ajax-elm' ),
                        esc_attr( wp_create_nonce( 'delete-' . $item['guid'] ) ),
                        esc_attr( 'delete_bunny_video' ),
                        esc_attr( $item['guid'] ),
                        sprintf(
                            esc_attr__( 'Are you sure you want to permanently delete the %s video? This action will also delete the associated video post.', 'streamtube-core' ),
                            $item['title']
                        ),
                        esc_html__( 'Delete', 'streamtube-core' )
                    );
                }

                if( $output ){
                    $output = sprintf( '<div class="d-flex gap-3 align-items-start">%s</div>', $output );
                }
            break;

            case 'collectionId':
                $output = sprintf(
                    '<a class="text-body" href="%1$s">%2$s</a>',
                    esc_url( 
                        add_query_arg( 
                            array( $column_name => $item[ $column_name ] ),
                            remove_query_arg( array_keys( $_REQUEST ), $_SERVER['REQUEST_URI'] )
                        ) 
                    ),
                    $item[ $column_name ]
                );
            break;            

            default:
                $output = $item[ $column_name ];
            break;
        }

        return $output;
    }

    /**
     * Define which columns are hidden
     *
     * @return array
     */
    public function get_hidden_columns() {
        $hidden_columns = (array)get_user_option( 'manage' . $this->screen->id . 'columnshidden' );

        return is_array( $hidden_columns ) ? $hidden_columns : array();
    }

    protected function get_items_per_page( $option = '', $default_value = 20 ) {
        $per_page = (int)get_user_option( 'edit_'. $this->screen->id .'_per_page' );

        return $per_page !== 0 ? absint( $per_page ) : $default_value;
    }

    /**
     * Define the sortable columns
     *
     * @return array
     */
    public function get_sortable_columns() {
        return array(
            'title'         => array( 'title', false ),
            'dateUploaded'  => array( 'dateUploaded', false )
        );
    }

    public function get_bulk_actions(){

        $action = array();

        if( Streamtube_Core_BunnyCDN_Perms::import_videos() ){
            $action = array(
                'import'    =>  esc_html__( 'Import', 'streamtube-core' )
            );
        }
        
        return $action; 
    }    

    /**
     * Render the bulk edit checkbox
     *
     * @param array $item
     *
     * @return string
     */
    function column_cb( $item ) {
        return sprintf(
            '<input type="checkbox" name="guid[]" value="%s" />', $item['guid']
        );
    }    

    /**
     * Get the table data
     *
     * @return array
     */
    private function table_data() {
        
        $args = array(
            'search'    => isset( $_REQUEST['s'] ) ? wp_unslash( $_REQUEST['s'] ) : '',
            'orderBy'   => 'date'
        );

        if( preg_match( '/^[0-9a-f]{8}-[0-9a-f]{4}-[1-5][0-9a-f]{3}-[89ab][0-9a-f]{3}-[0-9a-f]{12}$/i', $args['search'] ) === 1 ){
            $args['collection'] = $args['search'];
            unset( $args['search'] );
        }

        if( isset( $_REQUEST['collectionId'] ) ){
            $args['collection'] = wp_unslash( $_REQUEST['collectionId'] );
        }

        $data = $this->bunnycdn->bunnyAPI->get_videos( $args );

        if( is_wp_error( $data ) ){
            return array();
        }

        return $data['items'];
    }

    /**
     * Sort data based on the selected column and order
     *
     * @param array $a
     * @param array $b
     * @return int
     */
    private function sort_data( $a, $b ) {
        $orderby = isset( $_GET['orderby'] ) && is_string( $_GET['orderby'] ) ? sanitize_text_field( $_GET['orderby'] ) : 'dateUploaded'; 
        $order   = isset( $_GET['order'] ) && is_string( $_GET['order'] ) ? sanitize_text_field( $_GET['order'] ) : 'desc';

        if ( ! is_string( $a[$orderby] ) || ! is_string( $b[$orderby] ) || ! isset( $a[$orderby] ) || ! isset( $b[$orderby] ) ) {
            return 0; // Handle cases where the sort column doesn't exist in the data
        }

        $result = strcmp( $a[$orderby], $b[$orderby] ); 

        return ( $order === 'asc' ) ? $result : -$result;
    }

    public function prepare_items() {
        $columns  = $this->get_columns();
        $hidden   = $this->get_hidden_columns();
        $sortable = $this->get_sortable_columns();

        $data = $this->table_data();
        usort( $data, array( $this, 'sort_data' ) );

        $perPage     = $this->get_items_per_page();
        $currentPage = $this->get_pagenum();
        $totalItems  = count( $data );

        $this->set_pagination_args( array(
            'total_items' => $totalItems,
            'per_page'    => $perPage
        ) );

        $data = array_slice( $data, ( ( $currentPage - 1 ) * $perPage ), $perPage );

        $this->_column_headers = array( $columns, $hidden, $sortable );
        $this->items = $data;

        $this->process_bulk_action();
    }

    public function process_bulk_action() {

        if( ! Streamtube_Core_BunnyCDN_Perms::import_videos() ){
            return;
        }

        $action = $this->current_action();

        $guids = isset( $_REQUEST['guid'] ) ? wp_unslash( $_REQUEST['guid'] ) : false;

        switch ( $action ) {
            case 'import':
                for ( $i=0;  $i < count( $guids );  $i++) { 

                    $attachment_id = $this->bunnycdn->get_post_id_from_videoId( $guids[$i] );

                    if( ! $attachment_id ){
                        $this->bunnycdn->import_bunny_video( 
                            $guids[$i], 
                            apply_filters( 'streamtube/core/bunny/bulk_import/tax', false ) 
                        );
                    }
                }
            break;
        }
    }
}