<?php
/**
 * Define the BunnyCDN Permissions functionality
 *
 * Provides methods to handle permissions related to BunnyCDN features.
 *
 * @link       https://themeforest.net/user/phpface
 * @since      2.1
 *
 * @package    Streamtube_Core
 * @subpackage Streamtube_Core/includes
 */

/**
 * BunnyCDN Permissions Class
 *
 * Handles permission checks for various BunnyCDN functionalities.
 *
 * @since      2.1
 * @package    Streamtube_Core
 * @subpackage Streamtube_Core/includes
 * @author     phpface <nttoanbrvt@gmail.com>
 */

// Exit if accessed directly
if ( ! defined( 'ABSPATH' ) ) {
    exit;
}

class Streamtube_Core_BunnyCDN_Perms {

    /**
     * Get BunnyCDN capabilities
     *
     * Returns an object with capabilities mapped to specific permission keys.
     *
     * @return object
     */
    public static function get_capabilities() {
        return (object) array(
            'view_encode_logs'      => 'bunnycdn_view_encode_logs',
            'view_stream'           => 'bunnycdn_view_stream',
            'list_videos'           => 'bunnycdn_list_videos',
            'import_videos'         => 'bunnycdn_import_videos',
            'delete_videos'         => 'bunnycdn_delete_videos',
            'list_collections'      => 'bunnycdn_list_collections',
            'delete_collections'    => 'bunnycdn_delete_collections',
            'manual_sync'           => 'bunnycdn_manual_sync'
        );
    }

    /**
     *
     * Need an extra capability to check if current user is moderator
     * 
     */
    private static function is_moderator(){
        $capability = apply_filters( 'streamtube/core/bunnycdn/moderator_capability', 'activate_plugins' );

        return current_user_can( $capability );
    }

    /**
     * Check permission to view encode logs
     *
     * @return bool
     */
    public static function view_encode_logs() {
        return current_user_can( self::get_capabilities()->view_encode_logs );
    }

    /**
     * Check permission to view the stream
     *
     * @return bool
     */
    public static function manual_sync() {
        return current_user_can( self::get_capabilities()->manual_sync );
    }    

    /**
     * Check permission to view the stream
     *
     * @return bool
     */
    public static function view_stream() {
        return current_user_can( self::get_capabilities()->view_stream );
    }

    /**
     * Check permission to list videos
     *
     * @return bool
     */
    public static function list_videos() {
        return current_user_can( self::get_capabilities()->list_videos );
    }

    /**
     * Check permission to import videos
     *
     * @return bool
     */
    public static function import_videos() {
        return current_user_can( self::get_capabilities()->import_videos ) && current_user_can( get_post_type_object( 'video' )->cap->create_posts );
    }

    /**
     * Check permission to delete videos
     *
     * @return bool
     */
    public static function delete_videos() {
        return current_user_can( self::get_capabilities()->delete_videos ) && self::is_moderator();
    }

    /**
     * Check permission to list collections
     *
     * @return bool
     */
    public static function list_collections() {
        return current_user_can( self::get_capabilities()->list_collections );
    }

    /**
     * Check permission to delete collections
     *
     * @return bool
     */
    public static function delete_collections() {
        return current_user_can( self::get_capabilities()->delete_collections ) && self::is_moderator();
    }
}
