<?php
/**
 * Define the Form Post functionality
 *
 *
 * @link       https://themeforest.net/user/phpface
 * @since      1.1
 *
 * @package    Streamtube_Core
 * @subpackage Streamtube_Core/includes
 */

/**
 *
 * @since      1.0.0
 * @package    Streamtube_Core
 * @subpackage Streamtube_Core/includes
 * @author     phpface <nttoanbrvt@gmail.com>
 */

if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

class StreamTube_Core_ACF_Form_Post extends ACF_Form_Post {

	function __construct() {
	}

	public function init() {

		// enqueue scripts
		acf_enqueue_scripts(
			array(
				'uploader' => true,
			)
		);

		acf_nonce_input( 'post' );
	}

	/**
	 * add_meta_boxes
	 *
	 * Adds ACF metaboxes for the given $post_type and $post.
	 *
	 * @date    19/9/18
	 * @since   5.7.6
	 *
	 * @param   string  $post_type The post type.
	 * @param   WP_Post $post      The post being edited.
	 * @return  void
	 */
	public function add_meta_boxes( $screen = '', $post = null ) {

		if ( ! $post instanceof WP_Post ) {
			return;
		}

		$screen = 'dashboard_' . $post->post_type;

		// Storage for localized postboxes.
		$postboxes = array();

		// Get field groups for this screen.
		$field_groups = acf_get_field_groups(
			array(
				'post_id'   => $post->ID,
				'post_type' => $post->post_type
			)
		);

		// Loop over field groups.
		if ( $field_groups ) {
			foreach ( $field_groups as $field_group ) {

				if ( isset( $field_group['ui_location'] ) && $field_group['ui_location'] === 'backend' ) {
					continue;
				}

				// vars
				$id       = "acf-{$field_group['key']}";          // acf-group_123
				$title    = $field_group['title'];             // Group 1
				$context  = $field_group['position'];        // normal, side, acf_after_title
				$priority = 'high';                         // high, core, default, low

				if ( $context === 'acf_after_title' ) {
					$context = 'advanced';
				}

				// Reduce priority for sidebar metaboxes for best position.
				if ( $context == 'side' ) {
					$priority = 'core';
				}

				/**
				 * Filters the metabox priority.
				 *
				 * @date    23/06/12
				 * @since   3.1.8
				 *
				 * @param   string $priority The metabox priority (high, core, default, low).
				 * @param   array $field_group The field group array.
				 */
				$priority = apply_filters( 'acf/input/meta_box_priority', $priority, $field_group );

				// Localize data
				$postboxes[] = array(
					'id'    => $id,
					'key'   => $field_group['key'],
					'style' => $field_group['style'],
					'label' => $field_group['label_placement'],
					'edit'  => acf_get_field_group_edit_link( $field_group['ID'] ),
				);

				// Add the meta box.
				streamtube_add_meta_box(
					$id,
					acf_esc_html( $title ),
					array( $this, 'render_meta_box' ),
					$screen,
					$context,
					$priority,
					array( 'field_group' => $field_group )
				);
			}

			// Set style from first field group.
			$this->style = acf_get_field_group_style( $field_groups[0] );

			// Localize postboxes.
			acf_localize_data(
				array(
					'postboxes' => $postboxes,
				)
			);
		}

		// remove postcustom metabox (removes expensive SQL query)
		if ( acf_get_setting( 'remove_wp_meta_box' ) ) {
			remove_meta_box( 'postcustom', false, 'normal' );
		}

		// Add hidden input fields.
		add_action( 'edit_form_after_title', array( $this, 'edit_form_after_title' ) );

		/**
		 * Fires after metaboxes have been added.
		 *
		 * @date    13/12/18
		 * @since   5.8.0
		 *
		 * @param   string $post_type The post type.
		 * @param   WP_Post $post The post being edited.
		 * @param   array $field_groups The field groups added.
		 */
		do_action( 'acf/add_meta_boxes', $screen, $post, $field_groups );
	}

	/**
	 *
	 * Do not display metabox if its UI location is Frontend
	 * 
	 */
	public function remove_admin_meta_boxes( $post_type, $post ) {

		if ( ! is_admin() ) {
			return;
		}

		$field_groups = acf_get_field_groups(
			array(
				'post_id'   => $post->ID,
				'post_type' => $post->post_type
			)
		);

		if ( $field_groups ) {
			foreach ( $field_groups as $field_group ) {

				if ( isset( $field_group['ui_location'] ) && $field_group['ui_location'] === 'frontend' ) {
					remove_meta_box( "acf-{$field_group['key']}", $post->post_type, $field_group['position'] );
				}
			}
		}
	}

	public function render_additional_group_settings_tabs( $field_group = array() ) {
		acf_render_field_wrap(
			array(
				'label'         => __( 'UI Location', 'streamtube-core' ),
				'instructions'  => __( 'Where does this group appear in the UI?', 'streamtube-core' ),
				'type'          => 'button_group',
				'name'          => 'ui_location',
				'prefix'        => 'acf_field_group',
				'value'         => isset( $field_group['ui_location'] ) ? $field_group['ui_location'] : 'both',
				'choices'       => array(
					'backend'  => __( 'Backend', 'streamtube-core' ),
					'frontend' => __( 'Frontend (User Dashboard)', 'streamtube-core' ),
					'both'     => __( 'Both', 'streamtube-core' )
				),
				'default_value' => 'both'
			),
			'div',
			'field'
		);
	}

	/**
	 *
	 * Add Field Group additional setting tab
	 * 
	 */
	public function additional_group_settings_tabs( $tabs = array() ) {
		$tabs['additional'] = esc_html__( 'Additional', 'streamtube-core' );

		return $tabs;
	}

	/**
	 *
	 * Add additional settings to custom post type
	 * 
	 */
	public function filter_custom_post_type_settings( $args, $post ) {

		if ( isset( $post['show_ui_frontend'] ) && wp_validate_boolean( $post['show_ui_frontend'] ) ) {
			$args['show_ui_frontend'] = $post['show_ui_frontend'];
		}
		return $args;
	}

	/**
	 *
	 * Add `Show UI Frontend` setting
	 * And maybe other settings in future update
	 * 
	 */
	public function add_custom_post_type_settings( $acf_post_type ) {
		acf_render_field_wrap(
			array(
				'type'         => 'true_false',
				'name'         => 'show_ui_frontend',
				'key'          => 'show_ui_frontend',
				'prefix'       => 'acf_post_type',
				'value'        => $acf_post_type['show_ui_frontend'],
				'label'        => esc_html__( 'Show UI Frontend', 'streamtube-core' ),
				'instructions' => esc_html__( 'Visible on the frontend user dashboard.', 'streamtube-core' ),
				'ui'           => true,
				'default'      => 1
			),
			'div'
		);
	}

	/**
	 * Filter custom taxonomy settings.
	 *
	 * This function adds additional settings to a custom taxonomy based on the provided post data.
	 *
	 * @param array $args The original arguments for the custom taxonomy.
	 * @param array $post The post data containing additional settings.
	 * @return array The modified arguments for the custom taxonomy.
	 */
	public function filter_custom_taxonomy_settings( $args, $post ) {
		if ( isset( $post['show_ui_frontend'] ) && wp_validate_boolean( $post['show_ui_frontend'] ) ) {
			$args['show_ui_frontend'] = $post['show_ui_frontend'];
		}
		if ( isset( $post['show_in_filter'] ) && wp_validate_boolean( $post['show_in_filter'] ) ) {
			$args['show_in_filter'] = $post['show_in_filter'];
		}
		if ( isset( $post['show_in_filter_cap'] ) ) {
			$args['show_in_filter_cap'] = $post['show_in_filter_cap'];
		}

		$args['builtby'] = 'acf';

		return $args;
	}

	/**
	 *
	 * Add `Show UI Frontend` setting
	 * And maybe other settings in future update
	 * 
	 */
	public function add_custom_taxonomy_settings( $acf_taxonomy ) {
		acf_render_field_wrap(
			array(
				'type'         => 'true_false',
				'key'          => 'show_ui_frontend',
				'name'         => 'show_ui_frontend',
				'prefix'       => 'acf_taxonomy',
				'value'        => isset( $acf_taxonomy['show_ui_frontend'] ) ? $acf_taxonomy['show_ui_frontend'] : '',
				'label'        => esc_html__( 'Show UI Frontend', 'streamtube-core' ),
				'instructions' => esc_html__( 'Makes a taxonomy visible on the frontend user dashboard.', 'streamtube-core' ),
				'ui'           => true,
				'default'      => 1
			)
		);

		acf_render_field_wrap(
			array(
				'type'    => 'true_false',
				'key'     => 'show_in_filter',
				'name'    => 'show_in_filter',
				'prefix'  => 'acf_taxonomy',
				'value'   => isset( $acf_taxonomy['show_in_filter'] ) ? $acf_taxonomy['show_in_filter'] : '',
				'label'   => esc_html__( 'Show UI Filter', 'streamtube-core' ),
				'ui'      => true,
				'default' => 1
			)
		);

		acf_render_field_wrap(
			array(
				'type'         => 'text',
				'key'          => 'show_in_filter_cap',
				'name'         => 'show_in_filter_cap',
				'prefix'       => 'acf_taxonomy',
				'value'        => isset( $acf_taxonomy['show_in_filter_cap'] ) ? $acf_taxonomy['show_in_filter_cap'] : '',
				'label'        => esc_html__( 'Show UI Filter Capability', 'streamtube-core' ),
				'instructions' => esc_html__( 'The specific capability required to use the filters. If left empty, the "assign_terms" alias will be utilized.', 'streamtube-core' ),
				'ui'           => true,
				'default'      => 'read'
			)
		);
	}
}