<?php
if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

/**
 *
 * Get post statuses
 * 
 * @param  int|WP_Post $post
 * @return array
 */
function streamtube_core_get_post_statuses( $post = null ) {

	if ( is_int( $post ) ) {
		$post = get_post( $post );
	}

	$statuses = array_merge( get_post_statuses(), array(
		'unlist'  => esc_html__( 'Unlist', 'streamtube-core' ),
		'publish' => esc_html__( 'Publish', 'streamtube-core' )
	) );

	$post_type_object = get_post_type_object( $post->post_type );

	if ( current_user_can( $post_type_object->cap->edit_others_posts ) ) {
		$statuses['reject'] = esc_html__( 'Reject', 'streamtube-core' );
	}

	if ( ! current_user_can( $post_type_object->cap->publish_posts ) ) {
		unset( $statuses['publish'] );
		unset( $statuses['private'] );
	}

	return apply_filters( 'streamtube_core_get_post_statuses', $statuses, $post );
}

/**
 *
 * Get post visibility in human-readable format
 * 
 * @param  int|WP_Post $post Post ID or WP_Post object.
 * @return string            Post visibility (e.g., 'Published', 'Private', 'Password Protected').
 * 
 */
function streamtube_core_get_post_visibility( $post = null ) {
	// Retrieve the post object if only an ID is provided
	if ( is_int( $post ) ) {
		$post = get_post( $post );
	}

	// Ensure the post exists
	if ( ! $post ) {
		return '';
	}

	// Get all post statuses with visibility in the admin list
	$statuses = get_post_stati( array( 'show_in_admin_all_list' => true ), 'object' );

	// Check if the post status exists in the statuses list, or use the raw status
	$status_label = array_key_exists( $post->post_status, $statuses )
		? esc_html( $statuses[ $post->post_status ]->label )
		: esc_html( ucfirst( $post->post_status ) );

	// Return the formatted visibility badge
	return $status_label;
}

/**
 *
 * Get the Add new post url
 * 
 * @return string
 */
function streamtube_get_add_new_post_url( $post_type = '' ) {

	if ( ! is_user_logged_in() ) {
		return;
	}

	global $post_type_screen;

	if ( ! $post_type && $post_type_screen ) {
		$post_type = $post_type_screen;
	}

	if ( ! $post_type || ! post_type_exists( $post_type ) ) {
		return;
	}

	$base = sprintf(
		'%s/dashboard/%s',
		untrailingslashit( get_author_posts_url( get_current_user_id() ) ),
		$post_type
	);

	$url = add_query_arg( array(
		'view'  => 'add-post',
		'nonce' => wp_create_nonce( 'add-new-' . $post_type )
	), $base );

	return apply_filters( 'streamtube_get_add_new_post_url', $url, $post_type );
}

/**
 *
 * Get text tracks
 * 
 * @param  int $post_id
 * @return array|false
 */
function streamtube_core_get_text_tracks( $post_id ) {
	return streamtube_core()->get()->post->get_text_tracks( $post_id );
}

/**
 *
 * Check if current is edit post screen
 * 
 * @return true|false
 *
 * @since  1.0.0
 * 
 */
function streamtube_core_is_edit_post_screen() {
	return streamtube_core()->get()->post->is_edit_post_screen();
}

/**
 *
 * Get edit post URL
 * 
 * @param  string $endpoint
 * @return string
 *
 * @since  1.0.0
 * 
 */
function streamtube_core_get_edit_post_url( $post_id = 0, $endpoint = '' ) {
	return streamtube_core()->get()->post->get_edit_post_url( $post_id, $endpoint );
}

/**
 *
 * The Edit Post form
 * 
 * @param  WP_Post $post
 * 
 */
function streamtube_core_the_post_form( $post ) {
	ob_start();

	$GLOBALS['post_type_screen'] = $post->post_type;

	setup_postdata( $GLOBALS['post'] =& $post );

	?>
	<div class="row">
		<div class="col-12 col-lg-8 col-xl-8 col-xxl-8">
			<div id="postbox-container-1" class="postbox-container postbox-ajax w-100">
				<?php streamtube_core_load_template( 'post/edit/main.php', false ); ?>
			</div>
		</div>
		<div class="col-12 col-lg-4 col-xl-4 col-xxl-4">
			<div id="postbox-container-2" class="postbox-container postbox-ajax w-100">
				<?php streamtube_core_load_template( 'post/edit/metaboxes.php' ); ?>
			</div>
		</div><!--.col-3-->
	</div>
	<?php

	wp_reset_postdata();

	return ob_get_clean();
}
