<?php
/**
 * Define the custom flat posts elementor functionality
 *
 *
 * @link       https://themeforest.net/user/phpface
 * @since      1.0.0
 *
 * @package    Streamtube_Core
 * @subpackage Streamtube_Core/includes
 */

/**
 *
 * @since      1.0.0
 * @package    Streamtube_Core
 * @subpackage Streamtube_Core/includes
 * @author     phpface <nttoanbrvt@gmail.com>
 */
if( ! defined('ABSPATH' ) ){
    exit;
}

class Streamtube_Core_Flat_Posts_Elementor extends \Elementor\Widget_Base{

    public function get_name(){
        return 'flat-posts';
    }

    public function get_title(){
        return esc_html__( 'Tiled Posts', 'streamtube-core' );
    }

    public function get_icon(){
        return 'eicon-video-playlist';
    }

    public function get_keywords(){
        return array( 'streamtube', 'posts', 'flat', 'tiled' );
    }

    public function get_categories(){
        return array( 'streamtube' );
    }

    /**
     *
     * Get default supported post types
     * 
     * @return array
     *
     * @since  1.0.0
     * 
     */
    private function get_post_types(){
        $r = array(
            'post'      =>  esc_html__( 'Post', 'streamtube-core' ),
            'video'     =>  esc_html__( 'Video', 'streamtube-core' )
        );

        return $r;
    }    

    private function get_term_options( $terms ){
        $options = array();

        if( ! $terms ){
            return $options;
        }

        foreach( $terms as $term ){
            $options[ $term->slug ] = $term->name;
        }

        return $options;
    }

    protected function register_controls(){

        $this->start_controls_section(
            'section-appearance',
            array(
                'label'     =>  esc_html__( 'Appearance', 'streamtube-core' ),
                'tab'       =>  \Elementor\Controls_Manager::TAB_CONTENT
            )
        );

             $this->add_control(
                'thumbnail_size',
                array(
                    'label'     =>  esc_html__( 'Thumbnail Image Size', 'streamtube-core' ),
                    'type'      =>  \Elementor\Controls_Manager::SELECT,
                    'default'   =>  'large',
                    'options'   =>  streamtube_core_get_thumbnail_sizes()
                )
            );       

            $this->add_control(
                'post_categories',
                array(
                    'label'     =>  esc_html__( 'Show post categories', 'streamtube-core' ),
                    'type'      =>  \Elementor\Controls_Manager::SWITCHER,
                    'default'   =>  'yes'
                )
            );

            $this->add_control(
                'show_post_date',
                array(
                    'label'     =>  esc_html__( 'Show post date', 'streamtube-core' ),
                    'type'      =>  \Elementor\Controls_Manager::SELECT,
                    'default'   =>  'normal',
                    'options'   => array(
                        ''          =>  esc_html__( 'None', 'streamtube-core' ),
                        'normal'    =>  esc_html__( 'Normal', 'streamtube-core' ),
                        'diff'      =>  esc_html__( 'Diff', 'streamtube-core' ),
                    )
                )
            );

            $this->add_control(
                'show_post_comment',
                array(
                    'label'     =>  esc_html__( 'Show post comment', 'streamtube-core' ),
                    'type'      =>  \Elementor\Controls_Manager::SWITCHER,
                    'default'   =>  ''
                )
            );
     
            $this->add_control(
                'show_author_name',
                array(
                    'label'     =>  esc_html__( 'Show post author name', 'streamtube-core' ),
                    'type'      =>  \Elementor\Controls_Manager::SWITCHER,
                    'default'   =>  'yes'
                )
            );

            $this->add_control(
                'author_avatar',
                array(
                    'label'     =>  esc_html__( 'Show post author avatar', 'streamtube-core' ),
                    'type'      =>  \Elementor\Controls_Manager::SWITCHER,
                    'default'   =>  ''
                )
            );

            $this->add_control(
                'avatar_size',
                array(
                    'label'     =>  esc_html__( 'Avatar size', 'streamtube-core' ),
                    'type'      =>  \Elementor\Controls_Manager::SELECT,
                    'default'   =>  'md',
                    'options'   =>  array(
                        'sm'    =>  esc_html__( 'Small', 'streamtube-core' ),
                        'md'    =>  esc_html__( 'Medium', 'streamtube-core' ),
                        'lg'    =>  esc_html__( 'Large', 'streamtube-core' )
                    ),
                    'condition' =>  array(
                        'author_avatar' =>  'yes'
                    )
                )
            );    

        $this->end_controls_section();

        $this->start_controls_section(
            'section-datasource',
            array(
                'label'     =>  esc_html__( 'Data Source', 'streamtube-core' ),
                'tab'       =>  \Elementor\Controls_Manager::TAB_CONTENT
            )
        );

           $this->add_control(
                'upcoming_posts',
                array(
                    'label'     =>  esc_html__( 'Upcoming', 'streamtube-core' ),
                    'type'      =>  \Elementor\Controls_Manager::SWITCHER,
                    'default'   =>  '',
                    'description'   =>  esc_html__( 'Retrieve upcoming posts.', 'streamtube-core' )
                )
            );        

           $this->add_control(
                'live_stream',
                array(
                    'label'     =>  esc_html__( 'Live Stream', 'streamtube-core' ),
                    'type'      =>  \Elementor\Controls_Manager::SWITCHER,
                    'default'   =>  '',
                    'description'   =>  esc_html__( 'Retrieve Live Streams', 'streamtube-core' )
                )
            );

           $this->add_control(
                'live_status',
                array(
                    'label'     =>  esc_html__( 'Live Status', 'streamtube-core' ),
                    'type'      =>  \Elementor\Controls_Manager::SELECT2,
                    'default'   =>  array( 'connected' ),
                    'condition' =>  array(
                        'live_stream' =>  'yes'
                    ),                    
                    'multiple'  =>  true,
                    'options'   =>  Streamtube_Core_Widget_Posts::get_live_statuses()
                )
            );              

            $this->add_control(
                'post_type',
                array(
                    'label'     =>  esc_html__( 'Post Type', 'streamtube-core' ),
                    'type'      =>  \Elementor\Controls_Manager::SELECT,
                    'default'   =>  'video',
                    'options'   =>  $this->get_post_types()
                )
            );

            foreach( $this->get_post_types() as $post_type => $post_type_label ){
                if( is_post_type_viewable( $post_type )){
                    $taxonomies = get_object_taxonomies( $post_type, 'object' );

                    if( $taxonomies ){

                        foreach ( $taxonomies as $tax => $object ){

                            $terms = get_terms( array(
                                'taxonomy'      =>  $tax,
                                'hide_empty'    =>  false
                            ) );                            

                            $this->add_control(
                                'tax_query_' . $tax,
                                array(
                                    'label'     =>  sprintf( esc_html__( 'Inc %s', 'streamtube-core' ), $object->label ),
                                    'type'      =>  \Elementor\Controls_Manager::SELECT2,
                                    'multiple'  =>  true,
                                    'default'   =>  '',
                                    'condition' =>  array(
                                        'post_type' =>  $post_type
                                    ),
                                    'options'   =>  $this->get_term_options( $terms ),
                                    'description'   =>  sprintf( esc_html__( 'Include %s terms', 'streamtube-core' ), $object->label )
                                )
                            );

                            $this->add_control(
                                'exclude_tax_query_' . $tax,
                                array(
                                    'label'     =>  sprintf( esc_html__( 'Exc %s', 'streamtube-core' ), $object->label ),
                                    'type'      =>  \Elementor\Controls_Manager::SELECT2,
                                    'multiple'  =>  true,
                                    'default'   =>  '',
                                    'condition' =>  array(
                                        'post_type' =>  $post_type
                                    ),
                                    'options'   =>  $this->get_term_options( $terms ),
                                    'description'   =>  sprintf( esc_html__( 'Exclude %s terms', 'streamtube-core' ), $object->label )
                                )
                            );                            

                        }
                    }
                }
            }

            $this->add_control(
                'search',
                array(
                    'label'     =>  esc_html__( 'Keyword', 'streamtube-core' ),
                    'type'      =>  \Elementor\Controls_Manager::TEXT,
                    'default'   =>  '',
                    'description'   =>  esc_html__( 'Show posts based on a keyword search', 'streamtube-core' )
                )
            );

            $this->add_control(
                'author__in',
                array(
                    'label'     =>  esc_html__( 'Authors', 'streamtube-core' ),
                    'type'      =>  \Elementor\Controls_Manager::TEXT,
                    'default'   =>  '',
                    'description'   =>  esc_html__( 'Retrieve posts of given author IDs, separated by commas', 'streamtube-core' )
                )
            );            

            $this->add_control(
                'verified_users_only',
                array(
                    'label'     =>  esc_html__( 'Verified Users', 'streamtube-core' ),
                    'type'      =>  \Elementor\Controls_Manager::SWITCHER,
                    'default'   =>  '',
                    'description'   =>  esc_html__( 'Retrieve posts of verified users only', 'streamtube-core' )
                )
            );             
          
        $this->end_controls_section();

        $this->start_controls_section(
            'section-order',
            array(
                'label'     =>  esc_html__( 'Order', 'streamtube-core' ),
                'tab'       =>  \Elementor\Controls_Manager::TAB_CONTENT
            )
        );

            $this->add_control(
                'orderby',
                array(
                    'label'     =>  esc_html__( 'Order by', 'streamtube-core' ),
                    'type'      =>  \Elementor\Controls_Manager::SELECT,
                    'default'   =>  'date',
                    'options'   =>  streamtube_core_get_orderby_options()
                )
            );

            $this->add_control(
                'order',
                array(
                    'label'     =>  esc_html__( 'Order', 'streamtube-core' ),
                    'type'      =>  \Elementor\Controls_Manager::SELECT,
                    'default'   =>  'DESC',
                    'options'   =>  array(
                        'ASC'               =>  esc_html__( 'Ascending', 'streamtube-core' ),
                        'DESC'              =>  esc_html__( 'Descending (default).', 'streamtube-core' )
                    )
                )
            );               

        $this->end_controls_section();                

        if( function_exists( 'WP_Statistics' ) && function_exists( 'streamtube_core_get_statistics_metrics' ) ) :
            $this->start_controls_section(
                'section-statistics',
                array(
                    'label'     =>  esc_html__( 'Statistics', 'streamtube-core' ),
                    'tab'       =>  \Elementor\Controls_Manager::TAB_CONTENT
                )
            );

                $this->add_control(
                    'statistics_notice',
                    array(
                        'notice_type'   =>  'info',
                        'type'          =>  \Elementor\Controls_Manager::NOTICE,                    
                        'content'       =>  esc_html__( 'These options are utilized in combination with the \'Order by number of views\' functionality.', 'streamtube-core' )
                    )
                );        

                $this->add_control(
                    'statistics_from',
                    array(
                        'label'         =>  esc_html__( 'From Date', 'streamtube-core' ),
                        'type'          =>  \Elementor\Controls_Manager::DATE_TIME,
                        'default'       =>  '',
                        'description'   =>  esc_html__( 'The starting date to retrieve posts based on view count (hits).', 'streamtube-core' )
                    )
                );

                $this->add_control(
                    'statistics_to',
                    array(
                        'label'         =>  esc_html__( 'To Date', 'streamtube-core' ),
                        'type'          =>  \Elementor\Controls_Manager::DATE_TIME,
                        'default'       =>  '',
                        'description'   =>  esc_html__( 'The end date for retrieving posts based on view count (hits).', 'streamtube-core' )
                    )
                );

                $this->add_control(
                    'statistics_date',
                    array(
                        'label'         =>  esc_html__( 'Date Range', 'streamtube-core' ),
                        'type'          =>  \Elementor\Controls_Manager::TEXT,
                        'default'       =>  '',
                        'description'   =>  sprintf(
                            esc_html__( 'The date range to retrieve posts from, defined values are: %s', 'streamtube-core' ),
                            '<strong>'. join(', ', array_keys( Streamtube_Core_Widget_Filter_Post_Date::get_options() ) ) .'</strong>'
                        )
                    )
                );

                $this->add_control(
                    'statistics_metric',
                    array(
                        'label'         =>  esc_html__( 'Metric', 'streamtube-core' ),
                        'type'          =>  \Elementor\Controls_Manager::SELECT2,
                        'default'       =>  'default',
                        'options'       =>  array_merge(
                            array(
                                'default'  =>  esc_html__( 'Default', 'streamtube-core' )
                            ),
                            streamtube_core_get_statistics_metrics()
                        )
                    )
                );                 

                $this->add_control(
                    'statistics_hits_range',
                    array(
                        'label'         =>  esc_html__( 'Post Views', 'streamtube-core' ),
                        'type'          =>  \Elementor\Controls_Manager::SWITCHER,
                        'default'       =>  '',
                        'description'   =>  esc_html__( 'Display post views based on the configured date range, otherwise, all-time views will be shown.', 'streamtube-core' )
                    )
                );                                                    
            $this->end_controls_section();
        endif;        
    }

    protected function content_template() {}

    public function render_plain_content( $instance = array() ) {}

    protected function render(){

        global $displayed_post_ids, $displayed_author_ids;

        if( ! $displayed_post_ids ){
            $displayed_post_ids = array();
        }

        if( ! $displayed_author_ids ){
            $displayed_author_ids = array();
        }

        $settings = array_merge( $this->get_settings_for_display(), array(
            'widget_id'         => $this->get_name(),
            'show_post_view'    =>  apply_filters( 'streamtube_has_pageviews', false ) === true ? 'on' : ''
        ) );

        $settings = wp_parse_args( $settings, array(
            'upcoming_posts'    =>  '',
            'live_stream'       =>  '',
            'live_status'       =>  array( 'connected' ),
            'statistics_from'   =>  '',
            'statistics_to'     =>  '',
            'statistics_date'   =>  ''
        ) );

        $GLOBALS['widget_instance'] = $settings;

        $query_args = array(
            'post_type'         =>  $settings['post_type'],
            'post_status'       =>  'publish',
            'posts_per_page'    =>  4,
            's'                 =>  $settings['search'],
            'orderby'           =>  $settings['orderby'],
            'order'             =>  $settings['order'],
            'meta_query'        =>  array(),
            'author__in'        =>  array(),
            'meta_query'        =>  array()
        ); 

        if( $settings['upcoming_posts'] ){
            $query_args['meta_query'][] = array(
                'key'       =>  '_upcoming_date',
                'compare'   =>  '>',
                'value'     =>  current_datetime()->format('Y-m-d H:i:s'),
                'type'      =>  'DATETIME'
            );
        }

        if( $settings['live_stream'] ){
            if( is_string( $settings['live_status'] ) ){
                $settings['live_status'] = array_map( 'trim', explode( ',', $settings['live_status'] ) );
            }

            $query_args['meta_query'][] = array(
                'key'       =>  'live_status',
                'compare'   =>  'IN',
                'value'     =>  $settings['live_status']
            );            
        }

        if( $query_args['post_type'] == 'video' ){
            $query_args['meta_query'][] = array(
                'key'       =>  '_thumbnail_id',
                'compare'   =>  'EXISTS'
            );        
        }

        // Set taxonomies
        $taxonomies = get_object_taxonomies( $query_args['post_type'], 'object' );

        if( $taxonomies ){

            $tax_query = array();

            foreach ( $taxonomies as $tax => $object ) {
                
                if( array_key_exists( 'tax_query_' . $tax , $settings ) && $settings[ 'tax_query_' . $tax ] ){
                    $tax_query[] = array(
                        'taxonomy'  =>  $tax,
                        'field'     =>  'slug',
                        'terms'     =>  (array)$settings[ 'tax_query_' . $tax ]
                    );
                }

                if( array_key_exists( 'exclude_tax_query_' . $tax , $settings ) && $settings[ 'exclude_tax_query_' . $tax ] ){
                    $tax_query[] = array(
                        'taxonomy'  =>  $tax,
                        'field'     =>  'slug',
                        'terms'     =>  (array)$settings[ 'exclude_tax_query_' . $tax ],
                        'operator'  =>  'NOT IN'
                    );
                }                
            }

            if( $tax_query ){
                $query_args['tax_query'] = $tax_query;
            }
        }


        // Set orderby
        if( $query_args['orderby'] == 'post_view' ){
            $query_args['orderby_post_view']    = true;
            $query_args['meta_key']             = streamtube_core()->get()->post->get_post_views_meta();
            $query_args['orderby']              = 'meta_value_num';
        }

        if( $query_args['orderby'] == 'post_like' ){
            $query_args['meta_key'] = '_like_count';
            $query_args['orderby']  = 'meta_value_num';
        }

        if( array_key_exists( 'author__in', $settings ) && $settings['author__in'] ){
            $author__in = array_map( 'trim' , explode( ',' , $settings['author__in'] ));

            if( $author__in ){
                $query_args['author__in'] = $author__in;
            }
        }

        if( array_key_exists( 'verified_users_only', $settings ) && $settings['verified_users_only'] ){
            $query_args['author__in'] = array_merge( $query_args['author__in'], Streamtube_Core_User::get_users( array(
                'role__in'    =>  array( Streamtube_Core_Permission::ROLE_VERIFY )
            ) ) );
        }

        /**
         *
         * Filter the post query args
         *
         * @param  array $query_args
         * @param  array $instance
         *
         * @since  1.0.0
         * 
         */
        $query_args = apply_filters( 'streamtube/core/widget/posts_flat/query_args', $query_args, $settings );        

        /**
         *
         * Fires before sending query
         * 
         */
        do_action( 'streamtube/core/widget/posts_flat/query/before', $query_args, $settings );

        $post_query = new WP_Query( $query_args );

        /**
         *
         * Fires before sending query
         * 
         */
        do_action( 'streamtube/core/widget/posts_flat/query/after', $query_args, $settings );        

        if( ! $post_query->have_posts() ){
            return;
        }

        ?>

        <div class="flat-posts">

            <?php 
            $loop = 0;
            while ( $post_query->have_posts() ): 
            $loop++;
            $post_query->the_post(); 

            global $post;

            $displayed_post_ids[]   = $post->ID;
            $displayed_author_ids[] = $post->post_author;

            ?>

                <?php if( $loop == 1 ) :?>

                    <div class="large-post-wrap">
                        <?php 
                            get_template_part( 'template-parts/content/content', 'flat', $settings );
                        ?>
                    </div>

                    <div class="small-posts-wrap">
                <?php else: ?>

                    <?php printf(
                        '<div class="small-post %s">',
                        $loop == 2 ?'small-post-wide' : 'small-post-small'
                    );?>
                        <?php 
                            get_template_part( 'template-parts/content/content', 'flat', $settings );
                        ?>
                    </div>

                    <?php if( $loop == 4 ):?>
                        </div><!--.small-posts-wrap-->
                    <?php endif; ?>                    

                <?php endif; ?>


            <?php endwhile; ?>

        </div>

        <?php

        unset( $GLOBALS['widget_instance'] );

        wp_reset_postdata();
    }
}

if( defined( 'ELEMENTOR_VERSION' ) && version_compare( ELEMENTOR_VERSION , '3.5.0', '<' ) ){
    \Elementor\Plugin::instance()->widgets_manager->register_widget_type( new Streamtube_Core_Flat_Posts_Elementor() );
}
else{
    \Elementor\Plugin::instance()->widgets_manager->register( new Streamtube_Core_Flat_Posts_Elementor() );
}

