<?php

/**
 * The file that defines the core plugin class
 *
 * A class definition that includes attributes and functions used across both the
 * public-facing side of the site and the admin area.
 *
 * @link       https://themeforest.net/user/phpface
 * @since      1.0.0
 *
 * @package    Streamtube_Core
 * @subpackage Streamtube_Core/includes
 */

/**
 * The core plugin class.
 *
 * This is used to define internationalization, admin-specific hooks, and
 * public-facing site hooks.
 *
 * Also maintains the unique identifier of this plugin as well as the current
 * version of the plugin.
 *
 * @since      1.0.0
 * @package    Streamtube_Core
 * @subpackage Streamtube_Core/includes
 * @author     phpface <nttoanbrvt@gmail.com>
 */
if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

class Streamtube_Core {

	/**
	 * The loader that's responsible for maintaining and registering all hooks that power
	 * the plugin.
	 *
	 * @since    1.0.0
	 * @access   protected
	 * @var      Streamtube_Core_Loader    $loader    Maintains and registers all hooks for the plugin.
	 */
	protected $loader;

	/**
	 * The unique identifier of this plugin.
	 *
	 * @since    1.0.0
	 * @access   protected
	 * @var      string    $plugin_name    The string used to uniquely identify this plugin.
	 */
	protected $plugin_name;

	protected $plugin_setting_name;

	/**
	 * The current version of the plugin.
	 *
	 * @since    1.0.0
	 * @access   protected
	 * @var      string    $version    The current version of the plugin.
	 */
	protected $version;

	protected $plugin;

	protected $license = null;

	/**
	 * Define the core functionality of the plugin.
	 *
	 * Set the plugin name and the plugin version that can be used throughout the plugin.
	 * Load the dependencies, define the locale, and set the hooks for the admin area and
	 * the public-facing side of the site.
	 *
	 * @since    1.0.0
	 */
	public function __construct() {

		if ( defined( 'STREAMTUBE_CORE_VERSION' ) ) {
			$this->version = STREAMTUBE_CORE_VERSION;
		} else {
			$this->version = '2.0';
		}

		$this->plugin_name = 'StreamTube Core';

		$this->plugin = new stdClass();

		$this->load_dependencies();
		$this->request_license();
		$this->set_locale();
		$this->define_admin_hooks();
		$this->define_public_hooks();
		$this->define_core_hooks();
		$this->define_ads_hooks();
		$this->define_post_hooks();
		$this->define_video_hooks();
		$this->define_player_hooks();
		$this->define_timestamp_hooks();
		$this->define_comment_hooks();
		$this->define_user_hooks();
		$this->define_rest_hooks();

		$this->define_collection_hooks();

		$this->define_mobile_bottom_bar();

		$this->define_youtube_importer();

		$this->define_bunnycdn();

		$this->define_googlesitekit_hooks();

		$this->define_woothanks();

		$this->define_open_graph();

		$this->update_checker();

		$this->action_links();

		$this->define_storage_hooks();

		if ( class_exists( 'BigFileUploads' ) ) {
			$this->define_big_file_uploads_hooks();
		}

		if ( class_exists( 'myCRED_Core' ) ) {
			$this->define_mycred_hooks();
		}

		if ( class_exists( 'BuddyPress' ) ) {
			$this->define_buddypress_hooks();
		}

		if ( class_exists( 'BP_Better_Messages' ) ) {
			$this->define_better_messages_hooks();
		}

		if ( class_exists( 'ACF_Form_Post' ) && class_exists( 'ACF_Form_User' ) ) {
			$this->define_acf_hooks();
		}

		if ( class_exists( 'bbpress' ) ) {
			$this->define_bbpress();
		}

		if ( function_exists( 'pmpro_activation' ) ) {
			$this->define_pmpro();
		}

		if ( class_exists( 'Paid_Member_Subscriptions' ) ) {
			$this->define_pms();
		}

		if ( class_exists( 'Patreon_Wordpress' ) ) {
			$this->define_patreon();
		}

		if ( $this->is_plugin_active( 'woocommerce/woocommerce.php' ) ) {
			$this->define_woocommerce();

			if ( $this->is_plugin_active( 'woocommerce-bookings/woocommerce-bookings.php' ) ) {
				$this->define_woocommerce_booking();
			}

			if ( $this->is_plugin_active( 'yith-woocommerce-affiliates/init.php' ) ) {
				$this->define_yith_woocommerce_affiliates();
			}
		}

		if ( function_exists( 'dokan' ) && $this->is_dokan_active() ) {
			$this->define_dokan();
		}

		if ( $this->is_plugin_active( 'wp-fundraising-donation/wp-fundraising.php' ) ) {
			$this->define_fundengine();
		}

		if ( $this->is_plugin_active( 'directorist/directorist-base.php' ) ) {
			$this->define_directorist();
		}

		if (
			$this->is_plugin_active( 'wp-statistics/wp-statistics.php' ) &&
			$this->is_wp_statistics_active() ) {
			$this->define_wp_statistics();
		}

		if ( $this->is_plugin_active( 'jwt-authentication-for-wp-rest-api/jwt-auth.php' ) ) {
			$this->define_jwt_auth();
		}

		if ( defined( 'RCB_PATH' ) ) {
			$this->define_real_cookie_banner();
		}

	}

	/**
	 *
	 * Include file in WP environment
	 * 
	 * @param  string $file
	 *
	 * @since 1.0.9
	 * 
	 */
	private function include_file( $file ) {
		require_once trailingslashit( plugin_dir_path( dirname( __FILE__ ) ) ) . $file;
	}

	/**
	 *
	 * Include WP admin file
	 * 
	 */
	private function include_admin_file( $file ) {
		require_once ABSPATH . 'wp-admin/includes/' . $file;
	}

	/**
	 * Load the required dependencies for this plugin.
	 *
	 * Include the following files that make up the plugin:
	 *
	 * - Streamtube_Core_Loader. Orchestrates the hooks of the plugin.
	 * - Streamtube_Core_i18n. Defines internationalization functionality.
	 * - Streamtube_Core_Admin. Defines all hooks for the admin area.
	 * - Streamtube_Core_Public. Defines all hooks for the public side of the site.
	 *
	 * Create an instance of the loader which will be used to register the hooks
	 * with WordPress.
	 *
	 * @since    1.0.0
	 * @access   private
	 */
	private function load_dependencies() {

		$this->include_admin_file( 'plugin.php' );

		$this->include_admin_file( 'user.php' );

		$this->include_admin_file( 'class-wp-list-table.php' );
		// Includes the base class for creating custom list tables in the WordPress admin area.

		$this->include_admin_file( 'class-wp-terms-list-table.php' );
		// Includes the class for managing and displaying taxonomy terms in list table format.

		$this->include_admin_file( 'class-wp-posts-list-table.php' );
		// Includes the class for displaying and managing posts in list table format.

		$this->include_admin_file( 'class-wp-comments-list-table.php' );
		// Includes the class for handling comments display and management in the admin list table.

		$this->include_admin_file( 'class-wp-users-list-table.php' );
		// Includes the class for handling users display and management in the admin list table.		

		$this->include_admin_file( 'class-wp-screen.php' );
		// Includes the class that provides functionality for managing WordPress admin screens.

		$this->include_admin_file( 'screen.php' );
		// Includes core functions for screen management in the WordPress admin interface.

		$this->include_admin_file( 'template.php' );
		// Includes helper functions for working with WordPress admin templates.

		$this->include_admin_file( 'list-table.php' );
		// Includes additional utilities and functions for managing admin list tables.

		$this->include_admin_file( 'post.php' );
		// Includes functions and utilities related to handling post objects in the admin.

		$this->include_admin_file( 'taxonomy.php' );
		// Includes functions for handling taxonomy-related operations in the admin.

		$this->include_admin_file( 'comment.php' );
		// Includes functions for managing comments and their operations in the admin.

		$this->include_admin_file( 'theme.php' );
		// Includes functions related to theme management and operations in the admin.

		// Google OAuth
		$this->include_file( 'includes/class-streamtube-core-google-oauth.php' );

		global $streamtube_google_oauth;

		if ( ! $streamtube_google_oauth ) {
			$streamtube_google_oauth = new StreamTube_Core_Google_OAuth(
				array(
					'client_id'     => get_option( 'google_client_id' ),
					'client_secret' => get_option( 'google_client_secret' ),
					'redirect_uri'  => add_query_arg( array(
						'oauth' => 'google'
					), home_url( '/' ) )
				)
			);
		}

		/**
		 * The class responsible for defining license functionality
		 * of the plugin.
		 */
		$this->include_file( 'includes/class-streamtube-core-license.php' );

		/**
		 * The class responsible for orchestrating the actions and filters of the
		 * core plugin.
		 */
		$this->include_file( 'includes/class-streamtube-core-loader.php' );

		$this->loader = new Streamtube_Core_Loader();

		/**
		 * The class responsible for defining internationalization functionality
		 * of the plugin.
		 */
		$this->include_file( 'includes/class-streamtube-core-i18n.php' );

		/**
		 * The class responsible for defining cron job functionality
		 * of the plugin.
		 */
		$this->include_file( 'includes/class-streamtube-core-cron.php' );

		/**
		 * The class responsible for defining role functionality
		 * of the plugin.
		 */
		$this->include_file( 'includes/class-streamtube-core-role.php' );

		/**
		 * The class responsible for defining upload functionality
		 * of the plugin.
		 */
		$this->include_file( 'includes/class-streamtube-core-upload.php' );

		/**
		 * HTTP Request
		 */
		$this->include_file( 'includes/class-streamtube-core-http-request.php' );

		/**
		 * Misc
		 */
		$this->include_file( 'includes/class-streamtube-core-misc.php' );

		/**
		 * System permission
		 */
		$this->include_file( 'includes/class-streamtube-core-permission.php' );

		/**
		 * The class responsible for defining oEmbed functionality
		 * of the plugin.
		 */
		$this->include_file( 'includes/class-streamtube-core-oembed.php' );

		/**
		 * The class responsible for defining endpoint functionality
		 * of the plugin.
		 */
		$this->include_file( 'includes/class-streamtube-core-endpoint.php' );

		/**
		 * The class responsible for defining all metabox functionality
		 */
		$this->include_file( 'includes/class-streamtube-core-metabox.php' );

		/**
		 * The class responsible for defining all widget metabox functionality
		 */
		$this->include_file( 'includes/class-streamtube-core-widget-metabox.php' );

		/**
		 * The class responsible for defining custom menu functionality
		 * of the plugin.
		 */
		$this->include_file( 'includes/class-streamtube-core-menu.php' );

		/**
		 * The class responsible for defining WP menu functionality
		 * of the plugin.
		 */
		$this->include_file( 'includes/class-streamtube-core-wp-menu.php' );

		/**
		 * The class responsible for defining all post functionality
		 */
		$this->include_file( 'includes/class-streamtube-core-post.php' );

		/**
		 * The class responsible for defining all video functionality
		 */
		$this->include_file( 'includes/class-streamtube-core-video.php' );

		/**
		 * The class responsible for defining all player functionality
		 */
		$this->include_file( 'includes/class-streamtube-core-player.php' );

		/**
		 * The class responsible for defining all download functionality
		 */
		$this->include_file( 'includes/class-streamtube-core-download-files.php' );

		/**
		 * The class responsible for defining all comment functionality
		 */
		$this->include_file( 'includes/class-streamtube-core-comment.php' );

		/**
		 * The class responsible for defining all custom taxonomies
		 */
		$this->include_file( 'includes/class-streamtube-core-taxonomy.php' );

		/**
		 * The class responsible for defining all custom taxonomies
		 */
		$this->include_file( 'includes/class-streamtube-core-terms-list-table.php' );

		/**
		 * The class responsible for defining posts list table
		 */
		$this->include_file( 'includes/class-streamtube-core-posts-list-table.php' );

		/**
		 * The class responsible for defining comments list table
		 */
		$this->include_file( 'includes/class-streamtube-core-comments-list-table.php' );

		/**
		 * The class responsible for defining comments list table
		 */
		$this->include_file( 'includes/class-streamtube-core-users-list-table.php' );

		/**
		 * The class responsible for defining sidebar
		 */
		$this->include_file( 'includes/class-streamtube-core-sidebar.php' );

		/**
		 * The class responsible for defining upgrading functionality
		 * of the plugin.
		 */
		$this->include_file( 'includes/upgrade/class-streamtube-core-upgrade-40.php' );

		$this->include_file( 'includes/widgets/class-streamtube-core-widget-live-ajax-search.php' );

		/**
		 * The class responsible for defining custom posts widget
		 */
		$this->include_file( 'includes/widgets/class-streamtube-core-widget-posts.php' );

		/**
		 * The class responsible for defining custom taxonomy widget
		 */
		$this->include_file( 'includes/widgets/class-streamtube-core-widget-term-grid.php' );

		/**
		 * The class responsible for defining custom posts widget
		 */
		$this->include_file( 'includes/widgets/class-streamtube-core-widget-comments.php' );

		/**
		 * The class responsible for defining custom posts widget
		 */
		$this->include_file( 'includes/widgets/class-streamtube-core-widget-comments-template.php' );

		/**
		 * The class responsible for defining custom user list widget
		 */
		$this->include_file( 'includes/widgets/class-streamtube-core-widget-user-list.php' );

		/**
		 * The class responsible for defining custom user grid widget
		 */
		$this->include_file( 'includes/widgets/class-streamtube-core-widget-user-grid.php' );

		/**
		 * The class responsible for defining custom taxonomy widget
		 */
		$this->include_file( 'includes/widgets/class-streamtube-core-widget-video-categories.php' );

		/**
		 * The class responsible for defining custom taxonomy widget
		 */
		$this->include_file( 'includes/widgets/class-streamtube-core-widget-chatroom.php' );

		/**
		 * The class responsible for defining recorded videos widget
		 */
		$this->include_file( 'includes/widgets/class-streamtube-core-widget-recorded-videos.php' );

		/**
		 * The class responsible for defining playlist content widget
		 */
		$this->include_file( 'includes/widgets/class-streamtube-core-widget-playlist-content.php' );

		/**
		 * The class responsible for defining Content Type Filter widget
		 */
		$this->include_file( 'includes/widgets/class-streamtube-core-widget-filter-content-type.php' );

		/**
		 * The class responsible for defining advanced navigation widget
		 */
		$this->include_file( 'includes/widgets/class-streamtube-core-widget-nav.php' );

		/**
		 * The class responsible for defining Content Cost Filter widget
		 */
		$this->include_file( 'includes/widgets/class-streamtube-core-widget-filter-content-cost.php' );

		/**
		 * The class responsible for defining Submit Date Filter widget
		 */
		$this->include_file( 'includes/widgets/class-streamtube-core-widget-filter-post-date.php' );

		/**
		 * The class responsible for defining Sortby Filter widget
		 */
		$this->include_file( 'includes/widgets/class-streamtube-core-widget-filter-sortby.php' );

		/**
		 * The class responsible for defining Taxonomy Filter widget
		 */
		$this->include_file( 'includes/widgets/class-streamtube-core-widget-filter-taxonomy.php' );

		/**
		 * The class responsible for defining PMP Filter widget
		 */
		$this->include_file( 'includes/widgets/class-streamtube-core-widget-filter-paid-membership-pro.php' );

		/**
		 * The class responsible for defining custom elementor widgets
		 */
		$this->include_file( 'includes/class-streamtube-core-elementor.php' );

		/**
		 * The class responsible for defining user profile page
		 */
		$this->include_file( 'includes/class-streamtube-core-user.php' );

		/**
		 * The class responsible for defining user privacy
		 */
		$this->include_file( 'includes/class-streamtube-core-user-privacy.php' );

		/**
		 * The class responsible for defining user profile page
		 */
		$this->include_file( 'includes/class-streamtube-core-user-profile.php' );

		/**
		 * The class responsible for defining user profile page
		 */
		$this->include_file( 'includes/class-streamtube-core-user-dashboard.php' );

		/**
		 * The class responsible for defining shortcodes.
		 */
		$this->include_file( 'includes/class-streamtube-core-shortcode.php' );

		/**
		 * The class responsible for defining restrict conte t
		 */
		$this->include_file( 'includes/class-streamtube-core-restrict-content.php' );

		$this->plugin->restrict_content = new Streamtube_Core_Restrict_Content();

		$this->include_file( 'includes/class-streamtube-core-update.php' );

		$this->include_file( 'includes/class-streamtube-core-modules.php' );

		/**
		 * The class responsible for defining rest.
		 */
		$this->include_file( 'includes/rest_api/class-streamtube-core-rest-api.php' );

		/**
		 * Post rest
		 */
		$this->include_file( 'includes/rest_api/class-streamtube-core-post-rest-api.php' );

		/**
		 * The class responsible for defining generate image rest API
		 */
		$this->include_file( 'includes/rest_api/class-streamtube-core-generate-image-rest-controller.php' );

		/**
		 * The class responsible for defining generate subtitle rest API
		 */
		$this->include_file( 'includes/rest_api/class-streamtube-core-subtitles-rest-controller.php' );

		/**
		 * The class responsible for defining user rest API
		 */
		$this->include_file( 'includes/rest_api/class-streamtube-core-user-rest-controller.php' );

		/**
		 * The class responsible for defining upload rest API
		 */
		$this->include_file( 'includes/rest_api/class-streamtube-core-upload-rest-controller.php' );

		/**
		 * The class responsible for defining embed rest API
		 */
		$this->include_file( 'includes/rest_api/class-streamtube-core-embed-rest-controller.php' );

		/**
		 * The class responsible for defining all actions that occur in the admin area.
		 */
		$this->include_file( 'admin/class-streamtube-core-admin.php' );

		$this->include_file( 'admin/class-streamtube-core-task-spooler.php' );

		$this->include_file( 'admin/class-streamtube-core-admin-user.php' );

		$this->include_file( 'admin/class-streamtube-core-metabox.php' );

		$this->include_file( 'admin/class-streamtube-core-customizer.php' );

		/**
		 * The class responsible for defining all actions that occur in the public-facing
		 * side of the site.
		 */
		$this->include_file( 'public/class-streamtube-core-public.php' );

		/**
		 * The function responsible for defining post functions
		 */
		$this->include_file( 'includes/function-users.php' );

		/**
		 * The function responsible for defining post functions
		 */
		$this->include_file( 'includes/function-posts.php' );

		/**
		 * The function responsible for defining post functions
		 */
		$this->include_file( 'includes/function-comments.php' );

		/**
		 * The function responsible for defining user functions
		 */
		$this->include_file( 'includes/function-templates.php' );

		/**
		 * The template tags
		 */
		$this->include_file( 'includes/template-tags.php' );

		/**
		 * The function responsible for defining email functions
		 */
		$this->include_file( 'includes/function-notify.php' );

		/**
		 * The function responsible for defining options functions
		 */
		$this->include_file( 'includes/function-options.php' );

		/**
		 * The function responsible for defining filters
		 */
		$this->include_file( 'includes/function-filters.php' );

		/**
		 * The function responsible for defining metaboxes
		 */
		$this->include_file( 'includes/function-metaboxes.php' );
	}

	/**
	 *
	 * Request License
	 * 
	 */
	private function request_license() {

		$license = new Streamtube_Core_License();

		$this->license = $license->is_verified();

		if ( is_wp_error( $this->license ) || ! is_array( $this->license ) ) {
			$this->license = false;
		}

		wp_cache_set( "streamtube:license", $this->license );
	}

	/**
	 * Get license
	 */
	private function get_license() {
		return wp_cache_get( "streamtube:license" );
	}

	/**
	 * Define the locale for this plugin for internationalization.
	 *
	 * Uses the Streamtube_Core_i18n class in order to set the domain and to register the hook
	 * with WordPress.
	 *
	 * @since    1.0.0
	 * @access   private
	 */
	private function set_locale() {
		if ( version_compare( wp_get_wp_version(), '6.8', '<' ) ) {
			$plugin_i18n = new Streamtube_Core_i18n();
			$this->loader->add_action( 'init', $plugin_i18n, 'load_plugin_textdomain' );
		}
	}

	/**
	 * Check if a given plugin is active.
	 */
	private function is_plugin_active( $plugin = '' ) {
		return function_exists( 'is_plugin_active' ) && is_plugin_active( $plugin );
	}

	/**
	 * Check if WP Statistics is active.
	 */
	private function is_wp_statistics_active() {
		return ! defined( 'STREAMTUBE_WP_STATISTICS' ) || defined( 'STREAMTUBE_WP_STATISTICS' ) && STREAMTUBE_WP_STATISTICS === true;
	}

	/**
	 * Check if Dokan is active.
	 */
	private function is_dokan_active() {
		return ! defined( 'STREAMTUBE_DOKAN' ) || defined( 'STREAMTUBE_DOKAN' ) && STREAMTUBE_DOKAN === true;
	}

	/**
	 * Register all of the hooks related to the admin area functionality
	 * of the plugin.
	 *
	 * @since    1.0.0
	 * @access   private
	 */
	private function define_admin_hooks() {

		$this->plugin->customizer = new Streamtube_Core_Customizer();

		$this->loader->add_action(
			'customize_register',
			$this->plugin->customizer,
			'register'
		);

		$this->plugin->admin = new Streamtube_Core_Admin( $this->get_plugin_name(), $this->get_version() );

		$this->loader->add_action(
			'admin_enqueue_scripts',
			$this->plugin->admin,
			'enqueue_styles'
		);

		$this->loader->add_action(
			'admin_enqueue_scripts',
			$this->plugin->admin,
			'enqueue_scripts'
		);

		$this->loader->add_action(
			'wp_ajax_streamtube/core/upgrade_database',
			$this->plugin->admin,
			'ajax_upgrade_database'
		);

		$this->loader->add_action(
			'admin_init',
			$this->plugin->admin,
			'auto_upgrade'
		);

		$this->loader->add_action(
			'admin_notices',
			$this->plugin->admin,
			'notice_upgrade_database'
		);

		$this->plugin->admin_user = new Streamtube_Core_Admin_User();

		$this->loader->add_filter(
			'manage_users_columns',
			$this->plugin->admin_user,
			'user_table',
			10,
			1
		);

		$this->loader->add_filter(
			'manage_edit-users_columns',
			$this->plugin->admin_user,
			'user_table',
			10,
			1
		);

		$this->loader->add_filter(
			'manage_users_custom_column',
			$this->plugin->admin_user,
			'user_table_columns',
			20,
			3
		);

		$this->plugin->admin_metabox = new Streamtube_Core_Admin_MetaBox();

		$this->loader->add_action(
			'add_meta_boxes',
			$this->plugin->admin_metabox,
			'add_meta_boxes',
			1
		);

		$this->loader->add_action(
			'save_post_video',
			$this->plugin->admin_metabox,
			'video_data_save',
			10,
			1
		);

		$this->loader->add_action(
			'save_post',
			$this->plugin->admin_metabox,
			'template_options_save',
			10,
			1
		);

		$this->loader->add_action(
			'add_meta_boxes',
			$this->plugin->restrict_content,
			'metaboxes'
		);

		$this->loader->add_action(
			'save_post_video',
			$this->plugin->restrict_content,
			'save_data',
			10,
			1
		);

		$this->loader->add_action(
			'streamtube/player/file/output',
			$this->plugin->restrict_content,
			'filter_player_output',
			20,
			2
		);

		$this->loader->add_action(
			'streamtube/player/embed/output',
			$this->plugin->restrict_content,
			'filter_player_embed_output',
			20,
			2
		);

		$this->loader->add_filter(
			'streamtube/core/rest/video/source',
			$this->plugin->restrict_content,
			'filter_rest_video_source_field',
			50,
			2
		);

		$this->loader->add_action(
			'streamtube/core/video/can_user_download',
			$this->plugin->restrict_content,
			'filter_download_permission',
			10,
			1
		);

		$this->loader->add_action(
			'wp_ajax_join_us',
			$this->plugin->restrict_content,
			'ajax_request_join_us',
			10,
			1
		);

		$this->loader->add_action(
			'wp_footer',
			$this->plugin->restrict_content,
			'load_modal_join_us',
			10,
			1
		);

		$this->loader->add_filter(
			'manage_video_posts_columns',
			$this->plugin->restrict_content,
			'filter_post_table',
			10,
			1
		);

		$this->loader->add_action(
			'manage_video_posts_custom_column',
			$this->plugin->restrict_content,
			'filter_post_table_columns',
			10,
			2
		);

		$this->plugin->task_spooler = new Streamtube_Core_Task_Spooler();

		$this->loader->add_action(
			'admin_menu',
			$this->plugin->task_spooler,
			'admin_menu'
		);
	}

	/**
	 * Register all of the hooks related to the public-facing functionality
	 * of the plugin.
	 *
	 * @since    1.0.0
	 * @access   private
	 */
	private function define_public_hooks() {

		$this->plugin->public = new Streamtube_Core_Public();

		$this->loader->add_action(
			'wp_enqueue_scripts',
			$this->plugin->public,
			'enqueue_styles',
			20
		);

		$this->loader->add_action(
			'wp_enqueue_scripts',
			$this->plugin->public,
			'enqueue_scripts'
		);

		$this->loader->add_action(
			'login_enqueue_scripts',
			$this->plugin->public,
			'enqueue_scripts'
		);

		$this->loader->add_action(
			'enqueue_embed_scripts',
			$this->plugin->public,
			'enqueue_embed_scripts'
		);

		$this->loader->add_action(
			'streamtube/header/profile/before',
			$this->plugin->public,
			'the_upload_button'
		);

		$this->loader->add_action(
			'wp_footer',
			$this->plugin->public,
			'load_modals'
		);


		$this->loader->add_filter(
			'search_template',
			$this->plugin->public,
			'load_search_template'
		);

		$this->loader->add_action(
			'wp_head',
			$this,
			'generator'
		);

	}

	/**
	 * Register all of the hooks related to the core functionality
	 * of the plugin.
	 *
	 * @since    1.0.0
	 * @access   private
	 */
	private function define_core_hooks() {

		$this->plugin->modules = new Streamtube_Core_Modules();

		$this->loader->add_action(
			'init',
			$this->plugin->modules,
			'register',
			9999
		);

		$this->loader->add_action(
			'streamtube/core/register_modules',
			$this->plugin->modules,
			'add_post_types'
		);

		$this->loader->add_action(
			'streamtube/core/register_modules',
			$this->plugin->modules,
			'add_taxonomies'
		);

		$this->plugin->role = new StreamTube_Core_Role();

		$this->loader->add_action(
			'wp_ajax_streamtube/core/grant_role_capability',
			$this->plugin->role,
			'ajax_update_role_capability'
		);

		$this->loader->add_action(
			'wp_ajax_streamtube/core/add_role_capability',
			$this->plugin->role,
			'ajax_add_role_capability'
		);

		$this->loader->add_action(
			'wp_ajax_streamtube/core/add_role',
			$this->plugin->role,
			'ajax_add_role'
		);

		$this->loader->add_action(
			'wp_ajax_streamtube/core/delete_role',
			$this->plugin->role,
			'ajax_delete_role'
		);

		$this->loader->add_action(
			'wp_ajax_streamtube/core/delete_capability',
			$this->plugin->role,
			'ajax_delete_capability'
		);

		$this->loader->add_action(
			'wp_ajax_streamtube/core/search_capabilities',
			$this->plugin->role,
			'ajax_search_capabilities'
		);

		$this->loader->add_filter(
			'http_request_args',
			'Streamtube_Core_HTTP_Request',
			'filter_request_args',
			10,
			2
		);

		$this->plugin->cron = new Streamtube_Core_Cron();

		$this->loader->add_filter(
			'cron_schedules',
			$this->plugin->cron,
			'add_schedules',
			10,
			1
		);

		$this->loader->add_action(
			'init',
			'Streamtube_Core_Endpoint',
			'add_endpoints'
		);

		$this->plugin->metabox = new Streamtube_Core_MetaBox();

		$this->loader->add_action(
			'streamtube/core/post/edit/metaboxes/side',
			$this->plugin->metabox,
			'side'
		);

		$this->loader->add_action(
			'streamtube/core/post/edit/metaboxes/normal',
			$this->plugin->metabox,
			'normal'
		);

		$this->loader->add_action(
			'streamtube/core/post/edit/metaboxes/advanced',
			$this->plugin->metabox,
			'advanced'
		);

		$this->loader->add_action(
			'streamtube/core/user/dashboard/metabox',
			$this->plugin->metabox,
			'user_dashboard'
		);

		$this->loader->add_action(
			'streamtube/core/googlesitekit/metaboxes',
			$this->plugin->metabox,
			'googlesitekit'
		);

		$this->loader->add_action(
			'streamtube_add_meta_boxes',
			$this->plugin->metabox,
			'add_meta_boxes'
		);

		$this->loader->add_action(
			'wp_ajax_reset_metaboxes_order',
			$this->plugin->metabox,
			'ajax_reset_metaboxes_order'
		);

		$this->loader->add_action(
			'wp_enqueue_scripts',
			$this->plugin->metabox,
			'enqueue_scripts'
		);

		$this->plugin->taxonomy = new Streamtube_Core_Taxonomy();

		$this->loader->add_action(
			'init',
			$this->plugin->taxonomy,
			'video_category'
		);

		$this->loader->add_action(
			'init',
			$this->plugin->taxonomy,
			'video_tag'
		);

		$this->loader->add_action(
			'init',
			$this->plugin->taxonomy,
			'report_category'
		);

		$this->loader->add_action(
			'init',
			$this->plugin->taxonomy,
			'hand_pick'
		);

		$this->loader->add_action(
			'wp_ajax_search_terms',
			$this->plugin->taxonomy,
			'search_terms'
		);

		$this->loader->add_action(
			'wp_ajax_get_video_tag_terms',
			$this->plugin->taxonomy,
			'ajax_get_video_tag_terms'
		);

		$this->loader->add_action(
			'categories_add_form_fields',
			$this->plugin->taxonomy,
			'add_thumbnail_field',
			10,
			1
		);

		$this->loader->add_action(
			'categories_edit_form_fields',
			$this->plugin->taxonomy,
			'edit_thumbnail_field',
			10,
			2
		);

		$this->loader->add_action(
			'video_tag_add_form_fields',
			$this->plugin->taxonomy,
			'add_thumbnail_field',
			10,
			1
		);

		$this->loader->add_action(
			'video_tag_edit_form_fields',
			$this->plugin->taxonomy,
			'edit_thumbnail_field',
			10,
			2
		);

		$this->loader->add_action(
			'category_add_form_fields',
			$this->plugin->taxonomy,
			'add_thumbnail_field',
			10,
			1
		);

		$this->loader->add_action(
			'category_edit_form_fields',
			$this->plugin->taxonomy,
			'edit_thumbnail_field',
			10,
			2
		);

		$this->loader->add_action(
			'created_categories',
			$this->plugin->taxonomy,
			'update_thumbnail_field',
			10,
			1
		);

		$this->loader->add_action(
			'edited_categories',
			$this->plugin->taxonomy,
			'update_thumbnail_field',
			10,
			1
		);

		$this->loader->add_action(
			'created_video_tag',
			$this->plugin->taxonomy,
			'update_thumbnail_field',
			10,
			1
		);

		$this->loader->add_action(
			'edited_video_tag',
			$this->plugin->taxonomy,
			'update_thumbnail_field',
			10,
			1
		);

		$this->loader->add_action(
			'created_category',
			$this->plugin->taxonomy,
			'update_thumbnail_field',
			10,
			1
		);

		$this->loader->add_action(
			'edited_category',
			$this->plugin->taxonomy,
			'update_thumbnail_field',
			10,
			1
		);

		$this->loader->add_filter(
			'manage_edit-categories_columns',
			$this->plugin->taxonomy,
			'add_thumbnail_column',
			10,
			1
		);

		$this->loader->add_filter(
			'manage_categories_custom_column',
			$this->plugin->taxonomy,
			'add_thumbnail_column_content',
			10,
			3
		);

		$this->loader->add_filter(
			'manage_edit-video_tag_columns',
			$this->plugin->taxonomy,
			'add_thumbnail_column',
			10,
			1
		);

		$this->loader->add_filter(
			'manage_video_tag_custom_column',
			$this->plugin->taxonomy,
			'add_thumbnail_column_content',
			10,
			3
		);

		$this->loader->add_filter(
			'manage_edit-category_columns',
			$this->plugin->taxonomy,
			'add_thumbnail_column',
			10,
			1
		);

		$this->loader->add_filter(
			'manage_category_custom_column',
			$this->plugin->taxonomy,
			'add_thumbnail_column_content',
			10,
			3
		);

		$this->loader->add_filter(
			'get_edit_term_link',
			$this->plugin->taxonomy,
			'filter_edit_term_link',
			10,
			4
		);

		$this->loader->add_filter(
			'tag_row_actions',
			$this->plugin->taxonomy,
			'filter_tag_row_actions',
			10,
			2
		);

		$this->loader->add_filter(
			'register_taxonomy_args',
			$this->plugin->taxonomy,
			'filter_register_taxonomy_args',
			1,
			3
		);

		$this->loader->add_filter(
			'clean_url',
			$this->plugin->taxonomy,
			'filter_count_post_url',
			10,
			2
		);

		$this->loader->add_action(
			'restrict_manage_posts',
			$this->plugin->taxonomy,
			'add_dropdown_filters',
			10,
			2
		);

		$this->plugin->sidebar = new Streamtube_Core_Sidebar();

		$this->loader->add_action(
			'widgets_init',
			$this->plugin->sidebar,
			'widgets_init'
		);

		$this->loader->add_action(
			'widgets_init',
			'Streamtube_Core_Widget_Filter_Content_Type',
			'register'
		);

		$this->loader->add_action(
			'widgets_init',
			'Streamtube_Core_Widget_Filter_Content_Cost',
			'register'
		);

		$this->loader->add_action(
			'widgets_init',
			'Streamtube_Core_Widget_Filter_Taxonomy',
			'register'
		);

		$this->loader->add_action(
			'widgets_init',
			'Streamtube_Core_Widget_Filter_Post_Date',
			'register'
		);

		$this->loader->add_action(
			'widgets_init',
			'Streamtube_Core_Widget_Filter_Sortby',
			'register'
		);

		$this->loader->add_action(
			'widgets_init',
			'Streamtube_Core_Widget_Filter_Paid_Membership_Pro',
			'register'
		);

		$this->loader->add_action(
			'widgets_init',
			'Streamtube_Core_Widget_Live_Ajax_Search',
			'register'
		);

		$this->loader->add_action(
			'widgets_init',
			'Streamtube_Core_Widget_User_List',
			'register'
		);

		$this->loader->add_action(
			'widgets_init',
			'Streamtube_Core_Widget_User_Grid',
			'register'
		);

		$this->loader->add_action(
			'wp_ajax_load_more_users',
			'Streamtube_Core_Widget_User_Grid',
			'ajax_load_more_users'
		);

		$this->loader->add_action(
			'wp_ajax_nopriv_load_more_users',
			'Streamtube_Core_Widget_User_Grid',
			'ajax_load_more_users'
		);

		$this->loader->add_action(
			'widgets_init',
			'Streamtube_Core_Widget_Posts',
			'register'
		);

		$this->loader->add_action(
			'widgets_init',
			'Streamtube_Core_Widget_Term_Grid',
			'register'
		);

		$this->loader->add_action(
			'wp_ajax_load_more_tax_terms',
			'Streamtube_Core_Widget_Term_Grid',
			'load_more_terms'
		);

		$this->loader->add_action(
			'wp_ajax_nopriv_load_more_tax_terms',
			'Streamtube_Core_Widget_Term_Grid',
			'load_more_terms'
		);

		$this->loader->add_action(
			'widgets_init',
			'Streamtube_Core_Widget_Video_Category',
			'register'
		);

		$this->loader->add_action(
			'widgets_init',
			'Streamtube_Core_Widget_Recorded_Videos',
			'register'
		);

		$this->loader->add_action(
			'widgets_init',
			'Streamtube_Core_Widget_Comments',
			'register'
		);

		$this->loader->add_action(
			'widgets_init',
			'Streamtube_Core_Widget_Comments_Template',
			'register'
		);

		$this->loader->add_action(
			'widgets_init',
			'Streamtube_Core_Widget_Nav',
			'register'
		);

		$this->loader->add_action(
			'widgets_init',
			'Streamtube_Core_Widget_Playlist_Content',
			'register'
		);

		$this->loader->add_action(
			"wp_ajax_search_in_collection",
			'Streamtube_Core_Widget_Playlist_Content',
			'ajax_search_in_collection'
		);

		$this->loader->add_action(
			"wp_ajax_nopriv_search_in_collection",
			'Streamtube_Core_Widget_Playlist_Content',
			'ajax_search_in_collection'
		);

		$this->loader->add_action(
			'wp_ajax_nopriv_widget_load_more_posts',
			'Streamtube_Core_Widget_Posts',
			'ajax_load_more_posts'
		);

		$this->loader->add_action(
			'wp_ajax_widget_load_more_posts',
			'Streamtube_Core_Widget_Posts',
			'ajax_load_more_posts'
		);

		$this->loader->add_action(
			'wp_ajax_ajax_download_widget_json',
			'Streamtube_Core_Widget_Posts',
			'ajax_download_json'
		);

		/** Elementor  */
		$this->plugin->elementor = new Streamtube_Core_Elementor();

		$this->loader->add_action(
			'init',
			$this->plugin->elementor,
			'init'
		);

		$this->plugin->shortcode = new Streamtube_Core_ShortCode();

		$this->loader->add_action(
			'init',
			$this->plugin->shortcode,
			'is_logged_in'
		);

		$this->loader->add_action(
			'init',
			$this->plugin->shortcode,
			'is_not_logged_in'
		);

		$this->loader->add_action(
			'init',
			$this->plugin->shortcode,
			'can_upload'
		);

		$this->loader->add_action(
			'init',
			$this->plugin->shortcode,
			'can_not_upload'
		);

		$this->loader->add_action(
			'init',
			$this->plugin->shortcode,
			'user_name'
		);

		$this->loader->add_action(
			'init',
			$this->plugin->shortcode,
			'user_avatar'
		);

		$this->loader->add_action(
			'init',
			$this->plugin->shortcode,
			'user_data'
		);

		$this->loader->add_action(
			'init',
			$this->plugin->shortcode,
			'user_grid'
		);

		$this->loader->add_action(
			'init',
			$this->plugin->shortcode,
			'post_grid'
		);

		$this->loader->add_action(
			'init',
			$this->plugin->shortcode,
			'playlist'
		);

		$this->loader->add_action(
			'init',
			$this->plugin->shortcode,
			'player'
		);

		$this->loader->add_action(
			'init',
			$this->plugin->shortcode,
			'embed_media'
		);

		$this->loader->add_action(
			'init',
			$this->plugin->shortcode,
			'button_upload'
		);

		$this->loader->add_action(
			'init',
			$this->plugin->shortcode,
			'form_upload'
		);

		$this->loader->add_action(
			'init',
			$this->plugin->shortcode,
			'form_embed'
		);

		$this->loader->add_action(
			'wp',
			$this->plugin->shortcode,
			'redirect_nonlogged_to_login_on_upload'
		);

		$this->loader->add_action(
			'init',
			$this->plugin->shortcode,
			'term_grid'
		);

		$this->loader->add_action(
			'init',
			$this->plugin->shortcode,
			'user_library'
		);

		$this->loader->add_action(
			'init',
			$this->plugin->shortcode,
			'user_dashboard_url'
		);

		$this->loader->add_action(
			'init',
			$this->plugin->shortcode,
			'chapter'
		);

		$this->loader->add_action(
			'init',
			$this->plugin->shortcode,
			'term_menu'
		);

		$this->loader->add_action(
			'init',
			$this->plugin->shortcode,
			'the_term_menu'
		);

		$this->loader->add_action(
			'init',
			$this->plugin->shortcode,
			'term_play_all_url'
		);

		$this->plugin->oembed = new Streamtube_Core_oEmbed();

		$this->loader->add_action(
			'init',
			$this->plugin->oembed,
			'add_providers'
		);

		$this->loader->add_filter(
			'embed_oembed_html',
			$this->plugin->oembed,
			'filter_embed_oembed_html',
			10,
			4
		);

		$this->loader->add_action(
			'streamtube/player/embed/output',
			$this->plugin->oembed,
			'force_balance_tags_embed_html',
			5,
			2
		);

		$this->plugin->misc = new Streamtube_Core_Misc();

		$this->loader->add_filter(
			'register_post_type_args',
			$this->plugin->misc,
			'exclude_page_from_search',
			10,
			2
		);

		$this->loader->add_filter(
			'login_url',
			$this->plugin->misc,
			'filter_login_url',
			9999,
			3
		);

		$this->loader->add_filter(
			'register_url',
			$this->plugin->misc,
			'filter_register_url',
			9999,
			1
		);

		$this->loader->add_filter(
			'show_admin_bar',
			$this->plugin->misc,
			'hide_admin_bar',
			9999,
			1
		);

		$this->loader->add_action(
			'admin_init',
			$this->plugin->misc,
			'block_admin_access'
		);

		$this->loader->add_filter(
			'wp_editor_settings',
			$this->plugin->misc,
			'wp_editor_style',
			9999,
			2
		);

		$this->loader->add_action(
			'admin_init',
			'Streamtube_Core_Update',
			'add_roles'
		);

		$this->loader->add_action(
			'admin_init',
			'Streamtube_Core_Update',
			'add_default_widgets'
		);
	}

	/**
	 * Register all of the hooks related to the storage functionality
	 * of the plugin.
	 *
	 * @since    1.0.0
	 * @access   private
	 */
	private function define_storage_hooks() {
		$this->include_file( 'third-party/storage/class-streamtube-core-storage-loader.php' );

		$this->plugin->storage = new StreamTube_Core_Storage_Loader();

		$this->loader->add_action(
			'streamtube/core/do_upgrade_database',
			'StreamTube_Core_Storage_DB',
			'install_db'
		);

		$this->loader->add_filter(
			'upload_size_limit',
			$this->plugin->storage,
			'filter_upload_size_limit',
			20
		);

		$this->loader->add_filter(
			'wp_handle_upload_prefilter',
			$this->plugin->storage,
			'filter_wp_handle_upload_prefilter',
			20
		);

		$this->loader->add_filter(
			'streamtube/core/upload/video/errors',
			$this->plugin->storage,
			'filter_upload_video_error_handler',
			20,
			2
		);

		$this->loader->add_action(
			'add_attachment',
			$this->plugin->storage,
			'add_attachment_usage'
		);

		$this->loader->add_action(
			'delete_attachment',
			$this->plugin->storage,
			'delete_attachment_usage',
			10,
			2
		);

		$this->loader->add_action(
			'created_direct_upload_url',
			$this->plugin->storage,
			'added_direct_upload_service',
			10
		);

		$this->loader->add_action(
			'streamtube/core/bunny/imported/post',
			$this->plugin->storage,
			'remove_bunny_resumable_token',
			10,
			3
		);

		$this->loader->add_action(
			'wp_cloudflare_stream_post_upload_synced',
			$this->plugin->storage,
			'remove_cloudflare_resumable_token',
			10,
			3
		);

		$this->loader->add_action(
			'wp_ajax_streamtube/core/storage/update_roles_sizes',
			$this->plugin->storage,
			'ajax_update_roles_sizes'
		);

		$this->loader->add_action(
			'wp_ajax_streamtube/core/storage/update_roles_quotas',
			$this->plugin->storage,
			'ajax_update_roles_quotes'
		);

		$this->loader->add_action(
			'wp_ajax_streamtube/core/storage/update_user_quota',
			$this->plugin->storage,
			'ajax_update_user_storage_quota'
		);

		$this->loader->add_action(
			'wp_ajax_streamtube/core/storage/reset_users_storage_quota',
			$this->plugin->storage,
			'ajax_reset_users_storage_quota'
		);

		$this->loader->add_action(
			'wp_ajax_streamtube/core/storage/add_user_manual',
			$this->plugin->storage,
			'ajax_add_user_manual_quota'
		);

		$this->loader->add_action(
			'streamtube/core/woocommerce/xcapabilities/updated_capabilities',
			$this->plugin->storage,
			'wc_purchase_capacity_quota',
			10,
			4
		);

		$this->loader->add_action(
			'streamtube/core/woocommerce/xcapabilities/removed_capabilities',
			$this->plugin->storage,
			'wc_purchase_capacity_quota',
			10,
			4
		);

		$this->loader->add_action(
			'streamtube/core/removed_role_capability',
			$this->plugin->storage,
			'delete_quota_capacities',
			10,
			2
		);

		$this->loader->add_action(
			'wp_ajax_streamtube/core/storage/add_quota_capacity',
			$this->plugin->storage,
			'ajax_add_quota_capacity'
		);

		$this->loader->add_action(
			'deleted_user',
			$this->plugin->storage,
			'deleted_user',
			10,
			3
		);

		$this->loader->add_action(
			'after_delete_post',
			$this->plugin->storage,
			'deleted_post',
			10,
			2
		);

		$this->loader->add_action(
			'init',
			$this->plugin->storage,
			'register_shortcode_user_storage_progress'
		);

		$this->loader->add_action(
			'init',
			$this->plugin->storage,
			'register_shortcode_user_storage_table'
		);

		$this->loader->add_filter(
			'manage_edit-users_columns',
			$this->plugin->storage,
			'users_list_table_columns',
			10,
			1
		);

		$this->loader->add_filter(
			'manage_users_custom_column',
			$this->plugin->storage,
			'users_list_table_columns_content',
			10,
			3
		);

		$this->loader->add_filter(
			'manage_frontend_users_list_table_actions',
			$this->plugin->storage,
			'users_list_table_actions',
			10,
			1
		);
	}

	/**
	 * Register all of the hooks related to the ads functionality
	 * of the plugin.
	 *
	 * @since    1.0.0
	 * @access   private
	 */
	private function define_ads_hooks() {
		$this->include_file( 'third-party/advertising/class-streamtube-core-advertising.php' );

		$this->plugin->advertising = new Streamtube_Core_Advertising();

		if ( ! $this->get_license() ) {
			$this->loader->add_action(
				'admin_menu',
				$this->plugin->advertising->admin,
				'admin_menu_unregistered'
			);
		} else {

			$this->loader->add_action(
				'admin_init',
				$this->plugin->advertising,
				'update_htaccess'
			);

			$this->loader->add_action(
				'streamtube/core/register_modules',
				$this->plugin->advertising,
				'register_permission_module'
			);

			$this->loader->add_action(
				'admin_menu',
				$this->plugin->advertising->admin,
				'admin_menu'
			);
			$this->loader->add_action(
				'init',
				$this->plugin->advertising->ad_tag,
				'post_type'
			);

			$this->loader->add_action(
				'add_meta_boxes',
				$this->plugin->advertising->ad_tag,
				'add_meta_boxes'
			);

			$this->loader->add_action(
				'save_post',
				$this->plugin->advertising->ad_tag,
				'save_ad_content_box',
				10,
				1
			);

			$this->loader->add_action(
				'wp_ajax_import_vast',
				$this->plugin->advertising->ad_tag,
				'ajax_import_vast'
			);

			$this->loader->add_action(
				'template_redirect',
				$this->plugin->advertising->ad_tag,
				'template_redirect'
			);

			$this->loader->add_filter(
				'manage_ad_tag_posts_columns',
				$this->plugin->advertising->ad_tag,
				'admin_post_table',
				10,
				1
			);

			$this->loader->add_action(
				'manage_ad_tag_posts_custom_column',
				$this->plugin->advertising->ad_tag,
				'admin_post_table_columns',
				10,
				2
			);

			$this->loader->add_action(
				'init',
				$this->plugin->advertising->ad_schedule,
				'post_type'
			);

			$this->loader->add_action(
				'add_meta_boxes',
				$this->plugin->advertising->ad_schedule,
				'add_meta_boxes'
			);

			$this->loader->add_action(
				'save_post',
				$this->plugin->advertising->ad_schedule,
				'save_ad_tags_box',
				10,
				1
			);

			$this->loader->add_action(
				'wp_ajax_get_schedule_tax_terms',
				$this->plugin->advertising->ad_schedule,
				'ajax_get_tax_terms'
			);

			$this->loader->add_action(
				'save_post',
				$this->plugin->advertising->ad_schedule,
				'clear_cache',
				100,
				1
			);

			$this->loader->add_action(
				'template_redirect',
				$this->plugin->advertising->ad_schedule,
				'load_vmap_template'
			);

			$this->loader->add_filter(
				'manage_ad_schedule_posts_columns',
				$this->plugin->advertising->ad_schedule,
				'admin_post_table',
				10,
				1
			);

			$this->loader->add_action(
				'manage_ad_schedule_posts_custom_column',
				$this->plugin->advertising->ad_schedule,
				'admin_post_table_columns',
				10,
				2
			);

			$this->loader->add_filter(
				'streamtube/player/file/setup',
				$this->plugin->advertising,
				'request_ads',
				5,
				2
			);

			$this->loader->add_filter(
				'streamtube/core/advertising/vast_tag_url',
				$this->plugin->advertising,
				'filter_ads_visibility',
				5,
				3
			);
		}
	}

	/**
	 * Register all of the hooks related to the post functionality
	 * of the plugin.
	 *
	 * @since    1.0.0
	 * @access   private
	 */
	private function define_post_hooks() {
		$this->plugin->post = new Streamtube_Core_Post();

		$this->loader->add_action(
			'init',
			$this->plugin->post,
			'cpt_video'
		);

		$this->loader->add_action(
			'init',
			$this->plugin->post,
			'new_post_statuses'
		);

		$this->loader->add_action(
			'streamtube/core/post/update',
			$this->plugin->post,
			'update_post_meta'
		);

		$this->loader->add_action(
			'delete_scheduled_posts',
			$this->plugin->post,
			'delete_scheduled_posts'
		);

		$this->loader->add_action(
			'after_delete_post',
			$this->plugin->post,
			'delete_delete_scheduled_posts',
			10,
			2
		);

		$this->loader->add_action(
			'save_post_video',
			$this->plugin->post,
			'sync_post_attachment',
			10,
			2
		);

		$this->loader->add_action(
			'wp_ajax_import_embed',
			$this->plugin->post,
			'ajax_import_embed'
		);

		$this->loader->add_action(
			'wp_ajax_add_video',
			$this->plugin->post,
			'ajax_add_video'
		);

		$this->loader->add_action(
			'wp_ajax_upload_video',
			$this->plugin->post,
			'ajax_upload_video'
		);


		$this->loader->add_action(
			'wp_ajax_upload_video_chunk',
			$this->plugin->post,
			'ajax_upload_video_chunk'
		);

		$this->loader->add_action(
			'wp_ajax_upload_video_chunks',
			$this->plugin->post,
			'ajax_upload_video_chunks'
		);

		$this->loader->add_action(
			'wp_ajax_update_post',
			$this->plugin->post,
			'ajax_update_post'
		);

		$this->loader->add_action(
			'wp_ajax_inline-save',
			$this->plugin->post,
			'ajax_update_post_inline',
			0
		);

		$this->loader->add_action(
			'save_post_video',
			$this->plugin->post,
			'update_video_source',
			10,
			2
		);

		$this->loader->add_filter(
			'wp_get_attachment_url',
			$this->plugin->post,
			'filter_wp_get_attachment_url',
			5,
			2
		);

		$this->loader->add_action(
			'wp_ajax_trash_post',
			$this->plugin->post,
			'ajax_trash_post'
		);

		$this->loader->add_action(
			'wp_ajax_approve_post',
			$this->plugin->post,
			'ajax_approve_post'
		);

		$this->loader->add_action(
			'wp_ajax_reject_post',
			$this->plugin->post,
			'ajax_reject_post'
		);

		$this->loader->add_action(
			'wp_ajax_restore_post',
			$this->plugin->post,
			'ajax_restore_post'
		);

		$this->loader->add_action(
			'wp_ajax_search_posts',
			$this->plugin->post,
			'ajax_search_posts'
		);

		$this->loader->add_action(
			'wp_ajax_report_video',
			$this->plugin->post,
			'ajax_report_video'
		);

		$this->loader->add_action(
			'wp_ajax_upload_text_track',
			$this->plugin->post,
			'ajax_upload_text_track'
		);

		$this->loader->add_action(
			'wp_ajax_update_text_tracks',
			$this->plugin->post,
			'ajax_update_text_tracks'
		);

		$this->loader->add_action(
			'wp_ajax_update_altsources',
			$this->plugin->post,
			'ajax_update_altsources'
		);

		$this->loader->add_action(
			'wp_ajax_update_embed_privacy',
			$this->plugin->post,
			'ajax_update_embed_privacy'
		);

		$this->loader->add_action(
			'wp_ajax_get_post_thumbnail',
			$this->plugin->post,
			'ajax_get_post_thumbnail'
		);

		$this->loader->add_action(
			'wp_ajax_get_post_by_url',
			$this->plugin->post,
			'ajax_get_post_by_url'
		);

		$this->loader->add_action(
			'wp_ajax_nopriv_get_post_by_url',
			$this->plugin->post,
			'ajax_get_post_by_url'
		);

		$this->loader->add_action(
			'wp_ajax_search_autocomplete',
			$this->plugin->post,
			'ajax_search_autocomplete'
		);

		$this->loader->add_action(
			'wp_ajax_nopriv_search_autocomplete',
			$this->plugin->post,
			'ajax_search_autocomplete'
		);

		$this->loader->add_action(
			'template_redirect',
			$this->plugin->post,
			'load_edit_template'
		);

		$this->loader->add_action(
			'wp',
			$this->plugin->post,
			'redirect_to_edit_page'
		);

		$this->loader->add_action(
			'wp_head',
			$this->plugin->post,
			'load_video_schema',
			1
		);

		$this->loader->add_action(
			'ajax_query_attachments_args',
			$this->plugin->post,
			'filter_ajax_query_attachments_args',
			10,
			1
		);

		$this->loader->add_action(
			'wp_insert_post',
			$this->plugin->post,
			'wp_insert_post',
			10,
			3
		);

		$this->loader->add_action(
			'wp',
			$this->plugin->post,
			'update_last_seen'
		);

		$this->loader->add_action(
			'before_delete_post',
			$this->plugin->post,
			'delete_attached_files',
			10,
			2
		);

		$this->loader->add_action(
			'delete_attachment',
			$this->plugin->post,
			'delete_attached_files',
			10,
			2
		);

		$this->loader->add_action(
			'template_redirect',
			$this->plugin->post,
			'attachment_template_redirect',
			10,
			1
		);

		$this->loader->add_filter(
			'streamtube_pre_player_args',
			$this->plugin->post,
			'filter_altsource',
			10,
			1
		);

		$this->loader->add_action(
			'streamtube/single/video/control',
			$this->plugin->post,
			'the_trailer_button',
			10
		);

		$this->loader->add_action(
			'streamtube/single/video/control',
			$this->plugin->post,
			'the_altsources_navigator',
			500
		);

		$this->loader->add_filter(
			'streamtube/player/file/setup',
			$this->plugin->post,
			'filter_player_setup_text_tracks',
			20,
			2
		);

		$this->loader->add_filter(
			'streamtube/player/file/output',
			$this->plugin->post,
			'filter_player_output',
			10,
			2
		);

		$this->loader->add_filter(
			'streamtube/player/embed/output',
			$this->plugin->post,
			'filter_player_output',
			10,
			2
		);

		$this->loader->add_filter(
			'streamtube/player/file/output',
			$this->plugin->post,
			'display_embed_privacy_notice',
			100,
			2
		);

		$this->loader->add_filter(
			'streamtube/player/embed/output',
			$this->plugin->post,
			'display_embed_privacy_notice',
			100,
			2
		);

		$this->loader->add_filter(
			'streamtube/player/file/output',
			$this->plugin->post,
			'display_upcoming_notice',
			200,
			2
		);

		$this->loader->add_filter(
			'streamtube/player/embed/output',
			$this->plugin->post,
			'display_upcoming_notice',
			200,
			2
		);

		$this->loader->add_filter(
			'streamtube/core/player/upcoming/countdown/before',
			$this->plugin->post,
			'display_upcoming_notice_heading',
			10,
			3
		);

		$this->loader->add_action(
			'streamtube/post/meta/item/before',
			$this->plugin->post,
			'display_membership_badge',
			10
		);

		$this->loader->add_filter(
			'manage_video_posts_columns',
			$this->plugin->post,
			'filter_post_table',
			1000,
			1
		);

		$this->loader->add_filter(
			'manage_video_posts_custom_column',
			$this->plugin->post,
			'filter_post_table_columns',
			10,
			2
		);

		$this->loader->add_filter(
			'post_row_actions',
			$this->plugin->post,
			'filter_post_row_actions',
			10,
			2
		);

		$this->loader->add_filter(
			'page_row_actions',
			$this->plugin->post,
			'filter_post_row_actions',
			10,
			2
		);

		$this->loader->add_filter(
			'get_edit_post_link',
			$this->plugin->post,
			'filter_get_edit_post_link',
			9999,
			3
		);

		$this->loader->add_filter(
			'get_delete_post_link',
			$this->plugin->post,
			'filter_get_delete_post_link',
			9999,
			4
		);

		$this->loader->add_filter(
			'quick_edit_dropdown_pages_args',
			$this->plugin->post,
			'filter_quick_edit_dropdown_pages_args',
			20,
			2
		);

		$this->loader->add_filter(
			'theme_post_templates',
			$this->plugin->post,
			'filter_theme_post_templates',
			100,
			4
		);

		$this->loader->add_action(
			'init',
			$this->plugin->post,
			'add_post_type_supports',
			9999
		);

		$this->loader->add_filter(
			'register_post_type_args',
			$this->plugin->post,
			'filter_register_post_type_args',
			9999,
			2
		);

		$this->loader->add_action(
			'pre_get_posts',
			$this->plugin->post,
			'filter_order_by_date_desc',
		);

		$this->loader->add_action(
			'streamtube/core/register_modules',
			$this->plugin->post,
			'register_permission_module'
		);
	}

	/**
	 * Register all of the hooks related to the video functionality
	 * of the plugin.
	 *
	 * @since    1.0.0
	 * @access   private
	 */
	private function define_video_hooks() {
		$this->plugin->video = new Streamtube_Core_Video();

		$this->loader->add_filter(
			'post_embed_url',
			$this->plugin->video,
			'filter_embed_url',
			10,
			2
		);

		$this->loader->add_action(
			'embed_html',
			$this->plugin->video,
			'filter_embed_html',
			100,
			4
		);

		$this->loader->add_action(
			'oembed_response_data',
			$this->plugin->video,
			'filter_embed_type',
			100,
			4
		);

		$this->loader->add_action(
			'streamtube/single/video/control',
			$this->plugin->video,
			'load_button_share',
			100
		);

		$this->loader->add_action(
			'wp_footer',
			$this->plugin->video,
			'load_modal_share'
		);

		$this->loader->add_action(
			'streamtube/single/video/control',
			$this->plugin->video,
			'load_button_report',
			200
		);

		$this->loader->add_action(
			'wp_footer',
			$this->plugin->video,
			'load_modal_report'
		);

		$this->loader->add_action(
			'streamtube/single/video/meta',
			$this->plugin->video,
			'load_single_post_date'
		);

		$this->loader->add_action(
			'streamtube/single/video/meta',
			$this->plugin->video,
			'load_single_post_comment_count'
		);

		$this->loader->add_action(
			'streamtube/single/video/meta',
			$this->plugin->video,
			'load_single_post_terms'
		);

		$this->loader->add_action(
			'streamtube/archive/video/page_header/before',
			$this->plugin->video,
			'load_the_archive_term_menu'
		);

		$this->loader->add_filter(
			'streamtube/archive/video/query_args',
			$this->plugin->video,
			'load_portrait_video_tags',
			10,
			1
		);

		$this->plugin->download_file = new StreamTube_Core_Download_File();

		$this->loader->add_action(
			'streamtube/single/video/control',
			$this->plugin->download_file,
			'button_download',
			5
		);

		$this->loader->add_action(
			'template_redirect',
			$this->plugin->download_file,
			'process_download'
		);

	}

	/**
	 * Register all of the hooks related to the player functionality
	 * of the plugin.
	 *
	 * @since    1.0.0
	 * @access   private
	 */
	private function define_player_hooks() {
		$this->plugin->player = new Streamtube_Core_Player();

		$this->loader->add_filter(
			'streamtube/player/file/setup',
			$this->plugin->player,
			"set_builtin_events",
			10,
			2
		);

		$this->loader->add_filter(
			'streamtube/player/file/setup',
			$this->plugin->player,
			"set_topbar",
			10,
			2
		);

		$this->loader->add_filter(
			'streamtube/player/file/setup',
			$this->plugin->player,
			"set_share_box",
			10,
			2
		);

		$this->loader->add_filter(
			'streamtube/player/file/setup',
			$this->plugin->player,
			"set_skin",
			10,
			2
		);

		$this->loader->add_filter(
			'streamtube/player/file/setup',
			$this->plugin->player,
			"set_inactivity_timeout",
			10,
			2
		);

		$this->loader->add_filter(
			'streamtube/player/file/setup',
			$this->plugin->player,
			"set_language",
			10,
			2
		);

		$this->loader->add_filter(
			'streamtube/player/file/setup',
			$this->plugin->player,
			"set_watermark",
			10,
			2
		);

		$this->loader->add_filter(
			'streamtube/player/file/setup',
			$this->plugin->player,
			"set_controlbar_watermark",
			10,
			2
		);

		$this->loader->add_filter(
			'streamtube/player/file/setup',
			$this->plugin->player,
			"set_vr",
			10,
			2
		);

		$this->loader->add_filter(
			'streamtube/player/file/setup',
			$this->plugin->player,
			"set_right_click_blocker",
			10,
			2
		);

		$this->loader->add_filter(
			'streamtube/player/file/setup',
			$this->plugin->player,
			"set_playback_rates",
			10,
			2
		);

		$this->loader->add_filter(
			'streamtube/player/file/setup',
			$this->plugin->player,
			"set_landscape_mode",
			10,
			2
		);

		$this->loader->add_filter(
			'streamtube/player/file/setup',
			$this->plugin->player,
			"set_hotkeys",
			10,
			2
		);

		$this->loader->add_filter(
			'streamtube/player/file/setup',
			$this->plugin->player,
			"set_start_at",
			10,
			2
		);

		$this->loader->add_filter(
			'streamtube/player/file/setup',
			$this->plugin->player,
			"set_volume",
			10,
			2
		);

		$this->loader->add_filter(
			'streamtube/player/file/setup',
			$this->plugin->player,
			"set_pause_simultaneous",
			10,
			2
		);

		$this->loader->add_filter(
			'streamtube/player/file/setup',
			$this->plugin->player,
			"set_forward_backward",
			10,
			2
		);

		$this->loader->add_filter(
			'streamtube/player/file/setup',
			$this->plugin->player,
			"set_idm_detector",
			9999,
			2
		);

		$this->loader->add_filter(
			'wp_video_shortcode',
			$this->plugin->player,
			'override_wp_video_shortcode',
			10,
			4
		);

		$this->loader->add_filter(
			'render_block',
			$this->plugin->player,
			'override_wp_video_block',
			10,
			2
		);

		$this->loader->add_filter(
			'render_block',
			$this->plugin->player,
			'override_wp_youtube_block',
			10,
			2
		);

		$this->loader->add_filter(
			'the_content',
			$this->plugin->player,
			"convert_youtube_to_videojs",
			5,
			1
		);

		$this->loader->add_action(
			'wp_ajax_load_video_source',
			$this->plugin->player,
			"ajax_load_video_source"
		);

		$this->loader->add_action(
			'wp_ajax_nopriv_load_video_source',
			$this->plugin->player,
			"ajax_load_video_source"
		);
	}

	/**
	 * Register all of the hooks related to the content timestemp functionality
	 * of the plugin.
	 *
	 * @since    1.0.0
	 * @access   private
	 */
	private function define_timestamp_hooks() {

		if ( ! $this->get_license() ) {
			return false;
		}

		$this->include_file( 'third-party/content-timestamp/class-streamtube-core-content-timestamp.php' );

		$this->plugin->timestamp = new StreamTube_Core_Content_TimesTamp();

		$this->loader->add_filter(
			'the_content',
			$this->plugin->timestamp,
			'filter_content',
			100,
			1
		);

		$this->loader->add_filter(
			'comment_text',
			$this->plugin->timestamp,
			'filter_content',
			100,
			1
		);

		$this->loader->add_filter(
			'streamtube/player/file/setup',
			$this->plugin->timestamp,
			'filter_player_setup',
			30,
			2
		);
	}

	private function define_collection_hooks() {

		$this->include_file( 'third-party/collection/class-streamtube-core-collection.php' );

		$this->plugin->collection = new Streamtube_Core_Collection();

		$taxonomy = $this->plugin->collection::TAX_COLLECTION;

		$this->loader->add_action(
			'init',
			$this->plugin->collection,
			'register_taxonomy'
		);

		$this->loader->add_filter(
			'streamtube/core/taxonomy/builtin',
			$this->plugin->collection,
			'add_builtin_taxonomy'
		);

		$this->loader->add_action(
			'admin_init',
			$this->plugin->collection,
			'add_role_capabilities'
		);

		$this->loader->add_filter(
			'get_term',
			$this->plugin->collection,
			'_filter_term',
			10,
			2
		);

		$this->loader->add_filter(
			'pre_insert_term',
			$this->plugin->collection,
			'filter_pre_insert_term',
			10,
			3
		);

		$this->loader->add_action(
			"{$taxonomy}_add_form_fields",
			$this->plugin->taxonomy,
			'add_thumbnail_field',
			10,
			1
		);

		$this->loader->add_action(
			"{$taxonomy}_edit_form_fields",
			$this->plugin->taxonomy,
			'edit_thumbnail_field',
			10,
			2
		);

		$this->loader->add_action(
			"created_{$taxonomy}",
			$this->plugin->taxonomy,
			'update_thumbnail_field',
			10,
			1
		);

		$this->loader->add_action(
			"edited_{$taxonomy}",
			$this->plugin->taxonomy,
			'update_thumbnail_field',
			10,
			1
		);

		$this->loader->add_filter(
			"manage_edit-{$taxonomy}_columns",
			$this->plugin->taxonomy,
			'add_thumbnail_column',
			10,
			1
		);

		$this->loader->add_filter(
			"manage_{$taxonomy}_custom_column",
			$this->plugin->taxonomy,
			'add_thumbnail_column_content',
			10,
			3
		);

		// meta fields.
		$this->loader->add_action(
			"{$taxonomy}_add_form_fields",
			$this->plugin->collection,
			'admin_add_term_meta_field',
			10,
			1
		);

		$this->loader->add_action(
			"{$taxonomy}_edit_form_fields",
			$this->plugin->collection,
			'admin_edit_term_meta_field',
			10,
			2
		);

		$this->loader->add_action(
			"created_{$taxonomy}",
			$this->plugin->collection,
			'admin_update_term_meta_fields',
			10,
			1
		);

		$this->loader->add_action(
			"edited_{$taxonomy}",
			$this->plugin->collection,
			'admin_update_term_meta_fields',
			10,
			1
		);

		$this->loader->add_filter(
			"manage_edit-{$taxonomy}_columns",
			$this->plugin->collection,
			'filter_terms_list_table_columns',
			100,
			1
		);

		$this->loader->add_filter(
			"manage_{$taxonomy}_custom_column",
			$this->plugin->collection,
			'filter_terms_list_table_columns_content',
			100,
			3
		);

		$this->loader->add_filter(
			"term_name",
			$this->plugin->collection,
			'filter_term_name',
			10,
			2
		);

		$this->loader->add_filter(
			"get_terms_args",
			$this->plugin->collection,
			'filter_get_terms_args',
			10,
			2
		);

		$this->loader->add_action(
			'delete_user',
			$this->plugin->collection,
			'admin_delete_user_collections',
			10,
			3
		);

		$this->loader->add_action(
			'streamtube/post/thumbnail/after',
			$this->plugin->collection,
			'frontend_the_watch_later_button',
			10
		);

		$this->loader->add_action(
			'streamtube/flat_post/item',
			$this->plugin->collection,
			'frontend_the_watch_later_button',
			10
		);

		if ( $this->get_license() ) {
			$this->loader->add_action(
				'streamtube/single/video/control',
				$this->plugin->collection,
				'frontend_the_collection_button'
			);
		}

		$this->loader->add_action(
			'wp_footer',
			$this->plugin->collection,
			'frontend_the_collection_modal'
		);

		$this->loader->add_action(
			"wp",
			$this->plugin->collection,
			'frontend_add_post_history'
		);

		$this->loader->add_filter(
			"streamtube/core/post/edit/content/after",
			$this->plugin->collection,
			'frontend_post_form_collections_box',
			10,
			1
		);

		$this->loader->add_filter(
			"streamtube/core/user/dashboard/menu/items",
			$this->plugin->collection,
			'frontend_dashboard_menu',
			10,
			1
		);

		$this->loader->add_filter(
			"streamtube/core/user/profile/menu/items",
			$this->plugin->collection,
			'frontend_profile_menu',
			10,
			1
		);

		$this->loader->add_action(
			"streamtube/archive/video/page_title/after",
			$this->plugin->collection,
			'frontend_the_button_play_all',
			10
		);

		$this->loader->add_action(
			"added_term_relationship",
			$this->plugin->collection,
			'_add_term_order',
			10,
			3
		);

		$this->loader->add_action(
			"deleted_term_relationships",
			$this->plugin->collection,
			'_remove_term_order',
			10,
			3
		);

		$this->loader->add_action(
			"deleted_term_taxonomy",
			$this->plugin->collection,
			'_clean_term_orders',
			10,
			1
		);

		$this->loader->add_action(
			"wp_login",
			$this->plugin->collection,
			'auto_create_user_terms',
			10,
			2
		);

		$this->loader->add_action(
			"saved_term",
			$this->plugin->collection,
			'saved_term',
			10,
			3
		);

		$this->loader->add_action(
			"wp_ajax_create_collection",
			$this->plugin->collection,
			'ajax_create_collection'
		);

		$this->loader->add_action(
			"wp_ajax_delete_collection",
			$this->plugin->collection,
			'ajax_delete_collection'
		);

		$this->loader->add_action(
			"wp_ajax_get_collection_term",
			$this->plugin->collection,
			'ajax_get_collection_term'
		);

		$this->loader->add_action(
			"wp_ajax_set_post_collection",
			$this->plugin->collection,
			'ajax_set_post_collection'
		);

		$this->loader->add_action(
			"wp_ajax_update_collection_item_index",
			$this->plugin->collection,
			'ajax_update_collection_item_index'
		);

		$this->loader->add_action(
			"wp_ajax_set_post_watch_later",
			$this->plugin->collection,
			'ajax_set_post_watch_later'
		);

		$this->loader->add_action(
			"wp_ajax_set_image_collection",
			$this->plugin->collection,
			'ajax_set_image_collection'
		);

		$this->loader->add_action(
			"wp_ajax_upload_collection_thumbnail_image",
			$this->plugin->collection,
			'ajax_upload_collection_thumbnail_image'
		);

		$this->loader->add_action(
			"wp_ajax_set_collection_status",
			$this->plugin->collection,
			'ajax_set_collection_status'
		);

		$this->loader->add_action(
			"wp_ajax_set_collection_activity",
			$this->plugin->collection,
			'ajax_set_collection_activity'
		);

		$this->loader->add_action(
			"wp_ajax_clear_collection",
			$this->plugin->collection,
			'ajax_clear_collection'
		);

		$this->loader->add_action(
			"wp_ajax_search_videos",
			$this->plugin->collection,
			'ajax_search_videos'
		);

		$this->loader->add_filter(
			'streamtube/player/file/setup',
			$this->plugin->collection,
			'filter_player_setup',
			30,
			2
		);

		$this->loader->add_filter(
			'streamtube_get_share_embed_permalink',
			$this->plugin->collection,
			'filter_share_links',
			100,
			2
		);

		$this->loader->add_filter(
			'streamtube/core/share/permalink',
			$this->plugin->collection,
			'filter_share_links',
			100,
			2
		);

		$this->loader->add_filter(
			'post_embed_url',
			$this->plugin->collection,
			'filter_embed_url',
			10,
			2
		);

		$this->loader->add_filter(
			'streamtube/core/terms_list_table/args',
			$this->plugin->collection,
			'filter_terms_list_table_args'
		);

		$this->loader->add_filter(
			'streamtube/core/terms_list_table/errors',
			$this->plugin->collection,
			'check_term_list_table_edit_term'
		);
	}

	private function define_mobile_bottom_bar() {

		if ( ! $this->get_license() ) {
			return false;
		}

		$this->include_file( 'third-party/mobile-bottom-bar/class-streamtube-core-mobile-bottom-bar.php' );

		$this->plugin->mobile_bottom_bar = new StreamTube_Core_Mobile_Bottom_Bar();

		$this->loader->add_filter(
			'body_class',
			$this->plugin->mobile_bottom_bar,
			'filter_body_classes'
		);

		$this->loader->add_filter(
			'login_body_class',
			$this->plugin->mobile_bottom_bar,
			'filter_body_classes'
		);

		$this->loader->add_action(
			'after_setup_theme',
			$this->plugin->mobile_bottom_bar,
			'register_nav_menus'
		);

		$this->loader->add_action(
			'streamtube/footer/after',
			$this->plugin->mobile_bottom_bar,
			'the_bar'
		);
	}

	/**
	 * Register all of the hooks related to the comment functionality
	 * of the plugin.
	 *
	 * @since    1.0.0
	 * @access   private
	 */
	private function define_comment_hooks() {
		$this->plugin->comment = new Streamtube_Core_Comment();

		$this->loader->add_action(
			'streamtube/comment_list/comment/meta/right',
			$this->plugin->comment,
			'load_control_buttons',
			10,
			3
		);

		$this->loader->add_action(
			'wp_ajax_nopriv_post_comment',
			$this->plugin->comment,
			'ajax_post_comment'
		);

		$this->loader->add_action(
			'wp_ajax_post_comment',
			$this->plugin->comment,
			'ajax_post_comment'
		);

		$this->loader->add_action(
			'wp_ajax_get_comment',
			$this->plugin->comment,
			'ajax_get_comment'
		);

		$this->loader->add_action(
			'wp_ajax_get_comment_to_edit',
			$this->plugin->comment,
			'ajax_get_comment'
		);

		$this->loader->add_action(
			'wp_ajax_get_comment_to_report',
			$this->plugin->comment,
			'ajax_get_comment_to_report'
		);

		$this->loader->add_action(
			'wp_ajax_edit_comment',
			$this->plugin->comment,
			'ajax_edit_comment'
		);

		$this->loader->add_action(
			'wp_ajax_report_comment',
			$this->plugin->comment,
			'ajax_report_comment'
		);

		$this->loader->add_action(
			'streamtube/core/comment/reported',
			$this->plugin->comment,
			'report_comment_notify',
			10,
			2
		);

		$this->loader->add_action(
			'wp_ajax_remove_comment_report',
			$this->plugin->comment,
			'ajax_remove_comment_report'
		);

		$this->loader->add_action(
			'wp_ajax_moderate_comment',
			$this->plugin->comment,
			'ajax_moderate_comment'
		);

		$this->loader->add_action(
			'wp_ajax_trash_comment',
			$this->plugin->comment,
			'ajax_trash_comment'
		);

		$this->loader->add_action(
			'wp_ajax_spam_comment',
			$this->plugin->comment,
			'ajax_spam_comment'
		);

		$this->loader->add_action(
			'wp_ajax_load_more_comments',
			$this->plugin->comment,
			'ajax_load_more_comments'
		);

		$this->loader->add_action(
			'wp_ajax_nopriv_load_more_comments',
			$this->plugin->comment,
			'ajax_load_more_comments'
		);

		$this->loader->add_action(
			'wp_ajax_load_comments',
			$this->plugin->comment,
			'ajax_load_comments'
		);

		$this->loader->add_action(
			'wp_ajax_nopriv_load_comments',
			$this->plugin->comment,
			'ajax_load_comments'
		);

		$this->loader->add_filter(
			'comment_text',
			$this->plugin->comment,
			'filter_reported_comment_content',
			1,
			3
		);

		$this->loader->add_filter(
			'streamtube/comment/form_args',
			$this->plugin->comment,
			'filter_comment_form_args'
		);

		$this->loader->add_filter(
			'comment_class',
			$this->plugin->comment,
			'filter_comment_classes',
			10,
			5
		);

		$this->loader->add_filter(
			'comments_list_table_query_args',
			$this->plugin->comment,
			'filter_comments_list_table_query_args',
			10,
			1
		);

		$this->loader->add_filter(
			'comment_row_actions',
			$this->plugin->comment,
			'filter_comment_row_actions',
			10,
			2
		);

		$this->loader->add_filter(
			'manage_product_page_product-reviews_columns',
			$this->plugin->comment,
			'filter_product_reviews_table_columns',
			9999,
			1
		);

		$this->loader->add_filter(
			'comments_template',
			$this->plugin->comment,
			'load_ajax_comments_template'
		);

		$this->loader->add_filter(
			'comment_email',
			$this->plugin->comment,
			'hide_commenter_email_table_list',
			10,
			2
		);

	}

	/**
	 * Register all of the hooks related to the user functionality
	 * of the plugin.
	 *
	 * @since    1.0.0
	 * @access   private
	 */
	private function define_user_hooks() {

		$this->plugin->user = new Streamtube_Core_User();

		$this->loader->add_filter(
			'get_avatar_url',
			$this->plugin->user,
			'get_avatar_url',
			10,
			3
		);

		$this->loader->add_action(
			'register_form',
			$this->plugin->user,
			'build_form_registration'
		);

		$this->loader->add_action(
			'registration_errors',
			$this->plugin->user,
			'verify_registration_data',
			10,
			1
		);

		$this->loader->add_action(
			'register_new_user',
			$this->plugin->user,
			'save_form_registration',
			9999,
			1
		);

		$this->loader->add_action(
			'wp_ajax_verify_user',
			$this->plugin->user,
			'ajax_verify_user'
		);

		$this->loader->add_action(
			'wp_ajax_delete_user_photo',
			$this->plugin->user,
			'ajax_delete_user_photo'
		);

		$this->loader->add_action(
			'wp_ajax_update_advertising',
			$this->plugin->user,
			'ajax_update_advertising'
		);

		$this->loader->add_action(
			'streamtube/core/advertising/vast_tag_url',
			$this->plugin->user,
			'load_vast_tag_url',
			10,
			3
		);

		$this->loader->add_action(
			'parse_request',
			$this->plugin->user,
			'fix_paged_request',
			9999
		);

		$this->loader->add_action(
			'pre_user_query',
			$this->plugin->user,
			'filter_users_by_post_type_count',
			10,
			1
		);

		$this->plugin->user_profile = new Streamtube_Core_User_Profile();

		$this->loader->add_action(
			'init',
			$this->plugin->user_profile,
			'widgets_init',
			50
		);

		$this->loader->add_action(
			'init',
			$this->plugin->user_profile,
			'add_endpoints',
			100
		);

		$this->loader->add_action(
			'streamtube/core/user/header',
			$this->plugin->user_profile,
			'the_header',
			10
		);

		$this->loader->add_action(
			'streamtube/core/user/header',
			$this->plugin->user_profile,
			'the_navigation', 20
		);

		$this->loader->add_action(
			'streamtube/core/user/main',
			$this->plugin->user_profile,
			'the_main'
		);

		$this->loader->add_filter(
			'template_redirect',
			$this->plugin->user_profile,
			'the_index',
			999
		);

		$this->loader->add_action(
			'streamtube/core/user/header/display_name/after',
			$this->plugin->user_profile,
			'the_action_buttons'
		);

		$this->loader->add_action(
			'streamtube/single/video/author/after',
			$this->plugin->user_profile,
			'the_action_buttons'
		);

		$this->loader->add_action(
			'streamtube/core/user/profile/content',
			$this->plugin->user_profile,
			'display_user_bio',
			20
		);

		$this->loader->add_action(
			'wp_enqueue_scripts',
			$this->plugin->user_profile,
			'enqueue_scripts',
			20
		);

		$this->plugin->user_dashboard = new Streamtube_Core_User_Dashboard();

		$this->loader->add_action(
			'after_setup_theme',
			$this->plugin->user_dashboard,
			'register_dashboard_menu_location'
		);

		$this->loader->add_filter(
			'streamtube/core/user/dashboard/menu/items',
			$this->plugin->user_dashboard,
			'register_wp_menu_items'
		);

		$this->loader->add_filter(
			'streamtube/core/user/dashboard/menu/items',
			$this->plugin->user_dashboard,
			'register_menu_post_types'
		);

		$this->loader->add_filter(
			'streamtube/core/user/dashboard/menu/items',
			$this->plugin->user_dashboard,
			'register_profile_menu',
			0
		);

		$this->loader->add_action(
			'init',
			$this->plugin->user_dashboard,
			'add_endpoints',
			100
		);

		$this->loader->add_action(
			'streamtube/core/user/dashboard/sidebar',
			$this->plugin->user_dashboard,
			'the_menu_template'
		);

		$this->loader->add_action(
			'streamtube/core/user/dashboard/main',
			$this->plugin->user_dashboard,
			'the_main_template'
		);

		$this->loader->add_action(
			'template_redirect',
			$this->plugin->user_dashboard,
			'the_index_template',
			9999
		);

		$this->loader->add_action(
			'streamtube_add_meta_boxes',
			$this->plugin->user_dashboard,
			'add_widget_metaboxes'
		);

		$this->loader->add_action(
			'wp_ajax_streamtube/dismiss_permissions_warning',
			$this->plugin->user_dashboard,
			'dismiss_permissions_warning'
		);

		$this->loader->add_action(
			'streamtube/core/user/dashboard/page_content/before',
			$this->plugin->user_dashboard,
			'the_permission_warning'
		);

		$this->loader->add_action(
			'login_redirect',
			$this->plugin->user_dashboard,
			'login_redirect',
			10,
			3
		);

		$this->loader->add_action(
			'wp',
			$this->plugin->user_dashboard,
			'dashboard_redirect',
			9999
		);

		$this->loader->add_filter(
			'body_class',
			$this->plugin->user_dashboard,
			'add_js_body_class',
			9999
		);

		$this->loader->add_action(
			'wp_enqueue_scripts',
			$this->plugin->user_dashboard,
			'enqueue_scripts'
		);

		$this->plugin->user_privacy = new Streamtube_Core_User_Privacy();

		$this->loader->add_action(
			'wp_ajax_streamtube/delete_user',
			$this->plugin->user_privacy,
			'ajax_delete_user'
		);

		$this->loader->add_action(
			'wp_ajax_streamtube/delete_users',
			$this->plugin->user_privacy,
			'ajax_delete_users'
		);

		$this->loader->add_action(
			'wp_ajax_streamtube/change_user_password',
			$this->plugin->user_privacy,
			'ajax_change_user_password'
		);

		$this->loader->add_action(
			'wp_ajax_streamtube/update_user_capabilities',
			$this->plugin->user_privacy,
			'ajax_update_user_capabilities'
		);

		$this->loader->add_action(
			'wp_ajax_streamtube/add_user_roles',
			$this->plugin->user_privacy,
			'ajax_add_user_roles'
		);

		$this->loader->add_action(
			'wp_ajax_deactivate_account',
			$this->plugin->user_privacy,
			'ajax_deactivate'
		);

		$this->loader->add_action(
			'wp_ajax_reactivate_account',
			$this->plugin->user_privacy,
			'ajax_reactivate'
		);

		$this->loader->add_action(
			'wp_ajax_admin_deactivate_user',
			$this->plugin->user_privacy,
			'ajax_admin_deactivate_user'
		);

		$this->loader->add_action(
			'delete_expired_transients',
			$this->plugin->user_privacy,
			'schedule_delete_users'
		);

		$this->loader->add_action(
			'manage_frontend_users_list_table_do_action',
			$this->plugin->user_privacy,
			'handle_users_list_bulk_action',
			10,
			3
		);

		$this->loader->add_filter(
			'streamtube/core/user/dashboard/menu/items',
			$this->plugin->user_privacy,
			'add_dashboard_settings_menu'
		);

		$this->loader->add_filter(
			'streamtube/core/user/dashboard/menu/items',
			$this->plugin->user_privacy,
			'filter_dashboard_menu_item',
			9999
		);

		$this->loader->add_filter(
			'streamtube/core/user/profile/menu/items',
			$this->plugin->user_privacy,
			'filter_profile_menu_item',
			9999
		);

		$this->loader->add_filter(
			'streamtube/core/user/dashboard/page_header/before',
			$this->plugin->user_privacy,
			'display_notices'
		);

		$this->loader->add_filter(
			'user_row_actions',
			'StreamTube_Core_Users_List_Table',
			'filter_user_row_actions',
			9999,
			2
		);

	}

	/**
	 *
	 * Define rest hooks
	 * 
	 * @since 1.0.0
	 */
	private function define_rest_hooks() {

		$this->plugin->rest_api = array();

		$this->plugin->rest_api['post']           = new StreamTube_Core_Post_Rest_API();
		$this->plugin->rest_api['generate_image'] = new StreamTube_Core_Generate_Image_Rest_Controller();
		$this->plugin->rest_api['subtitles']      = new StreamTube_Core_Subtitles_Rest_Controller();
		$this->plugin->rest_api['user']           = new StreamTube_Core_User_Rest_Controller();
		$this->plugin->rest_api['upload']         = new StreamTube_Core_Rest_Upload_Controller();
		$this->plugin->rest_api['embed']          = new StreamTube_Core_Rest_Embed_Controller();

		foreach ( $this->plugin->rest_api as $rest => $object ) {
			$this->loader->add_action(
				'rest_api_init',
				$object,
				'rest_api_init'
			);
		}
	}

	private function define_googlesitekit_hooks() {
		/**
		 * The class responsible for defining Google Sitekit.
		 */
		$this->include_file( 'third-party/googlesitekit/class-streamtube-core-googlesitekit.php' );

		/**
		 * The class responsible for defining Analytics module of Google Sitekit.
		 */
		$this->include_file( 'third-party/googlesitekit/class-streamtube-core-googlesitekit-analytics.php' );

		/**
		 * The class responsible for defining Tag Manager module of Google Sitekit.
		 */
		$this->include_file( 'third-party/googlesitekit/class-streamtube-core-googlesitekit-tag-manager.php' );

		/**
		 * The class responsible for defining Search Console module of Google Sitekit.
		 */
		$this->include_file( 'third-party/googlesitekit/class-streamtube-core-googlesitekit-search-console.php' );

		/**
		 * The class responsible for defining sitekit rest API
		 */
		$this->include_file( 'third-party/googlesitekit/class-streamtube-core-rest-googlesitekit-controller.php' );

		$this->plugin->googlesitekit = new stdClass();

		$this->plugin->googlesitekit->analytics = new Streamtube_Core_GoogleSiteKit_Analytics();

		if ( $this->plugin->googlesitekit->analytics->is_connected() && $this->get_license() ) {

			$this->loader->add_action(
				'init',
				$this->plugin->googlesitekit->analytics,
				'add_post_type_support_analytics',
				9999
			);

			$this->loader->add_action(
				'wp_enqueue_scripts',
				$this->plugin->googlesitekit->analytics,
				'enqueue_scripts'
			);

			$this->loader->add_action(
				'enqueue_embed_scripts',
				$this->plugin->googlesitekit->analytics,
				'enqueue_embed_scripts'
			);

			$this->loader->add_action(
				'streamtube_check_pageviews',
				$this->plugin->googlesitekit->analytics,
				'cron_update_post_list_pageviews',
				10
			);

			$this->loader->add_action(
				'streamtube_check_videoviews',
				$this->plugin->googlesitekit->analytics,
				'cron_update_post_list_videoviews',
				10
			);

			if ( get_option( 'sitekit_heartbeat_tick', 'on' ) ) {
				$this->loader->add_filter(
					'heartbeat_send',
					$this->plugin->googlesitekit->analytics,
					'heartbeat_tick',
					10,
					2
				);
			}

			$this->loader->add_action(
				'streamtube/single/video/manage/control',
				$this->plugin->googlesitekit->analytics,
				'button_analytics',
				100
			);

			$this->loader->add_action(
				'streamtube_add_meta_boxes',
				$this->plugin->googlesitekit->analytics,
				'add_meta_boxes'
			);

			$this->loader->add_filter(
				'streamtube/core/user/dashboard/menu/items',
				$this->plugin->googlesitekit->analytics,
				'register_dashboard_menu'
			);

			$this->loader->add_filter(
				'streamtube_core_get_edit_post_nav_items',
				$this->plugin->googlesitekit->analytics,
				'add_post_nav_item',
				10,
				2
			);

			$this->loader->add_filter(
				'post_row_actions',
				$this->plugin->googlesitekit->analytics,
				'filter_post_row_actions',
				10,
				2
			);

			$this->loader->add_filter(
				'page_row_actions',
				$this->plugin->googlesitekit->analytics,
				'filter_post_row_actions',
				10,
				2
			);

			$this->loader->add_filter(
				'streamtube/core/get_current_screen',
				$this->plugin->googlesitekit->analytics,
				'filter_current_screen',
				10,
				2
			);

			$this->loader->add_action(
				'streamtube/single/video/meta',
				$this->plugin->googlesitekit->analytics,
				'load_single_post_views'
			);

			$this->loader->add_filter(
				'manage_video_posts_columns',
				$this->plugin->googlesitekit->analytics,
				'filter_post_table',
				10,
				1
			);

			$this->loader->add_action(
				'manage_video_posts_custom_column',
				$this->plugin->googlesitekit->analytics,
				'filter_post_table_columns',
				10,
				2
			);

			/**
			 * The class responsible for defining analytics rest API
			 */
			$this->include_file( 'third-party/googlesitekit/class-streamtube-core-rest-googlesitekit-analytics-controller.php' );

			$this->plugin->googlesitekit->analytics_rest_api = new StreamTube_Core_GoogleSiteKit_Analytics_Rest_Controller();

			$this->loader->add_action(
				'rest_api_init',
				$this->plugin->googlesitekit->analytics_rest_api,
				'rest_api_init'
			);
		}

		$this->plugin->googlesitekit->tag_manager = new Streamtube_Core_GoogleSiteKit_Tag_Manager();

		if ( $this->plugin->googlesitekit->tag_manager->is_connected() ) {

			$this->loader->add_action(
				'enqueue_embed_scripts',
				$this->plugin->googlesitekit->tag_manager,
				'enqueue_embed_scripts'
			);

			$this->loader->add_filter(
				'streamtube/player/file/setup',
				$this->plugin->googlesitekit->tag_manager,
				'player_tracker',
				10,
				2
			);

		}

		$this->plugin->googlesitekit->search_console = new Streamtube_Core_GoogleSiteKit_Search_Console();

		if ( $this->plugin->googlesitekit->search_console->is_connected() ) {
			$this->loader->add_action(
				'wp_enqueue_scripts',
				$this->plugin->googlesitekit->search_console,
				'enqueue_scripts'
			);

			$this->loader->add_action(
				'streamtube_add_meta_boxes',
				$this->plugin->googlesitekit->search_console,
				'add_meta_boxes'
			);
		}
	}

	/**
	 *
	 * Define Big File Uploads hook
	 * 
	 */
	private function define_big_file_uploads_hooks() {
		$this->include_file( 'third-party/big-file-uploads/class-streamtube-core-big-file-uploads.php' );

		$this->plugin->big_file_uploads = new StreamTube_Core_Big_File_Uploads();

		$this->loader->add_filter(
			'upload_size_limit',
			$this->plugin->big_file_uploads,
			'filter_max_upload_size',
			30
		);

	}

	/**
	 *
	 * myCred Hooks
	 * 
	 * @since 1.0.9
	 */
	private function define_mycred_hooks() {

		if ( ! $this->get_license() ) {
			return;
		}

		/**
		 * The class responsible for defining myCred functions
		 */
		$this->include_file( 'third-party/mycred/class-streamtube-core-mycred.php' );

		$this->plugin->myCRED = new Streamtube_Core_myCRED();

		$this->loader->add_action(
			'widgets_init',
			'Streamtube_Core_myCRED_Widget_Buy_Points',
			'register'
		);

		$this->loader->add_action(
			'mycred_log_row_classes',
			$this->plugin->myCRED,
			'filter_log_row_classes',
			10,
			2
		);

		$this->loader->add_action(
			'mycred_log_username',
			$this->plugin->myCRED,
			'filter_mycred_log_username',
			100,
			3
		);

		$this->loader->add_action(
			'mycred_log_entry',
			$this->plugin->myCRED,
			'filter_mycred_log_entry',
			10,
			3
		);

		$this->loader->add_action(
			'streamtube/user/profile_dropdown/avatar/after',
			$this->plugin->myCRED,
			'show_user_dropdown_profile_balances'
		);

		$this->loader->add_action(
			'streamtube/core/dashboard/transactions/table/before',
			$this->plugin->myCRED,
			'show_user_balances'
		);

		$this->loader->add_action(
			'streamtube/core/dashboard/withdraw/table/before',
			$this->plugin->myCRED,
			'show_user_balances'
		);

		$this->loader->add_filter(
			'streamtube/core/user/dashboard/menu/items',
			$this->plugin->myCRED,
			'add_dashboard_menu',
			10,
			1
		);

		$this->loader->add_filter(
			'streamtube/core/user/profile/menu/items',
			$this->plugin->myCRED,
			'add_profile_menu',
			10,
			1
		);

		$this->loader->add_action(
			'streamtube/core/elementor/widgets_registered',
			$this->plugin->myCRED,
			'widgets_registered',
			10,
			1
		);

		$this->loader->add_action(
			'wp',
			$this->plugin->myCRED,
			'redirect_buy_points_page'
		);

		$this->loader->add_filter(
			'mycred_buycred_checkout_cancel',
			$this->plugin->myCRED,
			'filter_cancel_checkout',
			10,
			1
		);

		$this->loader->add_filter(
			'streamtube/core/advertising/vast_tag_url',
			$this->plugin->myCRED->sell_content,
			'filter_advertisements',
			10,
			3
		);

		$this->loader->add_filter(
			'streamtube/player/file/output',
			$this->plugin->myCRED->sell_content,
			'filter_player_output',
			50,
			2
		);

		$this->loader->add_filter(
			'streamtube/player/embed/output',
			$this->plugin->myCRED->sell_content,
			'filter_player_embed_output',
			50,
			2
		);

		$this->loader->add_filter(
			'streamtube/core/rest/video/source',
			$this->plugin->myCRED->sell_content,
			'filter_rest_video_source_field',
			50,
			2
		);

		$this->loader->add_filter(
			'streamtube/core/video/can_user_download',
			$this->plugin->myCRED->sell_content,
			'filter_download_permission',
			10,
			1
		);

		$this->loader->add_action(
			'streamtube/core/post/added',
			$this->plugin->myCRED->sell_content,
			'update_price',
			10,
			1
		);

		$this->loader->add_action(
			'streamtube/core/post/updated',
			$this->plugin->myCRED->sell_content,
			'update_price',
			10,
			1
		);

		$this->loader->add_action(
			'streamtube_add_meta_boxes',
			$this->plugin->myCRED->sell_content,
			'add_meta_boxes',
			20
		);

		$this->loader->add_action(
			'wp_ajax_transfers_points',
			$this->plugin->myCRED->transfers,
			'ajax_transfers_points',
			10
		);

		$this->loader->add_action(
			'streamtube/core/user/header/action_buttons',
			$this->plugin->myCRED->transfers,
			'button_donate',
			10
		);

		$this->loader->add_action(
			'wp_footer',
			$this->plugin->myCRED->transfers,
			'modal_donate',
			10
		);

		$this->loader->add_action(
			'streamtube/single/video/control',
			$this->plugin->myCRED->gifts,
			'button_gift'
		);

		$this->loader->add_action(
			'streamtube/core/mycred/modal/gift/widget',
			$this->plugin->myCRED->gifts,
			'gift_widget_content',
			10
		);

		$this->loader->add_action(
			'wp_footer',
			$this->plugin->myCRED->gifts,
			'modal_gift',
			10
		);

		$this->loader->add_action(
			'mycred_pre_process_cashcred',
			$this->plugin->myCRED->cash_cred,
			'fix_withdrawal_404'
		);

		$this->loader->add_filter(
			'mycred_setup_hooks',
			'Streamtube_Core_myCRED_Hook_Watch_Video',
			'register',
			10,
			2
		);

		$this->loader->add_filter(
			'mycred_setup_hooks',
			'Streamtube_Core_myCRED_Hook_Like_Post',
			'register',
			10,
			2
		);

		$this->loader->add_filter(
			'mycred_all_references',
			$this->plugin->myCRED,
			'filter_references',
			10
		);

		$this->loader->add_action(
			'init',
			$this->plugin->myCRED,
			'add_post_type_supports',
			9999
		);

		$this->loader->add_action(
			'streamtube/core/register_modules',
			$this->plugin->myCRED,
			'register_permission_module'
		);
	}

	/**
	 *
	 * buddyPress hooks
	 * 
	 */
	private function define_buddypress_hooks() {

		if ( ! $this->get_license() ) {
			return;
		}

		$this->include_file( 'third-party/buddypress/class-streamtube-core-buddypress.php' );

		$this->plugin->buddypress = new StreamTube_Core_buddyPress();

		$this->loader->add_action(
			'parse_request',
			$this->plugin->buddypress,
			'parse_request'
		);

		$this->loader->add_action(
			'wp_enqueue_scripts',
			$this->plugin->buddypress,
			'enqueue_scripts'
		);

		$this->loader->add_action(
			'widgets_init',
			$this->plugin->buddypress,
			'register_sidebar'
		);

		$this->loader->add_action(
			'widgets_init',
			'StreamTube_Core_buddyPress_Widget_User_List',
			'register'
		);

		$this->loader->add_action(
			'bp_setup_nav',
			$this->plugin->buddypress,
			'remove_unnecessary_menu_items'
		);

		$this->loader->add_filter(
			'streamtube/core/dashboard/menu/current',
			$this->plugin->buddypress,
			'set_current_dashboard_menu',
			10,
			2
		);

		$this->loader->add_filter(
			'streamtube/core/user/dashboard/menu/items',
			$this->plugin->buddypress,
			'register_dashboard_menu',
			5
		);

		$this->loader->add_filter(
			'streamtube/core/user/profile/menu/items',
			$this->plugin->buddypress,
			'register_profile_menu'
		);

		$this->loader->add_filter(
			'body_class',
			$this->plugin->buddypress,
			'filter_body_class'
		);

		$this->loader->add_filter(
			'is_buddypress',
			$this->plugin->buddypress,
			'filter_is_buddypress'
		);

		$this->loader->add_filter(
			'bp_members_get_user_url',
			$this->plugin->buddypress,
			'filter_bp_members_get_user_url',
			10,
			4
		);

		$this->loader->add_action(
			'wp',
			$this->plugin->buddypress,
			'_set_displayed_user_id',
			10
		);

		$this->loader->add_action(
			'init',
			$this->plugin->buddypress,
			'set_displayed_user_id'
		);


		$this->loader->add_filter(
			'bp_get_loggedin_user_avatar',
			$this->plugin->buddypress,
			'filter_bp_get_loggedin_user_avatar',
			10,
			3
		);

		$this->loader->add_filter(
			'bp_get_add_friend_button',
			$this->plugin->buddypress,
			'filter_bp_button'
		);

		$this->loader->add_filter(
			'bp_get_send_message_button_args',
			$this->plugin->buddypress,
			'filter_bp_button'
		);

		$this->loader->add_filter(
			'bp_follow_get_add_follow_button',
			$this->plugin->buddypress,
			'filter_bp_button'
		);

		$this->loader->add_filter(
			'bp_get_button',
			$this->plugin->buddypress,
			'filter_bp_get_button',
			10,
			3
		);

		$this->loader->add_filter(
			'bp_embed_oembed_html',
			$this->plugin->buddypress,
			'filter_bp_embed_oembed_html',
			10,
			4
		);

		$this->loader->add_action(
			'bp_activity_post_type_published',
			$this->plugin->buddypress,
			'notify_followers_of_new_activity',
			10,
			3
		);

		$this->loader->add_action(
			'publish_video',
			$this->plugin->buddypress,
			'notify_followers_of_new_submit',
			10,
			3
		);

		$this->loader->add_action(
			'publish_post',
			$this->plugin->buddypress,
			'notify_followers_of_new_submit',
			10,
			3
		);

		$this->loader->add_action(
			'streamtube/core/user/card/name/after',
			$this->plugin->buddypress,
			'display_primary_button'
		);

		$this->loader->add_action(
			'streamtube/buddypress/activity_loop/before',
			$this->plugin->buddypress,
			'display_featured_activities'
		);

		$this->loader->add_action(
			'streamtube/core/user/dashboard/page_header/before',
			$this->plugin->buddypress,
			'display_notices'
		);

		$this->loader->add_action(
			'wp_body_open',
			$this->plugin->buddypress,
			'display_float_user_list'
		);

		$this->loader->add_action(
			'streamtube/core/user/profile/content',
			$this->plugin->buddypress,
			'display_user_bio'
		);

		$this->loader->add_action(
			'admin_enqueue_scripts',
			$this->plugin->buddypress,
			'admin_enqueue_scripts'
		);

		/******************** Groups ********************/

		$this->loader->add_action(
			'widgets_init',
			$this->plugin->buddypress->groups,
			'register_sidebar'
		);

		$this->loader->add_filter(
			'bp_get_group_join_button',
			$this->plugin->buddypress->groups,
			'filter_bp_get_group_join_button',
			10,
			2
		);

		$this->loader->add_filter(
			'bp_get_group_description',
			$this->plugin->buddypress->groups,
			'filter_bp_get_group_description',
			10,
			2
		);

		$this->loader->add_filter(
			'streamtube/core/user/profile/menu/items',
			$this->plugin->buddypress->groups,
			'register_profile_menu'
		);

		/******************** Members ********************/

		$this->loader->add_action(
			'streamtube/core/user/card/name/after',
			$this->plugin->buddypress->members,
			'display_last_active_time'
		);

		$this->loader->add_filter(
			'bp_get_member_avatar',
			$this->plugin->buddypress->members,
			'filter_bp_get_member_avatar',
			10,
			2
		);

		$this->loader->add_filter(
			'bp_get_members_invitations_send_invites_permalink',
			$this->plugin->buddypress->members,
			'filter_bp_get_members_invitations_send_invites_permalink',
			10,
			2
		);

		$this->loader->add_filter(
			'bp_get_members_invitations_list_invites_permalink',
			$this->plugin->buddypress->members,
			'filter_bp_get_members_invitations_list_invites_permalink',
			10,
			2
		);

		$this->loader->add_filter(
			'streamtube/core/user/dashboard/menu/items',
			$this->plugin->buddypress->members,
			'display_dashboard_menu_item'
		);

		/******************** Friends ********************/
		$this->loader->add_filter(
			'bp_ajax_querystring',
			$this->plugin->buddypress->friends,
			'filter_bp_ajax_querystring',
			40,
			2
		);

		$this->loader->add_action(
			'streamtube/core/user/header/action_buttons',
			$this->plugin->buddypress->friends,
			'display_the_single_add_friend_button'
		);

		$this->loader->add_filter(
			'streamtube/core/user/profile/menu/items',
			$this->plugin->buddypress->friends,
			'register_profile_menu'
		);

		/******************** Messages ********************/
		$this->loader->add_filter(
			'bp_get_message_css_class',
			$this->plugin->buddypress->messages,
			'filter_bp_get_message_css_class'
		);

		$this->loader->add_filter(
			'streamtube_filter_wp_menu_item_title_wpmi',
			$this->plugin->buddypress->messages,
			'filter_message_icon',
			10,
			4
		);

		$this->loader->add_action(
			'streamtube/core/user/header/action_buttons',
			$this->plugin->buddypress->messages,
			'display_send_message_button'
		);

		$this->loader->add_filter(
			'streamtube_filter_wp_menu_item_title',
			$this->plugin->buddypress->messages,
			'display_unread_messages_badge',
			10,
			5
		);

		$this->loader->add_action(
			'streamtube/core/user/dashboard/page_header/before',
			$this->plugin->buddypress->messages,
			'display_global_notice'
		);

		$this->loader->add_filter(
			'streamtube/core/user/dashboard/menu/items',
			$this->plugin->buddypress->messages,
			'display_dashboard_menu_item'
		);

		/******************** Notifications ********************/
		$this->loader->add_filter(
			'bp_notifications_get_registered_components',
			$this->plugin->buddypress->notifications,
			'filter_bp_notifications_get_registered_components',
			10,
			2
		);

		$this->loader->add_filter(
			'bp_get_the_notification_description',
			$this->plugin->buddypress->notifications,
			'filter_notification_description',
			10,
			2
		);

		$this->loader->add_action(
			'streamtube_core_post_approved',
			$this->plugin->buddypress->notifications,
			'notify_author_post_moderated',
			10,
			3
		);

		$this->loader->add_action(
			'streamtube_core_post_rejected',
			$this->plugin->buddypress->notifications,
			'notify_author_post_moderated',
			10,
			3
		);

		$this->loader->add_action(
			'streamtube/header/profile/before',
			$this->plugin->buddypress->notifications,
			'display_header_notification_button'
		);

		$this->loader->add_filter(
			'streamtube_filter_wp_menu_item_title',
			$this->plugin->buddypress->notifications,
			'display_unread_notifications_badge',
			10,
			5
		);

		$this->loader->add_filter(
			'streamtube_filter_wp_menu_item_title_wpmi',
			$this->plugin->buddypress->notifications,
			'filter_notification_icon',
			10,
			4
		);

		$this->loader->add_filter(
			'streamtube/core/user/dashboard/menu/items',
			$this->plugin->buddypress->notifications,
			'display_dashboard_menu_item'
		);

		/******************** Activity ********************/
		$this->loader->add_filter(
			'bp_ajax_querystring',
			$this->plugin->buddypress->activity,
			'filter_bp_ajax_querystring',
			50,
			2
		);

		$this->loader->add_filter(
			'bp_get_activity_css_class',
			$this->plugin->buddypress->activity,
			'filter_bp_get_activity_css_class'
		);

		$this->loader->add_filter(
			'bp_get_activity_delete_link',
			$this->plugin->buddypress->activity,
			'filter_bp_get_activity_delete_link'
		);

		$this->loader->add_filter(
			'bp_activity_entry_content',
			$this->plugin->buddypress->activity,
			'display_the_player'
		);

		$this->loader->add_filter(
			'streamtube/core/user/profile/menu/items',
			$this->plugin->buddypress->activity,
			'register_profile_menu'
		);

		$this->loader->add_action(
			'wp_ajax_dismiss_migrate_activity',
			$this->plugin->buddypress->activity,
			'ajax_dismiss_migrate_activity'
		);

		$this->loader->add_action(
			'wp_ajax_migrate_activity',
			$this->plugin->buddypress->activity,
			'ajax_migrate_activity'
		);

		$this->loader->add_action(
			'admin_notices',
			$this->plugin->buddypress->activity,
			'display_admin_notices'
		);

		/******************** Follow ********************/
		if ( $this->plugin->buddypress->follow->is_active() ) {
			$this->loader->add_action(
				'init',
				$this->plugin->buddypress->follow,
				'remove_hooks'
			);

			$this->loader->add_action(
				'admin_init',
				$this->plugin->buddypress->follow,
				'migrate_wpuf'
			);

			$this->loader->add_filter(
				'bp_ajax_querystring',
				$this->plugin->buddypress->follow,
				'filter_bp_ajax_querystring',
				30,
				2
			);

			$this->loader->add_filter(
				'bp_has_members',
				$this->plugin->buddypress->follow,
				'filter_bp_has_members_template',
				10,
				3
			);

			$this->loader->add_action(
				'streamtube/core/user/header/action_buttons',
				$this->plugin->buddypress->follow,
				'display_the_single_follow_button'
			);

			$this->loader->add_filter(
				'streamtube/core/user/profile/menu/items',
				$this->plugin->buddypress->follow,
				'register_profile_menu'
			);

			$this->loader->add_action(
				'admin_notices',
				$this->plugin->buddypress->follow,
				'display_admin_notices'
			);
		}
	}

	/**
	 *
	 * Advanced Custom Fields
	 * 
	 */
	private function define_acf_hooks() {
		$this->include_file( 'third-party/acf/class-streamtube-core-acf.php' );

		$this->plugin->ACF = new StreamTube_Core_ACF();

		$this->loader->add_action(
			'streamtube/core/post/edit_form/before',
			$this->plugin->ACF->Form_Post,
			'init'
		);

		$this->loader->add_action(
			'streamtube_add_meta_boxes',
			$this->plugin->ACF->Form_Post,
			'add_meta_boxes',
			10,
			2
		);

		$this->loader->add_filter(
			'wp_insert_post_empty_content',
			$this->plugin->ACF->Form_Post,
			'wp_insert_post_empty_content',
			10,
			2
		);

		$this->loader->add_action(
			'save_post',
			$this->plugin->ACF->Form_Post,
			'save_post',
			10,
			2
		);

		$this->loader->add_action(
			'add_meta_boxes',
			$this->plugin->ACF->Form_Post,
			'remove_admin_meta_boxes',
			20,
			2
		);

		$this->loader->add_action(
			'acf/field_group/render_group_settings_tab/additional',
			$this->plugin->ACF->Form_Post,
			'render_additional_group_settings_tabs'
		);

		$this->loader->add_filter(
			'acf/field_group/additional_group_settings_tabs',
			$this->plugin->ACF->Form_Post,
			'additional_group_settings_tabs'
		);

		$this->loader->add_filter(
			'acf/post_type/registration_args',
			$this->plugin->ACF->Form_Post,
			'filter_custom_post_type_settings',
			10,
			2
		);

		$this->loader->add_action(
			'acf/post_type/basic_settings',
			$this->plugin->ACF->Form_Post,
			'add_custom_post_type_settings'
		);

		$this->loader->add_filter(
			'acf/taxonomy/registration_args',
			$this->plugin->ACF->Form_Post,
			'filter_custom_taxonomy_settings',
			10,
			2
		);

		$this->loader->add_action(
			'acf/taxonomy/basic_settings',
			$this->plugin->ACF->Form_Post,
			'add_custom_taxonomy_settings'
		);

		$this->loader->add_action(
			'wp_enqueue_scripts',
			$this->plugin->ACF->Form_User,
			'enqueue_scripts'
		);

		$this->loader->add_action(
			'streamtube/core/user_edit_custom_fields',
			$this->plugin->ACF->Form_User,
			'user_form'
		);
	}

	/**
	 *
	 * Better Messages Hooks
	 * 
	 * @since 1.1.3
	 */
	private function define_better_messages_hooks() {

		if ( ! $this->get_license() ) {
			return;
		}

		$this->include_file( 'third-party/better-messages/class-streamtube-core-better-messages.php' );

		$this->plugin->better_messages = new StreamTube_Core_Better_Messages();

		if ( ! $this->get_license() ) {
			$this->loader->add_action(
				'add_meta_boxes',
				$this->plugin->better_messages->admin,
				'unregistered_meta_boxes'
			);
		} else {

			$this->loader->add_action(
				'add_meta_boxes',
				$this->plugin->better_messages,
				'add_meta_boxes'
			);

			$this->loader->add_action(
				'save_post',
				$this->plugin->better_messages,
				'update_post_settings',
				10,
				1
			);

			$this->loader->add_action(
				'wp_ajax_get_recipient_info',
				$this->plugin->better_messages,
				'get_recipient_info'
			);

			$this->loader->add_action(
				'wp_ajax_nopriv_get_recipient_info',
				$this->plugin->better_messages,
				'get_recipient_info'
			);

			$this->loader->add_action(
				'streamtube/avatar_dropdown/after',
				$this->plugin->better_messages,
				'show_unread_threads_badge_on_avatar'
			);

			$this->loader->add_filter(
				'streamtube/core/user/profile/menu/items',
				$this->plugin->better_messages,
				'add_profile_menu',
				10,
				1
			);

			$this->loader->add_filter(
				'streamtube/core/user/dashboard/menu/items',
				$this->plugin->better_messages,
				'add_dashboard_menu',
				10,
				1
			);

			$this->loader->add_action(
				'streamtube/core/user/header/action_buttons',
				$this->plugin->better_messages,
				'button_private_message',
				20
			);

			$this->loader->add_action(
				'streamtube/core/user/card/name/after',
				$this->plugin->better_messages,
				'user_list_button_private_message',
				20
			);

			$this->loader->add_action(
				'wp_footer',
				$this->plugin->better_messages,
				'modal_private_message',
				10
			);

			$this->loader->add_action(
				'wp',
				$this->plugin->better_messages,
				'goto_inbox',
				10
			);

			$this->loader->add_filter(
				'streamtube_core_get_edit_post_nav_items',
				$this->plugin->better_messages,
				'add_post_nav_item',
				10,
				2
			);

			$this->loader->add_filter(
				'body_class',
				$this->plugin->better_messages,
				'filter_body_class',
				10,
				1
			);

			$this->loader->add_filter(
				'streamtube_has_post_comments',
				$this->plugin->better_messages,
				'filter_has_post_comments',
				10,
				1
			);

			$this->loader->add_filter(
				'comments_template',
				$this->plugin->better_messages,
				'filter_comments_template',
				100,
				1
			);

			$this->loader->add_action(
				'streamtube/post/thumbnail/after',
				$this->plugin->better_messages,
				'add_post_thumbnail_livechat_icon'
			);

			$this->loader->add_action(
				'streamtube/flat_post/item',
				$this->plugin->better_messages,
				'add_post_thumbnail_livechat_icon'
			);

			$this->loader->add_filter(
				'bp_better_messages_can_send_message',
				$this->plugin->better_messages,
				'filter_disable_reply',
				100,
				3
			);

			$this->loader->add_filter(
				'better_messages_get_thread_type',
				$this->plugin->better_messages,
				'filter_thread_type',
				100,
				2
			);

			$this->loader->add_filter(
				'better_messages_rest_user_item',
				$this->plugin->better_messages,
				'filter_rest_user_item',
				100,
				3
			);

			$this->loader->add_action(
				'streamtube_core_post_approved',
				$this->plugin->better_messages,
				'send_pm_after_post_moderated',
				10,
				3
			);

			$this->loader->add_action(
				'streamtube_core_post_rejected',
				$this->plugin->better_messages,
				'send_pm_after_post_moderated',
				10,
				3
			);

			$this->loader->add_action(
				'init',
				$this->plugin->better_messages,
				'add_post_type_support_live_chat',
				9999
			);

			$this->loader->add_action(
				'widgets_init',
				'Streamtube_Core_Widget_LiveChat',
				'register'
			);
		}
	}

	/**
	 *
	 * bbPress Hooks
	 * 
	 * @since 1.1.9
	 */
	private function define_bbpress() {
		$this->include_file( 'third-party/bbpress/class-streamtube-core-bbpress.php' );

		$this->plugin->bbpress = new StreamTube_Core_bbPress();

		$this->loader->add_action(
			'init',
			$this->plugin->bbpress,
			'add_forum_thumbnail'
		);

		$this->loader->add_action(
			'init',
			$this->plugin->bbpress,
			'redirect_search_page'
		);
	}

	/**
	 *
	 * Youtube Importer Hooks
	 * 
	 * @since 1.1.9
	 */
	private function define_youtube_importer() {

		$this->include_file( 'third-party/youtube-importer/class-streamtube-core-youtube-importer.php' );

		$this->plugin->yt_importer = new StreamTube_Core_Youtube_Importer();

		if ( ! $this->get_license() ) {

			$this->loader->add_action(
				'admin_menu',
				$this->plugin->yt_importer->admin,
				'unregistered'
			);

		} else {

			$this->loader->add_action(
				'init',
				$this->plugin->yt_importer->post_type,
				'post_type'
			);

			$this->loader->add_action(
				'init',
				$this->plugin->yt_importer->post_type,
				'remove_post_type_support',
				10000
			);

			$this->loader->add_action(
				'init',
				$this->plugin->yt_importer,
				'revoke_oauth_token',
				20
			);

			$this->loader->add_action(
				'init',
				$this->plugin->yt_importer,
				'request_oauth_token',
				30
			);

			$this->loader->add_action(
				'wp_ajax_youtube_search',
				$this->plugin->yt_importer,
				'ajax_search_content'
			);

			$this->loader->add_action(
				'wp_ajax_youtube_import',
				$this->plugin->yt_importer,
				'ajax_import_content'
			);

			$this->loader->add_action(
				'wp_ajax_youtube_bulk_import',
				$this->plugin->yt_importer,
				'ajax_bulk_import_content'
			);

			$this->loader->add_action(
				'wp_ajax_youtube_cron_bulk_import',
				$this->plugin->yt_importer,
				'ajax_run_bulk_import_content'
			);

			$this->loader->add_filter(
				'template_include',
				$this->plugin->yt_importer,
				'template_run_bulk_import_content',
				10,
				1
			);

			$this->loader->add_action(
				'wp_ajax_get_yt_importer_tax_terms',
				$this->plugin->yt_importer,
				'ajax_get_tax_terms'
			);

			$this->loader->add_filter(
				'streamtube/core/post/import_embed/errors',
				$this->plugin->yt_importer,
				'import_youtube_embed_error_handler',
				10,
				2
			);

			$this->loader->add_action(
				'streamtube/core/embed/imported',
				$this->plugin->yt_importer,
				'import_youtube_embed',
				10,
				2
			);

			$this->loader->add_filter(
				'streamtube/core/youtube_importer/get_settings',
				$this->plugin->yt_importer,
				'filter_importer_settings',
				10,
				2
			);

			$this->loader->add_action(
				'save_post_youtube_importer',
				$this->plugin->yt_importer,
				'add_import_schedule',
				20,
				3
			);

			$this->loader->add_action(
				'deleted_post_youtube_importer',
				$this->plugin->yt_importer,
				'clear_import_schedules',
				10,
				2
			);

			$this->loader->add_action(
				'streamtube_youtube_imports',
				$this->plugin->yt_importer,
				'execute_import_schedule',
				10,
				2
			);

			$this->loader->add_action(
				'streamtube/core/youtube_importer/oauth/authenticated',
				$this->plugin->yt_importer,
				'fetch_mine_channels',
				10,
				2
			);

			$this->loader->add_action(
				'streamtube/core/youtube_importer/oauth/authenticated',
				$this->plugin->yt_importer,
				'fetch_mine_playlists',
				10,
				2
			);

			$this->loader->add_action(
				'add_meta_boxes',
				$this->plugin->yt_importer->admin,
				'add_meta_boxes',
				10,
				2
			);

			$this->loader->add_action(
				'streamtube_add_meta_boxes',
				$this->plugin->yt_importer->admin,
				'add_meta_boxes',
				10,
				2
			);

			$this->loader->add_action(
				'save_post_youtube_importer',
				$this->plugin->yt_importer->admin,
				'save_settings',
				10,
				1
			);

			$this->loader->add_filter(
				'manage_youtube_importer_posts_columns',
				$this->plugin->yt_importer->admin,
				'post_table',
				10,
				1
			);

			$this->loader->add_action(
				'manage_youtube_importer_posts_custom_column',
				$this->plugin->yt_importer->admin,
				'post_table_columns',
				10,
				2
			);

			$this->loader->add_action(
				'pre_get_posts',
				$this->plugin->yt_importer->admin,
				'pre_get_posts'
			);

			$this->loader->add_filter(
				'streamtube/core/posts_list_table/query_args',
				$this->plugin->yt_importer->admin,
				'filter_posts_list_table_query_args',
				10,
				3
			);

			$this->loader->add_action(
				'admin_enqueue_scripts',
				$this->plugin->yt_importer->admin,
				'enqueue_scripts'
			);

			$this->loader->add_action(
				'wp_enqueue_scripts',
				$this->plugin->yt_importer->admin,
				'enqueue_scripts'
			);
		}
	}

	/**
	 *
	 * Bunny Stream Hooks
	 * 
	 */
	private function define_bunnycdn() {

		$this->include_file( 'third-party/bunnycdn/class-streamtube-core-bunnycdn.php' );

		$this->plugin->bunnycdn = new Streamtube_Core_BunnyCDN();

		if ( ! $this->get_license() ) {
			$this->loader->add_action(
				'admin_menu',
				$this->plugin->bunnycdn->admin,
				'unregistered'
			);
		} else {

			$this->loader->add_action(
				'admin_menu',
				$this->plugin->bunnycdn->admin,
				'registered'
			);

			$this->loader->add_action(
				'add_attachment',
				$this->plugin->bunnycdn,
				'add_attachment',
				10,
				1
			);

			$this->loader->add_action(
				'attachment_updated',
				$this->plugin->bunnycdn,
				'attachment_updated',
				10,
				1
			);

			$this->loader->add_action(
				'delete_attachment',
				$this->plugin->bunnycdn,
				'delete_attachment',
				10,
				1
			);

			$this->loader->add_action(
				'wp_get_attachment_url',
				$this->plugin->bunnycdn,
				'filter_wp_get_attachment_url',
				10,
				2
			);

			$this->loader->add_filter(
				'streamtube/player/file/setup',
				$this->plugin->bunnycdn,
				'filter_player_setup',
				10,
				2
			);

			$this->loader->add_filter(
				'streamtube/core/player/load_video_source',
				$this->plugin->bunnycdn,
				'filter_player_load_source',
				10,
				3
			);

			$this->loader->add_filter(
				'streamtube/player/file/output',
				$this->plugin->bunnycdn,
				'filter_player_output',
				50,
				3
			);

			$this->loader->add_filter(
				'streamtube/core/post/player/pre_text_tracks',
				$this->plugin->bunnycdn,
				'filter_player_subtitles',
				10,
				4
			);

			$this->loader->add_filter(
				'streamtube/core/video/download_file_url',
				$this->plugin->bunnycdn,
				'filter_download_file_url',
				10,
				2
			);

			$this->loader->add_action(
				'streamtube/core/post/edit/thumbnail_content',
				$this->plugin->bunnycdn,
				'thumbnail_notice',
				10,
				1
			);

			$this->loader->add_action(
				'wp_ajax_get_bunnycdn_video_status',
				$this->plugin->bunnycdn,
				'ajax_get_video_status'
			);

			$this->loader->add_action(
				'wp_ajax_refresh_bunny_data',
				$this->plugin->bunnycdn,
				'ajax_refresh_bunny_data'
			);

			$this->loader->add_filter(
				'streamtube/core/public_scripts/localize',
				$this->plugin->bunnycdn,
				'disable_big_uploads'
			);

			$this->loader->add_action(
				'wp_ajax_delete_bunny_video',
				$this->plugin->bunnycdn,
				'ajax_delete_bunny_video'
			);

			$this->loader->add_action(
				'wp_ajax_import_bunny_video',
				$this->plugin->bunnycdn,
				'ajax_import_bunny_video'
			);

			$this->loader->add_action(
				'wp_ajax_import_all_bunny_videos',
				$this->plugin->bunnycdn,
				'ajax_import_bunny_videos'
			);

			$this->loader->add_action(
				'wp_ajax_delete_bunny_collection',
				$this->plugin->bunnycdn,
				'ajax_delete_bunny_collection'
			);

			$this->loader->add_action(
				'wp_ajax_bunnycdn_sync',
				$this->plugin->bunnycdn,
				'ajax_sync'
			);

			$this->loader->add_action(
				'wp_ajax_bunnycdn_retry_sync',
				$this->plugin->bunnycdn,
				'ajax_retry_sync'
			);

			$this->loader->add_action(
				'edit_post_video',
				$this->plugin->bunnycdn,
				'refresh_bunny_data'
			);

			$this->loader->add_action(
				'init',
				$this->plugin->bunnycdn,
				'webhook_callback'
			);

			$this->loader->add_action(
				'streamtube/core/bunny/imported/post',
				$this->plugin->bunnycdn,
				'update_bunny_video_metadata',
				10,
				3
			);

			$this->loader->add_action(
				'streamtube/core/bunny/existed/post',
				$this->plugin->bunnycdn,
				'update_bunny_video_metadata',
				10,
				3
			);

			$this->loader->add_action(
				'streamtube/core/bunny/webhook/update',
				$this->plugin->bunnycdn,
				'update_attachment_metadata',
				10,
				3
			);

			$this->loader->add_action(
				'streamtube/core/bunny/webhook/update',
				$this->plugin->bunnycdn,
				'update_user_collection_webhook',
				10,
				3
			);

			$this->loader->add_action(
				'streamtube/core/bunny/webhook/update',
				$this->plugin->bunnycdn,
				'update_thumbnail_images',
				10,
				3
			);

			$this->loader->add_action(
				'streamtube/core/bunny/webhook/update',
				$this->plugin->bunnycdn,
				'delete_original_file',
				10,
				3
			);

			$this->loader->add_action(
				'streamtube/core/bunny/webhook/update',
				$this->plugin->bunnycdn,
				'auto_categorization',
				50,
				3
			);

			$this->loader->add_action(
				'streamtube/core/bunny/webhook/update',
				$this->plugin->bunnycdn,
				'auto_publish_after_success_encoding',
				20,
				3
			);

			$this->loader->add_action(
				'streamtube/core/bunny/webhook/update',
				$this->plugin->bunnycdn,
				'notify_author_after_encoding_failed',
				20,
				3
			);

			$this->loader->add_filter(
				'streamtube/core/video/thumbnail_url_2',
				$this->plugin->bunnycdn,
				'filter_thumbnail_image_2',
				20,
				3
			);

			$this->loader->add_action(
				'profile_update',
				$this->plugin->bunnycdn,
				'update_user_collection',
				10,
				3
			);

			$this->loader->add_filter(
				'bulk_actions-upload',
				$this->plugin->bunnycdn,
				'admin_bulk_actions',
				10,
				2
			);

			$this->loader->add_filter(
				'handle_bulk_actions-edit-video',
				$this->plugin->bunnycdn,
				'admin_handle_bulk_actions',
				10,
				3
			);

			$this->loader->add_filter(
				'handle_bulk_actions-upload',
				$this->plugin->bunnycdn,
				'admin_handle_bulk_actions',
				10,
				3
			);

			$this->loader->add_action(
				'admin_notices',
				$this->plugin->bunnycdn,
				'admin_handle_bulk_admin_notices',
				10
			);


			$this->loader->add_filter(
				'manage_media_columns',
				$this->plugin->bunnycdn,
				'admin_media_table'
			);

			$this->loader->add_action(
				'manage_media_custom_column',
				$this->plugin->bunnycdn,
				'admin_media_table_columns',
				10,
				2
			);

			$this->loader->add_filter(
				'manage_video_posts_columns',
				$this->plugin->bunnycdn,
				'admin_post_table',
				10,
				1
			);

			$this->loader->add_action(
				'manage_video_posts_custom_column',
				$this->plugin->bunnycdn,
				'admin_post_table_columns',
				10,
				2
			);

			$this->loader->add_action(
				'add_meta_boxes',
				$this->plugin->bunnycdn->admin,
				'add_meta_boxes'
			);

			$this->loader->add_filter(
				'bulk_actions-edit-video',
				$this->plugin->bunnycdn,
				'admin_bulk_actions',
				10,
				2
			);

			$this->loader->add_action(
				'wp_ajax_check_videos_progress',
				$this->plugin->bunnycdn->admin,
				'ajax_check_videos_progress',
				10
			);

			$this->loader->add_action(
				'admin_footer',
				$this->plugin->bunnycdn->admin,
				'interval_check_videos_progress',
				10
			);

			$this->loader->add_action(
				'admin_notices',
				$this->plugin->bunnycdn->admin,
				'notices',
				10
			);

			$this->loader->add_action(
				'wp_ajax_read_file_log_content',
				$this->plugin->bunnycdn,
				'ajax_read_log_content'
			);

			$this->loader->add_action(
				'wp_ajax_read_task_log_content',
				$this->plugin->bunnycdn,
				'ajax_read_task_log_content'
			);

			$this->loader->add_filter(
				'wp_video_extensions',
				$this->plugin->bunnycdn,
				'filter_allow_formats',
				9999,
				1
			);

			$this->loader->add_filter(
				'streamtube/core/generate_image_from_file',
				$this->plugin->bunnycdn,
				'rest_generate_thumbnail_image',
				10,
				2
			);

			$this->loader->add_filter(
				'streamtube/core/generate_animated_image_from_file',
				$this->plugin->bunnycdn,
				'rest_generate_animated_image_from_file',
				10,
				2
			);

			$this->loader->add_filter(
				'streamtube/core/generate_subtitles',
				$this->plugin->bunnycdn,
				'rest_generate_subtitles',
				10,
				2
			);

			$this->loader->add_filter(
				'better_messages_rest_message_meta',
				$this->plugin->bunnycdn,
				'filter_better_messages_rest_message_meta',
				100,
				4
			);

			$this->loader->add_filter(
				'streamtube/core/user/dashboard/menu/items',
				$this->plugin->bunnycdn,
				'register_dashboard_menu'
			);

			$this->loader->add_action(
				'admin_enqueue_scripts',
				$this->plugin->bunnycdn,
				'enqueue_scripts'
			);

			$this->loader->add_action(
				'wp_enqueue_scripts',
				$this->plugin->bunnycdn,
				'enqueue_scripts'
			);

			$this->loader->add_action(
				'streamtube/core/register_modules',
				$this->plugin->bunnycdn,
				'register_permission_module'
			);

			$this->loader->add_action(
				'rest_api_init',
				$this->plugin->bunnycdn->restApi,
				'rest_api_init'
			);
		}
	}

	/**
	 *
	 * PMP Hooks
	 * 
	 */
	private function define_pmpro() {

		if ( ! $this->get_license() ) {
			return;
		}

		$this->include_file( 'third-party/pmpro/short-functions.php' );
		$this->include_file( 'third-party/pmpro/class-streamtube-core-pmpro.php' );

		$this->plugin->pmpro = new StreamTube_Core_PMPro();

		$this->loader->add_action(
			'add_meta_boxes',
			$this->plugin->pmpro->admin,
			'add_meta_boxes',
			10
		);

		$this->loader->add_action(
			'wp_enqueue_scripts',
			$this->plugin->pmpro,
			'enqueue_scripts'
		);

		$this->loader->add_action(
			'init',
			$this->plugin->pmpro,
			'shortcode_membership_levels'
		);

		$this->loader->add_action(
			'wp_ajax_get_pmpro_invoice_detail',
			$this->plugin->pmpro,
			'ajax_get_invoice_detail'
		);

		$this->loader->add_action(
			'wp',
			$this->plugin->pmpro,
			'redirect_default_pages'
		);

		$this->loader->add_action(
			'pmpro_membership_level_after_other_settings',
			$this->plugin->pmpro,
			'add_level_settings_box'
		);

		$this->loader->add_action(
			'pmpro_save_membership_level',
			$this->plugin->pmpro,
			'update_level_settings'
		);

		$this->loader->add_filter(
			'streamtube/post/meta/item/membership',
			$this->plugin->pmpro,
			'add_membership_badge',
			10,
			2
		);

		$this->loader->add_action(
			'init',
			$this->plugin->pmpro,
			'disable_comments_filter',
			10
		);

		$this->loader->add_filter(
			'streamtube/core/advertising/vast_tag_url',
			$this->plugin->pmpro,
			'filter_advertisements',
			10,
			3
		);

		$this->loader->add_filter(
			'streamtube/player/file/output',
			$this->plugin->pmpro,
			'filter_player_output',
			50,
			2
		);

		$this->loader->add_filter(
			'streamtube/player/embed/output',
			$this->plugin->pmpro,
			'filter_player_embed_output',
			50,
			2
		);

		$this->loader->add_filter(
			'streamtube/core/rest/video/source',
			$this->plugin->pmpro,
			'filter_rest_video_source_field',
			50,
			2
		);

		$this->loader->add_filter(
			'streamtube/core/video/can_user_download',
			$this->plugin->pmpro,
			'filter_download_permission',
			10,
			1
		);

		$this->loader->add_filter(
			'streamtube/core/widget/posts/posts_where',
			$this->plugin->pmpro,
			'filter_widget_posts_where',
			10,
			2
		);

		$this->loader->add_filter(
			'streamtube/core/widget/posts/posts_join',
			$this->plugin->pmpro,
			'filter_widget_posts_join',
			10,
			2
		);

		$this->loader->add_filter(
			'streamtube/core/widget/posts/posts_distinct',
			$this->plugin->pmpro,
			'filter_widget_posts_distinct',
			10,
			2
		);

		$this->loader->add_filter(
			'streamtube/core/user/dashboard/menu/items',
			$this->plugin->pmpro,
			'add_dashboard_menu',
			10,
			1
		);

		$this->loader->add_filter(
			'streamtube/core/user/profile/menu/items',
			$this->plugin->pmpro,
			'add_profile_menu',
			10,
			1
		);

		$this->loader->add_action(
			'streamtube_add_meta_boxes',
			$this->plugin->pmpro,
			'add_meta_boxes',
			10,
			2
		);

		$this->loader->add_action(
			'init',
			$this->plugin->pmpro,
			'save_membership_levels_widget'
		);

		$this->loader->add_action(
			'init',
			$this->plugin->pmpro,
			'add_post_type_support_membership',
			9999
		);

		$this->loader->add_action(
			'streamtube/core/elementor/widgets_registered',
			$this->plugin->pmpro,
			'elementor_widgets_registered'
		);

		$this->loader->add_filter(
			'streamtube/core/posts_list_table/status_links',
			$this->plugin->pmpro,
			'filter_posts_list_table_status_links',
			10,
			3
		);

		$this->loader->add_action(
			'streamtube/core/posts_list_table/before_query',
			$this->plugin->pmpro,
			'filter_posts_list_table_before_query',
			10
		);
	}

	/**
	 * 
	 * Define Paid Member Subscriptions hook
	 */
	private function define_pms() {
		if ( ! $this->get_license() ) {
			return;
		}

		$this->include_file( 'third-party/pms/class-stream-core-pms.php' );

		$this->plugin->pmsubscriptions = new StreamTube_Core_PMSubscriptions();

		$this->loader->add_action(
			'wp',
			$this->plugin->pmsubscriptions,
			'remove_attachment_url_filter'
		);

		$this->loader->add_filter(
			'streamtube/player/file/output',
			$this->plugin->pmsubscriptions,
			'filter_player_output',
			10,
			2
		);

		$this->loader->add_filter(
			'streamtube/player/embed/output',
			$this->plugin->pmsubscriptions,
			'filter_player_embed_output',
			10,
			2
		);

		$this->loader->add_filter(
			'pms_member_account_tabs',
			$this->plugin->pmsubscriptions,
			'filter_member_account_tabs',
			10,
			2
		);

		$this->loader->add_filter(
			'pms_account_shortcode_content',
			$this->plugin->pmsubscriptions,
			'add_shortcode_wrapper',
			10
		);

		$this->loader->add_filter(
			'pms_subscriptions_form_content',
			$this->plugin->pmsubscriptions,
			'add_shortcode_wrapper',
			10
		);

		$this->loader->add_filter(
			'streamtube/post/meta/item/membership',
			$this->plugin->pmsubscriptions,
			'display_membership_badge',
			10,
			2
		);

		$this->loader->add_filter(
			'streamtube/core/posts_list_table/status_links',
			$this->plugin->pmsubscriptions,
			'filter_posts_list_table_status_links',
			10,
			3
		);

		$this->loader->add_filter(
			'streamtube/core/posts_list_table/query_args',
			$this->plugin->pmsubscriptions,
			'filter_posts_list_table_query_args',
			10,
			1
		);

		$this->loader->add_filter(
			'streamtube/core/user/dashboard/menu/items',
			$this->plugin->pmsubscriptions,
			'add_dashboard_menu',
			10,
			1
		);

		$this->loader->add_action(
			'streamtube_add_meta_boxes',
			$this->plugin->pmsubscriptions,
			'add_meta_boxes'
		);

		$this->loader->add_action(
			'init',
			$this->plugin->pmsubscriptions,
			'add_post_type_support_membership'
		);

		$this->loader->add_action(
			'wp_enqueue_scripts',
			$this->plugin->pmsubscriptions,
			'enqueue_scripts'
		);
	}

	/**
	 * 
	 * Define Patreon hooks
	 * @return void
	 */
	private function define_patreon() {
		if ( ! $this->get_license() ) {
			return;
		}

		$this->include_file( 'third-party/patreon/class-streamtube-core-patreon.php' );

		$this->plugin->patreon = new StreamTube_Core_Patreon();

		$this->loader->add_filter(
			'streamtube/player/file/output',
			$this->plugin->patreon,
			'filter_player_output',
			50,
			2
		);

		$this->loader->add_filter(
			'streamtube/player/embed/output',
			$this->plugin->patreon,
			'filter_player_embed_output',
			50,
			2
		);

		$this->loader->add_filter(
			'streamtube/core/video/can_user_download',
			$this->plugin->patreon,
			'filter_download_permission',
			50
		);

		$this->loader->add_filter(
			'streamtube/core/rest/video/source',
			$this->plugin->patreon,
			'filter_rest_video_source_field',
			50,
			2
		);

		$this->loader->add_filter(
			'streamtube/post/meta/item/membership',
			$this->plugin->patreon,
			'add_membership_badge',
			10,
			2
		);

		$this->loader->add_action(
			'init',
			$this->plugin->patreon,
			'add_post_type_supports'
		);

		$this->loader->add_action(
			'wp_enqueue_scripts',
			$this->plugin->patreon,
			'enqueue_scripts',
		);

		$this->loader->add_action(
			'enqueue_embed_scripts',
			$this->plugin->patreon,
			'enqueue_scripts',
		);

		if ( $this->plugin->patreon->metabox ) {
			$this->loader->add_action(
				'streamtube_add_meta_boxes',
				$this->plugin->patreon->metabox,
				'add_dashboard_meta_boxes',
			);
		}
	}

	/**
	 *
	 * Define Woocommerce Hooks
	 * 
	 */
	private function define_woocommerce() {

		if ( ! $this->get_license() ) {
			return;
		}

		$this->include_file( 'third-party/woocommerce/class-streamtube-core-woocommerce.php' );
		$this->include_file( 'third-party/woocommerce/short-functions.php' );
		$this->plugin->woocommerce = new Streamtube_Core_Woocommerce();

		$this->loader->add_action(
			'init',
			$this->plugin->woocommerce,
			'remove_default'
		);

		$this->loader->add_action(
			'parse_request',
			$this->plugin->woocommerce,
			'parse_request'
		);

		$this->loader->add_filter(
			'body_class',
			$this->plugin->woocommerce,
			'filter_body_classes'
		);

		$this->loader->add_filter(
			'woocommerce_account_menu_items',
			$this->plugin->woocommerce,
			'filter_woocommerce_account_menu_items',
			9999,
			2
		);

		$this->loader->add_filter(
			'streamtube/core/dashboard/menu/current',
			$this->plugin->woocommerce,
			'set_current_dashboard_shopping_menu',
			10,
			2
		);

		$this->loader->add_action(
			'woocommerce_account_purchased-products_endpoint',
			$this->plugin->woocommerce,
			'display_purchased_products'
		);

		$this->loader->add_action(
			'woocommerce_account_purchased-videos_endpoint',
			$this->plugin->woocommerce,
			'display_purchased_videos'
		);

		$this->loader->add_action(
			'woocommerce_account_liked-products_endpoint',
			$this->plugin->woocommerce,
			'display_liked_products'
		);

		$this->loader->add_action(
			'woocommerce_before_shop_loop_item_title',
			$this->plugin->woocommerce,
			'display_template_loop_product_thumbnail'
		);

		$this->loader->add_action(
			'woocommerce_shop_loop_item_title',
			$this->plugin->woocommerce,
			'display_template_loop_product_title'
		);

		$this->loader->add_action(
			'wp_ajax_get_cart_total',
			$this->plugin->woocommerce,
			'ajax_get_cart_total'
		);

		$this->loader->add_action(
			'wp_ajax_nopriv_get_cart_total',
			$this->plugin->woocommerce,
			'ajax_get_cart_total'
		);

		$this->loader->add_action(
			'streamtube/header/profile/before',
			$this->plugin->woocommerce,
			'the_cart_button'
		);

		$this->loader->add_filter(
			'woocommerce_order_is_paid_statuses',
			$this->plugin->woocommerce,
			'filter_order_is_paid_statuses'
		);

		$this->loader->add_filter(
			'woocommerce_is_account_page',
			$this->plugin->woocommerce,
			'filter_is_account_page'
		);

		$this->loader->add_filter(
			'woocommerce_is_purchasable',
			$this->plugin->woocommerce,
			'filter_is_purchasable',
			9999,
			2
		);

		$this->loader->add_filter(
			'streamtube_filter_wp_menu_item_title',
			$this->plugin->woocommerce,
			'filter_wp_menu_item_title',
			10,
			5
		);

		$this->loader->add_filter(
			'shortcode_atts_products',
			$this->plugin->woocommerce,
			'filter_shortcode_atts_products',
			10,
			4
		);

		$this->loader->add_filter(
			'woocommerce_shortcode_products_query',
			$this->plugin->woocommerce,
			'filter_shortcode_products_query',
			10,
			3
		);

		$this->loader->add_action(
			'woocommerce_product_is_visible',
			$this->plugin->woocommerce,
			'filter_product_is_visible'
		);

		$this->loader->add_action(
			'streamtube/woocommerce/product_single_title/after',
			$this->plugin->woocommerce,
			'display_single_product_rating'
		);

		$this->loader->add_filter(
			'woocommerce_single_product_image_thumbnail_html',
			$this->plugin->woocommerce,
			'filter_single_product_image_thumbnail_html',
			10,
			2
		);

		$this->loader->add_filter(
			'single_product_archive_thumbnail_size',
			$this->plugin->woocommerce,
			'filter_thumbnail_image_size'
		);

		$this->loader->add_filter(
			'woocommerce_get_endpoint_url',
			$this->plugin->woocommerce,
			'filter_woocommerce_get_endpoint_url',
			9999,
			4
		);

		$this->loader->add_action(
			'wp',
			$this->plugin->woocommerce,
			'redirect_woocommerce_edit_account_url'
		);

		$this->loader->add_filter(
			'streamtube/core/dashboard/page_header/heading',
			$this->plugin->woocommerce,
			'filter_page_heading',
			10,
			2
		);

		$this->loader->add_filter(
			'woocommerce_json_search_found_products',
			$this->plugin->woocommerce,
			'filter_json_search_found_products'
		);

		$this->loader->add_action(
			'streamtube/core/user/dashboard/page_header/after',
			$this->plugin->woocommerce,
			'display_dashboard_notices'
		);

		$this->loader->add_filter(
			'streamtube/core/dashboard/menu/current',
			$this->plugin->woocommerce,
			'filter_dashboard_current_menu',
			10,
			2
		);

		$this->loader->add_filter(
			'streamtube/core/user/dashboard/menu/items',
			$this->plugin->woocommerce,
			'add_dashboard_menu'
		);

		$this->loader->add_action(
			'add_meta_boxes',
			$this->plugin->woocommerce,
			'add_meta_boxes',
			20,
			2
		);

		$this->loader->add_action(
			'streamtube_add_meta_boxes',
			$this->plugin->woocommerce,
			'add_meta_boxes',
			20,
			2
		);

		$this->loader->add_action(
			'init',
			$this->plugin->woocommerce,
			'add_post_type_supports',
			9999
		);

		$this->loader->add_filter(
			'register_post_type_args',
			$this->plugin->woocommerce,
			'filter_register_post_type_args',
			10,
			2
		);

		$this->loader->add_action(
			'streamtube/core/register_modules',
			$this->plugin->woocommerce,
			'register_permission_module'
		);

		$this->loader->add_action(
			'wp_enqueue_scripts',
			$this->plugin->woocommerce,
			'enqueue_admin_scripts'
		);

		$this->loader->add_action(
			'add_meta_boxes',
			$this->plugin->woocommerce->sell_content,
			'add_meta_boxes',
			20,
			2
		);

		$this->loader->add_action(
			'streamtube_add_meta_boxes',
			$this->plugin->woocommerce->sell_content,
			'add_meta_boxes',
			20,
			2
		);

		$this->loader->add_action(
			'save_post_video',
			$this->plugin->woocommerce->sell_content,
			'update_relevant_product',
			10,
			1
		);

		$this->loader->add_action(
			'save_post_video',
			$this->plugin->woocommerce->sell_content,
			'do_add_builtin_product',
			20,
			3
		);

		$this->loader->add_action(
			'streamtube_core_post_approved',
			$this->plugin->woocommerce->sell_content,
			'do_update_builtin_product',
			10
		);

		$this->loader->add_action(
			'streamtube_core_post_rejected',
			$this->plugin->woocommerce->sell_content,
			'do_update_builtin_product',
			10
		);

		$this->loader->add_action(
			'streamtube_core_post_pending',
			$this->plugin->woocommerce->sell_content,
			'do_update_builtin_product',
			10
		);

		$this->loader->add_action(
			'streamtube_core_post_trashed',
			$this->plugin->woocommerce->sell_content,
			'do_update_builtin_product',
			10
		);

		$this->loader->add_action(
			'streamtube_core_post_restored',
			$this->plugin->woocommerce->sell_content,
			'do_update_builtin_product',
			10
		);

		$this->loader->add_action(
			'before_delete_post',
			$this->plugin->woocommerce->sell_content,
			'do_delete_builtin_product',
			20,
			2
		);

		$this->loader->add_filter(
			'streamtube/core/woocommerce/sell_content/ref_product',
			$this->plugin->woocommerce->sell_content,
			'set_default_relevant_product',
			10,
			2
		);

		$this->loader->add_filter(
			'streamtube/core/woocommerce/sell_content/ref_product',
			$this->plugin->woocommerce->sell_content,
			'set_builtin_product',
			20,
			2
		);

		$this->loader->add_filter(
			'streamtube/core/advertising/vast_tag_url',
			$this->plugin->woocommerce->sell_content,
			'filter_advertisements',
			10,
			3
		);

		$this->loader->add_filter(
			'streamtube/player/file/output',
			$this->plugin->woocommerce->sell_content,
			'filter_player_output',
			40,
			2
		);

		$this->loader->add_filter(
			'streamtube/player/embed/output',
			$this->plugin->woocommerce->sell_content,
			'filter_player_embed_output',
			40,
			2
		);

		$this->loader->add_filter(
			'streamtube/core/rest/video/source',
			$this->plugin->woocommerce->sell_content,
			'filter_rest_video_source_field',
			50,
			2
		);

		$this->loader->add_filter(
			'post_class',
			$this->plugin->woocommerce->sell_content,
			'filter_post_classes',
			10,
			1
		);

		$this->loader->add_filter(
			'streamtube/core/video/can_user_download',
			$this->plugin->woocommerce->sell_content,
			'filter_download_permission',
			10,
			1
		);

		$this->loader->add_filter(
			'streamtube/core/widget/posts/query_args',
			$this->plugin->woocommerce->sell_content,
			'filter_widget_post_list_query',
			10,
			2
		);

		$this->loader->add_filter(
			'streamtube/post/meta/item/membership',
			$this->plugin->woocommerce->sell_content,
			'add_price_badge',
			10,
			2
		);

		$this->loader->add_action(
			'wp_enqueue_scripts',
			$this->plugin->woocommerce->sell_content,
			'enqueue_scripts'
		);

		$this->loader->add_filter(
			'streamtube/core/get_full_post_data',
			$this->plugin->woocommerce->sell_content,
			'filter_get_full_post_data',
			10,
			2
		);

		$this->loader->add_action(
			'after_delete_post',
			$this->plugin->woocommerce->sell_content,
			'delete_product_metadata',
			10,
			2
		);

		$this->loader->add_filter(
			'woocommerce_product_data_tabs',
			$this->plugin->woocommerce->sell_capability,
			'add_tab'
		);

		$this->loader->add_action(
			'woocommerce_product_data_panels',
			$this->plugin->woocommerce->sell_capability,
			'tab_content'
		);

		$this->loader->add_action(
			'woocommerce_process_product_meta',
			$this->plugin->woocommerce->sell_capability,
			'save_fields'
		);

		$this->loader->add_action(
			'woocommerce_order_status_changed',
			$this->plugin->woocommerce->sell_capability,
			'update_user_capabilities'
		);

		$this->loader->add_action(
			'woocommerce_order_fully_refunded',
			$this->plugin->woocommerce->sell_capability,
			'remove_capabilities'
		);

		$this->loader->add_action(
			'woocommerce_order_partially_refunded',
			$this->plugin->woocommerce->sell_capability,
			'remove_capabilities'
		);

		$this->loader->add_action(
			'admin_enqueue_scripts',
			$this->plugin->woocommerce->sell_capability,
			'enqueue_scripts'
		);

		$this->loader->add_action(
			'wp_enqueue_scripts',
			$this->plugin->woocommerce->sell_capability,
			'enqueue_scripts'
		);

		$this->loader->add_filter(
			'wc_stripe_upe_params',
			$this->plugin->woocommerce->stripe,
			'filter_stripe_upe_params'
		);

	}

	/**
	 *
	 * Define Woocommerce Booking Hooks
	 * 
	 */
	private function define_woocommerce_booking() {
		if ( ! $this->get_license() ) {
			return;
		}

		$this->include_file( 'third-party/woocommerce-booking/class-streamtube-core-woocommerce-booking.php' );

		$this->plugin->woocommerce_booking = new Streamtube_Core_Woocommerce_Booking();

		$this->loader->add_filter(
			'get_edit_post_link',
			$this->plugin->woocommerce_booking,
			'filter_get_edit_post_link',
			10,
			3
		);

		$this->loader->add_action(
			'init',
			$this->plugin->woocommerce_booking,
			'init_tabs'
		);

		$this->loader->add_action(
			'wp_enqueue_scripts',
			$this->plugin->woocommerce_booking,
			'enqueue_scripts'
		);
	}

	/**
	 * Define YITH WooCommerce Affiliates
	 */
	private function define_yith_woocommerce_affiliates() {
		if ( ! $this->get_license() ) {
			return;
		}

		$this->include_file( 'third-party/yith-woocommerce-affiliates/class-streamtube-core-yith-woocommerce-affiliates.php' );

		$this->plugin->woocommerce_YITH_affiliates = new StreamTube_Core_WooCommerce_YITH_Affiliates();

		$this->loader->add_action(
			'parse_request',
			$this->plugin->woocommerce_YITH_affiliates,
			'parse_request',
			10,
			1
		);

		$this->loader->add_filter(
			'yith_wcaf_get_endpoint_url',
			$this->plugin->woocommerce_YITH_affiliates,
			'filter_yith_wcaf_get_endpoint_url',
			10,
			4
		);

		$this->loader->add_filter(
			'yith_wcaf_become_an_affiliate_redirection',
			$this->plugin->woocommerce_YITH_affiliates,
			'filter_yith_wcaf_become_an_affiliate_redirection'
		);

		$this->loader->add_filter(
			'streamtube/core/dashboard/page_header/heading',
			$this->plugin->woocommerce_YITH_affiliates,
			'filter_page_heading',
			10,
			2
		);

		$this->loader->add_filter(
			'streamtube/core/user/dashboard/menu/items',
			$this->plugin->woocommerce_YITH_affiliates,
			'add_dashboard_menu',
			20,
			1
		);

		$this->loader->add_action(
			'wp_enqueue_scripts',
			$this->plugin->woocommerce_YITH_affiliates,
			'enqueue_scripts'
		);
	}

	/**
	 *
	 * Define Dokan hooks
	 * 
	 */
	private function define_dokan() {

		if ( ! $this->get_license() ) {
			return;
		}

		$this->include_file( 'third-party/dokan/class-streamtube-core-dokan.php' );

		$this->plugin->dokan = new StreamTube_Core_Dokan();

		$this->loader->add_action(
			'parse_request',
			$this->plugin->dokan,
			'parse_request'
		);

		$this->loader->add_action(
			'dokan_dashboard_shortcode_query_vars',
			$this->plugin->dokan,
			'parse_shortcode_query_vars'
		);

		$this->loader->add_action(
			'get_header',
			$this->plugin->dokan,
			'set_store_query_var',
			1
		);

		$this->loader->add_action(
			'wp',
			$this->plugin->dokan,
			'set_seller_dashboard'
		);

		$this->loader->add_filter(
			'dokan_get_page_url',
			$this->plugin->dokan,
			'filter_get_page_url',
			10,
			4
		);

		$this->loader->add_filter(
			'woocommerce_get_endpoint_url',
			$this->plugin->dokan,
			'filter_woocommerce_get_endpoint_url',
			9999,
			4
		);

		$this->loader->add_filter(
			'streamtube/core/dashboard/menu/current',
			$this->plugin->dokan,
			'set_current_dashboard_store_menu',
			10,
			2
		);

		$this->loader->add_filter(
			'streamtube/core/dashboard/menu/default_submenu',
			$this->plugin->dokan,
			'filter_default_submenu',
			10,
			2
		);

		$this->loader->add_filter(
			'wp_list_comments_args',
			$this->plugin->dokan,
			'filter_support_comments_callback'
		);

		$this->loader->add_filter(
			'comments_open',
			$this->plugin->dokan,
			'filter_support_comments_open',
			10,
			2
		);

		$this->loader->add_filter(
			'dokan_get_dashboard_page_id',
			$this->plugin->dokan,
			'filter_get_dashboard_page_id'
		);

		$this->loader->add_filter(
			'streamtube/core/profile_image_url',
			$this->plugin->dokan,
			'filter_store_banner_img',
			10,
			3
		);

		$this->loader->add_filter(
			'dokan_get_store_url',
			$this->plugin->dokan,
			'filter_store_url',
			10,
			3
		);

		$this->loader->add_filter(
			'body_class',
			$this->plugin->dokan,
			'filter_body_class'
		);

		$this->loader->add_filter(
			'dokan_get_navigation_url',
			$this->plugin->dokan,
			'filter_navigation_url',
			10,
			2
		);

		$this->loader->add_filter(
			'dokan_dashboard_nav_active',
			$this->plugin->dokan,
			'filter_nav_active',
			10,
			3
		);

		$this->loader->add_action(
			'wp',
			$this->plugin->dokan,
			'redirect_dashboard_page'
		);

		$this->loader->add_action(
			'init',
			$this->plugin->dokan,
			'remove_register_page_redirect'
		);

		$this->loader->add_filter(
			'dokan_dashboard_nav_common_link',
			$this->plugin->dokan,
			'remove_dashboard_common_link'
		);

		$this->loader->add_filter(
			'dokan_store_sidebar_args',
			$this->plugin->dokan,
			'filter_store_sidebar_args'
		);

		$this->loader->add_filter(
			'dokan_store_widget_args',
			$this->plugin->dokan,
			'filter_store_widget_args'
		);

		$this->loader->add_filter(
			'dokan_dashboard_shortcode_query_vars',
			$this->plugin->dokan,
			'filter_shortcode_query_vars'
		);

		$this->loader->add_filter(
			'get_edit_post_link',
			$this->plugin->dokan,
			'filter_edit_product_url',
			10,
			3
		);

		$this->loader->add_filter(
			'dokan_pre_product_listing_args',
			$this->plugin->dokan,
			'filter_pre_product_listing_args'
		);

		$this->loader->add_filter(
			'streamtube/core/woocommerce/sell_content/query_product_args',
			$this->plugin->dokan,
			'filter_query_product_args'
		);

		$this->loader->add_action(
			'streamtube/core/woocommerce/add_builtin_product/before_save',
			$this->plugin->dokan,
			'filter_builtin_product_status',
			10,
			2
		);

		$this->loader->add_filter(
			'streamtube/core/woocommerce/sell_content/player/is_purchasable',
			$this->plugin->dokan,
			'filter_is_content_purchasable',
			10,
			3
		);

		$this->loader->add_filter(
			'bp_ajax_querystring',
			$this->plugin->dokan,
			'filter_bp_ajax_querystring',
			20,
			2
		);

		$this->loader->add_action(
			'init',
			$this->plugin->dokan,
			'become_seller_apply_form'
		);

		$this->loader->add_action(
			'admin_init',
			$this->plugin->dokan,
			'manual_approve_seller'
		);

		$this->loader->add_filter(
			'wp_ajax_apply_become_seller',
			$this->plugin->dokan,
			'ajax_process_apply_become_seller'
		);

		$this->loader->add_action(
			'streamtube/core/dokan/become_seller_content',
			$this->plugin->dokan,
			'display_become_seller_content'
		);

		$this->loader->add_action(
			'streamtube/core/woocommerce/sell_content_box/before',
			$this->plugin->dokan,
			'display_seller_not_enabled_notice'
		);

		$this->loader->add_action(
			'streamtube/core/user/profile/products/content',
			$this->plugin->dokan,
			'display_store_products'
		);

		$this->loader->add_filter(
			'streamtube/core/user/profile/about/bio',
			$this->plugin->dokan,
			'display_store_location',
			20,
			1
		);

		$this->loader->add_action(
			'streamtube/core/user/card/profile_image/before',
			$this->plugin->dokan,
			'display_store_featured_badge'
		);

		$this->loader->add_action(
			'streamtube/core/user/header/profile_photo/before',
			$this->plugin->dokan,
			'display_store_featured_badge'
		);

		$this->loader->add_action(
			'streamtube/core/user/card/name/after',
			$this->plugin->dokan,
			'display_store_rating',
			9
		);

		$this->loader->add_action(
			'streamtube/core/user/header/display_name/after',
			$this->plugin->dokan,
			'display_store_rating',
			9
		);

		$this->loader->add_action(
			'streamtube/single/video/author/name/after',
			$this->plugin->dokan,
			'display_store_rating',
			9
		);

		$this->loader->add_filter(
			'streamtube/core/user/dashboard/menu/items',
			$this->plugin->dokan,
			'display_dashboard_menu_item'
		);

		$this->loader->add_filter(
			'streamtube/core/user/profile/menu/items',
			$this->plugin->dokan,
			'register_profile_menu'
		);

		$this->loader->add_action(
			'bp_members_directory_member_types',
			$this->plugin->dokan,
			'display_bp_stores_tab'
		);

		$this->loader->add_action(
			'bp_follow_start_following',
			$this->plugin->dokan,
			'do_bp_follow_following'
		);

		$this->loader->add_action(
			'bp_follow_stop_following',
			$this->plugin->dokan,
			'do_bp_follow_following'
		);

		$this->loader->add_action(
			'wp_user_followed',
			$this->plugin->dokan,
			'wp_user_following',
			10,
			2
		);

		$this->loader->add_action(
			'wp_user_unfollowed',
			$this->plugin->dokan,
			'wp_user_unfollowing',
			10,
			2
		);

		$this->loader->add_filter(
			'streamtube_core_upload_types',
			$this->plugin->dokan,
			'display_add_product_menu',
			10,
			2
		);

		$this->loader->add_filter(
			'init',
			$this->plugin->dokan,
			'create_empty_product',
			10
		);

		$this->loader->add_action(
			'wp_enqueue_scripts',
			$this->plugin->dokan,
			'enqueue_scripts',
			9999
		);
	}

	/**
	 *
	 * Define WooThanks hooks
	 * 
	 */
	private function define_woothanks() {

		$this->include_file( 'third-party/woothanks/class-streamtube-core-woothanks.php' );

		$this->loader->add_action(
			'streamtube/core/user/header/action_buttons',
			'Streamtube_Core_WooThanks',
			'the_button_buy',
			100
		);

		$this->loader->add_action(
			'wp_footer',
			'Streamtube_Core_WooThanks',
			'the_modal_buyform'
		);
	}

	private function define_fundengine() {

		$this->include_file( 'third-party/wp-fundraising-donation/class-streamtube-core-wp-fundraising-donation.php' );

		$this->plugin->FundEngine = new StreamTube_Core_FundEngine();

		$this->loader->add_filter(
			'streamtube/core/user/dashboard/menu/items',
			$this->plugin->FundEngine,
			'add_dashboard_menu'
		);

		$this->loader->add_action(
			'wp_enqueue_scripts',
			$this->plugin->FundEngine,
			'dequeue_flatpickr',
			20
		);

		$this->loader->add_action(
			'wp_enqueue_scripts',
			$this->plugin->FundEngine,
			'enqueue_scripts',
			9
		);
	}

	/**
	 * directorist hooks
	 */
	private function define_directorist() {
		$this->include_file( 'third-party/directorist/class-streamtube-core-directorist.php' );

		$this->plugin->directorist = new StreamTube_Core_Directorist();

		$this->loader->add_action(
			'parse_request',
			$this->plugin->directorist,
			'parse_request'
		);

		$this->loader->add_filter(
			'directorist_pagination',
			$this->plugin->directorist,
			'filter_author_directorist_pagination',
			10,
			4
		);

		$this->loader->add_filter(
			'streamtube/core/user/profile/menu/items',
			$this->plugin->directorist,
			'register_profile_menu_listing'
		);

		$this->loader->add_action(
			'wp_enqueue_scripts',
			$this->plugin->directorist,
			'enqueue_scripts'
		);
	}

	/**
	 *
	 * WP Statistics
	 * 
	 */
	private function define_wp_statistics() {

		if ( ! $this->get_license() ) {
			return;
		}

		$this->include_file( 'third-party/wp-statistics/class-streamtube-core-statistics.php' );

		$this->plugin->wp_statistics = new StreamTube_Core_WP_Statistics();

		$this->loader->add_filter(
			'wp_statistics_metabox_list',
			$this->plugin->wp_statistics,
			'register_custom_meta_boxes'
		);

		$this->loader->add_action(
			'streamtube_add_meta_boxes',
			$this->plugin->wp_statistics,
			'add_meta_boxes'
		);

		$this->loader->add_action(
			'streamtube/core/wpstatistics/metabox',
			$this->plugin->wp_statistics,
			'register_meta_box_screens'
		);

		$this->loader->add_filter(
			'wp_statistics_admin_menu_list',
			$this->plugin->wp_statistics,
			'filter_overview_views',
			100
		);

		$this->loader->add_filter(
			'wp_statistics_content_analytics_views',
			$this->plugin->wp_statistics,
			'filter_content_analytics_views'
		);

		$this->loader->add_filter(
			'wp_statistics_visitors_views',
			$this->plugin->wp_statistics,
			'filter_visitors_views'
		);

		$this->loader->add_filter(
			'wp_statistics_pages_views',
			$this->plugin->wp_statistics,
			'filter_pages_insight_views'
		);

		$this->loader->add_filter(
			'wp_statistics_referrals_views',
			$this->plugin->wp_statistics,
			'filter_referrals_views'
		);

		$this->loader->add_filter(
			'wp_statistics_author_analytics_views',
			$this->plugin->wp_statistics,
			'filter_author_analytics_views'
		);

		$this->loader->add_filter(
			'wp_statistics_category_analytics_views',
			$this->plugin->wp_statistics,
			'filter_category_analytics_views'
		);

		$this->loader->add_filter(
			'wp_statistics_geographic_views',
			$this->plugin->wp_statistics,
			'filter_geographic_views'
		);

		$this->loader->add_filter(
			'wp_statistics_devices_views',
			$this->plugin->wp_statistics,
			'filter_devices_views'
		);

		$this->loader->add_action(
			'streamtube/core/widget/posts/query/before',
			$this->plugin->wp_statistics,
			'handler_orderby_views'
		);

		$this->loader->add_action(
			'streamtube/core/widget/posts/query/after',
			$this->plugin->wp_statistics,
			'remove_handler_orderby_views'
		);

		$this->loader->add_action(
			'streamtube/core/widget/posts_flat/query/before',
			$this->plugin->wp_statistics,
			'handler_orderby_views'
		);

		$this->loader->add_action(
			'streamtube/core/widget/posts_flat/query/after',
			$this->plugin->wp_statistics,
			'remove_handler_orderby_views'
		);

		$this->loader->add_action(
			'init',
			$this->plugin->wp_statistics,
			'add_post_table_columns',
			20
		);

		$this->loader->add_filter(
			'streamtube_core_get_edit_post_nav_items',
			$this->plugin->wp_statistics,
			'add_post_nav_item',
			10,
			2
		);

		$this->loader->add_filter(
			'post_row_actions',
			$this->plugin->wp_statistics,
			'filter_post_row_actions',
			10,
			2
		);

		$this->loader->add_filter(
			'page_row_actions',
			$this->plugin->wp_statistics,
			'filter_post_row_actions',
			10,
			2
		);

		$this->loader->add_action(
			'wp_statistics_after_admin_page_title',
			$this->plugin->wp_statistics,
			'add_header_component_single_type'
		);

		$this->loader->add_filter(
			'streamtube/core/user/dashboard/menu/items',
			$this->plugin->wp_statistics,
			'register_dashboard_menu'
		);

		$this->loader->add_filter(
			'streamtube_has_pageviews',
			$this->plugin->wp_statistics,
			'enable_display_post_views',
			20
		);

		$this->loader->add_filter(
			'streamtube/core/post/views',
			$this->plugin->wp_statistics,
			'display_post_view_count',
			10,
			2
		);

		$this->loader->add_action(
			'streamtube/single/video/meta',
			$this->plugin->wp_statistics,
			'display_single_post_view_count'
		);

		$this->loader->add_action(
			'streamtube/single/video/control',
			$this->plugin->wp_statistics,
			'display_single_post_statistics_button'
		);

		$this->loader->add_action(
			'streamtube/core/archive/sortby/after',
			$this->plugin->wp_statistics,
			'display_sortby_date_range'
		);

		$this->loader->add_action(
			'streamtube/core/wpstatistics/view',
			$this->plugin->wp_statistics,
			'remove_view_header'
		);

		$this->loader->add_action(
			'streamtube/core/register_modules',
			$this->plugin->wp_statistics,
			'register_permission_module'
		);

		$this->loader->add_action(
			'admin_init',
			$this->plugin->wp_statistics,
			'filter_admin_request',
			0
		);

		$this->loader->add_filter(
			'body_class',
			$this->plugin->wp_statistics,
			'filter_body_classes'
		);

		$this->loader->add_action(
			'init',
			$this->plugin->wp_statistics,
			'add_post_type_support',
			9999
		);

		$this->loader->add_action(
			'rest_api_init',
			$this->plugin->wp_statistics,
			'rest_api_init'
		);

		$this->loader->add_filter(
			'streamtube/core/widget/posts/tabs',
			$this->plugin->wp_statistics,
			'add_widget_settings_tab',
			10,
			2
		);

		$this->loader->add_action(
			'wp_enqueue_scripts',
			$this->plugin->wp_statistics,
			'enqueue_scripts'
		);

		$this->loader->add_action(
			'enqueue_embed_scripts',
			$this->plugin->wp_statistics,
			'enqueue_embed_scripts'
		);
	}

	private function define_jwt_auth() {
		$this->include_file( 'third-party/jwt-authentication/class-streamtube-core-jwt-authentication.php' );

		$this->loader->add_filter(
			'streamtube/core/user/dashboard/menu/items',
			'StreamTube_Core_JWT_Auth',
			'register_dashboard_menu'
		);
	}

	/**
	 *
	 * Real Cookie Banner
	 * 
	 */
	private function define_real_cookie_banner() {
		$this->include_file( 'third-party/real-cookie-banner/class-streamtube-core-real-cookie-banner.php' );

		$this->loader->add_filter(
			'streamtube/player/file/output',
			'StreamTube_Core_Real_Cookie_Banner',
			'filter_player_output',
			20,
			2
		);
	}

	/**
	 *
	 * OG plugin
	 *
	 * @link https://wordpress.org/plugins/og/
	 * 
	 */
	private function define_open_graph() {

		$this->include_file( 'third-party/open-graph/class-streamtube-core-open-graph.php' );

		$this->loader->add_filter(
			'og/term/meta/thumbnail_id_name',
			'StreamTube_Core_OpenGraph',
			'filter_term_thumbnail_name'
		);
	}

	private function update_checker() {
		if ( ! did_action( 'license_checker_loaded' ) ) {
			exit( 'Core Modifier was found' );
		}
	}

	/**
	 *
	 * Action links
	 * 
	 */
	private function action_links() {
		add_filter( 'plugin_action_links_' . STREAMTUBE_CORE_BASE, array( $this, '_action_links' ), 10, 1 );
	}

	/**
	 *
	 * Action links
	 * 
	 */
	public function _action_links( $actions ) {
		$links = array();

		$links[] = sprintf(
			'<a href="%s">%s</a>',
			esc_url( admin_url( '/customize.php?autofocus[panel]=streamtube' ) ),
			esc_html__( 'Theme Options', 'streamtube-core' )
		);

		$links[] = sprintf(
			'<a target="_blank" href="https://1.envato.market/DVqxZo">%s</a>',
			esc_html__( 'Support', 'streamtube-core' )
		);

		$links[] = sprintf(
			'<a target="_blank" href="https://streamtube.marstheme.com/documentation/">%s</a>',
			esc_html__( 'Documentation', 'streamtube-core' )
		);

		return array_merge( $actions, $links );
	}

	/**
	 *
	 * Generator meta tag
	 * 
	 * @since 1.0.8
	 */
	public function generator() {

		printf(
			'<meta name="generator" content="%1$s | %2$s | %3$s">',
			'StreamTube',
			'Video Streaming WordPress Theme',
			'https://1.envato.market/qny3O5'
		);
	}

	/**
	 * Run the loader to execute all of the hooks with WordPress.
	 *
	 * @since    1.0.0
	 */
	public function run() {
		$this->loader->run();
	}

	/**
	 *
	 * @since     1.0.0
	 * @return    string    The name of the plugin.
	 */
	public function get() {
		return $this->plugin;
	}

	/**
	 * The name of the plugin used to uniquely identify it within the context of
	 * WordPress and to define internationalization functionality.
	 *
	 * @since     1.0.0
	 * @return    string    The name of the plugin.
	 */
	public function get_plugin_name() {
		return $this->plugin_name;
	}

	/**
	 * The reference to the class that orchestrates the hooks with the plugin.
	 *
	 * @since     1.0.0
	 * @return    Streamtube_Core_Loader    Orchestrates the hooks of the plugin.
	 */
	public function get_loader() {
		return $this->loader;
	}

	/**
	 * Retrieve the version number of the plugin.
	 *
	 * @since     1.0.0
	 * @return    string    The version number of the plugin.
	 */
	public function get_version() {
		return $this->version;
	}

}