<?php

/**
 * WP menu
 *
 * Convert WP menu object to streamtube menu
 *
 * @link       https://themeforest.net/user/phpface
 * @since      1.0.0
 *
 * @package    Streamtube_Core
 * @subpackage Streamtube_Core/includes
 */

/**
 * Fired during plugin activation.
 *
 * This class defines all code necessary to run during the plugin's activation.
 *
 * @since      1.0.0
 * @package    Streamtube_Core
 * @subpackage Streamtube_Core/includes
 * @author     phpface <nttoanbrvt@gmail.com>
 */


/**
 * Class to structure and handle unlimited levels of nested menu items.
 */
class StreamTube_Core_WP_Menu {

    /**
     * Holds the structured menu items.
     *
     * @var array
     */
    private $items = [];

    private $callback = false;

    /**
     * Constructor to initialize and structure the menu items.
     *
     * @param array $items Menu items retrieved from wp_get_nav_menu_items().
     */
    public function __construct( $items = array(), $callback = false ) {

    	$this->callback = $callback;

        $this->build_menu_structure($items);
    }

    /**
     * Build the hierarchical structure of the menu items.
     *
     * @param array $items Menu items.
     */
    private function build_menu_structure( $items = array()) {
        $indexed_items = [];

        // Index the items by their IDs for fast lookups and initialize submenu slots
        foreach ($items as $item) {
            $item->submenu = [];
            $indexed_items[$item->ID] = $item;
        }

        // Build parent-child relationships
        foreach ($indexed_items as $item) {
            if ( $item->post_status !== 'publish') {
                continue;
            }

            if( $item->post_type === 'nav_menu_item' && 
                $item->type === 'post_type' &&
                ! in_array( get_post_status( $item->object_id ) , array( 'publish', 'private', 'unlist' ) ) ){
                continue;
            }

            if ( $item->type === 'post_type' ) {
                $item->url = '';
            }

            if ( ! property_exists( $item, 'wpmi' ) || ! is_object( $item->wpmi ) ) {
                $item->wpmi = new stdClass();
            }

            if ( empty( $item->wpmi->icon ) ) {
                $item->wpmi->icon = 'dashicons dashicons-admin-post';
            }

            if( ! $item->callback ){
            	$item->callback = array( $this , 'menu_cb' );
            }

            if ( $item->menu_item_parent && isset($indexed_items[$item->menu_item_parent])) {
                $indexed_items[$item->menu_item_parent]->submenu[$item->ID] = $item;
            } else {
                $this->items[$item->ID] = $item;
            }
        }
    }

    /**
     *
     * Get Menu item priority using custom meta field.
     * 
     */
    public static function get_menu_priority( $item ){
        $priority = (int)get_post_meta( $item->ID, 'priority', true );

        return $priority ?? $item->menu_order;
    }

    /**
     *
     * Get menu item capability
     * 
     */
    public static function get_menu_cap( $item ){
        $capability = get_post_meta( $item->ID, 'capability', true );

        return ! empty($capability ) ? $capability : 'read';
    }

    /**
     * Get the structured menu items.
     *
     * @return array The structured menu items with parent-child relationships.
     */
    public function get_menu(){
        return $this->items;
    }

    /**
     *
     * Callback function to generate menu page content
     * 
     * @param  WP_Post $item
     * 
     */
    public function menu_cb( $item ){
    	if( is_callable( $this->callback ) ){
    		call_user_func( $this->callback, $item );
    	}else{
    		esc_html_e( 'No callback', 'streamtube-core' );
    	}
    }

    /**
     * Recursively build submenu structure.
     *
     * @param array $menu Submenu items.
     * @return array Structured submenu items.
     */
    public function build_submenu( $menu = array() ){
        $structured_menu = [];

        foreach ($menu as $key) {
            $submenu_items = $key->submenu ? $this->build_submenu($key->submenu) : array();

            $structured_menu[sanitize_title($key->title)] = $this->create_menu_item($key, $submenu_items);
        }

        return $structured_menu;
    }

    /**
     * Create a menu item structure.
     *
     * @param object $item The menu item object.
     * @param array $submenu Submenu items.
     * @return array Structured menu item.
     */
    private function create_menu_item( $item, $submenu = array() ){
        return array(
            'title'    		=> $item->title,
            'url'      		=> $item->url ?? '',
            'priority' 		=> self::get_menu_priority( $item ),
            'cap'           => self::get_menu_cap( $item ),
            'icon'	   		=> $item->wpmi->icon,
            'callback' 		=> $item->callback,
            'callback_args'	=> $item,
            'submenu'  		=> $submenu ?? array()
        );
    }
}