<?php
/**
 * Convert Widgets to Metaboxes
 *
 * @link       https://themeforest.net/user/phpface
 * @since      1.0.0
 *
 * @package    Streamtube_Core
 * @subpackage Streamtube_Core/includes
 */

/**
 * @package    Streamtube_Core
 * @subpackage Streamtube_Core/includes
 * @author     phpface <nttoanbrvt@gmail.com>
 */

if( ! defined('ABSPATH' ) ){
    exit;
}

/**
 * Class to handle widget rendering in meta boxes.
 */
class StreamTube_Core_Widget_Meta_Box {

	/**
	 * Holds the sidebar ID
	 * @var string
	 */
	protected $sidebar_id;

	/**
	 * Holds the screen
	 * 
	 * @var string
	 */
	protected $screen;

	/**
	 * Holds the context
	 * 
	 * @var string
	 */
	protected $context;	

	/**
	 * Holds the priority
	 * 
	 * @var string
	 */
	protected $priority;		

	/**
	 * Whether to hide the widget title
	 *
	 * @var bool
	 */
	protected $hide_widget_title;

	public function __construct( $args = array() ) {

		$args = wp_parse_args( $args, array(
			'sidebar'			=>	'',
			'screen'			=>	'',
			'context'			=>	'',
			'priority'			=>	'default',
			'hide_widget_title'	=>	true,
		) );

		$this->sidebar_id        	= $args['sidebar'];

		$this->screen            	= $args['screen'];

		$this->context            	= $args['context'];

		$this->priority 			= $args['priority'];

		$this->hide_widget_title 	= $args['hide_widget_title'];
	}

	/**
	 * Get widget class by ID base.
	 *
	 * @param string $widget_id The widget ID base.
	 * @return string|null The widget class name, or null if not found.
	 */
	public function get_widget_class( $widget_id ) {
		global $wp_widget_factory;

		$widget = $wp_widget_factory->get_widget_object( _get_widget_id_base( $widget_id ) );

		return $widget ? get_class( $widget ) : null;
	}

	/**
	 * Get widget instance settings.
	 *
	 * @param string $widget_id The widget ID.
	 * @return array Widget settings.
	 */
	public function get_widget_instance( $widget_id ) {
		global $wp_registered_widgets;

		if ( empty( $wp_registered_widgets[$widget_id]['callback'] ) ) {
			return array();
		}

		/** @var WP_Widget $widget */
		$widget = $wp_registered_widgets[$widget_id]['callback'][0];

		$settings = $widget->get_settings();

		if ( preg_match( '/-(\d+)$/', $widget_id, $matches ) ) {
			$widget->number = (int) $matches[1] ?? $widget->number;
		}

		return ! empty( $settings[$widget->number] ) ? $settings[$widget->number] : array();
	}

	/**
	 * Get widget class by ID.
	 *
	 * @param string $widget_id The widget ID base.
	 * @return string|null The widget class name, or null if not found.
	 */
	public function get_widget_object( $widget_id ) {
		global $wp_widget_factory;

		$widget = $wp_widget_factory->get_widget_object( _get_widget_id_base( $widget_id ) );

		return $widget ?? false;
	}	

	/**
	 * Add widget meta boxes.
	 *
	 * @param string $sidebar_id The ID of the sidebar.
	 */
	public function add_meta_boxes( $sidebar_id = '' ) {
		if ( empty( $sidebar_id ) ) {
			$sidebar_id = $this->sidebar_id;
		}

		$sidebars = wp_get_sidebars_widgets();

		if ( ! isset( $sidebars[ $sidebar_id ] ) ) {
			return;
		}

		if ( $this->hide_widget_title ) {
			add_filter( 'widget_title', '__return_false' );
		}

		foreach ( $sidebars[ $sidebar_id ] as $widget_id ) {
			$this->add_single_widget_meta_box( $widget_id );
		}
	}

	/**
	 * Add a single widget meta box.
	 *
	 * @param string $widget_id The widget ID.
	 */
	protected function add_single_widget_meta_box( $widget_id ) {
		$settings = $this->get_widget_instance( $widget_id );

		if ( empty( $settings ) ) {
			return;
		}

		$widget_object = $this->get_widget_object( $widget_id );

		if ( ! $widget_object || ! class_exists( get_class( $widget_object ) ) ) {
			return;
		}

		streamtube_add_meta_box(
			$widget_id,
			$settings['title'] ?? esc_html__(  'Widget', 'streamtube-core' ),
			array( $this, 'render_widget_meta_box' ),
			$this->screen ?: 'user_dashboard', 
			$this->context,
			$this->priority
		);
	}

	/**
	 * Render the widget meta box content.
	 *
	 * @param WP_Post|null $post The post object.
	 * @param array $meta_box The meta box array.
	 */
	public function render_widget_meta_box( $post, $meta_box ) {
		$widget_id = $meta_box['id'];
		$settings = $this->get_widget_instance( $widget_id );
		$widget_object = $this->get_widget_object( $widget_id );

		the_widget(
			get_class( $widget_object ),
			array_merge( $settings, array(
				'title'       	=> false,
				'widget_title' 	=> false
			) ),
			array(
				'before_widget' => sprintf(
					'<div id="%1$s" class="widget widget-metabox %2$s">',
					esc_attr( $widget_id ),
					esc_attr( $widget_object->widget_options['classname'] )
				),
				'after_widget'  => '</div>',
			)
		);
	}
}