<?php
/**
 * Define the internationalization functionality
 *
 * Loads and defines the internationalization files for this plugin
 * so that it is ready for translation.
 *
 * @package    Streamtube_Core
 * @subpackage Streamtube_Core/includes
 *
 * @link       https://themeforest.net/user/phpface
 * @since      1.0.0
 */

/**
 * Define the internationalization functionality.
 *
 * Loads and defines the internationalization files for this plugin
 * so that it is ready for translation.
 *
 * @package    Streamtube_Core
 * @subpackage Streamtube_Core/includes
 *
 * @author     phpface <nttoanbrvt@gmail.com>
 *
 * @since      1.0.0
 */
if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

class Streamtube_Core_Taxonomy {

	const TAX_CATEGORY = 'categories';

	const TAX_TAG = 'video_tag';

	const TAX_REPORT = 'report_category';

	const TAX_HAND_PICK = 'hand_pick';

	/**
	 *
	 * Video Category taxonomy
	 *
	 * @since 1.0.0
	 */
	public function video_category() {
		$labels = array(
			"name"          => esc_html__( "Video Categories", "streamtube-core" ),
			"singular_name" => esc_html__( "Video Category", "streamtube-core" ),
			"search_items"  => esc_html__( "Search Categories", "streamtube-core" )
		);

		$args = array(
			"label"                 => esc_html__( "Video Categories", "streamtube-core" ),
			"labels"                => $labels,
			"public"                => true,
			"publicly_queryable"    => true,
			"hierarchical"          => true,
			"show_ui"               => true,
			"show_in_menu"          => true,
			"show_in_nav_menus"     => true,
			"query_var"             => true,
			"rewrite"               => array(
				'slug'         => sanitize_key( strtolower( get_option( 'taxonomy_' . self::TAX_CATEGORY . '_slug', self::TAX_CATEGORY ) ) ),
				'with_front'   => true,
				'hierarchical' => true
			),
			"show_admin_column"     => false,
			"show_in_rest"          => true,
			"rest_base"             => 'video_category',
			"rest_controller_class" => "WP_REST_Terms_Controller",
			"show_in_quick_edit"    => true,
			'show_tagcloud'         => true,
			'capabilities'          => array(
				'manage_terms' => 'manage_video_' . self::TAX_CATEGORY,
				'edit_terms'   => 'edit_video_' . self::TAX_CATEGORY,
				'delete_terms' => 'delete_video_' . self::TAX_CATEGORY,
				'assign_terms' => 'edit_videos'
			),
			'show_ui_frontend'      => wp_validate_boolean( get_option( 'taxonomy_' . self::TAX_CATEGORY, 'on' ) ),
			'show_in_filter'        => true
		);

		register_taxonomy( self::TAX_CATEGORY, array( 'video' ), $args );
	}

	/**
	 *
	 * Video Tag taxonomy
	 *
	 * @since 1.0.0
	 */
	public function video_tag() {

		$labels = array(
			"name"          => esc_html__( "Video Tags", "streamtube-core" ),
			"singular_name" => esc_html__( "Video Tag", "streamtube-core" ),
			"search_items"  => esc_html__( "Search Tags", "streamtube-core" )
		);

		$args = array(
			"label"                 => esc_html__( "Video Tags", "streamtube-core" ),
			"labels"                => $labels,
			"public"                => true,
			"publicly_queryable"    => true,
			"hierarchical"          => false,
			"show_ui"               => true,
			"show_in_menu"          => true,
			"show_in_nav_menus"     => true,
			"query_var"             => true,
			"rewrite"               => array(
				'slug' => sanitize_key( strtolower( get_option( 'taxonomy_' . self::TAX_TAG . '_slug', self::TAX_TAG ) ) )
			),
			"show_admin_column"     => false,
			"show_in_rest"          => true,
			"rest_base"             => self::TAX_TAG,
			"rest_controller_class" => "WP_REST_Terms_Controller",
			"show_in_quick_edit"    => true,
			'show_tagcloud'         => true,
			'capabilities'          => array(
				'manage_terms' => 'manage_' . self::TAX_TAG,
				'edit_terms'   => 'edit_' . self::TAX_TAG,
				'delete_terms' => 'delete_' . self::TAX_TAG,
				'assign_terms' => 'edit_videos'
			),
			'show_ui_frontend'      => wp_validate_boolean( get_option( 'taxonomy_' . self::TAX_TAG, 'on' ) )
		);

		register_taxonomy( self::TAX_TAG, array( 'video' ), $args );
	}

	/**
	 *
	 * report_category taxonomy
	 *
	 * @since 1.0.0
	 */
	public function report_category() {
		$labels = array(
			"name"          => esc_html__( "Video Reports", "streamtube-core" ),
			"singular_name" => esc_html__( "Video Report", "streamtube-core" )
		);

		$args = array(
			"label"                 => esc_html__( "Video Report", "streamtube-core" ),
			"labels"                => $labels,
			"public"                => true,
			"publicly_queryable"    => true,
			"hierarchical"          => true,
			"show_ui"               => true,
			"show_in_menu"          => true,
			"show_in_nav_menus"     => true,
			"query_var"             => true,
			"rewrite"               => array(
				'slug' => sanitize_key( strtolower( get_option( 'taxonomy_' . self::TAX_REPORT . '_slug', self::TAX_REPORT ) ) )
			),
			"show_admin_column"     => false,
			"show_in_rest"          => true,
			"rest_base"             => self::TAX_REPORT,
			"rest_controller_class" => "WP_REST_Terms_Controller",
			"show_in_quick_edit"    => false,
			'capabilities'          => array(
				'manage_terms' => 'manage_' . self::TAX_REPORT,
				'edit_terms'   => 'edit_' . self::TAX_REPORT,
				'delete_terms' => 'delete_' . self::TAX_REPORT,
				'assign_terms' => 'assign_' . self::TAX_REPORT
			)
		);

		register_taxonomy( self::TAX_REPORT, array( 'video' ), $args );
	}

	/**
	 *
	 * hand_pick taxonomy
	 *
	 * @since 1.0.0
	 */
	public function hand_pick() {
		$labels = array(
			"name"          => esc_html__( "Video Hand Picks", "streamtube-core" ),
			"singular_name" => esc_html__( "Video Hand Pick", "streamtube-core" )
		);

		$args = array(
			"label"                 => esc_html__( "Video Hand Pick", "streamtube-core" ),
			"labels"                => $labels,
			"public"                => true,
			"publicly_queryable"    => true,
			"hierarchical"          => true,
			"show_ui"               => true,
			"show_in_menu"          => true,
			"show_in_nav_menus"     => true,
			"query_var"             => true,
			"rewrite"               => array(
				'slug' => sanitize_key( strtolower( get_option( 'taxonomy_' . self::TAX_HAND_PICK . '_slug', self::TAX_HAND_PICK ) ) )
			),
			"show_admin_column"     => false,
			"show_in_rest"          => true,
			"rest_base"             => self::TAX_HAND_PICK,
			"rest_controller_class" => "WP_REST_Terms_Controller",
			"show_in_quick_edit"    => false,
			'capabilities'          => array(
				'manage_terms' => 'manage_' . self::TAX_HAND_PICK,
				'edit_terms'   => 'edit_' . self::TAX_HAND_PICK,
				'delete_terms' => 'delete_' . self::TAX_HAND_PICK,
				'assign_terms' => 'assign_' . self::TAX_HAND_PICK
			)
		);

		register_taxonomy( self::TAX_HAND_PICK, array( 'video' ), $args );
	}

	/**
	 *
	 * Get built-in taxonomies
	 *
	 * @return array
	 */
	public static function get_builtin_taxonomies() {
		$taxonomies   = array();
		$taxonomies[] = self::TAX_CATEGORY;
		$taxonomies[] = self::TAX_TAG;
		$taxonomies[] = self::TAX_REPORT;
		$taxonomies[] = self::TAX_HAND_PICK;

		return apply_filters( 'streamtube/core/taxonomy/builtin', $taxonomies );
	}

	/**
	 *
	 * Search terms
	 *
	 */
	public function search_terms() {

		check_ajax_referer( '_wpnonce' );

		$request = wp_parse_args( $_GET, array(
			'taxonomy'     => self::TAX_TAG,
			'responseType' => '',
			'search'       => '',
			'number'       => 10,
			'orderby'      => 'name',
			'order'        => 'ASC',
			'hide_empty'   => false
		) );

		extract( $request );

		if ( is_string( $taxonomy ) ) {
			$taxonomy = array_map( 'trim', explode( ',', $taxonomy ) );
		}

		$query_args = compact( 'orderby', 'order', 'number', 'search', 'hide_empty', 'taxonomy' );
		/**
		 *
		 * Filter query args
		 *
		 */
		$query_args = apply_filters( 'streamtube/core/search_terms', $query_args, $request );

		$terms = get_terms( $query_args );

		if ( is_wp_error( $terms ) ) {
			wp_send_json_error( $terms );
		}

		wp_send_json_success( $terms );
	}

	public function ajax_get_video_tag_terms() {

		$http_data = wp_parse_args( $_REQUEST, array(
			'search'  => '',
			'orderby' => 'name',
			'order'   => 'ASC',
			'number'  => 20
		) );

		extract( $http_data );

		wp_send_json_success( get_terms( array(
			'taxonomy'   => self::TAX_TAG,
			'hide_empty' => false,
			'number'     => $number,
			'search'     => $search,
			'orderby'    => $orderby,
			'order'      => $order
		) ) );
	}

	/**
	 *
	 * Get thumbnail ID
	 *
	 * @param WP_Term $term
	 */
	public function get_thumbnail_id( $term ) {
		return get_term_meta( $term->term_id, 'thumbnail_id', true );
	}

	/**
	 *
	 * Get term thumbnail image URL
	 *
	 * @since 2.2.1
	 *
	 * @param  WP_Term  $term
	 * @return string
	 */
	public function get_thumbnail_url( $term, $size = 'large' ) {

		if ( is_int( $term ) ) {
			$term = get_term( $term );
		}

		if ( ! $term ) {
			return;
		}

		$thumbnail_url = $this->get_thumbnail_id( $term );

		if ( wp_http_validate_url( $thumbnail_url ) ) {
			return $thumbnail_url;
		}

		if ( wp_attachment_is_image( $thumbnail_url ) ) {
			$thumbnail_url = wp_get_attachment_image_url( $thumbnail_url, $size );
		}

		/**
		 *
		 * Filter the thumbnail URL
		 *
		 * @param string  $thumbnail_url
		 * @param WP_Term $term
		 * @param string  $size
		 */
		return apply_filters( 'streamtube/core/taxonomy/thumbnail_url', $thumbnail_url, $term, $size );
	}

	/**
	 *
	 * Add thumbnail field
	 *
	 * @since 2.2.1
	 *
	 * @param string $taxonomy
	 */
	public function add_thumbnail_field( $taxonomy ) {

		wp_enqueue_media();

		?>
		<div class="form-field term-field-wrap">
			<div class="metabox-wrap">
				<label for="thumbnail_id">
					<?php esc_html_e( 'Featured Image', 'streamtube-core' ); ?>
				</label>

				<div class="field-group field-group-upload">

					<?php printf(
						'<button type="button" class="button-upload %s" data-media-type="image" data-media-source="url">',
						is_admin() ? 'button button-secondary button-small' : 'btn btn-secondary btn-sm'
					); ?>
					<?php esc_html_e( 'Upload An Image', 'streamtube-core' ); ?>
					</button>

					<div class="placeholder-image bg-white my-4 no-image ratio ratio-16x9">
						<button type="button" class="button button-secondary btn btn-danger btn-sm button-delete">
							<span class="dashicons dashicons-no position-absolute top-50 start-50 translate-middle"></span>
						</button>
					</div>

					<input class="input-field" name="tax_meta[thumbnail_id]" id="thumbnail_id" type="hidden" value="">
				</div>
			</div>
		</div>
		<?php

		wp_nonce_field( 'update_thumbnail_image', 'update_thumbnail_image' );
	}

	/**
	 *
	 * Add thumbnail field
	 *
	 * @since 2.2.1
	 *
	 * @param string $taxonomy
	 */
	public function edit_thumbnail_field( $term, $taxonomy ) {

		wp_enqueue_media();

		$thumbnail_url = $this->get_thumbnail_url( $term->term_id, 'large' );
		?>

		<tr class="form-field term-description-wrap">
			<th scope="row">
				<label for="thumbnail_id">
					<?php esc_html_e( 'Featured Image', 'streamtube-core' ); ?>
				</label>
			</th>

			<td>
				<div class="metabox-wrap">
					<div class="field-group field-group-upload">
						<?php printf(
							'<button type="button" class="button-upload %s" data-media-type="image" data-media-source="url">',
							is_admin() ? 'button button-secondary button-small' : 'btn btn-secondary btn-sm'
						); ?>
						<?php esc_html_e( 'Upload An Image', 'streamtube-core' ); ?>
						</button>

						<?php printf(
							'<div class="placeholder-image bg-white my-4 ratio ratio-16x9 %s">',
							! $thumbnail_url ? 'no-image' : ''
						); ?>

						<button type="button" class="button button-secondary button-delete btn btn-danger btn-sm">
							<span class="dashicons dashicons-no position-absolute top-50 start-50 translate-middle"></span>
						</button>

						<?php if ( $thumbnail_url ) {
							printf(
								'<img src="%s">',
								esc_url( $thumbnail_url )
							);
						} ?>

					</div>

					<?php printf(
						'<input class="input-field" name="tax_meta[thumbnail_id]" id="thumbnail_id" type="hidden" value="%s">',
						esc_attr( $thumbnail_url )
					) ?>
				</div>
				</div>
			</td>
		</tr>
		<?php

		wp_nonce_field( 'update_thumbnail_image', 'update_thumbnail_image' );
	}

	/**
	 *
	 * Update thumbnail image
	 *
	 * @since 2.2.1
	 *
	 * @param int    $term
	 * @param string $taxonomy
	 */
	public function update_thumbnail_field( $term ) {

		if ( ! current_user_can( 'administrator' ) ) {
			return;
		}

		if ( ! isset( $_POST['update_thumbnail_image'] ) ) {
			return;
		}

		if ( ! wp_verify_nonce( $_POST['update_thumbnail_image'], 'update_thumbnail_image' ) ) {
			return;
		}

		if ( isset( $_POST['tax_meta'] ) ) {
			$tax_meta = wp_unslash( $_POST['tax_meta'] );

			foreach ( $tax_meta as $key => $value ) {

				$value = sanitize_text_field( $value );

				if ( $key == 'thumbnail_id' ) {
					if ( ! wp_attachment_is_image( $value ) ) {
						$maybe_attachment_id = attachment_url_to_postid( $value );

						if ( wp_attachment_is_image( $maybe_attachment_id ) ) {
							$value = $maybe_attachment_id;
						}
					}
				}

				update_term_meta( $term, $key, $value );
			}
		}
	}

	/**
	 *
	 * Add Thumbnail column
	 *
	 * @since 2.2.1
	 */
	public function add_thumbnail_column( $columns ) {
		return array_merge( $columns, array(
			'thumbnail' => esc_html__( 'Thumbnail', 'streamtube-core' )
		) );
	}

	/**
	 *
	 * Add Thumbnail content
	 *
	 * @since 2.2.1
	 */
	public function add_thumbnail_column_content( $content, $column_name, $term_id ) {
		if ( $column_name == 'thumbnail' ) {
			$thumbnail_url = $this->get_thumbnail_url( $term_id );

			if ( $thumbnail_url ) {

				$content = sprintf(
					'<div class="ratio ratio-16x9"><img src="%s"></div>',
					esc_url( $thumbnail_url )
				);

			}
		}

		return $content;
	}

	/**
	 * Adds dropdown filters for taxonomies in the admin post list.
	 *
	 *                          Default is an empty string.
	 * @param string $post_type The post type for which to add the dropdown filters.
	 * @param string $which     Optional. The location of the extra table nav markup: 'top' or 'bottom'.
	 */
	public function add_dropdown_filters( $post_type, $which = '' ) {

		$taxonomies = get_object_taxonomies( $post_type, 'object' );

		if ( ! $taxonomies ) {
			return;
		}

		$is_moderate = Streamtube_Core_Permission::moderate_posts( 0, $post_type );

		foreach ( $taxonomies as $taxonomy => $object ) {

			$can_use_filter = apply_filters(
				'streamtube/core/taxonomy_filter',
				! empty( $object->show_in_filter_cap ) && current_user_can( $object->show_in_filter_cap ),
				$taxonomy,
				$object
			);

			if ( wp_validate_boolean( $object->show_in_filter ) && $can_use_filter ) {

				$q                = is_admin() ? $taxonomy : 'tax_' . $taxonomy;
				$current          = isset( $_REQUEST[ $q ] ) ? wp_unslash( $_REQUEST[ $q ] ) : '';
				$dropdown_options = array(
					'show_option_all'   => get_taxonomy( $taxonomy )->labels->all_items,
					'option_none_value' => '0',
					'hide_empty'        => 1,
					'hide_if_empty'     => true,
					'hierarchical'      => is_taxonomy_hierarchical( $taxonomy ),
					'show_count'        => $is_moderate,
					'orderby'           => 'name',
					'value_field'       => 'slug',
					'taxonomy'          => $taxonomy,
					'name'              => $q,
					'selected'          => $current,
					'class'             => 'postform select-select2'
				);

				/**
				 * Filters the arguments for the taxonomy filter dropdown.
				 *
				 * This filter allows customization of the dropdown options for taxonomy filters.
				 *
				 * @param array  $dropdown_options The current dropdown options.
				 * @param string $taxonomy         The taxonomy slug.
				 * @param string $post_type        The post type slug.
				 * @param string $which            The location of the dropdown ('top' or 'bottom').
				 */
				$dropdown_options = apply_filters(
					'streamtube/core/taxonomy_filter_dropdown_args',
					$dropdown_options,
					$taxonomy,
					$post_type,
					$which
				);

				wp_dropdown_categories( $dropdown_options );

				if ( strpos( $dropdown_options['class'], 'select-select2' ) !== false ) {
					wp_enqueue_style( 'select2' );
					wp_enqueue_script( 'select2' );

					wp_add_inline_script( 'streamtube-core-scripts', "
						jQuery(function($) {
							try {
								$('select.select-select2').select2();
							} catch (e) {
								console.error('Select2 initialization failed:', e);
							}
						});
					" );
				}
			}
		}
	}

	/**
	 *
	 * Filter edit term URL
	 * Only apply to frontend side
	 *
	 */
	public function filter_edit_term_link( $location, $term_id, $taxonomy, $object_type ) {

		if ( ! is_admin() || ( wp_doing_ajax() && isset( $_REQUEST['is_frontend'] ) ) ) {
			$location = add_query_arg(
				array(
					'tag_ID' => $term_id
				),
				wp_doing_ajax() ? $_SERVER['HTTP_REFERER'] : $_SERVER['REQUEST_URI']
			);
		}

		return $location;
	}

	/**
	 *
	 * Filter the actions in WP terms list table
	 * Only apply to frontend table
	 */
	public function filter_tag_row_actions( $actions, $tag ) {

		if ( streamtube_core_is_frontend_request() ) {
			if ( is_array( $actions ) && array_key_exists( 'delete', $actions ) ) {

				$url = wp_doing_ajax() ? $_SERVER['HTTP_REFERER'] : $_SERVER['REQUEST_URI'];

				$actions['delete'] = sprintf(
					'<a href="%s" class="delete-tag aria-button-if-js" aria-label="%s">%s</a>',
					esc_url( add_query_arg( array(
						'action'   => 'delete',
						'taxonomy' => $tag->taxonomy,
						'tag_ID'   => $tag->term_id,
						'_wpnonce' => wp_create_nonce( 'delete-tag_' . $tag->term_id )
					), $url ) ),
					esc_attr( sprintf( esc_html__( 'Delete &#8220;%s&#8221;', 'streamtube-core' ), $tag->name ) ),
					esc_html__( 'Delete', 'streamtube-core' )
				);
			}
		}

		return $actions;

	}

	/**
	 *
	 * Filter `register_taxonomy_args`
	 * Set `show_ui_frontend` to false
	 *
	 */
	public function filter_register_taxonomy_args( $args, $name, $object_type ) {
		$taxonomies = array( 'post_tag', 'category', 'product_cat', 'product_tag' );

		$args = wp_parse_args( $args, array(
			'_builtin'           => false,
			'public'             => false,
			'show_in_filter'     => false,
			'show_in_filter_cap' => 'read',
			'capabilities'       => array(),
			'builtby'            => ''
		) );

		$post_type = is_array( $object_type ) ? $object_type[0] : 'post';

		/**
		 * Modifies the capabilities for custom taxonomies.
		 *
		 * Custom capabilities added:
		 * - `manage_terms`: Capability to manage terms for the taxonomy.
		 * - `edit_terms`: Capability to edit terms for the taxonomy.
		 * - `delete_terms`: Capability to delete terms for the taxonomy.
		 * - `assign_terms`: Capability to assign terms to posts of the associated post type.
		 */
		if (
			! in_array( $name, array( self::TAX_CATEGORY, self::TAX_TAG ) ) &&
			! $args['_builtin'] &&
			$args['builtby'] === 'acf' ) {
			$args['capabilities'] = array_merge( $args['capabilities'], array(
				'manage_terms' => "manage_{$name}",
				'edit_terms'   => "edit_{$name}",
				'delete_terms' => "delete_{$name}",
				'assign_terms' => "edit_{$post_type}s"
			) );
		}

		if ( in_array( $name, $taxonomies ) ) {
			$args['public'] = true;
		}

		if ( ! isset( $args['show_ui_frontend'] ) ) {
			$args['show_ui_frontend'] = $args['public'];
		}

		if ( empty( $args['show_in_filter_cap'] ) && isset( $args['capabilities']['assign_terms'] ) ) {
			$args['show_in_filter_cap'] = $args['capabilities']['assign_terms'];
		}

		return $args;
	}

	/**
	 *
	 * Filter Count Post url using `clean_url` filter hook
	 *
	 */
	public function filter_count_post_url( $good_protocol_url, $original_url ) {
		if ( wp_doing_ajax() && isset( $_REQUEST['action'] ) && $_REQUEST['action'] === 'inline-save-tax' && isset( $_REQUEST['is_frontend'] ) ) {

			if ( isset( $_REQUEST['taxonomy'] ) && strpos( $original_url, 'edit.php?' . $_REQUEST['taxonomy'] . '=' ) === 0 ) {

				$http_data = wp_parse_args( $_REQUEST, array(
					'post_type' => 'post',
					'taxonomy'  => 'category',
					'tax_ID'    => 0
				) );

				$url = sprintf(
					'%s/dashboard/%s',
					untrailingslashit( get_author_posts_url( get_current_user_id() ) ),
					$http_data['post_type']
				);

				$good_protocol_url = add_query_arg(
					array(
						'taxonomy' => $http_data['taxonomy'],
						'term_id'  => $http_data['tax_ID']
					),
					$url
				);
			}
		}

		return $good_protocol_url;
	}
}