<?php
/**
 * WP Post List Table
 *
 * @link       https://themeforest.net/user/phpface
 * @since      1.0.0
 *
 * @package    Streamtube_Core
 * @subpackage Streamtube_Core/includes
 */

/**
 * @since      1.0.0
 * @package    Streamtube_Core
 * @subpackage Streamtube_Core/includes
 * @author     phpface <nttoanbrvt@gmail.com>
 */

if ( ! defined( 'ABSPATH' ) ) {
	exit;
}
class StreamTube_Core_Posts_List_Table extends WP_Posts_List_Table {

	/**
	 * Holds the number of posts for this user.
	 *
	 * @since 3.1.0
	 * @var int
	 */
	private $user_posts_count;

	/**
	 * Holds the number of posts which are sticky.
	 *
	 * @since 3.1.0
	 * @var int
	 */
	private $sticky_posts_count = 0;

	private $is_trash;

	private $post_type;

	private $errors = false;

	function __construct( $args = array() ) {

		if ( ! post_type_exists( $args['post_type'] ) ) {
			$this->errors = new WP_Error( 'Invalid_post_type', esc_html__( 'Invalid post type specified.', 'streamtube-core' ) );
		} else {
			$this->post_type = $args['post_type'];
			parent::__construct( $args );
		}
	}

	/**
	 *
	 * Check if instance returns error
	 *
	 * @return WP_Error
	 */
	public function has_errors() {
		return $this->errors;
	}

	public function get_post_type_object() {
		return get_post_type_object( $this->post_type );
	}

	/**
	 *
	 * Count posts
	 *
	 * @return object
	 */
	private function wp_count_posts() {
		global $wpdb;

		$type = $this->post_type;
		$perm = 'readable';

		$cache_key = _count_posts_cache_key( $type, $perm );

		$counts = wp_cache_get( $cache_key, 'counts' );
		if ( false !== $counts ) {
			// We may have cached this before every status was registered.
			foreach ( get_post_stati() as $status ) {
				if ( ! isset( $counts->{$status} ) ) {
					$counts->{$status} = 0;
				}
			}

			/** This filter is documented in wp-includes/post.php */
			return apply_filters( 'streamtube/core/list_table/wp_count_posts', $counts, $type, $perm );
		}

		$query = "SELECT post_status, COUNT( * ) AS num_posts FROM {$wpdb->posts} WHERE post_type = %s AND post_author = %d";

		if ( 'readable' === $perm && is_user_logged_in() ) {
			$post_type_object = get_post_type_object( $type );
			if ( ! current_user_can( $post_type_object->cap->edit_others_posts ) ) {
				$query .= $wpdb->prepare(
					" AND (post_status != 'private' OR ( post_author = %d AND post_status = 'private' ))",
					get_current_user_id()
				);
			}
		}

		$query .= ' GROUP BY post_status';

		$results = (array) $wpdb->get_results( $wpdb->prepare( $query, $type, get_current_user_id() ), ARRAY_A );
		$counts  = array_fill_keys( get_post_stati(), 0 );

		foreach ( $results as $row ) {
			$counts[ $row['post_status'] ] = $row['num_posts'];
		}

		$counts = (object) $counts;
		wp_cache_set( $cache_key, $counts, 'counts' );

		/**
		 * Filters the post counts by status for the current post type.
		 *
		 * @since 3.7.0
		 *
		 * @param stdClass $counts An object containing the current post_type's post
		 *                         counts by status.
		 * @param string   $type   Post type.
		 * @param string   $perm   The permission to determine if the posts are 'readable'
		 *                         by the current user.
		 */
		return apply_filters( 'streamtube/core/list_table/wp_count_posts', $counts, $type, $perm );
	}

	/**
	 * @global array $locked_post_status This seems to be deprecated.
	 * @global array $avail_post_stati
	 * @return array
	 */
	protected function get_views() {
		global $locked_post_status, $avail_post_stati;

		$post_type = $this->screen->post_type;

		if ( ! empty( $locked_post_status ) ) {
			return array();
		}

		$url = remove_query_arg( array_keys( $_GET ), $_SERVER['REQUEST_URI'] );

		$status_links = array();
		$num_posts    = $this->wp_count_posts();

		$total_posts = array_sum( (array) $num_posts );

		$class = '';

		$current_user_id = get_current_user_id();
		$all_args        = array( 'post_status' => 'all' );
		$mine            = '';

		// Subtract post types that are not included in the admin all list.
		foreach ( get_post_stati( array( 'show_in_admin_all_list' => false ) ) as $state ) {
			$total_posts -= $num_posts->$state;
		}

		if ( $this->user_posts_count && $this->user_posts_count !== $total_posts ) {
			if ( isset( $_GET['author'] ) && ( $current_user_id === (int) $_GET['author'] ) ) {
				$class = 'current';
			}

			$mine_args = array(
				'author' => $current_user_id
			);

			$mine_inner_html = sprintf(
				/* translators: %s: Number of posts. */
				_nx(
					'Mine <span class="count">(%s)</span>',
					'Mine <span class="count">(%s)</span>',
					$this->user_posts_count,
					'posts'
				),
				number_format_i18n( $this->user_posts_count )
			);

			$mine = array(
				'url'     => esc_url( add_query_arg( $mine_args, '' ) ),
				'label'   => $mine_inner_html,
				'current' => isset( $_GET['author'] ) && ( $current_user_id === (int) $_GET['author'] )
			);

			$all_args['all_posts'] = 1;
			$class                 = '';
		}

		$all_inner_html = sprintf(
			/* translators: %s: Number of posts. */
			_nx(
				'All <span class="count">(%s)</span>',
				'All <span class="count">(%s)</span>',
				$total_posts,
				'posts'
			),
			number_format_i18n( $total_posts )
		);

		$status_links['all'] = array(
			'url'     => esc_url( $url ),
			'label'   => $all_inner_html,
			'current' => empty( $class ) && ( $this->is_base_request() || isset( $_REQUEST['all_posts'] ) )
		);

		if ( $mine ) {
			$status_links['mine'] = $mine;
		}

		foreach ( get_post_stati( array( 'show_in_admin_status_list' => true ), 'objects' ) as $status ) {
			$class = '';

			$status_name = $status->name;

			if ( ! in_array( $status_name, $avail_post_stati, true ) || empty( $num_posts->$status_name ) ) {
				continue;
			}

			if ( isset( $_REQUEST['post_status'] ) && $status_name === $_REQUEST['post_status'] ) {
				$class = 'current';
			}

			$status_args = array(
				'post_status' => $status_name
			);

			$status_label = sprintf(
				translate_nooped_plural( $status->label_count, $num_posts->$status_name ),
				number_format_i18n( $num_posts->$status_name )
			);

			$status_links[ $status_name ] = array(
				'url'     => esc_url( add_query_arg( $status_args, $url ) ),
				'label'   => $status_label,
				'current' => isset( $_REQUEST['post_status'] ) && $status_name === $_REQUEST['post_status']
			);
		}

		if ( ! empty( $this->sticky_posts_count ) ) {
			$class = ! empty( $_REQUEST['show_sticky'] ) ? 'current' : '';

			$sticky_args = array(
				'show_sticky' => 1
			);

			$sticky_inner_html = sprintf(
				/* translators: %s: Number of posts. */
				_nx(
					'Sticky <span class="count">(%s)</span>',
					'Sticky <span class="count">(%s)</span>',
					$this->sticky_posts_count,
					'posts'
				),
				number_format_i18n( $this->sticky_posts_count )
			);

			$sticky_link = array(
				'sticky' => array(
					'url'     => esc_url( add_query_arg( $sticky_args, $url ) ),
					'label'   => $sticky_inner_html,
					'current' => ! empty( $_REQUEST['show_sticky'] )
				)
			);

			// Sticky comes after Publish, or if not listed, after All.
			$split        = 1 + array_search( ( isset( $status_links['publish'] ) ? 'publish' : 'all' ), array_keys( $status_links ), true );
			$status_links = array_merge( array_slice( $status_links, 0, $split ), $sticky_link, array_slice( $status_links, $split ) );
		}

		if ( post_type_supports( $post_type, 'post_password' ) ) {
			$status_links['password_protected'] = array(
				'url'     => add_query_arg( array( 'post_status' => 'password_protected' ), $url ),
				'label'   => esc_html__( 'Password Protected', 'wp-cloudflare-stream' ),
				'current' => isset( $_REQUEST['post_status'] ) && $_REQUEST['post_status'] == 'password_protected' ? true : false
			);
		}

		/**
		 *
		 * Filter status links
		 *
		 * @param array $status_links
		 * @param string $post_type
		 */
		$status_links = apply_filters( 'streamtube/core/posts_list_table/status_links', $status_links, $post_type, $url );

		return $this->get_views_links( $status_links );
	}

	/**
	 * Creates a link to edit.php with params.
	 *
	 * @since 4.4.0
	 *
	 * @param string[] $args      Associative array of URL parameters for the link.
	 * @param string   $link_text Link text.
	 * @param string   $css_class Optional. Class attribute. Default empty string.
	 * @return string The formatted link string.
	 */
	protected function get_edit_link( $args, $link_text, $css_class = '' ) {

		// Remove post type as it would occur error.
		if ( array_key_exists( 'post_type', $args ) ) {
			unset( $args['post_type'] );
		}

		$taxonomies = get_object_taxonomies( $this->screen->post_type, 'object' );

		if ( $taxonomies ) {
			foreach ( $taxonomies as $tax => $object ) {

				$tax_query_var = $object->query_var ? $object->query_var : $tax;

				if ( array_key_exists( $tax_query_var, $args ) ) {
					$args[ 'tax_' . $tax_query_var ] = $args[ $tax_query_var ];
					unset( $args[ $tax_query_var ] );
				}
			}
		}

		$url = add_query_arg( $args, $_SERVER['REQUEST_URI'] );

		$class_html   = '';
		$aria_current = '';

		if ( ! empty( $css_class ) ) {
			$class_html = sprintf(
				' class="%s"',
				esc_attr( $css_class )
			);

			if ( 'current' === $css_class ) {
				$aria_current = ' aria-current="page"';
			}
		}

		return sprintf(
			'<a href="%s"%s%s>%s</a>',
			esc_url( $url ),
			$class_html,
			$aria_current,
			$link_text
		);
	}

	/**
	 * Displays a comment count bubble.
	 *
	 * @since 3.1.0
	 *
	 * @param int $post_id          The post ID.
	 * @param int $pending_comments Number of pending comments.
	 */
	protected function comments_bubble( $post_id, $pending_comments ) {
		$post_object   = get_post( $post_id );
		$edit_post_cap = $post_object ? 'edit_post' : 'edit_posts';

		if (
			! current_user_can( $edit_post_cap, $post_id )
			&& ( post_password_required( $post_id )
				|| ! current_user_can( 'read_post', $post_id ) )
		) {
			// The user has no access to the post and thus cannot see the comments.
			return false;
		}

		$approved_comments = get_comments_number();

		$approved_comments_number = number_format_i18n( $approved_comments );
		$pending_comments_number  = number_format_i18n( $pending_comments );

		$approved_only_phrase = sprintf(
			/* translators: %s: Number of comments. */
			_n( '%s comment', '%s comments', $approved_comments ),
			$approved_comments_number
		);

		$approved_phrase = sprintf(
			/* translators: %s: Number of comments. */
			_n( '%s approved comment', '%s approved comments', $approved_comments ),
			$approved_comments_number
		);

		$pending_phrase = sprintf(
			/* translators: %s: Number of comments. */
			_n( '%s pending comment', '%s pending comments', $pending_comments ),
			$pending_comments_number
		);

		if ( ! $approved_comments && ! $pending_comments ) {
			// No comments at all.
			printf(
				'<span aria-hidden="true">&#8212;</span>' .
				'<span class="screen-reader-text">%s</span>',
				__( 'No comments' )
			);
		} elseif ( $approved_comments && 'trash' === get_post_status( $post_id ) ) {
			// Don't link the comment bubble for a trashed post.
			printf(
				'<span class="post-com-count post-com-count-approved">' .
				'<span class="comment-count-approved" aria-hidden="true">%s</span>' .
				'<span class="screen-reader-text">%s</span>' .
				'</span>',
				$approved_comments_number,
				$pending_comments ? $approved_phrase : $approved_only_phrase
			);
		} elseif ( $approved_comments ) {
			// Link the comment bubble to approved comments.
			printf(
				'<a href="%s" class="post-com-count post-com-count-approved">' .
				'<span class="comment-count-approved" aria-hidden="true">%s</span>' .
				'<span class="screen-reader-text">%s</span>' .
				'</a>',
				esc_url(
					add_query_arg(
						array(
							'post_id'        => $post_id,
							'comment_status' => 'approved'
						),
						trailingslashit( get_author_posts_url( get_current_user_id() ) ) . 'dashboard/comments'
					)
				),
				$approved_comments_number,
				$pending_comments ? $approved_phrase : $approved_only_phrase
			);
		} else {
			// Don't link the comment bubble when there are no approved comments.
			printf(
				'<span class="post-com-count post-com-count-no-comments">' .
				'<span class="comment-count comment-count-no-comments" aria-hidden="true">%s</span>' .
				'<span class="screen-reader-text">%s</span>' .
				'</span>',
				$approved_comments_number,
				$pending_comments ?
				/* translators: Hidden accessibility text. */
				__( 'No approved comments' ) :
				/* translators: Hidden accessibility text. */
				__( 'No comments' )
			);
		}

		if ( $pending_comments ) {
			printf(
				'<a href="%s" class="post-com-count post-com-count-pending">' .
				'<span class="comment-count-pending" aria-hidden="true">%s</span>' .
				'<span class="screen-reader-text">%s</span>' .
				'</a>',
				esc_url(
					add_query_arg(
						array(
							'post_id'        => $post_id,
							'comment_status' => 'moderated'
						),
						trailingslashit( get_author_posts_url( get_current_user_id() ) ) . 'dashboard/comments'
					)
				),
				$pending_comments_number,
				$pending_phrase
			);
		} else {
			printf(
				'<span class="post-com-count post-com-count-pending post-com-count-no-pending">' .
				'<span class="comment-count comment-count-no-pending" aria-hidden="true">%s</span>' .
				'<span class="screen-reader-text">%s</span>' .
				'</span>',
				$pending_comments_number,
				$approved_comments ?
				/* translators: Hidden accessibility text. */
				__( 'No pending comments' ) :
				/* translators: Hidden accessibility text. */
				__( 'No comments' )
			);
		}
	}

	/**
	 * Handles the post author column output.
	 *
	 * @since 4.3.0
	 *
	 * @param WP_Post $post The current WP_Post object.
	 */
	public function column_author( $post ) {
		$args = array(
			'author_id' => get_the_author_meta( 'ID' )
		);
		echo $this->get_edit_link( $args, get_the_author() );
	}

	/**
	 * Displays a dropdown for filtering items in the list table by month.
	 *
	 * @since 3.1.0
	 *
	 * @global wpdb      $wpdb      WordPress database abstraction object.
	 * @global WP_Locale $wp_locale WordPress date and time locale object.
	 *
	 * @param string $post_type The post type.
	 */
	protected function months_dropdown( $post_type ) {
		global $wpdb, $wp_locale;

		/**
		 * Filters whether to remove the 'Months' drop-down from the post list table.
		 *
		 * @since 4.2.0
		 *
		 * @param bool   $disable   Whether to disable the drop-down. Default false.
		 * @param string $post_type The post type.
		 */
		if ( apply_filters( 'disable_months_dropdown', false, $post_type ) ) {
			return;
		}

		/**
		 * Filters whether to short-circuit performing the months dropdown query.
		 *
		 * @since 5.7.0
		 *
		 * @param object[]|false $months   'Months' drop-down results. Default false.
		 * @param string         $post_type The post type.
		 */
		$months = apply_filters( 'pre_months_dropdown_query', false, $post_type );

		if ( ! is_array( $months ) ) {
			$extra_checks = "AND post_status != 'auto-draft'";
			if ( ! isset( $_GET['post_status'] ) || 'trash' !== $_GET['post_status'] ) {
				$extra_checks .= " AND post_status != 'trash'";
			} elseif ( isset( $_GET['post_status'] ) ) {
				$extra_checks = $wpdb->prepare( ' AND post_status = %s', $_GET['post_status'] );
			}

			$months = $wpdb->get_results(
				$wpdb->prepare(
					"SELECT DISTINCT YEAR( post_date ) AS year, MONTH( post_date ) AS month
					FROM $wpdb->posts
					WHERE post_type = %s
					$extra_checks
					ORDER BY post_date DESC",
					$post_type
				)
			);
		}

		/**
		 * Filters the 'Months' drop-down results.
		 *
		 * @since 3.7.0
		 *
		 * @param object[] $months    Array of the months drop-down query results.
		 * @param string   $post_type The post type.
		 */
		$months = apply_filters( 'months_dropdown_results', $months, $post_type );

		$month_count = count( $months );

		if ( ! $month_count || ( 1 === $month_count && 0 === (int) $months[0]->month ) ) {
			return;
		}

		$m = isset( $_GET['month'] ) ? (int) $_GET['month'] : 0;

		printf(
			'<label for="filter-by-date" class="screen-reader-text">%s</label>',
			get_post_type_object( $post_type )->labels->filter_by_date
		);

		echo '<select name="month" id="filter-by-date">';

		printf(
			'<option %s value="0">%s</option>',
			selected( $m, 0, false ),
			esc_html__( 'All dates', 'streamtube-core' )
		);

		foreach ( $months as $arc_row ) :
			if ( 0 === (int) $arc_row->year ) {
				continue;
			}

			$month = zeroise( $arc_row->month, 2 );
			$year  = $arc_row->year;

			printf(
				"<option %s value='%s'>%s</option>\n",
				selected( $m, $year . $month, false ),
				esc_attr( $arc_row->year . $month ),
				/* translators: 1: Month name, 2: 4-digit year. */
				sprintf( __( '%1$s %2$d' ), $wp_locale->get_month( $month ), $year )
			);
		endforeach;

		echo '</select>';
	}

	/**
	 * @return array
	 */
	protected function get_bulk_actions() {
		$actions       = array();
		$post_type_obj = get_post_type_object( $this->screen->post_type );
		if ( current_user_can( $post_type_obj->cap->edit_posts ) ) {
			if ( $this->is_trash ) {
				$actions['untrash'] = esc_html__( 'Restore', 'streamtube-core' );
			} else {
				$actions['edit'] = esc_html__( 'Edit', 'streamtube-core' );
			}
		}

		if ( current_user_can( $post_type_obj->cap->delete_posts ) ) {
			if ( $this->is_trash || ! EMPTY_TRASH_DAYS ) {
				$actions['delete'] = esc_html__( 'Delete permanently', 'streamtube-core' );
			} else {
				$actions['trash'] = esc_html__( 'Move to Trash', 'streamtube-core' );
			}
		}

		return $actions;
	}
	private function items_per_page() {

		$screen_id = $this->screen->id ? $this->screen->id : $this->post_type;

		$per_page       = "edit_{$screen_id}_per_page";
		$posts_per_page = (int) get_user_option( $per_page );
		if ( empty( $posts_per_page ) || $posts_per_page < 1 ) {
			$posts_per_page = 20;
		}

		/**
		 * Filters the number of items per page to show for a specific 'per_page' type.
		 *
		 * The dynamic portion of the hook name, `$post_type`, refers to the post type.
		 *
		 * Possible hook names include:
		 *
		 *  - `edit_post_per_page`
		 *  - `edit_page_per_page`
		 *  - `edit_attachment_per_page`
		 *
		 * @since 3.0.0
		 *
		 * @param int $posts_per_page Number of posts to display per page for the given post
		 *                            type. Default 20.
		 */
		$posts_per_page = apply_filters( "edit_{$screen_id}_per_page", $posts_per_page );

		/**
		 * Filters the number of posts displayed per page when specifically listing "posts".
		 *
		 * @since 2.8.0
		 *
		 * @param int    $posts_per_page Number of posts to be displayed. Default 20.
		 * @param string $post_type      The post type.
		 */
		$posts_per_page = apply_filters( 'edit_posts_per_page', $posts_per_page, $screen_id );

		return $posts_per_page;
	}

	/**
	 *
	 * Prepare query, overrides default $wp_query
	 *
	 * @return WP_Query
	 */
	private function prepare_query() {

		global $wp, $wp_query, $per_page;

		$per_page = $this->items_per_page();

		$tax_query = array();

		$q = wp_parse_args( $_GET, array(
			'post_status'    => 'any',
			'posts_per_page' => $per_page,
			'orderby'        => isset( $_REQUEST['orderby'] ) ? $_REQUEST['orderby'] : 'date',
			'order'          => isset( $_REQUEST['order'] ) ? $_REQUEST['order'] : 'desc',
			's'              => isset( $_REQUEST['s'] ) ? wp_unslash( $_REQUEST['s'] ) : '',
			'm'              => isset( $_REQUEST['month'] ) ? wp_unslash( $_REQUEST['month'] ) : ''
		) );

		if ( $q['post_status'] == 'all' ) {
			$q['post_status'] = 'any';
		}

		if ( is_post_type_hierarchical( $this->post_type ) ) {
			$q['fields']         = 'id=>parent';
			$q['posts_per_page'] = -1;
		}

		if ( ! current_user_can( $this->get_post_type_object()->cap->edit_others_posts ) ) {
			$q['author'] = get_current_user_id();
		}

		if ( current_user_can( $this->get_post_type_object()->cap->edit_others_posts ) ) {
			if ( isset( $_REQUEST['author_id'] ) && absint( $_REQUEST['author_id'] ) > 0 ) {
				$q['author'] = absint( $_REQUEST['author_id'] );
			}
		}

		if ( isset( $_REQUEST['cpaged'] ) && absint( $_REQUEST['cpaged'] ) > 0 ) {
			$q['paged'] = absint( $_REQUEST['cpaged'] );
		}

		if ( isset( $_REQUEST['post_status'] ) && wp_unslash( $_REQUEST['post_status'] ) === 'password_protected' ) {
			$q['has_password'] = true;
		}

		if ( isset( $_REQUEST['taxonomy'] ) && is_string( $_REQUEST['taxonomy'] ) ) {

			$term_id = isset( $_REQUEST['term_id'] ) ? absint( $_REQUEST['term_id'] ) : 0;

			if ( $term_id ) {
				$tax_query[] = array(
					'taxonomy' => wp_unslash( $_REQUEST['taxonomy'] ),
					'field'    => 'term_id',
					'terms'    => $term_id
				);
			}
		}

		$taxonomies = get_object_taxonomies( $this->post_type, 'object' );

		if ( $taxonomies ) {
			foreach ( $taxonomies as $tax => $object ) {
				$tax_query_var = $object->query_var ? $object->query_var : $tax;

				if ( array_key_exists( 'tax_' . $tax_query_var, $_REQUEST ) && $_REQUEST[ 'tax_' . $tax_query_var ] ) {
					$terms       = $_REQUEST[ 'tax_' . $tax_query_var ];
					$tax_query[] = array(
						'taxonomy' => $tax,
						'field'    => 'slug',
						'terms'    => wp_unslash( $terms )
					);
				}
			}
		}

		if ( count( $tax_query ) > 1 ) {
			$tax_query['relation'] = 'AND';
		}

		$q = array_merge( $q, compact( 'tax_query' ) );

		$q = array_merge( $q, array( 'post_type' => $this->post_type ) );

		/**
		 *
		 * Filter query var
		 * @param array $q
		 * @param string $post_type
		 */
		$q = apply_filters( 'streamtube/core/posts_list_table/query_args', $q, $this->post_type, $wp_query );

		/**
		 *
		 * Fires before sending query
		 *
		 */
		do_action( 'streamtube/core/posts_list_table/before_query', $q, $this->post_type, $wp_query );

		$wp_query = new WP_Query( $q );

		/**
		 *
		 * Fires after sending query
		 *
		 */
		do_action( 'streamtube/core/posts_list_table/after_query', $q, $this->post_type, $wp_query );

		return $wp_query;
	}

	/**
	 * @global string   $mode             List table view mode.
	 * @global array    $avail_post_stati
	 * @global WP_Query $wp_query         WordPress Query object.
	 * @global int      $per_page
	 */
	public function prepare_items() {
		global $mode, $avail_post_stati, $wp_query, $per_page;

		if ( ! empty( $_REQUEST['mode'] ) ) {
			$mode = 'excerpt' === $_REQUEST['mode'] ? 'excerpt' : 'list';
			set_user_setting( 'posts_list_mode', $mode );
		} else {
			$mode = get_user_setting( 'posts_list_mode', 'list' );
		}

		$this->prepare_query();

		// Is going to call wp().
		$avail_post_stati = get_available_post_statuses( $this->post_type );

		$this->set_hierarchical_display(
			is_post_type_hierarchical( $this->post_type )
		);

		if ( $this->hierarchical_display ) {
			$total_items = $wp_query->found_posts;
		} elseif ( $wp_query->found_posts || $this->get_pagenum() === 1 ) {
			$total_items = $wp_query->found_posts;
		} else {
			$post_counts = (array) $this->wp_count_posts();

			if ( isset( $_REQUEST['post_status'] ) && in_array( $_REQUEST['post_status'], $avail_post_stati, true ) ) {
				$total_items = $post_counts[ $_REQUEST['post_status'] ];
			} elseif ( isset( $_REQUEST['show_sticky'] ) && $_REQUEST['show_sticky'] ) {
				$total_items = $this->sticky_posts_count;
			} elseif ( isset( $_GET['author'] ) && get_current_user_id() === (int) $_GET['author'] ) {
				$total_items = $this->user_posts_count;
			} else {
				$total_items = array_sum( $post_counts );

				// Subtract post types that are not included in the admin all list.
				foreach ( get_post_stati( array( 'show_in_admin_all_list' => false ) ) as $state ) {
					$total_items -= $post_counts[ $state ];
				}
			}
		}

		$this->is_trash = isset( $_REQUEST['post_status'] ) && 'trash' === $_REQUEST['post_status'];

		$this->set_pagination_args(
			array(
				'total_items' => $total_items,
				'per_page'    => $per_page
			)
		);
	}

	public function output_inline_edit() {

		ob_start();

		$this->inline_edit();

		$output = ob_get_clean();

		$field = '<input type="hidden" name="is_frontend" value="on" />';

		$pattern = '/(<div class="submit inline-edit-save">)/';

		echo preg_replace( $pattern, '$1' . $field, $output );
	}

	/**
	 * Enqueue admin js
	 */
	public function enqueue_scripts() {
		if ( current_user_can( $this->get_post_type_object()->cap->edit_posts ) ) {
			wp_enqueue_script( 'wp-lists' );
			wp_enqueue_script( 'wp-tags-box' );
			wp_enqueue_script( 'wp-tags-suggest' );
			wp_enqueue_script( 'wp-inline-edit-post' );
		}
	}
}