<?php
/**
 * Define the Permission functionality
 *
 *
 * @link       https://themeforest.net/user/phpface
 * @since      1.0.0
 *
 * @package    Streamtube_Core
 * @subpackage Streamtube_Core/includes
 */

/**
 *
 * @since      1.0.0
 * @package    Streamtube_Core
 * @subpackage Streamtube_Core/includes
 * @author     phpface <nttoanbrvt@gmail.com>
 */

if( ! defined('ABSPATH' ) ){
    exit;
}

class Streamtube_Core_Permission{

    /**
     *
     * Define deactivate role
     * 
     */
    const ROLE_DEACTIVATE       =   'role_deactivated';

    /**
     *
     * Define verify role
     * 
     */
    const ROLE_VERIFY           =   'role_verified';

    /**
     *
     * Define spam role
     * 
     */
    const ROLE_SPAMMER          =   'role_spammer';

    /**
     *
     * Publish Posts Cap
     *
     * @since 2.2.3
     * 
     */
    const CAP_EDIT_OTHER_POSTS  =   'edit_others_posts';// editor or higher    

    /**
     *
     * Publish Posts Cap
     *
     * @since 2.2.3
     * 
     */
    const CAP_PUBLISH_POSTS     =   'publish_posts';// author or higher

    /**
     *
     * Edit Posts Cap
     *
     * @since 2.2.3
     * 
     */
    const CAP_EDIT_POSTS        =   'edit_posts';// Contributor or higher

    /**
     *
     * Edit Post Cap
     *
     * @since 2.2.3
     * 
     */
    const CAP_EDIT_POST         =   'edit_post';// Contributor or higher

    /**
     *
     * Delete Post Cap
     *
     * @since 2.2.3
     * 
     */
    const CAP_DELETE_POST       =   'delete_post';// Contributor or higher    

    /**
     *
     * Delete Post Cap
     *
     * @since 2.2.3
     * 
     */
    const CAP_ADD_VAST_TAG      =   'manage_vast_tag';

    /**
     *
     * Add custom roles
     * 
     */
    public static function add_roles(){

        add_role( 
            self::ROLE_VERIFY, 
            esc_html__( 'Verified', 'streamtube-core' ), 
            array(
                'read'  =>  true
            ) 
        );

        add_role( 
            self::ROLE_DEACTIVATE, 
            esc_html__( 'Deactivated', 'streamtube-core' ), 
            array(
                'read'  =>  true
            ) 
        );

        add_role( 
            self::ROLE_SPAMMER, 
            esc_html__( 'Spammer', 'streamtube-core' ), 
            array(
                'read'  =>  true
            ) 
        );         
    }

    /**
     *
     * Adds role to user.
     * 
     */
    public static function add_user_role( $user_id, $role ){
        return get_user_by( 'ID', $user_id )->add_role( $role );
    }

    /**
     *
     * Remove user role
     * 
     */
    public static function remove_user_role( $user_id, $role ){
        return get_user_by( 'ID', $user_id )->remove_role( $role );
    }    

    /**
     *
     * Do verify user
     * 
     */
    public static function verify_user( $user_id ){
        return self::add_user_role( $user_id, self::ROLE_VERIFY );
    }

    /**
     *
     * Revoke verify role for given user
     * 
     */
    public static function unverify_user( $user_id ){
        return self::remove_user_role( $user_id, self::ROLE_VERIFY );
    }

    /**
     *
     * Check if user is verified
     * 
     */
    public static function is_verified( $user_id ){

        $user = new WP_User( $user_id );

        return is_array( $user->roles ) && in_array( self::ROLE_VERIFY, $user->roles );
    }

    /**
     *
     * Check if given user is administrator
     * 
     */
    public static function can_admin( $user_id = 0 ){
        if( ! $user_id ){
            $user_id = get_current_user_id();
        }

        return user_can( $user_id, 'administrator' ) ? true : false;
    }

    /**
     *
     * Check if current user can moderate posts
     * 
     * @return true|false
     */
    public static function moderate_posts( $user_id = 0, $post_type = 'post' ){

        if( ! $user_id ){
            $user_id = get_current_user_id();
        }

        $capability = self::CAP_EDIT_OTHER_POSTS;

        if( post_type_exists( $post_type ) ){
            $capability = get_post_type_object( $post_type )->cap->edit_others_posts;
        }

        if( user_can( $user_id, 'administrator' ) || user_can( $user_id, $capability ) ){
            return true;
        }

        return false;
    }

    public static function can_create_posts( $post_type_screen = null, $user_id = 0 ){
        if( ! $post_type_screen || ! post_type_exists( $post_type_screen ) ){
            return false;
        }

        if( ! $user_id ){
            $user_id = get_current_user_id();
        }

        $post_type_object = get_post_type_object( $post_type_screen );

        return user_can( $user_id, $post_type_object->cap->create_posts );
    }

    /**
     *
     * Check if current user can edit post, requires Contributor or higher
     * 
     * @return true|false
     */
    public static function can_edit_posts( $user_id = 0, $post = null ){

        if( ! $user_id ){
            $user_id = get_current_user_id();
        }

        if( ! is_null( $post ) ){
            if( ! is_object( $post ) ){
                $post = get_post( $post );
            }

            if( ! $post || ! post_type_exists( $post->post_type ) ){
                return false;
            }            

            $post_type_object = get_post_type_object( $post->post_type );

            return user_can( $user_id, $post_type_object->cap->edit_posts );
        }

        return user_can( $user_id, self::CAP_EDIT_POSTS );
    }

    /**
     *
     * Check if current user can edit given post, requires Contributor or higher
     * 
     * @return true|false
     */
    public static function can_edit_post( $post = null, $user_id = 0 ){

        if( ! $user_id ){
            $user_id = get_current_user_id();
        }

        if( ! is_object( $post ) ){
            $post = get_post( $post );
        }        

        if( ! $post || ! post_type_exists( $post->post_type ) ){
            return false;
        }

        $post_type_object = get_post_type_object( $post->post_type );

        return user_can( $user_id, $post_type_object->cap->edit_post, $post->ID );
    }

    /**
     *
     * Check if current user can delete given post, requires Contributor or higher
     * 
     * @return true|false
     */
    public static function can_delete_post( $post = null , $user_id = 0 ){

        if( ! $user_id ){
            $user_id = get_current_user_id();
        }

        if( ! is_object( $post ) ){
            $post = get_post( $post );
        }

        if( ! post_type_exists( $post->post_type ) ){
            return false;
        }        

        $post_type_object = get_post_type_object( $post->post_type );

        return user_can( $user_id, $post_type_object->cap->delete_post, $post->ID );
    }     

    /**
     *
     * Check if current user can upload videos
     * 
     * @return true|false
     */
    public static function can_upload( $user_id = 0 ){

        if( ! $user_id ){
            $user_id = get_current_user_id();
        }

        $post_type_object   = get_post_type_object( 'video' );

        if( ! $post_type_object ){
            return false;
        }

        return user_can( $user_id, $post_type_object->cap->create_posts ) && user_can( $user_id, 'upload_files' );
    }

    /**
     *
     * Check if current user can embed videos
     * 
     * @return true|false
     */
    public static function can_embed( $user_id = 0 ){

        if( ! $user_id ){
            $user_id = get_current_user_id();
        }

        $post_type_object   = get_post_type_object( 'video' );

        if( ! $post_type_object ){
            return false;
        }
        
        return user_can( $user_id, $post_type_object->cap->create_posts ) && user_can( $user_id, 'embed_videos' );
    }

    /**
     *
     * Ask if given user is onwer of given post
     * 
     * @return boolean
     */
    public static function is_post_owner( $post = null, $user_id = 0 ){

        if( ! $user_id ){
            $user_id = get_current_user_id();
        }

        if( ! $post || ! $user_id ){
            return false;
        }

        if( is_int( $post ) ){
            $post = get_post( $post );
        }

        if( ! is_object( $post ) ){
            return false;
        }

        return $post->post_author == $user_id ? true : false;
    }

    /**
     *
     * Check if current user can manage given taxonomy
     *
     * @param string $taxonomy
     * @param int $user_id
     *
     * @return boolean
     * 
     */
    public static function can_manage_term( $taxonomy, $user_id = 0 ){

        if( ! taxonomy_exists( $taxonomy ) ){
            return false;
        }

        if( ! $user_id ){
            $user_id = get_current_user_id();
        }

        return user_can( $user_id, get_taxonomy( $taxonomy )->cap->manage_terms );
    }

    public static function can_manage_vast_tag( $user_id = 0 ){

        if( ! $user_id ){
            $user_id = get_current_user_id();
        }

        if( self::moderate_posts( $user_id ) ){
            return true;
        }

        return user_can( $user_id, self::CAP_ADD_VAST_TAG );
    }

    /**
     *
     * Check if current user can moderate posts
     * 
     * @return true|false
     */
    public static function moderate_cdn_sync( $user_id = 0 ){

        if( ! $user_id ){
            $user_id = get_current_user_id();
        }

        return user_can( $user_id, 'administrator' );
    }
}