<?php
/**
 * Define the MetaBox functionality
 *
 *
 * @link       https://themeforest.net/user/phpface
 * @since      1.0.0
 *
 * @package    Streamtube_Core
 * @subpackage Streamtube_Core/includes
 */

/**
 *
 * @since      1.0.0
 * @package    Streamtube_Core
 * @subpackage Streamtube_Core/includes
 * @author     phpface <nttoanbrvt@gmail.com>
 */

if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

class Streamtube_Core_MetaBox {

	/**
	 *
	 * Get current screen ID
	 * 
	 */
	public static function get_current_screen( $screen = '' ) {
		global $post_type_screen;

		$_current_screen = '';

		if ( $screen ) {
			$_current_screen = $screen;
		} else {
			$_current_screen = $post_type_screen ?? $_current_screen;
		}

		if ( ! $_current_screen ) {
			$_current_screen = get_query_var( 'dashboard' );
		}

		if ( $_current_screen && strpos( $_current_screen, 'dashboard_' ) === false ) {
			$_current_screen = 'dashboard_' . $_current_screen;
		} else {
			$_current_screen = 'user_dashboard';
		}

		/**
		 *
		 * Filter current screen
		 * 
		 */
		return apply_filters( 'streamtube/core/get_current_screen', $_current_screen, $screen );
	}

	/**
	 *
	 * Add metabox
	 * 
	 * @param  string $location
	 * 
	 */
	public static function _add( $screen = '', $context = 'normal' ) {
		global $post, $wp_meta_boxes;

		$screen = $screen ?? self::get_current_screen();

		if ( function_exists( 'do_meta_boxes' ) ) {
			do_action( 'streamtube_add_meta_boxes', $screen, $post );
			do_meta_boxes( $screen, $context, $post );
		}
	}

	/**
	 *
	 * Register the aside metabox
	 * 
	 */
	public function side() {
		return self::_add( null, 'side' );
	}

	/**
	 *
	 * Register the normal metabox
	 * 
	 */
	public function normal() {
		return self::_add( null, 'normal' );
	}

	/**
	 *
	 * Register the advanced metabox
	 * 
	 */
	public function advanced() {
		return self::_add( null, 'advanced' );
	}

	/**
	 *
	 * Register the Dashboard metabox
	 * 
	 */
	public function user_dashboard( $context ) {
		return self::_add( 'user_dashboard', $context );
	}

	/**
	 *
	 * Register the GoogleSitekit metabox
	 * 
	 */
	public function googlesitekit() {
		return self::_add( 'dashboard_analytics' );
	}

	/**
	 * Adds a meta box to one or more screens on the frontend dashboard
	 * Similiar to add_meta_box()
	 *
	 * @see add_meta_box()
	 * 
	 */
	public static function add( $id, $title, $callback, $screen = null, $context = 'advanced', $priority = 'default', $callback_args = null ) {

		if ( is_array( $screen ) ) {
			$_uid = md5( json_encode( $screen ) );
		} else {
			$_uid = $screen;
		}

		if ( ! $id || did_action( $id . '-' . $_uid ) ) {
			return;
		}

		add_meta_box( $id, $title, $callback, $screen, $context, $priority, $callback_args );

		do_action( $id . '-' . $_uid );
	}

	/**
	 *
	 * Remove metabox
	 *
	 * @see remove_meta_box()
	 * 
	 */
	public static function remove( $id, $screen, $context = 'normal' ) {
		remove_meta_box( $id, $screen, $context );
	}

	/**
	 *
	 * Publish metabox
	 * 
	 */
	public function box_publish( $post = null ) {

		$post_status = is_object( $post ) ? $post->post_status : '';

		if ( $post_status === 'future' ) {
			$post_status = 'publish';
		}

		streamtube_core_the_field_control( array(
			'label'   => esc_html__( 'Visibility', 'streamtube-core' ),
			'type'    => 'select',
			'name'    => 'post_status',
			'current' => $post_status,
			'options' => streamtube_core_get_post_statuses( $post )
		) );

		if ( post_type_supports( $post->post_type, 'post_password' ) ) {
			streamtube_core_the_field_control( array(
				'label' => esc_html__( 'Password Protected', 'streamtube-core' ),
				'type'  => 'password',
				'name'  => 'post_password',
				'id'    => 'post_password',
				'value' => $post ? $post->post_password : ''
			) );
		}

		streamtube_core_the_field_control( array(
			'label' => esc_html__( 'Publish Date', 'streamtube-core' ),
			'type'  => 'datetime-local',
			'name'  => 'post_date',
			'value' => $post ? date( 'Y-m-d\TH:i', strtotime( $post->post_date ) ) : ''
		) );
	}

	/**
	 *
	 * Excerpt metabox
	 * 
	 */
	public function box_excerpt( $post ) {
		printf(
			'<textarea name="excerpt" class="form-control">%s</textarea>',
			esc_textarea( $post->post_excerpt )
		);
	}

	/**
	 *
	 * Video Source metabox
	 * 
	 */
	public function box_video_sources( $post ) {

		global $streamtube;

		if ( current_user_can( 'edit_video_trailer' ) ) {
			streamtube_core_the_field_control( array(
				'label'   => esc_html__( 'Trailer', 'streamtube-core' ),
				'type'    => 'text',
				'name'    => 'video_trailer',
				'value'   => $post ? esc_attr( $streamtube->get()->post->get_video_trailer( $post->ID ) ) : '',
				'wpmedia' => true
			) );
		}

		if ( current_user_can( 'edit_video_source' ) ) {
			streamtube_core_the_field_control( array(
				'label'   => esc_html__( 'Main Source', 'streamtube-core' ),
				'type'    => 'text',
				'name'    => 'video_source',
				'value'   => $post ? esc_attr( $streamtube->get()->post->get_source( $post->ID ) ) : '',
				'wpmedia' => true
			) );
		}
	}

	/**
	 *
	 * Additional video meta box
	 * 
	 */
	public function box_additional_video_meta( $post ) {

		global $streamtube, $post_type_screen;

		?>
		<div class="row field-group-meta-1">

			<?php if ( post_type_supports( $post_type_screen, 'upcoming' ) ) : ?>
				<div class="col-12 col-md-6">
					<?php
					$upcoming_date = $streamtube->get()->post->get_upcoming_date( $post->ID );
					streamtube_core_the_field_control( array(
						'label' => esc_html__( 'Upcoming Date', 'streamtube-core' ),
						'type'  => 'datetime-local',
						'name'  => 'meta_input[_upcoming_date]',
						'value' => $post && $upcoming_date ? date( 'Y-m-d\TH:i', strtotime( $upcoming_date ) ) : ''
					) );
					?>
				</div>
			<?php endif; ?>

			<?php if ( post_type_supports( $post_type_screen, 'aspect_ratio' ) ) : ?>
				<div class="col-12 col-md-6">
					<?php

					$default_ratios = streamtube_core_get_ratio_options();

					/**
					 *
					 * Filter default allowed ratio options
					 * 
					 */
					$default_ratios = apply_filters( 'streamtube/core/post/edit/meta/default_ratios', $default_ratios );

					streamtube_core_the_field_control( array(
						'label'   => esc_html__( 'Aspect Ratio', 'streamtube-core' ),
						'type'    => 'select',
						'name'    => 'meta_input[_aspect_ratio]',
						'current' => $post ? $streamtube->get()->post->get_aspect_ratio( $post->ID ) : '',
						'options' => array_merge( array(
							'' => esc_html__( 'Default', 'streamtube-core' )
						), $default_ratios )
					) );
					?>
				</div>
			<?php endif; ?>

			<?php if ( post_type_supports( $post_type_screen, 'length' ) ) : ?>
				<div class="col-12 col-md-6">
					<?php
					streamtube_core_the_field_control( array(
						'label' => esc_html__( 'Video Length', 'streamtube-core' ),
						'type'  => 'text',
						'name'  => 'meta_input[_length]',
						'value' => $post ? $streamtube->get()->post->get_length( $post->ID ) : ''
					) );
					?>
				</div>
			<?php endif; ?>

			<?php if ( post_type_supports( $post_type_screen, '360_degree' ) ) : ?>
				<div class="col-12 col-md-6">
					<?php
					streamtube_core_the_field_control( array(
						'label'       => esc_html__( '360 Degree Video', 'streamtube-core' ),
						'type'        => 'checkbox',
						'name'        => 'meta_input[_vr]',
						'value'       => 'vr',
						'current'     => $post ? $streamtube->get()->post->is_video_vr( $post->ID ) : '',
						'id'          => is_object( $post ) ? "vr_{$post->ID}" : "vr",
						'description' => esc_html__( 'Is this video 360 degree?', 'streamtube' )
					) );
					?>
				</div>
			<?php endif; ?>

		</div>
		<?php

	}

	/**
	 *
	 * The featured image metabox template
	 * 
	 */
	public function box_featured_image( $post ) {
		streamtube_core_load_template( 'metaboxes/thumbnail.php' );
	}

	/**
	 *
	 * Attributes box, useful for hierarchical post_type
	 * 
	 */
	public function box_attributes( $post ) {
		streamtube_core_load_template( 'metaboxes/attributes.php' );
	}

	/**
	 *
	 * Discussion metabox template
	 * 
	 */
	public function box_discussion( $post ) {
		if ( function_exists( 'post_comment_status_meta_box' ) ) {
			post_comment_status_meta_box( $post );
		}
	}

	/**
	 *
	 * The hierarchical taxonomy metabox template
	 * 
	 */
	public function box_hierarchical_taxonomy( $post, $box ) {
		streamtube_core_load_template( 'metaboxes/taxonomies-hierarchical.php', false, $box );
	}

	/**
	 *
	 * The tag metabox template
	 * 
	 */
	public function box_tag( $post, $box ) {
		streamtube_core_load_template( 'metaboxes/taxonomies-tags.php', false, $box );
	}

	/**
	 *
	 * Register default metaboxes
	 * 
	 */
	public function add_meta_boxes() {

		global $post_type_screen;

		if ( ! $post_type_screen || ! post_type_exists( $post_type_screen ) ) {
			return;
		}

		$prefix = $post_type_screen . '-'; // no prefix

		if ( ! function_exists( 'post_submit_meta_box' ) ) {
			require_once ABSPATH . 'wp-admin/includes/meta-boxes.php';
		}

		self::add(
			$prefix . 'publish',
			esc_html__( 'Publish', 'streamtube-core' ),
			array( $this, 'box_publish' ),
			'dashboard_' . $post_type_screen,
			'side',
			'core'
		);

		// Thumbnail
		if ( post_type_supports( $post_type_screen, 'thumbnail' ) ) {
			self::add(
				$prefix . 'featured-image',
				esc_html__( 'Featured Image', 'streamtube-core' ),
				array( $this, 'box_featured_image' ),
				'dashboard_' . $post_type_screen,
				'side',
				'core'
			);
		}

		if ( is_post_type_hierarchical( $post_type_screen ) ) {
			self::add(
				$prefix . 'attributes',
				esc_html__( 'Attributes', 'streamtube-core' ),
				array( $this, 'box_attributes' ),
				'dashboard_' . $post_type_screen,
				'side',
				'core'
			);
		}

		if ( post_type_supports( $post_type_screen, 'comments' ) ) {
			self::add(
				$prefix . 'discussion',
				esc_html__( 'Discussion', 'streamtube-core' ),
				array( $this, 'box_discussion' ),
				'dashboard_' . $post_type_screen,
				'normal'
			);
		}

		if ( $post_type_screen === 'video' ) {

			if ( current_user_can( 'edit_video_trailer' ) || current_user_can( 'edit_video_source' ) ) {
				self::add(
					$prefix . 'video-sources',
					esc_html__( 'Sources', 'streamtube-core' ),
					array( $this, 'box_video_sources' ),
					'dashboard_' . $post_type_screen,
					'advanced'
				);
			}

			self::add(
				$prefix . 'additional-meta',
				esc_html__( 'Additional meta', 'streamtube-core' ),
				array( $this, 'box_additional_video_meta' ),
				'dashboard_' . $post_type_screen,
				'normal'
			);
		}

		if ( post_type_supports( $post_type_screen, 'excerpt' ) ) {
			self::add(
				$prefix . 'excerpt',
				esc_html__( 'Short description', 'streamtube-core' ),
				array( $this, 'box_excerpt' ),
				'dashboard_' . $post_type_screen,
				'advanced'
			);
		}

		// Taxonomies
		$taxonomies = get_object_taxonomies( $post_type_screen, 'objects' );

		if ( $taxonomies ) {
			foreach ( $taxonomies as $taxonomy => $object ) {

				if ( in_array( $taxonomy, array( 'post_format' ) ) ) {
					continue;
				}

				if (
					$object->show_ui_frontend &&
					get_option( sprintf( 'taxonomy_%s_%s', $post_type_screen, $taxonomy ), 'on' ) &&
					current_user_can( $object->cap->assign_terms ) ) {

					if ( is_taxonomy_hierarchical( $taxonomy ) ) {
						self::add(
							$prefix . 'tax-' . $taxonomy,
							$object->label,
							array( $this, 'box_hierarchical_taxonomy' ),
							'dashboard_' . $post_type_screen,
							'side',
							'default',
							$object
						);
					} else {
						self::add(
							$prefix . 'tax-' . $taxonomy,
							$object->label,
							array( $this, 'box_tag' ),
							'dashboard_' . $post_type_screen,
							'normal',
							'default',
							$object
						);
					}
				}
			}
		}
	}

	/**
	 *
	 * Enqueue scripts
	 * 
	 */
	public function enqueue_scripts() {

		global $wp_query;

		if ( is_author() && isset( $wp_query->query_vars['dashboard'] ) ) {
			wp_enqueue_script( 'wp-lists' );
			wp_enqueue_script( 'clipboard' );
			wp_enqueue_script( 'word-count' );
			wp_enqueue_script( 'wp-postbox' );
			wp_enqueue_script( 'wp-admin-post' );
		}
	}

	/**
	 *
	 * AJAX reset metaboxes
	 * 
	 */
	public function ajax_reset_metaboxes_order() {

		$http_data = wp_parse_args( $_REQUEST, array(
			'data'     => '',
			'_wpnonce' => ''
		) );

		extract( $http_data );

		check_ajax_referer( '_wpnonce' );

		delete_user_meta( get_current_user_id(), 'meta-box-order_' . wp_unslash( trim( $data ) ) );

		wp_send_json_success();
	}
}