<?php
/**
 * Menu
 *
 * @link       https://themeforest.net/user/phpface
 * @since      1.0.0
 *
 * @package    Streamtube_Core
 * @subpackage Streamtube_Core/includes
 */

/**
 * Register all actions and filters for the plugin.
 *
 * Maintain a list of all hooks that are registered throughout
 * the plugin, and register them with the WordPress API. Call the
 * run function to execute the list of actions and filters.
 *
 * @package    Streamtube_Core
 * @subpackage Streamtube_Core/includes
 * @author     phpface <nttoanbrvt@gmail.com>
 */

if( ! defined('ABSPATH' ) ){
    exit;
}

class Streamtube_Core_Menu{

	protected $user_id 			=	0;

	protected $base_url			=	'';

	protected $current 			=	'';

	protected $menu_items 		=	array();

	protected $menu_classes		=	array( 'nav' );

	protected $item_classes		=	array( 'nav-link' );

	protected $icon 			=	false;

	protected $icon_position	=	'left';

	protected $location 		= 	'';

    /**
     * Constructor for Streamtube_Core_Menu class.
     *
     * @param array $args Array of arguments for initializing menu properties.
     */
	public function __construct( $args = array() ){
		$args = wp_parse_args( $args, array(
			'user_id'		=>	0,
			'base_url'		=>	'',
			'current'		=>	'',
			'menu_items'	=>	array(),
			'menu_classes'	=>	array(),
			'item_classes'	=>	array(),
			'icon'			=>	false,
			'icon_position'	=>	'left',
			'location'		=>	''
		) );

		$this->location 			= $args['location'];

		$this->user_id 				= $args['user_id'];

		$this->base_url 			= $args['base_url'];

		$this->current 				= $args['current'];

		$this->menu_items 			= $args['menu_items'];

		$this->icon 				= $args['icon'];

		if( $args['icon_position'] ){
			$this->icon_position = $args['icon_position'];	
		}

		if( is_string( $args['menu_classes'] ) ){
			$args['menu_classes'] 	= explode( " " , $args['menu_classes'] );
		}

		$this->menu_classes 		= array_merge( $args['menu_classes'], $this->menu_classes );

		if( $this->icon === true ){
			$this->menu_classes[] = 'nav-has-icon';

			if( $this->icon_position ){
				$this->menu_classes[] = 'nav-icon-' . $this->icon_position;
			}
		}

		if( is_string( $args['item_classes'] ) ){
			$args['item_classes'] 	= explode( " " , $args['item_classes'] );
		}

		$this->item_classes 		= array_merge( $args['item_classes'], $this->item_classes );
	}

    /**
     *
     * Extract class from dokan menu icon
     * 
     * @param  string $icon
     * @return array
     */
    public function extract_icon_class( $icon = '' ){
    	$classes = array();
        // Extract the class attribute value
        if ( preg_match('/class="([^"]+)"/', $icon, $matches )) {
            $classes = explode(' ', $matches[1] ); // Split the class names into an array
        }

        return $classes ? join( ' ', $classes ) : $icon;
    }	

    /**
     * Sorts menu items by priority.
     *
     * @param array &$items Reference to the menu items array.
     */
	protected function uasort( &$menu_items ){
		uasort( $menu_items, function( $item1, $item2 ){
			return $item1['priority'] <=> $item2['priority'] ?? 0;
		} );
	}

	/**
	 *
	 * Parse Menu.
	 * Fix missing params
	 * 
	 * @param  array &$menu_items
	 * 
	 */
	protected function parse_menu( &$menu_items ){
		foreach( $menu_items as $menu_id => $menu ){

			$menu = wp_parse_args( (array)$menu, array(
				'title'			=>	'',
				'desc'			=>	'',
				'url'			=>	'',
				'callback'		=>	'',
				'icon'			=>	'',
				'parent'		=>	'',
				'cap'			=>	'read',
				'priority'		=>	array_key_exists( 'pos', $menu ) ? absint( $menu['pos'] ) : 10,
				'submenu'		=>	array()
			) );

			$menu_items[ $menu_id ] = $menu;
		}

		return $menu_items;
	}

    /**
     * Constructs URL for menu items.
     *
     * @param string $endpoint Endpoint for URL.
     * @param string $parent Parent for endpoint if applicable.
     * @return string URL for the menu item.
     */
	protected function get_url( $endpoint = '', $parent = '' ){

		$url = $this->base_url;

		if( ! $endpoint ){
			return $url;
		}

		$parsed_parent_slug = explode( '/', $parent );

		if( is_array( $parsed_parent_slug ) && in_array( $endpoint , $parsed_parent_slug ) ){
			$endpoint = '';
		}

		if( ! get_option( 'permalink_structure' ) ){
			if( ! $parent ){
				$url = add_query_arg( array(
					$endpoint 	=>	1
				), $url );
			}
			else{
				$url = add_query_arg( array(
					$parent 	=>	$endpoint
				), $url );
			}
		}
		else{

			$path = $endpoint;

			if( $parent ){
				$path = $parent . '/' . $endpoint;	
			}

			$url = trailingslashit( $url ) . $path;
		}

		return $url;
	}

	/**
	 *
	 * Check if given menu is current
	 * 
	 */
	protected function is_current( $menu_id = '', $parent = '' ){

		if( $this->current === $menu_id ){
			return true;
		}

		global $wp_query;

		$endpoint = '';

		if( $parent === 'dashboard' ){
			$parent = '';
		}

		if( array_key_exists( 'dashboard', $wp_query->query_vars ) ){
			$endpoint = 'dashboard/' . untrailingslashit( $wp_query->query_vars[ 'dashboard' ] );
		}

		if( ! $parent && $menu_id === $endpoint ){
			return true;
		}

		if( $parent && untrailingslashit( $parent ) . '/' . $menu_id ===  $endpoint ){
			return true;
		}

		return false;
	}

    /**
     * Outputs the menu.
     */
	public function the_menu( $menu_items = array(), $echo = true ){

		$menu_li 		= '';
		$menu_classes 	= $this->menu_classes;

		if( $menu_items ){
			$this->menu_items = $menu_items;
			$menu_classes 	= array( 'dropdown-menu', 'animate', 'slideIn' );
		}

		$this->parse_menu( $this->menu_items );

		$this->uasort( $this->menu_items );

		foreach( $this->menu_items as $menu_id => $menu ):

			$menu_li_class 		= array( 'nav-item' );
			$menu_link_class 	= $this->item_classes;

			$menu = wp_parse_args( $menu, array(
				'title'			=>	'',
				'url'			=>	'',
				'desc'			=>	'',
				'badge'			=>	'',
				'parent'		=>	'',
				'cap'			=>	'read',
				'private'		=>	false,
				'id'			=>	'',
				'icon'			=>	'',
				'icon_color'	=>	'',
				'priority'		=>	1,
				'submenu'		=>	array()
			) );

			if( ! $menu['id'] ){
				$menu['id'] = 'nav-' . $menu_id;
			}

			$menu['icon'] = $this->extract_icon_class( $menu['icon'] );

			if( ! $menu['title'] ){
				continue;
			}

			$tooltip = $menu['desc'] ? $menu['desc'] : $menu['title'];

			if( is_callable( $menu['badge'] ) ){
				$menu['badge'] = call_user_func( $menu['badge'], $this->user_id );
			}

			if( ( 
					( is_string( $menu['cap'] ) && user_can( $this->user_id, $menu['cap'] ) ) ||
					( is_callable( $menu['cap'] ) && call_user_func( $menu['cap'], $this->user_id ) === true )
				) 
				&& ! is_object( $menu['url'] ) ):

				$is_visible = $menu['private'] === false || $this->location != 'main' ? true : false;

				if( apply_filters( 'streamtube/core/menu/is_visible', $is_visible , $menu, $this->menu_items ) === true ){

					$menu_li_class[] 		= $this->icon && $menu['icon'] 	? 'nav-item-icon' 									: '';
					$menu_li_class[] 		= 'nav-' . sanitize_html_class( $menu['id'] );
					$menu_li_class[] 		= $this->is_current( $menu_id, $menu['parent'] )	? 'current selected' 			: '';
					$menu_link_class[] 		= $this->is_current( $menu_id, $menu['parent'] )	? 'active current-menu-item' 	: '';

					if( $menu['submenu'] ){
						$menu_li_class[] 	= 'dropdown';
						$menu_li_class[] 	= 'drop-end';
						$menu_li_class[] 	= 'menu-item-has-children';
						$menu_link_class[] 	= 'dropdown-toggle';
					}

					$menu_li .= sprintf(
						'<li class="%1$s" id="%2$s" data-priority="%3$s">',
						trim( esc_attr( implode( ' ', array_unique( array_map('trim', $menu_li_class))) ) ),
						esc_attr( $menu['id'] ),
						(int)$menu['priority']
					);

						$menu_li .= sprintf(
							'<a class="%s" aria-current="page" href="%s" %s>',
							esc_attr( join( ' ', $menu_link_class ) ),
							$menu['submenu'] ? '#' : ( $menu['url'] ? esc_url( $menu['url'] ) : esc_url( $this->get_url( $menu_id, $menu['parent'] ) ) ),
							$menu['submenu'] ? ( ! $menu_items ? ' data-bs-toggle="dropdown" data-bs-display="static"' : ' data-bs-display="static"' ) : ''
						);

						if( $this->icon === true && array_key_exists( 'icon' , $menu ) ){

							$icon_class = array( 'menu-icon' );

							$icon_class[] = $menu['icon'];

							if( is_string( $this->icon_position ) && $this->icon_position ){
								$icon_class[] = 'icon-position-' . $this->icon_position;
							}

							if( $this->icon_position === 'left' ){
								$icon_class[] = 'me-3';
							}
							if( $this->icon_position === 'right' ){
								$icon_class[] = 'ms-3';
							}
							if( $this->icon_position === 'top' ){
								$icon_class[] = "mb-2 mx-auto h6";
							}

							$menu_li .= sprintf(
								'<span data-bs-toggle="tooltip" data-bs-placement="%s" data-bs-title="%s" class="%s" style="%s"></span>',
								is_string( $this->icon_position ) ? $this->icon_position : 'right',
								esc_attr( wp_strip_all_tags( $tooltip, true ) ),
								esc_attr( join( ' ', $icon_class ) ),
								$menu['icon_color'] ? 'color:' . esc_attr( $menu['icon_color'] ) : ''
							);
						}

						$menu_li .= sprintf(
							' <span class="menu-text">%s %s</span>',
							$menu['title'],
							$menu['badge']
						);

						$menu_li .= '</a>';

						if( is_array( $menu['submenu'] ) && $menu['submenu'] ){
							$menu_li .= $this->the_menu( $menu['submenu'], false );
						}

					$menu_li .= '</li>';
				}

			endif;

		endforeach;

		$output = sprintf(
			'<ul class="%s">%s</ul>',
			esc_attr( join( ' ', $menu_classes ) ),
			$menu_li
		);

		if( $echo ){
			echo $output;
		}else{
			return $output;
		}

	}
}