<?php

/**
 * The admin-specific functionality of the plugin.
 *
 * @link       https://themeforest.net/user/phpface
 * @since      1.0.0
 *
 * @package    Streamtube_Core
 * @subpackage Streamtube_Core/admin
 */

/**
 * The admin-specific functionality of the plugin.
 *
 * Defines the plugin name, version, and two examples hooks for how to
 * enqueue the admin-specific stylesheet and JavaScript.
 *
 * @package    Streamtube_Core
 * @subpackage Streamtube_Core/admin
 * @author     phpface <nttoanbrvt@gmail.com>
 */

if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

class Streamtube_Core_Admin {

	/**
	 * The ID of this plugin.
	 *
	 * @since    1.0.0
	 * @access   private
	 * @var      string    $plugin_name    The ID of this plugin.
	 */
	private $plugin_name;

	/**
	 * The version of this plugin.
	 *
	 * @since    1.0.0
	 * @access   private
	 * @var      string    $version    The current version of this plugin.
	 */
	private $version;

	/**
	 * Initialize the class and set its properties.
	 *
	 * @since    1.0.0
	 * @param      string    $plugin_name       The name of this plugin.
	 * @param      string    $version    The version of this plugin.
	 */
	public function __construct( $plugin_name, $version ) {

		$this->plugin_name = $plugin_name;
		$this->version     = $version;

	}

	/**
	 * Register the stylesheets for the admin area.
	 *
	 * @since    1.0.0
	 */
	public function enqueue_styles() {

		wp_register_style(
			'select2',
			plugin_dir_url( dirname( __FILE__ ) ) . 'public/assets/vendor/select2/select2.min.css',
			array(),
			filemtime( plugin_dir_path( dirname( __FILE__ ) ) . 'public/assets/vendor/select2/select2.min.css' )
		);

		wp_enqueue_style(
			'streamtube-core-admin',
			plugin_dir_url( __FILE__ ) . 'assets/css/streamtube-core-admin.css',
			array(),
			filemtime( plugin_dir_path( __FILE__ ) . 'assets/css/streamtube-core-admin.css' ),
			'all'
		);

	}

	/**
	 * Register the JavaScript for the admin area.
	 *
	 * @since    1.0.0
	 */
	public function enqueue_scripts() {

		wp_register_script(
			'select2',
			plugin_dir_url( dirname( __FILE__ ) ) . 'public/assets/vendor/select2/select2.min.js',
			array( 'jquery' ),
			filemtime( plugin_dir_path( dirname( __FILE__ ) ) . 'public/assets/vendor/select2/select2.min.js' ),
			true
		);

		wp_enqueue_script(
			'streamtube-core-admin',
			plugin_dir_url( __FILE__ ) . 'assets/js/streamtube-core-admin.js',
			array( 'jquery' ),
			filemtime( plugin_dir_path( __FILE__ ) . 'assets/js/streamtube-core-admin.js' ),
			true
		);

		wp_enqueue_script(
			'streamtube-core-user',
			plugin_dir_url( __FILE__ ) . 'assets/js/user.js',
			array( 'jquery' ),
			filemtime( plugin_dir_path( __FILE__ ) . 'assets/js/user.js' ),
			true
		);

		wp_localize_script( 'streamtube-core-admin', 'streamtube', array(
			'is_admin'              => is_admin(),
			'home_url'              => home_url( '/' ),
			'admin_url'             => esc_url_raw( admin_url( '/' ) ),
			'ajax_url'              => esc_url_raw( admin_url( 'admin-ajax.php' ) ),
			'rest_url'              => esc_url_raw( rest_url( 'streamtube/v1' ) ),
			'nonce'                 => wp_create_nonce( 'wp_rest' ),
			'_wpnonce'              => wp_create_nonce( '_wpnonce' ),
			'generate'              => esc_html__( 'Generate', 'streamtube-core' ),
			'mediaid_not_found'     => esc_html__( 'Media Id was not found', 'streamtube-core' ),
			'remove_featured_image' => esc_html__( 'Remove featured image', 'streamtube-core' ),
			'confirm_remove_ad'     => esc_html__( 'Do you want to remove this Ad Tag?', 'streamtube-core' ),
			'confirm_import_yt'     => esc_html__( 'Do you want to import all checked items?', 'streamtube-core' ),
			'number_posts_imported' => esc_html__( '%s posts have been imported successfully.', 'streamtube-core' ),
			'cannot_generate_image' => esc_html__( 'Cannot generate animated image from an embed/URL.', 'streamtube-core' )
		) );
	}

	/**
	 *
	 * Get admin notice id
	 * 
	 * @param  string $key
	 * @return string
	 *
	 * @since 2.1
	 * 
	 */
	private function _get_notice_id( $key ) {
		$current_user_id = get_current_user_id();
		return "_notice_{$key}_{$current_user_id}_{$this->version}";
	}

	/**
	 *
	 * Check if notice was dismissed
	 * 
	 * @param  string  $key
	 * @return boolean     
	 *
	 * @since 2.1
	 * 
	 */
	private function _is_notice_dismissed( $key ) {
		return get_option( $this->_get_notice_id( $key ) );
	}

	/**
	 *
	 * Check if notice was dismissed
	 * 
	 * @param  string  $key
	 * @return boolean     
	 *
	 * @since 2.1
	 * 
	 */
	private function _dismiss_notice( $key ) {
		return update_option( $this->_get_notice_id( $key ), 'on' );
	}

	/**
	 * Determines whether the database should be upgraded.
	 *
	 * This method uses a filter to allow future modifications or custom logic
	 * for triggering database upgrades.
	 *
	 * @return mixed The result of the filter 'streamtube/core/should_upgrade_database'.
	 */
	private function should_upgrade_database() {
		return apply_filters( 'streamtube/core/should_upgrade_database', false );
	}

	/**
	 *
	 * Fires action to upgrade database
	 * 
	 */
	private function upgrade_database() {
		do_action( 'streamtube/core/do_upgrade_database' );
	}

	/**
	 *
	 * AJAX do upgrade dababase
	 * 
	 */
	public function ajax_upgrade_database() {

		check_ajax_referer( 'streamtube/core/upgrade_database', 'nonce' );

		if ( ! current_user_can( 'administrator' ) ) {
			wp_send_json_error(
				new WP_Error(
					'no_permission',
					esc_html__( 'You do not have permission to do this action.', 'streamtube-core' )
				)
			);
		}

		$this->upgrade_database();

		do_action( 'streamtube/core/did_upgrade_database' );

		wp_send_json_success( esc_html__( 'Done!', 'streamtube-core' ) );
	}

	/**
	 *
	 * Auto upgrade once administrator open WP admin
	 * 
	 */
	public function auto_upgrade() {
		if (
			! wp_doing_ajax() &&
			! did_action( 'streamtube/core/do_upgrade_database' ) &&
			apply_filters( 'streamtube/core/should_upgrade_database', false ) ) {
			if ( current_user_can( 'administrator' ) ) {
				$this->upgrade_database();
				do_action( 'streamtube/core/did_upgrade_database' );
			}
		}
	}

	/**
	 *
	 * Add custom capabilities to admin role
	 * 
	 */
	public function notice_upgrade_database() {

		if ( ! $this->should_upgrade_database() || ! current_user_can( 'administrator' ) ) {
			return;
		}

		?>
		<div class="notice notice-info">
			<p>
				<strong><?php esc_html_e( 'Important! Upgrade your StreamTube database to ensure compatibility with the new version.', 'streamtube-core' ); ?></strong>
			</p>

			<form method="post" name="streamtube-upgrade-form">

				<input type="hidden" name="action" value="streamtube/core/upgrade_database">

				<?php printf(
					'<input type="hidden" name="nonce" value="%s">',
					wp_create_nonce( 'streamtube/core/upgrade_database' )
				); ?>

				<p>
					<button type="submit" class="button button-primary button-large">
						<?php esc_html_e( 'Upgrade Now', 'streamtube-core' ); ?>
					</button>

					<span class="spinner" style="float: none"></span>
				</p>

			</form>
		</div>
		<?php
	}
}
