<?php
/**
 * The table list class
 * 
 * @link       https://1.envato.market/mgXE4y
 * @since      2.1
 *
 * @package    WordPress
 * @subpackage StreamTube
 * @author     phpface <nttoanbrvt@gmail.com>
 */

if ( ! defined( 'ABSPATH' ) ) {
    exit;
}

if ( ! class_exists( 'WP_List_Table' ) ) {
    require_once ABSPATH . 'wp-admin/includes/class-wp-list-table.php';
}

class WP_Cloudflare_Stream_Videos_List_Table extends WP_List_Table {

    private $cloudflare;

    protected $screen_id;

    public function __construct( $args = array() ) {

        $args = wp_parse_args( $args, array(
            'screen'    =>  'edit-cloudflare-stream',
            'singular'  => esc_html__( 'Video', 'wp-cloudflare-stream' ),
            'plural'    => esc_html__( 'Videos', 'wp-cloudflare-stream' ),
            'ajax'      => true            
        ) );

        parent::__construct( $args );

        $this->cloudflare = $GLOBALS['wp_cloudflare_stream']->get();

        $this->screen_id = $args['screen'];
    }

    /**
     *
     * Video statuses
     * 
     * @return array
     * 
     */
    public function get_statuses(){
        return array(
            'ready'             =>  esc_html__( 'Ready', 'wp-cloudflare-stream' ),
            'pendingupload'     =>  esc_html__( 'Pending Upload', 'wp-cloudflare-stream' ),
            'downloading'       =>  esc_html__( 'Downloading', 'wp-cloudflare-stream' ),
            'queued'            =>  esc_html__( 'Queued', 'wp-cloudflare-stream' ),
            'inprogress'        =>  esc_html__( 'Inprogress', 'wp-cloudflare-stream' ),
            'error'             =>  esc_html__( 'Error', 'wp-cloudflare-stream' )
        );
    }

    /**
     *
     * Video types
     * 
     * @return array
     */
    public function get_types(){
        return array(
            'vod'       =>  esc_html__( 'Vod', 'wp-cloudflare-stream' ),
            'live'      =>  esc_html__( 'Live', 'wp-cloudflare-stream' )
        );
    }

    public function get_state_badge( $state ){
        $class = 'secondary';

        switch ( $state ) {
            case 'ready':
                $class = 'success';
            break;

            case 'queued':
                $class = 'black';
            break;

            case 'inprogress':
                $class = 'info';
            break;

            case 'downloading':
                $class = 'primary';
            break;

            case 'pendingupload':
                $class = 'warning';
            break;

            case 'pendingupload':
                $class = 'danger';
            break;                                                                                                      
            
            default:
                $class = 'secondary';
            break;
        }  
        return $class;      
    }    

    public function get_columns() {
        $columns = array(
            'cb'                => '<input type="checkbox" />',
            'title'             => esc_html__( 'Title', 'wp-cloudflare-stream' ),
            'thumbnail'         => esc_html__( 'Thumbnail', 'wp-cloudflare-stream' ),
            'creator'           => esc_html__( 'Creator', 'wp-cloudflare-stream' ),
            'status'            => esc_html__( 'Status', 'wp-cloudflare-stream' ),
            'size'              => esc_html__( 'Size', 'wp-cloudflare-stream' ),
            'duration'          => esc_html__( 'Duration', 'wp-cloudflare-stream' ),
            'scheduledDeletion' => esc_html__( 'Scheduled Deletion', 'wp-cloudflare-stream' ),
            'uid'               => esc_html__( 'UID', 'wp-cloudflare-stream' ),
            'allowedOrigins'    => esc_html__( 'Allowed Origins', 'wp-cloudflare-stream' ),
            'requireSignedURLs' => esc_html__( 'Signed URL', 'wp-cloudflare-stream' ),
            'uploaded'          => esc_html__( 'Date', 'wp-cloudflare-stream' ),
            'action'            => esc_html__( 'Action', 'wp-cloudflare-stream' )
        );

        if( ! WP_Cloudflare_Stream_Permission::list_others_videos() ){
            unset( $columns['uid'] );
            unset( $columns['creator'] );
            unset( $columns['allowedOrigins'] );
            unset( $columns['requireSignedURLs'] );
        }

        return apply_filters( 'manage_cloudflare_stream_columns', $columns );
    }

    /**
     * Define what data to show on each column of the table
     *
     * @param  Array $item        Data
     * @param  String $column_name - Current column name
     *
     * @return Mixed
     */
    public function column_default( $item, $column_name ){

        switch( $column_name ) {

            case 'title':

                $title = isset( $item['meta']['name'] ) ? $item['meta']['name'] : '';

                return sprintf(
                    '<p><a class="row-title" target="_blank" href="%s">%s</a></p>',
                    esc_url( $item[ 'preview' ] ),
                    esc_html( $title )
                );                
            break; 

            case 'thumbnail':
                $output = sprintf(
                    '<div class="post-thumbnail ratio ratio-16x9 rounded overflow-hidden bg-dark"><img class="img-thumbnail" src="%s"></div>',
                    esc_url( $item[ $column_name ] )
                );                
            break;

            case 'size':
                return size_format( $item[$column_name], 2 );
            break;             

            case 'uploaded':
                return sprintf(
                    esc_html__( '%s ago', 'wp-cloudflare-stream' ),
                    human_time_diff( strtotime( $item[ $column_name ] ), current_time( 'U' ) )
                );
            break;

            case 'status':

                $status = $item[ $column_name ]['state'];

                if( is_string( $status ) && array_key_exists( $status, $this->get_statuses() ) ){
                    $status = $this->get_statuses()[ $status ];
                }

                return sprintf(
                    '<span class="badge bg-%1$s">%2$s</span>',
                    esc_attr( $this->get_state_badge( $item[ $column_name ]['state'] ) ),
                    is_string( $status ) ? $status : esc_html__( 'Unknown', 'wp-cloudflare-stream' )
                );
            break;

            case 'creator':

                $maybe_user = false;
                $creator    = $this->cloudflare->post->get_creator_id( $item );

                if( is_integer( $creator ) ){
                    $maybe_user = get_userdata( $creator  );
                }

                return sprintf(
                    '<a class="text-%s" href="%s">%s</a>',
                    $maybe_user ? 'body' : 'danger',
                    $maybe_user ? esc_url( get_author_posts_url( $maybe_user->ID )) : '#',
                    $maybe_user ? $maybe_user->display_name : esc_html__( 'Unknown', 'wp-cloudflare-stream' )
                );

            break;

            case 'scheduledDeletion':

                if( $item[ $column_name ] ){
                    return sprintf(
                        '<span class="text-danger">%s</span>',
                        $item[ $column_name ]
                    );
                }

            break;

            case 'allowedOrigins':

                if( is_array( $item[ $column_name ] ) ){
                    return implode(', ', $item[ $column_name ] );
                }

            break;    

            case 'requireSignedURLs':

                $is_enabled = wp_validate_boolean( $item[ $column_name ] );

                return sprintf(
                    '<span class="badge bg-%s">%s</span>',
                    $is_enabled ? 'success' : 'secondary',
                    $is_enabled ? esc_html__( 'On', 'wp-cloudflare-stream' ) : esc_html__( 'Off', 'wp-cloudflare-stream' )
                );

            break;

            case 'action':

                $output = '';

                $attachment_id = $this->cloudflare->post->get_attachment_id_from_uid( $item['uid'] );

                if( $attachment_id ){

                    $parent = get_post_parent( $attachment_id );

                    $output .= sprintf(
                        '<span class="badge bg-success"><a class="text-white" target="_blank" href="%s">%s</a></span>',
                        $parent ? esc_url( get_permalink( $parent ) ) : '#',
                        esc_html__( 'Imported', 'wp-cloudflare-stream' )
                    );
                }else{
                    if( WP_Cloudflare_Stream_Permission::import_videos() || ( $item['creator'] == get_current_user_id() && WP_Cloudflare_Stream_Permission::import_others_videos() ) ){
                        $output = sprintf(
                            '<button class="btn btn-sm button button-small button-primary ajax-elm" data-params="%s" type="button" data-action="%s" data-endpoint="%s">%s</button>',
                            esc_attr( json_encode( array( 'uid' => $item['uid'] ) ) ),
                            'import_cloudflare_stream_video',
                            esc_url_raw( $this->cloudflare->rest_api->get_rest_url( '/import' ) ),
                            esc_html__( 'Import', 'wp-cloudflare-stream' )
                        );
                    }                    
                }

                if( WP_Cloudflare_Stream_Permission::delete_videos() || ( $item['creator'] == get_current_user_id() && WP_Cloudflare_Stream_Permission::delete_others_videos() ) ){
                    $output .= sprintf(
                        '<button 
                            type="button" 
                            class="%s" 
                            data-endpoint="%s"
                            data-action="%s" 
                            data-params="%s"
                            data-confirm="%s"
                        >%s</button>',
                        esc_attr( 'btn btn-sm bg-danger button button-small text-white delete-cloudflare-video ajax-elm' ),
                        esc_url_raw( $this->cloudflare->rest_api->get_rest_url( '/delete' ) ),
                        esc_attr( 'delete_cloudflare_video' ),
                        esc_attr( json_encode( array( 'uid' => $item['uid'] ) ) ),
                        sprintf(
                            esc_attr__( 'Are you sure you want to permanently delete the %s video? This action will also delete the associated video post.', 'wp-cloudflare-stream' ),
                            isset( $item['meta']['name'] ) ? $item['meta']['name'] : ''
                        ),
                        esc_html__( 'Delete', 'wp-cloudflare-stream' )
                    );
                }

                if( $output ){
                    return sprintf( '<div class="d-flex gap-3 align-items-start">%s</div>', $output );
                }

            break;        

            default:
                return $item[ $column_name ];
            break;
        }
    }

    /**
     * Define which columns are hidden
     *
     * @return array
     */
    public function get_hidden_columns() {
        $hidden_columns = (array)get_user_option( 'manage' . $this->screen->id . 'columnshidden' );

        return is_array( $hidden_columns ) ? $hidden_columns : array();
    }

    protected function get_items_per_page( $option = '', $default_value = 20 ) {
        $per_page = (int)get_user_option( 'edit_'. $this->screen->id .'_per_page' );

        return $per_page !== 0 ? absint( $per_page ) : $default_value;
    }
    
    /**
     * Define the sortable columns
     *
     * @return array
     */
    public function get_sortable_columns() {
        return array(
            'creator'       => array( 'creator', false ),
            'size'          => array( 'size', false ),
            'duration'      => array( 'duration', false ),
            'uploaded'      => array( 'uploaded', false )
        );
    }

    protected function get_views() {

        $url    = $_SERVER['REQUEST_URI'];

        $views = array();

        $views['all']  = array(
            'url'     => remove_query_arg( array_keys( $_REQUEST ), $url ),
            'label'   => esc_html__( 'All', 'wp-cloudflare-stream' ),
            'current' => ! isset( $_REQUEST['status'] ) && ! isset( $_REQUEST['type'] )
        );

        foreach ( $this->get_statuses() as $key => $value) {
            $views[$key] = array(
                'url'     =>  esc_url( add_query_arg( array(
                    'status'    =>  $key
                ), remove_query_arg( array_keys( $_REQUEST ), $url ) ) ),
                'label'   => ucwords( $value ),
                'current' => isset( $_REQUEST['status'] ) && $key === $_REQUEST['status']
            );
        }

        if( WP_Cloudflare_Stream_Permission::list_others_videos() ){
            foreach ( $this->get_types() as $key => $value) {
                $views[$key] = array(
                    'url'     =>  esc_url( add_query_arg( array(
                        'type'    =>  $key
                    ),  remove_query_arg( array_keys( $_REQUEST ), $url ) ) ),
                    'label'   => ucwords( $value ),
                    'current' => isset( $_REQUEST['type'] ) && $key === $_REQUEST['type']
                );
            }   
        }    

        return $this->get_views_links( $views );
    }

    public function get_bulk_actions(){

        $action = array();

        $action = array(
            'import'    =>  esc_html__( 'Import', 'wp-cloudflare-stream' )
        );
        
        return apply_filters( 'bulk_actions-' . $this->screen_id, $action ); 
    }    

    /**
     * Render the bulk edit checkbox
     *
     * @param array $item
     *
     * @return string
     */
    function column_cb( $item ) {
        return sprintf(
            '<input type="checkbox" name="uid[]" value="%s" />', $item['uid']
        );
    }    

    /**
     * Get the table data
     *
     * @return array
     */
    private function table_data() {

        $args = array();

        $args = wp_parse_args( $args, array(
            'search'    =>  isset( $_REQUEST['s'] ) ? wp_unslash( $_REQUEST['s'] ) : '',
            'limit'     =>  -1
        ) );

        if( isset( $_REQUEST['status'] ) && array_key_exists( $_REQUEST['status'], $this->get_statuses() ) ){
            $args['status'] = $_REQUEST['status'];
        }

        if( isset( $_REQUEST['type'] ) && array_key_exists( $_REQUEST['type'], $this->get_types() ) ){
            $args['type'] = $_REQUEST['type'];
        }              

        $response = $this->cloudflare->post->get_cloudflare_videos( $args );

        if( is_wp_error( $response ) ){
            return array();
        }

        return $response['result'] ? $response['result']['videos'] : array();
    }

    /**
     * Sort data based on the selected column and order
     *
     * @param array $a
     * @param array $b
     * @return int
     */
    private function sort_data( $a, $b ) {
        $orderby = isset( $_GET['orderby'] ) && is_string( $_GET['orderby'] ) ? sanitize_text_field( $_GET['orderby'] ) : ''; 
        $order   = isset( $_GET['order'] ) && is_string( $_GET['order'] ) ? sanitize_text_field( $_GET['order'] ) : 'asc';

        if ( ! isset( $a[$orderby] ) || ! isset( $b[$orderby] ) || ! is_string( $a[$orderby] ) || ! is_string( $b[$orderby] ) ) {
            return 0; // Handle cases where the sort column doesn't exist in the data
        }

        $result = strcmp( $a[$orderby], $b[$orderby] ); 

        return ( $order === 'asc' ) ? $result : -$result;
    }

    public function prepare_items() {
        $columns  = $this->get_columns();
        $hidden   = $this->get_hidden_columns();
        $sortable = $this->get_sortable_columns();

        $data = $this->table_data();
        usort( $data, array( $this, 'sort_data' ) );

        $perPage     = $this->get_items_per_page();
        $currentPage = $this->get_pagenum();
        $totalItems  = count( $data );

        $this->set_pagination_args( array(
            'total_items' => $totalItems,
            'per_page'    => $perPage,
        ) );

        $data = array_slice( $data, ( ( $currentPage - 1 ) * $perPage ), $perPage );

        $this->_column_headers = array( $columns, $hidden, $sortable );
        $this->items = $data;

        $this->process_bulk_action();
    }

    public function process_bulk_action() {
        if( ! WP_Cloudflare_Stream_Permission::import_videos() ){
            return;
        }

        $action = $this->current_action();

        $uids = isset( $_REQUEST['uid'] ) ? wp_unslash( $_REQUEST['uid'] ) : false;

        switch ( $action ) {
            case 'import':
                for ( $i=0;  $i < count( $uids );  $i++) { 

                    $attachment_id = $this->cloudflare->post->get_attachment_id_from_uid( $uids[$i] );

                    if( ! $attachment_id ){
                       $this->cloudflare->post->_import_video( array(
                            'uid'           =>  $uids[$i],
                            'post_status'   =>  'pending',
                            'post_author'   =>  get_current_user_id()
                       ));
                    }
                }
            break;
        }                
    }
}