<?php
/**
 *
 * @link              https://1.envato.market/mgXE4y
 * @since             1.0.0
 * @package           WP_User_Channel
 */

// If this file is called directly, abort.
if ( ! defined( 'WPINC' ) ) {
	die;
}

class WP_User_Channel {

	public $settings;

	function __construct() {
		$this->load_dependencies();
		$this->load_hooks();
	}

	private function load_dependencies() {
		require_once( WP_USER_CHANNEL_PATH . 'admin/wp-user-channel-admin.php' );

		$this->settings = WP_User_Channel_Admin::get_settings();
	}

	/**
	 * Load necessarily hooks
	 */
	private function load_hooks() {

		add_action(
			'init',
			array( $this, 'load_plugin_textdomain' )
		);

		add_action(
			'wp_ajax_wp_user_channel/create_channel',
			array( $this, 'ajax_create_channel' )
		);

		add_action(
			'wp_ajax_wp_user_channel/get_channel_list',
			array( $this, 'ajax_get_channel_list' )
		);

		add_action(
			'wp_ajax_wp_user_channel/switch_to',
			array( $this, 'ajax_do_switch_to' )
		);

		add_filter(
			'comment_notification_recipients',
			array( $this, 'send_comment_notification' ),
			10,
			2
		);

		add_action(
			'user_profile_update_errors',
			array( $this, 'omit_update_email_address_error' ),
			10,
			3
		);

		add_action(
			'wp_user_channel/channel_meta/after',
			array( $this, 'display_channel_last_activity' )
		);

		add_action(
			'wp_user_channel/channel_avatar/after',
			array( $this, 'display_channel_unread_notification_count' )
		);

		add_filter(
			'streamtube/core/user/profile/menu/items',
			array( $this, 'display_dropdown_menu' ),
			10,
			1
		);

		add_filter(
			'streamtube/core/user/dashboard/menu/items',
			array( $this, 'display_dashboard_menu' ),
			10,
			1
		);

		add_action(
			'wp_footer',
			array( $this, 'load_modal_accounts' )
		);

		add_action(
			'wp_enqueue_scripts',
			array( $this, 'enqueue_scripts' )
		);

		add_action(
			'streamtube/core/register_modules',
			array( $this, 'register_permission_module' )
		);

		add_action(
			'wp_ajax_wp_user_channel/assign_channel_cap',
			array( 'WP_User_Channel_Admin', 'ajax_assign_channel_cap' )
		);

		add_action(
			'wp_ajax_wp_user_channel/assign_role_cap',
			array( 'WP_User_Channel_Admin', 'ajax_assign_role_cap' )
		);

		add_action(
			'admin_init',
			array( 'WP_User_Channel_Admin', 'save_settings' )
		);

		add_action(
			'admin_menu',
			array( 'WP_User_Channel_Admin', 'setting_page' )
		);

		add_filter(
			'manage_users_columns',
			array( 'WP_User_Channel_Admin', 'table_user_column' )
		);

		add_filter(
			'manage_users_custom_column',
			array( 'WP_User_Channel_Admin', 'table_user_column_content' ),
			10,
			3
		);

		add_action(
			'admin_enqueue_scripts',
			array( 'WP_User_Channel_Admin', 'enqueue_scripts' )
		);

		add_filter(
			'plugin_action_links_' . WP_USER_CHANNEL_BASE,
			array( 'WP_User_Channel_Admin', 'action_links' )
		);
	}

	/**
	 *
	 * Plugin activation
	 * 
	 */
	public static function activate() {
		self::add_role();
	}

	/**
	 *
	 * Add "Channel" role
	 * 
	 */
	private static function add_role() {
		add_role( 'wpuc_channel', esc_html__( 'Channel', 'wp-user-channel' ), array(
			'read' => true
		) );
	}

	/**
	 *
	 * Check if current user can switch to given user
	 * 
	 * @param  integer $user_id
	 * @return boolean
	 * 
	 */
	public function can_switch_to( $user_id = 0 ) {

		$user_id = (int) $user_id;

		if ( ! is_user_logged_in() || ! $user_id ) {
			return false;
		}

		$current_user = get_current_user_id();

		// Check if the currently logged-in user is the same as the given user ID
		// If so, there's no need to switch to themselves
		if ( $user_id === $current_user ) {
			return false;
		}

		// Maybe logged in is a sub account and try to switch to main account
		if ( $user_id === (int) get_user_meta( $current_user, 'wpuc_user_id', true ) ) {
			return true;
		}

		// Retrieve given user wpuc_user_id value
		$maybe_main_user = (int) get_user_meta( $user_id, 'wpuc_user_id', true );

		// Maybe logged in is the main account
		if ( $maybe_main_user ) {
			if ( $maybe_main_user === $current_user || $maybe_main_user === (int) get_user_meta( $current_user, 'wpuc_user_id', true ) ) {
				return true;
			}
		}

		return false;
	}

	/**
	 *
	 * Do switch to action
	 * 
	 * @param  int $user_id
	 */
	private function do_switch_to( $user_id ) {

		if ( self::can_switch_to( $user_id ) ) {
			wp_set_auth_cookie( $user_id );

			/**
			 * Fires after switching account
			 */
			do_action( 'wp_user_channel/switched_to', $user_id );

			return true;
		}

		return false;
	}

	/**
	 *
	 * AJAX switch to requested user
	 * 
	 */
	public function ajax_do_switch_to() {

		$user_id = isset( $_REQUEST['user_id'] ) ? absint( $_REQUEST['user_id'] ) : 0;
		$nonce   = isset( $_REQUEST['_wpnonce'] ) ? wp_unslash( $_REQUEST['_wpnonce'] ) : '';

		if ( ! wp_verify_nonce( $nonce, 'wpuc_switch_to_user_' . $user_id ) ) {
			wp_send_json_error( new WP_Error(
				'invalid_request',
				esc_html__( 'Invalid Request', 'wp-user-channel' )
			) );
		}

		$results = $this->do_switch_to( $user_id );

		if ( $results === true ) {
			wp_send_json_success( 'OK' );
		}

		wp_send_json_error( new WP_Error(
			'cannot_switch',
			esc_html__( 'Cannot switch to this user', 'wp-user-channel' )
		) );
	}

	/**
	 *
	 * Send a comment notification to the main channel if a subchannel receives one
	 * 
	 * @param  array $emails
	 * @param  int $comment_id
	 * @return array
	 */
	public function send_comment_notification( $emails, $comment_id ) {
		$comment = get_comment( $comment_id );

		$post = get_post( $comment->comment_post_ID );

		if ( $maybe_main_account = get_user_meta( $post->post_author, 'wpuc_user_id', true ) ) {
			$main_account = get_userdata( $post->post_author );

			if ( $main_account && $main_account->user_email ) {
				$emails[] = $main_account->user_email;
			}
		}

		return $emails;
	}

	/**
	 *
	 * Create Channel
	 * 
	 * @return WP_User|WP_Error
	 */
	public function create_channel( $channel_name = '' ) {

		$maybe_main_account = 0;
		$default_roles      = array( 'wpuc_channel' );
		$errors             = new WP_Error();

		if ( ! $channel_name ) {
			$errors->add(
				'empty_name',
				esc_html__( 'Channel Name is required.', 'wp-user-channel' )
			);
		}

		$user = wp_get_current_user();

		$maybe_main_account = (int) get_user_meta( $user->ID, 'wpuc_user_id', true );

		if ( ! $maybe_main_account || ! get_userdata( $maybe_main_account ) ) {
			$maybe_main_account = $user->ID;
		}

		if ( ! $user || ! user_can( $user, 'add_channels' ) ) {
			$errors->add(
				'no_permission',
				esc_html__( 'Sorry, You don\'t have permission to create channels.', 'wp-user-channel' )
			);
		}

		if ( ! user_can( $maybe_main_account, 'add_users' ) ) {
			$max_channels = absint( $this->settings['max_channels'] );
			if ( $max_channels != 0 && count( $this->get_channels() ) > $max_channels - 1 ) {
				$errors->add(
					'max_channels_exceeded',
					sprintf(
						esc_html__( 'Sorry, you have exceeded the maximum allowed channels (%s)', 'wp-user-channel' ),
						$max_channels
					)
				);
			}
		}

		/**
		 * Filter errors
		 */
		$errors = apply_filters( 'wp_user_channel/create_channel/errors', $errors, $channel_name );

		if ( $errors->get_error_codes() ) {
			return $errors;
		}

		$channel_id = wp_create_user( $channel_name, wp_generate_password() );


		if ( is_wp_error( $channel_id ) ) {
			return $channel_id;
		}

		$channel = get_userdata( $channel_id );

		// Remove all default roles
		$channel->set_role( '' );

		if ( (array) $this->settings['default_roles'] ) {
			$default_roles = array_merge( $default_roles, (array) $this->settings['default_roles'] );
		} else {
			$default_roles = array_merge( $default_roles, $user->roles );
		}

		$default_roles = array_unique( $default_roles );

		if ( $default_roles ) {
			for ( $i = 0; $i < count( $default_roles ); $i++ ) {
				$channel->add_role( $default_roles[ $i ] );
			}
		}

		update_user_meta( $channel_id, 'wpuc_user_id', $maybe_main_account );

		/**
		 *
		 * Fires after Channel created
		 *
		 * @param WP_User $channel
		 * @param WP_User $user
		 * 
		 */
		do_action( 'wp_user_channel/created_channel', $channel, $user, $maybe_main_account );

		return $channel;
	}

	/**
	 *
	 * Get channels
	 * 
	 * @param  array  $args
	 * @return get_users()
	 * 
	 */
	public function get_channels( $args = array() ) {

		$user_data = array();

		$args = wp_parse_args( $args, array(
			'user_id' => get_current_user_id(),
			'orderby' => 'registered',
			'order'   => 'DESC'
		) );

		extract( $args );

		$user_args = compact( 'orderby', 'order' );

		// Check if current logged in user is a channel
		$maybe_main_user = (int) get_user_meta( $user_id, 'wpuc_user_id', true );

		if ( $maybe_main_user ) {
			$user_id = $maybe_main_user;
		}

		if ( $user_data = get_userdata( $user_id ) ) {
			$user_args = array_merge( $user_args, array(
				'meta_query' => array(
					array(
						'key'   => 'wpuc_user_id',
						'value' => $user_id
					)
				)
			) );
		}

		if ( $user_data ) {
			return array_filter( array_merge( array( $user_data ), (array) get_users( $user_args ) ) );
		} else {
			return (array) get_users( $user_args );
		}

	}

	/**
	 *
	 * Get Channels list
	 * 
	 * @param  array  $args
	 * @return string|false
	 * 
	 */
	public function get_channel_list( $args = array() ) {
		$users = $this->get_channels( $args );

		if ( $users ) {
			ob_start();

			for ( $i = 0; $i < count( $users ); $i++ ) {
				$user = $users[ $i ];
				include WP_USER_CHANNEL_PATH . 'partials/channel.php';
			}

			return ob_get_clean();
		}
		return false;
	}

	/**
	 *
	 * AJAX create channel
	 * 
	 */
	public function ajax_create_channel() {

		check_ajax_referer( 'wp_user_channel/create_channel_' . get_current_user_id(), 'create_channel' );

		$http_data = wp_parse_args( $_REQUEST, array(
			'channel_name' => ''
		) );

		$response = $this->create_channel( $http_data['channel_name'] );

		if ( is_wp_error( $response ) ) {
			wp_send_json_error( $response );
		}

		wp_send_json_success( sprintf(
			esc_html__( '%s has been created successfully.', 'wp-user-channel' ),
			$response->display_name
		) );
	}

	/**
	 *
	 * AJAX get list channel
	 * 
	 */
	public function ajax_get_channel_list() {
		check_ajax_referer( '_wpnonce' );

		$response = $this->get_channel_list();

		if ( ! $response ) {
			wp_send_json_error( new WP_Error(
				'empty',
				esc_html__( 'You have not created any channels yet.', 'wp-user-channel' )
			) );
		}

		wp_send_json_success( $response );
	}

	/**
	 *
	 * Do not return empty_email error if current user is a channel
	 * 
	 */
	public function omit_update_email_address_error( $errors, $update, $user ) {

		if ( user_can( $user->ID, 'wpuc_channel' ) ) {
			if ( in_array( 'empty_email', $errors->get_error_codes() ) ) {
				$errors->remove( 'empty_email' );
			}
		}
	}

	/**
	 *
	 * Display last activity of given user
	 * Requires buddyPress
	 * 
	 * @param  WP_User $user
	 * 
	 */
	public function display_channel_last_activity( $user ) {
		if ( function_exists( 'bp_get_user_last_activity' ) ) {
			$last_activity = bp_get_user_last_activity( $user->ID );
			if ( $last_activity ) {
				$last_activity = sprintf(
					esc_html__( 'Active %s ago', 'wp-user-channel' ),
					human_time_diff( strtotime( bp_get_user_last_activity( $user->ID ) ), current_time( 'U' ) )
				);
			} else {
				$last_activity = esc_html__( 'Never active', 'wp-user-channel' );
			}
			printf(
				'<span class="activity text-muted small">%s</span>',
				$last_activity
			);
		}
	}

	/**
	 *
	 * Display unread notification count of given user
	 * Requires buddyPress
	 * 
	 * @param  WP_User $user
	 * 
	 */
	public function display_channel_unread_notification_count( $user ) {
		if ( ! function_exists( 'bp_notifications_get_unread_notification_count' ) ) {
			return;
		}

		$count = bp_notifications_get_unread_notification_count( $user->ID );

		if ( $count ) {
			printf(
				'<span class="badge bg-danger notification-count">%s<span>',
				apply_filters( 'wp_user_channel/format_number', number_format_i18n( $count ) )
			);
		}
	}

	/**
	 *
	 * Display in dropdown menu
	 * 
	 */
	public function display_dropdown_menu( $items ) {
		$items['channels'] = array(
			'title'    => esc_html__( 'Switch Account', 'wp-user-channel' ),
			'icon'     => 'icon-arrows-cw',
			'url'      => '#',
			'priority' => 5,
			'private'  => true,
			'cap'      => 'list_channels'
		);

		return $items;
	}

	/**
	 *
	 * Display the "Channels" menu item within frontend dashboard
	 * 
	 */
	public function display_dashboard_menu( $items ) {
		$items['channels'] = array(
			'title'    => esc_html__( 'Channels', 'wp-user-channel' ),
			'desc'     => esc_html__( 'All Channels', 'wp-user-channel' ),
			'icon'     => 'icon-arrows-cw',
			'callback' => function () {
				load_template( WP_USER_CHANNEL_PATH . 'partials/dashboard-channels.php' );
			},
			'parent'   => 'dashboard',
			'cap'      => 'list_channels',
			'priority' => 5
		);

		return $items;
	}

	/**
	 *
	 * Load the modal Accounts
	 * 
	 */
	public function load_modal_accounts() {
		if ( is_user_logged_in() ) {
			load_template( WP_USER_CHANNEL_PATH . 'partials/modal-channels.php' );
		}
	}

	/**
	 * Load the plugin text domain for translation.
	 *
	 */
	public function load_plugin_textdomain() {
		if ( version_compare( wp_get_wp_version(), '6.8', '<' ) ) {
			load_plugin_textdomain(
				'wp-user-channel',
				false,
				dirname( dirname( plugin_basename( __FILE__ ) ) ) . '/languages/'
			);
		}
	}

	/**
	 * Load assets
	 */
	public function enqueue_scripts() {
		wp_enqueue_script(
			WP_USER_CHANNEL,
			WP_USER_CHANNEL_URL . 'scripts.js',
			array( 'jquery' ),
			filemtime( WP_USER_CHANNEL_PATH . 'scripts.js' ),
			true
		);

		wp_enqueue_style(
			WP_USER_CHANNEL,
			WP_USER_CHANNEL_URL . 'style.css',
			array( 'bootstrap' ),
			filemtime( WP_USER_CHANNEL_PATH . 'style.css' ),
		);
	}

	public function register_permission_module() {
		if ( function_exists( 'streamtube_core_register_module' ) ) {
			streamtube_core_register_module(
				'wp-user-channel',
				esc_html__( 'WP User Channel', 'wp-user-channel' ),
				array(
					'add_channels',
					'list_channels'
				) );
		}
	}
}