<?php
/**
 * The admin-specific functionality of the plugin.
 *
 * @link       https://santapress.halloecho.de
 * @since      1.0.0
 *
 * @package    SantaPress
 * @subpackage SantaPress/admin
 */

class SantaPress_Admin {

	/**
	 * The ID of this plugin.
	 *
	 * @since    1.0.0
	 * @access   private
	 * @var      string    $plugin_name    The ID of this plugin.
	 */
	private $plugin_name;

	/**
	 * The version of this plugin.
	 *
	 * @since    1.0.0
	 * @access   private
	 * @var      string    $version    The current version of this plugin.
	 */
	private $version;

	/**
	 * Check if doors exists
	 *
	 * @since    1.0.0
	 * @access   private
	 * @var      boolean    $doorsExist
	 */
	private $doorsExist;	

	/**
	 * Initialize the class and set its properties.
	 *
	 * @since    1.0.0
	 * @param      string    $plugin_name       The name of this plugin.
	 * @param      string    $version    The version of this plugin.
	 */
	public function __construct( $plugin_name, $version ) {

		$this->plugin_name = $plugin_name;
		$this->version = $version;

		// Migrate and update obsolete acf fields prior to 1.6.1 
		$this->updateObsoletFields();

	}

	/**
	 * Register the stylesheets for the admin area.
	 *
	 * @since    1.0.0
	 */
	public function enqueue_styles() {

		/**
		 * An instance of this class should be passed to the run() function
		 * defined in SantaPress_Loader as all of the hooks are defined
		 * in that particular class.
		 *
		 * The SantaPress_Loader will then create the relationship
		 * between the defined hooks and the functions defined in this
		 * class.
		 */

		// Css rules for color picker
    	wp_enqueue_style( 'wp-color-picker' );
    	// jQuery UI CSS for date picker
		wp_enqueue_style( 'jquery-ui-css', plugin_dir_url( __FILE__ ) . 'css/vendor/jquery-ui.min.css', array(), '1.11.4', 'all' );
		wp_enqueue_style( $this->plugin_name, plugin_dir_url( __FILE__ ) . 'css/santapress-admin.css', array(), $this->version, 'all' );

	}

	/**
	 * Register the JavaScript for the admin area.
	 *
	 * @since    1.0.0
	 */
	public function enqueue_scripts() {

		/**
		 * An instance of this class should be passed to the run() function
		 * defined in SantaPress_Loader as all of the hooks are defined
		 * in that particular class.
		 *
		 * The SantaPress_Loader will then create the relationship
		 * between the defined hooks and the functions defined in this
		 * class.
		 */
		
		wp_enqueue_script('jquery-ui-datepicker');
		wp_enqueue_script('wp-color-picker');

		// Register script
		wp_register_script( $this->plugin_name, plugin_dir_url( __FILE__ ) . 'js/santapress-admin.js', array( 'jquery' ), $this->version, false );

		// Localize the script with new data
		$translation_array = array(
			'plugin_admin_dir_url' => plugin_dir_url( __FILE__ ),
		);
		wp_localize_script( $this->plugin_name, 'data', $translation_array );

		// Enqueued script with localized data.
		wp_enqueue_script( $this->plugin_name );

	}

	/**
	 * Set ACF Path
	 *
	 * @since    1.0.0
	 */
	public function acf_settings_path( $path ) {

	    // update path
	    $path = plugin_dir_path( __FILE__ ) . '../includes/vendor/acf/';
	    
	    // return
	    return $path;
	    
	}

	/**
	 * Set ACF Dir
	 *
	 * @since    1.0.0
	 */
	public function acf_settings_dir( $dir ) {

	    // update path
	    $dir = plugin_dir_url( __FILE__ ) . '../includes/vendor/acf/';
	    
	    // return
	    return $dir;

	}

	/**
	 * Set ACF URL
	 *
	 * @since    1.4.1
	 */
	public function acf_settings_url( $dir ) {

	    // update path
	    $dir = plugin_dir_url( __FILE__ ) . '../includes/vendor/acf/';
	    
	    // return
	    return $dir;

	}

	/**
	 * Hide ACF if plugin is not already active on the client site
	 *
	 * @since    1.0.0
	 */
	public function acf_show_admin( ) {
		return false;
	}

	/**
	 * Setup ACF fields
	 *
	 * @since    1.0.0
	 */
	public function setup_acf_fields() {
		
		if( function_exists('acf_add_local_field_group') ):
	
			$door_type_choices = array();
			$door_type_choices['global'] = esc_html__('Use global setting', 'santapress');	
			$door_type_choices['regular'] = esc_html__('Content should open in a regular window', 'santapress');
			$door_type_choices['popup'] = esc_html__('Content should open in popup/modal window', 'santapress');
			if ( in_array( 'woocommerce/woocommerce.php', apply_filters( 'active_plugins', get_option( 'active_plugins' ) ) ) )
				$door_type_choices['wc'] = esc_html__('Link to WooCommerce product', 'santapress');
			$door_type_choices['external'] = esc_html__('External link (opens a new tab)', 'santapress');

			acf_add_local_field_group(array(
				'key' => 'group_5bceec4a740b6',
				'title' => 'SantaPress Door Setting',
				'fields' => array(
					array(
						'key' => 'field_5bceec4f3ace9',
						'label' => 'Door Image',
						'name' => 'santapress_door_image',
						'type' => 'text',
						'instructions' => '',
						'required' => 0,
						'conditional_logic' => 0,
						'wrapper' => array(
							'width' => '',
							'class' => 'santapress_door_image',
							'id' => '',
						),
						'default_value' => '',
						'placeholder' => '',
						'prepend' => '',
						'append' => '',
						'maxlength' => '',
					),
					array(
						'key' => 'field_5bd8812461b9f',
						'label' => 'Custom Door Image',
						'name' => 'santapress_custom_door_image',
						'type' => 'image',
						'instructions' => sprintf( wp_kses ( __('This overrides the door image set above.<br />Recommended is either an SVG file with a ratio of 1:1 or a PNG/JPG file with a ratio of 1:1 (e.g. 500px x 500px).<br />Other resolutions are also supported though you probably need to adjust the layout via.', 'santapress'), array('br' => array(), 'a' => array('href'))), '<a href="admin.php?page=santapress_settings">custom css</a>'),
						'required' => 0,
						'conditional_logic' => 0,
						'wrapper' => array(
							'width' => '',
							'class' => '',
							'id' => '',
						),
						'return_format' => 'array',
						'preview_size' => 'thumbnail',
						'library' => 'all',
						'min_width' => '',
						'min_height' => '',
						'min_size' => '',
						'max_width' => '',
						'max_height' => '',
						'max_size' => '',
						'mime_types' => '',
					),
					array(
						'key' => 'field_5bceec663acea',
						'label' => 'Door Type',
						'name' => 'santapress_door_type',
						'type' => 'select',
						'instructions' => esc_html__('Select the door type', 'santapress'),
						'required' => 0,
						'conditional_logic' => 0,
						'wrapper' => array(
							'width' => '',
							'class' => '',
							'id' => '',
						),
						'choices' => $door_type_choices,
						'default_value' => array(
							0 => 'global',
						),
						'allow_null' => 0,
						'multiple' => 0,
						'ui' => 0,
						'return_format' => 'value',
						'ajax' => 0,
						'placeholder' => '',
					),
					array(
						'key' => 'field_5bceecec3aceb',
						'label' => 'Link',
						'name' => 'santapress_door_type_link_url',
						'type' => 'url',
						'instructions' => esc_html__('Please provide the url', 'santapress'),
						'required' => 0,
						'conditional_logic' => array(
							array(
								array(
									'field' => 'field_5bceec663acea',
									'operator' => '==',
									'value' => 'external',
								),
							),
						),
						'wrapper' => array(
							'width' => '',
							'class' => '',
							'id' => '',
						),
						'default_value' => '',
						'placeholder' => '',
					),
					array(
						'key' => 'field_5be178d918b2f',
						'label' => 'WooCommerce Product',
						'name' => 'santapress_door_type_wc',
						'type' => 'post_object',
						'instructions' => '',
						'required' => 0,
						'conditional_logic' => array(
							array(
								array(
									'field' => 'field_5bceec663acea',
									'operator' => '==',
									'value' => 'wc',
								),
							),
						),
						'wrapper' => array(
							'width' => '',
							'class' => '',
							'id' => '',
						),
						'post_type' => array(
							0 => 'product',
						),
						'taxonomy' => '',
						'allow_null' => 1,
						'multiple' => 0,
						'return_format' => 'object',
						'ui' => 1,
					),
					array(
						'key' => 'field_5be3df21de5cb',
						'label' => 'Custom Javascript',
						'name' => 'santapress_custom_javascript',
						'type' => 'textarea',
						'instructions' => esc_html__('You can put any custom javascript in here (e.g. tracking codes, google tag event etc.). This only works if the door type opens is either in regular or modal/popup.', 'santapress'),
						'required' => 0,
						'conditional_logic' => array(
							array(
								array(
									'field' => 'field_5bceec663acea',
									'operator' => '==',
									'value' => 'regular',
								),
							),
							array(
								array(
									'field' => 'field_5bceec663acea',
									'operator' => '==',
									'value' => 'popup',
								),
							),
							array(
								array(
									'field' => 'field_5bceec663acea',
									'operator' => '==',
									'value' => 'global',
								),
							),
						),
						'wrapper' => array(
							'width' => '',
							'class' => '',
							'id' => '',
						),
						'default_value' => '',
						'placeholder' => '<script><!-- your code --></script>',
						'maxlength' => '',
						'rows' => '',
						'new_lines' => '',
					),
				),
				'location' => array(
					array(
						array(
							'param' => 'post_type',
							'operator' => '==',
							'value' => 'door',
						),
					),
				),
				'menu_order' => 0,
				'position' => 'normal',
				'style' => 'default',
				'label_placement' => 'top',
				'instruction_placement' => 'label',
				'hide_on_screen' => '',
				'active' => 1,
				'description' => '',
			));

			acf_add_local_field_group(array(
				'key' => 'group_5dc18bf0d8a10',
				'title' => 'Quiz Settings',
				'fields' => array(
					array(
						'key' => 'field_5dc18bf5481e0',
						'label' => esc_html__('Enable Quiz', 'santapress'),
						'name' => 'santapress_activate_quiz',
						'type' => 'true_false',
						'instructions' => '',
						'required' => 0,
						'conditional_logic' => 0,
						'wrapper' => array(
							'width' => '',
							'class' => '',
							'id' => '',
						),
						'message' => esc_html__('Enable Quiz for this door', 'santapress'),
						'default_value' => 0,
						'ui' => 0,
						'ui_on_text' => '',
						'ui_off_text' => '',
					),
					array(
						'key' => 'field_5dc18c0c481e1',
						'label' => esc_html__('Question', 'santapress'),
						'name' => 'santapress_quiz_question',
						'type' => 'text',
						'instructions' => '',
						'required' => 0,
						'conditional_logic' => array(
							array(
								array(
									'field' => 'field_5dc18bf5481e0',
									'operator' => '==',
									'value' => '1',
								),
							),
						),
						'wrapper' => array(
							'width' => '',
							'class' => '',
							'id' => '',
						),
						'default_value' => '',
						'placeholder' => '',
						'prepend' => '',
						'append' => '',
						'maxlength' => '',
					),
					array(
						'key' => 'field_5dc18c2f481e2',
						'label' => esc_html__('Answers', 'santapress'),
						'name' => 'santapress_quiz_answers_161',
						'type' => 'textarea',
						'instructions' => wp_kses( __('Please provide your answers with following schema - one answer per row:<br /><strong>Your answer|true</strong> or <strong>Your answer|false</strong><br /><div><u>Example</u><br /><em>Question: In the song Frosty the Snowman, what made Frosty come to life?</em><br />An old silk hat|true<br />An carrot|false<br />A dirty broom|false</div>', 'santapress'), array('br' => array(), 'strong' => array(), 'div' => array(), 'u' => array())),
						'required' => 0,
						'conditional_logic' => array(
							array(
								array(
									'field' => 'field_5dc18bf5481e0',
									'operator' => '==',
									'value' => '1',
								),
							),
						),
						'wrapper' => array(
							'width' => '',
							'class' => '',
							'id' => '',
						),
						'default_value' => '',
						'maxlength' => '',
						'rows' => '',
						'placeholder' => "An old silk hat|true\nAn carrot|false\nA dirty broom|false",
						'new_lines' => '',
					),
					array(
						'key' => 'field_5dec00dbdf7c5',
						'label' => esc_html__('Correct Response', 'santapress'),
						'name' => 'santapress_correct_response',
						'type' => 'wysiwyg',
						'instructions' => esc_html__('Insert content that should show up after the user has clicked on a correct answer.') . '<br />' . esc_html__('If you leave this field empty the response set under SantaPress > Quiz Settings will show up.', 'santapress') . '<br />' . 
							sprintf( wp_kses( __('You can also place your %s shortcode here.', 'santapress'), array('a' => array('href', 'target'))), '<a href="https://de.wordpress.org/plugins/contact-form-7/" target="_blank">Contact Form 7</a>'),
						'required' => 0,
						'conditional_logic' => array(
							array(
								array(
									'field' => 'field_5dc18bf5481e0',
									'operator' => '==',
									'value' => '1',
								),
							),
						),
						'wrapper' => array(
							'width' => '',
							'class' => '',
							'id' => '',
						),
						'default_value' => '',
						'tabs' => 'all',
						'toolbar' => 'full',
						'media_upload' => 1,
						'delay' => 0,
					),
					array(
						'key' => 'field_5dec0171df7c6',
						'label' => 'Incorect Response',
						'name' => 'santapress_incorrect_response',
						'type' => 'wysiwyg',
						'instructions' => esc_html__('Insert content that should show up after the user has clicked on a wrong answer.') . '<br />' . esc_html__('If you leave this field empty the response set under SantaPress > Quiz Settings will show up.', 'santapress') . '<br />' .
							sprintf( wp_kses( __('You can also place your %s shortcode here.', 'santapress'), array('a' => array('href', 'target'))), '<a href="https://de.wordpress.org/plugins/contact-form-7/" target="_blank">Contact Form 7</a>'),
						'required' => 0,
						'conditional_logic' => array(
							array(
								array(
									'field' => 'field_5dc18bf5481e0',
									'operator' => '==',
									'value' => '1',
								),
							),
						),
						'wrapper' => array(
							'width' => '',
							'class' => '',
							'id' => '',
						),
						'default_value' => '',
						'tabs' => 'all',
						'toolbar' => 'full',
						'media_upload' => 1,
						'delay' => 0,
					),
				),
				'location' => array(
					array(
						array(
							'param' => 'post_type',
							'operator' => '==',
							'value' => 'door',
						),
					),
				),
				'menu_order' => 10,
				'position' => 'normal',
				'style' => 'default',
				'label_placement' => 'top',
				'instruction_placement' => 'label',
				'hide_on_screen' => '',
				'active' => 1,
				'description' => '',
			));

		endif;
	}

	/**
	 * Check doors
	 *
	 * @since    1.0.0
	 */
	public function check_doors() {

		$args = array(
			'posts_per_page'   => -1,
			'orderby'          => 'menu_order',
			'order'            => 'ASC',
			'post_type'        => 'door',
			'post_status'      => 'publish, future',
			'suppress_filters' => true
		);
		$doors = get_posts( $args );

		if($doors)
			return true;
		else
			return false;

	}

	/**
	 * Add Admin CSS
	 *
	 * @since    1.0.0
	 */
	public function add_admin_css() {

		if(!$this->check_doors()) {

			echo '
			<style>
			#toplevel_page_santapress .wp-submenu li:nth-child(3),
			#toplevel_page_santapress .wp-submenu li:nth-child(4),
			#toplevel_page_santapress .wp-submenu li:nth-child(5) {
				display:none;
			}
			</style>
			';

		}

		if(ISSET($_GET['page']) && (
			$_GET['page'] == 'santapress_installation' ||
			$_GET['page'] == 'santapress_layout_settings' ||
			$_GET['page'] == 'santapress_config' ||
			$_GET['page'] == 'santapress_quiz'
			)) {
			echo '
			<style>
			#wpcontent {
			    background-image: url('.plugin_dir_url( __FILE__ ).'img/bg_pattern_admin.png);
			    background-size: 100%;
			    background-repeat: no-repeat;
			    background-position-y: -350px;
			   }
			</style>
			';
		}

	}

	/**
	 * Menu page
	 *
	 * @since  1.0.0
	 */
	public function add_admin_menu() {

		// add menu page
		add_menu_page(
			esc_html__( 'SantaPress', 'santapress' ),
			esc_html__( 'SantaPress','santapress' ),
			'manage_options',
			'santapress',
			array( $this, 'santapress_admin_parent' ),
			'dashicons-star-filled'
		);

		// add submenu page
		add_submenu_page(
			'santapress',
			esc_html__( 'Installation', 'santapress'), 
			esc_html__( 'Installation', 'santapress'), 
			'manage_options', 
			'santapress_installation', 
			array( $this, 'santapress_admin_settings' )
		);

		// remove parent menu item from submenu
		remove_submenu_page('santapress', 'santapress');

		// add submenu page
		add_submenu_page(
			'santapress',
			esc_html__( 'Layout', 'santapress'), 
			esc_html__( 'Layout', 'santapress'), 
			'manage_options', 
			'santapress_layout_settings', 
			array( $this, 'santapress_admin_layout' )
		);

		// add submenu page
		add_submenu_page(
			'santapress',
			esc_html__( 'Settings', 'santapress'), 
			esc_html__( 'Global Settings', 'santapress'), 
			'manage_options', 
			'santapress_config', 
			array( $this, 'santapress_admin_config' )
		);

		add_submenu_page(
			'santapress',
			esc_html__( 'Quiz', 'santapress'), 
			esc_html__( 'Quiz Settings', 'santapress'), 
			'manage_options', 
			'santapress_quiz', 
			array( $this, 'santapress_admin_quiz' )
		);


	}

	/**
	 * Empty callback function for menu
	 *
	 * @since    1.0.0
	 */
	public function santapress_admin_parent() {
		// just an empty callback function
		return;
    }

	/**
	 * SantaPress Installation
	 *
	 * @since    1.0.0
	 */
	public function santapress_admin_settings() {
         

        if( ISSET($_GET['settings-updated']) && $_GET['settings-updated'] == true ) {
     		
     		$start_date = get_option('santapress_date_start');
     		$end_date = get_option('santapress_date_end');
     		$santapress_layout = get_option('santapress_layout');
     		$santapress_doortype = get_option('santapress_doortype');

     		$dStart = new DateTime($start_date);
     		$dStart->setTime(0, 0);
     		$post_date = $dStart;
			$days = $end_date;

			$_day = (int)$dStart->format('d');

			$santa_quotes = array(
				array('Gifts of time and love are surely the basic ingredients of a truly merry Christmas.', 'Peg Bracken'),
				array('We are better throughout the year for having, in spirit, become a child again at Christmastime.', 'Laura Ingalls Wilder'),
				array('He who has not Christmas in his heart will never find it under a tree.', 'Roy L. Smith'),
				array('Christmas waves a magic wand over this world, and behold, everything is softer and more beautiful.', 'Norman Vincent Peale'),
				array('My idea of Christmas, whether old-fashioned or modern, is very simple: loving others.', 'Bob Hope'),
				array('One of the most glorious messes in the world is the mess created in the living room on Christmas Day. Don’t clean it up too quickly.', 'Andy Rooney'),
				array('At Christmas, all roads lead home.', 'Marjorie Holmes'),
				array('Christmas is a season for kindling the fire for hospitality in the hall, the genial flame of charity in the heart.', 'Washington Irving'),
				array('A good conscience is a continual Christmas.', 'Benjamin Franklin'),
				array('Christmas is a day of meaning and traditions, a special day spent in the warm circle of family and friends.', 'Margaret Thatcher'),
				array('Christmas isn’t a season. It’s a feeling.', 'Edna Ferber'),
				array('Christmas is a necessity. There has to be at least one day of the year to remind us that we’re here for something else besides ourselves.', 'Eric Sevareid'),
				array('Christmas is a bridge. We need bridges as the river of time flows past. Today’s Christmas should mean creating happy hours for tomorrow and reliving those of yesterday.', 'Gladys Taber'),
				array('Christmas is a season not only of rejoicing, but of reflection.', 'Winston Churchill'),
				array('Christmas is a tonic for our souls. It moves us to think of others rather than of ourselves. It directs our thoughts to giving.', 'B.C. Forbes'),			
				array('We send out Christmas greetings to friends and family near and far, we put up the tree in hopes Santa will arrive, but most of all we take the time to thank God for His Son.', 'Kate Summers'),
				array('Christmas is for everyone, adults and children alike. Allow this season fill your heart, and let go of the things you dislike.', 'Julie Hebert'),
				array('Christmas can\'t be bought from a store - Maybe Christmas means a little bit more,', 'Dr.Seuss'),
				array('May your hearth be warm, your holidays grand, and your heart held gently in the good Lord\'s hand.', 'Irish Saying'),
				array('Christmas is the celebration of the event that set heaven to singing, an event that gave the stars of the night sky a new brilliance.', 'Billy Graham'),
				array('Christmas is joy, religious joy, an inner joy of light and peace.', 'Pope Francis'),
				array('This is the message of Christmas: We are never alone.', 'Taylor Caldwell'),
				array('It is Christmas in the heart that puts Christmas in the air.', 'W. T. Ellis'),
				array('Christmas is the day that holds all time together.', 'Alexander Smith'),
				array('Blessed is the season which engages the whole world in a conspiracy of love.', 'Hamilton Wright Mabie')
			);

			shuffle($santa_quotes);

			$santa_quiz_questions = array(
				array( 'In the song Frosty the Snowman, what made Frosty come to life?', array( 'An old silk hat|true', 'A carrot|false', 'A dirty broom|false' ) ),
				array( 'What Christmas-themed ballet premiered in Saint Petersburg, Russia in 1892?', array( 'The swan|false', 'The Nutcracker|true', 'The Polar Express|false' ) ),
				array( 'What Bing Crosby song is the best-selling single ever?', array( 'White Christmas|true', 'Jingle Bells|false', 'Holy Night, Silent Night|false' ) ),
				array( 'Who was the star of the movie titled White Christmas?', array( 'Bing Crosby|true', 'Tom Hanks|false', 'Matt Damon|false') ),
				array( 'What was the name of the animated movie featuring Tom Hanks?', array( 'Ice Age|false', 'The Polar Express|true') ),
				array( 'The red and white costume of Father Christmas wa  first introduced by which company?', array( 'Microsoft|false', 'Coca-Cola|true', 'Apple|false') ),
				array( 'Approximately how many turkeys are purchased every year in the UK, 1 million, 10 million, or 20 million?', array( '1 million|false', '10 million|true', '20 millon|false') ),
				array( 'How do you say Merry Christmas in Spanish?', array( 'Feliz Navidad|true', 'Buenas noches|false', 'Buongiorno|false') ),
				array( 'True or False: The first Christmas tree appeared in the United States.', array( 'False|true', 'True|false') ),
				array( 'Who created Rudolph the rednose reindeer?', array( 'Michael S. in 1983|false', 'Robert L. May in 1939|true', 'Martin S. White in 1956|false') ),
				array( 'How many reindeer drive Santa Claus’ sleigh (counting Rudolph)?', array( 'Twelve|false', 'Nine|true', 'Six|false') ),
				array( 'Who once stole Christmas?', array( 'The Gronch|false', 'The Grinch|true', 'A grizzly bear|false') ),
				array( 'Which country was the first to use the tradition of Christmas trees?', array( 'Austria|false', 'Germany|true', 'Spain|false') ),
				array( 'Who invented Christmas lights?', array( 'Thomas Edison|true', 'Einstein|false', 'Benjamin Franklin|false') ),
				array( 'What is Santa Claus name in France?', array( 'Père Noël|true', 'Frère Jacques|false') ),
				array( 'How many days do we have between Christmas and the Epiphany?', array( 'Nine|false', 'Ten|false', 'Twelve|true') ),
				array( 'Where was Jesus born?', array( 'Bethlehem|true', 'Jersusalem|false') ),
				array( 'What is the color of mistletoe berries of the plant?', array( 'White|true', 'Red|false', 'Yellow|false') ),
				array( 'Which was the last state in the United States that declared Christmas a legal holiday?', array( 'Oklahoma|true', 'Pennsylvania|false', 'Alamaba|false') ),
				array( 'In the song Jingle Bells, what kind of a sleigh is mentioned?', array( 'A one-horse open sleigh|true','A two-horse open sleigh|false') ),
				array( 'Which is the most famous Christmas ballet of all time?', array( 'Cinderella|false', 'The Nutcracker|true', 'Swan Lake|false') ),
				array( 'What song does Lucy ask Schroeder to play on his piano in “A Charlie Brown Christmas”?', array( 'Jingle Bells|true', 'White Christmas|false', 'Feliz Navidad|false') ),
				array( 'Where was the song Silent Night by Franz Xaver Gruber composed?', array( 'Amsterdam|Netherlands|false', 'Oberndorf/Austria|true', 'Munich/Germany|false') ),
				array( 'How many points does a snowflake traditionally have?', array( 'Five|false', 'Six|true', 'Seven|false') ),
				array( 'Which country does St. Nicholas originally belong to?', array( 'Turkey|true', 'The Netherlands|false', 'Spain|false') )
			);

			shuffle($santa_quiz_questions);

			// Initialize Quiz
			$santapress_init_quiz = get_option('santapress_init_quiz');

			for($i=1;$i<=$days;$i++) {	
     		
         		// Initialize the post ID to -1. This indicates no action has been taken.
				$post_id = -1;

				// Setup the author, slug, and title for the post
				$slug = $_day;
				$title = 'Day ' . $_day;

				// If the page doesn't already exist, then create it
				if( null == get_page_by_title( $title ) ) {

					// shuffle santa quotes
					//$random_key = mt_rand(0, count($santa_quotes)-1);
					$santa_quote_key = $i - 1;
					if(ISSET($santa_quotes[$santa_quote_key]))
						$santa_quote = '<blockquote>'.$santa_quotes[$santa_quote_key][0].'<footer>'.$santa_quotes[$santa_quote_key][1].'</footer></blockquote>';
					else
						$santa_quote = '';

					$prependContent = $santa_quote;

					// if quiz is initialized no quotes should be inserted, instead the [sp_quiz] shortcode should be prepended
					if($santapress_init_quiz) {
						$prependContent = '<p>[sp_quiz]</p>';
					}

					// Set the page ID so that we know the page was created successfully
					$post_id = wp_insert_post(
						array(
							'comment_status'	=>	'closed',
							'ping_status'		=>	'closed',
							'post_name'			=>	$slug,
							'post_title'		=>	$title,
							'post_status'		=>	'publish',
							'post_content'		=>  $prependContent,
							'post_date'      	=> 	$post_date->format('Y-m-d H:i:s'),
							'post_type'			=>	'door',
							'menu_order'		=> 	$_day
						)
					);

					// add original day to post meta
					add_post_meta( $post_id, 'santapress_day', $i, true );

					// Save door graphics
					if($santapress_layout) {
						if($_day <= 25) {
							$imgUrl = plugins_url( '/public/img/'.$santapress_layout.'/'.$_day.'.png', dirname(__FILE__) );
							update_field('santapress_door_image', $imgUrl, $post_id);
						}
					}

					if($santapress_doortype) {
						update_field('santapress_door_type', 'global', $post_id); // for each post
						update_option('santapress_doortype', $santapress_doortype); // for the global option
					}

					// Add quiz data
					if($santapress_init_quiz) {
						update_field('santapress_activate_quiz', true, $post_id);

						$santa_question_key = $i - 1;
						if(ISSET($santa_quiz_questions[$santa_question_key])) {
							$question = $santa_quiz_questions[$santa_question_key][0];
							update_field('santapress_quiz_question', $question, $post_id);
							if(is_array($santa_quiz_questions[$santa_question_key][1])) {
								$answers = $santa_quiz_questions[$santa_question_key][1];
								$answerStr = '';
								foreach($answers as $answer) {
									$answerStr .= $answer;
									$answerStr .= "\n";
								}
								update_field('santapress_quiz_answers_161', $answerStr, $post_id);
							}
						}

					}

				}

				$_day++;
				$post_date->modify('+1 day');

			}

			$this->doorsExist = true;

			echo '
			<style>
			#toplevel_page_santapress .wp-submenu li:nth-child(3),
			#toplevel_page_santapress .wp-submenu li:nth-child(4),
			#toplevel_page_santapress .wp-submenu li:nth-child(5) {
				display:block;
			}
			.dashicons-star-empty:before {
			    content: "\f155";
			}
			#setting-error-settings_updated {
				display:none;
			}
			</style>
			';

			$message = sprintf( wp_kses ( __('Settings saved. You have successfully set up SantaPress. You can now go to %s to see your doors.<br />To include SantaPress you can use the shortcode [santapress].<br />For more options, please go to the %s.<br />For help please see the %s and the %s.', 'santapress'), array('a' => array('href', 'target'), 'br' => array())), '<a href="edit.php?post_type=door">Doors</a>', '<a href="admin.php?page=santapress_config">Global Settings</a>', '<a href="https://santapress.halloecho.de/docs/" target="_blank">docs</a>', '<a href="https://santapress.halloecho.de/faq/" target="_blank">FAQ</a>');
			add_settings_error('santapress_init_notice', 'santapress_init_notice', $message, 'updated');

			$santapress_init = get_option('santapress_init');
			if(!$santapress_init) {

				// set default settings
				update_option('santapress_previewmode', 1);
				update_option('santapress_showtitle', 1);
				update_option('santapress_modal_width', 500);
				update_option('santapress_modal_autolayout', 1);
				update_option('santapress_modal_bgcolor', '#ffffff');
				update_option('santapress_modal_bgcolor2', '#a0bedc');
				update_option('santapress_modal_showhat', 1);
				update_option('santapress_modal_showpattern', 1);
				update_option('santapress_enablesnow', 'modal');
				update_option('santapress_tracking', 'logged-out-users');
				update_option('santapress_modal_mode', 'ajax'); // since 1.4.2
				update_option('santapress_updated_to_161', true); // since 1.6.1 to prevent checking for repeater field update

				// set default quiz settings
				update_option('santapress_quizcookie_msg', '<p>Sorry, your limit of tries is reached for today.</p><p>Please try again tomorrow :-) Good luck!</p>');
				update_option('santapress_quiztrue', '<p>Your answer was correct!</p><p>Please fill in the contact form.</p><p>[sp_quiz_back message="Try again »"]</p>');
				update_option('santapress_quizfalse', '<p>Your answer was not correct!</p><p>[sp_quiz_back message="Try again »"]</p>');

			}

			// init done
			update_option('santapress_init', 1); 

			// flush rewrite rules after init
			flush_rewrite_rules();

         	include 'partials/santapress-init.php';

        }

        else {
        	
	    	if( !$this->check_doors() || ISSET($_GET['forceinit']) ) {
	        	include 'partials/santapress-init.php';
	    	}
	    	else {
	    		include 'partials/santapress-init-exists.php';
	    	}
        
		}

    }

	/**
	 * Setup Sections
	 *
	 * @since    1.0.0
	 */
    public function setup_sections() {
		add_settings_section( 'santapress_settings_section', esc_html__( 'Initial Setup', 'santapress'), array( $this, 'section_callback' ), 'santapress_settings' );
		add_settings_section( 'santapress_layout_settings_section', '', array( $this, 'section_callback' ), 'santapress_layout_settings' );
		add_settings_section( 'santapress_config_section', '', array( $this, 'section_callback' ), 'santapress_config' );
		add_settings_section( 'santapress_quiz_section', '', array( $this, 'section_callback' ), 'santapress_quiz' );
 	}

	/**
	 * Empty callback function for sections
	 *
	 * @since    1.0.0
	 */
	public function section_callback( $arguments ) {
		
		if($arguments['id'] == 'santapress_config_section_general') {
			echo '<h2>'.esc_html__( 'General settings', 'santapress').'</h2>';
		}

		if($arguments['id'] == 'santapress_config_section_modal') {
			echo '<h2>'.esc_html__( 'Modal settings', 'santapress').'</h2>';
		}

		return;
	}

	/**
	 * Setup Fields
	 *
	 * @since    1.0.0
	 */
	public function setup_fields() {
	    
	    /* Installation Fields */
	    register_setting( 'santapress_settings', 'santapress_date_start' );
	    add_settings_field( 'santapress_date_start', esc_html__( 'Start date', 'santapress'), array( $this, 'santapress_date_start_callback' ), 'santapress_settings', 'santapress_settings_section' );

	    register_setting( 'santapress_settings', 'santapress_date_end' );
	    add_settings_field( 'santapress_date_end', esc_html__( 'Number of doors', 'santapress'), array( $this, 'santapress_date_end_callback' ), 'santapress_settings', 'santapress_settings_section' );

	    register_setting( 'santapress_settings', 'santapress_layout' );
	    add_settings_field( 'santapress_layout', esc_html__( 'Initial Layout', 'santapress'), array( $this, 'santapress_layout_callback' ), 'santapress_settings', 'santapress_settings_section' );

	    register_setting( 'santapress_settings', 'santapress_doortype' );
	    add_settings_field( 'santapress_doortype', esc_html__( 'Global door type', 'santapress'), array( $this, 'santapress_doortype_callback' ), 'santapress_settings', 'santapress_settings_section' );

	    register_setting( 'santapress_settings', 'santapress_init_quiz' );
	    add_settings_field( 'santapress_init_quiz', esc_html__( 'Initialize Quiz', 'santapress'), array( $this, 'santapress_init_quiz_callback' ), 'santapress_settings', 'santapress_settings_section' );

	    /* Layout Fields */
	    register_setting( 'santapress_layout_settings', 'santapress_layout' );
	    add_settings_field( 'santapress_layout', esc_html__( 'Choose a layout', 'santapress'), array( $this, 'santapress_layout_callback' ), 'santapress_layout_settings', 'santapress_layout_settings_section' );

	    /* Config Fields */
	    register_setting( 'santapress_config', 'santapress_previewmode' );
	    add_settings_field( 'santapress_previewmode', esc_html__( 'Preview Mode', 'santapress'), array( $this, 'santapress_previewmode_callback' ), 'santapress_config', 'santapress_config_section' );

	    register_setting( 'santapress_config', 'santapress_showtitle' );
	    add_settings_field( 'santapress_showtitle', esc_html__( 'Show Title', 'santapress'), array( $this, 'santapress_showtitle_callback' ), 'santapress_config', 'santapress_config_section' );  
	
	    register_setting( 'santapress_config', 'santapress_shuffledoors' );
	    add_settings_field( 'santapress_shuffledoors', esc_html__( 'Shuffle doors', 'santapress'), array( $this, 'santapress_shuffledoors_callback' ), 'santapress_config', 'santapress_config_section' ); 

    	register_setting( 'santapress_config', 'santapress_disablepastdoors' );
	    add_settings_field( 'santapress_disablepastdoors', esc_html__( 'Disable past doors', 'santapress'), array( $this, 'santapress_disablepastdoors_callback' ), 'santapress_config', 'santapress_config_section' ); 

    	register_setting( 'santapress_config', 'santapress_doortype' );
	    add_settings_field( 'santapress_doortype', esc_html__( 'Global door type', 'santapress'), array( $this, 'santapress_doortype_callback' ), 'santapress_config', 'santapress_config_section' ); 

	    register_setting( 'santapress_config', 'santapress_modal_width' );
	    add_settings_field( 'santapress_modal_width', esc_html__( 'Modal max width in px', 'santapress'), array( $this, 'santapress_modal_width_callback' ), 'santapress_config', 'santapress_config_section' ); 

	    register_setting( 'santapress_config', 'santapress_modal_autolayout' );
	    add_settings_field( 'santapress_modal_autolayout', esc_html__( 'Auto Layout for Modal', 'santapress'), array( $this, 'santapress_modal_autolayout_callback' ), 'santapress_config', 'santapress_config_section' );

	    register_setting( 'santapress_config', 'santapress_modal_bgcolor' );
	    add_settings_field( 'santapress_modal_bgcolor', esc_html__( 'Modal background color', 'santapress'), array( $this, 'santapress_modal_bgcolor_callback' ), 'santapress_config', 'santapress_config_section', array('class' => 'santapress_manual_layout')  );

	    register_setting( 'santapress_config', 'santapress_modal_bgcolor2' );
	    add_settings_field( 'santapress_modal_bgcolor2', esc_html__( 'Modal background color 2', 'santapress'), array( $this, 'santapress_modal_bgcolor2_callback' ), 'santapress_config', 'santapress_config_section', array('class' => 'santapress_manual_layout') );

	    //register_setting( 'santapress_config', 'santapress_modal_showhat' );
	    //add_settings_field( 'santapress_modal_showhat', esc_html__( 'Show Santa hat in modal', 'santapress'), array( $this, 'santapress_modal_showhat_callback' ), 'santapress_config', 'santapress_config_section', array('class' => 'santapress_manual_layout') ); 

	    register_setting( 'santapress_config', 'santapress_modal_showpattern' );
	    add_settings_field( 'santapress_modal_showpattern', esc_html__( 'Show Pattern in modal', 'santapress'), array( $this, 'santapress_modal_showpattern_callback' ), 'santapress_config', 'santapress_config_section', array('class' => 'santapress_manual_layout') ); 

	    register_setting( 'santapress_config', 'santapress_modal_mode' );
	    add_settings_field( 'santapress_modal_mode', esc_html__( 'Modal mode', 'santapress'), array( $this, 'santapress_modal_mode_callback' ), 'santapress_config', 'santapress_config_section' );

	    register_setting( 'santapress_config', 'santapress_enablesnow' );
	    add_settings_field( 'santapress_enablesnow', esc_html__( 'Enable snow', 'santapress'), array( $this, 'santapress_enablesnow_callback' ), 'santapress_config', 'santapress_config_section' );

    	register_setting( 'santapress_config', 'santapress_tracking' );
	    add_settings_field( 'santapress_tracking', esc_html__( 'Click Tracking', 'santapress'), array( $this, 'santapress_tracking_callback' ), 'santapress_config', 'santapress_config_section' ); 

    	register_setting( 'santapress_config', 'santapress_customcss' );
	    add_settings_field( 'santapress_customcss', esc_html__( 'Custom CSS', 'santapress'), array( $this, 'santapress_customcss_callback' ), 'santapress_config', 'santapress_config_section' );

    	register_setting( 'santapress_config', 'santapress_resettracking' );
	    add_settings_field( 'santapress_resettracking', esc_html__( 'Reset tracking count', 'santapress'), array( $this, 'santapress_resettracking_callback' ), 'santapress_config', 'santapress_config_section', array('class' => 'santapress_resettracking') );  
 
 		// SantaPress Quiz Fields
     	register_setting( 'santapress_quiz', 'santapress_quizcookie' );
	    add_settings_field( 'santapress_quizcookie', esc_html__( 'One-Try only /<br />Only allow one try per day', 'santapress'), array( $this, 'santapress_quizcookie_callback' ), 'santapress_quiz', 'santapress_quiz_section' );     	

     	register_setting( 'santapress_quiz', 'santapress_quizcookie_msg' );
	    add_settings_field( 'santapress_quizcookie_msg', esc_html__( 'Message to the user', 'santapress'), array( $this, 'santapress_quizcookie_msg_callback' ), 'santapress_quiz', 'santapress_quiz_section', array('class' => 'santapress_quizcookie_msg') );  

     	register_setting( 'santapress_quiz', 'santapress_quizcookie_type' );
	    add_settings_field( 'santapress_quizcookie_type', esc_html__( 'When to set the "One-Try" cookie', 'santapress'), array( $this, 'santapress_quizcookie_type_callback' ), 'santapress_quiz', 'santapress_quiz_section', array('class' => 'santapress_quizcookie_msg') );  

     	register_setting( 'santapress_quiz', 'santapress_quiztrue' );
	    add_settings_field( 'santapress_quiztrue', esc_html__( 'Correct response', 'santapress'), array( $this, 'santapress_quiztrue_callback' ), 'santapress_quiz', 'santapress_quiz_section' );     	
     	register_setting( 'santapress_quiz', 'santapress_quizfalse' );
	    add_settings_field( 'santapress_quizfalse', esc_html__( 'Incorrect response', 'santapress'), array( $this, 'santapress_quizfalse_callback' ), 'santapress_quiz', 'santapress_quiz_section' );     	

	}

	/**
	 * Callback functions for fields
	 *
	 * @since    1.0.0
	 */
	
	public function santapress_date_start_callback( $arguments ) {
		$start_date = get_option('santapress_date_start');
		if(!$start_date)
			$start_date = date('Y') . '-12-01';
		echo '<input type="text" class="santapress_datepicker" name="santapress_date_start" value="'.esc_attr($start_date).'" />';
	}

	public function santapress_date_end_callback( $arguments ) {
		$end_date = get_option('santapress_date_end');
		if(!$end_date)
			$end_date = 25;
		echo '<select name="santapress_date_end">';
		for($i=1;$i<=25;$i++) {
			echo '<option value="'.$i.'"';
			if($i == $end_date)
				echo ' selected';
			echo '>'.$i.'</option>';
		}
		echo '</select>';
	}

	public function santapress_layout_callback( $arguments ) {

		// set default layout for installation
		$santapress_init = get_option('santapress_init');

		// create santapress_selected_classes array
		for($i=0;$i<8;$i++) {
			$santapress_selected_classes[] = '';
		}
		if(!$santapress_init) {
			update_option('santapress_layout', '01');
			$santapress_selected_classes[0] = 'selected';
		}
		else {
			$santapress_layout = get_option('santapress_layout');
			$santapress_selected_classes[(int)$santapress_layout-1] = 'selected';
		}

		echo '<div class="santapress_layout_row">';
		echo '<div class="santapress_layout_col"><input type="radio" name="santapress_layout" id="santapress_layout_01" value="01" '.checked('01', get_option('santapress_layout'), false).'><label for="santapress_layout_01"><img src="'.plugin_dir_url( __FILE__ ) . 'img/santapress_layout_01.jpg" class="'.$santapress_selected_classes[0].'"></label></div>';
		echo '<div class="santapress_layout_col"><input type="radio"  name="santapress_layout" id="santapress_layout_02" value="02" '.checked('02', get_option('santapress_layout'), false).'><label for="santapress_layout_02"><img src="'.plugin_dir_url( __FILE__ ) . 'img/santapress_layout_02.jpg" class="'.$santapress_selected_classes[1].'"></label></div>';
		echo '<div class="santapress_layout_col"><input type="radio"  name="santapress_layout" id="santapress_layout_03" value="03" '.checked('03', get_option('santapress_layout'), false).'><label for="santapress_layout_03"><img src="'.plugin_dir_url( __FILE__ ) . 'img/santapress_layout_03.jpg" class="'.$santapress_selected_classes[2].'"></label></div>';
		echo '<div class="santapress_layout_col"><input type="radio"  name="santapress_layout" id="santapress_layout_04" value="04" '.checked('04', get_option('santapress_layout'), false).'><label for="santapress_layout_04"><img src="'.plugin_dir_url( __FILE__ ) . 'img/santapress_layout_04.jpg" class="'.$santapress_selected_classes[3].'"></label></div>';
		echo '</div> <!-- /.santapress_layout_row -->';
		
	}

	public function santapress_doortype_callback( $arguments ) {
		$santapress_doortype = get_option('santapress_doortype');
		echo '<select name="santapress_doortype">';
		echo '<option value="popup"'.selected('popup', $santapress_doortype, false).' selected>Content should open in popup/modal window</option>';
		echo '<option value="regular" '.selected('regular', $santapress_doortype, false).'>Content should open in a regular window</option>';
		echo '</select>';
		echo '<p class="description santapress_desc_small">'.wp_kses( __('You can set a global door type which can be manually overridden later while editing each door.<br />Further options will also appear while editing each door.', 'santapress'), array('br' => array()) ).'</p>';
	}

	public function santapress_init_quiz_callback( $arguments ) {
		echo '<input type="checkbox" name="santapress_init_quiz" value="1" '.checked(1, get_option('santapress_init_quiz'), false). '/><br />';
		echo '<p class="description santapress_desc_small">'.esc_html__('Populate each door with sample trivia data instead filling the doors with sample quotes.', 'santapress').'</p>';
	}

	public function santapress_previewmode_callback( $arguments ) {
		echo '<input type="checkbox" name="santapress_previewmode" value="1" '.checked(1, get_option('santapress_previewmode'), false). '/><br>';
		echo '<p class="description santapress_desc_small">'.wp_kses( __('If this box is checked, all doors will be visible and clickable for preview.<br /><strong>Only for logged-in administrators</strong>.', 'santapress'), array('br' => array(), 'strong' => array()) ).'</p>';
	}

	public function santapress_modal_width_callback( $arguments ) {
		$santapress_modal_width = get_option('santapress_modal_width');
		echo '<input type="text" class="santapress_modal_width" name="santapress_modal_width" value="'.esc_attr($santapress_modal_width).'" />';
		echo '<p class="description santapress_desc_small">'.esc_html__('Modal width (max.) in px.', 'santapress').'</p>';
	}

	public function santapress_modal_autolayout_callback( $arguments ) {
		echo '<input type="checkbox" name="santapress_modal_autolayout" value="1" '.checked(1, get_option('santapress_modal_autolayout'), false). '/><br />';
		echo '<p class="description santapress_desc_small">'.esc_html__('Auto Layout for Modal (= the modal layout will adjust to the layout of the calendar).', 'santapress').'</p>';
	}

	public function santapress_modal_bgcolor_callback( $arguments ) {
		$santapress_modal_bgcolor = get_option('santapress_modal_bgcolor');
		echo '<input type="text" class="santapress_modal_bgcolor color-field" name="santapress_modal_bgcolor" value="'.esc_attr($santapress_modal_bgcolor).'" />';
		echo '<p class="description santapress_desc_small">'.esc_html__('Modal background color.', 'santapress').'</p>';
	}

	public function santapress_modal_bgcolor2_callback( $arguments ) {
		$santapress_modal_bgcolor2 = get_option('santapress_modal_bgcolor2');
		echo '<input type="text" class="santapress_modal_bgcolor2 color-field" name="santapress_modal_bgcolor2" value="'.esc_attr($santapress_modal_bgcolor2).'" />';
		echo '<p class="description santapress_desc_small">'.esc_html__('If a second color is chosen here, the background will be displayed as a radial gradient.', 'santapress').'</p>';
	}

	public function santapress_modal_showhat_callback( $arguments ) {
		echo '<input type="checkbox" name="santapress_modal_showhat" value="1" '.checked(1, get_option('santapress_modal_showhat'), false). '/><br />';
		echo '<p class="description santapress_desc_small">'.esc_html__('Show a Santa hat in modal.', 'santapress').'</p>';
	}

	public function santapress_modal_showpattern_callback( $arguments ) {
		echo '<input type="checkbox" name="santapress_modal_showpattern" value="1" '.checked(1, get_option('santapress_modal_showpattern'), false). '/><br />';
		echo '<p class="description santapress_desc_small">'.esc_html__('Show snow flakes pattern in modal.', 'santapress').'</p>';
	}

	public function santapress_showtitle_callback( $arguments ) {
		echo '<input type="checkbox" name="santapress_showtitle" value="1" '.checked(1, get_option('santapress_showtitle'), false). '/><br />';
		echo '<p class="description santapress_desc_small">'.esc_html__('Show post title within the modal and regular pages.', 'santapress').'</p>';
	}

	public function santapress_modal_mode_callback( $arguments ) {
		$santapress_modal_mode = get_option('santapress_modal_mode');
		echo '<select name="santapress_modal_mode">';
		echo '<option value="ajax"'.selected('ajax', $santapress_modal_mode, false).' selected>Ajax (works in most cases)</option>';
		echo '<option value="iframe" '.selected('iframe', $santapress_modal_mode, false).'>iFrame</option>';
		echo '</select>';
		echo '<p class="description santapress_desc_small">'.wp_kses( __('Ajax is the default and should work in most cases. If you experience problems with the modal contents or use custom plugins that are not officially supported yet (e.g. gallery, forms, quiz plugins etc.) you might want to try iFrame instead.', 'santapress'), array('br' => array()) ).'</p>';
	}

	public function santapress_enablesnow_callback( $arguments ) {
		
		$santapress_enablesnow = get_option('santapress_enablesnow');
		echo '<select name="santapress_enablesnow">';
		echo '<option value="">'.esc_html__('Disable snow', 'santapress').'</option>';
		//echo '<option value="everywhere" '.selected('everywhere', $santapress_enablesnow, false).'>'.esc_html__('Enable snow on the calendar page', 'santapress').'</option>';
		echo '<option value="modal" '.selected('modal', $santapress_enablesnow, false).'>'.esc_html__('Enable Snow', 'santapress').'</option>';
		echo '</select>';
		//echo '<p class="description santapress_desc_small">'.esc_html__('If you enable snow on the calendar page, snow will be displayed on the page where the shortcode is embedded.', 'santapress').'</p>';

	}

	public function santapress_shuffledoors_callback( $arguments ) {
		echo '<input type="checkbox" name="santapress_shuffledoors" value="1" '.checked(1, get_option('santapress_shuffledoors'), false). '/><br />';
		echo '<p class="description santapress_desc_small">'.esc_html__('Shuffle doors each time a user visits the page.', 'santapress').'</p>';
	}

	public function santapress_disablepastdoors_callback( $arguments ) {
		echo '<input type="checkbox" name="santapress_disablepastdoors" value="1" '.checked(1, get_option('santapress_disablepastdoors'), false). '/><br />';
		echo '<p class="description santapress_desc_small">'.esc_html__('Disable past doors.', 'santapress').'<br />'.esc_html__('This function will disable past doors in the calendar view and will block the content on regular door pages.', 'santapress').'</p>';
	}

	public function santapress_tracking_callback( $arguments ) {
		$santapress_tracking = get_option('santapress_tracking');
		echo '<select name="santapress_tracking">';
		echo '<option value="logged-out-users" '.selected('logged-out-users', $santapress_tracking, false).'>track only logged out users</option>';
		echo '<option value="all-users" '.selected('all-users', $santapress_tracking, false).'>track all users (except administrators)</option>';
		echo '<option value="disable-tracking" '.selected('disable-tracking', $santapress_tracking, false).'>disable click tracking</option>';
		echo '</select>';
	}

	public function santapress_customcss_callback( $arguments ) {
		$santapress_customcss = get_option('santapress_customcss');
		echo '<div class="sp-custom-css"><textarea rows="15" cols="80" class="code" name="santapress_customcss">'.esc_html($santapress_customcss).'</textarea>';
		echo '<div class="sp-code-example"><pre><code>/* santapress custom style examples */

.santapress-post.modal {
/* custom styles for modal window */
}

.santapress-post {
/* custom styles for door post */
}

.santapress-post.modal.post-1269 {
/* you can also target a specific day using the post id */
}</pre></code></div></div>';
	}

	public function santapress_resettracking_callback( $arguments ) {
		echo '<input type="checkbox" name="santapress_resettracking" value="1" '.checked(1, get_option('santapress_resettracking'), false). '/><br />';
		echo '<p class="description santapress_desc_small">'.wp_kses( __('<strong>Use with caution:</strong> If this box is checked, it will reset the tracking count on each door after you hit save.<br />It will only reset the tracking count - nothing else.', 'santapress'), array('strong' => array() )).'</p>';
	}

	/**
	 * Quiz Fields Callback function
	 *
	 * @since    1.3.0
	 */
	public function santapress_quizcookie_callback( $arguments ) {
		echo '<input type="checkbox" name="santapress_quizcookie" value="1" '.checked(1, get_option('santapress_quizcookie'), false). '/><br />';
		echo '<p class="description santapress_desc_small">'.wp_kses( __('If this box is checked, users will only have try/answer per day.<br />Otherwise they can try as much as they want.', 'santapress'), array('br' => array()) ).'</p>';
		echo '<div class="sp-gdrp-notice"><p class="description santapress_desc_small"><strong>'.esc_html__('GDPR Notice', 'santapress').':</strong> '.esc_html__('This function creates a cookie called "sp_quiz_cookie" - no personal information is stored whatsoever. You might want to make sure you have a Cookie Notice and explain the use of cookies in your data privacy page.', 'santapress').'</p></div>';
	}
	public function santapress_quizcookie_msg_callback( $arguments ) {
		$santapress_quizcookie_msg = get_option('santapress_quizcookie_msg');
		wp_editor( $santapress_quizcookie_msg, 'santapress_quizcookie_msg', array('textarea_rows' => 10, 'wpautop' => false, 'textarea_name' => 'santapress_quizcookie_msg') );
		echo '<p class="description sp-mw">'.esc_html__('Message to the user after the One-Try was made and the cookie was set.', 'santapress').'</p>';
	}
	public function santapress_quizcookie_type_callback( $arguments ) {
		$santapress_quizcookie_type = get_option('santapress_quizcookie_type');
		echo '<select name="santapress_quizcookie_type">';
		echo '<option value="after-an-answer" '.selected('after-an-answer', $santapress_quizcookie_type, false).'>After an answer</option>';
		echo '<option value="after-wpcf7-submission" '.selected('after-wpcf7-submission', $santapress_quizcookie_type, false).'>After the contact form was sent</option>';
		echo '</select>';
		echo '<p class="description sp-mw">';
		printf( wp_kses( __('The option "After the contact form was sent" requires to have a %s placed within the correct or incorrect content.', 'santapress'), array('a' => array('href', 'target'))), '<a href="https://de.wordpress.org/plugins/contact-form-7/" target="_blank">Contact form 7 Shortcode</a>' );
		echo '</p>';
	}
	public function santapress_quiztrue_callback( $arguments ) {
		$santapress_quiztrue = get_option('santapress_quiztrue');
		wp_editor( $santapress_quiztrue, 'santapress_quiztrue', array('textarea_rows' => 10, 'wpautop' => false, 'textarea_name' => 'santapress_quiztrue') );
		echo '<p class="description sp-mw">';
		printf( wp_kses( __('Insert content that should show up after the user has clicked on a correct answer.<br />You can also place your %s shortcode here.', 'santapress'), array('a' => array('href', 'target'))), '<a href="https://de.wordpress.org/plugins/contact-form-7/" target="_blank">Contact Form 7</a>');
		echo '<br />';
		printf( wp_kses( __('Please also see the %s for reference', 'santapress'), array( 'a' => array( 'href', 'target' ))), '<a href="https://santapress.halloecho.de/docs" target="_blank">docs</a>' );
		echo '</p>';
	}
	public function santapress_quizfalse_callback( $arguments ) {
		$santapress_quizfalse = get_option('santapress_quizfalse');
		wp_editor( $santapress_quizfalse, 'santapress_quizfalse', array('textarea_rows' => 10, 'wpautop' => false, 'textarea_name' => 'santapress_quizfalse') );
		echo '<p class="description sp-mw">'.esc_html__('Insert content that should show up after the user has clicked on a wrong answer.', 'santapress').'</p>';
	}

	/**
	 * Register Door CPT
	 *
	 * @since    1.0.0
	 */
    public function register_cpt() {

		if(!$this->check_doors())
			$icon = 'dashicons-star-empty';
		else
			$icon = 'dashicons-star-filled';

		/**
		 * CPT: Doors
		 */

		$labels = array(
			'name'                  => _x( 'Doors', 'Post Type General Name', 'santapress' ),
			'singular_name'         => _x( 'door', 'Post Type Singular Name', 'santapress' ),
			'menu_name'             => esc_html__( 'Doors', 'santapress' ),
			'name_admin_bar'        => esc_html__( 'Doors', 'santapress' ),
			'archives'              => esc_html__( 'Item Archives', 'santapress' ),
			'attributes'            => esc_html__( 'Item Attributes', 'santapress' ),
			'parent_item_colon'     => esc_html__( 'Parent Item:', 'santapress' ),
			'all_items'             => esc_html__( 'All Items', 'santapress' ),
			'add_new_item'          => esc_html__( 'Add New Item', 'santapress' ),
			'add_new'               => esc_html__( 'Add New', 'santapress' ),
			'new_item'              => esc_html__( 'New Item', 'santapress' ),
			'edit_item'             => esc_html__( 'Edit Item', 'santapress' ),
			'update_item'           => esc_html__( 'Update Item', 'santapress' ),
			'view_item'             => esc_html__( 'View Item', 'santapress' ),
			'view_items'            => esc_html__( 'View Items', 'santapress' ),
			'search_items'          => esc_html__( 'Search Item', 'santapress' ),
			'not_found'             => esc_html__( 'Not found', 'santapress' ),
			'not_found_in_trash'    => esc_html__( 'Not found in Trash', 'santapress' ),
			'featured_image'        => esc_html__( 'Featured Image', 'santapress' ),
			'set_featured_image'    => esc_html__( 'Set featured image', 'santapress' ),
			'remove_featured_image' => esc_html__( 'Remove featured image', 'santapress' ),
			'use_featured_image'    => esc_html__( 'Use as featured image', 'santapress' ),
			'insert_into_item'      => esc_html__( 'Insert into item', 'santapress' ),
			'uploaded_to_this_item' => esc_html__( 'Uploaded to this item', 'santapress' ),
			'items_list'            => esc_html__( 'Items list', 'santapress' ),
			'items_list_navigation' => esc_html__( 'Items list navigation', 'santapress' ),
			'filter_items_list'     => esc_html__( 'Filter items list', 'santapress' ),
		);

		$args = array(
			'label'                 => esc_html__( 'door', 'santapress' ),
			'description'           => esc_html__( 'Doors for SantaPress', 'santapress' ),
			'labels'                => $labels,
			'supports'              => array( 'title', 'editor', 'page-attributes' ),
			'hierarchical'          => false,
			'public'                => true,
			'show_ui'               => true,
			'show_in_menu'          => true,
			'menu_position'         => 50,
			'show_in_admin_bar'     => true,
			'show_in_nav_menus'     => true,
			'can_export'            => true,
			'has_archive'           => false,
			'exclude_from_search'   => false,
			'publicly_queryable'    => true,
			'show_in_rest' 			=> true, // adds Gutenberg support (since 1.6.3)
			'menu_icon'   			=> $icon,
			'capability_type'       => 'post',
		);

		register_post_type( 'door', $args );

		/**
		 * Taxonomy: Door Categories.
		 */

		$labels = array(
			"name" => __( "Door Categories", "santapress" ),
			"singular_name" => __( "Door Category", "santapress" ),
		);

		$args = array(
			"label" => __( "Door Categories", "santapress" ),
			"labels" => $labels,
			"public" => true,
			"publicly_queryable" => true,
			"hierarchical" => true,
			"show_ui" => true,
			"show_in_menu" => true,
			"show_in_nav_menus" => true,
			"query_var" => true,
			"rewrite" => [ 'slug' => 'doorcat', 'with_front' => true, ],
			"show_admin_column" => false,
			"show_in_rest" => true,
			"rest_base" => "doorcat",
			"rest_controller_class" => "WP_REST_Terms_Controller",
			"show_in_quick_edit" => false,
		);

		register_taxonomy( "doorcat", [ "door" ], $args );

    }

	/**
	 * Callback function for Layout page
	 *
	 * @since    1.0.0
	 */
	public function santapress_admin_layout() {

		if( ISSET($_GET['settings-updated']) && $_GET['settings-updated'] == true ) {

         	$santapress_layout = get_option('santapress_layout');

			$args = array(
				'posts_per_page'   => -1,
				'orderby'          => 'menu_order',
				'order'            => 'ASC',
				'post_type'        => 'door',
				'post_status'      => 'publish, future',
				'suppress_filters' => true
			);
			$doors = get_posts( $args );

			foreach($doors as $door) {

				$santapress_day = get_post_meta( $door->ID, 'santapress_day', true );

				if($santapress_day) {
					// Save door graphics
					$imgUrl = esc_url( plugins_url( '/public/img/'.$santapress_layout.'/'.$santapress_day.'.png' , dirname(__FILE__) ) );
					update_field('santapress_door_image', $imgUrl, $door->ID);
				}

			}

		}

		include 'partials/santapress-layout.php';

	}

	/**
	 * Callback function for Config page
	 *
	 * @since    1.0.0
	 */
	public function santapress_admin_config() {

		// reset tracking
		if( ISSET($_GET['settings-updated']) && $_GET['settings-updated'] == true ) {
			
			$santapress_resettracking = get_option('santapress_resettracking');
			
			if($santapress_resettracking) {
				
				$args = array(
					'posts_per_page'   => -1,
					'orderby'          => 'menu_order',
					'order'            => 'ASC',
					'post_type'        => 'door',
					'post_status'      => 'publish, future',
					'suppress_filters' => true
				);
				$doors = get_posts( $args );
				
				foreach($doors as $door) {
					delete_post_meta($door->ID, 'santapress_clicks');
				}

				update_option( 'santapress_resettracking', '0' ); 

			}
		}
		

		include 'partials/santapress-config.php';

	}

	/**
	 * Callback function for Quiz page
	 *
	 * @since    1.3.0
	 */
	public function santapress_admin_quiz() {
		

		include 'partials/santapress-quiz.php';

	}

	/**
	 * Callback function for Info page
	 * (currently not active)
	 *
	 * @since    1.0.0
	 */
	public function santapress_admin_info() {

		include 'partials/santapress-info.php';

	}

	/**
	 * Register Admin columns
	 *
	 * @since  1.0.0
	 */
	public function door_posts_columns( $columns ) {

		$columns = array(
		  'cb' => $columns['cb'],
		  'title' => esc_html__( 'Title' ),
		  'date' => esc_html__( 'Date' ),
		  'door_image' => esc_html__( 'Door Image', 'santapress' ),
		  'door_order' => esc_html__( 'Order' ),
		  'door_clicks' => esc_html__( 'Door Clicks', 'santapress' ),
		);

		return $columns;
	}

	/**
	 * Admin columns column functions
	 *
	 * @since  1.0.0
	 */
	public function door_column( $column, $post_id ) {
		// Image column
		if ( 'door_image' === $column ) {
			$santapress_door_image = get_field('santapress_door_image', $post_id);
			$santapress_custom_door_image = get_field('santapress_custom_door_image', $post_id);
			if($santapress_custom_door_image) {
				$santapress_door_image = $santapress_custom_door_image['url'];
			}
			// svg fix
			if(!$santapress_custom_door_image)
				$santapress_door_image = str_replace('.svg', '.png', $santapress_door_image);
			echo '<div style="width:50px"><img src="'.esc_url($santapress_door_image).'" style="max-width:100%;height:auto;" /></div>';
		}
		if ( 'door_order' === $column ) {
			echo get_post_field( 'menu_order', $post_id );
		}
		if ( 'door_clicks' === $column ) {
			$santapress_clicks = (int)get_post_meta( $post_id, 'santapress_clicks', true );
			if(!$santapress_clicks)
				echo '0';
			else
				echo $santapress_clicks;
		}
	}

	/**
	 * Admin columns sort colums
	 *
	 * @since  1.0.0
	 */
	public function santapress_sortable_columns( $columns ) {
	  $columns['door_order'] = 'price_per_month';
	  return $columns;
	}

	/**
	 * Admin columns default sort
	 *
	 * @since  1.0.0
	 */
	public function santapress_default_order( $query ){
	    
	    global $pagenow;
	    
	    if( is_admin()
	        && 'edit.php' == $pagenow
	        && !isset( $_GET['orderby'] )
	        && (ISSET( $_GET['post_type'] ) && $_GET['post_type'] == 'door') ) {
	            $query->set( 'orderby', 'menu_order' );
	            $query->set( 'order', 'ASC' );
	    }

	}

	/**
	 * Migrate and update obsolete acf fields prior to 1.6.1
	 *
	 * @since  1.6.1
	 */
	public function updateObsoletFields() {

		if( !get_option('santapress_updated_to_161') ) {

			$args = array(
				'posts_per_page'   => -1,
				'post_type'        => 'door',
				'suppress_filters' => false,
				'post_status' 	   => 'any'
			);

			$doors = get_posts( $args );

			if( $doors ) {

				foreach($doors as $door) {
					$custom = get_post_custom($door->ID);
					if( array_key_exists('santapress_quiz_answers', $custom) ) {
						if($custom['santapress_quiz_answers'][0] > 0) {
							$answerStr = '';
							for($i=0;$i<$custom['santapress_quiz_answers'][0];$i++) {
								$answer = $custom['santapress_quiz_answers_'.$i.'_santapress_quiz_answer'][0];
								$correct = $custom['santapress_quiz_answers_'.$i.'_santapress_quiz_answer_correct'][0];
								$answerStr .= $answer;
								$answerStr .= '|';
								$answerStr .= $correct ? 'true' : 'false';
								$answerStr .= "\n";
							}
						}
						update_post_meta($door->ID, 'santapress_quiz_answers_161', $answerStr);
					}
				}

				update_option('santapress_updated_to_161', true);

			}

		}

	}


}
