<?php
/**
 * The public-facing functionality of the plugin.
 *
 * @link       https://santapress.halloecho.de
 * @since      1.0.0
 *
 * @package    SantaPress
 * @subpackage SantaPress/public
 */

class SantaPress_Public {

	/**
	 * The ID of this plugin.
	 *
	 * @since    1.0.0
	 * @access   private
	 * @var      string    $plugin_name    The ID of this plugin.
	 */
	private $plugin_name;

	/**
	 * The version of this plugin.
	 *
	 * @since    1.0.0
	 * @access   private
	 * @var      string    $version    The current version of this plugin.
	 */
	private $version;

	/**
	 * Initialize the class and set its properties.
	 *
	 * @since    1.0.0
	 * @param      string    $plugin_name       The name of the plugin.
	 * @param      string    $version    The version of this plugin.
	 */
	public function __construct( $plugin_name, $version ) {

		$this->plugin_name = $plugin_name;
		$this->version = $version;

	}

	/**
	 * Register the stylesheets for the public-facing side of the site.
	 *
	 * @since    1.0.0
	 */
	public function enqueue_styles() {

		/**
		 * Enqueue Styles
		 *
		 */

		wp_enqueue_style( $this->plugin_name, plugin_dir_url( __FILE__ ) . 'css/santapress-public-all.min.css', array(), $this->version, 'all' );
		
		// Add custom css
		$santapress_customcss = get_option('santapress_customcss');
		if($santapress_customcss)
			wp_add_inline_style( $this->plugin_name, esc_html($santapress_customcss) );

		// Add more styles
		
		// Modal width
		$santapress_modal_width = get_option('santapress_modal_width');
		if( is_numeric($santapress_modal_width) )
			wp_add_inline_style( $this->plugin_name, '.santapress-post.modal, .sp-modal-iframe.modal { max-width: '.(int)$santapress_modal_width.'px !important }' );

		$santapress_modal_autolayout = get_option('santapress_modal_autolayout');

		// Modal background color
		$santapress_modal_bgcolor = get_option('santapress_modal_bgcolor');
		if( !$santapress_modal_autolayout && $santapress_modal_bgcolor )
			wp_add_inline_style( $this->plugin_name, '.santapress-post.modal { background-color: '.esc_html($santapress_modal_bgcolor).' !important }' );
		
		// Modal background color 2
		$santapress_modal_bgcolor2 = get_option('santapress_modal_bgcolor2');
		if( !$santapress_modal_autolayout && $santapress_modal_bgcolor && $santapress_modal_bgcolor2 ) {
			$_style = '.santapress-post.modal { 
background: '.$santapress_modal_bgcolor.' !important;
background: -moz-radial-gradient(center, ellipse cover, '.$santapress_modal_bgcolor.' 1%, '.$santapress_modal_bgcolor2.' 100%) !important; /* FF3.6-15 */
background: -webkit-radial-gradient(center, ellipse cover, '.$santapress_modal_bgcolor.' 1%,'.$santapress_modal_bgcolor2.' 100%) !important; /* Chrome10-25,Safari5.1-6 */
background: radial-gradient(ellipse at center, '.$santapress_modal_bgcolor.' 1%,'.$santapress_modal_bgcolor2.' 100%) !important; /* W3C, IE10+, FF16+, Chrome26+, Opera12+, Safari7+ */
filter: progid:DXImageTransform.Microsoft.gradient( startColorstr=\''.$santapress_modal_bgcolor.'\', endColorstr=\''.$santapress_modal_bgcolor2.'\',GradientType=1 ) !important; /* IE6-9 fallback on horizontal gradient */
}';
			wp_add_inline_style( $this->plugin_name, esc_html($_style) );
		}

		// Show santa hat
		$santapress_modal_showhat = get_option('santapress_modal_showhat');
		if( !$santapress_modal_showhat )
			wp_add_inline_style( $this->plugin_name, '.santapress-post.modal:before { display:none; }' );
		if( $santapress_modal_autolayout )
			wp_add_inline_style( $this->plugin_name, '.santapress-post.modal:before { display:none; }' );

		// Show snowflakes pattern
		$santapress_modal_showpattern = get_option('santapress_modal_showpattern');
		if( !$santapress_modal_showpattern )
			wp_add_inline_style( $this->plugin_name, '.santapress-post.modal:after { display:none; }' );
		if( $santapress_modal_autolayout )
			wp_add_inline_style( $this->plugin_name, '.santapress-post.modal:after { display:none; }' );

	}

	/**
	 * Register the JavaScript for the public-facing side of the site.
	 *
	 * @since    1.0.0
	 */
	public function enqueue_scripts() {

		/**
		 * Enqueue Scripts
		 *
		 */

		// Register script
		wp_register_script( $this->plugin_name, plugin_dir_url( __FILE__ ) . 'js/santapress-public-all.min.js', array( 'jquery' ), $this->version, true );

		// Localize the script with new data
		$translation_array = array(
			'plugin_public_dir_url' => plugin_dir_url( __FILE__ ),
			'ajaxurl' => admin_url( 'admin-ajax.php' )
		);
		wp_localize_script( $this->plugin_name, 'data', $translation_array );

		// Enqueued script with localized data.
		wp_enqueue_script( $this->plugin_name );

	}

	/**
	 * Register Shortcodes
	 *
	 * @since    1.0.0
	 */
	public function register_shortcodes() {
		add_shortcode( 'santapress', array( $this, 'santapress_shortcode' ) );
		add_shortcode( 'sp_icon', array( $this, 'sp_icon_shortcode' ) );
		add_shortcode( 'sp_quiz', array( $this, 'sp_quiz_shortcode' ) );
		add_shortcode( 'sp_quiz_back', array( $this, 'sp_quiz_back_shortcode' ) );
	}

	/**
	 * santapress Shortcode
	 *
	 * @since    1.0.0
	 */
	public static function santapress_shortcode( $atts, $content = ""  ) {

		// Shortcode will be used like this:
		// [santapress hiddenmode="true" timeout="800" title="It's Christmas time!" subtitle="We want to celebrate Christmas with our friends and clients. Here's a special gift for you. Have great holidays!" trigger_class=".show-calendar" show_builtin_trigger="true" hide_after_close="false"]

		$atts = shortcode_atts(
			array(
				'hiddenmode' => false,
				'timeout' => 800,
				'title' => '',
				'subtitle' => '',
				'trigger_class' => '.show-calendar',
				'show_builtin_trigger' => false,
				'hide_after_close' => false,
				'cat_id' => false,
				'cols' => false
			), $atts, 'santapress' );

		$args = array(
			'posts_per_page'   => -1,
			'orderby'          => 'menu_order',
			'order'            => 'ASC',
			'post_type'        => 'door',
			'post_status'      => 'publish, future',
			'suppress_filters' => false
		);

		// filter for custom taxonomy
		if( $atts['cat_id'] ) {
				$args['tax_query'] = array(
					array(
						'taxonomy' => 'doorcat',
						'field'    => 'term_id',
						'terms'    => $atts['cat_id'],
					),
				);
		}

		$doors = get_posts( $args );

		$returnStr = '';

		if(!empty($doors)) {

			// Preview mode
			$preview = false;
			$addPreviewClass = '';
			$santapress_previewmode = get_option('santapress_previewmode');
			if( current_user_can( 'manage_options' ) && $santapress_previewmode ) {
				$preview = true;
				$addPreviewClass = ' preview';
			}

			// Shuffle doors
			$santapress_shuffledoors = get_option('santapress_shuffledoors');
			if($santapress_shuffledoors) {
				shuffle($doors);
			}

			// Disable past doors
			$santapress_disablepastdoors = get_option('santapress_disablepastdoors');

			// Auto layout
			$santapress_modal_autolayout = get_option('santapress_modal_autolayout');

			// Snowtype
			$santapress_snowtype = 'santapress';

			// Enable snow
			$addSnowClass = '';
			$dataSnow = '';
			$dataSnowtype = '';
			$santapress_enablesnow = get_option('santapress_enablesnow');
			if($santapress_enablesnow) {
				$addSnowClass = ' with-snow';
				$dataSnowtype = ' data-snowcond="'.esc_js($santapress_enablesnow).'" data-snowtype="'.esc_js($santapress_snowtype).'"';
			}

			// Click Tracking
			$addTrackingClass = '';
			$santapress_tracking = get_option('santapress_tracking');
			if( $santapress_tracking == 'logged-out-users' && !is_user_logged_in() ) { // track only logged out users 
				$addTrackingClass = ' tr';
			}
			if( $santapress_tracking == 'all-users' && !current_user_can( 'manage_options' ) ) { // track all users except admins 
				$addTrackingClass = ' tr';
			}

			// Hidden Mode
			$addHiddenModeClass = '';
			$addDataAttributes = '';
			if($atts['hiddenmode']) {
				$addHiddenModeClass = ' hiddenmode';
				$addDataAttributes = ' data-timeout='.esc_js($atts['timeout']).'';
				if($atts['trigger_class'])
					$addDataAttributes .= ' data-trigger='.esc_js($atts['trigger_class']).'';
				if($atts['hide_after_close'])
					$addDataAttributes .= ' data-hideafterclose='.esc_js($atts['hide_after_close']).'';
			}

			// Cols (since 1.5.0)
			$colsClass = '';
			if($atts['cols']) {
				if(is_numeric($atts['cols'])) {
					$colsClass = ' cols-'.$atts['cols'];
				}
			}

			$returnStr .= '<div class="santapress-container w-'.count($doors).'-doors'.esc_attr($addSnowClass).''.esc_attr($addTrackingClass).''.esc_attr($addHiddenModeClass).''.esc_attr($colsClass).'"'.esc_js($addDataAttributes).''.$dataSnowtype.'>';

			if($atts['hiddenmode'] && $atts['title'])
				$returnStr .= '<h1 class="santapress-title">'.esc_html($atts['title']).'</h1>';

			if($atts['hiddenmode'] && $atts['subtitle'])
				$returnStr .= '<h2 class="santapress-subtitle">'.esc_html($atts['subtitle']).'</h2>';

			if($atts['hiddenmode'])
				$returnStr .= '<a href="javascript:void(0);" id="sp-close-hidden-mode"></a>';

			if($atts['show_builtin_trigger']) {
				$_tc = str_replace('.', '', $atts['trigger_class']);
				$returnStr .= '<div id="santapress-trigger"><span class="'.esc_attr($_tc).'"></span></div>';
			}

			// modal mode (since 1.4.2)
			$santapress_modal_mode = get_option('santapress_modal_mode');

			foreach($doors as $door) {
				
				$link = get_permalink($door);
				
				$santapress_door_image = get_field('santapress_door_image', $door);
				$santapress_custom_door_image = get_field('santapress_custom_door_image', $door);
				$santapress_door_type = get_field('santapress_door_type', $door);

				$addClass = '';
				$addTarget = '';
				if($santapress_door_type == 'global') {
					$santapress_door_type = get_option('santapress_doortype');
				}
				if($santapress_door_type == 'popup') {
					$addClass = 'santapress-modal';
				}
				if($santapress_door_type == 'wc') {
					$wc_product = get_field('santapress_door_type_wc', $door);
					$link = get_permalink($wc_product);
				}
				if($santapress_door_type == 'external') {
					$addTarget = ' target="_blank"';
					$link = get_field('santapress_door_type_link_url', $door);
				}

				// custom door image
				if($santapress_custom_door_image) {
					$santapress_door_image = $santapress_custom_door_image['url'];
				}

				// disable link for future posts
				if( $door->post_status == 'future' && !$preview) {
					$link = '#';
				}

				// add class to past doors
				$addPastOrCurrentClass = '';
				$doorDateTime = DateTime::createFromFormat( 'Y-m-d H:i:s', get_the_date('Y-m-d', $door->ID) . '00:00:00');
				$todayDateTime = DateTime::createFromFormat( 'Y-m-d H:i:s', current_time('Y-m-d') . '00:00:00' );
				if($doorDateTime < $todayDateTime) {
					$addPastOrCurrentClass = ' door-past';
				}
				if($doorDateTime == $todayDateTime) {
					$addPastOrCurrentClass = ' door-current';
				}

				// disable past doors
				if($santapress_disablepastdoors) {
					if( get_the_date('Y-m-d', $door->ID) != current_time('Y-m-d') ) {
						$door->post_status = 'future';
						if(!$preview)
							$link = '#';
					}
				}

				// get door layout number
				$door_layout = 0;
				$santapress_custom_door_image = get_field('santapress_custom_door_image', $door);
				if(!$santapress_custom_door_image) {
					$santapress_door_image = get_field('santapress_door_image', $door);
					preg_match('/(\d{2})/', $santapress_door_image, $matches, PREG_OFFSET_CAPTURE);
					if($matches[0]) {
						$door_layout = $matches[0][0];
					}
				}
				else {
					$santapress_layout = get_option('santapress_layout');
					$door_layout = $santapress_layout;
				}
				if(!$santapress_modal_autolayout)
					$door_layout = 'man';

				// svg fix
				if(!$santapress_custom_door_image) {
					$santapress_door_image = str_replace('.svg', '.png', $santapress_door_image);
				}

				// modal mode (since 1.4.2)
				if($santapress_modal_mode == 'iframe') {
					if(strpos($link, '?') !== false)
						$link = $link . '&embed';
					else
						$link = $link . '?embed';
				}

				$returnStr .= '<div class="santapress-door status-'.esc_attr($door->post_status).''.esc_attr($addPreviewClass).''.esc_attr($addPastOrCurrentClass).' santapress-layout-'.esc_attr($door_layout).'" data-postid="'.esc_js($door->ID).'"><div class="santapress-door-inner"><a href="'.esc_url($link).'" class="'.esc_attr($addClass).'"'.esc_attr($addTarget).' data-e-disable-page-transition="true">';
				$returnStr .= '<img src="'.esc_url($santapress_door_image).'" alt="door-'.esc_attr($door->post_title).'" />';
				$returnStr .= '</a></div></div>';

			}

			// attach nonce for ajax calls
			$nonce = wp_create_nonce("santapress_nonce");
			$returnStr .= '<input type="hidden" name="sp-nonce" id="sp-nonce" value="'.$nonce.'">';

			$returnStr .= '</div> <!-- /.santapress-container -->';

		}

		return $returnStr;

	}

	/**
	 * sp_icon Shortcode
	 * Shortcode for the SantaPress Icon
	 *
	 * @since    1.2.0
	 */
	public static function sp_icon_shortcode( $atts, $content = ""  ) {

		global $post;

		// for ajax calls, since 1.4.2
		if( ISSET($_GET['postid']) ) {
			$postid = $_GET['postid'];
			if( !is_numeric($postid) )
				return;
			$post = get_post($postid);
		}

		// get door layout number
		$door_layout = 0;
		$santapress_custom_door_image = get_field('santapress_custom_door_image', $post);
		if(!$santapress_custom_door_image) {
			$santapress_door_image = get_field('santapress_door_image', $post);
			preg_match('/public\/img\/(\d{2})/', $santapress_door_image, $matches, PREG_OFFSET_CAPTURE);
			if($matches[0]) {
				$door_layout = $matches[1][0];
				return '<img src="'.esc_url(plugin_dir_url( __FILE__ ) . 'img/icons/sp_icon_'.$door_layout.'.png').'" width="100" height="100" alt="icon_'.esc_attr($door_layout).'" class="sp_icon" />';
			}
		}
		else {
			return false; // false will be returned if user has set a custom door image
		}

	}

	/**
	 * sp_quiz Shortcode
	 * Shortcode for the SantaPress Quiz
	 *
	 * @since    1.3.0
	 */
	public static function sp_quiz_shortcode( $atts, $content = "" ) {

		global $post;

		// for ajax calls, since 1.4.2
		if( ISSET($_GET['postid']) ) {
			$postid = $_GET['postid'];
			if( !is_numeric($postid) )
				return;
			$post = get_post($postid);
		}

		// Add Quiz
		if( get_field('santapress_activate_quiz', $post->ID) ) {
			
			$santapress_quizcookie = get_option('santapress_quizcookie');

			$santapress_quizcookie_type = get_option('santapress_quizcookie_type');
			$santapress_quizcookie_type_as_number = false;
			if($santapress_quizcookie) {
				if($santapress_quizcookie_type == 'after-an-answer') {
					$santapress_quizcookie_type_as_number = 1;
				}
				else {
					$santapress_quizcookie_type_as_number = 2;
				}
			}

			$return = '<div class="spquiz" data-postid="'.esc_js($post->ID).'" data-qc="'.esc_js($santapress_quizcookie).'" data-qctype="'.esc_js($santapress_quizcookie_type_as_number).'">';

			$question = get_field('santapress_quiz_question', $post->ID);
			$return .= '<h2 class="spquiz-question">'.esc_html($question).'</h2>';

			if( $santapress_quiz_answers = get_field('santapress_quiz_answers_161', $post->ID) ) { // since 1.6.1

				$return .= '<div class="spquiz-answers">';

				$santapress_quiz_answers = explode("\n", $santapress_quiz_answers);

				foreach($santapress_quiz_answers as $answer) {
					if (strpos($answer, '|') !== false) {
						$_answer = explode('|', $answer);
						$_answer[1] = filter_var($_answer[1], FILTER_VALIDATE_BOOLEAN);
						$return .= '<div class="spquiz-answer">'.esc_html($_answer[0]).'<span class="spquiz-loader"></span></div>';
					}
				}

				$return .= '</div> <!-- /.spquiz-answers -->';
				$return .= '<div class="spquiz-result"></div> <!-- /.spquiz-result -->';

			}

			$return .= '</div> <!-- /.spquiz -->';

			return $return;

		}

		return;

	}

	/**
	 * sp_quiz_back Shortcode
	 * Shortcode for the SantaPress Quiz
	 *
	 * @since    1.3.0
	 */
	public static function sp_quiz_back_shortcode( $atts, $content = ""  ) {
		
		$atts = shortcode_atts(
			array(
				'message' => 'Go back'
			), $atts, 'sp_quiz_back' );

		return '<p><span class="spquiz-back">'.esc_html($atts['message']).'</span></p>';

	}

	/**
	 * Content Filter for single door posts
	 *
	 * @since    1.0.0
	 */
	public function wrap_div_around_door_post( $content, $ajax_request_post_id = false ) {

		// only apply content filter if it's single door post or this function was called manually after an ajax request (see ajax_sp_fetch_door())
		if (is_singular('door') || ($ajax_request_post_id && ISSET($_GET['action'])) ) {

			if( $ajax_request_post_id && is_numeric($ajax_request_post_id) ) {
				$doorPost = get_post($ajax_request_post_id);
			}
			else {
				$doorPost = get_post(get_the_ID());
			}
			
			// show/hide title
			$santapress_showtitle = get_option('santapress_showtitle', $doorPost);
			if($santapress_showtitle) {
				$content = '<h1 class="santapress-title">'.esc_html($doorPost->post_title).'</h1>' . $content;
			}

			// attach custom javascript to post
			$santapress_custom_javascript = get_field('santapress_custom_javascript', $doorPost);
			$addScript = '';
			if($santapress_custom_javascript) {
				$addScript = $santapress_custom_javascript;
			}

			// get door layout number
			$door_layout = 0;
			$santapress_custom_door_image = get_field('santapress_custom_door_image', $doorPost);
			if(!$santapress_custom_door_image) {
				$santapress_door_image = get_field('santapress_door_image', $doorPost);
				preg_match('/public\/img\/(\d{2})/', $santapress_door_image, $matches, PREG_OFFSET_CAPTURE);
				if($matches[0]) {
					$door_layout = $matches[1][0];
				}
			}
			else {
				$santapress_layout = get_option('santapress_layout');
				$door_layout = $santapress_layout;
			}
			if(!get_option('santapress_modal_autolayout'))
				$door_layout = 'man';

			// Add Quiz
			if( get_field('santapress_activate_quiz', $doorPost->ID) ) {
				// append quiz only if the shortcode is not inserted within the content
				if ( !preg_match('/\[sp_quiz\]/', $doorPost->post_content ) ) {
					$quiz_html = do_shortcode('[sp_quiz]');
					$content .= $quiz_html;
				}
			}

			// Disable past doors
			$santapress_disablepastdoors = get_option('santapress_disablepastdoors');
			if($santapress_disablepastdoors) {
				$preview = false;
				if( current_user_can( 'manage_options' ) && get_option('santapress_previewmode') )
					$preview = true;
				if( ( get_the_date('Y-m-d', $doorPost->ID) != current_time('Y-m-d') ) && !$preview ) {
					$content = esc_html__('Sorry, Santa has disabled this door.', 'santapress');
				}
			}
			// modal mode (since 1.4.2) 
			$santapress_modal_mode = get_option('santapress_modal_mode');
			$addModalClass = '';
			if( $santapress_modal_mode == 'iframe' && ISSET($_GET['embed']) ) {
				$addModalClass = ' modal';
			}

			// wrap div around content and add $content to it
			$custom_content = '<div class="santapress-post santapress-layout-'.esc_attr($door_layout).' post-'.$doorPost->ID.''.esc_attr($addModalClass).'"><div class="santapress-post-inner">'.$content.''.$addScript.'</div></div>';

			if( is_singular( 'door' ) || ($santapress_modal_mode == 'iframe' && ISSET($_GET['embed'])) ) {
				// attach nonce for ajax calls
				$nonce = wp_create_nonce("santapress_nonce");
				echo '<input type="hidden" name="sp-nonce" id="sp-nonce" value="'.$nonce.'">'; // TODO:: echo should return
			}
			
		} else {
			$custom_content = $content;
		}
		return $custom_content;

	}

	/**
	 * Remove post titles for single door posts
	 *
	 * @since    1.0.0
	 */
	public function remove_post_titles( $title ) {
		if( is_singular( 'door' ) && in_the_loop() ):
			return '';
		else:
			return $title;
		endif;
	}

	/**
	 * Ajax Function to fetch door posts via wp-admin/admin-ajax.php
	 *
	 * @since    1.5.0
	 */
	public function ajax_sp_fetch_door() {

		$postid = $_GET['postid'];
		if( !is_numeric($postid) )
			return;

		// check nonce
		if ( !wp_verify_nonce( $_REQUEST['nonce'], "santapress_nonce") ) {
		 	exit("Santa's security guards blocked this request");
		}

		$post = get_post($postid);

		if( $post->post_status == 'publish' || (current_user_can( 'manage_options' ) && get_option('santapress_previewmode')) ) {

			// click tracking
			$santapress_tracking = get_option('santapress_tracking');
			if( ($santapress_tracking == 'logged-out-users' && !is_user_logged_in()) ||  
				($santapress_tracking == 'all-users' && !current_user_can( 'manage_options' )) ) {
				$this->clickTrack($postid);
			}
		
			$content = apply_filters('the_content', $post->post_content);
			$returnStr = $this->wrap_div_around_door_post($content, $postid);

			echo trim($returnStr);
			
		}
		else {
			$content = esc_html__('Sorry, Santa has disabled this door.', 'santapress');
			echo $content;
		}
		die();

	}

	/**
	 * Click Tracking
	 *
	 * @since    1.5.0 (rebuild)
	 */
	public function clickTrack($post_id) {

		$santapress_clicks = (int)get_post_meta( $post_id, 'santapress_clicks', true );

		if(!$santapress_clicks) {
			add_post_meta( $post_id, 'santapress_clicks', 1, true );
		}
		else {
			$santapress_clicks++;
			update_post_meta( $post_id, 'santapress_clicks', $santapress_clicks );
		}

		return true;

	}

	/**
	 * Ajax Function to get quiz answer via wp-admin/admin-ajax.php
	 *
	 * @since    1.5.0
	 */
	public function ajax_sp_quiz_answer() {

		$post_id = $_GET['postid'];
		if( !is_numeric($post_id) )
			return;

		// check nonce
		if ( !wp_verify_nonce( $_REQUEST['nonce'], "santapress_nonce") ) {
		 	exit("No naughty business please");
		}

		// since 1.6.1
		$userAnswer = urldecode($_GET['answer']); 
		$userAnswer = stripslashes($userAnswer);

		$santapress_quiz_answers = get_field('santapress_quiz_answers_161', $post_id);
		$santapress_quiz_answers = explode("\n", $santapress_quiz_answers);

		foreach($santapress_quiz_answers as $answer) {
			if (strpos($answer, '|') !== false) {
				$_answer = explode('|', $answer);
				$_answer[1] = filter_var($_answer[1], FILTER_VALIDATE_BOOLEAN);
				if($userAnswer == $_answer[0]) {
					$sp_quiz_cookie_arr = [];
					if(ISSET($_COOKIE['sp_quiz_cookie'])) {
						$sp_quiz_cookie = $_COOKIE['sp_quiz_cookie'];
						if($sp_quiz_cookie) {
							$sp_quiz_cookie_arr = json_decode($sp_quiz_cookie);
						}
					}
					if( get_option('santapress_quizcookie') && ISSET($_COOKIE['sp_quiz_cookie']) && in_array($post_id, $sp_quiz_cookie_arr) ) {
						echo do_shortcode(get_option('santapress_quizcookie_msg'));
					}
					else {
						if($_answer[1]) {
							$santapress_quiztrue = get_field('santapress_correct_response', $post_id);
							if($santapress_quiztrue == '')
								$santapress_quiztrue = get_option('santapress_quiztrue');
							echo do_shortcode($santapress_quiztrue);
						}
						else {
							$santapress_quizfalse = get_field('santapress_incorrect_response', $post_id);
							if($santapress_quizfalse == '')
								$santapress_quizfalse = get_option('santapress_quizfalse');
							echo do_shortcode($santapress_quizfalse);
						}
					}
				}
				
			}
		}
		
		die();

	}

	/**
	 * Include single door template if door type is popup and embed mode is iframe
	 *
	 * @since    1.4.2
	 */
	public function sp_template_include($template){
	    global $post;
	    if($post) {
		    $santapress_door_type = get_field('santapress_door_type', $post->ID);
		    if($santapress_door_type == 'global') {
		    	$santapress_door_type = get_option('santapress_doortype');
		    }
		    if( is_singular('door') && $santapress_door_type == 'popup' && ISSET($_GET['embed']) ) {
		    	$pathToTemplate = plugin_dir_path( dirname( __FILE__ ) ) . 'templates/single-door-embed.php';
		    	$template = $pathToTemplate;
		    }
	    }
	    return $template;
	}

	/**
	 * Click tracking for iFrames
	 *
	 * @since    1.5.0
	 */
	public function ajax_sp_track_iframe() {

		$post_id = $_GET['postid'];
		if( !is_numeric($post_id) )
			return;

		// check nonce
		if ( !wp_verify_nonce( $_REQUEST['nonce'], "santapress_nonce") ) {
		 	exit("Santa's security guards blocked this request");
		}

		$santapress_tracking = get_option('santapress_tracking');
		if( ($santapress_tracking == 'logged-out-users' && !is_user_logged_in()) ||  
			($santapress_tracking == 'all-users' && !current_user_can( 'manage_options' )) ) {
			$this->clickTrack($post_id);
		}
		
		die();
		
	}

}
