<?php
/**
 * The file that defines the core plugin class
 *
 * A class definition that includes attributes and functions used across both the
 * public-facing side of the site and the admin area.
 *
 * @link       https://santapress.halloecho.de
 * @since      1.0.0
 *
 * @package    SantaPress
 * @subpackage SantaPress/includes
 */

class SantaPress {

	/**
	 * The loader that's responsible for maintaining and registering all hooks that power
	 * the plugin.
	 *
	 * @since    1.0.0
	 * @access   protected
	 * @var      SantaPress_Loader    $loader    Maintains and registers all hooks for the plugin.
	 */
	protected $loader;

	/**
	 * The unique identifier of this plugin.
	 *
	 * @since    1.0.0
	 * @access   protected
	 * @var      string    $plugin_name    The string used to uniquely identify this plugin.
	 */
	protected $plugin_name;

	/**
	 * The current version of the plugin.
	 *
	 * @since    1.0.0
	 * @access   protected
	 * @var      string    $version    The current version of the plugin.
	 */
	protected $version;

	/**
	 * Define the core functionality of the plugin.
	 *
	 * Set the plugin name and the plugin version that can be used throughout the plugin.
	 * Load the dependencies, define the locale, and set the hooks for the admin area and
	 * the public-facing side of the site.
	 *
	 * @since    1.0.0
	 */
	public function __construct() {
		if ( defined( 'SANTAPRESS_VERSION' ) ) {
			$this->version = SANTAPRESS_VERSION;
		} else {
			$this->version = '1.0.0';
		}
		$this->plugin_name = 'santapress';

		$this->load_dependencies();
		$this->set_locale();
		$this->define_admin_hooks();
		$this->define_public_hooks();

	}

	/**
	 * Load the required dependencies for this plugin.
	 *
	 * Include the following files that make up the plugin:
	 *
	 * - SantaPress_Loader. Orchestrates the hooks of the plugin.
	 * - SantaPress_i18n. Defines internationalization functionality.
	 * - SantaPress_Admin. Defines all hooks for the admin area.
	 * - SantaPress_Public. Defines all hooks for the public side of the site.
	 *
	 * Create an instance of the loader which will be used to register the hooks
	 * with WordPress.
	 *
	 * @since    1.0.0
	 * @access   private
	 */
	private function load_dependencies() {

		/**
		 * The class responsible for orchestrating the actions and filters of the
		 * core plugin.
		 */
		require_once plugin_dir_path( dirname( __FILE__ ) ) . 'includes/class-santapress-loader.php';

		/**
		 * The class responsible for defining internationalization functionality
		 * of the plugin.
		 */
		require_once plugin_dir_path( dirname( __FILE__ ) ) . 'includes/class-santapress-i18n.php';

		/**
		 * The class responsible for defining all actions that occur in the admin area.
		 */
		require_once plugin_dir_path( dirname( __FILE__ ) ) . 'admin/class-santapress-admin.php';

		/**
		 * The class responsible for defining all actions that occur in the public-facing
		 * side of the site.
		 */
		require_once plugin_dir_path( dirname( __FILE__ ) ) . 'public/class-santapress-public.php';

		$this->loader = new SantaPress_Loader();

	}

	/**
	 * Define the locale for this plugin for internationalization.
	 *
	 * Uses the SantaPress_i18n class in order to set the domain and to register the hook
	 * with WordPress.
	 *
	 * @since    1.0.0
	 * @access   private
	 */
	private function set_locale() {

		$plugin_i18n = new SantaPress_i18n();

		$this->loader->add_action( 'plugins_loaded', $plugin_i18n, 'load_plugin_textdomain' );

	}

	/**
	 * Register all of the hooks related to the admin area functionality
	 * of the plugin.
	 *
	 * @since    1.0.0
	 * @access   private
	 */
	private function define_admin_hooks() {

		$plugin_admin = new SantaPress_Admin( $this->get_plugin_name(), $this->get_version() );

		// load ACF
		if ( !class_exists('acf') ) { // Detect ACF plugin. Include if not present.
			// include ACF
			include_once( plugin_dir_path( dirname( __FILE__ ) ) . 'includes/vendor/acf/acf.php' );
			// set ACF path and dir
			$this->loader->add_filter('acf/settings/url', $plugin_admin, 'acf_settings_url');
			$this->loader->add_filter('acf/settings/path', $plugin_admin, 'acf_settings_path');
			$this->loader->add_filter('acf/settings/dir', $plugin_admin, 'acf_settings_dir');
			// hide ACF
			$this->loader->add_filter('acf/settings/show_admin', $plugin_admin, 'acf_show_admin');
			// stop ACF from hiding custom fields for other post types
			$this->loader->add_filter('acf/settings/remove_wp_meta_box', $plugin_admin, 'acf_show_admin');
		}
		else {
			// ACF is already activated on site
		}
		// setup ACF fields
		$plugin_admin->setup_acf_fields();

		// enqueue styles and scripts
		$this->loader->add_action( 'admin_enqueue_scripts', $plugin_admin, 'enqueue_styles' );
		$this->loader->add_action( 'admin_enqueue_scripts', $plugin_admin, 'enqueue_scripts' );

		// admin actions
		$this->loader->add_action('admin_head', $plugin_admin, 'add_admin_css');
		$this->loader->add_action( 'admin_menu', $plugin_admin, 'add_admin_menu' );
		$this->loader->add_action( 'admin_init', $plugin_admin, 'setup_sections' );
		$this->loader->add_action( 'admin_init', $plugin_admin, 'setup_fields' );
		$this->loader->add_action( 'init', $plugin_admin, 'register_cpt' );
		$this->loader->add_filter( 'manage_door_posts_columns', $plugin_admin, 'door_posts_columns' ); // register columns
		$this->loader->add_action( 'manage_door_posts_custom_column', $plugin_admin, 'door_column', 10, 2); // column functions
		$this->loader->add_filter( 'manage_edit-door_sortable_columns', $plugin_admin, 'santapress_sortable_columns'); // sort columns
		$this->loader->add_action( 'pre_get_posts', $plugin_admin, 'santapress_default_order' ); // set default order

	}

	/**
	 * Register all of the hooks related to the public-facing functionality
	 * of the plugin.
	 *
	 * @since    1.0.0
	 * @access   private
	 */
	private function define_public_hooks() {

		$plugin_public = new SantaPress_Public( $this->get_plugin_name(), $this->get_version() );

		// public actions
		$this->loader->add_action( 'wp_enqueue_scripts', $plugin_public, 'enqueue_styles' );
		$this->loader->add_action( 'wp_enqueue_scripts', $plugin_public, 'enqueue_scripts' );
		$this->loader->add_filter( 'the_content', $plugin_public, 'wrap_div_around_door_post' );
		$this->loader->add_filter( 'the_title', $plugin_public, 'remove_post_titles', 10, 2 );

		// register shortcodes
		$plugin_public->register_shortcodes();

		// iframe template, click tracking for iframes
		$this->loader->add_filter('template_include', $plugin_public, 'sp_template_include', 99);
		$this->loader->add_action( 'wp_ajax_nopriv_sp_track_iframe', $plugin_public, 'ajax_sp_track_iframe' );
		$this->loader->add_action( 'wp_ajax_sp_track_iframe', $plugin_public, 'ajax_sp_track_iframe' );

		// ajax calls (since 1.5.0)
		$this->loader->add_action( 'wp_ajax_nopriv_sp_fetch_door', $plugin_public, 'ajax_sp_fetch_door' );
		$this->loader->add_action( 'wp_ajax_sp_fetch_door', $plugin_public, 'ajax_sp_fetch_door' );
		$this->loader->add_action( 'wp_ajax_nopriv_sp_quiz_answer', $plugin_public, 'ajax_sp_quiz_answer' );
		$this->loader->add_action( 'wp_ajax_sp_quiz_answer', $plugin_public, 'ajax_sp_quiz_answer' );

	}

	/**
	 * Run the loader to execute all of the hooks with WordPress.
	 *
	 * @since    1.0.0
	 */
	public function run() {
		$this->loader->run();
	}

	/**
	 * The name of the plugin used to uniquely identify it within the context of
	 * WordPress and to define internationalization functionality.
	 *
	 * @since     1.0.0
	 * @return    string    The name of the plugin.
	 */
	public function get_plugin_name() {
		return $this->plugin_name;
	}

	/**
	 * The reference to the class that orchestrates the hooks with the plugin.
	 *
	 * @since     1.0.0
	 * @return    SantaPress_Loader    Orchestrates the hooks of the plugin.
	 */
	public function get_loader() {
		return $this->loader;
	}

	/**
	 * Retrieve the version number of the plugin.
	 *
	 * @since     1.0.0
	 * @return    string    The version number of the plugin.
	 */
	public function get_version() {
		return $this->version;
	}


}
