<?php
if ( ! defined( 'ABSPATH' ) ) {
	die( '-1' );
}

/**
 *
 * Class Vc_IconPicker - nectar addition - whole file
 * @since 4.4
 * See example usage in shortcode 'vc_icon'
 *
 *      `` example
 *        array(
 *            'type' => 'iconpicker',
 *            'heading' => esc_html__( 'Icon', 'js_composer' ),
 *            'param_name' => 'icon_fontawesome',
 *            'settings' => array(
 *                'emptyIcon' => false, // default true, display an "EMPTY"
 *     icon? - if false it will display first icon from set as default.
 *                'iconsPerPage' => 200, // default 100, how many icons
 *     per/page to display
 *            ),
 *            'dependency' => array(
 *                'element' => 'type',
 *                'value' => 'fontawesome',
 *            ),
 *        ),
 * vc_filter: vc_iconpicker-type-{your_icon_font_name} - filter to add new icon
 *     font type. see example for vc_iconpicker-type-fontawesome in bottom of
 *     this file Also // SEE HOOKS FOLDER FOR FONTS REGISTERING/ENQUEUE IN BASE
 * @path "/include/autoload/hook-vc-iconpicker-param.php"
 */
class Vc_IconPicker {
	/**
	 * @since 4.4
	 * @var array - save current param data array from vc_map
	 */
	protected $settings;
	/**
	 * @since 4.4
	 * @var string - save a current field value
	 */
	protected $value;
	/**
	 * @since 4.4
	 * @var array - optional, can be used as self source from self array., you
	 *     can pass it also with filter see Vc_IconPicker::setDefaults
	 */
	protected $source = [];

	/**

	 * @since 4.4
	 *
	 * @param $settings - param field data array
	 * @param $value - param field value
	 */
	public function __construct( $settings, $value ) {
		$this->settings = $settings;
		$this->setDefaults();

		$this->value = $value; // param field value
	}

	/**
	 * Set default function will extend current settings with defaults
	 * It can be used in Vc_IconPicker::render, but also it is passed to input
	 * field and was hooked in composer-atts.js file See vc.atts.iconpicker in
	 * wp-content/plugins/js_composer/assets/js/params/composer-atts.js init
	 * method
	 *  - it initializes javascript logic, you can provide ANY default param to
	 * it with 'settings' key
	 * @since 4.4
	 */
	protected function setDefaults() {
		if ( ! isset( $this->settings['settings'], $this->settings['settings']['type'] ) ) {
			$this->settings['settings']['type'] = 'fontawesome'; // Default type for icons
		}

		// More about this you can read in http://codeb.it/fonticonpicker/
		if ( ! isset( $this->settings['settings'], $this->settings['settings']['hasSearch'] ) ) {
			// Whether or not to show the search bar.
			$this->settings['settings']['hasSearch'] = true;
		}
		if ( ! isset( $this->settings['settings'], $this->settings['settings']['emptyIcon'] ) ) {
			// Whether or not empty icon should be shown on the icon picker
			$this->settings['settings']['emptyIcon'] = true;
		}
		if ( ! isset( $this->settings['settings'], $this->settings['settings']['allCategoryText'] ) ) {
			// If categorized then use this option
			$this->settings['settings']['allCategoryText'] = esc_html__( 'From all categories', 'js_composer' );
		}
		if ( ! isset( $this->settings['settings'], $this->settings['settings']['unCategorizedText'] ) ) {
			// If categorized then use this option
			$this->settings['settings']['unCategorizedText'] = esc_html__( 'Uncategorized', 'js_composer' );
		}

		/**
		 * Source for icons, can be passed via "mapping" or with filter vc_iconpicker-type-{your_type} (default fontawesome)
		 * vc_filter: vc_iconpicker-type-{your_type} (default fontawesome)
		 */
		if ( isset( $this->settings['settings'], $this->settings['settings']['source'] ) ) {
			$this->source = $this->settings['settings']['source'];
			unset( $this->settings['settings']['source'] ); // We don't need this on frontend.(js)
		}
	}

	/**
	 * Render edit form field type 'iconpicker' with selected settings and
	 * provided value. It uses javascript file vc-icon-picker
	 * (vc_iconpicker_base_register_js, vc_iconpicker_editor_jscss), see
	 * wp-content/plugins/js_composer/include/autoload/hook-vc-iconpicker-param.php
	 * folder

	 * @since 4.4
	 * @return string - rendered param field for editor panel
	 */
	public function render() {

		$output = '<div class="vc-iconpicker-wrapper"><select class="vc-iconpicker">';

		// call filter vc_iconpicker-type-{your_type}, e.g. vc_iconpicker-type-fontawesome with passed source from shortcode(default empty array). to get icons
		$arr = apply_filters( 'vc_iconpicker-type-' . esc_attr( $this->settings['settings']['type'] ), $this->source );
		if ( isset( $this->settings['settings'], $this->settings['settings']['emptyIcon'] ) && true === $this->settings['settings']['emptyIcon'] ) {
			array_unshift( $arr, [] );
		}
		if ( ! empty( $arr ) ) {
			foreach ( $arr as $group => $icons ) {
				if ( ! is_array( $icons ) || ! is_array( current( $icons ) ) ) {
					$class_key = empty( $icons ) ? '' : key( $icons );
					$output .= vc_get_template( 'params/iconpicker/single_icon.php',
						[
							'class_key' => $class_key,
							'selected' => null !== $this->value && 0 === strcmp( $class_key, $this->value ) ? 'selected' : '',
							'icon' => current( $icons ),
						]
					);
				} else {
					$output .= vc_get_template( 'params/iconpicker/icon_group.php',
						[
							'icons' => $icons,
							'group' => $group,
							'value' => $this->value,
						]
					);
				}
			}
		}
		$output .= '</select></div>';

		$output .= '<input name="' . esc_attr( $this->settings['param_name'] ) . '" class="wpb_vc_param_value  ' . esc_attr( $this->settings['param_name'] ) . ' ' . esc_attr( $this->settings['type'] ) . '_field" type="hidden" value="' . esc_attr( $this->value ) . '" ' . ( ( isset( $this->settings['settings'] ) && ! empty( $this->settings['settings'] ) ) ? ' data-settings="' . esc_attr( wp_json_encode( $this->settings['settings'] ) ) . '" ' : '' ) . ' />';

		return $output;
	}
}

/**
 * Function for rendering param in edit form (add element)
 * Parse settings from vc_map and entered values.
 *
 * @param $settings
 * @param $value
 * @param $tag
 *
 * @since 4.4
 * @return string - rendered template for params in edit form
 *
 */
function vc_iconpicker_form_field( $settings, $value, $tag ) {

	$icon_picker = new Vc_IconPicker( $settings, $value );

	return apply_filters( 'vc_iconpicker_render_filter', $icon_picker->render() );
}

// SEE HOOKS FOLDER FOR FONTS REGISTERING/ENQUEUE IN BASE @path "/include/autoload/hook-vc-iconpicker-param.php"

add_filter( 'vc_iconpicker-type-fontawesome', 'vc_iconpicker_type_fontawesome' );

/**
 * Fontawesome icons from FontAwesome :)
 *
 * @param $icons - taken from filter - vc_map param field settings['source']
 *     provided icons (default empty array). If array categorized it will
 *     auto-enable category dropdown
 *

 * @since 4.4
 * @return array - of icons for iconpicker, can be categorized, or not.
 */

function vc_iconpicker_type_fontawesome( $icons ) {
	// Categorized icons ( you can also output simple array ( key=> value ), where key = icon class, value = icon readable name ).
	/**
	 * @version 4.7
	 */
	$fontawesome_icons = array(
		'New in 4.7' => array(
			array( 'fa fa-handshake-o' => 'Handshake Outlined' ),
			array( 'fa fa-envelope-open' => 'Envelope Open(email, e-mail, letter, support, mail, message, notification)' ),
			array( 'fa fa-envelope-open-o' => 'Envelope Open Outlined(email, e-mail, letter, support, mail, message, notification)' ),
			array( 'fa fa-linode' => 'Linode' ),
			array( 'fa fa-address-book' => 'Address Book' ),
			array( 'fa fa-address-book-o' => 'Address Book Outlined' ),
			array( 'fa fa-address-card' => 'Address Card(vcard)' ),
			array( 'fa fa-address-card-o' => 'Address Card Outlined(vcard-o)' ),
			array( 'fa fa-user-circle' => 'User Circle' ),
			array( 'fa fa-user-circle-o' => 'User Circle Outlined' ),
			array( 'fa fa-user-o' => 'User Outlined' ),
			array( 'fa fa-id-badge' => 'Identification Badge' ),
			array( 'fa fa-id-card' => 'Identification Card(drivers-license)' ),
			array( 'fa fa-id-card-o' => 'Identification Card Outlined(drivers-license-o)' ),
			array( 'fa fa-quora' => 'Quora' ),
			array( 'fa fa-free-code-camp' => 'Free Code Camp' ),
			array( 'fa fa-telegram' => 'Telegram' ),
			array( 'fa fa-thermometer-full' => 'Thermometer Full(thermometer-4, thermometer)' ),
			array( 'fa fa-thermometer-three-quarters' => 'Thermometer 3/4 Full(thermometer-3)' ),
			array( 'fa fa-thermometer-half' => 'Thermometer 1/2 Full(thermometer-2)' ),
			array( 'fa fa-thermometer-quarter' => 'Thermometer 1/4 Full(thermometer-1)' ),
			array( 'fa fa-thermometer-empty' => 'Thermometer Empty(thermometer-0)' ),
			array( 'fa fa-shower' => 'Shower' ),
			array( 'fa fa-bath' => 'Bath(bathtub, s15)' ),
			array( 'fa fa-podcast' => 'Podcast' ),
			array( 'fa fa-window-maximize' => 'Window Maximize' ),
			array( 'fa fa-window-minimize' => 'Window Minimize' ),
			array( 'fa fa-window-restore' => 'Window Restore' ),
			array( 'fa fa-window-close' => 'Window Close(times-rectangle)' ),
			array( 'fa fa-window-close-o' => 'Window Close Outline(times-rectangle-o)' ),
			array( 'fa fa-bandcamp' => 'Bandcamp' ),
			array( 'fa fa-grav' => 'Grav' ),
			array( 'fa fa-etsy' => 'Etsy' ),
			array( 'fa fa-imdb' => 'IMDB' ),
			array( 'fa fa-ravelry' => 'Ravelry' ),
			array( 'fa fa-eercast' => 'Eercast' ),
			array( 'fa fa-microchip' => 'Microchip' ),
			array( 'fa fa-snowflake-o' => 'Snowflake Outlined' ),
			array( 'fa fa-superpowers' => 'Superpowers' ),
			array( 'fa fa-wpexplorer' => 'WPExplorer' ),
			array( 'fa fa-meetup' => 'Meetup' ),
		),
		'Web Application Icons' => array(
			array( 'fa fa-glass' => 'Glass(martini, drink, bar, alcohol, liquor)' ),
			array( 'fa fa-music' => 'Music(note, sound)' ),
			array( 'fa fa-search' => 'Search(magnify, zoom, enlarge, bigger)' ),
			array( 'fa fa-envelope-o' => 'Envelope Outlined(email, e-mail, letter, support, mail, message, notification)' ),
			array( 'fa fa-heart' => 'Heart(love, like, favorite)' ),
			array( 'fa fa-star' => 'Star(award, achievement, night, rating, score, favorite)' ),
			array( 'fa fa-star-o' => 'Star Outlined(award, achievement, night, rating, score, favorite)' ),
			array( 'fa fa-user' => 'User(person, man, head, profile)' ),
			array( 'fa fa-film' => 'Film(movie)' ),
			array( 'fa fa-check' => 'Check(checkmark, done, todo, agree, accept, confirm, tick, ok)' ),
			array( 'fa fa-times' => 'Times(close, exit, x, cross)(remove, close)' ),
			array( 'fa fa-search-plus' => 'Search Plus(magnify, zoom, enlarge, bigger)' ),
			array( 'fa fa-search-minus' => 'Search Minus(magnify, minify, zoom, smaller)' ),
			array( 'fa fa-power-off' => 'Power Off(on)' ),
			array( 'fa fa-signal' => 'signal(graph, bars)' ),
			array( 'fa fa-cog' => 'cog(settings)(gear)' ),
			array( 'fa fa-trash-o' => 'Trash Outlined(garbage, delete, remove, trash, hide)' ),
			array( 'fa fa-home' => 'home(main, house)' ),
			array( 'fa fa-clock-o' => 'Clock Outlined(watch, timer, late, timestamp)' ),
			array( 'fa fa-road' => 'road(street)' ),
			array( 'fa fa-download' => 'Download(import)' ),
			array( 'fa fa-inbox' => 'inbox' ),
			array( 'fa fa-refresh' => 'refresh(reload, sync)' ),
			array( 'fa fa-lock' => 'lock(protect, admin, security)' ),
			array( 'fa fa-flag' => 'flag(report, notification, notify)' ),
			array( 'fa fa-headphones' => 'headphones(sound, listen, music, audio)' ),
			array( 'fa fa-volume-off' => 'volume-off(audio, mute, sound, music)' ),
			array( 'fa fa-volume-down' => 'volume-down(audio, lower, quieter, sound, music)' ),
			array( 'fa fa-volume-up' => 'volume-up(audio, higher, louder, sound, music)' ),
			array( 'fa fa-qrcode' => 'qrcode(scan)' ),
			array( 'fa fa-barcode' => 'barcode(scan)' ),
			array( 'fa fa-tag' => 'tag(label)' ),
			array( 'fa fa-tags' => 'tags(labels)' ),
			array( 'fa fa-book' => 'book(read, documentation)' ),
			array( 'fa fa-bookmark' => 'bookmark(save)' ),
			array( 'fa fa-print' => 'print' ),
			array( 'fa fa-camera' => 'camera(photo, picture, record)' ),
			array( 'fa fa-video-camera' => 'Video Camera(film, movie, record)' ),
			array( 'fa fa-picture-o' => 'Picture Outlined(photo, image)' ),
			array( 'fa fa-pencil' => 'pencil(write, edit, update)' ),
			array( 'fa fa-map-marker' => 'map-marker(map, pin, location, coordinates, localize, address, travel, where, place)' ),
			array( 'fa fa-adjust' => 'adjust(contrast)' ),
			array( 'fa fa-tint' => 'tint(raindrop, waterdrop, drop, droplet)' ),
			array( 'fa fa-pencil-square-o' => 'Pencil Square Outlined(write, edit, update)(edit)' ),
			array( 'fa fa-share-square-o' => 'Share Square Outlined(social, send, arrow)' ),
			array( 'fa fa-check-square-o' => 'Check Square Outlined(todo, done, agree, accept, confirm, ok)' ),
			array( 'fa fa-arrows' => 'Arrows(move, reorder, resize)' ),
			array( 'fa fa-plus-circle' => 'Plus Circle(add, new, create, expand)' ),
			array( 'fa fa-minus-circle' => 'Minus Circle(delete, remove, trash, hide)' ),
			array( 'fa fa-times-circle' => 'Times Circle(close, exit, x)' ),
			array( 'fa fa-check-circle' => 'Check Circle(todo, done, agree, accept, confirm, ok)' ),
			array( 'fa fa-question-circle' => 'Question Circle(help, information, unknown, support)' ),
			array( 'fa fa-info-circle' => 'Info Circle(help, information, more, details)' ),
			array( 'fa fa-crosshairs' => 'Crosshairs(picker)' ),
			array( 'fa fa-times-circle-o' => 'Times Circle Outlined(close, exit, x)' ),
			array( 'fa fa-check-circle-o' => 'Check Circle Outlined(todo, done, agree, accept, confirm, ok)' ),
			array( 'fa fa-ban' => 'ban(delete, remove, trash, hide, block, stop, abort, cancel)' ),
			array( 'fa fa-share' => 'Share(mail-forward)' ),
			array( 'fa fa-plus' => 'plus(add, new, create, expand)' ),
			array( 'fa fa-minus' => 'minus(hide, minify, delete, remove, trash, hide, collapse)' ),
			array( 'fa fa-asterisk' => 'asterisk(details)' ),
			array( 'fa fa-exclamation-circle' => 'Exclamation Circle(warning, error, problem, notification, alert)' ),
			array( 'fa fa-gift' => 'gift(present)' ),
			array( 'fa fa-leaf' => 'leaf(eco, nature, plant)' ),
			array( 'fa fa-fire' => 'fire(flame, hot, popular)' ),
			array( 'fa fa-eye' => 'Eye(show, visible, views)' ),
			array( 'fa fa-eye-slash' => 'Eye Slash(toggle, show, hide, visible, visiblity, views)' ),
			array( 'fa fa-exclamation-triangle' => 'Exclamation Triangle(warning, error, problem, notification, alert)(warning)' ),
			array( 'fa fa-plane' => 'plane(travel, trip, location, destination, airplane, fly, mode)' ),
			array( 'fa fa-calendar' => 'calendar(date, time, when, event)' ),
			array( 'fa fa-random' => 'random(sort, shuffle)' ),
			array( 'fa fa-comment' => 'comment(speech, notification, note, chat, bubble, feedback, message, texting, sms, conversation)' ),
			array( 'fa fa-magnet' => 'magnet' ),
			array( 'fa fa-retweet' => 'retweet(refresh, reload, share)' ),
			array( 'fa fa-shopping-cart' => 'shopping-cart(checkout, buy, purchase, payment)' ),
			array( 'fa fa-folder' => 'Folder' ),
			array( 'fa fa-folder-open' => 'Folder Open' ),
			array( 'fa fa-arrows-v' => 'Arrows Vertical(resize)' ),
			array( 'fa fa-arrows-h' => 'Arrows Horizontal(resize)' ),
			array( 'fa fa-bar-chart' => 'Bar Chart(graph, analytics, statistics)(bar-chart-o)' ),
			array( 'fa fa-camera-retro' => 'camera-retro(photo, picture, record)' ),
			array( 'fa fa-key' => 'key(unlock, password)' ),
			array( 'fa fa-cogs' => 'cogs(settings)(gears)' ),
			array( 'fa fa-comments' => 'comments(speech, notification, note, chat, bubble, feedback, message, texting, sms, conversation)' ),
			array( 'fa fa-thumbs-o-up' => 'Thumbs Up Outlined(like, approve, favorite, agree, hand)' ),
			array( 'fa fa-thumbs-o-down' => 'Thumbs Down Outlined(dislike, disapprove, disagree, hand)' ),
			array( 'fa fa-star-half' => 'star-half(award, achievement, rating, score)' ),
			array( 'fa fa-heart-o' => 'Heart Outlined(love, like, favorite)' ),
			array( 'fa fa-sign-out' => 'Sign Out(log out, logout, leave, exit, arrow)' ),
			array( 'fa fa-thumb-tack' => 'Thumb Tack(marker, pin, location, coordinates)' ),
			array( 'fa fa-external-link' => 'External Link(open, new)' ),
			array( 'fa fa-sign-in' => 'Sign In(enter, join, log in, login, sign up, sign in, signin, signup, arrow)' ),
			array( 'fa fa-trophy' => 'trophy(award, achievement, cup, winner, game)' ),
			array( 'fa fa-upload' => 'Upload(import)' ),
			array( 'fa fa-lemon-o' => 'Lemon Outlined(food)' ),
			array( 'fa fa-phone' => 'Phone(call, voice, number, support, earphone, telephone)' ),
			array( 'fa fa-square-o' => 'Square Outlined(block, square, box)' ),
			array( 'fa fa-bookmark-o' => 'Bookmark Outlined(save)' ),
			array( 'fa fa-phone-square' => 'Phone Square(call, voice, number, support, telephone)' ),
			array( 'fa fa-unlock' => 'unlock(protect, admin, password, lock)' ),
			array( 'fa fa-credit-card' => 'credit-card(money, buy, debit, checkout, purchase, payment)' ),
			array( 'fa fa-rss' => 'rss(blog)(feed)' ),
			array( 'fa fa-hdd-o' => 'HDD(harddrive, hard drive, storage, save)' ),
			array( 'fa fa-bullhorn' => 'bullhorn(announcement, share, broadcast, louder, megaphone)' ),
			array( 'fa fa-bell' => 'bell(alert, reminder, notification)' ),
			array( 'fa fa-certificate' => 'certificate(badge, star)' ),
			array( 'fa fa-globe' => 'Globe(world, planet, map, place, travel, earth, global, translate, all, language, localize, location, coordinates, country)' ),
			array( 'fa fa-wrench' => 'Wrench(settings, fix, update, spanner)' ),
			array( 'fa fa-tasks' => 'Tasks(progress, loading, downloading, downloads, settings)' ),
			array( 'fa fa-filter' => 'Filter(funnel, options)' ),
			array( 'fa fa-briefcase' => 'Briefcase(work, business, office, luggage, bag)' ),
			array( 'fa fa-users' => 'Users(people, profiles, persons)(group)' ),
			array( 'fa fa-cloud' => 'Cloud(save)' ),
			array( 'fa fa-flask' => 'Flask(science, beaker, experimental, labs)' ),
			array( 'fa fa-square' => 'Square(block, box)' ),
			array( 'fa fa-bars' => 'Bars(menu, drag, reorder, settings, list, ul, ol, checklist, todo, list, hamburger)(navicon, reorder)' ),
			array( 'fa fa-magic' => 'magic(wizard, automatic, autocomplete)' ),
			array( 'fa fa-truck' => 'truck(shipping)' ),
			array( 'fa fa-money' => 'Money(cash, money, buy, checkout, purchase, payment)' ),
			array( 'fa fa-sort' => 'Sort(order)(unsorted)' ),
			array( 'fa fa-sort-desc' => 'Sort Descending(dropdown, more, menu, arrow)(sort-down)' ),
			array( 'fa fa-sort-asc' => 'Sort Ascending(arrow)(sort-up)' ),
			array( 'fa fa-envelope' => 'Envelope(email, e-mail, letter, support, mail, message, notification)' ),
			array( 'fa fa-gavel' => 'Gavel(judge, lawyer, opinion)(legal)' ),
			array( 'fa fa-tachometer' => 'Tachometer(speedometer, fast)(dashboard)' ),
			array( 'fa fa-comment-o' => 'comment-o(speech, notification, note, chat, bubble, feedback, message, texting, sms, conversation)' ),
			array( 'fa fa-comments-o' => 'comments-o(speech, notification, note, chat, bubble, feedback, message, texting, sms, conversation)' ),
			array( 'fa fa-bolt' => 'Lightning Bolt(lightning, weather)(flash)' ),
			array( 'fa fa-sitemap' => 'Sitemap(directory, hierarchy, organization)' ),
			array( 'fa fa-umbrella' => 'Umbrella' ),
			array( 'fa fa-lightbulb-o' => 'Lightbulb Outlined(idea, inspiration)' ),
			array( 'fa fa-exchange' => 'Exchange(transfer, arrows, arrow)' ),
			array( 'fa fa-cloud-download' => 'Cloud Download(import)' ),
			array( 'fa fa-cloud-upload' => 'Cloud Upload(import)' ),
			array( 'fa fa-suitcase' => 'Suitcase(trip, luggage, travel, move, baggage)' ),
			array( 'fa fa-bell-o' => 'Bell Outlined(alert, reminder, notification)' ),
			array( 'fa fa-coffee' => 'Coffee(morning, mug, breakfast, tea, drink, cafe)' ),
			array( 'fa fa-cutlery' => 'Cutlery(food, restaurant, spoon, knife, dinner, eat)' ),
			array( 'fa fa-building-o' => 'Building Outlined(work, business, apartment, office, company)' ),
			array( 'fa fa-fighter-jet' => 'fighter-jet(fly, plane, airplane, quick, fast, travel)' ),
			array( 'fa fa-beer' => 'beer(alcohol, stein, drink, mug, bar, liquor)' ),
			array( 'fa fa-plus-square' => 'Plus Square(add, new, create, expand)' ),
			array( 'fa fa-desktop' => 'Desktop(monitor, screen, desktop, computer, demo, device)' ),
			array( 'fa fa-laptop' => 'Laptop(demo, computer, device)' ),
			array( 'fa fa-tablet' => 'tablet(ipad, device)' ),
			array( 'fa fa-mobile' => 'Mobile Phone(cell phone, cellphone, text, call, iphone, number, telephone)(mobile-phone)' ),
			array( 'fa fa-circle-o' => 'Circle Outlined' ),
			array( 'fa fa-quote-left' => 'quote-left' ),
			array( 'fa fa-quote-right' => 'quote-right' ),
			array( 'fa fa-spinner' => 'Spinner(loading, progress)' ),
			array( 'fa fa-circle' => 'Circle(dot, notification)' ),
			array( 'fa fa-reply' => 'Reply(mail-reply)' ),
			array( 'fa fa-folder-o' => 'Folder Outlined' ),
			array( 'fa fa-folder-open-o' => 'Folder Open Outlined' ),
			array( 'fa fa-smile-o' => 'Smile Outlined(face, emoticon, happy, approve, satisfied, rating)' ),
			array( 'fa fa-frown-o' => 'Frown Outlined(face, emoticon, sad, disapprove, rating)' ),
			array( 'fa fa-meh-o' => 'Meh Outlined(face, emoticon, rating, neutral)' ),
			array( 'fa fa-gamepad' => 'Gamepad(controller)' ),
			array( 'fa fa-keyboard-o' => 'Keyboard Outlined(type, input)' ),
			array( 'fa fa-flag-o' => 'Flag Outlined(report, notification)' ),
			array( 'fa fa-flag-checkered' => 'flag-checkered(report, notification, notify)' ),
			array( 'fa fa-terminal' => 'Terminal(command, prompt, code)' ),
			array( 'fa fa-code' => 'Code(html, brackets)' ),
			array( 'fa fa-reply-all' => 'reply-all(mail-reply-all)' ),
			array( 'fa fa-star-half-o' => 'Star Half Outlined(award, achievement, rating, score)(star-half-empty, star-half-full)' ),
			array( 'fa fa-location-arrow' => 'location-arrow(map, coordinates, location, address, place, where)' ),
			array( 'fa fa-crop' => 'crop' ),
			array( 'fa fa-code-fork' => 'code-fork(git, fork, vcs, svn, github, rebase, version, merge)' ),
			array( 'fa fa-question' => 'Question(help, information, unknown, support)' ),
			array( 'fa fa-info' => 'Info(help, information, more, details)' ),
			array( 'fa fa-exclamation' => 'exclamation(warning, error, problem, notification, notify, alert)' ),
			array( 'fa fa-eraser' => 'eraser(remove, delete)' ),
			array( 'fa fa-puzzle-piece' => 'Puzzle Piece(addon, add-on, section)' ),
			array( 'fa fa-microphone' => 'microphone(record, voice, sound)' ),
			array( 'fa fa-microphone-slash' => 'Microphone Slash(record, voice, sound, mute)' ),
			array( 'fa fa-shield' => 'shield(award, achievement, security, winner)' ),
			array( 'fa fa-calendar-o' => 'calendar-o(date, time, when, event)' ),
			array( 'fa fa-fire-extinguisher' => 'fire-extinguisher' ),
			array( 'fa fa-rocket' => 'rocket(app)' ),
			array( 'fa fa-anchor' => 'Anchor(link)' ),
			array( 'fa fa-unlock-alt' => 'Unlock Alt(protect, admin, password, lock)' ),
			array( 'fa fa-bullseye' => 'Bullseye(target)' ),
			array( 'fa fa-ellipsis-h' => 'Ellipsis Horizontal(dots)' ),
			array( 'fa fa-ellipsis-v' => 'Ellipsis Vertical(dots)' ),
			array( 'fa fa-rss-square' => 'RSS Square(feed, blog)' ),
			array( 'fa fa-ticket' => 'Ticket(movie, pass, support)' ),
			array( 'fa fa-minus-square' => 'Minus Square(hide, minify, delete, remove, trash, hide, collapse)' ),
			array( 'fa fa-minus-square-o' => 'Minus Square Outlined(hide, minify, delete, remove, trash, hide, collapse)' ),
			array( 'fa fa-level-up' => 'Level Up(arrow)' ),
			array( 'fa fa-level-down' => 'Level Down(arrow)' ),
			array( 'fa fa-check-square' => 'Check Square(checkmark, done, todo, agree, accept, confirm, ok)' ),
			array( 'fa fa-pencil-square' => 'Pencil Square(write, edit, update)' ),
			array( 'fa fa-external-link-square' => 'External Link Square(open, new)' ),
			array( 'fa fa-share-square' => 'Share Square(social, send)' ),
			array( 'fa fa-compass' => 'Compass(safari, directory, menu, location)' ),
			array( 'fa fa-caret-square-o-down' => 'Caret Square Outlined Down(more, dropdown, menu)(toggle-down)' ),
			array( 'fa fa-caret-square-o-up' => 'Caret Square Outlined Up(toggle-up)' ),
			array( 'fa fa-caret-square-o-right' => 'Caret Square Outlined Right(next, forward)(toggle-right)' ),
			array( 'fa fa-sort-alpha-asc' => 'Sort Alpha Ascending' ),
			array( 'fa fa-sort-alpha-desc' => 'Sort Alpha Descending' ),
			array( 'fa fa-sort-amount-asc' => 'Sort Amount Ascending' ),
			array( 'fa fa-sort-amount-desc' => 'Sort Amount Descending' ),
			array( 'fa fa-sort-numeric-asc' => 'Sort Numeric Ascending(numbers)' ),
			array( 'fa fa-sort-numeric-desc' => 'Sort Numeric Descending(numbers)' ),
			array( 'fa fa-thumbs-up' => 'thumbs-up(like, favorite, approve, agree, hand)' ),
			array( 'fa fa-thumbs-down' => 'thumbs-down(dislike, disapprove, disagree, hand)' ),
			array( 'fa fa-female' => 'Female(woman, user, person, profile)' ),
			array( 'fa fa-male' => 'Male(man, user, person, profile)' ),
			array( 'fa fa-sun-o' => 'Sun Outlined(weather, contrast, lighter, brighten, day)' ),
			array( 'fa fa-moon-o' => 'Moon Outlined(night, darker, contrast)' ),
			array( 'fa fa-archive' => 'Archive(box, storage)' ),
			array( 'fa fa-bug' => 'Bug(report, insect)' ),
			array( 'fa fa-caret-square-o-left' => 'Caret Square Outlined Left(previous, back)(toggle-left)' ),
			array( 'fa fa-dot-circle-o' => 'Dot Circle Outlined(target, bullseye, notification)' ),
			array( 'fa fa-wheelchair' => 'Wheelchair(handicap, person)' ),
			array( 'fa fa-plus-square-o' => 'Plus Square Outlined(add, new, create, expand)' ),
			array( 'fa fa-space-shuttle' => 'Space Shuttle' ),
			array( 'fa fa-envelope-square' => 'Envelope Square(email, e-mail, letter, support, mail, message, notification)' ),
			array( 'fa fa-university' => 'University(institution, bank)' ),
			array( 'fa fa-graduation-cap' => 'Graduation Cap(learning, school, student)(mortar-board)' ),
			array( 'fa fa-language' => 'Language(translate)' ),
			array( 'fa fa-fax' => 'Fax' ),
			array( 'fa fa-building' => 'Building(work, business, apartment, office, company)' ),
			array( 'fa fa-child' => 'Child' ),
			array( 'fa fa-paw' => 'Paw(pet)' ),
			array( 'fa fa-spoon' => 'spoon' ),
			array( 'fa fa-cube' => 'Cube' ),
			array( 'fa fa-cubes' => 'Cubes' ),
			array( 'fa fa-recycle' => 'Recycle' ),
			array( 'fa fa-car' => 'Car(vehicle)(automobile)' ),
			array( 'fa fa-taxi' => 'Taxi(vehicle)(cab)' ),
			array( 'fa fa-tree' => 'Tree' ),
			array( 'fa fa-database' => 'Database' ),
			array( 'fa fa-file-pdf-o' => 'PDF File Outlined' ),
			array( 'fa fa-file-word-o' => 'Word File Outlined' ),
			array( 'fa fa-file-excel-o' => 'Excel File Outlined' ),
			array( 'fa fa-file-powerpoint-o' => 'Powerpoint File Outlined' ),
			array( 'fa fa-file-image-o' => 'Image File Outlined(file-photo-o, file-picture-o)' ),
			array( 'fa fa-file-archive-o' => 'Archive File Outlined(file-zip-o)' ),
			array( 'fa fa-file-audio-o' => 'Audio File Outlined(file-sound-o)' ),
			array( 'fa fa-file-video-o' => 'Video File Outlined(file-movie-o)' ),
			array( 'fa fa-file-code-o' => 'Code File Outlined' ),
			array( 'fa fa-life-ring' => 'Life Ring(life-bouy, life-buoy, life-saver, support)' ),
			array( 'fa fa-circle-o-notch' => 'Circle Outlined Notched' ),
			array( 'fa fa-paper-plane' => 'Paper Plane(send)' ),
			array( 'fa fa-paper-plane-o' => 'Paper Plane Outlined(send-o)' ),
			array( 'fa fa-history' => 'History' ),
			array( 'fa fa-circle-thin' => 'Circle Outlined Thin' ),
			array( 'fa fa-sliders' => 'Sliders(settings)' ),
			array( 'fa fa-share-alt' => 'Share Alt' ),
			array( 'fa fa-share-alt-square' => 'Share Alt Square' ),
			array( 'fa fa-bomb' => 'Bomb' ),
			array( 'fa fa-futbol-o' => 'Futbol Outlined(soccer-ball-o)' ),
			array( 'fa fa-tty' => 'TTY' ),
			array( 'fa fa-binoculars' => 'Binoculars' ),
			array( 'fa fa-plug' => 'Plug(power, connect)' ),
			array( 'fa fa-newspaper-o' => 'Newspaper Outlined(press)' ),
			array( 'fa fa-wifi' => 'WiFi' ),
			array( 'fa fa-calculator' => 'Calculator' ),
			array( 'fa fa-bell-slash' => 'Bell Slash' ),
			array( 'fa fa-bell-slash-o' => 'Bell Slash Outlined' ),
			array( 'fa fa-trash' => 'Trash(garbage, delete, remove, hide)' ),
			array( 'fa fa-copyright' => 'Copyright' ),
			array( 'fa fa-at' => 'At(email, e-mail)' ),
			array( 'fa fa-eyedropper' => 'Eyedropper' ),
			array( 'fa fa-paint-brush' => 'Paint Brush' ),
			array( 'fa fa-birthday-cake' => 'Birthday Cake' ),
			array( 'fa fa-area-chart' => 'Area Chart(graph, analytics, statistics)' ),
			array( 'fa fa-pie-chart' => 'Pie Chart(graph, analytics, statistics)' ),
			array( 'fa fa-line-chart' => 'Line Chart(graph, analytics, statistics)' ),
			array( 'fa fa-toggle-off' => 'Toggle Off' ),
			array( 'fa fa-toggle-on' => 'Toggle On' ),
			array( 'fa fa-bicycle' => 'Bicycle(vehicle, bike)' ),
			array( 'fa fa-bus' => 'Bus(vehicle)' ),
			array( 'fa fa-cc' => 'Closed Captions' ),
			array( 'fa fa-cart-plus' => 'Add to Shopping Cart(add, shopping)' ),
			array( 'fa fa-cart-arrow-down' => 'Shopping Cart Arrow Down(shopping)' ),
			array( 'fa fa-diamond' => 'Diamond(gem, gemstone)' ),
			array( 'fa fa-ship' => 'Ship(boat, sea)' ),
			array( 'fa fa-user-secret' => 'User Secret(whisper, spy, incognito, privacy)' ),
			array( 'fa fa-motorcycle' => 'Motorcycle(vehicle, bike)' ),
			array( 'fa fa-street-view' => 'Street View(map)' ),
			array( 'fa fa-heartbeat' => 'Heartbeat(ekg)' ),
			array( 'fa fa-server' => 'Server' ),
			array( 'fa fa-user-plus' => 'Add User(sign up, signup)' ),
			array( 'fa fa-user-times' => 'Remove User' ),
			array( 'fa fa-bed' => 'Bed(travel)(hotel)' ),
			array( 'fa fa-battery-full' => 'Battery Full(power)(battery-4, battery)' ),
			array( 'fa fa-battery-three-quarters' => 'Battery 3/4 Full(power)(battery-3)' ),
			array( 'fa fa-battery-half' => 'Battery 1/2 Full(power)(battery-2)' ),
			array( 'fa fa-battery-quarter' => 'Battery 1/4 Full(power)(battery-1)' ),
			array( 'fa fa-battery-empty' => 'Battery Empty(power)(battery-0)' ),
			array( 'fa fa-mouse-pointer' => 'Mouse Pointer' ),
			array( 'fa fa-i-cursor' => 'I Beam Cursor' ),
			array( 'fa fa-object-group' => 'Object Group' ),
			array( 'fa fa-object-ungroup' => 'Object Ungroup' ),
			array( 'fa fa-sticky-note' => 'Sticky Note' ),
			array( 'fa fa-sticky-note-o' => 'Sticky Note Outlined' ),
			array( 'fa fa-clone' => 'Clone(copy)' ),
			array( 'fa fa-balance-scale' => 'Balance Scale' ),
			array( 'fa fa-hourglass-o' => 'Hourglass Outlined' ),
			array( 'fa fa-hourglass-start' => 'Hourglass Start(hourglass-1)' ),
			array( 'fa fa-hourglass-half' => 'Hourglass Half(hourglass-2)' ),
			array( 'fa fa-hourglass-end' => 'Hourglass End(hourglass-3)' ),
			array( 'fa fa-hourglass' => 'Hourglass' ),
			array( 'fa fa-hand-rock-o' => 'Rock (Hand)(hand-grab-o)' ),
			array( 'fa fa-hand-paper-o' => 'Paper (Hand)(stop)(hand-stop-o)' ),
			array( 'fa fa-hand-scissors-o' => 'Scissors (Hand)' ),
			array( 'fa fa-hand-lizard-o' => 'Lizard (Hand)' ),
			array( 'fa fa-hand-spock-o' => 'Spock (Hand)' ),
			array( 'fa fa-hand-pointer-o' => 'Hand Pointer' ),
			array( 'fa fa-hand-peace-o' => 'Hand Peace' ),
			array( 'fa fa-trademark' => 'Trademark' ),
			array( 'fa fa-registered' => 'Registered Trademark' ),
			array( 'fa fa-creative-commons' => 'Creative Commons' ),
			array( 'fa fa-television' => 'Television(display, computer, monitor)(tv)' ),
			array( 'fa fa-calendar-plus-o' => 'Calendar Plus Outlined' ),
			array( 'fa fa-calendar-minus-o' => 'Calendar Minus Outlined' ),
			array( 'fa fa-calendar-times-o' => 'Calendar Times Outlined' ),
			array( 'fa fa-calendar-check-o' => 'Calendar Check Outlined(ok)' ),
			array( 'fa fa-industry' => 'Industry(factory)' ),
			array( 'fa fa-map-pin' => 'Map Pin' ),
			array( 'fa fa-map-signs' => 'Map Signs' ),
			array( 'fa fa-map-o' => 'Map Outlined' ),
			array( 'fa fa-map' => 'Map' ),
			array( 'fa fa-commenting' => 'Commenting(speech, notification, note, chat, bubble, feedback, message, texting, sms, conversation)' ),
			array( 'fa fa-commenting-o' => 'Commenting Outlined(speech, notification, note, chat, bubble, feedback, message, texting, sms, conversation)' ),
			array( 'fa fa-credit-card-alt' => 'Credit Card(money, buy, debit, checkout, purchase, payment, credit card)' ),
			array( 'fa fa-shopping-bag' => 'Shopping Bag' ),
			array( 'fa fa-shopping-basket' => 'Shopping Basket' ),
			array( 'fa fa-hashtag' => 'Hashtag' ),
			array( 'fa fa-bluetooth' => 'Bluetooth' ),
			array( 'fa fa-bluetooth-b' => 'Bluetooth' ),
			array( 'fa fa-percent' => 'Percent' ),
			array( 'fa fa-universal-access' => 'Universal Access' ),
			array( 'fa fa-wheelchair-alt' => 'Wheelchair Alt(handicap, person)' ),
			array( 'fa fa-question-circle-o' => 'Question Circle Outlined' ),
			array( 'fa fa-blind' => 'Blind' ),
			array( 'fa fa-audio-description' => 'Audio Description' ),
			array( 'fa fa-volume-control-phone' => 'Volume Control Phone(telephone)' ),
			array( 'fa fa-braille' => 'Braille' ),
			array( 'fa fa-assistive-listening-systems' => 'Assistive Listening Systems' ),
			array( 'fa fa-american-sign-language-interpreting' => 'American Sign Language Interpreting(asl-interpreting)' ),
			array( 'fa fa-deaf' => 'Deaf(deafness, hard-of-hearing)' ),
			array( 'fa fa-sign-language' => 'Sign Language(signing)' ),
			array( 'fa fa-low-vision' => 'Low Vision' ),
			array( 'fa fa-handshake-o' => 'Handshake Outlined' ),
			array( 'fa fa-envelope-open' => 'Envelope Open(email, e-mail, letter, support, mail, message, notification)' ),
			array( 'fa fa-envelope-open-o' => 'Envelope Open Outlined(email, e-mail, letter, support, mail, message, notification)' ),
			array( 'fa fa-address-book' => 'Address Book' ),
			array( 'fa fa-address-book-o' => 'Address Book Outlined' ),
			array( 'fa fa-address-card' => 'Address Card(vcard)' ),
			array( 'fa fa-address-card-o' => 'Address Card Outlined(vcard-o)' ),
			array( 'fa fa-user-circle' => 'User Circle' ),
			array( 'fa fa-user-circle-o' => 'User Circle Outlined' ),
			array( 'fa fa-user-o' => 'User Outlined' ),
			array( 'fa fa-id-badge' => 'Identification Badge' ),
			array( 'fa fa-id-card' => 'Identification Card(drivers-license)' ),
			array( 'fa fa-id-card-o' => 'Identification Card Outlined(drivers-license-o)' ),
			array( 'fa fa-thermometer-full' => 'Thermometer Full(thermometer-4, thermometer)' ),
			array( 'fa fa-thermometer-three-quarters' => 'Thermometer 3/4 Full(thermometer-3)' ),
			array( 'fa fa-thermometer-half' => 'Thermometer 1/2 Full(thermometer-2)' ),
			array( 'fa fa-thermometer-quarter' => 'Thermometer 1/4 Full(thermometer-1)' ),
			array( 'fa fa-thermometer-empty' => 'Thermometer Empty(thermometer-0)' ),
			array( 'fa fa-shower' => 'Shower' ),
			array( 'fa fa-bath' => 'Bath(bathtub, s15)' ),
			array( 'fa fa-podcast' => 'Podcast' ),
			array( 'fa fa-window-maximize' => 'Window Maximize' ),
			array( 'fa fa-window-minimize' => 'Window Minimize' ),
			array( 'fa fa-window-restore' => 'Window Restore' ),
			array( 'fa fa-window-close' => 'Window Close(times-rectangle)' ),
			array( 'fa fa-window-close-o' => 'Window Close Outline(times-rectangle-o)' ),
			array( 'fa fa-microchip' => 'Microchip' ),
			array( 'fa fa-snowflake-o' => 'Snowflake Outlined' ),
		),
		'Medical Icons' => array(
			array( 'fa fa-heart' => 'Heart(love, like, favorite)' ),
			array( 'fa fa-heart-o' => 'Heart Outlined(love, like, favorite)' ),
			array( 'fa fa-user-md' => 'user-md(doctor, profile, medical, nurse)' ),
			array( 'fa fa-stethoscope' => 'Stethoscope' ),
			array( 'fa fa-hospital-o' => 'hospital Outlined(building)' ),
			array( 'fa fa-ambulance' => 'ambulance(vehicle, support, help)' ),
			array( 'fa fa-medkit' => 'medkit(first aid, firstaid, help, support, health)' ),
			array( 'fa fa-h-square' => 'H Square(hospital, hotel)' ),
			array( 'fa fa-plus-square' => 'Plus Square(add, new, create, expand)' ),
			array( 'fa fa-wheelchair' => 'Wheelchair(handicap, person)' ),
			array( 'fa fa-heartbeat' => 'Heartbeat(ekg)' ),
			array( 'fa fa-wheelchair-alt' => 'Wheelchair Alt(handicap, person)' ),
		),
		'Text Editor Icons' => array(
			array( 'fa fa-th-large' => 'th-large(blocks, squares, boxes, grid)' ),
			array( 'fa fa-th' => 'th(blocks, squares, boxes, grid)' ),
			array( 'fa fa-th-list' => 'th-list(ul, ol, checklist, finished, completed, done, todo)' ),
			array( 'fa fa-file-o' => 'File Outlined(new, page, pdf, document)' ),
			array( 'fa fa-repeat' => 'Repeat(redo, forward)(rotate-right)' ),
			array( 'fa fa-list-alt' => 'list-alt(ul, ol, checklist, finished, completed, done, todo)' ),
			array( 'fa fa-font' => 'font(text)' ),
			array( 'fa fa-bold' => 'bold' ),
			array( 'fa fa-italic' => 'italic(italics)' ),
			array( 'fa fa-text-height' => 'text-height' ),
			array( 'fa fa-text-width' => 'text-width' ),
			array( 'fa fa-align-left' => 'align-left(text)' ),
			array( 'fa fa-align-center' => 'align-center(middle, text)' ),
			array( 'fa fa-align-right' => 'align-right(text)' ),
			array( 'fa fa-align-justify' => 'align-justify(text)' ),
			array( 'fa fa-list' => 'list(ul, ol, checklist, finished, completed, done, todo)' ),
			array( 'fa fa-outdent' => 'Outdent(dedent)' ),
			array( 'fa fa-indent' => 'Indent' ),
			array( 'fa fa-link' => 'Link(chain)(chain)' ),
			array( 'fa fa-scissors' => 'Scissors(cut)' ),
			array( 'fa fa-files-o' => 'Files Outlined(duplicate, clone, copy)(copy)' ),
			array( 'fa fa-paperclip' => 'Paperclip(attachment)' ),
			array( 'fa fa-floppy-o' => 'Floppy Outlined(save)' ),
			array( 'fa fa-list-ul' => 'list-ul(ul, ol, checklist, todo, list)' ),
			array( 'fa fa-list-ol' => 'list-ol(ul, ol, checklist, list, todo, list, numbers)' ),
			array( 'fa fa-strikethrough' => 'Strikethrough' ),
			array( 'fa fa-underline' => 'Underline' ),
			array( 'fa fa-table' => 'table(data, excel, spreadsheet)' ),
			array( 'fa fa-columns' => 'Columns(split, panes)' ),
			array( 'fa fa-undo' => 'Undo(back)(rotate-left)' ),
			array( 'fa fa-clipboard' => 'Clipboard(copy)(paste)' ),
			array( 'fa fa-file-text-o' => 'File Text Outlined(new, page, pdf, document)' ),
			array( 'fa fa-chain-broken' => 'Chain Broken(remove)(unlink)' ),
			array( 'fa fa-superscript' => 'superscript(exponential)' ),
			array( 'fa fa-subscript' => 'subscript' ),
			array( 'fa fa-eraser' => 'eraser(remove, delete)' ),
			array( 'fa fa-file' => 'File(new, page, pdf, document)' ),
			array( 'fa fa-file-text' => 'File Text(new, page, pdf, document)' ),
			array( 'fa fa-header' => 'header(heading)' ),
			array( 'fa fa-paragraph' => 'paragraph' ),
		),
		'Spinner Icons' => array(
			array( 'fa fa-cog' => 'cog(settings)(gear)' ),
			array( 'fa fa-refresh' => 'refresh(reload, sync)' ),
			array( 'fa fa-spinner' => 'Spinner(loading, progress)' ),
			array( 'fa fa-circle-o-notch' => 'Circle Outlined Notched' ),
		),
		'File Type Icons' => array(
			array( 'fa fa-file-o' => 'File Outlined(new, page, pdf, document)' ),
			array( 'fa fa-file-text-o' => 'File Text Outlined(new, page, pdf, document)' ),
			array( 'fa fa-file' => 'File(new, page, pdf, document)' ),
			array( 'fa fa-file-text' => 'File Text(new, page, pdf, document)' ),
			array( 'fa fa-file-pdf-o' => 'PDF File Outlined' ),
			array( 'fa fa-file-word-o' => 'Word File Outlined' ),
			array( 'fa fa-file-excel-o' => 'Excel File Outlined' ),
			array( 'fa fa-file-powerpoint-o' => 'Powerpoint File Outlined' ),
			array( 'fa fa-file-image-o' => 'Image File Outlined(file-photo-o, file-picture-o)' ),
			array( 'fa fa-file-archive-o' => 'Archive File Outlined(file-zip-o)' ),
			array( 'fa fa-file-audio-o' => 'Audio File Outlined(file-sound-o)' ),
			array( 'fa fa-file-video-o' => 'Video File Outlined(file-movie-o)' ),
			array( 'fa fa-file-code-o' => 'Code File Outlined' ),
		),
		'Directional Icons' => array(
			array( 'fa fa-arrow-circle-o-down' => 'Arrow Circle Outlined Down(download)' ),
			array( 'fa fa-arrow-circle-o-up' => 'Arrow Circle Outlined Up' ),
			array( 'fa fa-arrows' => 'Arrows(move, reorder, resize)' ),
			array( 'fa fa-chevron-left' => 'chevron-left(bracket, previous, back)' ),
			array( 'fa fa-chevron-right' => 'chevron-right(bracket, next, forward)' ),
			array( 'fa fa-arrow-left' => 'arrow-left(previous, back)' ),
			array( 'fa fa-arrow-right' => 'arrow-right(next, forward)' ),
			array( 'fa fa-arrow-up' => 'arrow-up' ),
			array( 'fa fa-arrow-down' => 'arrow-down(download)' ),
			array( 'fa fa-chevron-up' => 'chevron-up' ),
			array( 'fa fa-chevron-down' => 'chevron-down' ),
			array( 'fa fa-arrows-v' => 'Arrows Vertical(resize)' ),
			array( 'fa fa-arrows-h' => 'Arrows Horizontal(resize)' ),
			array( 'fa fa-hand-o-right' => 'Hand Outlined Right(point, right, next, forward, finger)' ),
			array( 'fa fa-hand-o-left' => 'Hand Outlined Left(point, left, previous, back, finger)' ),
			array( 'fa fa-hand-o-up' => 'Hand Outlined Up(point, finger)' ),
			array( 'fa fa-hand-o-down' => 'Hand Outlined Down(point, finger)' ),
			array( 'fa fa-arrow-circle-left' => 'Arrow Circle Left(previous, back)' ),
			array( 'fa fa-arrow-circle-right' => 'Arrow Circle Right(next, forward)' ),
			array( 'fa fa-arrow-circle-up' => 'Arrow Circle Up' ),
			array( 'fa fa-arrow-circle-down' => 'Arrow Circle Down(download)' ),
			array( 'fa fa-arrows-alt' => 'Arrows Alt(expand, enlarge, fullscreen, bigger, move, reorder, resize, arrow)' ),
			array( 'fa fa-caret-down' => 'Caret Down(more, dropdown, menu, triangle down, arrow)' ),
			array( 'fa fa-caret-up' => 'Caret Up(triangle up, arrow)' ),
			array( 'fa fa-caret-left' => 'Caret Left(previous, back, triangle left, arrow)' ),
			array( 'fa fa-caret-right' => 'Caret Right(next, forward, triangle right, arrow)' ),
			array( 'fa fa-exchange' => 'Exchange(transfer, arrows, arrow)' ),
			array( 'fa fa-angle-double-left' => 'Angle Double Left(laquo, quote, previous, back, arrows)' ),
			array( 'fa fa-angle-double-right' => 'Angle Double Right(raquo, quote, next, forward, arrows)' ),
			array( 'fa fa-angle-double-up' => 'Angle Double Up(arrows)' ),
			array( 'fa fa-angle-double-down' => 'Angle Double Down(arrows)' ),
			array( 'fa fa-angle-left' => 'angle-left(previous, back, arrow)' ),
			array( 'fa fa-angle-right' => 'angle-right(next, forward, arrow)' ),
			array( 'fa fa-angle-up' => 'angle-up(arrow)' ),
			array( 'fa fa-angle-down' => 'angle-down(arrow)' ),
			array( 'fa fa-chevron-circle-left' => 'Chevron Circle Left(previous, back, arrow)' ),
			array( 'fa fa-chevron-circle-right' => 'Chevron Circle Right(next, forward, arrow)' ),
			array( 'fa fa-chevron-circle-up' => 'Chevron Circle Up(arrow)' ),
			array( 'fa fa-chevron-circle-down' => 'Chevron Circle Down(more, dropdown, menu, arrow)' ),
			array( 'fa fa-caret-square-o-down' => 'Caret Square Outlined Down(more, dropdown, menu)(toggle-down)' ),
			array( 'fa fa-caret-square-o-up' => 'Caret Square Outlined Up(toggle-up)' ),
			array( 'fa fa-caret-square-o-right' => 'Caret Square Outlined Right(next, forward)(toggle-right)' ),
			array( 'fa fa-long-arrow-down' => 'Long Arrow Down' ),
			array( 'fa fa-long-arrow-up' => 'Long Arrow Up' ),
			array( 'fa fa-long-arrow-left' => 'Long Arrow Left(previous, back)' ),
			array( 'fa fa-long-arrow-right' => 'Long Arrow Right' ),
			array( 'fa fa-arrow-circle-o-right' => 'Arrow Circle Outlined Right(next, forward)' ),
			array( 'fa fa-arrow-circle-o-left' => 'Arrow Circle Outlined Left(previous, back)' ),
			array( 'fa fa-caret-square-o-left' => 'Caret Square Outlined Left(previous, back)(toggle-left)' ),
		),
		'Video Player Icons' => array(
			array( 'fa fa-play-circle-o' => 'Play Circle Outlined' ),
			array( 'fa fa-step-backward' => 'step-backward(rewind, previous, beginning, start, first)' ),
			array( 'fa fa-fast-backward' => 'fast-backward(rewind, previous, beginning, start, first)' ),
			array( 'fa fa-backward' => 'backward(rewind, previous)' ),
			array( 'fa fa-play' => 'play(start, playing, music, sound)' ),
			array( 'fa fa-pause' => 'pause(wait)' ),
			array( 'fa fa-stop' => 'stop(block, box, square)' ),
			array( 'fa fa-forward' => 'forward(forward, next)' ),
			array( 'fa fa-fast-forward' => 'fast-forward(next, end, last)' ),
			array( 'fa fa-step-forward' => 'step-forward(next, end, last)' ),
			array( 'fa fa-eject' => 'eject' ),
			array( 'fa fa-expand' => 'Expand(enlarge, bigger, resize)' ),
			array( 'fa fa-compress' => 'Compress(collapse, combine, contract, merge, smaller)' ),
			array( 'fa fa-random' => 'random(sort, shuffle)' ),
			array( 'fa fa-arrows-alt' => 'Arrows Alt(expand, enlarge, fullscreen, bigger, move, reorder, resize, arrow)' ),
			array( 'fa fa-play-circle' => 'Play Circle(start, playing)' ),
			array( 'fa fa-youtube-play' => 'YouTube Play(start, playing)' ),
			array( 'fa fa-pause-circle' => 'Pause Circle' ),
			array( 'fa fa-pause-circle-o' => 'Pause Circle Outlined' ),
			array( 'fa fa-stop-circle' => 'Stop Circle' ),
			array( 'fa fa-stop-circle-o' => 'Stop Circle Outlined' ),
		),
		'Form Control Icons' => array(
			array( 'fa fa-check-square-o' => 'Check Square Outlined(todo, done, agree, accept, confirm, ok)' ),
			array( 'fa fa-square-o' => 'Square Outlined(block, square, box)' ),
			array( 'fa fa-square' => 'Square(block, box)' ),
			array( 'fa fa-plus-square' => 'Plus Square(add, new, create, expand)' ),
			array( 'fa fa-circle-o' => 'Circle Outlined' ),
			array( 'fa fa-circle' => 'Circle(dot, notification)' ),
			array( 'fa fa-minus-square' => 'Minus Square(hide, minify, delete, remove, trash, hide, collapse)' ),
			array( 'fa fa-minus-square-o' => 'Minus Square Outlined(hide, minify, delete, remove, trash, hide, collapse)' ),
			array( 'fa fa-check-square' => 'Check Square(checkmark, done, todo, agree, accept, confirm, ok)' ),
			array( 'fa fa-dot-circle-o' => 'Dot Circle Outlined(target, bullseye, notification)' ),
			array( 'fa fa-plus-square-o' => 'Plus Square Outlined(add, new, create, expand)' ),
		),
		'Transportation Icons' => array(
			array( 'fa fa-plane' => 'plane(travel, trip, location, destination, airplane, fly, mode)' ),
			array( 'fa fa-truck' => 'truck(shipping)' ),
			array( 'fa fa-ambulance' => 'ambulance(vehicle, support, help)' ),
			array( 'fa fa-fighter-jet' => 'fighter-jet(fly, plane, airplane, quick, fast, travel)' ),
			array( 'fa fa-rocket' => 'rocket(app)' ),
			array( 'fa fa-wheelchair' => 'Wheelchair(handicap, person)' ),
			array( 'fa fa-space-shuttle' => 'Space Shuttle' ),
			array( 'fa fa-car' => 'Car(vehicle)(automobile)' ),
			array( 'fa fa-taxi' => 'Taxi(vehicle)(cab)' ),
			array( 'fa fa-bicycle' => 'Bicycle(vehicle, bike)' ),
			array( 'fa fa-bus' => 'Bus(vehicle)' ),
			array( 'fa fa-ship' => 'Ship(boat, sea)' ),
			array( 'fa fa-motorcycle' => 'Motorcycle(vehicle, bike)' ),
			array( 'fa fa-train' => 'Train' ),
			array( 'fa fa-subway' => 'Subway' ),
			array( 'fa fa-wheelchair-alt' => 'Wheelchair Alt(handicap, person)' ),
		),
		'Chart Icons' => array(
			array( 'fa fa-bar-chart' => 'Bar Chart(graph, analytics, statistics)(bar-chart-o)' ),
			array( 'fa fa-area-chart' => 'Area Chart(graph, analytics, statistics)' ),
			array( 'fa fa-pie-chart' => 'Pie Chart(graph, analytics, statistics)' ),
			array( 'fa fa-line-chart' => 'Line Chart(graph, analytics, statistics)' ),
		),
		'Brand Icons' => array(
			array( 'fa fa-twitter-square' => 'Twitter Square(tweet, social network)' ),
			array( 'fa fa-facebook-square' => 'Facebook Square(social network)' ),
			array( 'fa fa-linkedin-square' => 'LinkedIn Square' ),
			array( 'fa fa-github-square' => 'GitHub Square(octocat)' ),
			array( 'fa fa-twitter' => 'Twitter(tweet, social network)' ),
			array( 'fa fa-facebook' => 'Facebook(social network)(facebook-f)' ),
			array( 'fa fa-github' => 'GitHub(octocat)' ),
			array( 'fa fa-pinterest' => 'Pinterest' ),
			array( 'fa fa-pinterest-square' => 'Pinterest Square' ),
			array( 'fa fa-google-plus-square' => 'Google Plus Square(social network)' ),
			array( 'fa fa-google-plus' => 'Google Plus(social network)' ),
			array( 'fa fa-linkedin' => 'LinkedIn' ),
			array( 'fa fa-github-alt' => 'GitHub Alt(octocat)' ),
			array( 'fa fa-maxcdn' => 'MaxCDN' ),
			array( 'fa fa-html5' => 'HTML 5 Logo' ),
			array( 'fa fa-css3' => 'CSS 3 Logo(code)' ),
			array( 'fa fa-btc' => 'Bitcoin (BTC)(bitcoin)' ),
			array( 'fa fa-youtube-square' => 'YouTube Square(video, film)' ),
			array( 'fa fa-youtube' => 'YouTube(video, film)' ),
			array( 'fa fa-xing' => 'Xing' ),
			array( 'fa fa-xing-square' => 'Xing Square' ),
			array( 'fa fa-youtube-play' => 'YouTube Play(start, playing)' ),
			array( 'fa fa-dropbox' => 'Dropbox' ),
			array( 'fa fa-stack-overflow' => 'Stack Overflow' ),
			array( 'fa fa-instagram' => 'Instagram' ),
			array( 'fa fa-flickr' => 'Flickr' ),
			array( 'fa fa-adn' => 'App.net' ),
			array( 'fa fa-bitbucket' => 'Bitbucket(git)' ),
			array( 'fa fa-bitbucket-square' => 'Bitbucket Square(git)' ),
			array( 'fa fa-tumblr' => 'Tumblr' ),
			array( 'fa fa-tumblr-square' => 'Tumblr Square' ),
			array( 'fa fa-apple' => 'Apple(osx, food)' ),
			array( 'fa fa-windows' => 'Windows(microsoft)' ),
			array( 'fa fa-android' => 'Android(robot)' ),
			array( 'fa fa-linux' => 'Linux(tux)' ),
			array( 'fa fa-dribbble' => 'Dribbble' ),
			array( 'fa fa-skype' => 'Skype' ),
			array( 'fa fa-foursquare' => 'Foursquare' ),
			array( 'fa fa-trello' => 'Trello' ),
			array( 'fa fa-gratipay' => 'Gratipay (Gittip)(heart, like, favorite, love)(gittip)' ),
			array( 'fa fa-vk' => 'VK' ),
			array( 'fa fa-weibo' => 'Weibo' ),
			array( 'fa fa-renren' => 'Renren' ),
			array( 'fa fa-pagelines' => 'Pagelines(leaf, leaves, tree, plant, eco, nature)' ),
			array( 'fa fa-stack-exchange' => 'Stack Exchange' ),
			array( 'fa fa-vimeo-square' => 'Vimeo Square' ),
			array( 'fa fa-slack' => 'Slack Logo(hashtag, anchor, hash)' ),
			array( 'fa fa-wordpress' => 'WordPress Logo' ),
			array( 'fa fa-openid' => 'OpenID' ),
			array( 'fa fa-yahoo' => 'Yahoo Logo' ),
			array( 'fa fa-google' => 'Google Logo' ),
			array( 'fa fa-reddit' => 'reddit Logo' ),
			array( 'fa fa-reddit-square' => 'reddit Square' ),
			array( 'fa fa-stumbleupon-circle' => 'StumbleUpon Circle' ),
			array( 'fa fa-stumbleupon' => 'StumbleUpon Logo' ),
			array( 'fa fa-delicious' => 'Delicious Logo' ),
			array( 'fa fa-digg' => 'Digg Logo' ),
			array( 'fa fa-pied-piper-pp' => 'Pied Piper PP Logo (Old)' ),
			array( 'fa fa-pied-piper-alt' => 'Pied Piper Alternate Logo' ),
			array( 'fa fa-drupal' => 'Drupal Logo' ),
			array( 'fa fa-joomla' => 'Joomla Logo' ),
			array( 'fa fa-behance' => 'Behance' ),
			array( 'fa fa-behance-square' => 'Behance Square' ),
			array( 'fa fa-steam' => 'Steam' ),
			array( 'fa fa-steam-square' => 'Steam Square' ),
			array( 'fa fa-spotify' => 'Spotify' ),
			array( 'fa fa-deviantart' => 'deviantART' ),
			array( 'fa fa-soundcloud' => 'SoundCloud' ),
			array( 'fa fa-vine' => 'Vine' ),
			array( 'fa fa-codepen' => 'Codepen' ),
			array( 'fa fa-jsfiddle' => 'jsFiddle' ),
			array( 'fa fa-rebel' => 'Rebel Alliance(ra, resistance)' ),
			array( 'fa fa-empire' => 'Galactic Empire(ge)' ),
			array( 'fa fa-git-square' => 'Git Square' ),
			array( 'fa fa-git' => 'Git' ),
			array( 'fa fa-hacker-news' => 'Hacker News(y-combinator-square, yc-square)' ),
			array( 'fa fa-tencent-weibo' => 'Tencent Weibo' ),
			array( 'fa fa-qq' => 'QQ' ),
			array( 'fa fa-weixin' => 'Weixin (WeChat)(wechat)' ),
			array( 'fa fa-share-alt' => 'Share Alt' ),
			array( 'fa fa-share-alt-square' => 'Share Alt Square' ),
			array( 'fa fa-slideshare' => 'Slideshare' ),
			array( 'fa fa-twitch' => 'Twitch' ),
			array( 'fa fa-yelp' => 'Yelp' ),
			array( 'fa fa-paypal' => 'Paypal' ),
			array( 'fa fa-google-wallet' => 'Google Wallet' ),
			array( 'fa fa-cc-visa' => 'Visa Credit Card' ),
			array( 'fa fa-cc-mastercard' => 'MasterCard Credit Card' ),
			array( 'fa fa-cc-discover' => 'Discover Credit Card' ),
			array( 'fa fa-cc-amex' => 'American Express Credit Card(amex)' ),
			array( 'fa fa-cc-paypal' => 'Paypal Credit Card' ),
			array( 'fa fa-cc-stripe' => 'Stripe Credit Card' ),
			array( 'fa fa-lastfm' => 'last.fm' ),
			array( 'fa fa-lastfm-square' => 'last.fm Square' ),
			array( 'fa fa-ioxhost' => 'ioxhost' ),
			array( 'fa fa-angellist' => 'AngelList' ),
			array( 'fa fa-meanpath' => 'meanpath' ),
			array( 'fa fa-buysellads' => 'BuySellAds' ),
			array( 'fa fa-connectdevelop' => 'Connect Develop' ),
			array( 'fa fa-dashcube' => 'DashCube' ),
			array( 'fa fa-forumbee' => 'Forumbee' ),
			array( 'fa fa-leanpub' => 'Leanpub' ),
			array( 'fa fa-sellsy' => 'Sellsy' ),
			array( 'fa fa-shirtsinbulk' => 'Shirts in Bulk' ),
			array( 'fa fa-simplybuilt' => 'SimplyBuilt' ),
			array( 'fa fa-skyatlas' => 'skyatlas' ),
			array( 'fa fa-facebook-official' => 'Facebook Official' ),
			array( 'fa fa-pinterest-p' => 'Pinterest P' ),
			array( 'fa fa-whatsapp' => 'What\'s App' ),
			array( 'fa fa-viacoin' => 'Viacoin' ),
			array( 'fa fa-medium' => 'Medium' ),
			array( 'fa fa-y-combinator' => 'Y Combinator(yc)' ),
			array( 'fa fa-optin-monster' => 'Optin Monster' ),
			array( 'fa fa-opencart' => 'OpenCart' ),
			array( 'fa fa-expeditedssl' => 'ExpeditedSSL' ),
			array( 'fa fa-cc-jcb' => 'JCB Credit Card' ),
			array( 'fa fa-cc-diners-club' => 'Diner\'s Club Credit Card' ),
			array( 'fa fa-gg' => 'GG Currency' ),
			array( 'fa fa-gg-circle' => 'GG Currency Circle' ),
			array( 'fa fa-tripadvisor' => 'TripAdvisor' ),
			array( 'fa fa-odnoklassniki' => 'Odnoklassniki' ),
			array( 'fa fa-odnoklassniki-square' => 'Odnoklassniki Square' ),
			array( 'fa fa-get-pocket' => 'Get Pocket' ),
			array( 'fa fa-wikipedia-w' => 'Wikipedia W' ),
			array( 'fa fa-safari' => 'Safari(browser)' ),
			array( 'fa fa-chrome' => 'Chrome(browser)' ),
			array( 'fa fa-firefox' => 'Firefox(browser)' ),
			array( 'fa fa-opera' => 'Opera' ),
			array( 'fa fa-internet-explorer' => 'Internet-explorer(browser, ie)' ),
			array( 'fa fa-contao' => 'Contao' ),
			array( 'fa fa-500px' => '500px' ),
			array( 'fa fa-amazon' => 'Amazon' ),
			array( 'fa fa-houzz' => 'Houzz' ),
			array( 'fa fa-vimeo' => 'Vimeo' ),
			array( 'fa fa-black-tie' => 'Font Awesome Black Tie' ),
			array( 'fa fa-fonticons' => 'Fonticons' ),
			array( 'fa fa-reddit-alien' => 'reddit Alien' ),
			array( 'fa fa-edge' => 'Edge Browser(browser, ie)' ),
			array( 'fa fa-codiepie' => 'Codie Pie' ),
			array( 'fa fa-modx' => 'MODX' ),
			array( 'fa fa-fort-awesome' => 'Fort Awesome' ),
			array( 'fa fa-usb' => 'USB' ),
			array( 'fa fa-product-hunt' => 'Product Hunt' ),
			array( 'fa fa-mixcloud' => 'Mixcloud' ),
			array( 'fa fa-scribd' => 'Scribd' ),
			array( 'fa fa-bluetooth' => 'Bluetooth' ),
			array( 'fa fa-bluetooth-b' => 'Bluetooth' ),
			array( 'fa fa-gitlab' => 'GitLab' ),
			array( 'fa fa-wpbeginner' => 'WPBeginner' ),
			array( 'fa fa-wpforms' => 'WPForms' ),
			array( 'fa fa-envira' => 'Envira Gallery(leaf)' ),
			array( 'fa fa-glide' => 'Glide' ),
			array( 'fa fa-glide-g' => 'Glide G' ),
			array( 'fa fa-viadeo' => 'Viadeo' ),
			array( 'fa fa-viadeo-square' => 'Viadeo Square' ),
			array( 'fa fa-snapchat' => 'Snapchat' ),
			array( 'fa fa-snapchat-ghost' => 'Snapchat Ghost' ),
			array( 'fa fa-snapchat-square' => 'Snapchat Square' ),
			array( 'fa fa-pied-piper' => 'Pied Piper Logo' ),
			array( 'fa fa-first-order' => 'First Order' ),
			array( 'fa fa-yoast' => 'Yoast' ),
			array( 'fa fa-themeisle' => 'ThemeIsle' ),
			array( 'fa fa-google-plus-official' => 'Google Plus Official(google-plus-circle)' ),
			array( 'fa fa-font-awesome' => 'Font Awesome(fa)' ),
			array( 'fa fa-linode' => 'Linode' ),
			array( 'fa fa-quora' => 'Quora' ),
			array( 'fa fa-free-code-camp' => 'Free Code Camp' ),
			array( 'fa fa-telegram' => 'Telegram' ),
			array( 'fa fa-bandcamp' => 'Bandcamp' ),
			array( 'fa fa-grav' => 'Grav' ),
			array( 'fa fa-etsy' => 'Etsy' ),
			array( 'fa fa-imdb' => 'IMDB' ),
			array( 'fa fa-ravelry' => 'Ravelry' ),
			array( 'fa fa-eercast' => 'Eercast' ),
			array( 'fa fa-superpowers' => 'Superpowers' ),
			array( 'fa fa-wpexplorer' => 'WPExplorer' ),
			array( 'fa fa-meetup' => 'Meetup' ),
		),
		'Hand Icons' => array(
			array( 'fa fa-thumbs-o-up' => 'Thumbs Up Outlined(like, approve, favorite, agree, hand)' ),
			array( 'fa fa-thumbs-o-down' => 'Thumbs Down Outlined(dislike, disapprove, disagree, hand)' ),
			array( 'fa fa-hand-o-right' => 'Hand Outlined Right(point, right, next, forward, finger)' ),
			array( 'fa fa-hand-o-left' => 'Hand Outlined Left(point, left, previous, back, finger)' ),
			array( 'fa fa-hand-o-up' => 'Hand Outlined Up(point, finger)' ),
			array( 'fa fa-hand-o-down' => 'Hand Outlined Down(point, finger)' ),
			array( 'fa fa-thumbs-up' => 'thumbs-up(like, favorite, approve, agree, hand)' ),
			array( 'fa fa-thumbs-down' => 'thumbs-down(dislike, disapprove, disagree, hand)' ),
			array( 'fa fa-hand-rock-o' => 'Rock (Hand)(hand-grab-o)' ),
			array( 'fa fa-hand-paper-o' => 'Paper (Hand)(stop)(hand-stop-o)' ),
			array( 'fa fa-hand-scissors-o' => 'Scissors (Hand)' ),
			array( 'fa fa-hand-lizard-o' => 'Lizard (Hand)' ),
			array( 'fa fa-hand-spock-o' => 'Spock (Hand)' ),
			array( 'fa fa-hand-pointer-o' => 'Hand Pointer' ),
			array( 'fa fa-hand-peace-o' => 'Hand Peace' ),
		),
		'Payment Icons' => array(
			array( 'fa fa-credit-card' => 'credit-card(money, buy, debit, checkout, purchase, payment)' ),
			array( 'fa fa-paypal' => 'Paypal' ),
			array( 'fa fa-google-wallet' => 'Google Wallet' ),
			array( 'fa fa-cc-visa' => 'Visa Credit Card' ),
			array( 'fa fa-cc-mastercard' => 'MasterCard Credit Card' ),
			array( 'fa fa-cc-discover' => 'Discover Credit Card' ),
			array( 'fa fa-cc-amex' => 'American Express Credit Card(amex)' ),
			array( 'fa fa-cc-paypal' => 'Paypal Credit Card' ),
			array( 'fa fa-cc-stripe' => 'Stripe Credit Card' ),
			array( 'fa fa-cc-jcb' => 'JCB Credit Card' ),
			array( 'fa fa-cc-diners-club' => 'Diner\'s Club Credit Card' ),
			array( 'fa fa-credit-card-alt' => 'Credit Card(money, buy, debit, checkout, purchase, payment, credit card)' ),
		),
		'Currency Icons' => array(
			array( 'fa fa-money' => 'Money(cash, money, buy, checkout, purchase, payment)' ),
			array( 'fa fa-eur' => 'Euro (EUR)(euro)' ),
			array( 'fa fa-gbp' => 'GBP' ),
			array( 'fa fa-usd' => 'US Dollar(dollar)' ),
			array( 'fa fa-inr' => 'Indian Rupee (INR)(rupee)' ),
			array( 'fa fa-jpy' => 'Japanese Yen (JPY)(cny, rmb, yen)' ),
			array( 'fa fa-rub' => 'Russian Ruble (RUB)(ruble, rouble)' ),
			array( 'fa fa-krw' => 'Korean Won (KRW)(won)' ),
			array( 'fa fa-btc' => 'Bitcoin (BTC)(bitcoin)' ),
			array( 'fa fa-try' => 'Turkish Lira (TRY)(turkish-lira)' ),
			array( 'fa fa-ils' => 'Shekel (ILS)(shekel, sheqel)' ),
			array( 'fa fa-gg' => 'GG Currency' ),
			array( 'fa fa-gg-circle' => 'GG Currency Circle' ),
		),
		'Accessibility Icons' => array(
			array( 'fa fa-wheelchair' => 'Wheelchair(handicap, person)' ),
			array( 'fa fa-tty' => 'TTY' ),
			array( 'fa fa-cc' => 'Closed Captions' ),
			array( 'fa fa-universal-access' => 'Universal Access' ),
			array( 'fa fa-wheelchair-alt' => 'Wheelchair Alt(handicap, person)' ),
			array( 'fa fa-question-circle-o' => 'Question Circle Outlined' ),
			array( 'fa fa-blind' => 'Blind' ),
			array( 'fa fa-audio-description' => 'Audio Description' ),
			array( 'fa fa-volume-control-phone' => 'Volume Control Phone(telephone)' ),
			array( 'fa fa-braille' => 'Braille' ),
			array( 'fa fa-assistive-listening-systems' => 'Assistive Listening Systems' ),
			array( 'fa fa-american-sign-language-interpreting' => 'American Sign Language Interpreting(asl-interpreting)' ),
			array( 'fa fa-deaf' => 'Deaf(deafness, hard-of-hearing)' ),
			array( 'fa fa-sign-language' => 'Sign Language(signing)' ),
			array( 'fa fa-low-vision' => 'Low Vision' ),
		),
		'Gender Icons' => array(
			array( 'fa fa-venus' => 'Venus(female)' ),
			array( 'fa fa-mars' => 'Mars(male)' ),
			array( 'fa fa-mercury' => 'Mercury(transgender)' ),
			array( 'fa fa-transgender' => 'Transgender(intersex)' ),
			array( 'fa fa-transgender-alt' => 'Transgender Alt' ),
			array( 'fa fa-venus-double' => 'Venus Double' ),
			array( 'fa fa-mars-double' => 'Mars Double' ),
			array( 'fa fa-venus-mars' => 'Venus Mars' ),
			array( 'fa fa-mars-stroke' => 'Mars Stroke' ),
			array( 'fa fa-mars-stroke-v' => 'Mars Stroke Vertical' ),
			array( 'fa fa-mars-stroke-h' => 'Mars Stroke Horizontal' ),
			array( 'fa fa-neuter' => 'Neuter' ),
			array( 'fa fa-genderless' => 'Genderless' ),
		),
	);

	return array_merge( $icons, $fontawesome_icons );
}

add_filter( 'vc_iconpicker-type-openiconic', 'vc_iconpicker_type_openiconic' );

/**
 * Openicons icons from fontello.com
 *
 * @param $icons - taken from filter - vc_map param field settings['source']
 *     provided icons (default empty array). If array categorized it will
 *     auto-enable category dropdown
 *
 * @since 4.4
 * @return array - of icons for iconpicker, can be categorized, or not.
 */
function vc_iconpicker_type_openiconic( $icons ) {
	$openiconic_icons = [
		[ 'vc-oi vc-oi-dial' => 'Dial' ],
		[ 'vc-oi vc-oi-pilcrow' => 'Pilcrow' ],
		[ 'vc-oi vc-oi-at' => 'At' ],
		[ 'vc-oi vc-oi-hash' => 'Hash' ],
		[ 'vc-oi vc-oi-key-inv' => 'Key-inv' ],
		[ 'vc-oi vc-oi-key' => 'Key' ],
		[ 'vc-oi vc-oi-chart-pie-alt' => 'Chart-pie-alt' ],
		[ 'vc-oi vc-oi-chart-pie' => 'Chart-pie' ],
		[ 'vc-oi vc-oi-chart-bar' => 'Chart-bar' ],
		[ 'vc-oi vc-oi-umbrella' => 'Umbrella' ],
		[ 'vc-oi vc-oi-moon-inv' => 'Moon-inv' ],
		[ 'vc-oi vc-oi-mobile' => 'Mobile' ],
		[ 'vc-oi vc-oi-cd' => 'Cd' ],
		[ 'vc-oi vc-oi-split' => 'Split' ],
		[ 'vc-oi vc-oi-exchange' => 'Exchange' ],
		[ 'vc-oi vc-oi-block' => 'Block' ],
		[ 'vc-oi vc-oi-resize-full' => 'Resize-full' ],
		[ 'vc-oi vc-oi-article-alt' => 'Article-alt' ],
		[ 'vc-oi vc-oi-article' => 'Article' ],
		[ 'vc-oi vc-oi-pencil-alt' => 'Pencil-alt' ],
		[ 'vc-oi vc-oi-undo' => 'Undo' ],
		[ 'vc-oi vc-oi-attach' => 'Attach' ],
		[ 'vc-oi vc-oi-link' => 'Link' ],
		[ 'vc-oi vc-oi-search' => 'Search' ],
		[ 'vc-oi vc-oi-mail' => 'Mail' ],
		[ 'vc-oi vc-oi-heart' => 'Heart' ],
		[ 'vc-oi vc-oi-comment' => 'Comment' ],
		[ 'vc-oi vc-oi-resize-full-alt' => 'Resize-full-alt' ],
		[ 'vc-oi vc-oi-lock' => 'Lock' ],
		[ 'vc-oi vc-oi-book-open' => 'Book-open' ],
		[ 'vc-oi vc-oi-arrow-curved' => 'Arrow-curved' ],
		[ 'vc-oi vc-oi-equalizer' => 'Equalizer' ],
		[ 'vc-oi vc-oi-heart-empty' => 'Heart-empty' ],
		[ 'vc-oi vc-oi-lock-empty' => 'Lock-empty' ],
		[ 'vc-oi vc-oi-comment-inv' => 'Comment-inv' ],
		[ 'vc-oi vc-oi-folder' => 'Folder' ],
		[ 'vc-oi vc-oi-resize-small' => 'Resize-small' ],
		[ 'vc-oi vc-oi-play' => 'Play' ],
		[ 'vc-oi vc-oi-cursor' => 'Cursor' ],
		[ 'vc-oi vc-oi-aperture' => 'Aperture' ],
		[ 'vc-oi vc-oi-play-circle2' => 'Play-circle2' ],
		[ 'vc-oi vc-oi-resize-small-alt' => 'Resize-small-alt' ],
		[ 'vc-oi vc-oi-folder-empty' => 'Folder-empty' ],
		[ 'vc-oi vc-oi-comment-alt' => 'Comment-alt' ],
		[ 'vc-oi vc-oi-lock-open' => 'Lock-open' ],
		[ 'vc-oi vc-oi-star' => 'Star' ],
		[ 'vc-oi vc-oi-user' => 'User' ],
		[ 'vc-oi vc-oi-lock-open-empty' => 'Lock-open-empty' ],
		[ 'vc-oi vc-oi-box' => 'Box' ],
		[ 'vc-oi vc-oi-resize-vertical' => 'Resize-vertical' ],
		[ 'vc-oi vc-oi-stop' => 'Stop' ],
		[ 'vc-oi vc-oi-aperture-alt' => 'Aperture-alt' ],
		[ 'vc-oi vc-oi-book' => 'Book' ],
		[ 'vc-oi vc-oi-steering-wheel' => 'Steering-wheel' ],
		[ 'vc-oi vc-oi-pause' => 'Pause' ],
		[ 'vc-oi vc-oi-to-start' => 'To-start' ],
		[ 'vc-oi vc-oi-move' => 'Move' ],
		[ 'vc-oi vc-oi-resize-horizontal' => 'Resize-horizontal' ],
		[ 'vc-oi vc-oi-rss-alt' => 'Rss-alt' ],
		[ 'vc-oi vc-oi-comment-alt2' => 'Comment-alt2' ],
		[ 'vc-oi vc-oi-rss' => 'Rss' ],
		[ 'vc-oi vc-oi-comment-inv-alt' => 'Comment-inv-alt' ],
		[ 'vc-oi vc-oi-comment-inv-alt2' => 'Comment-inv-alt2' ],
		[ 'vc-oi vc-oi-eye' => 'Eye' ],
		[ 'vc-oi vc-oi-pin' => 'Pin' ],
		[ 'vc-oi vc-oi-video' => 'Video' ],
		[ 'vc-oi vc-oi-picture' => 'Picture' ],
		[ 'vc-oi vc-oi-camera' => 'Camera' ],
		[ 'vc-oi vc-oi-tag' => 'Tag' ],
		[ 'vc-oi vc-oi-chat' => 'Chat' ],
		[ 'vc-oi vc-oi-cog' => 'Cog' ],
		[ 'vc-oi vc-oi-popup' => 'Popup' ],
		[ 'vc-oi vc-oi-to-end' => 'To-end' ],
		[ 'vc-oi vc-oi-book-alt' => 'Book-alt' ],
		[ 'vc-oi vc-oi-brush' => 'Brush' ],
		[ 'vc-oi vc-oi-eject' => 'Eject' ],
		[ 'vc-oi vc-oi-down' => 'Down' ],
		[ 'vc-oi vc-oi-wrench' => 'Wrench' ],
		[ 'vc-oi vc-oi-chat-inv' => 'Chat-inv' ],
		[ 'vc-oi vc-oi-tag-empty' => 'Tag-empty' ],
		[ 'vc-oi vc-oi-ok' => 'Ok' ],
		[ 'vc-oi vc-oi-ok-circle' => 'Ok-circle' ],
		[ 'vc-oi vc-oi-download' => 'Download' ],
		[ 'vc-oi vc-oi-location' => 'Location' ],
		[ 'vc-oi vc-oi-share' => 'Share' ],
		[ 'vc-oi vc-oi-left' => 'Left' ],
		[ 'vc-oi vc-oi-target' => 'Target' ],
		[ 'vc-oi vc-oi-brush-alt' => 'Brush-alt' ],
		[ 'vc-oi vc-oi-cancel' => 'Cancel' ],
		[ 'vc-oi vc-oi-upload' => 'Upload' ],
		[ 'vc-oi vc-oi-location-inv' => 'Location-inv' ],
		[ 'vc-oi vc-oi-calendar' => 'Calendar' ],
		[ 'vc-oi vc-oi-right' => 'Right' ],
		[ 'vc-oi vc-oi-signal' => 'Signal' ],
		[ 'vc-oi vc-oi-eyedropper' => 'Eyedropper' ],
		[ 'vc-oi vc-oi-layers' => 'Layers' ],
		[ 'vc-oi vc-oi-award' => 'Award' ],
		[ 'vc-oi vc-oi-up' => 'Up' ],
		[ 'vc-oi vc-oi-calendar-inv' => 'Calendar-inv' ],
		[ 'vc-oi vc-oi-location-alt' => 'Location-alt' ],
		[ 'vc-oi vc-oi-download-cloud' => 'Download-cloud' ],
		[ 'vc-oi vc-oi-cancel-circle' => 'Cancel-circle' ],
		[ 'vc-oi vc-oi-plus' => 'Plus' ],
		[ 'vc-oi vc-oi-upload-cloud' => 'Upload-cloud' ],
		[ 'vc-oi vc-oi-compass' => 'Compass' ],
		[ 'vc-oi vc-oi-calendar-alt' => 'Calendar-alt' ],
		[ 'vc-oi vc-oi-down-circle' => 'Down-circle' ],
		[ 'vc-oi vc-oi-award-empty' => 'Award-empty' ],
		[ 'vc-oi vc-oi-layers-alt' => 'Layers-alt' ],
		[ 'vc-oi vc-oi-sun' => 'Sun' ],
		[ 'vc-oi vc-oi-list' => 'List' ],
		[ 'vc-oi vc-oi-left-circle' => 'Left-circle' ],
		[ 'vc-oi vc-oi-mic' => 'Mic' ],
		[ 'vc-oi vc-oi-trash' => 'Trash' ],
		[ 'vc-oi vc-oi-quote-left' => 'Quote-left' ],
		[ 'vc-oi vc-oi-plus-circle' => 'Plus-circle' ],
		[ 'vc-oi vc-oi-minus' => 'Minus' ],
		[ 'vc-oi vc-oi-quote-right' => 'Quote-right' ],
		[ 'vc-oi vc-oi-trash-empty' => 'Trash-empty' ],
		[ 'vc-oi vc-oi-volume-off' => 'Volume-off' ],
		[ 'vc-oi vc-oi-right-circle' => 'Right-circle' ],
		[ 'vc-oi vc-oi-list-nested' => 'List-nested' ],
		[ 'vc-oi vc-oi-sun-inv' => 'Sun-inv' ],
		[ 'vc-oi vc-oi-bat-empty' => 'Bat-empty' ],
		[ 'vc-oi vc-oi-up-circle' => 'Up-circle' ],
		[ 'vc-oi vc-oi-volume-up' => 'Volume-up' ],
		[ 'vc-oi vc-oi-doc' => 'Doc' ],
		[ 'vc-oi vc-oi-quote-left-alt' => 'Quote-left-alt' ],
		[ 'vc-oi vc-oi-minus-circle' => 'Minus-circle' ],
		[ 'vc-oi vc-oi-cloud' => 'Cloud' ],
		[ 'vc-oi vc-oi-rain' => 'Rain' ],
		[ 'vc-oi vc-oi-bat-half' => 'Bat-half' ],
		[ 'vc-oi vc-oi-cw' => 'Cw' ],
		[ 'vc-oi vc-oi-headphones' => 'Headphones' ],
		[ 'vc-oi vc-oi-doc-inv' => 'Doc-inv' ],
		[ 'vc-oi vc-oi-quote-right-alt' => 'Quote-right-alt' ],
		[ 'vc-oi vc-oi-help' => 'Help' ],
		[ 'vc-oi vc-oi-info' => 'Info' ],
		[ 'vc-oi vc-oi-pencil' => 'Pencil' ],
		[ 'vc-oi vc-oi-doc-alt' => 'Doc-alt' ],
		[ 'vc-oi vc-oi-clock' => 'Clock' ],
		[ 'vc-oi vc-oi-loop' => 'Loop' ],
		[ 'vc-oi vc-oi-bat-full' => 'Bat-full' ],
		[ 'vc-oi vc-oi-flash' => 'Flash' ],
		[ 'vc-oi vc-oi-moon' => 'Moon' ],
		[ 'vc-oi vc-oi-bat-charge' => 'Bat-charge' ],
		[ 'vc-oi vc-oi-loop-alt' => 'Loop-alt' ],
		[ 'vc-oi vc-oi-lamp' => 'Lamp' ],
		[ 'vc-oi vc-oi-doc-inv-alt' => 'Doc-inv-alt' ],
		[ 'vc-oi vc-oi-pencil-neg' => 'Pencil-neg' ],
		[ 'vc-oi vc-oi-home' => 'Home' ],
	];

	return array_merge( $icons, $openiconic_icons );
}

add_filter( 'vc_iconpicker-type-typicons', 'vc_iconpicker_type_typicons' );

/**
 * Typicons icons from github.com/stephenhutchings/typicons.font
 *
 * @param $icons - taken from filter - vc_map param field settings['source']
 *     provided icons (default empty array). If array categorized it will
 *     auto-enable category dropdown
 *
 * @since 4.4
 * @return array - of icons for iconpicker, can be categorized, or not.
 */
function vc_iconpicker_type_typicons( $icons ) {
	$typicons_icons = [
		[ 'typcn typcn-adjust-brightness' => 'Adjust Brightness' ],
		[ 'typcn typcn-adjust-contrast' => 'Adjust Contrast' ],
		[ 'typcn typcn-anchor-outline' => 'Anchor Outline' ],
		[ 'typcn typcn-anchor' => 'Anchor' ],
		[ 'typcn typcn-archive' => 'Archive' ],
		[ 'typcn typcn-arrow-back-outline' => 'Arrow Back Outline' ],
		[ 'typcn typcn-arrow-back' => 'Arrow Back' ],
		[ 'typcn typcn-arrow-down-outline' => 'Arrow Down Outline' ],
		[ 'typcn typcn-arrow-down-thick' => 'Arrow Down Thick' ],
		[ 'typcn typcn-arrow-down' => 'Arrow Down' ],
		[ 'typcn typcn-arrow-forward-outline' => 'Arrow Forward Outline' ],
		[ 'typcn typcn-arrow-forward' => 'Arrow Forward' ],
		[ 'typcn typcn-arrow-left-outline' => 'Arrow Left Outline' ],
		[ 'typcn typcn-arrow-left-thick' => 'Arrow Left Thick' ],
		[ 'typcn typcn-arrow-left' => 'Arrow Left' ],
		[ 'typcn typcn-arrow-loop-outline' => 'Arrow Loop Outline' ],
		[ 'typcn typcn-arrow-loop' => 'Arrow Loop' ],
		[ 'typcn typcn-arrow-maximise-outline' => 'Arrow Maximise Outline' ],
		[ 'typcn typcn-arrow-maximise' => 'Arrow Maximise' ],
		[ 'typcn typcn-arrow-minimise-outline' => 'Arrow Minimise Outline' ],
		[ 'typcn typcn-arrow-minimise' => 'Arrow Minimise' ],
		[ 'typcn typcn-arrow-move-outline' => 'Arrow Move Outline' ],
		[ 'typcn typcn-arrow-move' => 'Arrow Move' ],
		[ 'typcn typcn-arrow-repeat-outline' => 'Arrow Repeat Outline' ],
		[ 'typcn typcn-arrow-repeat' => 'Arrow Repeat' ],
		[ 'typcn typcn-arrow-right-outline' => 'Arrow Right Outline' ],
		[ 'typcn typcn-arrow-right-thick' => 'Arrow Right Thick' ],
		[ 'typcn typcn-arrow-right' => 'Arrow Right' ],
		[ 'typcn typcn-arrow-shuffle' => 'Arrow Shuffle' ],
		[ 'typcn typcn-arrow-sorted-down' => 'Arrow Sorted Down' ],
		[ 'typcn typcn-arrow-sorted-up' => 'Arrow Sorted Up' ],
		[ 'typcn typcn-arrow-sync-outline' => 'Arrow Sync Outline' ],
		[ 'typcn typcn-arrow-sync' => 'Arrow Sync' ],
		[ 'typcn typcn-arrow-unsorted' => 'Arrow Unsorted' ],
		[ 'typcn typcn-arrow-up-outline' => 'Arrow Up Outline' ],
		[ 'typcn typcn-arrow-up-thick' => 'Arrow Up Thick' ],
		[ 'typcn typcn-arrow-up' => 'Arrow Up' ],
		[ 'typcn typcn-at' => 'At' ],
		[ 'typcn typcn-attachment-outline' => 'Attachment Outline' ],
		[ 'typcn typcn-attachment' => 'Attachment' ],
		[ 'typcn typcn-backspace-outline' => 'Backspace Outline' ],
		[ 'typcn typcn-backspace' => 'Backspace' ],
		[ 'typcn typcn-battery-charge' => 'Battery Charge' ],
		[ 'typcn typcn-battery-full' => 'Battery Full' ],
		[ 'typcn typcn-battery-high' => 'Battery High' ],
		[ 'typcn typcn-battery-low' => 'Battery Low' ],
		[ 'typcn typcn-battery-mid' => 'Battery Mid' ],
		[ 'typcn typcn-beaker' => 'Beaker' ],
		[ 'typcn typcn-beer' => 'Beer' ],
		[ 'typcn typcn-bell' => 'Bell' ],
		[ 'typcn typcn-book' => 'Book' ],
		[ 'typcn typcn-bookmark' => 'Bookmark' ],
		[ 'typcn typcn-briefcase' => 'Briefcase' ],
		[ 'typcn typcn-brush' => 'Brush' ],
		[ 'typcn typcn-business-card' => 'Business Card' ],
		[ 'typcn typcn-calculator' => 'Calculator' ],
		[ 'typcn typcn-calendar-outline' => 'Calendar Outline' ],
		[ 'typcn typcn-calendar' => 'Calendar' ],
		[ 'typcn typcn-camera-outline' => 'Camera Outline' ],
		[ 'typcn typcn-camera' => 'Camera' ],
		[ 'typcn typcn-cancel-outline' => 'Cancel Outline' ],
		[ 'typcn typcn-cancel' => 'Cancel' ],
		[ 'typcn typcn-chart-area-outline' => 'Chart Area Outline' ],
		[ 'typcn typcn-chart-area' => 'Chart Area' ],
		[ 'typcn typcn-chart-bar-outline' => 'Chart Bar Outline' ],
		[ 'typcn typcn-chart-bar' => 'Chart Bar' ],
		[ 'typcn typcn-chart-line-outline' => 'Chart Line Outline' ],
		[ 'typcn typcn-chart-line' => 'Chart Line' ],
		[ 'typcn typcn-chart-pie-outline' => 'Chart Pie Outline' ],
		[ 'typcn typcn-chart-pie' => 'Chart Pie' ],
		[ 'typcn typcn-chevron-left-outline' => 'Chevron Left Outline' ],
		[ 'typcn typcn-chevron-left' => 'Chevron Left' ],
		[ 'typcn typcn-chevron-right-outline' => 'Chevron Right Outline' ],
		[ 'typcn typcn-chevron-right' => 'Chevron Right' ],
		[ 'typcn typcn-clipboard' => 'Clipboard' ],
		[ 'typcn typcn-cloud-storage' => 'Cloud Storage' ],
		[ 'typcn typcn-cloud-storage-outline' => 'Cloud Storage Outline' ],
		[ 'typcn typcn-code-outline' => 'Code Outline' ],
		[ 'typcn typcn-code' => 'Code' ],
		[ 'typcn typcn-coffee' => 'Coffee' ],
		[ 'typcn typcn-cog-outline' => 'Cog Outline' ],
		[ 'typcn typcn-cog' => 'Cog' ],
		[ 'typcn typcn-compass' => 'Compass' ],
		[ 'typcn typcn-contacts' => 'Contacts' ],
		[ 'typcn typcn-credit-card' => 'Credit Card' ],
		[ 'typcn typcn-css3' => 'Css3' ],
		[ 'typcn typcn-database' => 'Database' ],
		[ 'typcn typcn-delete-outline' => 'Delete Outline' ],
		[ 'typcn typcn-delete' => 'Delete' ],
		[ 'typcn typcn-device-desktop' => 'Device Desktop' ],
		[ 'typcn typcn-device-laptop' => 'Device Laptop' ],
		[ 'typcn typcn-device-phone' => 'Device Phone' ],
		[ 'typcn typcn-device-tablet' => 'Device Tablet' ],
		[ 'typcn typcn-directions' => 'Directions' ],
		[ 'typcn typcn-divide-outline' => 'Divide Outline' ],
		[ 'typcn typcn-divide' => 'Divide' ],
		[ 'typcn typcn-document-add' => 'Document Add' ],
		[ 'typcn typcn-document-delete' => 'Document Delete' ],
		[ 'typcn typcn-document-text' => 'Document Text' ],
		[ 'typcn typcn-document' => 'Document' ],
		[ 'typcn typcn-download-outline' => 'Download Outline' ],
		[ 'typcn typcn-download' => 'Download' ],
		[ 'typcn typcn-dropbox' => 'Dropbox' ],
		[ 'typcn typcn-edit' => 'Edit' ],
		[ 'typcn typcn-eject-outline' => 'Eject Outline' ],
		[ 'typcn typcn-eject' => 'Eject' ],
		[ 'typcn typcn-equals-outline' => 'Equals Outline' ],
		[ 'typcn typcn-equals' => 'Equals' ],
		[ 'typcn typcn-export-outline' => 'Export Outline' ],
		[ 'typcn typcn-export' => 'Export' ],
		[ 'typcn typcn-eye-outline' => 'Eye Outline' ],
		[ 'typcn typcn-eye' => 'Eye' ],
		[ 'typcn typcn-feather' => 'Feather' ],
		[ 'typcn typcn-film' => 'Film' ],
		[ 'typcn typcn-filter' => 'Filter' ],
		[ 'typcn typcn-flag-outline' => 'Flag Outline' ],
		[ 'typcn typcn-flag' => 'Flag' ],
		[ 'typcn typcn-flash-outline' => 'Flash Outline' ],
		[ 'typcn typcn-flash' => 'Flash' ],
		[ 'typcn typcn-flow-children' => 'Flow Children' ],
		[ 'typcn typcn-flow-merge' => 'Flow Merge' ],
		[ 'typcn typcn-flow-parallel' => 'Flow Parallel' ],
		[ 'typcn typcn-flow-switch' => 'Flow Switch' ],
		[ 'typcn typcn-folder-add' => 'Folder Add' ],
		[ 'typcn typcn-folder-delete' => 'Folder Delete' ],
		[ 'typcn typcn-folder-open' => 'Folder Open' ],
		[ 'typcn typcn-folder' => 'Folder' ],
		[ 'typcn typcn-gift' => 'Gift' ],
		[ 'typcn typcn-globe-outline' => 'Globe Outline' ],
		[ 'typcn typcn-globe' => 'Globe' ],
		[ 'typcn typcn-group-outline' => 'Group Outline' ],
		[ 'typcn typcn-group' => 'Group' ],
		[ 'typcn typcn-headphones' => 'Headphones' ],
		[ 'typcn typcn-heart-full-outline' => 'Heart Full Outline' ],
		[ 'typcn typcn-heart-half-outline' => 'Heart Half Outline' ],
		[ 'typcn typcn-heart-outline' => 'Heart Outline' ],
		[ 'typcn typcn-heart' => 'Heart' ],
		[ 'typcn typcn-home-outline' => 'Home Outline' ],
		[ 'typcn typcn-home' => 'Home' ],
		[ 'typcn typcn-html5' => 'Html5' ],
		[ 'typcn typcn-image-outline' => 'Image Outline' ],
		[ 'typcn typcn-image' => 'Image' ],
		[ 'typcn typcn-infinity-outline' => 'Infinity Outline' ],
		[ 'typcn typcn-infinity' => 'Infinity' ],
		[ 'typcn typcn-info-large-outline' => 'Info Large Outline' ],
		[ 'typcn typcn-info-large' => 'Info Large' ],
		[ 'typcn typcn-info-outline' => 'Info Outline' ],
		[ 'typcn typcn-info' => 'Info' ],
		[ 'typcn typcn-input-checked-outline' => 'Input Checked Outline' ],
		[ 'typcn typcn-input-checked' => 'Input Checked' ],
		[ 'typcn typcn-key-outline' => 'Key Outline' ],
		[ 'typcn typcn-key' => 'Key' ],
		[ 'typcn typcn-keyboard' => 'Keyboard' ],
		[ 'typcn typcn-leaf' => 'Leaf' ],
		[ 'typcn typcn-lightbulb' => 'Lightbulb' ],
		[ 'typcn typcn-link-outline' => 'Link Outline' ],
		[ 'typcn typcn-link' => 'Link' ],
		[ 'typcn typcn-location-arrow-outline' => 'Location Arrow Outline' ],
		[ 'typcn typcn-location-arrow' => 'Location Arrow' ],
		[ 'typcn typcn-location-outline' => 'Location Outline' ],
		[ 'typcn typcn-location' => 'Location' ],
		[ 'typcn typcn-lock-closed-outline' => 'Lock Closed Outline' ],
		[ 'typcn typcn-lock-closed' => 'Lock Closed' ],
		[ 'typcn typcn-lock-open-outline' => 'Lock Open Outline' ],
		[ 'typcn typcn-lock-open' => 'Lock Open' ],
		[ 'typcn typcn-mail' => 'Mail' ],
		[ 'typcn typcn-map' => 'Map' ],
		[ 'typcn typcn-media-eject-outline' => 'Media Eject Outline' ],
		[ 'typcn typcn-media-eject' => 'Media Eject' ],
		[ 'typcn typcn-media-fast-forward-outline' => 'Media Fast Forward Outline' ],
		[ 'typcn typcn-media-fast-forward' => 'Media Fast Forward' ],
		[ 'typcn typcn-media-pause-outline' => 'Media Pause Outline' ],
		[ 'typcn typcn-media-pause' => 'Media Pause' ],
		[ 'typcn typcn-media-play-outline' => 'Media Play Outline' ],
		[ 'typcn typcn-media-play-reverse-outline' => 'Media Play Reverse Outline' ],
		[ 'typcn typcn-media-play-reverse' => 'Media Play Reverse' ],
		[ 'typcn typcn-media-play' => 'Media Play' ],
		[ 'typcn typcn-media-record-outline' => 'Media Record Outline' ],
		[ 'typcn typcn-media-record' => 'Media Record' ],
		[ 'typcn typcn-media-rewind-outline' => 'Media Rewind Outline' ],
		[ 'typcn typcn-media-rewind' => 'Media Rewind' ],
		[ 'typcn typcn-media-stop-outline' => 'Media Stop Outline' ],
		[ 'typcn typcn-media-stop' => 'Media Stop' ],
		[ 'typcn typcn-message-typing' => 'Message Typing' ],
		[ 'typcn typcn-message' => 'Message' ],
		[ 'typcn typcn-messages' => 'Messages' ],
		[ 'typcn typcn-microphone-outline' => 'Microphone Outline' ],
		[ 'typcn typcn-microphone' => 'Microphone' ],
		[ 'typcn typcn-minus-outline' => 'Minus Outline' ],
		[ 'typcn typcn-minus' => 'Minus' ],
		[ 'typcn typcn-mortar-board' => 'Mortar Board' ],
		[ 'typcn typcn-news' => 'News' ],
		[ 'typcn typcn-notes-outline' => 'Notes Outline' ],
		[ 'typcn typcn-notes' => 'Notes' ],
		[ 'typcn typcn-pen' => 'Pen' ],
		[ 'typcn typcn-pencil' => 'Pencil' ],
		[ 'typcn typcn-phone-outline' => 'Phone Outline' ],
		[ 'typcn typcn-phone' => 'Phone' ],
		[ 'typcn typcn-pi-outline' => 'Pi Outline' ],
		[ 'typcn typcn-pi' => 'Pi' ],
		[ 'typcn typcn-pin-outline' => 'Pin Outline' ],
		[ 'typcn typcn-pin' => 'Pin' ],
		[ 'typcn typcn-pipette' => 'Pipette' ],
		[ 'typcn typcn-plane-outline' => 'Plane Outline' ],
		[ 'typcn typcn-plane' => 'Plane' ],
		[ 'typcn typcn-plug' => 'Plug' ],
		[ 'typcn typcn-plus-outline' => 'Plus Outline' ],
		[ 'typcn typcn-plus' => 'Plus' ],
		[ 'typcn typcn-point-of-interest-outline' => 'Point Of Interest Outline' ],
		[ 'typcn typcn-point-of-interest' => 'Point Of Interest' ],
		[ 'typcn typcn-power-outline' => 'Power Outline' ],
		[ 'typcn typcn-power' => 'Power' ],
		[ 'typcn typcn-printer' => 'Printer' ],
		[ 'typcn typcn-puzzle-outline' => 'Puzzle Outline' ],
		[ 'typcn typcn-puzzle' => 'Puzzle' ],
		[ 'typcn typcn-radar-outline' => 'Radar Outline' ],
		[ 'typcn typcn-radar' => 'Radar' ],
		[ 'typcn typcn-refresh-outline' => 'Refresh Outline' ],
		[ 'typcn typcn-refresh' => 'Refresh' ],
		[ 'typcn typcn-rss-outline' => 'Rss Outline' ],
		[ 'typcn typcn-rss' => 'Rss' ],
		[ 'typcn typcn-scissors-outline' => 'Scissors Outline' ],
		[ 'typcn typcn-scissors' => 'Scissors' ],
		[ 'typcn typcn-shopping-bag' => 'Shopping Bag' ],
		[ 'typcn typcn-shopping-cart' => 'Shopping Cart' ],
		[ 'typcn typcn-social-at-circular' => 'Social At Circular' ],
		[ 'typcn typcn-social-dribbble-circular' => 'Social Dribbble Circular' ],
		[ 'typcn typcn-social-dribbble' => 'Social Dribbble' ],
		[ 'typcn typcn-social-facebook-circular' => 'Social Facebook Circular' ],
		[ 'typcn typcn-social-facebook' => 'Social Facebook' ],
		[ 'typcn typcn-social-flickr-circular' => 'Social Flickr Circular' ],
		[ 'typcn typcn-social-flickr' => 'Social Flickr' ],
		[ 'typcn typcn-social-github-circular' => 'Social Github Circular' ],
		[ 'typcn typcn-social-github' => 'Social Github' ],
		[ 'typcn typcn-social-google-plus-circular' => 'Social Google Plus Circular' ],
		[ 'typcn typcn-social-google-plus' => 'Social Google Plus' ],
		[ 'typcn typcn-social-instagram-circular' => 'Social Instagram Circular' ],
		[ 'typcn typcn-social-instagram' => 'Social Instagram' ],
		[ 'typcn typcn-social-last-fm-circular' => 'Social Last Fm Circular' ],
		[ 'typcn typcn-social-last-fm' => 'Social Last Fm' ],
		[ 'typcn typcn-social-linkedin-circular' => 'Social Linkedin Circular' ],
		[ 'typcn typcn-social-linkedin' => 'Social Linkedin' ],
		[ 'typcn typcn-social-pinterest-circular' => 'Social Pinterest Circular' ],
		[ 'typcn typcn-social-pinterest' => 'Social Pinterest' ],
		[ 'typcn typcn-social-skype-outline' => 'Social Skype Outline' ],
		[ 'typcn typcn-social-skype' => 'Social Skype' ],
		[ 'typcn typcn-social-tumbler-circular' => 'Social Tumbler Circular' ],
		[ 'typcn typcn-social-tumbler' => 'Social Tumbler' ],
		[ 'typcn typcn-social-twitter-circular' => 'Social Twitter Circular' ],
		[ 'typcn typcn-social-twitter' => 'Social Twitter' ],
		[ 'typcn typcn-social-vimeo-circular' => 'Social Vimeo Circular' ],
		[ 'typcn typcn-social-vimeo' => 'Social Vimeo' ],
		[ 'typcn typcn-social-youtube-circular' => 'Social Youtube Circular' ],
		[ 'typcn typcn-social-youtube' => 'Social Youtube' ],
		[ 'typcn typcn-sort-alphabetically-outline' => 'Sort Alphabetically Outline' ],
		[ 'typcn typcn-sort-alphabetically' => 'Sort Alphabetically' ],
		[ 'typcn typcn-sort-numerically-outline' => 'Sort Numerically Outline' ],
		[ 'typcn typcn-sort-numerically' => 'Sort Numerically' ],
		[ 'typcn typcn-spanner-outline' => 'Spanner Outline' ],
		[ 'typcn typcn-spanner' => 'Spanner' ],
		[ 'typcn typcn-spiral' => 'Spiral' ],
		[ 'typcn typcn-star-full-outline' => 'Star Full Outline' ],
		[ 'typcn typcn-star-half-outline' => 'Star Half Outline' ],
		[ 'typcn typcn-star-half' => 'Star Half' ],
		[ 'typcn typcn-star-outline' => 'Star Outline' ],
		[ 'typcn typcn-star' => 'Star' ],
		[ 'typcn typcn-starburst-outline' => 'Starburst Outline' ],
		[ 'typcn typcn-starburst' => 'Starburst' ],
		[ 'typcn typcn-stopwatch' => 'Stopwatch' ],
		[ 'typcn typcn-support' => 'Support' ],
		[ 'typcn typcn-tabs-outline' => 'Tabs Outline' ],
		[ 'typcn typcn-tag' => 'Tag' ],
		[ 'typcn typcn-tags' => 'Tags' ],
		[ 'typcn typcn-th-large-outline' => 'Th Large Outline' ],
		[ 'typcn typcn-th-large' => 'Th Large' ],
		[ 'typcn typcn-th-list-outline' => 'Th List Outline' ],
		[ 'typcn typcn-th-list' => 'Th List' ],
		[ 'typcn typcn-th-menu-outline' => 'Th Menu Outline' ],
		[ 'typcn typcn-th-menu' => 'Th Menu' ],
		[ 'typcn typcn-th-small-outline' => 'Th Small Outline' ],
		[ 'typcn typcn-th-small' => 'Th Small' ],
		[ 'typcn typcn-thermometer' => 'Thermometer' ],
		[ 'typcn typcn-thumbs-down' => 'Thumbs Down' ],
		[ 'typcn typcn-thumbs-ok' => 'Thumbs Ok' ],
		[ 'typcn typcn-thumbs-up' => 'Thumbs Up' ],
		[ 'typcn typcn-tick-outline' => 'Tick Outline' ],
		[ 'typcn typcn-tick' => 'Tick' ],
		[ 'typcn typcn-ticket' => 'Ticket' ],
		[ 'typcn typcn-time' => 'Time' ],
		[ 'typcn typcn-times-outline' => 'Times Outline' ],
		[ 'typcn typcn-times' => 'Times' ],
		[ 'typcn typcn-trash' => 'Trash' ],
		[ 'typcn typcn-tree' => 'Tree' ],
		[ 'typcn typcn-upload-outline' => 'Upload Outline' ],
		[ 'typcn typcn-upload' => 'Upload' ],
		[ 'typcn typcn-user-add-outline' => 'User Add Outline' ],
		[ 'typcn typcn-user-add' => 'User Add' ],
		[ 'typcn typcn-user-delete-outline' => 'User Delete Outline' ],
		[ 'typcn typcn-user-delete' => 'User Delete' ],
		[ 'typcn typcn-user-outline' => 'User Outline' ],
		[ 'typcn typcn-user' => 'User' ],
		[ 'typcn typcn-vendor-android' => 'Vendor Android' ],
		[ 'typcn typcn-vendor-apple' => 'Vendor Apple' ],
		[ 'typcn typcn-vendor-microsoft' => 'Vendor Microsoft' ],
		[ 'typcn typcn-video-outline' => 'Video Outline' ],
		[ 'typcn typcn-video' => 'Video' ],
		[ 'typcn typcn-volume-down' => 'Volume Down' ],
		[ 'typcn typcn-volume-mute' => 'Volume Mute' ],
		[ 'typcn typcn-volume-up' => 'Volume Up' ],
		[ 'typcn typcn-volume' => 'Volume' ],
		[ 'typcn typcn-warning-outline' => 'Warning Outline' ],
		[ 'typcn typcn-warning' => 'Warning' ],
		[ 'typcn typcn-watch' => 'Watch' ],
		[ 'typcn typcn-waves-outline' => 'Waves Outline' ],
		[ 'typcn typcn-waves' => 'Waves' ],
		[ 'typcn typcn-weather-cloudy' => 'Weather Cloudy' ],
		[ 'typcn typcn-weather-downpour' => 'Weather Downpour' ],
		[ 'typcn typcn-weather-night' => 'Weather Night' ],
		[ 'typcn typcn-weather-partly-sunny' => 'Weather Partly Sunny' ],
		[ 'typcn typcn-weather-shower' => 'Weather Shower' ],
		[ 'typcn typcn-weather-snow' => 'Weather Snow' ],
		[ 'typcn typcn-weather-stormy' => 'Weather Stormy' ],
		[ 'typcn typcn-weather-sunny' => 'Weather Sunny' ],
		[ 'typcn typcn-weather-windy-cloudy' => 'Weather Windy Cloudy' ],
		[ 'typcn typcn-weather-windy' => 'Weather Windy' ],
		[ 'typcn typcn-wi-fi-outline' => 'Wi Fi Outline' ],
		[ 'typcn typcn-wi-fi' => 'Wi Fi' ],
		[ 'typcn typcn-wine' => 'Wine' ],
		[ 'typcn typcn-world-outline' => 'World Outline' ],
		[ 'typcn typcn-world' => 'World' ],
		[ 'typcn typcn-zoom-in-outline' => 'Zoom In Outline' ],
		[ 'typcn typcn-zoom-in' => 'Zoom In' ],
		[ 'typcn typcn-zoom-out-outline' => 'Zoom Out Outline' ],
		[ 'typcn typcn-zoom-out' => 'Zoom Out' ],
		[ 'typcn typcn-zoom-outline' => 'Zoom Outline' ],
		[ 'typcn typcn-zoom' => 'Zoom' ],
	];

	return array_merge( $icons, $typicons_icons );
}

add_filter( 'vc_iconpicker-type-entypo', 'vc_iconpicker_type_entypo' );
/**
 * Entypo icons from github.com/danielbruce/entypo
 *
 * @param $icons - taken from filter - vc_map param field settings['source']
 *     provided icons (default empty array). If array categorized it will
 *     auto-enable category dropdown
 *
 * @since 4.4
 * @return array - of icons for iconpicker, can be categorized, or not.
 */
function vc_iconpicker_type_entypo( $icons ) {
	$entypo_icons = [
		[ 'entypo-icon entypo-icon-note' => 'Note' ],
		[ 'entypo-icon entypo-icon-note-beamed' => 'Note Beamed' ],
		[ 'entypo-icon entypo-icon-music' => 'Music' ],
		[ 'entypo-icon entypo-icon-search' => 'Search' ],
		[ 'entypo-icon entypo-icon-flashlight' => 'Flashlight' ],
		[ 'entypo-icon entypo-icon-mail' => 'Mail' ],
		[ 'entypo-icon entypo-icon-heart' => 'Heart' ],
		[ 'entypo-icon entypo-icon-heart-empty' => 'Heart Empty' ],
		[ 'entypo-icon entypo-icon-star' => 'Star' ],
		[ 'entypo-icon entypo-icon-star-empty' => 'Star Empty' ],
		[ 'entypo-icon entypo-icon-user' => 'User' ],
		[ 'entypo-icon entypo-icon-users' => 'Users' ],
		[ 'entypo-icon entypo-icon-user-add' => 'User Add' ],
		[ 'entypo-icon entypo-icon-video' => 'Video' ],
		[ 'entypo-icon entypo-icon-picture' => 'Picture' ],
		[ 'entypo-icon entypo-icon-camera' => 'Camera' ],
		[ 'entypo-icon entypo-icon-layout' => 'Layout' ],
		[ 'entypo-icon entypo-icon-menu' => 'Menu' ],
		[ 'entypo-icon entypo-icon-check' => 'Check' ],
		[ 'entypo-icon entypo-icon-cancel' => 'Cancel' ],
		[ 'entypo-icon entypo-icon-cancel-circled' => 'Cancel Circled' ],
		[ 'entypo-icon entypo-icon-cancel-squared' => 'Cancel Squared' ],
		[ 'entypo-icon entypo-icon-plus' => 'Plus' ],
		[ 'entypo-icon entypo-icon-plus-circled' => 'Plus Circled' ],
		[ 'entypo-icon entypo-icon-plus-squared' => 'Plus Squared' ],
		[ 'entypo-icon entypo-icon-minus' => 'Minus' ],
		[ 'entypo-icon entypo-icon-minus-circled' => 'Minus Circled' ],
		[ 'entypo-icon entypo-icon-minus-squared' => 'Minus Squared' ],
		[ 'entypo-icon entypo-icon-help' => 'Help' ],
		[ 'entypo-icon entypo-icon-help-circled' => 'Help Circled' ],
		[ 'entypo-icon entypo-icon-info' => 'Info' ],
		[ 'entypo-icon entypo-icon-info-circled' => 'Info Circled' ],
		[ 'entypo-icon entypo-icon-back' => 'Back' ],
		[ 'entypo-icon entypo-icon-home' => 'Home' ],
		[ 'entypo-icon entypo-icon-link' => 'Link' ],
		[ 'entypo-icon entypo-icon-attach' => 'Attach' ],
		[ 'entypo-icon entypo-icon-lock' => 'Lock' ],
		[ 'entypo-icon entypo-icon-lock-open' => 'Lock Open' ],
		[ 'entypo-icon entypo-icon-eye' => 'Eye' ],
		[ 'entypo-icon entypo-icon-tag' => 'Tag' ],
		[ 'entypo-icon entypo-icon-bookmark' => 'Bookmark' ],
		[ 'entypo-icon entypo-icon-bookmarks' => 'Bookmarks' ],
		[ 'entypo-icon entypo-icon-flag' => 'Flag' ],
		[ 'entypo-icon entypo-icon-thumbs-up' => 'Thumbs Up' ],
		[ 'entypo-icon entypo-icon-thumbs-down' => 'Thumbs Down' ],
		[ 'entypo-icon entypo-icon-download' => 'Download' ],
		[ 'entypo-icon entypo-icon-upload' => 'Upload' ],
		[ 'entypo-icon entypo-icon-upload-cloud' => 'Upload Cloud' ],
		[ 'entypo-icon entypo-icon-reply' => 'Reply' ],
		[ 'entypo-icon entypo-icon-reply-all' => 'Reply All' ],
		[ 'entypo-icon entypo-icon-forward' => 'Forward' ],
		[ 'entypo-icon entypo-icon-quote' => 'Quote' ],
		[ 'entypo-icon entypo-icon-code' => 'Code' ],
		[ 'entypo-icon entypo-icon-export' => 'Export' ],
		[ 'entypo-icon entypo-icon-pencil' => 'Pencil' ],
		[ 'entypo-icon entypo-icon-feather' => 'Feather' ],
		[ 'entypo-icon entypo-icon-print' => 'Print' ],
		[ 'entypo-icon entypo-icon-retweet' => 'Retweet' ],
		[ 'entypo-icon entypo-icon-keyboard' => 'Keyboard' ],
		[ 'entypo-icon entypo-icon-comment' => 'Comment' ],
		[ 'entypo-icon entypo-icon-chat' => 'Chat' ],
		[ 'entypo-icon entypo-icon-bell' => 'Bell' ],
		[ 'entypo-icon entypo-icon-attention' => 'Attention' ],
		[ 'entypo-icon entypo-icon-alert' => 'Alert' ],
		[ 'entypo-icon entypo-icon-vcard' => 'Vcard' ],
		[ 'entypo-icon entypo-icon-address' => 'Address' ],
		[ 'entypo-icon entypo-icon-location' => 'Location' ],
		[ 'entypo-icon entypo-icon-map' => 'Map' ],
		[ 'entypo-icon entypo-icon-direction' => 'Direction' ],
		[ 'entypo-icon entypo-icon-compass' => 'Compass' ],
		[ 'entypo-icon entypo-icon-cup' => 'Cup' ],
		[ 'entypo-icon entypo-icon-trash' => 'Trash' ],
		[ 'entypo-icon entypo-icon-doc' => 'Doc' ],
		[ 'entypo-icon entypo-icon-docs' => 'Docs' ],
		[ 'entypo-icon entypo-icon-doc-landscape' => 'Doc Landscape' ],
		[ 'entypo-icon entypo-icon-doc-text' => 'Doc Text' ],
		[ 'entypo-icon entypo-icon-doc-text-inv' => 'Doc Text Inv' ],
		[ 'entypo-icon entypo-icon-newspaper' => 'Newspaper' ],
		[ 'entypo-icon entypo-icon-book-open' => 'Book Open' ],
		[ 'entypo-icon entypo-icon-book' => 'Book' ],
		[ 'entypo-icon entypo-icon-folder' => 'Folder' ],
		[ 'entypo-icon entypo-icon-archive' => 'Archive' ],
		[ 'entypo-icon entypo-icon-box' => 'Box' ],
		[ 'entypo-icon entypo-icon-rss' => 'Rss' ],
		[ 'entypo-icon entypo-icon-phone' => 'Phone' ],
		[ 'entypo-icon entypo-icon-cog' => 'Cog' ],
		[ 'entypo-icon entypo-icon-tools' => 'Tools' ],
		[ 'entypo-icon entypo-icon-share' => 'Share' ],
		[ 'entypo-icon entypo-icon-shareable' => 'Shareable' ],
		[ 'entypo-icon entypo-icon-basket' => 'Basket' ],
		[ 'entypo-icon entypo-icon-bag' => 'Bag' ],
		[ 'entypo-icon entypo-icon-calendar' => 'Calendar' ],
		[ 'entypo-icon entypo-icon-login' => 'Login' ],
		[ 'entypo-icon entypo-icon-logout' => 'Logout' ],
		[ 'entypo-icon entypo-icon-mic' => 'Mic' ],
		[ 'entypo-icon entypo-icon-mute' => 'Mute' ],
		[ 'entypo-icon entypo-icon-sound' => 'Sound' ],
		[ 'entypo-icon entypo-icon-volume' => 'Volume' ],
		[ 'entypo-icon entypo-icon-clock' => 'Clock' ],
		[ 'entypo-icon entypo-icon-hourglass' => 'Hourglass' ],
		[ 'entypo-icon entypo-icon-lamp' => 'Lamp' ],
		[ 'entypo-icon entypo-icon-light-down' => 'Light Down' ],
		[ 'entypo-icon entypo-icon-light-up' => 'Light Up' ],
		[ 'entypo-icon entypo-icon-adjust' => 'Adjust' ],
		[ 'entypo-icon entypo-icon-block' => 'Block' ],
		[ 'entypo-icon entypo-icon-resize-full' => 'Resize Full' ],
		[ 'entypo-icon entypo-icon-resize-small' => 'Resize Small' ],
		[ 'entypo-icon entypo-icon-popup' => 'Popup' ],
		[ 'entypo-icon entypo-icon-publish' => 'Publish' ],
		[ 'entypo-icon entypo-icon-window' => 'Window' ],
		[ 'entypo-icon entypo-icon-arrow-combo' => 'Arrow Combo' ],
		[ 'entypo-icon entypo-icon-down-circled' => 'Down Circled' ],
		[ 'entypo-icon entypo-icon-left-circled' => 'Left Circled' ],
		[ 'entypo-icon entypo-icon-right-circled' => 'Right Circled' ],
		[ 'entypo-icon entypo-icon-up-circled' => 'Up Circled' ],
		[ 'entypo-icon entypo-icon-down-open' => 'Down Open' ],
		[ 'entypo-icon entypo-icon-left-open' => 'Left Open' ],
		[ 'entypo-icon entypo-icon-right-open' => 'Right Open' ],
		[ 'entypo-icon entypo-icon-up-open' => 'Up Open' ],
		[ 'entypo-icon entypo-icon-down-open-mini' => 'Down Open Mini' ],
		[ 'entypo-icon entypo-icon-left-open-mini' => 'Left Open Mini' ],
		[ 'entypo-icon entypo-icon-right-open-mini' => 'Right Open Mini' ],
		[ 'entypo-icon entypo-icon-up-open-mini' => 'Up Open Mini' ],
		[ 'entypo-icon entypo-icon-down-open-big' => 'Down Open Big' ],
		[ 'entypo-icon entypo-icon-left-open-big' => 'Left Open Big' ],
		[ 'entypo-icon entypo-icon-right-open-big' => 'Right Open Big' ],
		[ 'entypo-icon entypo-icon-up-open-big' => 'Up Open Big' ],
		[ 'entypo-icon entypo-icon-down' => 'Down' ],
		[ 'entypo-icon entypo-icon-left' => 'Left' ],
		[ 'entypo-icon entypo-icon-right' => 'Right' ],
		[ 'entypo-icon entypo-icon-up' => 'Up' ],
		[ 'entypo-icon entypo-icon-down-dir' => 'Down Dir' ],
		[ 'entypo-icon entypo-icon-left-dir' => 'Left Dir' ],
		[ 'entypo-icon entypo-icon-right-dir' => 'Right Dir' ],
		[ 'entypo-icon entypo-icon-up-dir' => 'Up Dir' ],
		[ 'entypo-icon entypo-icon-down-bold' => 'Down Bold' ],
		[ 'entypo-icon entypo-icon-left-bold' => 'Left Bold' ],
		[ 'entypo-icon entypo-icon-right-bold' => 'Right Bold' ],
		[ 'entypo-icon entypo-icon-up-bold' => 'Up Bold' ],
		[ 'entypo-icon entypo-icon-down-thin' => 'Down Thin' ],
		[ 'entypo-icon entypo-icon-left-thin' => 'Left Thin' ],
		[ 'entypo-icon entypo-icon-right-thin' => 'Right Thin' ],
		[ 'entypo-icon entypo-icon-up-thin' => 'Up Thin' ],
		[ 'entypo-icon entypo-icon-ccw' => 'Ccw' ],
		[ 'entypo-icon entypo-icon-cw' => 'Cw' ],
		[ 'entypo-icon entypo-icon-arrows-ccw' => 'Arrows Ccw' ],
		[ 'entypo-icon entypo-icon-level-down' => 'Level Down' ],
		[ 'entypo-icon entypo-icon-level-up' => 'Level Up' ],
		[ 'entypo-icon entypo-icon-shuffle' => 'Shuffle' ],
		[ 'entypo-icon entypo-icon-loop' => 'Loop' ],
		[ 'entypo-icon entypo-icon-switch' => 'Switch' ],
		[ 'entypo-icon entypo-icon-play' => 'Play' ],
		[ 'entypo-icon entypo-icon-stop' => 'Stop' ],
		[ 'entypo-icon entypo-icon-pause' => 'Pause' ],
		[ 'entypo-icon entypo-icon-record' => 'Record' ],
		[ 'entypo-icon entypo-icon-to-end' => 'To End' ],
		[ 'entypo-icon entypo-icon-to-start' => 'To Start' ],
		[ 'entypo-icon entypo-icon-fast-forward' => 'Fast Forward' ],
		[ 'entypo-icon entypo-icon-fast-backward' => 'Fast Backward' ],
		[ 'entypo-icon entypo-icon-progress-0' => 'Progress 0' ],
		[ 'entypo-icon entypo-icon-progress-1' => 'Progress 1' ],
		[ 'entypo-icon entypo-icon-progress-2' => 'Progress 2' ],
		[ 'entypo-icon entypo-icon-progress-3' => 'Progress 3' ],
		[ 'entypo-icon entypo-icon-target' => 'Target' ],
		[ 'entypo-icon entypo-icon-palette' => 'Palette' ],
		[ 'entypo-icon entypo-icon-list' => 'List' ],
		[ 'entypo-icon entypo-icon-list-add' => 'List Add' ],
		[ 'entypo-icon entypo-icon-signal' => 'Signal' ],
		[ 'entypo-icon entypo-icon-trophy' => 'Trophy' ],
		[ 'entypo-icon entypo-icon-battery' => 'Battery' ],
		[ 'entypo-icon entypo-icon-back-in-time' => 'Back In Time' ],
		[ 'entypo-icon entypo-icon-monitor' => 'Monitor' ],
		[ 'entypo-icon entypo-icon-mobile' => 'Mobile' ],
		[ 'entypo-icon entypo-icon-network' => 'Network' ],
		[ 'entypo-icon entypo-icon-cd' => 'Cd' ],
		[ 'entypo-icon entypo-icon-inbox' => 'Inbox' ],
		[ 'entypo-icon entypo-icon-install' => 'Install' ],
		[ 'entypo-icon entypo-icon-globe' => 'Globe' ],
		[ 'entypo-icon entypo-icon-cloud' => 'Cloud' ],
		[ 'entypo-icon entypo-icon-cloud-thunder' => 'Cloud Thunder' ],
		[ 'entypo-icon entypo-icon-flash' => 'Flash' ],
		[ 'entypo-icon entypo-icon-moon' => 'Moon' ],
		[ 'entypo-icon entypo-icon-flight' => 'Flight' ],
		[ 'entypo-icon entypo-icon-paper-plane' => 'Paper Plane' ],
		[ 'entypo-icon entypo-icon-leaf' => 'Leaf' ],
		[ 'entypo-icon entypo-icon-lifebuoy' => 'Lifebuoy' ],
		[ 'entypo-icon entypo-icon-mouse' => 'Mouse' ],
		[ 'entypo-icon entypo-icon-briefcase' => 'Briefcase' ],
		[ 'entypo-icon entypo-icon-suitcase' => 'Suitcase' ],
		[ 'entypo-icon entypo-icon-dot' => 'Dot' ],
		[ 'entypo-icon entypo-icon-dot-2' => 'Dot 2' ],
		[ 'entypo-icon entypo-icon-dot-3' => 'Dot 3' ],
		[ 'entypo-icon entypo-icon-brush' => 'Brush' ],
		[ 'entypo-icon entypo-icon-magnet' => 'Magnet' ],
		[ 'entypo-icon entypo-icon-infinity' => 'Infinity' ],
		[ 'entypo-icon entypo-icon-erase' => 'Erase' ],
		[ 'entypo-icon entypo-icon-chart-pie' => 'Chart Pie' ],
		[ 'entypo-icon entypo-icon-chart-line' => 'Chart Line' ],
		[ 'entypo-icon entypo-icon-chart-bar' => 'Chart Bar' ],
		[ 'entypo-icon entypo-icon-chart-area' => 'Chart Area' ],
		[ 'entypo-icon entypo-icon-tape' => 'Tape' ],
		[ 'entypo-icon entypo-icon-graduation-cap' => 'Graduation Cap' ],
		[ 'entypo-icon entypo-icon-language' => 'Language' ],
		[ 'entypo-icon entypo-icon-ticket' => 'Ticket' ],
		[ 'entypo-icon entypo-icon-water' => 'Water' ],
		[ 'entypo-icon entypo-icon-droplet' => 'Droplet' ],
		[ 'entypo-icon entypo-icon-air' => 'Air' ],
		[ 'entypo-icon entypo-icon-credit-card' => 'Credit Card' ],
		[ 'entypo-icon entypo-icon-floppy' => 'Floppy' ],
		[ 'entypo-icon entypo-icon-clipboard' => 'Clipboard' ],
		[ 'entypo-icon entypo-icon-megaphone' => 'Megaphone' ],
		[ 'entypo-icon entypo-icon-database' => 'Database' ],
		[ 'entypo-icon entypo-icon-drive' => 'Drive' ],
		[ 'entypo-icon entypo-icon-bucket' => 'Bucket' ],
		[ 'entypo-icon entypo-icon-thermometer' => 'Thermometer' ],
		[ 'entypo-icon entypo-icon-key' => 'Key' ],
		[ 'entypo-icon entypo-icon-flow-cascade' => 'Flow Cascade' ],
		[ 'entypo-icon entypo-icon-flow-branch' => 'Flow Branch' ],
		[ 'entypo-icon entypo-icon-flow-tree' => 'Flow Tree' ],
		[ 'entypo-icon entypo-icon-flow-line' => 'Flow Line' ],
		[ 'entypo-icon entypo-icon-flow-parallel' => 'Flow Parallel' ],
		[ 'entypo-icon entypo-icon-rocket' => 'Rocket' ],
		[ 'entypo-icon entypo-icon-gauge' => 'Gauge' ],
		[ 'entypo-icon entypo-icon-traffic-cone' => 'Traffic Cone' ],
		[ 'entypo-icon entypo-icon-cc' => 'Cc' ],
		[ 'entypo-icon entypo-icon-cc-by' => 'Cc By' ],
		[ 'entypo-icon entypo-icon-cc-nc' => 'Cc Nc' ],
		[ 'entypo-icon entypo-icon-cc-nc-eu' => 'Cc Nc Eu' ],
		[ 'entypo-icon entypo-icon-cc-nc-jp' => 'Cc Nc Jp' ],
		[ 'entypo-icon entypo-icon-cc-sa' => 'Cc Sa' ],
		[ 'entypo-icon entypo-icon-cc-nd' => 'Cc Nd' ],
		[ 'entypo-icon entypo-icon-cc-pd' => 'Cc Pd' ],
		[ 'entypo-icon entypo-icon-cc-zero' => 'Cc Zero' ],
		[ 'entypo-icon entypo-icon-cc-share' => 'Cc Share' ],
		[ 'entypo-icon entypo-icon-cc-remix' => 'Cc Remix' ],
		[ 'entypo-icon entypo-icon-github' => 'Github' ],
		[ 'entypo-icon entypo-icon-github-circled' => 'Github Circled' ],
		[ 'entypo-icon entypo-icon-flickr' => 'Flickr' ],
		[ 'entypo-icon entypo-icon-flickr-circled' => 'Flickr Circled' ],
		[ 'entypo-icon entypo-icon-vimeo' => 'Vimeo' ],
		[ 'entypo-icon entypo-icon-vimeo-circled' => 'Vimeo Circled' ],
		[ 'entypo-icon entypo-icon-twitter' => 'Twitter' ],
		[ 'entypo-icon entypo-icon-twitter-circled' => 'Twitter Circled' ],
		[ 'entypo-icon entypo-icon-facebook' => 'Facebook' ],
		[ 'entypo-icon entypo-icon-facebook-circled' => 'Facebook Circled' ],
		[ 'entypo-icon entypo-icon-facebook-squared' => 'Facebook Squared' ],
		[ 'entypo-icon entypo-icon-gplus' => 'Gplus' ],
		[ 'entypo-icon entypo-icon-gplus-circled' => 'Gplus Circled' ],
		[ 'entypo-icon entypo-icon-pinterest' => 'Pinterest' ],
		[ 'entypo-icon entypo-icon-pinterest-circled' => 'Pinterest Circled' ],
		[ 'entypo-icon entypo-icon-tumblr' => 'Tumblr' ],
		[ 'entypo-icon entypo-icon-tumblr-circled' => 'Tumblr Circled' ],
		[ 'entypo-icon entypo-icon-linkedin' => 'Linkedin' ],
		[ 'entypo-icon entypo-icon-linkedin-circled' => 'Linkedin Circled' ],
		[ 'entypo-icon entypo-icon-dribbble' => 'Dribbble' ],
		[ 'entypo-icon entypo-icon-dribbble-circled' => 'Dribbble Circled' ],
		[ 'entypo-icon entypo-icon-stumbleupon' => 'Stumbleupon' ],
		[ 'entypo-icon entypo-icon-stumbleupon-circled' => 'Stumbleupon Circled' ],
		[ 'entypo-icon entypo-icon-lastfm' => 'Lastfm' ],
		[ 'entypo-icon entypo-icon-lastfm-circled' => 'Lastfm Circled' ],
		[ 'entypo-icon entypo-icon-rdio' => 'Rdio' ],
		[ 'entypo-icon entypo-icon-rdio-circled' => 'Rdio Circled' ],
		[ 'entypo-icon entypo-icon-spotify' => 'Spotify' ],
		[ 'entypo-icon entypo-icon-spotify-circled' => 'Spotify Circled' ],
		[ 'entypo-icon entypo-icon-qq' => 'Qq' ],
		[ 'entypo-icon entypo-icon-instagrem' => 'Instagrem' ],
		[ 'entypo-icon entypo-icon-dropbox' => 'Dropbox' ],
		[ 'entypo-icon entypo-icon-evernote' => 'Evernote' ],
		[ 'entypo-icon entypo-icon-flattr' => 'Flattr' ],
		[ 'entypo-icon entypo-icon-skype' => 'Skype' ],
		[ 'entypo-icon entypo-icon-skype-circled' => 'Skype Circled' ],
		[ 'entypo-icon entypo-icon-renren' => 'Renren' ],
		[ 'entypo-icon entypo-icon-sina-weibo' => 'Sina Weibo' ],
		[ 'entypo-icon entypo-icon-paypal' => 'Paypal' ],
		[ 'entypo-icon entypo-icon-picasa' => 'Picasa' ],
		[ 'entypo-icon entypo-icon-soundcloud' => 'Soundcloud' ],
		[ 'entypo-icon entypo-icon-mixi' => 'Mixi' ],
		[ 'entypo-icon entypo-icon-behance' => 'Behance' ],
		[ 'entypo-icon entypo-icon-google-circles' => 'Google Circles' ],
		[ 'entypo-icon entypo-icon-vkontakte' => 'Vkontakte' ],
		[ 'entypo-icon entypo-icon-smashing' => 'Smashing' ],
		[ 'entypo-icon entypo-icon-sweden' => 'Sweden' ],
		[ 'entypo-icon entypo-icon-db-shape' => 'Db Shape' ],
		[ 'entypo-icon entypo-icon-logo-db' => 'Logo Db' ],
	];

	return array_merge( $icons, $entypo_icons );
}

add_filter( 'vc_iconpicker-type-linecons', 'vc_iconpicker_type_linecons' );

/**
 * Linecons icons from fontello.com
 *
 * @param $icons - taken from filter - vc_map param field settings['source']
 *     provided icons (default empty array). If array categorized it will
 *     auto-enable category dropdown
 *
 * @since 4.4
 * @return array - of icons for iconpicker, can be categorized, or not.
 */
function vc_iconpicker_type_linecons( $icons ) {
	$linecons_icons = [
		[ 'vc_li vc_li-heart' => 'Heart' ],
		[ 'vc_li vc_li-cloud' => 'Cloud' ],
		[ 'vc_li vc_li-star' => 'Star' ],
		[ 'vc_li vc_li-tv' => 'Tv' ],
		[ 'vc_li vc_li-sound' => 'Sound' ],
		[ 'vc_li vc_li-video' => 'Video' ],
		[ 'vc_li vc_li-trash' => 'Trash' ],
		[ 'vc_li vc_li-user' => 'User' ],
		[ 'vc_li vc_li-key' => 'Key' ],
		[ 'vc_li vc_li-search' => 'Search' ],
		[ 'vc_li vc_li-settings' => 'Settings' ],
		[ 'vc_li vc_li-camera' => 'Camera' ],
		[ 'vc_li vc_li-tag' => 'Tag' ],
		[ 'vc_li vc_li-lock' => 'Lock' ],
		[ 'vc_li vc_li-bulb' => 'Bulb' ],
		[ 'vc_li vc_li-pen' => 'Pen' ],
		[ 'vc_li vc_li-diamond' => 'Diamond' ],
		[ 'vc_li vc_li-display' => 'Display' ],
		[ 'vc_li vc_li-location' => 'Location' ],
		[ 'vc_li vc_li-eye' => 'Eye' ],
		[ 'vc_li vc_li-bubble' => 'Bubble' ],
		[ 'vc_li vc_li-stack' => 'Stack' ],
		[ 'vc_li vc_li-cup' => 'Cup' ],
		[ 'vc_li vc_li-phone' => 'Phone' ],
		[ 'vc_li vc_li-news' => 'News' ],
		[ 'vc_li vc_li-mail' => 'Mail' ],
		[ 'vc_li vc_li-like' => 'Like' ],
		[ 'vc_li vc_li-photo' => 'Photo' ],
		[ 'vc_li vc_li-note' => 'Note' ],
		[ 'vc_li vc_li-clock' => 'Clock' ],
		[ 'vc_li vc_li-paperplane' => 'Paperplane' ],
		[ 'vc_li vc_li-params' => 'Params' ],
		[ 'vc_li vc_li-banknote' => 'Banknote' ],
		[ 'vc_li vc_li-data' => 'Data' ],
		[ 'vc_li vc_li-music' => 'Music' ],
		[ 'vc_li vc_li-megaphone' => 'Megaphone' ],
		[ 'vc_li vc_li-study' => 'Study' ],
		[ 'vc_li vc_li-lab' => 'Lab' ],
		[ 'vc_li vc_li-food' => 'Food' ],
		[ 'vc_li vc_li-t-shirt' => 'T Shirt' ],
		[ 'vc_li vc_li-fire' => 'Fire' ],
		[ 'vc_li vc_li-clip' => 'Clip' ],
		[ 'vc_li vc_li-shop' => 'Shop' ],
		[ 'vc_li vc_li-calendar' => 'Calendar' ],
		[ 'vc_li vc_li-vallet' => 'Vallet' ],
		[ 'vc_li vc_li-vynil' => 'Vynil' ],
		[ 'vc_li vc_li-truck' => 'Truck' ],
		[ 'vc_li vc_li-world' => 'World' ],
	];

	return array_merge( $icons, $linecons_icons );
}

add_filter( 'vc_iconpicker-type-monosocial', 'vc_iconpicker_type_monosocial' );

/**
 * monosocial icons from drinchev.github.io/monosocialiconsfont
 *
 * @param $icons - taken from filter - vc_map param field settings['source']
 *     provided icons (default empty array). If array categorized it will
 *     auto-enable category dropdown
 *
 * @since 4.4
 * @return array - of icons for iconpicker, can be categorized, or not.
 */
function vc_iconpicker_type_monosocial( $icons ) {
	$monosocial = [
		[ 'vc-mono vc-mono-fivehundredpx' => 'Five Hundred px' ],
		[ 'vc-mono vc-mono-aboutme' => 'About me' ],
		[ 'vc-mono vc-mono-addme' => 'Add me' ],
		[ 'vc-mono vc-mono-amazon' => 'Amazon' ],
		[ 'vc-mono vc-mono-aol' => 'Aol' ],
		[ 'vc-mono vc-mono-appstorealt' => 'App-store-alt' ],
		[ 'vc-mono vc-mono-appstore' => 'Appstore' ],
		[ 'vc-mono vc-mono-apple' => 'Apple' ],
		[ 'vc-mono vc-mono-bebo' => 'Bebo' ],
		[ 'vc-mono vc-mono-behance' => 'Behance' ],
		[ 'vc-mono vc-mono-bing' => 'Bing' ],
		[ 'vc-mono vc-mono-blip' => 'Blip' ],
		[ 'vc-mono vc-mono-blogger' => 'Blogger' ],
		[ 'vc-mono vc-mono-coroflot' => 'Coroflot' ],
		[ 'vc-mono vc-mono-daytum' => 'Daytum' ],
		[ 'vc-mono vc-mono-delicious' => 'Delicious' ],
		[ 'vc-mono vc-mono-designbump' => 'Design bump' ],
		[ 'vc-mono vc-mono-designfloat' => 'Design float' ],
		[ 'vc-mono vc-mono-deviantart' => 'Deviant-art' ],
		[ 'vc-mono vc-mono-diggalt' => 'Digg-alt' ],
		[ 'vc-mono vc-mono-digg' => 'Digg' ],
		[ 'vc-mono vc-mono-dribble' => 'Dribble' ],
		[ 'vc-mono vc-mono-drupal' => 'Drupal' ],
		[ 'vc-mono vc-mono-ebay' => 'Ebay' ],
		[ 'vc-mono vc-mono-email' => 'Email' ],
		[ 'vc-mono vc-mono-emberapp' => 'Ember app' ],
		[ 'vc-mono vc-mono-etsy' => 'Etsy' ],
		[ 'vc-mono vc-mono-facebook' => 'Facebook' ],
		[ 'vc-mono vc-mono-feedburner' => 'Feed burner' ],
		[ 'vc-mono vc-mono-flickr' => 'Flickr' ],
		[ 'vc-mono vc-mono-foodspotting' => 'Food spotting' ],
		[ 'vc-mono vc-mono-forrst' => 'Forrst' ],
		[ 'vc-mono vc-mono-foursquare' => 'Fours quare' ],
		[ 'vc-mono vc-mono-friendsfeed' => 'Friends feed' ],
		[ 'vc-mono vc-mono-friendstar' => 'Friend star' ],
		[ 'vc-mono vc-mono-gdgt' => 'Gdgt' ],
		[ 'vc-mono vc-mono-github' => 'Github' ],
		[ 'vc-mono vc-mono-githubalt' => 'Github-alt' ],
		[ 'vc-mono vc-mono-googlebuzz' => 'Google buzz' ],
		[ 'vc-mono vc-mono-googleplus' => 'Google plus' ],
		[ 'vc-mono vc-mono-googletalk' => 'Google talk' ],
		[ 'vc-mono vc-mono-gowallapin' => 'Gowallapin' ],
		[ 'vc-mono vc-mono-gowalla' => 'Gowalla' ],
		[ 'vc-mono vc-mono-grooveshark' => 'Groove shark' ],
		[ 'vc-mono vc-mono-heart' => 'Heart' ],
		[ 'vc-mono vc-mono-hyves' => 'Hyves' ],
		[ 'vc-mono vc-mono-icondock' => 'Icondock' ],
		[ 'vc-mono vc-mono-icq' => 'Icq' ],
		[ 'vc-mono vc-mono-identica' => 'Identica' ],
		[ 'vc-mono vc-mono-imessage' => 'I message' ],
		[ 'vc-mono vc-mono-itunes' => 'I-tunes' ],
		[ 'vc-mono vc-mono-lastfm' => 'Lastfm' ],
		[ 'vc-mono vc-mono-linkedin' => 'Linkedin' ],
		[ 'vc-mono vc-mono-meetup' => 'Meetup' ],
		[ 'vc-mono vc-mono-metacafe' => 'Metacafe' ],
		[ 'vc-mono vc-mono-mixx' => 'Mixx' ],
		[ 'vc-mono vc-mono-mobileme' => 'Mobile me' ],
		[ 'vc-mono vc-mono-mrwong' => 'Mrwong' ],
		[ 'vc-mono vc-mono-msn' => 'Msn' ],
		[ 'vc-mono vc-mono-myspace' => 'Myspace' ],
		[ 'vc-mono vc-mono-newsvine' => 'Newsvine' ],
		[ 'vc-mono vc-mono-paypal' => 'Paypal' ],
		[ 'vc-mono vc-mono-photobucket' => 'Photo bucket' ],
		[ 'vc-mono vc-mono-picasa' => 'Picasa' ],
		[ 'vc-mono vc-mono-pinterest' => 'Pinterest' ],
		[ 'vc-mono vc-mono-podcast' => 'Podcast' ],
		[ 'vc-mono vc-mono-posterous' => 'Posterous' ],
		[ 'vc-mono vc-mono-qik' => 'Qik' ],
		[ 'vc-mono vc-mono-quora' => 'Quora' ],
		[ 'vc-mono vc-mono-reddit' => 'Reddit' ],
		[ 'vc-mono vc-mono-retweet' => 'Retweet' ],
		[ 'vc-mono vc-mono-rss' => 'Rss' ],
		[ 'vc-mono vc-mono-scribd' => 'Scribd' ],
		[ 'vc-mono vc-mono-sharethis' => 'Sharethis' ],
		[ 'vc-mono vc-mono-skype' => 'Skype' ],
		[ 'vc-mono vc-mono-slashdot' => 'Slashdot' ],
		[ 'vc-mono vc-mono-slideshare' => 'Slideshare' ],
		[ 'vc-mono vc-mono-smugmug' => 'Smugmug' ],
		[ 'vc-mono vc-mono-soundcloud' => 'Soundcloud' ],
		[ 'vc-mono vc-mono-spotify' => 'Spotify' ],
		[ 'vc-mono vc-mono-squidoo' => 'Squidoo' ],
		[ 'vc-mono vc-mono-stackoverflow' => 'Stackoverflow' ],
		[ 'vc-mono vc-mono-star' => 'Star' ],
		[ 'vc-mono vc-mono-stumbleupon' => 'Stumble upon' ],
		[ 'vc-mono vc-mono-technorati' => 'Technorati' ],
		[ 'vc-mono vc-mono-tumblr' => 'Tumblr' ],
		[ 'vc-mono vc-mono-twitterbird' => 'Twitterbird' ],
		[ 'vc-mono vc-mono-twitter' => 'Twitter' ],
		[ 'vc-mono vc-mono-viddler' => 'Viddler' ],
		[ 'vc-mono vc-mono-vimeo' => 'Vimeo' ],
		[ 'vc-mono vc-mono-virb' => 'Virb' ],
		[ 'vc-mono vc-mono-www' => 'Www' ],
		[ 'vc-mono vc-mono-wikipedia' => 'Wikipedia' ],
		[ 'vc-mono vc-mono-windows' => 'Windows' ],
		[ 'vc-mono vc-mono-wordpress' => 'WordPress' ],
		[ 'vc-mono vc-mono-xing' => 'Xing' ],
		[ 'vc-mono vc-mono-yahoobuzz' => 'Yahoo buzz' ],
		[ 'vc-mono vc-mono-yahoo' => 'Yahoo' ],
		[ 'vc-mono vc-mono-yelp' => 'Yelp' ],
		[ 'vc-mono vc-mono-youtube' => 'Youtube' ],
		[ 'vc-mono vc-mono-instagram' => 'Instagram' ],
	];

	return array_merge( $icons, $monosocial );
}

add_filter( 'vc_iconpicker-type-material', 'vc_iconpicker_type_material' );
/**
 * Material icon set from Google
 * @since 5.0
 *
 * @param $icons
 *
 * @return array

 */
function vc_iconpicker_type_material( $icons ) {
	$material = [
		[ 'vc-material vc-material-3d_rotation' => '3d rotation' ],
		[ 'vc-material vc-material-ac_unit' => 'ac unit' ],
		[ 'vc-material vc-material-alarm' => 'alarm' ],
		[ 'vc-material vc-material-access_alarms' => 'access alarms' ],
		[ 'vc-material vc-material-schedule' => 'schedule' ],
		[ 'vc-material vc-material-accessibility' => 'accessibility' ],
		[ 'vc-material vc-material-accessible' => 'accessible' ],
		[ 'vc-material vc-material-account_balance' => 'account balance' ],
		[ 'vc-material vc-material-account_balance_wallet' => 'account balance wallet' ],
		[ 'vc-material vc-material-account_box' => 'account box' ],
		[ 'vc-material vc-material-account_circle' => 'account circle' ],
		[ 'vc-material vc-material-adb' => 'adb' ],
		[ 'vc-material vc-material-add' => 'add' ],
		[ 'vc-material vc-material-add_a_photo' => 'add a photo' ],
		[ 'vc-material vc-material-alarm_add' => 'alarm add' ],
		[ 'vc-material vc-material-add_alert' => 'add alert' ],
		[ 'vc-material vc-material-add_box' => 'add box' ],
		[ 'vc-material vc-material-add_circle' => 'add circle' ],
		[ 'vc-material vc-material-control_point' => 'control point' ],
		[ 'vc-material vc-material-add_location' => 'add location' ],
		[ 'vc-material vc-material-add_shopping_cart' => 'add shopping cart' ],
		[ 'vc-material vc-material-queue' => 'queue' ],
		[ 'vc-material vc-material-add_to_queue' => 'add to queue' ],
		[ 'vc-material vc-material-adjust' => 'adjust' ],
		[ 'vc-material vc-material-airline_seat_flat' => 'airline seat flat' ],
		[ 'vc-material vc-material-airline_seat_flat_angled' => 'airline seat flat angled' ],
		[ 'vc-material vc-material-airline_seat_individual_suite' => 'airline seat individual suite' ],
		[ 'vc-material vc-material-airline_seat_legroom_extra' => 'airline seat legroom extra' ],
		[ 'vc-material vc-material-airline_seat_legroom_normal' => 'airline seat legroom normal' ],
		[ 'vc-material vc-material-airline_seat_legroom_reduced' => 'airline seat legroom reduced' ],
		[ 'vc-material vc-material-airline_seat_recline_extra' => 'airline seat recline extra' ],
		[ 'vc-material vc-material-airline_seat_recline_normal' => 'airline seat recline normal' ],
		[ 'vc-material vc-material-flight' => 'flight' ],
		[ 'vc-material vc-material-airplanemode_inactive' => 'airplanemode inactive' ],
		[ 'vc-material vc-material-airplay' => 'airplay' ],
		[ 'vc-material vc-material-airport_shuttle' => 'airport shuttle' ],
		[ 'vc-material vc-material-alarm_off' => 'alarm off' ],
		[ 'vc-material vc-material-alarm_on' => 'alarm on' ],
		[ 'vc-material vc-material-album' => 'album' ],
		[ 'vc-material vc-material-all_inclusive' => 'all inclusive' ],
		[ 'vc-material vc-material-all_out' => 'all out' ],
		[ 'vc-material vc-material-android' => 'android' ],
		[ 'vc-material vc-material-announcement' => 'announcement' ],
		[ 'vc-material vc-material-apps' => 'apps' ],
		[ 'vc-material vc-material-archive' => 'archive' ],
		[ 'vc-material vc-material-arrow_back' => 'arrow back' ],
		[ 'vc-material vc-material-arrow_downward' => 'arrow downward' ],
		[ 'vc-material vc-material-arrow_drop_down' => 'arrow drop down' ],
		[ 'vc-material vc-material-arrow_drop_down_circle' => 'arrow drop down circle' ],
		[ 'vc-material vc-material-arrow_drop_up' => 'arrow drop up' ],
		[ 'vc-material vc-material-arrow_forward' => 'arrow forward' ],
		[ 'vc-material vc-material-arrow_upward' => 'arrow upward' ],
		[ 'vc-material vc-material-art_track' => 'art track' ],
		[ 'vc-material vc-material-aspect_ratio' => 'aspect ratio' ],
		[ 'vc-material vc-material-poll' => 'poll' ],
		[ 'vc-material vc-material-assignment' => 'assignment' ],
		[ 'vc-material vc-material-assignment_ind' => 'assignment ind' ],
		[ 'vc-material vc-material-assignment_late' => 'assignment late' ],
		[ 'vc-material vc-material-assignment_return' => 'assignment return' ],
		[ 'vc-material vc-material-assignment_returned' => 'assignment returned' ],
		[ 'vc-material vc-material-assignment_turned_in' => 'assignment turned in' ],
		[ 'vc-material vc-material-assistant' => 'assistant' ],
		[ 'vc-material vc-material-flag' => 'flag' ],
		[ 'vc-material vc-material-attach_file' => 'attach file' ],
		[ 'vc-material vc-material-attach_money' => 'attach money' ],
		[ 'vc-material vc-material-attachment' => 'attachment' ],
		[ 'vc-material vc-material-audiotrack' => 'audiotrack' ],
		[ 'vc-material vc-material-autorenew' => 'autorenew' ],
		[ 'vc-material vc-material-av_timer' => 'av timer' ],
		[ 'vc-material vc-material-backspace' => 'backspace' ],
		[ 'vc-material vc-material-cloud_upload' => 'cloud upload' ],
		[ 'vc-material vc-material-battery_alert' => 'battery alert' ],
		[ 'vc-material vc-material-battery_charging_full' => 'battery charging full' ],
		[ 'vc-material vc-material-battery_std' => 'battery std' ],
		[ 'vc-material vc-material-battery_unknown' => 'battery unknown' ],
		[ 'vc-material vc-material-beach_access' => 'beach access' ],
		[ 'vc-material vc-material-beenhere' => 'beenhere' ],
		[ 'vc-material vc-material-block' => 'block' ],
		[ 'vc-material vc-material-bluetooth' => 'bluetooth' ],
		[ 'vc-material vc-material-bluetooth_searching' => 'bluetooth searching' ],
		[ 'vc-material vc-material-bluetooth_connected' => 'bluetooth connected' ],
		[ 'vc-material vc-material-bluetooth_disabled' => 'bluetooth disabled' ],
		[ 'vc-material vc-material-blur_circular' => 'blur circular' ],
		[ 'vc-material vc-material-blur_linear' => 'blur linear' ],
		[ 'vc-material vc-material-blur_off' => 'blur off' ],
		[ 'vc-material vc-material-blur_on' => 'blur on' ],
		[ 'vc-material vc-material-class' => 'class' ],
		[ 'vc-material vc-material-turned_in' => 'turned in' ],
		[ 'vc-material vc-material-turned_in_not' => 'turned in not' ],
		[ 'vc-material vc-material-border_all' => 'border all' ],
		[ 'vc-material vc-material-border_bottom' => 'border bottom' ],
		[ 'vc-material vc-material-border_clear' => 'border clear' ],
		[ 'vc-material vc-material-border_color' => 'border color' ],
		[ 'vc-material vc-material-border_horizontal' => 'border horizontal' ],
		[ 'vc-material vc-material-border_inner' => 'border inner' ],
		[ 'vc-material vc-material-border_left' => 'border left' ],
		[ 'vc-material vc-material-border_outer' => 'border outer' ],
		[ 'vc-material vc-material-border_right' => 'border right' ],
		[ 'vc-material vc-material-border_style' => 'border style' ],
		[ 'vc-material vc-material-border_top' => 'border top' ],
		[ 'vc-material vc-material-border_vertical' => 'border vertical' ],
		[ 'vc-material vc-material-branding_watermark' => 'branding watermark' ],
		[ 'vc-material vc-material-brightness_1' => 'brightness 1' ],
		[ 'vc-material vc-material-brightness_2' => 'brightness 2' ],
		[ 'vc-material vc-material-brightness_3' => 'brightness 3' ],
		[ 'vc-material vc-material-brightness_4' => 'brightness 4' ],
		[ 'vc-material vc-material-brightness_low' => 'brightness low' ],
		[ 'vc-material vc-material-brightness_medium' => 'brightness medium' ],
		[ 'vc-material vc-material-brightness_high' => 'brightness high' ],
		[ 'vc-material vc-material-brightness_auto' => 'brightness auto' ],
		[ 'vc-material vc-material-broken_image' => 'broken image' ],
		[ 'vc-material vc-material-brush' => 'brush' ],
		[ 'vc-material vc-material-bubble_chart' => 'bubble chart' ],
		[ 'vc-material vc-material-bug_report' => 'bug report' ],
		[ 'vc-material vc-material-build' => 'build' ],
		[ 'vc-material vc-material-burst_mode' => 'burst mode' ],
		[ 'vc-material vc-material-domain' => 'domain' ],
		[ 'vc-material vc-material-business_center' => 'business center' ],
		[ 'vc-material vc-material-cached' => 'cached' ],
		[ 'vc-material vc-material-cake' => 'cake' ],
		[ 'vc-material vc-material-phone' => 'phone' ],
		[ 'vc-material vc-material-call_end' => 'call end' ],
		[ 'vc-material vc-material-call_made' => 'call made' ],
		[ 'vc-material vc-material-merge_type' => 'merge type' ],
		[ 'vc-material vc-material-call_missed' => 'call missed' ],
		[ 'vc-material vc-material-call_missed_outgoing' => 'call missed outgoing' ],
		[ 'vc-material vc-material-call_received' => 'call received' ],
		[ 'vc-material vc-material-call_split' => 'call split' ],
		[ 'vc-material vc-material-call_to_action' => 'call to action' ],
		[ 'vc-material vc-material-camera' => 'camera' ],
		[ 'vc-material vc-material-photo_camera' => 'photo camera' ],
		[ 'vc-material vc-material-camera_enhance' => 'camera enhance' ],
		[ 'vc-material vc-material-camera_front' => 'camera front' ],
		[ 'vc-material vc-material-camera_rear' => 'camera rear' ],
		[ 'vc-material vc-material-camera_roll' => 'camera roll' ],
		[ 'vc-material vc-material-cancel' => 'cancel' ],
		[ 'vc-material vc-material-redeem' => 'redeem' ],
		[ 'vc-material vc-material-card_membership' => 'card membership' ],
		[ 'vc-material vc-material-card_travel' => 'card travel' ],
		[ 'vc-material vc-material-casino' => 'casino' ],
		[ 'vc-material vc-material-cast' => 'cast' ],
		[ 'vc-material vc-material-cast_connected' => 'cast connected' ],
		[ 'vc-material vc-material-center_focus_strong' => 'center focus strong' ],
		[ 'vc-material vc-material-center_focus_weak' => 'center focus weak' ],
		[ 'vc-material vc-material-change_history' => 'change history' ],
		[ 'vc-material vc-material-chat' => 'chat' ],
		[ 'vc-material vc-material-chat_bubble' => 'chat bubble' ],
		[ 'vc-material vc-material-chat_bubble_outline' => 'chat bubble outline' ],
		[ 'vc-material vc-material-check' => 'check' ],
		[ 'vc-material vc-material-check_box' => 'check box' ],
		[ 'vc-material vc-material-check_box_outline_blank' => 'check box outline blank' ],
		[ 'vc-material vc-material-check_circle' => 'check circle' ],
		[ 'vc-material vc-material-navigate_before' => 'navigate before' ],
		[ 'vc-material vc-material-navigate_next' => 'navigate next' ],
		[ 'vc-material vc-material-child_care' => 'child care' ],
		[ 'vc-material vc-material-child_friendly' => 'child friendly' ],
		[ 'vc-material vc-material-chrome_reader_mode' => 'chrome reader mode' ],
		[ 'vc-material vc-material-close' => 'close' ],
		[ 'vc-material vc-material-clear_all' => 'clear all' ],
		[ 'vc-material vc-material-closed_caption' => 'closed caption' ],
		[ 'vc-material vc-material-wb_cloudy' => 'wb cloudy' ],
		[ 'vc-material vc-material-cloud_circle' => 'cloud circle' ],
		[ 'vc-material vc-material-cloud_done' => 'cloud done' ],
		[ 'vc-material vc-material-cloud_download' => 'cloud download' ],
		[ 'vc-material vc-material-cloud_off' => 'cloud off' ],
		[ 'vc-material vc-material-cloud_queue' => 'cloud queue' ],
		[ 'vc-material vc-material-code' => 'code' ],
		[ 'vc-material vc-material-photo_library' => 'photo library' ],
		[ 'vc-material vc-material-collections_bookmark' => 'collections bookmark' ],
		[ 'vc-material vc-material-palette' => 'palette' ],
		[ 'vc-material vc-material-colorize' => 'colorize' ],
		[ 'vc-material vc-material-comment' => 'comment' ],
		[ 'vc-material vc-material-compare' => 'compare' ],
		[ 'vc-material vc-material-compare_arrows' => 'compare arrows' ],
		[ 'vc-material vc-material-laptop' => 'laptop' ],
		[ 'vc-material vc-material-confirmation_number' => 'confirmation number' ],
		[ 'vc-material vc-material-contact_mail' => 'contact mail' ],
		[ 'vc-material vc-material-contact_phone' => 'contact phone' ],
		[ 'vc-material vc-material-contacts' => 'contacts' ],
		[ 'vc-material vc-material-content_copy' => 'content copy' ],
		[ 'vc-material vc-material-content_cut' => 'content cut' ],
		[ 'vc-material vc-material-content_paste' => 'content paste' ],
		[ 'vc-material vc-material-control_point_duplicate' => 'control point duplicate' ],
		[ 'vc-material vc-material-copyright' => 'copyright' ],
		[ 'vc-material vc-material-mode_edit' => 'mode edit' ],
		[ 'vc-material vc-material-create_new_folder' => 'create new folder' ],
		[ 'vc-material vc-material-payment' => 'payment' ],
		[ 'vc-material vc-material-crop' => 'crop' ],
		[ 'vc-material vc-material-crop_16_9' => 'crop 16 9' ],
		[ 'vc-material vc-material-crop_3_2' => 'crop 3 2' ],
		[ 'vc-material vc-material-crop_landscape' => 'crop landscape' ],
		[ 'vc-material vc-material-crop_7_5' => 'crop 7 5' ],
		[ 'vc-material vc-material-crop_din' => 'crop din' ],
		[ 'vc-material vc-material-crop_free' => 'crop free' ],
		[ 'vc-material vc-material-crop_original' => 'crop original' ],
		[ 'vc-material vc-material-crop_portrait' => 'crop portrait' ],
		[ 'vc-material vc-material-crop_rotate' => 'crop rotate' ],
		[ 'vc-material vc-material-crop_square' => 'crop square' ],
		[ 'vc-material vc-material-dashboard' => 'dashboard' ],
		[ 'vc-material vc-material-data_usage' => 'data usage' ],
		[ 'vc-material vc-material-date_range' => 'date range' ],
		[ 'vc-material vc-material-dehaze' => 'dehaze' ],
		[ 'vc-material vc-material-delete' => 'delete' ],
		[ 'vc-material vc-material-delete_forever' => 'delete forever' ],
		[ 'vc-material vc-material-delete_sweep' => 'delete sweep' ],
		[ 'vc-material vc-material-description' => 'description' ],
		[ 'vc-material vc-material-desktop_mac' => 'desktop mac' ],
		[ 'vc-material vc-material-desktop_windows' => 'desktop windows' ],
		[ 'vc-material vc-material-details' => 'details' ],
		[ 'vc-material vc-material-developer_board' => 'developer board' ],
		[ 'vc-material vc-material-developer_mode' => 'developer mode' ],
		[ 'vc-material vc-material-device_hub' => 'device hub' ],
		[ 'vc-material vc-material-phonelink' => 'phonelink' ],
		[ 'vc-material vc-material-devices_other' => 'devices other' ],
		[ 'vc-material vc-material-dialer_sip' => 'dialer sip' ],
		[ 'vc-material vc-material-dialpad' => 'dialpad' ],
		[ 'vc-material vc-material-directions' => 'directions' ],
		[ 'vc-material vc-material-directions_bike' => 'directions bike' ],
		[ 'vc-material vc-material-directions_boat' => 'directions boat' ],
		[ 'vc-material vc-material-directions_bus' => 'directions bus' ],
		[ 'vc-material vc-material-directions_car' => 'directions car' ],
		[ 'vc-material vc-material-directions_railway' => 'directions railway' ],
		[ 'vc-material vc-material-directions_run' => 'directions run' ],
		[ 'vc-material vc-material-directions_transit' => 'directions transit' ],
		[ 'vc-material vc-material-directions_walk' => 'directions walk' ],
		[ 'vc-material vc-material-disc_full' => 'disc full' ],
		[ 'vc-material vc-material-dns' => 'dns' ],
		[ 'vc-material vc-material-not_interested' => 'not interested' ],
		[ 'vc-material vc-material-do_not_disturb_alt' => 'do not disturb alt' ],
		[ 'vc-material vc-material-do_not_disturb_off' => 'do not disturb off' ],
		[ 'vc-material vc-material-remove_circle' => 'remove circle' ],
		[ 'vc-material vc-material-dock' => 'dock' ],
		[ 'vc-material vc-material-done' => 'done' ],
		[ 'vc-material vc-material-done_all' => 'done all' ],
		[ 'vc-material vc-material-donut_large' => 'donut large' ],
		[ 'vc-material vc-material-donut_small' => 'donut small' ],
		[ 'vc-material vc-material-drafts' => 'drafts' ],
		[ 'vc-material vc-material-drag_handle' => 'drag handle' ],
		[ 'vc-material vc-material-time_to_leave' => 'time to leave' ],
		[ 'vc-material vc-material-dvr' => 'dvr' ],
		[ 'vc-material vc-material-edit_location' => 'edit location' ],
		[ 'vc-material vc-material-eject' => 'eject' ],
		[ 'vc-material vc-material-markunread' => 'markunread' ],
		[ 'vc-material vc-material-enhanced_encryption' => 'enhanced encryption' ],
		[ 'vc-material vc-material-equalizer' => 'equalizer' ],
		[ 'vc-material vc-material-error' => 'error' ],
		[ 'vc-material vc-material-error_outline' => 'error outline' ],
		[ 'vc-material vc-material-euro_symbol' => 'euro symbol' ],
		[ 'vc-material vc-material-ev_station' => 'ev station' ],
		[ 'vc-material vc-material-insert_invitation' => 'insert invitation' ],
		[ 'vc-material vc-material-event_available' => 'event available' ],
		[ 'vc-material vc-material-event_busy' => 'event busy' ],
		[ 'vc-material vc-material-event_note' => 'event note' ],
		[ 'vc-material vc-material-event_seat' => 'event seat' ],
		[ 'vc-material vc-material-exit_to_app' => 'exit to app' ],
		[ 'vc-material vc-material-expand_less' => 'expand less' ],
		[ 'vc-material vc-material-expand_more' => 'expand more' ],
		[ 'vc-material vc-material-explicit' => 'explicit' ],
		[ 'vc-material vc-material-explore' => 'explore' ],
		[ 'vc-material vc-material-exposure' => 'exposure' ],
		[ 'vc-material vc-material-exposure_neg_1' => 'exposure neg 1' ],
		[ 'vc-material vc-material-exposure_neg_2' => 'exposure neg 2' ],
		[ 'vc-material vc-material-exposure_plus_1' => 'exposure plus 1' ],
		[ 'vc-material vc-material-exposure_plus_2' => 'exposure plus 2' ],
		[ 'vc-material vc-material-exposure_zero' => 'exposure zero' ],
		[ 'vc-material vc-material-extension' => 'extension' ],
		[ 'vc-material vc-material-face' => 'face' ],
		[ 'vc-material vc-material-fast_forward' => 'fast forward' ],
		[ 'vc-material vc-material-fast_rewind' => 'fast rewind' ],
		[ 'vc-material vc-material-favorite' => 'favorite' ],
		[ 'vc-material vc-material-favorite_border' => 'favorite border' ],
		[ 'vc-material vc-material-featured_play_list' => 'featured play list' ],
		[ 'vc-material vc-material-featured_video' => 'featured video' ],
		[ 'vc-material vc-material-sms_failed' => 'sms failed' ],
		[ 'vc-material vc-material-fiber_dvr' => 'fiber dvr' ],
		[ 'vc-material vc-material-fiber_manual_record' => 'fiber manual record' ],
		[ 'vc-material vc-material-fiber_new' => 'fiber new' ],
		[ 'vc-material vc-material-fiber_pin' => 'fiber pin' ],
		[ 'vc-material vc-material-fiber_smart_record' => 'fiber smart record' ],
		[ 'vc-material vc-material-get_app' => 'get app' ],
		[ 'vc-material vc-material-file_upload' => 'file upload' ],
		[ 'vc-material vc-material-filter' => 'filter' ],
		[ 'vc-material vc-material-filter_1' => 'filter 1' ],
		[ 'vc-material vc-material-filter_2' => 'filter 2' ],
		[ 'vc-material vc-material-filter_3' => 'filter 3' ],
		[ 'vc-material vc-material-filter_4' => 'filter 4' ],
		[ 'vc-material vc-material-filter_5' => 'filter 5' ],
		[ 'vc-material vc-material-filter_6' => 'filter 6' ],
		[ 'vc-material vc-material-filter_7' => 'filter 7' ],
		[ 'vc-material vc-material-filter_8' => 'filter 8' ],
		[ 'vc-material vc-material-filter_9' => 'filter 9' ],
		[ 'vc-material vc-material-filter_9_plus' => 'filter 9 plus' ],
		[ 'vc-material vc-material-filter_b_and_w' => 'filter b and w' ],
		[ 'vc-material vc-material-filter_center_focus' => 'filter center focus' ],
		[ 'vc-material vc-material-filter_drama' => 'filter drama' ],
		[ 'vc-material vc-material-filter_frames' => 'filter frames' ],
		[ 'vc-material vc-material-terrain' => 'terrain' ],
		[ 'vc-material vc-material-filter_list' => 'filter list' ],
		[ 'vc-material vc-material-filter_none' => 'filter none' ],
		[ 'vc-material vc-material-filter_tilt_shift' => 'filter tilt shift' ],
		[ 'vc-material vc-material-filter_vintage' => 'filter vintage' ],
		[ 'vc-material vc-material-find_in_page' => 'find in page' ],
		[ 'vc-material vc-material-find_replace' => 'find replace' ],
		[ 'vc-material vc-material-fingerprint' => 'fingerprint' ],
		[ 'vc-material vc-material-first_page' => 'first page' ],
		[ 'vc-material vc-material-fitness_center' => 'fitness center' ],
		[ 'vc-material vc-material-flare' => 'flare' ],
		[ 'vc-material vc-material-flash_auto' => 'flash auto' ],
		[ 'vc-material vc-material-flash_off' => 'flash off' ],
		[ 'vc-material vc-material-flash_on' => 'flash on' ],
		[ 'vc-material vc-material-flight_land' => 'flight land' ],
		[ 'vc-material vc-material-flight_takeoff' => 'flight takeoff' ],
		[ 'vc-material vc-material-flip' => 'flip' ],
		[ 'vc-material vc-material-flip_to_back' => 'flip to back' ],
		[ 'vc-material vc-material-flip_to_front' => 'flip to front' ],
		[ 'vc-material vc-material-folder' => 'folder' ],
		[ 'vc-material vc-material-folder_open' => 'folder open' ],
		[ 'vc-material vc-material-folder_shared' => 'folder shared' ],
		[ 'vc-material vc-material-folder_special' => 'folder special' ],
		[ 'vc-material vc-material-font_download' => 'font download' ],
		[ 'vc-material vc-material-format_align_center' => 'format align center' ],
		[ 'vc-material vc-material-format_align_justify' => 'format align justify' ],
		[ 'vc-material vc-material-format_align_left' => 'format align left' ],
		[ 'vc-material vc-material-format_align_right' => 'format align right' ],
		[ 'vc-material vc-material-format_bold' => 'format bold' ],
		[ 'vc-material vc-material-format_clear' => 'format clear' ],
		[ 'vc-material vc-material-format_color_fill' => 'format color fill' ],
		[ 'vc-material vc-material-format_color_reset' => 'format color reset' ],
		[ 'vc-material vc-material-format_color_text' => 'format color text' ],
		[ 'vc-material vc-material-format_indent_decrease' => 'format indent decrease' ],
		[ 'vc-material vc-material-format_indent_increase' => 'format indent increase' ],
		[ 'vc-material vc-material-format_italic' => 'format italic' ],
		[ 'vc-material vc-material-format_line_spacing' => 'format line spacing' ],
		[ 'vc-material vc-material-format_list_bulleted' => 'format list bulleted' ],
		[ 'vc-material vc-material-format_list_numbered' => 'format list numbered' ],
		[ 'vc-material vc-material-format_paint' => 'format paint' ],
		[ 'vc-material vc-material-format_quote' => 'format quote' ],
		[ 'vc-material vc-material-format_shapes' => 'format shapes' ],
		[ 'vc-material vc-material-format_size' => 'format size' ],
		[ 'vc-material vc-material-format_strikethrough' => 'format strikethrough' ],
		[ 'vc-material vc-material-format_textdirection_l_to_r' => 'format textdirection l to r' ],
		[ 'vc-material vc-material-format_textdirection_r_to_l' => 'format textdirection r to l' ],
		[ 'vc-material vc-material-format_underlined' => 'format underlined' ],
		[ 'vc-material vc-material-question_answer' => 'question answer' ],
		[ 'vc-material vc-material-forward' => 'forward' ],
		[ 'vc-material vc-material-forward_10' => 'forward 10' ],
		[ 'vc-material vc-material-forward_30' => 'forward 30' ],
		[ 'vc-material vc-material-forward_5' => 'forward 5' ],
		[ 'vc-material vc-material-free_breakfast' => 'free breakfast' ],
		[ 'vc-material vc-material-fullscreen' => 'fullscreen' ],
		[ 'vc-material vc-material-fullscreen_exit' => 'fullscreen exit' ],
		[ 'vc-material vc-material-functions' => 'functions' ],
		[ 'vc-material vc-material-g_translate' => 'g translate' ],
		[ 'vc-material vc-material-games' => 'games' ],
		[ 'vc-material vc-material-gavel' => 'gavel' ],
		[ 'vc-material vc-material-gesture' => 'gesture' ],
		[ 'vc-material vc-material-gif' => 'gif' ],
		[ 'vc-material vc-material-goat' => 'goat' ],
		[ 'vc-material vc-material-golf_course' => 'golf course' ],
		[ 'vc-material vc-material-my_location' => 'my location' ],
		[ 'vc-material vc-material-location_searching' => 'location searching' ],
		[ 'vc-material vc-material-location_disabled' => 'location disabled' ],
		[ 'vc-material vc-material-star' => 'star' ],
		[ 'vc-material vc-material-gradient' => 'gradient' ],
		[ 'vc-material vc-material-grain' => 'grain' ],
		[ 'vc-material vc-material-graphic_eq' => 'graphic eq' ],
		[ 'vc-material vc-material-grid_off' => 'grid off' ],
		[ 'vc-material vc-material-grid_on' => 'grid on' ],
		[ 'vc-material vc-material-people' => 'people' ],
		[ 'vc-material vc-material-group_add' => 'group add' ],
		[ 'vc-material vc-material-group_work' => 'group work' ],
		[ 'vc-material vc-material-hd' => 'hd' ],
		[ 'vc-material vc-material-hdr_off' => 'hdr off' ],
		[ 'vc-material vc-material-hdr_on' => 'hdr on' ],
		[ 'vc-material vc-material-hdr_strong' => 'hdr strong' ],
		[ 'vc-material vc-material-hdr_weak' => 'hdr weak' ],
		[ 'vc-material vc-material-headset' => 'headset' ],
		[ 'vc-material vc-material-headset_mic' => 'headset mic' ],
		[ 'vc-material vc-material-healing' => 'healing' ],
		[ 'vc-material vc-material-hearing' => 'hearing' ],
		[ 'vc-material vc-material-help' => 'help' ],
		[ 'vc-material vc-material-help_outline' => 'help outline' ],
		[ 'vc-material vc-material-high_quality' => 'high quality' ],
		[ 'vc-material vc-material-highlight' => 'highlight' ],
		[ 'vc-material vc-material-highlight_off' => 'highlight off' ],
		[ 'vc-material vc-material-restore' => 'restore' ],
		[ 'vc-material vc-material-home' => 'home' ],
		[ 'vc-material vc-material-hot_tub' => 'hot tub' ],
		[ 'vc-material vc-material-local_hotel' => 'local hotel' ],
		[ 'vc-material vc-material-hourglass_empty' => 'hourglass empty' ],
		[ 'vc-material vc-material-hourglass_full' => 'hourglass full' ],
		[ 'vc-material vc-material-http' => 'http' ],
		[ 'vc-material vc-material-lock' => 'lock' ],
		[ 'vc-material vc-material-photo' => 'photo' ],
		[ 'vc-material vc-material-image_aspect_ratio' => 'image aspect ratio' ],
		[ 'vc-material vc-material-import_contacts' => 'import contacts' ],
		[ 'vc-material vc-material-import_export' => 'import export' ],
		[ 'vc-material vc-material-important_devices' => 'important devices' ],
		[ 'vc-material vc-material-inbox' => 'inbox' ],
		[ 'vc-material vc-material-indeterminate_check_box' => 'indeterminate check box' ],
		[ 'vc-material vc-material-info' => 'info' ],
		[ 'vc-material vc-material-info_outline' => 'info outline' ],
		[ 'vc-material vc-material-input' => 'input' ],
		[ 'vc-material vc-material-insert_comment' => 'insert comment' ],
		[ 'vc-material vc-material-insert_drive_file' => 'insert drive file' ],
		[ 'vc-material vc-material-tag_faces' => 'tag faces' ],
		[ 'vc-material vc-material-link' => 'link' ],
		[ 'vc-material vc-material-invert_colors' => 'invert colors' ],
		[ 'vc-material vc-material-invert_colors_off' => 'invert colors off' ],
		[ 'vc-material vc-material-iso' => 'iso' ],
		[ 'vc-material vc-material-keyboard' => 'keyboard' ],
		[ 'vc-material vc-material-keyboard_arrow_down' => 'keyboard arrow down' ],
		[ 'vc-material vc-material-keyboard_arrow_left' => 'keyboard arrow left' ],
		[ 'vc-material vc-material-keyboard_arrow_right' => 'keyboard arrow right' ],
		[ 'vc-material vc-material-keyboard_arrow_up' => 'keyboard arrow up' ],
		[ 'vc-material vc-material-keyboard_backspace' => 'keyboard backspace' ],
		[ 'vc-material vc-material-keyboard_capslock' => 'keyboard capslock' ],
		[ 'vc-material vc-material-keyboard_hide' => 'keyboard hide' ],
		[ 'vc-material vc-material-keyboard_return' => 'keyboard return' ],
		[ 'vc-material vc-material-keyboard_tab' => 'keyboard tab' ],
		[ 'vc-material vc-material-keyboard_voice' => 'keyboard voice' ],
		[ 'vc-material vc-material-kitchen' => 'kitchen' ],
		[ 'vc-material vc-material-label' => 'label' ],
		[ 'vc-material vc-material-label_outline' => 'label outline' ],
		[ 'vc-material vc-material-language' => 'language' ],
		[ 'vc-material vc-material-laptop_chromebook' => 'laptop chromebook' ],
		[ 'vc-material vc-material-laptop_mac' => 'laptop mac' ],
		[ 'vc-material vc-material-laptop_windows' => 'laptop windows' ],
		[ 'vc-material vc-material-last_page' => 'last page' ],
		[ 'vc-material vc-material-open_in_new' => 'open in new' ],
		[ 'vc-material vc-material-layers' => 'layers' ],
		[ 'vc-material vc-material-layers_clear' => 'layers clear' ],
		[ 'vc-material vc-material-leak_add' => 'leak add' ],
		[ 'vc-material vc-material-leak_remove' => 'leak remove' ],
		[ 'vc-material vc-material-lens' => 'lens' ],
		[ 'vc-material vc-material-library_books' => 'library books' ],
		[ 'vc-material vc-material-library_music' => 'library music' ],
		[ 'vc-material vc-material-lightbulb_outline' => 'lightbulb outline' ],
		[ 'vc-material vc-material-line_style' => 'line style' ],
		[ 'vc-material vc-material-line_weight' => 'line weight' ],
		[ 'vc-material vc-material-linear_scale' => 'linear scale' ],
		[ 'vc-material vc-material-linked_camera' => 'linked camera' ],
		[ 'vc-material vc-material-list' => 'list' ],
		[ 'vc-material vc-material-live_help' => 'live help' ],
		[ 'vc-material vc-material-live_tv' => 'live tv' ],
		[ 'vc-material vc-material-local_play' => 'local play' ],
		[ 'vc-material vc-material-local_airport' => 'local airport' ],
		[ 'vc-material vc-material-local_atm' => 'local atm' ],
		[ 'vc-material vc-material-local_bar' => 'local bar' ],
		[ 'vc-material vc-material-local_cafe' => 'local cafe' ],
		[ 'vc-material vc-material-local_car_wash' => 'local car wash' ],
		[ 'vc-material vc-material-local_convenience_store' => 'local convenience store' ],
		[ 'vc-material vc-material-restaurant_menu' => 'restaurant menu' ],
		[ 'vc-material vc-material-local_drink' => 'local drink' ],
		[ 'vc-material vc-material-local_florist' => 'local florist' ],
		[ 'vc-material vc-material-local_gas_station' => 'local gas station' ],
		[ 'vc-material vc-material-shopping_cart' => 'shopping cart' ],
		[ 'vc-material vc-material-local_hospital' => 'local hospital' ],
		[ 'vc-material vc-material-local_laundry_service' => 'local laundry service' ],
		[ 'vc-material vc-material-local_library' => 'local library' ],
		[ 'vc-material vc-material-local_mall' => 'local mall' ],
		[ 'vc-material vc-material-theaters' => 'theaters' ],
		[ 'vc-material vc-material-local_offer' => 'local offer' ],
		[ 'vc-material vc-material-local_parking' => 'local parking' ],
		[ 'vc-material vc-material-local_pharmacy' => 'local pharmacy' ],
		[ 'vc-material vc-material-local_pizza' => 'local pizza' ],
		[ 'vc-material vc-material-print' => 'print' ],
		[ 'vc-material vc-material-local_shipping' => 'local shipping' ],
		[ 'vc-material vc-material-local_taxi' => 'local taxi' ],
		[ 'vc-material vc-material-location_city' => 'location city' ],
		[ 'vc-material vc-material-location_off' => 'location off' ],
		[ 'vc-material vc-material-room' => 'room' ],
		[ 'vc-material vc-material-lock_open' => 'lock open' ],
		[ 'vc-material vc-material-lock_outline' => 'lock outline' ],
		[ 'vc-material vc-material-looks' => 'looks' ],
		[ 'vc-material vc-material-looks_3' => 'looks 3' ],
		[ 'vc-material vc-material-looks_4' => 'looks 4' ],
		[ 'vc-material vc-material-looks_5' => 'looks 5' ],
		[ 'vc-material vc-material-looks_6' => 'looks 6' ],
		[ 'vc-material vc-material-looks_one' => 'looks one' ],
		[ 'vc-material vc-material-looks_two' => 'looks two' ],
		[ 'vc-material vc-material-sync' => 'sync' ],
		[ 'vc-material vc-material-loupe' => 'loupe' ],
		[ 'vc-material vc-material-low_priority' => 'low priority' ],
		[ 'vc-material vc-material-loyalty' => 'loyalty' ],
		[ 'vc-material vc-material-mail_outline' => 'mail outline' ],
		[ 'vc-material vc-material-map' => 'map' ],
		[ 'vc-material vc-material-markunread_mailbox' => 'markunread mailbox' ],
		[ 'vc-material vc-material-memory' => 'memory' ],
		[ 'vc-material vc-material-menu' => 'menu' ],
		[ 'vc-material vc-material-message' => 'message' ],
		[ 'vc-material vc-material-mic' => 'mic' ],
		[ 'vc-material vc-material-mic_none' => 'mic none' ],
		[ 'vc-material vc-material-mic_off' => 'mic off' ],
		[ 'vc-material vc-material-mms' => 'mms' ],
		[ 'vc-material vc-material-mode_comment' => 'mode comment' ],
		[ 'vc-material vc-material-monetization_on' => 'monetization on' ],
		[ 'vc-material vc-material-money_off' => 'money off' ],
		[ 'vc-material vc-material-monochrome_photos' => 'monochrome photos' ],
		[ 'vc-material vc-material-mood_bad' => 'mood bad' ],
		[ 'vc-material vc-material-more' => 'more' ],
		[ 'vc-material vc-material-more_horiz' => 'more horiz' ],
		[ 'vc-material vc-material-more_vert' => 'more vert' ],
		[ 'vc-material vc-material-motorcycle' => 'motorcycle' ],
		[ 'vc-material vc-material-mouse' => 'mouse' ],
		[ 'vc-material vc-material-move_to_inbox' => 'move to inbox' ],
		[ 'vc-material vc-material-movie_creation' => 'movie creation' ],
		[ 'vc-material vc-material-movie_filter' => 'movie filter' ],
		[ 'vc-material vc-material-multiline_chart' => 'multiline chart' ],
		[ 'vc-material vc-material-music_note' => 'music note' ],
		[ 'vc-material vc-material-music_video' => 'music video' ],
		[ 'vc-material vc-material-nature' => 'nature' ],
		[ 'vc-material vc-material-nature_people' => 'nature people' ],
		[ 'vc-material vc-material-navigation' => 'navigation' ],
		[ 'vc-material vc-material-near_me' => 'near me' ],
		[ 'vc-material vc-material-network_cell' => 'network cell' ],
		[ 'vc-material vc-material-network_check' => 'network check' ],
		[ 'vc-material vc-material-network_locked' => 'network locked' ],
		[ 'vc-material vc-material-network_wifi' => 'network wifi' ],
		[ 'vc-material vc-material-new_releases' => 'new releases' ],
		[ 'vc-material vc-material-next_week' => 'next week' ],
		[ 'vc-material vc-material-nfc' => 'nfc' ],
		[ 'vc-material vc-material-no_encryption' => 'no encryption' ],
		[ 'vc-material vc-material-signal_cellular_no_sim' => 'signal cellular no sim' ],
		[ 'vc-material vc-material-note' => 'note' ],
		[ 'vc-material vc-material-note_add' => 'note add' ],
		[ 'vc-material vc-material-notifications' => 'notifications' ],
		[ 'vc-material vc-material-notifications_active' => 'notifications active' ],
		[ 'vc-material vc-material-notifications_none' => 'notifications none' ],
		[ 'vc-material vc-material-notifications_off' => 'notifications off' ],
		[ 'vc-material vc-material-notifications_paused' => 'notifications paused' ],
		[ 'vc-material vc-material-offline_pin' => 'offline pin' ],
		[ 'vc-material vc-material-ondemand_video' => 'ondemand video' ],
		[ 'vc-material vc-material-opacity' => 'opacity' ],
		[ 'vc-material vc-material-open_in_browser' => 'open in browser' ],
		[ 'vc-material vc-material-open_with' => 'open with' ],
		[ 'vc-material vc-material-pages' => 'pages' ],
		[ 'vc-material vc-material-pageview' => 'pageview' ],
		[ 'vc-material vc-material-pan_tool' => 'pan tool' ],
		[ 'vc-material vc-material-panorama' => 'panorama' ],
		[ 'vc-material vc-material-radio_button_unchecked' => 'radio button unchecked' ],
		[ 'vc-material vc-material-panorama_horizontal' => 'panorama horizontal' ],
		[ 'vc-material vc-material-panorama_vertical' => 'panorama vertical' ],
		[ 'vc-material vc-material-panorama_wide_angle' => 'panorama wide angle' ],
		[ 'vc-material vc-material-party_mode' => 'party mode' ],
		[ 'vc-material vc-material-pause' => 'pause' ],
		[ 'vc-material vc-material-pause_circle_filled' => 'pause circle filled' ],
		[ 'vc-material vc-material-pause_circle_outline' => 'pause circle outline' ],
		[ 'vc-material vc-material-people_outline' => 'people outline' ],
		[ 'vc-material vc-material-perm_camera_mic' => 'perm camera mic' ],
		[ 'vc-material vc-material-perm_contact_calendar' => 'perm contact calendar' ],
		[ 'vc-material vc-material-perm_data_setting' => 'perm data setting' ],
		[ 'vc-material vc-material-perm_device_information' => 'perm device information' ],
		[ 'vc-material vc-material-person_outline' => 'person outline' ],
		[ 'vc-material vc-material-perm_media' => 'perm media' ],
		[ 'vc-material vc-material-perm_phone_msg' => 'perm phone msg' ],
		[ 'vc-material vc-material-perm_scan_wifi' => 'perm scan wifi' ],
		[ 'vc-material vc-material-person' => 'person' ],
		[ 'vc-material vc-material-person_add' => 'person add' ],
		[ 'vc-material vc-material-person_pin' => 'person pin' ],
		[ 'vc-material vc-material-person_pin_circle' => 'person pin circle' ],
		[ 'vc-material vc-material-personal_video' => 'personal video' ],
		[ 'vc-material vc-material-pets' => 'pets' ],
		[ 'vc-material vc-material-phone_android' => 'phone android' ],
		[ 'vc-material vc-material-phone_bluetooth_speaker' => 'phone bluetooth speaker' ],
		[ 'vc-material vc-material-phone_forwarded' => 'phone forwarded' ],
		[ 'vc-material vc-material-phone_in_talk' => 'phone in talk' ],
		[ 'vc-material vc-material-phone_iphone' => 'phone iphone' ],
		[ 'vc-material vc-material-phone_locked' => 'phone locked' ],
		[ 'vc-material vc-material-phone_missed' => 'phone missed' ],
		[ 'vc-material vc-material-phone_paused' => 'phone paused' ],
		[ 'vc-material vc-material-phonelink_erase' => 'phonelink erase' ],
		[ 'vc-material vc-material-phonelink_lock' => 'phonelink lock' ],
		[ 'vc-material vc-material-phonelink_off' => 'phonelink off' ],
		[ 'vc-material vc-material-phonelink_ring' => 'phonelink ring' ],
		[ 'vc-material vc-material-phonelink_setup' => 'phonelink setup' ],
		[ 'vc-material vc-material-photo_album' => 'photo album' ],
		[ 'vc-material vc-material-photo_filter' => 'photo filter' ],
		[ 'vc-material vc-material-photo_size_select_actual' => 'photo size select actual' ],
		[ 'vc-material vc-material-photo_size_select_large' => 'photo size select large' ],
		[ 'vc-material vc-material-photo_size_select_small' => 'photo size select small' ],
		[ 'vc-material vc-material-picture_as_pdf' => 'picture as pdf' ],
		[ 'vc-material vc-material-picture_in_picture' => 'picture in picture' ],
		[ 'vc-material vc-material-picture_in_picture_alt' => 'picture in picture alt' ],
		[ 'vc-material vc-material-pie_chart' => 'pie chart' ],
		[ 'vc-material vc-material-pie_chart_outlined' => 'pie chart outlined' ],
		[ 'vc-material vc-material-pin_drop' => 'pin drop' ],
		[ 'vc-material vc-material-play_arrow' => 'play arrow' ],
		[ 'vc-material vc-material-play_circle_filled' => 'play circle filled' ],
		[ 'vc-material vc-material-play_circle_outline' => 'play circle outline' ],
		[ 'vc-material vc-material-play_for_work' => 'play for work' ],
		[ 'vc-material vc-material-playlist_add' => 'playlist add' ],
		[ 'vc-material vc-material-playlist_add_check' => 'playlist add check' ],
		[ 'vc-material vc-material-playlist_play' => 'playlist play' ],
		[ 'vc-material vc-material-plus_one' => 'plus one' ],
		[ 'vc-material vc-material-polymer' => 'polymer' ],
		[ 'vc-material vc-material-pool' => 'pool' ],
		[ 'vc-material vc-material-portable_wifi_off' => 'portable wifi off' ],
		[ 'vc-material vc-material-portrait' => 'portrait' ],
		[ 'vc-material vc-material-power' => 'power' ],
		[ 'vc-material vc-material-power_input' => 'power input' ],
		[ 'vc-material vc-material-power_settings_new' => 'power settings new' ],
		[ 'vc-material vc-material-pregnant_woman' => 'pregnant woman' ],
		[ 'vc-material vc-material-present_to_all' => 'present to all' ],
		[ 'vc-material vc-material-priority_high' => 'priority high' ],
		[ 'vc-material vc-material-public' => 'public' ],
		[ 'vc-material vc-material-publish' => 'publish' ],
		[ 'vc-material vc-material-queue_music' => 'queue music' ],
		[ 'vc-material vc-material-queue_play_next' => 'queue play next' ],
		[ 'vc-material vc-material-radio' => 'radio' ],
		[ 'vc-material vc-material-radio_button_checked' => 'radio button checked' ],
		[ 'vc-material vc-material-rate_review' => 'rate review' ],
		[ 'vc-material vc-material-receipt' => 'receipt' ],
		[ 'vc-material vc-material-recent_actors' => 'recent actors' ],
		[ 'vc-material vc-material-record_voice_over' => 'record voice over' ],
		[ 'vc-material vc-material-redo' => 'redo' ],
		[ 'vc-material vc-material-refresh' => 'refresh' ],
		[ 'vc-material vc-material-remove' => 'remove' ],
		[ 'vc-material vc-material-remove_circle_outline' => 'remove circle outline' ],
		[ 'vc-material vc-material-remove_from_queue' => 'remove from queue' ],
		[ 'vc-material vc-material-visibility' => 'visibility' ],
		[ 'vc-material vc-material-remove_shopping_cart' => 'remove shopping cart' ],
		[ 'vc-material vc-material-reorder' => 'reorder' ],
		[ 'vc-material vc-material-repeat' => 'repeat' ],
		[ 'vc-material vc-material-repeat_one' => 'repeat one' ],
		[ 'vc-material vc-material-replay' => 'replay' ],
		[ 'vc-material vc-material-replay_10' => 'replay 10' ],
		[ 'vc-material vc-material-replay_30' => 'replay 30' ],
		[ 'vc-material vc-material-replay_5' => 'replay 5' ],
		[ 'vc-material vc-material-reply' => 'reply' ],
		[ 'vc-material vc-material-reply_all' => 'reply all' ],
		[ 'vc-material vc-material-report' => 'report' ],
		[ 'vc-material vc-material-warning' => 'warning' ],
		[ 'vc-material vc-material-restaurant' => 'restaurant' ],
		[ 'vc-material vc-material-restore_page' => 'restore page' ],
		[ 'vc-material vc-material-ring_volume' => 'ring volume' ],
		[ 'vc-material vc-material-room_service' => 'room service' ],
		[ 'vc-material vc-material-rotate_90_degrees_ccw' => 'rotate 90 degrees ccw' ],
		[ 'vc-material vc-material-rotate_left' => 'rotate left' ],
		[ 'vc-material vc-material-rotate_right' => 'rotate right' ],
		[ 'vc-material vc-material-rounded_corner' => 'rounded corner' ],
		[ 'vc-material vc-material-router' => 'router' ],
		[ 'vc-material vc-material-rowing' => 'rowing' ],
		[ 'vc-material vc-material-rss_feed' => 'rss feed' ],
		[ 'vc-material vc-material-rv_hookup' => 'rv hookup' ],
		[ 'vc-material vc-material-satellite' => 'satellite' ],
		[ 'vc-material vc-material-save' => 'save' ],
		[ 'vc-material vc-material-scanner' => 'scanner' ],
		[ 'vc-material vc-material-school' => 'school' ],
		[ 'vc-material vc-material-screen_lock_landscape' => 'screen lock landscape' ],
		[ 'vc-material vc-material-screen_lock_portrait' => 'screen lock portrait' ],
		[ 'vc-material vc-material-screen_lock_rotation' => 'screen lock rotation' ],
		[ 'vc-material vc-material-screen_rotation' => 'screen rotation' ],
		[ 'vc-material vc-material-screen_share' => 'screen share' ],
		[ 'vc-material vc-material-sd_storage' => 'sd storage' ],
		[ 'vc-material vc-material-search' => 'search' ],
		[ 'vc-material vc-material-security' => 'security' ],
		[ 'vc-material vc-material-select_all' => 'select all' ],
		[ 'vc-material vc-material-send' => 'send' ],
		[ 'vc-material vc-material-sentiment_dissatisfied' => 'sentiment dissatisfied' ],
		[ 'vc-material vc-material-sentiment_neutral' => 'sentiment neutral' ],
		[ 'vc-material vc-material-sentiment_satisfied' => 'sentiment satisfied' ],
		[ 'vc-material vc-material-sentiment_very_dissatisfied' => 'sentiment very dissatisfied' ],
		[ 'vc-material vc-material-sentiment_very_satisfied' => 'sentiment very satisfied' ],
		[ 'vc-material vc-material-settings' => 'settings' ],
		[ 'vc-material vc-material-settings_applications' => 'settings applications' ],
		[ 'vc-material vc-material-settings_backup_restore' => 'settings backup restore' ],
		[ 'vc-material vc-material-settings_bluetooth' => 'settings bluetooth' ],
		[ 'vc-material vc-material-settings_brightness' => 'settings brightness' ],
		[ 'vc-material vc-material-settings_cell' => 'settings cell' ],
		[ 'vc-material vc-material-settings_ethernet' => 'settings ethernet' ],
		[ 'vc-material vc-material-settings_input_antenna' => 'settings input antenna' ],
		[ 'vc-material vc-material-settings_input_composite' => 'settings input composite' ],
		[ 'vc-material vc-material-settings_input_hdmi' => 'settings input hdmi' ],
		[ 'vc-material vc-material-settings_input_svideo' => 'settings input svideo' ],
		[ 'vc-material vc-material-settings_overscan' => 'settings overscan' ],
		[ 'vc-material vc-material-settings_phone' => 'settings phone' ],
		[ 'vc-material vc-material-settings_power' => 'settings power' ],
		[ 'vc-material vc-material-settings_remote' => 'settings remote' ],
		[ 'vc-material vc-material-settings_system_daydream' => 'settings system daydream' ],
		[ 'vc-material vc-material-settings_voice' => 'settings voice' ],
		[ 'vc-material vc-material-share' => 'share' ],
		[ 'vc-material vc-material-shop' => 'shop' ],
		[ 'vc-material vc-material-shop_two' => 'shop two' ],
		[ 'vc-material vc-material-shopping_basket' => 'shopping basket' ],
		[ 'vc-material vc-material-short_text' => 'short text' ],
		[ 'vc-material vc-material-show_chart' => 'show chart' ],
		[ 'vc-material vc-material-shuffle' => 'shuffle' ],
		[ 'vc-material vc-material-signal_cellular_4_bar' => 'signal cellular 4 bar' ],
		[ 'vc-material vc-material-signal_cellular_connected_no_internet_4_bar' => 'signal_cellular_connected_no internet 4 bar' ],
		[ 'vc-material vc-material-signal_cellular_null' => 'signal cellular null' ],
		[ 'vc-material vc-material-signal_cellular_off' => 'signal cellular off' ],
		[ 'vc-material vc-material-signal_wifi_4_bar' => 'signal wifi 4 bar' ],
		[ 'vc-material vc-material-signal_wifi_4_bar_lock' => 'signal wifi 4 bar lock' ],
		[ 'vc-material vc-material-signal_wifi_off' => 'signal wifi off' ],
		[ 'vc-material vc-material-sim_card' => 'sim card' ],
		[ 'vc-material vc-material-sim_card_alert' => 'sim card alert' ],
		[ 'vc-material vc-material-skip_next' => 'skip next' ],
		[ 'vc-material vc-material-skip_previous' => 'skip previous' ],
		[ 'vc-material vc-material-slideshow' => 'slideshow' ],
		[ 'vc-material vc-material-slow_motion_video' => 'slow motion video' ],
		[ 'vc-material vc-material-stay_primary_portrait' => 'stay primary portrait' ],
		[ 'vc-material vc-material-smoke_free' => 'smoke free' ],
		[ 'vc-material vc-material-smoking_rooms' => 'smoking rooms' ],
		[ 'vc-material vc-material-textsms' => 'textsms' ],
		[ 'vc-material vc-material-snooze' => 'snooze' ],
		[ 'vc-material vc-material-sort' => 'sort' ],
		[ 'vc-material vc-material-sort_by_alpha' => 'sort by alpha' ],
		[ 'vc-material vc-material-spa' => 'spa' ],
		[ 'vc-material vc-material-space_bar' => 'space bar' ],
		[ 'vc-material vc-material-speaker' => 'speaker' ],
		[ 'vc-material vc-material-speaker_group' => 'speaker group' ],
		[ 'vc-material vc-material-speaker_notes' => 'speaker notes' ],
		[ 'vc-material vc-material-speaker_notes_off' => 'speaker notes off' ],
		[ 'vc-material vc-material-speaker_phone' => 'speaker phone' ],
		[ 'vc-material vc-material-spellcheck' => 'spellcheck' ],
		[ 'vc-material vc-material-star_border' => 'star border' ],
		[ 'vc-material vc-material-star_half' => 'star half' ],
		[ 'vc-material vc-material-stars' => 'stars' ],
		[ 'vc-material vc-material-stay_primary_landscape' => 'stay primary landscape' ],
		[ 'vc-material vc-material-stop' => 'stop' ],
		[ 'vc-material vc-material-stop_screen_share' => 'stop screen share' ],
		[ 'vc-material vc-material-storage' => 'storage' ],
		[ 'vc-material vc-material-store_mall_directory' => 'store mall directory' ],
		[ 'vc-material vc-material-straighten' => 'straighten' ],
		[ 'vc-material vc-material-streetview' => 'streetview' ],
		[ 'vc-material vc-material-strikethrough_s' => 'strikethrough s' ],
		[ 'vc-material vc-material-style' => 'style' ],
		[ 'vc-material vc-material-subdirectory_arrow_left' => 'subdirectory arrow left' ],
		[ 'vc-material vc-material-subdirectory_arrow_right' => 'subdirectory arrow right' ],
		[ 'vc-material vc-material-subject' => 'subject' ],
		[ 'vc-material vc-material-subscriptions' => 'subscriptions' ],
		[ 'vc-material vc-material-subtitles' => 'subtitles' ],
		[ 'vc-material vc-material-subway' => 'subway' ],
		[ 'vc-material vc-material-supervisor_account' => 'supervisor account' ],
		[ 'vc-material vc-material-surround_sound' => 'surround sound' ],
		[ 'vc-material vc-material-swap_calls' => 'swap calls' ],
		[ 'vc-material vc-material-swap_horiz' => 'swap horiz' ],
		[ 'vc-material vc-material-swap_vert' => 'swap vert' ],
		[ 'vc-material vc-material-swap_vertical_circle' => 'swap vertical circle' ],
		[ 'vc-material vc-material-switch_camera' => 'switch camera' ],
		[ 'vc-material vc-material-switch_video' => 'switch video' ],
		[ 'vc-material vc-material-sync_disabled' => 'sync disabled' ],
		[ 'vc-material vc-material-sync_problem' => 'sync problem' ],
		[ 'vc-material vc-material-system_update' => 'system update' ],
		[ 'vc-material vc-material-system_update_alt' => 'system update alt' ],
		[ 'vc-material vc-material-tab' => 'tab' ],
		[ 'vc-material vc-material-tab_unselected' => 'tab unselected' ],
		[ 'vc-material vc-material-tablet' => 'tablet' ],
		[ 'vc-material vc-material-tablet_android' => 'tablet android' ],
		[ 'vc-material vc-material-tablet_mac' => 'tablet mac' ],
		[ 'vc-material vc-material-tap_and_play' => 'tap and play' ],
		[ 'vc-material vc-material-text_fields' => 'text fields' ],
		[ 'vc-material vc-material-text_format' => 'text format' ],
		[ 'vc-material vc-material-texture' => 'texture' ],
		[ 'vc-material vc-material-thumb_down' => 'thumb down' ],
		[ 'vc-material vc-material-thumb_up' => 'thumb up' ],
		[ 'vc-material vc-material-thumbs_up_down' => 'thumbs up down' ],
		[ 'vc-material vc-material-timelapse' => 'timelapse' ],
		[ 'vc-material vc-material-timeline' => 'timeline' ],
		[ 'vc-material vc-material-timer' => 'timer' ],
		[ 'vc-material vc-material-timer_10' => 'timer 10' ],
		[ 'vc-material vc-material-timer_3' => 'timer 3' ],
		[ 'vc-material vc-material-timer_off' => 'timer off' ],
		[ 'vc-material vc-material-title' => 'title' ],
		[ 'vc-material vc-material-toc' => 'toc' ],
		[ 'vc-material vc-material-today' => 'today' ],
		[ 'vc-material vc-material-toll' => 'toll' ],
		[ 'vc-material vc-material-tonality' => 'tonality' ],
		[ 'vc-material vc-material-touch_app' => 'touch app' ],
		[ 'vc-material vc-material-toys' => 'toys' ],
		[ 'vc-material vc-material-track_changes' => 'track changes' ],
		[ 'vc-material vc-material-traffic' => 'traffic' ],
		[ 'vc-material vc-material-train' => 'train' ],
		[ 'vc-material vc-material-tram' => 'tram' ],
		[ 'vc-material vc-material-transfer_within_a_station' => 'transfer within a station' ],
		[ 'vc-material vc-material-transform' => 'transform' ],
		[ 'vc-material vc-material-translate' => 'translate' ],
		[ 'vc-material vc-material-trending_down' => 'trending down' ],
		[ 'vc-material vc-material-trending_flat' => 'trending flat' ],
		[ 'vc-material vc-material-trending_up' => 'trending up' ],
		[ 'vc-material vc-material-tune' => 'tune' ],
		[ 'vc-material vc-material-tv' => 'tv' ],
		[ 'vc-material vc-material-unarchive' => 'unarchive' ],
		[ 'vc-material vc-material-undo' => 'undo' ],
		[ 'vc-material vc-material-unfold_less' => 'unfold less' ],
		[ 'vc-material vc-material-unfold_more' => 'unfold more' ],
		[ 'vc-material vc-material-update' => 'update' ],
		[ 'vc-material vc-material-usb' => 'usb' ],
		[ 'vc-material vc-material-verified_user' => 'verified user' ],
		[ 'vc-material vc-material-vertical_align_bottom' => 'vertical align bottom' ],
		[ 'vc-material vc-material-vertical_align_center' => 'vertical align center' ],
		[ 'vc-material vc-material-vertical_align_top' => 'vertical align top' ],
		[ 'vc-material vc-material-vibration' => 'vibration' ],
		[ 'vc-material vc-material-video_call' => 'video call' ],
		[ 'vc-material vc-material-video_label' => 'video label' ],
		[ 'vc-material vc-material-video_library' => 'video library' ],
		[ 'vc-material vc-material-videocam' => 'videocam' ],
		[ 'vc-material vc-material-videocam_off' => 'videocam off' ],
		[ 'vc-material vc-material-videogame_asset' => 'videogame asset' ],
		[ 'vc-material vc-material-view_agenda' => 'view agenda' ],
		[ 'vc-material vc-material-view_array' => 'view array' ],
		[ 'vc-material vc-material-view_carousel' => 'view carousel' ],
		[ 'vc-material vc-material-view_column' => 'view column' ],
		[ 'vc-material vc-material-view_comfy' => 'view comfy' ],
		[ 'vc-material vc-material-view_compact' => 'view compact' ],
		[ 'vc-material vc-material-view_day' => 'view day' ],
		[ 'vc-material vc-material-view_headline' => 'view headline' ],
		[ 'vc-material vc-material-view_list' => 'view list' ],
		[ 'vc-material vc-material-view_module' => 'view module' ],
		[ 'vc-material vc-material-view_quilt' => 'view quilt' ],
		[ 'vc-material vc-material-view_stream' => 'view stream' ],
		[ 'vc-material vc-material-view_week' => 'view week' ],
		[ 'vc-material vc-material-vignette' => 'vignette' ],
		[ 'vc-material vc-material-visibility_off' => 'visibility off' ],
		[ 'vc-material vc-material-voice_chat' => 'voice chat' ],
		[ 'vc-material vc-material-voicemail' => 'voicemail' ],
		[ 'vc-material vc-material-volume_down' => 'volume down' ],
		[ 'vc-material vc-material-volume_mute' => 'volume mute' ],
		[ 'vc-material vc-material-volume_off' => 'volume off' ],
		[ 'vc-material vc-material-volume_up' => 'volume up' ],
		[ 'vc-material vc-material-vpn_key' => 'vpn key' ],
		[ 'vc-material vc-material-vpn_lock' => 'vpn lock' ],
		[ 'vc-material vc-material-wallpaper' => 'wallpaper' ],
		[ 'vc-material vc-material-watch' => 'watch' ],
		[ 'vc-material vc-material-watch_later' => 'watch later' ],
		[ 'vc-material vc-material-wb_auto' => 'wb auto' ],
		[ 'vc-material vc-material-wb_incandescent' => 'wb incandescent' ],
		[ 'vc-material vc-material-wb_iridescent' => 'wb iridescent' ],
		[ 'vc-material vc-material-wb_sunny' => 'wb sunny' ],
		[ 'vc-material vc-material-wc' => 'wc' ],
		[ 'vc-material vc-material-web' => 'web' ],
		[ 'vc-material vc-material-web_asset' => 'web asset' ],
		[ 'vc-material vc-material-weekend' => 'weekend' ],
		[ 'vc-material vc-material-whatshot' => 'whatshot' ],
		[ 'vc-material vc-material-widgets' => 'widgets' ],
		[ 'vc-material vc-material-wifi' => 'wifi' ],
		[ 'vc-material vc-material-wifi_lock' => 'wifi lock' ],
		[ 'vc-material vc-material-wifi_tethering' => 'wifi tethering' ],
		[ 'vc-material vc-material-work' => 'work' ],
		[ 'vc-material vc-material-wrap_text' => 'wrap text' ],
		[ 'vc-material vc-material-youtube_searched_for' => 'youtube searched for' ],
		[ 'vc-material vc-material-zoom_in' => 'zoom in' ],
		[ 'vc-material vc-material-zoom_out' => 'zoom out' ],
		[ 'vc-material vc-material-zoom_out_map' => 'zoom out map' ],
	];

	return array_merge( $icons, $material );
}
