<?php

/**
 * Previous Orders
 */
if (!defined('ABSPATH')) {
	exit; // Exit if accessed directly.
}

if (!class_exists('FP_RAC_Previous_Order')) {

	/**
	 * FP_RAC_Previous_Order Class.
	 */
	class FP_RAC_Previous_Order {

		/**
		 * FP_RAC_Previous_Order Class initialization.
		 */
		public static function init() {
			add_action('wp_ajax_rac_get_order_ids', array( __CLASS__, 'get_order_ids' ));
			add_action('wp_ajax_rac_add_order_cart_lists_by_order_ids', array( __CLASS__, 'add_order_cart_lists_by_order_ids' ));
			add_action('wp_ajax_rac_get_old_order_cart_list_data', array( __CLASS__, 'get_old_order_cart_list_data' ));
			add_action('wp_ajax_rac_update_user_data_old_order_cart_list', array( __CLASS__, 'update_user_data_old_order_cart_list' ));
		}

		/**
		 * Get the old order cat list data.
		 *
		 * @sine 1.0.0
		 */
		public static function get_old_order_cart_list_data() {
			check_ajax_referer('oldorder-update', 'rac_security');

			try {
				// Return if the current user does not have permission.
				if (!current_user_can('manage_options')) {
					throw new exception(__("You don't have permission to do this action", 'recoverabandoncart'));
				}

				$cart_list_ids=fp_rac_check_query_having_posts(array(
					'post_type' => 'raccartlist',
					'posts_per_page' => '-1',
					'post_status' => array( 'rac-cart-new', 'rac-cart-abandon', 'rac-cart-recovered', 'trash' ),
					'meta_query' => array(
						array(
							'key' => 'rac_user_details',
							'value' => 'old_order',
						),
					),
					'fields' => 'ids',
					'cache_results' => false,
				));

				wp_send_json_success(array( 'ids'=>$cart_list_ids ));
			} catch (Exception $ex) {
				wp_send_json_error(array( 'error' => $ex->getMessage() ));
			}
		}

		/**
		 * Update the user data in old order cart lists.
		 *
		 * @since 1.0.0
		 */
		public static function update_user_data_old_order_cart_list() {
			check_ajax_referer('oldorder-update', 'rac_security');

			try {
				$cart_list_ids = isset($_POST['ids']) ? wc_clean(wp_unslash($_POST['ids'])) : '';
				if (!$cart_list_ids) {
					throw new exception(__('Invalid arguments', 'recoverabandoncart'));
				}

				// Return if the current user does not have permission.
				if (!current_user_can('manage_options')) {
					throw new exception(__("You don't have permission to do this action", 'recoverabandoncart'));
				}

				if ('none'!==$cart_list_ids && rac_check_is_array($cart_list_ids)) {
					foreach ($cart_list_ids as $cart_id) {
						$cartlist = fp_rac_create_cart_list_obj($cart_id);
						$order = fp_rac_format_cart_details($cartlist->cart_details, $cartlist);
						if (is_object($order) && ( 'old_order' == $cartlist->user_id )) {
							$order_id = fp_rac_get_order_obj_data($order, 'id');
							if ($order_id) {
								$user_email = fp_rac_get_order_obj_data($order, 'billing_email');
								$first_name = fp_rac_get_order_obj_data($order, 'billing_first_name');
								$last_name = fp_rac_get_order_obj_data($order, 'billing_last_name');
								$phone_number = fp_rac_get_order_obj_data($order, 'billing_phone');
								$product_ids = fp_rac_extract_cartlist_content($order, $cartlist, true);
								$product_ids = implode(',', $product_ids);
								$update_meta = array(
									'rac_product_details' => $product_ids,
									'rac_user_info' => $first_name . ',' . $last_name,
									'rac_phone_number' => $phone_number,
									'rac_cart_email_id' => $user_email,
								);
								fp_rac_update_cartlist_post_meta($update_meta, $cart_id);
							}
						}
					}
				}

				wp_send_json_success();
			} catch (Exception $ex) {
				wp_send_json_error(array( 'error' => $ex->getMessage() ));
			}           
		}

		/**
		 * Get the order IDs that are not yet added to the cart list.
		 *
		 * @since 1.0.0
		 */
		public static function get_order_ids() {
			check_ajax_referer('oldorder-cartlist', 'rac_security');

			try {
				$order_time = isset($_POST['order_time']) ? wc_clean(wp_unslash($_POST['order_time'])) : '';
				if (!$order_time) {
					throw new exception(__('Invalid arguments', 'recoverabandoncart'));
				}

				// Return if the current user does not have permission.
				if (!current_user_can('manage_options')) {
					throw new exception(__("You don't have permission to do this action", 'recoverabandoncart'));
				}

				$order_ids=array();
				$user_permission = fp_rac_get_cartlist_entry_restriction('user');
				$guest_permission = fp_rac_get_cartlist_entry_restriction('guest');
				if ('no' !== $user_permission || 'no' !== $guest_permission) {
					$args = array(
						'type' => 'shop_order',
						'numberposts' => '-1',
						'status' => isset($_POST['order_statuses']) ? wc_clean(wp_unslash($_POST['order_statuses'])) : array(),
						'return' => 'ids',
						'meta_query' => array(
							'relation' => 'AND',
							array(
								'key' => 'rac_order_placed',
								'compare' => 'NOT EXISTS',
							),
							array(
								'key' => 'guest_cart',
								'compare' => 'NOT EXISTS',
							),
							array(
								'key' => 'old_order_updated',
								'compare' => 'NOT EXISTS',
							),
						),
					);

					if ('no' == $user_permission && 'no' != $guest_permission) {
						$args['customer_id'] = '0';
					} elseif ('no' != $user_permission && 'no' == $guest_permission) {
						$args['customer_id'] = '0';
					}

					if ('all' !== $order_time) {
						if (!empty($_POST['from_time']) || !empty($_POST['to_time'])) {
							$from_strtotime = strtotime(wc_clean(wp_unslash($_POST['from_time'])));
							$to_strtotime = strtotime(wc_clean(wp_unslash($_POST['to_time'])));
							//convert std format
							$convert_from_date = gmdate('Y-m-d', $from_strtotime);
							$convert_to_date = gmdate('Y-m-d', $to_strtotime);
							$from_time_array = explode('-', $convert_from_date);
							$to_time_array = explode('-', $convert_to_date);
							if (!empty($_POST['to_time'])) {
								$date_query = array(
									'before' => array(
										'year' => $to_time_array[0],
										'month' => $to_time_array[1],
										'day' => $to_time_array[2],
									),
									'inclusive' => true,
								);
							} elseif (!empty($_POST['from_time'])) {
								$date_query = array(
									'after' => array(
										'year' => $from_time_array[0],
										'month' => $from_time_array[1],
										'day' => $from_time_array[2],
									),
									'inclusive' => true,
								);
							} else {
								$date_query = array(
									'after' => array(
										'year' => $from_time_array[0],
										'month' => $from_time_array[1],
										'day' => $from_time_array[2],
									),
									'before' => array(
										'year' => $to_time_array[0],
										'month' => $to_time_array[1],
										'day' => $to_time_array[2],
									),
									'inclusive' => true,
								);
							}
							$args = array_merge($args, array( 'date_query' => $date_query ));
						}
					}

					$order_ids = wc_get_orders($args);
				}

				delete_option('rac_abandon_previous_count');

				wp_send_json_success(array( 'ids'=>$order_ids ));
			} catch (Exception $ex) {
				wp_send_json_error(array( 'error' => $ex->getMessage() ));
			}   
		}

		/**
		 * Add order cart lists by order IDs.
		 *
		 * @since 1.0.0
		 */
		public static function add_order_cart_lists_by_order_ids() {
			check_ajax_referer('oldorder-cartlist', 'rac_security');

			try {
				$order_ids = isset($_POST['ids']) ? wc_clean(wp_unslash($_POST['ids'])) : '';
				$order_statuses = isset($_POST['order_statuses']) ? wc_clean(wp_unslash($_POST['order_statuses'])) : '';
				if (!$order_ids || !$order_statuses) {
					throw new exception(__('Invalid arguments', 'recoverabandoncart'));
				}

				// Return if the current user does not have permission.
				if (!current_user_can('manage_options')) {
					throw new exception(__("You don't have permission to do this action", 'recoverabandoncart'));
				}

				$bulk_process='done';
				if ('none'!==$order_ids) {
					$updated_count = 0;
					$looking_order_status = wc_clean(wp_unslash($_POST['order_statuses']));
					$order_statuses = fp_rac_format_previous_order_statuses($looking_order_status);
					$check_previous_data = get_option('rac_abandon_previous_count');
					if (rac_check_is_array($order_ids)) {
						foreach ($order_ids as $order_id) {
							$order = fp_rac_get_order_obj($order_id);
							$order_id = fp_rac_get_order_obj_data($order, 'id');
							$subscription_check = fp_rac_check_is_subscription($order_id);
							$paymentplan_check = fp_rac_check_is_payment_plan($order_id);
							if (!$subscription_check && !$paymentplan_check) {
								//check to, not importing order which are recovered and captured on place order

								if (in_array($order->get_status(), $order_statuses)) {
									$insert_id = FP_RAC_Insert_CartList_Entry::fp_rac_insert_old_order_entry($order_id);
									if ($insert_id) {
										$order->update_meta_data('old_order_updated', 'yes'); // this makes sure for no duplication
										$order->save();
										$updated_count++;
									}
								}
							}
						}
					}

					update_option('rac_abandon_previous_count', $check_previous_data + $updated_count);
				}

				wp_send_json_success(array( 'count' => get_option('rac_abandon_previous_count') ));
			} catch (Exception $ex) {
				wp_send_json_error(array( 'error' => $ex->getMessage() ));
			}
		}
	}

	FP_RAC_Previous_Order::init();
}
