<?php
/**
 * Admin Ajax.
 * 
 * @since 1.0.0
 */
if ( ! defined( 'ABSPATH' ) ) {
	exit ; // Exit if accessed directly.
}
if ( ! class_exists( 'FP_RAC_Admin_Ajax' ) ) {

	/**
	 * Class.
	 * 
	 * @since 1.0.0
	 */
	class FP_RAC_Admin_Ajax {

		/**
		 * Actions.
		 *
		 * @string 1.0.0
		 * @var array
		 */
		private static $actions = array(
			'recover_cart_list' => false,
			'update_guest_cartlist_email'=>false,
			'get_cartlist_popup_content'=>false,
			'get_cartlist_recover_popup_content'=>false,
			'add_cartlist_recovered_order_id'=>false,
			'toggle_email_template_status' => false,
			'toggle_cartlist_email_sending_status' => false,
			'rearrange_email_product_info_columns' => false,
			'save_guest_cartlist_entry'=>true,
			'set_guest_details_in_cookie'=>true,
			'set_guest_popup_viewed_cookie'=>true,
			'set_guest_popup_delay_time'=>true,
			'handle_checkout_gdpr'=>true,
			'toggle_customer_email_subscription'=>true,
		);

		/**
		 * Class initialization.
		 *
		 * @since 1.0.0
		 */
		public static function init() {
			foreach (self::$actions as $action => $nopriv) {
				add_action('wp_ajax_rac_' . $action, array( __CLASS__, $action ));

				if (!$nopriv) {
					continue;
				}

				add_action('wp_ajax_nopriv_rac_' . $action, array( __CLASS__, $action ));
			}
		}

		/**
		 * Toggle cart list email sending status.
		 * 
		 * @since 12.8.0
		 */
		public static function toggle_cartlist_email_sending_status() {
			check_ajax_referer( 'cartlist', 'rac_security' ) ;

			try {
				$cart_list_id = isset($_POST['cart_list_id']) ? absint($_POST['cart_list_id']) : 0;
				$current_status = isset($_POST['status']) ? wc_clean(wp_unslash($_POST['status'])) : '';
				if (!$cart_list_id || !$current_status) {
					throw new exception(__('Invalid arguments', 'recoverabandoncart'));
				}

				$cart_list=get_post($cart_list_id);
				if (!is_object($cart_list) || 'raccartlist'!== $cart_list->post_type) {
					throw new exception(__('Invalid arguments', 'recoverabandoncart'));
				}

				if (!in_array($current_status, array( 'DONT', 'SEND' ))) {
					throw new exception(__('Unable to update - current status is invalid.', 'recoverabandoncart'));
				}

				// Return if the current user does not have permission.
				if (!current_user_can('edit_post', $cart_list_id)) {
					throw new exception(__("You don't have permission to do this action", 'recoverabandoncart'));
				}

				if ($current_status === $cart_list->sending_status) {
					throw new exception(__('Cart list already has the same status.', 'recoverabandoncart'));
				}
					
				update_post_meta( $cart_list_id, 'rac_cart_sending_status', $current_status ) ;

				if ('SEND' === $current_status) {
					$status = 'DONT';
					$button_text=__('Stop Mailing', 'recoverabandoncart');
				} else {
					$status = 'SEND' ;
					$button_text=__('Start Mailing', 'recoverabandoncart');
				}

				wp_send_json_success(array( 'status' => $status, 'button_text'=> $button_text ));
			} catch (Exception $ex) {
				wp_send_json_error(array( 'error' => $ex->getMessage() ));
			}           
		}

		/**
		 * Recover cart list.
		 * 
		 * @since 12.8.0
		 */
		public static function recover_cart_list() {
			check_ajax_referer( 'cartlist', 'rac_security' ) ;

			try {
				$cart_list_id = isset($_POST['cart_list_id']) ? absint($_POST['cart_list_id']) : '';
				if (!$cart_list_id) {
					throw new exception(__('Invalid arguments', 'recoverabandoncart'));
				}

				$cart_list=get_post($cart_list_id);
				if (!is_object($cart_list) || 'raccartlist'!== $cart_list->post_type) {
					throw new exception(__('Invalid arguments', 'recoverabandoncart'));
				}

				// Return if the current user does not have permission.
				if (!current_user_can('edit_post', $cart_list_id)) {
					throw new exception(__("You don't have permission to do this action", 'recoverabandoncart'));
				}

				if ('rac-cart-recovered'===$cart_list->post_status) {
					throw new exception(__('This cartlist has already been recovered', 'recoverabandoncart'));
				}
					
				wp_update_post( array(
						'ID'          => $cart_list_id,
						'post_status' => 'rac-cart-recovered',
							)  ) ;

				update_post_meta( $cart_list_id, 'rac_recover_method', '1' ) ;

				wp_send_json_success();
			} catch (Exception $ex) {
				wp_send_json_error(array( 'error' => $ex->getMessage() ));
			}
		}

		/**
		 * Update the guest email manually in cart list table.
		 * 
		 * @since 12.8.0
		 */
		public static function update_guest_cartlist_email() {
			check_ajax_referer( 'cartlist', 'rac_security' ) ;

			try {
				$cart_list_id = isset($_POST['cart_list_id']) ? absint($_POST['cart_list_id']) : '';
				if (!$cart_list_id) {
					throw new exception(__('Invalid arguments', 'recoverabandoncart'));
				}

				if ('raccartlist'!==get_post_type($cart_list_id)) {
					throw new exception( esc_html__( 'Invalid Arguments', 'recoverabandoncart' ) ) ;
				}

				$email = isset($_POST['email']) ? wc_clean( wp_unslash($_POST['email'])) : '';
				if (!$email) {
					throw new exception(__('Please enter the email ID', 'recoverabandoncart'));
				}

				if (sanitize_email($email) !== $email) {
					throw new exception(__('Please enter a valid email address.', 'recoverabandoncart'));
				}

				// Return if the current user does not have permission.
				if (!current_user_can('edit_post', $cart_list_id)) {
					throw new exception(__("You don't have permission to do this action", 'recoverabandoncart'));
				}

				$cart_list                      = fp_rac_create_cart_list_obj( $cart_list_id ) ;
				$cart_details                   = fp_rac_format_cart_details( $cart_list->cart_details, $cart_list ) ;
				$cart_details[ 'visitor_mail' ] = $email ;
				$details                        = base64_encode( maybe_serialize( $cart_details ) ) ;
				
				update_post_meta( $cart_list_id, 'rac_cart_details', $details ) ;

				wp_send_json_success();
			} catch (Exception $ex) {
				wp_send_json_error(array( 'error' => $ex->getMessage() ));
			}
		}

		/**
		 * Toggle email template status.
		 * 
		 * @since 12.8.0
		 */
		public static function toggle_email_template_status() {
			check_ajax_referer( 'email-template', 'rac_security' ) ;

			try {
				$email_template_id = isset($_POST['email_template_id']) ? absint($_POST['email_template_id']) : 0;
				$current_status = isset($_POST['status']) ? wc_clean(wp_unslash($_POST['status'])) : '';
				if (!$email_template_id || !$current_status) {
					throw new exception(__('Invalid arguments', 'recoverabandoncart'));
				}

				$email_template=get_post($email_template_id);
				if (!is_object($email_template) || 'racemailtemplate'!== $email_template->post_type) {
					throw new exception(__('Invalid arguments', 'recoverabandoncart'));
				}

				// Return if the current user does not have permission.
				if (!current_user_can('edit_post', $email_template_id)) {
					throw new exception(__("You don't have permission to do this action", 'recoverabandoncart'));
				}

				if ('rac' . $current_status !== $email_template->post_status) {
					throw new exception(__('Email template already has the same status.', 'recoverabandoncart'));
				}
					
				wp_update_post( array(
						'ID'          => $email_template_id,
						'post_status' => 'active' !== $current_status ? 'racactive' : 'racinactive',
							) ) ;

				wp_send_json_success();
			} catch (Exception $ex) {
				wp_send_json_error(array( 'error' => $ex->getMessage() ));
			}
		}

		/**
		 * Rearrange the email product info columns.
		 * 
		 * @since 12.8.0
		 */
		public static function rearrange_email_product_info_columns() {
			check_ajax_referer( 'fp-rac-sortable', 'rac_security' ) ;

			try {
				$sorted_columns = isset($_POST['sorted_columns']) ? wc_clean(wp_unslash($_POST['sorted_columns'])) : '';
				if (!$sorted_columns || ! fp_rac_check_is_array($sorted_columns)) {
					throw new exception(__('Invalid arguments', 'recoverabandoncart'));
				}

				// Return if the current user does not have permission.
				if (!current_user_can('manage_options')) {
					throw new exception(__("You don't have permission to do this action", 'recoverabandoncart'));
				}
					
				update_option( 'drag_and_drop_product_info_sortable_column', $sorted_columns ) ;

				wp_send_json_success();
			} catch (Exception $ex) {
				wp_send_json_error(array( 'error' => $ex->getMessage() ));
			}
		}

		/**
		 * Get the cart list popup content.
		 * 
		 * @since 12.8.0
		 */
		public static function get_cartlist_popup_content() {
			check_ajax_referer( 'cartlist', 'rac_security' ) ;

			try {
				if ( ! isset( $_POST[ 'cart_list_id' ] ) ) {
					throw new exception( __( 'Invalid Arguments', 'recoverabandoncart' ) ) ;
				}

				$cart_list_id       = absint( $_POST[ 'cart_list_id' ] ) ;
				if ('raccartlist'!==get_post_type($cart_list_id)) {
					throw new exception( esc_html__( 'Invalid Arguments', 'recoverabandoncart' ) ) ;
				}

				// Return if the current user does not have permission.
				if (!current_user_can('read_post', $cart_list_id)) {
					throw new exception(__("You don't have permission to do this action", 'recoverabandoncart'));
				}

				$cart_list          = fp_rac_create_cart_list_obj( $cart_list_id ) ;
				fp_rac_wpml_switch_lang( $cart_list->wpml_lang ) ;
				ob_start() ;
				?>
				<div class="fp_rac_popup_wrapper">
					<div class="fp_rac_email_info_popup_content">
						<div class="fp_rac_email_info_popup_header">
							<label class="rac_email_info_popup_label">
								<?php echo esc_html( __( 'Cart', 'recoverabandoncart' ) . ' #' . $cart_list_id ) ; ?>
							</label> </div>
						<div class="fp_rac_email_info_popup_close"> <img src=<?php echo esc_url( RAC_PLUGIN_URL . '/assets/images/close.png' ) ; ?> class="rac_popup_close"> </div>
						<div class="fp_rac_email_info_popup_body">
							<div class="fp_rac_email_info_popup_body_content">

								<!-- To Display Abandoned product information -->
								<div class="fp_rac_email_info_product">
									<?php echo do_shortcode( FP_RAC_Polish_Product_Info::fp_rac_extract_cart_details( $cart_list, false ) ) ; ?>
								</div>

								<!-- To Display email status information -->
								<div class="fp_rac_email_info_status">
									<table class="fp_rac_email_info_table">
										<tr>
											<th><?php esc_html_e( 'Email Template', 'recoverabandoncart' ) ; ?></th>
											<th><?php esc_html_e( 'Email Status', 'recoverabandoncart' ) ; ?></th>
											<th><?php esc_html_e( 'Cart Link Status', 'recoverabandoncart' ) ; ?></th>
										</tr>
										<?php
										$mail_sent          = maybe_unserialize( $cart_list->mail_template_id ) ;
										$arg                = array( 'posts_per_page' => -1, 'post_status' => array( 'racactive', 'racinactive' ), 'post_type' => 'racemailtemplate', 'order' => 'ASC', 'orderby' => 'ID' ) ;
										$email_template_all = fp_rac_check_query_having_posts( $arg ) ;

										if ( rac_check_is_array( $email_template_all ) ) {
											foreach ( $email_template_all as $check_all_email_temp ) {
												$old_email_id = get_post_meta( $check_all_email_temp->ID, 'rac_old_template_id', true ) ;
												?>
												<tr>
													<td><?php echo esc_html( $check_all_email_temp->post_title ) ; ?></td>

													<td>
														<?php
														if ( ! empty( $mail_sent ) ) {
															if ( in_array( $old_email_id, ( array ) $mail_sent ) ) {
																esc_html_e( 'Sent', 'recoverabandoncart' ) ;
															} else {
																esc_html_e( 'Not Sent', 'recoverabandoncart' ) ;
															}
														} else {
															esc_html_e( 'Not Sent', 'recoverabandoncart' ) ;
														}
														?>
													</td>

													<td>
														<?php
														if ( ! empty( $cart_list->link_status ) ) {
															$mails_clicked = maybe_unserialize( $cart_list->link_status ) ;

															if ( in_array( $old_email_id, ( array ) $mails_clicked ) ) {
																esc_html_e( 'Clicked', 'recoverabandoncart' ) ;
															} else {
																esc_html_e( 'Not Clicked', 'recoverabandoncart' ) ;
															}
														} else {
															esc_html_e( 'Not Clicked', 'recoverabandoncart' ) ;
														}
														?>
													</td>

													<?php
											}
										}
										?>
										</tr>
									</table>
								</div>
							</div>
							<div class="fp_rac_email_info_popup_footer">
								<input type="button" class="rac_email_info_popup_close_btn" value="<?php esc_html_e( 'Close', 'recoverabandoncart' ) ; ?>">
							</div>
						</div>
					</div>
				</div>
				<?php
				fp_rac_wpml_switch_lang() ;

				$popup = ob_get_clean() ;
				wp_send_json_success( array( 'content' => $popup ) ) ;
			} catch ( Exception $e ) {
				wp_send_json_error( array( 'error' => $e->getMessage() ) ) ;
			}
		}

		/**
		 * Get the cartlist recover popup content.
		 * 
		 * @since 12.8.0
		 */
		public static function get_cartlist_recover_popup_content() {
			check_ajax_referer( 'cartlist', 'rac_security' ) ;
			
			try {
				if ( ! isset( $_POST[ 'cart_list_id' ] ) ) {
					throw new exception( __( 'Invalid Arguments', 'recoverabandoncart' ) ) ;
				}

				$cart_list_id = absint( $_POST[ 'cart_list_id' ] ) ;
				if ('raccartlist'!==get_post_type($cart_list_id)) {
					throw new exception( esc_html__( 'Invalid Arguments', 'recoverabandoncart' ) ) ;
				}

				// Return if the current user does not have permission.
				if (!current_user_can('read_post', $cart_list_id)) {
					throw new exception(__("You don't have permission to do this action", 'recoverabandoncart'));
				}

				ob_start() ;
				?>
				<div class="fp_rac_popup_wrapper">
					<div class="fp_rac_popup_content">
						<div class="fp_rac_popup_header"><label class="rac_popup_label"> <?php esc_html_e( 'Manual Recover Form', 'recoverabandoncart' ) ; ?></label> </div>
						<div class="fp_rac_popup_close"> <img src=<?php echo esc_url( RAC_PLUGIN_URL . '/assets/images/close.png' ) ; ?> class="rac_popup_close"> </div>
						<div class="fp_rac_popup_body">
							<div class="fp_rac_popup_body_content">
								<label class="rac_manual_order_id_label"><?php esc_html_e( 'Enter Order ID', 'recoverabandoncart' ) ; ?></label>
								<input type="number" class="rac_manual_order_id_num"><br>
								<input type="hidden" class="rac_cart_list_id" value ="<?php echo esc_attr( $cart_list_id ) ; ?>"/>
							</div>
						</div>
						<div class="fp_rac_popup_footer">
							<input type="button" class="rac_manual_order_entry_btn" value="<?php esc_attr_e( 'Enter Order ID', 'recoverabandoncart' ) ; ?>">
						</div>
					</div>
				</div>
				<?php
				$popup        = ob_get_clean() ;
				wp_send_json_success( array( 'content' => $popup ) ) ;
			} catch ( Exception $e ) {
				wp_send_json_error( array( 'error' => $e->getMessage() ) ) ;
			}
		}

		/**
		 * Add the cartlist recovered order ID.
		 * 
		 * @since 12.8.0
		 */
		public static function add_cartlist_recovered_order_id() {
			check_ajax_referer( 'cartlist', 'rac_security' ) ;

			try {
				if ( ! isset( $_POST[ 'order_id' ] ) || ! isset( $_POST[ 'cart_list_id' ] ) ) {
					throw new exception( __( 'Invalid Arguments', 'recoverabandoncart' ) ) ;
				}

				$order_id     = absint( $_POST[ 'order_id' ] ) ;
				$cart_list_id = absint( $_POST[ 'cart_list_id' ] ) ;
				if ('raccartlist'!==get_post_type($cart_list_id)) {
					throw new exception( esc_html__( 'Invalid Arguments', 'recoverabandoncart' ) ) ;
				}

				$order_obj = fp_rac_get_order_obj( $order_id ) ;
				if ( ! $order_obj ) {
					throw new exception( __( 'Invalid Order ID', 'recoverabandoncart' ) ) ;
				}

				// Return if the current user does not have permission.
				if (!current_user_can('edit_post', $cart_list_id)) {
					throw new exception(__("You don't have permission to do this action", 'recoverabandoncart'));
				}

				$saved_order_id = get_post_meta( $cart_list_id, 'order_id', true ) ;
				if ( ! empty( $saved_order_id ) ) {
					throw new exception( __( 'Order ID already exists', 'recoverabandoncart' ) ) ;
				}

				$product_ids = array() ;
				$get_items   = $order_obj->get_items() ;

				if ( rac_check_is_array( $get_items ) ) {
					foreach ( $get_items as $product ) {
						$product_ids[] = $product[ 'product_id' ] ;
					}
				}
				$args = array(
					'rac_order_id'              => $order_id,
					'rac_cart_id'               => $cart_list_id,
					'rac_product_details'       => implode( ',', $product_ids ),
					'rac_recovered_sales_total' => fp_rac_get_order_obj_data( $order_obj, 'order_total' ),
					'rac_recovered_date'        => strtotime( fp_rac_get_order_obj_data( $order_obj, 'order_date' ) ),
						) ;

				fp_rac_insert_recovered_order_post( $args ) ;

				update_post_meta( $cart_list_id, 'rac_recovered_order_id', $order_id ) ;
				update_post_meta( $cart_list_id, 'rac_cart_payment_details', true ) ;

				wp_send_json_success( array( 'content' => __( 'Order ID Updated', 'recoverabandoncart' ) ) ) ;
			} catch ( Exception $e ) {
				wp_send_json_error( array( 'error' => $e->getMessage() ) ) ;
			}
		}

		/**
		 * Set the guest popup delay time in the cookie.
		 * 
		 * @since 12.8.0
		 */
		public static function set_guest_popup_delay_time() {
			check_ajax_referer( 'rac_popup_delay', 'rac_security' ) ;

			try {
				setcookie( 'rac_guest_popup_clicked_time', time(), time() + 3600, '/' ) ;
				
				wp_send_json_success() ;
			} catch ( Exception $ex ) {
				wp_send_json_error( array( 'error' => $ex->getMessage() ) ) ;
			}
		}

		/**
		 * Accept/Reject the GDPR by the user in the checkout.
		 *
		 * @since 1.0.0
		 */
		public static function handle_checkout_gdpr() {
			check_ajax_referer( 'gdpr-nonce', 'rac_security' ) ;

			try {
				$gdpr_accepted = isset( $_REQUEST[ 'gdpr_accepted' ] ) ? wc_clean( wp_unslash( $_REQUEST[ 'gdpr_accepted' ] ) ) : '' ; // @codingStandardsIgnoreLine.
				if ( empty( $gdpr_accepted ) ) {
					throw new exception( esc_html__( 'Invalid Arguments', 'recoverabandoncart' ) ) ;
				}

				if ( 'no' === $gdpr_accepted ) {
					$cookie_value = 'no' ;
					if ( isset( $_COOKIE[ 'rac_checkout_entry' ] ) ) {
						$abandoned_cart_id=absint( $_COOKIE[ 'rac_checkout_entry' ] );
						if ('raccartlist'!==get_post_type($abandoned_cart_id)) {
							throw new exception( esc_html__( 'Invalid Arguments', 'recoverabandoncart' ) ) ;
						}

						wp_delete_post( $abandoned_cart_id , true ) ;
						setcookie( 'rac_checkout_entry' , '' , -1 , '/' ) ;
					}
				} else {
					$cookie_value = 'yes' ;
				}

				setcookie( 'rac_gdpr', $cookie_value, time() + 3600, '/' ) ;

				wp_send_json_success() ;
			} catch ( Exception $ex ) {
				wp_send_json_error( array( 'error' => $ex->getMessage() ) ) ;
			}
		}

		/**
		 * Add/update guest cartlist entry from the checkout.
		 * 
		 * @since 12.8.0
		 */
		public static function save_guest_cartlist_entry() {
			check_ajax_referer( 'guest-entry', 'rac_security' ) ;

			try {
				$email = isset($_POST['rac_email']) ? wc_clean(wp_unslash($_POST['rac_email'])) : '';
				if (!$email) {
					throw new exception(__('Email cannot be empty', 'recoverabandoncart'));
				}

				if (sanitize_email($email) !== $email) {
					throw new exception(__('Please enter a valid email address.', 'recoverabandoncart'));
				}

				if (is_user_logged_in() || !FP_RAC_Insert_CartList_Entry::get_user_permission()) {
					throw new exception(__('Insufficient permissions.', 'recoverabandoncart'));
				}

				$guest_details_from_cookie= FP_RAC_Insert_CartList_Entry::get_guest_details_from_cookie();
				$first_name = isset($_POST['rac_first_name']) ? wc_clean(wp_unslash($_POST['rac_first_name'])) : '';
				$last_name = isset($_POST['rac_last_name']) ? wc_clean(wp_unslash($_POST['rac_last_name'])) : '';
				$phone = isset($_POST['rac_phone']) ? wc_clean(wp_unslash($_POST['rac_phone'])) : '';
				$user_details = array( $first_name, $last_name, $phone, $email );
				$only_update = isset($_COOKIE['rac_checkout_entry']) || ( isset($guest_details_from_cookie['email'])&&!empty($guest_details_from_cookie['email']) ) ? 'yes' : '';
				$current_user_lang = isset($_POST['rac_lang']) ? wc_clean(wp_unslash($_POST['rac_lang'])) : fp_rac_get_current_language();
				
				FP_RAC_Insert_CartList_Entry::fp_rac_insert_update_cartlist_functionality($user_details, $current_user_lang, 'ajax', $only_update, 2);

				wp_send_json_success();
			} catch (Exception $ex) {
				wp_send_json_error(array( 'error' => $ex->getMessage() ));
			}
		}

		/**
		 * Set the guest details in the cookie from the popup.
		 * 
		 * @since 12.8.0
		 */
		public static function set_guest_details_in_cookie() {
			check_ajax_referer( 'guest-entry', 'rac_security' ) ;

			try {
				$email = isset($_POST['cookie_guest_email']) ? wc_clean(wp_unslash($_POST['cookie_guest_email'])) : '';
				if (!$email) {
					throw new exception(__('Email cannot be empty', 'recoverabandoncart'));
				}

				if (sanitize_email($email) !== $email) {
					throw new exception(__('Please enter a valid email address.', 'recoverabandoncart'));
				}

				if (is_user_logged_in()) {
					throw new exception(__('Insufficient permissions.', 'recoverabandoncart'));
				}

				$first_name = isset($_POST['cookie_guest_fname']) ? wc_clean(wp_unslash($_POST['cookie_guest_fname'])) : '';
				$last_name = isset($_POST['cookie_guest_lname']) ? wc_clean(wp_unslash($_POST['cookie_guest_lname'])) : '';
				$phone_no = isset($_POST['cookie_guest_phone_no']) ? wc_clean(wp_unslash($_POST['cookie_guest_phone_no'])) : '';
				$details = json_encode(array( 'email' => $email, 'firstname' => $first_name, 'lastname' => $last_name, 'phone_no' => $phone_no ));

				if ('yes' === get_option('rac_guest_popup_gdpr_enabled')) {
					setcookie('rac_gdpr', 'yes', time() + 3600, '/');
				}

				setcookie('raccookie_guest_email', $details, time() + 3600, '/');

				wp_send_json_success();
			} catch (Exception $ex) {
				wp_send_json_error(array( 'error' => $ex->getMessage() ));
			}
		}

		/**
		 * Set the guest popup viewed cookie.
		 * 
		 * @since 12.8.0
		 */
		public static function set_guest_popup_viewed_cookie() {
			check_ajax_referer( 'guest-entry', 'rac_security' ) ;

			try {
				if (is_user_logged_in()) {
					throw new exception(__('Insufficient permissions.', 'recoverabandoncart'));
				}
				
				setcookie('rac_guest_popup_already_displayed', 'yes', time() + 3600, '/');

				wp_send_json_success();
			} catch (Exception $ex) {
				wp_send_json_error(array( 'error' => $ex->getMessage() ));
			}
		}

		/**
		 * Toggle the customer email subscription.
		 * 
		 * @since 12.8.0
		 */
		public static function toggle_customer_email_subscription() {
			check_ajax_referer( 'unsubscribe-email', 'rac_security' ) ;

			try {

				$userid      = isset($_POST['user_id']) ? absint( $_POST[ 'user_id' ] ): 0 ;
				$unsubscribe = isset($_POST['unsubscribe']) ? wc_clean(wp_unslash($_POST['unsubscribe'])) : '';
				if (!$userid || !$unsubscribe) {
					throw new exception(__('Invalid arguments.', 'recoverabandoncart'));
				}

				if ($userid && !get_userdata($userid)) {
					throw new exception(__('Provided user ID is not valid.', 'recoverabandoncart'));
				}

				$email = isset($_POST['email']) ? wc_clean(wp_unslash($_POST['email'])) : '';
				if ($email && sanitize_email($email) !== $email) {
					throw new exception(__('Provided email address is not valid.', 'recoverabandoncart'));
				}

				$userid = ( !$userid ) ? rac_check_is_member_or_guest( $email, true ) : $userid ;
				if ( $userid ) {
					if ( 'yes' === $unsubscribe ) {
						update_user_meta( $userid, 'fp_rac_mail_unsubscribed', 'yes' ) ;
					} else {
						delete_user_meta( $userid, 'fp_rac_mail_unsubscribed' ) ;
					}
				} else {
					$unsubscribed_emails = ( array ) get_option( 'fp_rac_mail_unsubscribed' ) ;
					if ( 'yes' === $unsubscribe ) {
						if ( ! in_array( $email_id, $unsubscribed_emails ) ) {
							$unsubscribed_emails[] = $email_id ;
						}
					} else {
						$key = array_search( $email_id, $unsubscribed_emails ) ;
						if ( false !== $key ) {
								unset( $unsubscribed_emails[ $key ] ) ;
						}
					}

					update_option( 'fp_rac_mail_unsubscribed', $unsubscribed_emails ) ;
				}

				$message=( 'yes' === $unsubscribe )? __( 'Successfully Unsubscribed.', 'recoverabandoncart' ): __( 'Successfully Subscribed.', 'recoverabandoncart' );

				wp_send_json_success(array( 'message' => $message ));
			} catch (Exception $ex) {
				wp_send_json_error(array( 'error' => $ex->getMessage() ));
			}
		}
	}

	FP_RAC_Admin_Ajax::init() ;
}
